/* GAdmin-Rsync - An easy to use GTK+ frontend for the rsync backup client and server.
 * Copyright (C) 2007 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/



#include "../config.h"
#include <string.h>
#include <gtk/gtk.h>
#include "gettext.h"
#include "support.h"
#include "widgets.h"
#include "gadmin-rsync.h"
#include "create_main_window.h"
#include "create_rsync_tab.h"
#include "create_progress_tab.h"
#include "create_disc_tab.h"
#include "populate_rsync_tab.h"
#include "populate_disc_tab.h"
#include "commands.h"
#include "status_update.h"
#include "set_version.h"
#include "show_info.h"

/* For getuid */
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>

#include <stdlib.h> // new
#include <key_handling.h>

int MAX_READ_POPEN = 16384;
int info_window_exit_main = 0;

/* Cancel backup process */
int cancel_backup = 0;

/* admin mode = 1
   user mode  = 0
   Only mode 1 can be used for now */
int admin_mode = 1;

/* Settings and output directories */
gchar *global_settings_dir;
gchar *global_scripts_dir;
gchar *global_script_log_dir;
gchar *global_key_path;
gchar *global_backup_name;
G_CONST_RETURN gchar *global_home_dir;



int main(int argc, char *argv[])
{
    int i = 0;
    gchar *cmd, *info;
    int fatal_error = 0;
//    const char *askpass="SSH_ASKPASS";

#ifdef ENABLE_NLS
    bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
    bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
    textdomain(GETTEXT_PACKAGE);
#endif

    gtk_set_locale();
    gtk_init(&argc, &argv);

    /* Unset SSH_ASKPASS environment variable */
//    if( ! unsetenv(askpass) )
//      printf("Unable to unset SSH_ASKPASS environment variable\n");


    wid* widgets = g_malloc(sizeof(wid));

    /* Non root usage */
    if( ! getuid() == 0 )
    {
	/* For setting a different exit method in the info window */
	info_window_exit_main = 1;

        info = g_strdup_printf("You must be root to run: %s\nThis window will close in 10 seconds.\n", PACKAGE);
        show_info(info);
        g_free(info);

	/* Users can close the info window earlier then the timeout */
	for(i=0; i<10; i++)
	{
    	    while(gtk_events_pending())
                  gtk_main_iteration();

	    /* Set when close info window is clicked */
	    if( info_window_exit_main == 2 )
	      break;

    	    usleep(100000*10);
    	}

	g_free(widgets);
	return 0;
    }


    /* Set the directories to use for input and output */ 
    global_home_dir = g_get_home_dir();
    if( global_home_dir == NULL )
    {
	printf("Cant get your home directory.\nUse: \"su - UserName\" to set your environment.\n");
	g_free(widgets);
	return 0;
    }

    /* Settings directories based on mode */
    if( admin_mode )
    {
	global_settings_dir   = g_strdup_printf("%s/gadmin-rsync/settings", SYSCONFDIR);
	global_scripts_dir    = g_strdup_printf("%s/gadmin-rsync/scripts", SYSCONFDIR);
	global_script_log_dir = g_strdup_printf("%s/log/gadmin-rsync", LOCALSTATEDIR);
    }
    else
    {
	global_settings_dir   = g_strdup_printf("%s/.gadmin-rsync/settings", global_home_dir);
	global_scripts_dir    = g_strdup_printf("%s/.gadmin-rsync/scripts", global_home_dir);
	global_script_log_dir = g_strdup_printf("%s/.gadmin-rsync/logs", global_home_dir);
    }
    /* We want keys stored in a directory under users homes */
    global_key_path = g_strdup_printf("%s/.gadmin-rsync/keys", global_home_dir);


    /* Create the settings directory and global_key_path */
    cmd = g_strdup_printf("mkdir -p %s %s", global_settings_dir, global_key_path);
    if( ! run_command(cmd) )
    {
	fatal_error = 1;

        info = g_strdup_printf("Error: Can not create directory:\n%s\n", global_settings_dir);
        show_info(info);
        g_free(info);
    }
    g_free(cmd);

    /* Create the scripts directory */
    cmd = g_strdup_printf("mkdir -p %s", global_scripts_dir);
    if( ! run_command(cmd) )
    {
	fatal_error = 1;

        info = g_strdup_printf("Error: Can not create directory:\n%s\n", global_scripts_dir);
        show_info(info);
        g_free(info);
    }
    g_free(cmd);

    /* Create the script log directory */
    cmd = g_strdup_printf("mkdir -p %s", global_script_log_dir);
    if( ! run_command(cmd) )
    {
	fatal_error = 1;

        info = g_strdup_printf("Error: Can not create directory:\n%s\n", global_script_log_dir);
        show_info(info);
        g_free(info);
    }
    g_free(cmd);


    /* Create the main window */
    create_main_window(widgets);
    /* Set rsync version */
    set_version(widgets);

    if( ! fatal_error )
    {
	/* Create the tabs */
	create_rsync_tab(widgets);
	create_progress_tab(widgets);
	create_disc_tab(widgets);

	/* Populate the rsync client tab */
	populate_rsync_tab(widgets);

	/* Populate the disc tab */
	populate_disc_tab(widgets);

	/* Get the selected backup name from the backup selection combo */
	global_backup_name = gtk_combo_box_get_active_text(GTK_COMBO_BOX(widgets->rsync_set_combo[0]));
    }

    /* Window close button (x) */   
    g_signal_connect(GTK_OBJECT(widgets->main_window), "destroy",
	             GTK_SIGNAL_FUNC(gtk_main_quit), NULL);


    gtk_timeout_add(1000, (GtkFunction) status_update, widgets);


    gtk_main();

    g_free(widgets);
    g_free(global_settings_dir);
    g_free(global_scripts_dir);
    g_free(global_script_log_dir);
    g_free(global_key_path);
    g_free(global_backup_name);

    return 0;    
}
