/**************************************************************************
     fotox.cpp     image processing program
     Free software licensed under GNU General Public License v.2
     Source URL: kornelix.squarespace.com/fotox
***************************************************************************/

#include <gdk/gdkkeysyms.h>
#include "zfuncs.h"

#define fversion "fotox v.42"
#define flicense "Free software - GNU General Public License v.2"
#define fsource "http://kornelix.squarespace.com/fotox"

#define mega (1048576.0)                                                   //  1024 * 1024
#define undomax 20                                                         //  undo stack size
#define nodither GDK_RGB_DITHER_NONE,0,0
#define colorspace GDK_COLORSPACE_RGB
#define interp GDK_INTERP_BILINEAR
#define dottedline GDK_LINE_DOUBLE_DASH,GDK_CAP_BUTT,GDK_JOIN_MITER

#define pixbuf_new(a,b,c,d,e) gdk_pixbuf_new(a,b,c,d,e); incr_pixbufs();
#define pixbuf_scale_simple(a,b,c,d) gdk_pixbuf_scale_simple(a,b,c,d); incr_pixbufs();
#define pixbuf_new_from_file(a,b) gdk_pixbuf_new_from_file(a,b); incr_pixbufs();
#define pixbuf_copy(a) gdk_pixbuf_copy(a); incr_pixbufs();
#define pixbuf_rotate(a,b) gdk_pixbuf_rotate(a,b); incr_pixbufs();
#define pixbuf_copy_area(a,b,c,d,e,f,g,h) gdk_pixbuf_copy_area(a,b,c,d,e,f,g,h);
#define pixbuf_test(p) if (!p) zappcrash(xtext("memory allocation failure"));

#define pixbuf_poop(ID)                                                    \
   ww##ID = gdk_pixbuf_get_width(pxb##ID);                                 \
   hh##ID = gdk_pixbuf_get_height(pxb##ID);                                \
   rs##ID = gdk_pixbuf_get_rowstride(pxb##ID);                             \
   ppix##ID = gdk_pixbuf_get_pixels(pxb##ID);

#define getparm(pname) parm_##pname = getParm(#pname);
#define setparm(pname,value) parm_##pname = value; setParm(#pname,parm_##pname);

extern char    JPGquality[4];                                              //  quality param for gdk_pixbuf_save()

GtkWidget      *mwin, *drmwin;                                             //  main and drawing window
GtkWidget      *mtbar, *mmbar, *stbar;                                     //  menu bar, tool bar, status bar
GError         **gerror = 0;
GdkGC          *gdkgc = 0;                                                 //  graphics context
GdkColor       black, white;
GdkColormap    *colormap = 0;
uint           maxcolor = 0xffff;
pthread_t      tid;

//  pixbufs: pxb1, pxb2 = unchanged input images
//           pxb3 = modified output image
//           pxbM = main/drawing window image
//  pxb3 (org3x,org3y) * Rscale >> pxbM >> main/drawing window (posMx,posMy)

GdkPixbuf   *pxb1 = 0, *pxb2 = 0;                                          //  input image1 and image2
GdkPixbuf   *pxb3 = 0;                                                     //  modified / output image3
GdkPixbuf   *pxb1A = 0, *pxb2A = 0;                                        //  alignment images
GdkPixbuf   *pxb1C = 0, *pxb2C = 0;                                        //  alignment images, curved
GdkPixbuf   *pxbM = 0;                                                     //  main window drawing image

int      nch, nbits;                                                       //  pixbuf/image attributes
int      ww1, hh1, rs1;                                                    //  pxb1  width, height, rowstride
int      ww2, hh2, rs2;                                                    //  pxb2  width, height, rowstride
int      ww3, hh3, rs3;                                                    //  pxb3  width, height, rowstride
int      ww1A, hh1A, rs1A;                                                 //  pxb1A width, height, rowstride
int      ww2A, hh2A, rs2A;                                                 //  pxb2A width, height, rowstride
int      ww1C, hh1C, rs1C;                                                 //  pxb1C width, height, rowstride
int      ww2C, hh2C, rs2C;                                                 //  pxb2C width, height, rowstride
int      wwM, hhM, rsM;                                                    //  pxbM  width, height, rowstride

int      mwinW = 800, mwinH = 600;                                         //  main window initial size
int      org3x = 0, org3y = 0;                                             //  image3/pxb3 origin in pxbM 
int      posMx = 0, posMy = 0;                                             //  pxbM position in main/drawing window 
int      Mrefresh = 0;                                                     //  flag: refresh/rescale main window
int      Mscale = 0;                                                       //  image3 scale: 0 (fit window), 1x, 2x
double   Rscale = 1;                                                       //  pxb3 to pxbM scaling factor
int      Fredraw = 0;                                                      //  flag: main window was updated
int      Fslideshow = 0;                                                   //  flag: slideshow mode

int      mpWx, mpWy;                                                       //  mouse position in main window
int      mpMx, mpMy;                                                       //  corresp. position in pxbM
int      mp3x, mp3y;                                                       //  corresp. position in pxb3
int      mp3xcen = 0, mp3ycen = 0;                                         //  last click position >> window center
int      LMclick = 0, RMclick = 0;                                         //  L/R-mouse button clicked
int      LMdown = 0;                                                       //  L-mouse button down (drag underway)
int      LMcapture = 0;                                                    //  flag: L-mouse --> application handler
int      mdx1, mdy1, mdx2, mdy2;                                           //  mouse drag displacement
int      KBkey = 0;                                                        //  keyboard key

typedef  uchar *pixel;                                                     //  3 RGB values, 0-255 each
pixel    ppix1, ppix2, ppix3;                                              //  pxb1, pxb2, pxb3  pixels
pixel    ppix1A, ppix2A, ppix1C, ppix2C;                                   //  pxb1A, pxb2A, pxb1C, pxb2C  pixels
pixel    ppixM;                                                            //  pxbM pixels

char     clfile[maxfcc] = "";                                              //  command line image file
char     imagedirk[maxfcc] = "";                                           //  image file directory
char     *file1 = 0, *file2 = 0;                                           //  image1 and image2 pathnames
char     fname1[100] = "", fname3[100] = "";                               //  image1 and image3 file names
int      f1size, f3size;                                                   //  image1 and image3 file sizes

int      debug = 0;                                                        //  set by command line -d parameter
int      thread_busy = 0;                                                  //  thread function active
int      kill_thread = 0;                                                  //  thread kill flag
int      computing = 0;                                                    //  computation ongoing
int      fdestroy = 0;                                                     //  quit / destroy flag
int      fullSize, alignSize;                                              //  full and align image sizes
int      modified = 0;                                                     //  image3 modified (vs image1)
double   rotate_angle = 0;                                                 //  image3 net rotatation vs. image1

int      crop_stat = 0, cropw, croph;                                      //  image crop status
int      Nalign = 0;                                                       //  alignment progress counter
int      showRedpix = 0;                                                   //  flag, highlight alignment pixels
int      Fautolens = 0;                                                    //  flag, autolens function running
int      pxL, pxH, pyL, pyH;                                               //  image overlap rectangle
int      pixsamp;                                                          //  pixel sample size (HDR, pano)
int      dragx, dragy, dragw, dragh;                                       //  mouse drag rectangle

bitmap   *BMpixels = 0;                                                    //  flags edge pixels for image alignment

double   Bratios[3][256];                                                  //  brightness ratios /color /brightness
double   brightfix0[256], brightfix1[256], brightfix2[256];                //  calculated color adjustment factors 
double   lens_curve, lens_mm, lens_bow, lens_mmB, lens_bowB;               //  lens parameters: focal length, bow
double   xoff, yoff, toff, yst1, yst2;                                     //  align offsets: x/y/theta/y-stretch
double   xoffB, yoffB, toffB, yst1B, yst2B;                                //  align offsets: current best values
double   xystep, tstep, xylim, tlim, yststep, ystlim;                      //  align step size and search range
double   matchlev, matchB;                                                 //  image alignment match level
double   blend;                                                            //  image blend width (pano)

int      nsapg, maxsapg = 1000, sapgx[1000], sapgy[1000];                  //  select area polygon points
int      sablend;                                                          //  select area blend width
int      sastat;                                                           //  select area dialog status

struct   sapix1 { int16  px, py, dist; };                                  //  list of pixels in select area
sapix1  *sapix = 0;
int      Nsapix;

double   parm_pixel_sample_size;                                           //  adjustable parameters
double   parm_jpg_save_quality;
double   parm_pano_lens_mm;
double   parm_pano_lens_bow;
double   parm_pano_prealign_size;
double   parm_pano_mouse_leverage;
double   parm_pano_align_size_increase;
double   parm_pano_blend_reduction;
double   parm_pano_minimum_blend;
double   parm_pano_image_stretch;

int   gtkinitfunc(void *data);                                             //  GTK initz. function
void  mwpaint();                                                           //  window paint function - expose event
void  mwpaint2(int refresh);                                               //  cause window repaint
void  update_status_bar();                                                 //  update status bar

int   KBevent(GtkWidget *, GdkEventKey *, void *);                         //  KB event function
void  mouse_event(GtkWidget *, GdkEventButton *, void *);                  //  mouse event function
void  menufunc(GtkWidget *, const char *menu);                             //  menu function, main window

void  m_index();                                                           //  thumbnail index window  v.19
void  m_open(char *file);                                                  //  open new image file
void  m_prev();                                                            //  open previous file    v.13
void  m_next();                                                            //  open next file        v.13
void  m_undo();                                                            //  undo changes
void  m_redo();                                                            //  redo changes
void  m_magnify(int mag);                                                  //  magnify: 1x/2x/fit window       v.40
void  m_slideshow();                                                       //  slideshow mode (toggle)         v.41
void  m_kill();                                                            //  kill running thread
void  m_exif();                                                            //  show EXIF data   v.31
void  m_save();                                                            //  save modified image
void  m_trash();                                                           //  move image to trash    v.15
void  m_clone();                                                           //  start another fotox instance    v.39
void  m_quit();                                                            //  exit application

int   mod_keep();                                                          //  query to discard mods
void  destroy();                                                           //  main window destroy signal

void  m_parms();                                                           //  adjust parameters
void  m_montest();                                                         //  check monitor          v.40
void  m_help(const char *menu);                                            //  display help
void  m_thumbs();                                                          //  generate thumbnails    v.41

void  m_tune();                                                            //  adjust brightness/colors
void  m_crop();                                                            //  crop image
void  m_redeye();                                                          //  redeye removal
void  m_rotate(double angle);                                              //  rotate image
void  m_resize();                                                          //  resize image
void  m_HDR();                                                             //  make HDR image
void  m_pano();                                                            //  make panorama image
void  m_unbend();                                                          //  unbend panorama image        v.32
void  m_stretch();                                                         //  stretch/distort image area   v.38
void  m_sharpen();                                                         //  sharpen image                v.38
void  m_denoise();                                                         //  image noise reduction        v.38
void  m_colors();                                                          //  set image color depth        v.38

pixel  vpixel(pixel, double px, double py, int, int, int);                 //  get virtual pixel (px,py)
int    sigdiff(double d1, double d2, double signf);                        //  test for significant difference
void   get_overlap_region(GdkPixbuf *pxb1, GdkPixbuf *pxb2);               //  get image overlap rectangle
void   get_Bratios(GdkPixbuf *pxb1, GdkPixbuf *pxb2);                      //  get brightness ratios for overlap 
void   flag_edge_pixels(GdkPixbuf *pixbuf);                                //  flag high-contrast pixels in overlap 
double match_images(GdkPixbuf *pxb1, GdkPixbuf *pxb2);                     //  match images in overlap region
double match_pixels(pixel pix1, pixel pix2);                               //  match two pixels

void  push_image3();                                                       //  push image3 into undo stack
void  pull_image3();                                                       //  pull image3 from undo stack (undo)
void  prior_image3();                                                      //  get last image3, no stack removal
void  redo_image3();                                                       //  advance undo stack without push
void  clearmem_image3();                                                   //  clear undo stack

GdkPixbuf * load_pixbuf(const char *file);                                 //  load pixbuf from file, remove alpha 
void  incr_pixbufs();                                                      //  track pixbufs allocated
void  pixbuf_free(GdkPixbuf *&pixbuf);                                     //  track pixbufs freed

int   exif_show(char *file);                                               //  display EXIF data                v.31
int   exif_fetch(char *file);                                              //  extract EXIF data from image file
int   exif_stuff(char *file1, char *file2, int xx, int yy);                //  stuff EXIF data from file1 into file2

void * select_area_thread(void *);                                         //  select image area for editing

const char  *Bcancel;                                                      //  dialog box buttons and labels (xtext)
const char  *Bdone;
const char  *Bundo;
const char  *Bredo;
const char  *Bapply;
const char  *Breset;
const char  *Bselarea;
const char  *Bseldone;
const char  *Bselblend;
const char  *Bcrop;
const char  *Bproceed;
const char  *Bsearch;
const char  *Bwidth;
const char  *Bheight;
const char  *Bpercent;
const char  *Bpreset;
const char  *Bred;
const char  *Bgreen;
const char  *Bblue;
const char  *Bbrightness;
const char  *Bwhiteness;
const char  *Bblendwidth;

/**************************************************************************/

//  main program

int main(int argc, char *argv[])
{
   GtkWidget   *vbox;
   GtkWidget   *mEdit, *mAdmin, *mHelp;
   int8        dashes[2] = { 2, 2 };
   int         ii;

   gtk_init(&argc,&argv);                                                  //  GTK command line options
   if (! g_thread_supported())                                             //  suddenly required for new gtk   v.30
         g_thread_init(null);                                              //  initz. GTK for threads
   gdk_threads_init();
   zlockInit();

   initz_appfiles("fotox","parameters",null);                              //  get app directories, parameter file

   for (ii = 1; ii < argc; ii++)                                           //  fotox command line options   v.16
   {
      if (strEqu(argv[ii],"-d")) debug = 1;                                //  -d (debug flag)
      else if (strEqu(argv[ii],"-i") && argc > ii+1)                       //  -i imageDirectory
            strcpy(imagedirk,argv[++ii]);
      else if (strEqu(argv[ii],"-f") && argc > ii+1)                       //  -f imageFile     v.17
            strcpy(clfile,argv[++ii]);
      else strcpy(clfile,argv[ii]);                                        //  imageFile        v.18
   }

   mwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);                             //  create main window
   gtk_window_set_title(GTK_WINDOW(mwin),fversion);
   gtk_window_set_position(GTK_WINDOW(mwin),GTK_WIN_POS_CENTER);
   gtk_window_set_default_size(GTK_WINDOW(mwin),mwinW,mwinH);

   vbox = gtk_vbox_new(0,0);                                               //  add vert. packing box
   gtk_container_add(GTK_CONTAINER(mwin),vbox);

   mmbar = create_menubar(vbox);                                           //  add menu bar and menu items

   mAdmin = add_menubar_item(mmbar,xtext("Admin"),menufunc);
      add_submenu_item(mAdmin,xtext("edit parameters"),menufunc);
      add_submenu_item(mAdmin,xtext("check monitor"),menufunc);
      add_submenu_item(mAdmin,xtext("create thumbnails"),menufunc);

   mHelp = add_menubar_item(mmbar,xtext("Help"),menufunc);
      add_submenu_item(mHelp,xtext("about fotox"),menufunc);
      add_submenu_item(mHelp,xtext("fotox user guide"),menufunc);
      add_submenu_item(mHelp,xtext("README file"),menufunc);
      add_submenu_item(mHelp,xtext("change log"),menufunc);

   mEdit = add_menubar_item(mmbar,xtext("Edit Image"),menufunc);
      add_submenu_item(mEdit,xtext("tune"),menufunc);
      add_submenu_item(mEdit,xtext("crop"),menufunc);
      add_submenu_item(mEdit,xtext("red eye"),menufunc);
      add_submenu_item(mEdit,xtext("rotate"),menufunc);
      add_submenu_item(mEdit,xtext("resize"),menufunc);
      add_submenu_item(mEdit,xtext("HDR"),menufunc);
      add_submenu_item(mEdit,xtext("panorama"),menufunc);
      add_submenu_item(mEdit,xtext("unbend"),menufunc);
      add_submenu_item(mEdit,xtext("stretch"),menufunc);
      add_submenu_item(mEdit,xtext("sharpen"),menufunc);
      add_submenu_item(mEdit,xtext("reduce noise"),menufunc);
      add_submenu_item(mEdit,xtext("color depth"),menufunc);

   mtbar = create_toolbar(vbox,40);                                        //  add toolbar and buttons  
   
   add_toolbar_button(mtbar,xtext("index"),xtext("thumbnail index"),"index.png",menufunc);
   add_toolbar_button(mtbar,xtext("open"),xtext("open new image file"),"open.png",menufunc);
   add_toolbar_button(mtbar,xtext("prev"),xtext("open previous image file"),"prev.png",menufunc);
   add_toolbar_button(mtbar,xtext("next"),xtext("open next image file"),"next.png",menufunc);
   add_toolbar_button(mtbar,xtext("undo"),xtext("undo image changes"),"undo.png",menufunc);
   add_toolbar_button(mtbar,xtext("redo"),xtext("redo image changes"),"redo.png",menufunc);
   add_toolbar_button(mtbar,xtext("1x/2x"),xtext("toggle size: 1x/2x"),"magnify.png",menufunc);
   add_toolbar_button(mtbar,xtext("fit"),xtext("fit image to window"),"fit-win.png",menufunc);
   add_toolbar_button(mtbar,xtext("kill"),xtext("kill running function"),"kill.png",menufunc);
   add_toolbar_button(mtbar,xtext("EXIF"),xtext("show image EXIF data"),"exif.png",menufunc);
   add_toolbar_button(mtbar,xtext("save"),xtext("save image to a file"),"save.png",menufunc);
   add_toolbar_button(mtbar,xtext("trash"),xtext("move image file to trash"),"trash.png",menufunc);
   add_toolbar_button(mtbar,xtext("clone"),xtext("start parallel fotox instance"),"clone.png",menufunc);
   add_toolbar_button(mtbar,xtext("quit"),xtext("quit fotox"),"quit.png",menufunc);

   drmwin = gtk_drawing_area_new();                                        //  add drawing window
   gtk_box_pack_start(GTK_BOX(vbox),drmwin,1,1,0);

   stbar = create_stbar(vbox);                                             //  add status bar

   G_SIGNAL(mwin,"destroy",destroy,0)                                      //  connect signals to window
   G_SIGNAL(drmwin,"expose-event",mwpaint,0)
   G_SIGNAL(mwin,"key-release-event",KBevent,0)

   Bcancel = xtext("cancel");                                              //  dialog box buttons and labels
   Bdone = xtext("done");
   Bundo = xtext("undo");
   Bredo = xtext("redo");
   Bapply = xtext("apply");
   Breset = xtext("reset");
   Bselarea = xtext("select area");
   Bseldone = xtext("select done");
   Bselblend = xtext("blend width");
   Bcrop = xtext("crop");
   Bproceed = xtext("proceed");
   Bsearch = xtext("search");
   Bwidth = xtext("width");
   Bheight = xtext("height");
   Bpercent = xtext("percent");
   Bpreset = xtext("presets");
   Bred = xtext("red");
   Bgreen = xtext("green");
   Bblue = xtext("blue");
   Bbrightness = xtext("brightness");
   Bwhiteness = xtext("whiteness");
   Bblendwidth = xtext("blend width");

   gtk_widget_add_events(drmwin,GDK_BUTTON_PRESS_MASK);                    //  connect mouse events
   gtk_widget_add_events(drmwin,GDK_BUTTON_RELEASE_MASK);
   gtk_widget_add_events(drmwin,GDK_BUTTON_MOTION_MASK);
   gtk_widget_add_events(drmwin,GDK_POINTER_MOTION_MASK);
   G_SIGNAL(drmwin,"button-press-event",mouse_event,0)
   G_SIGNAL(drmwin,"button-release-event",mouse_event,0)
   G_SIGNAL(drmwin,"motion-notify-event",mouse_event,0)
   
   gtk_widget_show_all(mwin);                                              //  show all widgets

   gdkgc = gdk_gc_new(drmwin->window);                                     //  initz. graphics context

   black.red = black.green = black.blue = 0;                               //  set up colors black, white
   white.red = white.green = white.blue = maxcolor;
   colormap = gtk_widget_get_colormap(drmwin);
   gdk_rgb_find_color(colormap,&black);
   gdk_rgb_find_color(colormap,&white);

   gdk_gc_set_foreground(gdkgc,&black);                                    //  set up to draw dotted lines
   gdk_gc_set_background(gdkgc,&white);
   gdk_gc_set_dashes(gdkgc,0,(gint8 *) dashes,2);
   gdk_gc_set_line_attributes(gdkgc,1,dottedline);

   gtk_init_add((GtkFunction) gtkinitfunc,0);                              //  setup initz. call from gtk_main()

   gdk_threads_enter();
   gtk_main();                                                             //  process window events
   gdk_threads_leave();

   return 0;
}


//  initial function called from gtk_main() at startup

int gtkinitfunc(void * data)
{
   char     *pp;
   int      qual;

   initParmlist(20);
                                                                           //  set default parameters
   setparm(pixel_sample_size,5)
   setparm(jpg_save_quality,80)                                            //  default JPG quality  v.28
   setparm(pano_lens_mm,40)                                                //  default 40mm lens
   setparm(pano_lens_bow,0)                                                //  default zero bow
   setparm(pano_prealign_size,500)
   setparm(pano_mouse_leverage,2)
   setparm(pano_align_size_increase,1.6)
   setparm(pano_blend_reduction,0.7)
   setparm(pano_minimum_blend,10)
   setparm(pano_image_stretch,0)                                           //  disable by default     v.23
   
   initz_userParms();                                                      //  load or initz. user parm file

   getparm(pixel_sample_size)                                              //  get final parm values
   getparm(jpg_save_quality)
   getparm(pano_lens_mm)
   getparm(pano_lens_bow)
   getparm(pano_prealign_size)
   getparm(pano_mouse_leverage)
   getparm(pano_align_size_increase)
   getparm(pano_blend_reduction)
   getparm(pano_minimum_blend)
   getparm(pano_image_stretch)

   qual = int(parm_jpg_save_quality);                                      //  set JPG file save quality    v.28
   if (qual < 0 || qual > 100) qual = 80;
   snprintf(JPGquality,4,"%d",qual);
   
   if (! *imagedirk) getcwd(imagedirk,maxfcc-1);                           //  if no image dirk, use current

   if (*clfile) {                                                          //  command line image file
      if (*clfile != '/') {
         pp = zmalloc(strlen(imagedirk)+strlen(clfile)+5);                 //  clfile is relative to imagedirk
         strcpy(pp,imagedirk);
         strcat(pp,"/");
         strcat(pp,clfile);
      }
      else  pp = strdupz(clfile);                                          //  use clfile

      m_open(pp);                                                          //  open command line file
   }
   
   return 0;
}


//  paint window when created, exposed, or resized       v.27 overhaul
//  uses pxb3 (current modified image3)
//  no zlock(), called only from main thread

void mwpaint()
{
   GdkPixbuf         *pxbT = 0;
   static int        oldW = 0, oldH = 0;
   double            scalew, scaleh;
   int               ww, hh;
   GdkRectangle      rect;
   
   if (fdestroy) return;                                                   //  shutdown underway
   
   if (! pxb3) {                                                           //  no image to show
      gdk_window_clear(drmwin->window);
      return;
   }

   mwinW = drmwin->allocation.width;                                       //  window size
   mwinH = drmwin->allocation.height;
   if (mwinW < 20 || mwinH < 20) return;                                   //  reject tiny window

   if (mwinW != oldW || mwinH != oldH) {
      Mrefresh++;                                                          //  if size changed, force refresh
      oldW = mwinW;
      oldH = mwinH;
   }

   if (Mrefresh)                                                           //  refresh pxbM from image3
   {
      pixbuf_free(pxbM);
   
      if (Mscale == 0)                                                     //  scale image3 to fit window
      {
         scalew = 1.0 * mwinW / ww3;
         scaleh = 1.0 * mwinH / hh3;
         if (scalew < scaleh) Rscale = scalew;
         else Rscale = scaleh;
         ww = int(Rscale * ww3);
         hh = int(Rscale * hh3);
         pxbM = pixbuf_scale_simple(pxb3,ww,hh,interp);
         pixbuf_test(pxbM);
         org3x = org3y = 0;                                                //  image3 origin = (0,0)
      }
      
      else                                                                 //  image3 scale = 1x or 2x     v.40
      {   
         Rscale = Mscale;
         ww = int(mwinW / Rscale);                                         //  image3 pixels fitting in window
         hh = int(mwinH / Rscale);
         org3x = mp3xcen - ww / 2;                                         //  move image3 origin so that
         org3y = mp3ycen - hh / 2;                                         //    x/y center at middle of window

         ww = int(mwinW / Rscale);                                         //  image3 pixels fitting in window
         hh = int(mwinH / Rscale);
         if (ww > ww3) ww = ww3;
         if (hh > hh3) hh = hh3;

         if (org3x + ww > ww3) org3x = ww3 - ww;                           //  move image3 origin to keep
         if (org3y + hh > hh3) org3y = hh3 - hh;                           //    window full, if needed
         if (org3x < 0) org3x = 0;
         if (org3y < 0) org3y = 0;

         pxbM = pixbuf_new(colorspace,0,8,ww,hh);                          //  get portion of image3
         pixbuf_test(pxbM);                                                //    fitting in window
         pixbuf_copy_area(pxb3,org3x,org3y,ww,hh,pxbM,0,0);

         ww = int(ww * Rscale);                                            //  scale to window
         hh = int(hh * Rscale);
         pxbT = pixbuf_scale_simple(pxbM,ww,hh,interp);
         pixbuf_test(pxbT);
         pixbuf_free(pxbM);
         pxbM = pxbT;
      }
   }

   rect.x = rect.y = 0;                                                    //  paint rectangle = all window
   rect.width = mwinW;
   rect.height = mwinH;

   pixbuf_poop(M);                                                         //  pxbM width, height
   posMx = (mwinW - wwM) / 2;                                              //  center pxbM in drawing window   v.27
   posMy = (mwinH - hhM) / 2;

   gdk_window_begin_paint_rect(drmwin->window,&rect);                      //  pxbM >> drawing window
   gdk_draw_pixbuf(drmwin->window,0,pxbM,0,0,posMx,posMy,-1,-1,nodither);
   gdk_window_end_paint(drmwin->window);

   Mrefresh = 0;                                                           //  reset flag
   Fredraw++;                                                              //  flag, thread may need to update
   update_status_bar();                                                    //  update status bar
   return;
}


//  invalidate window - cause window to get repainted immediately          //  v.22
//  called only from threads

void mwpaint2(int refresh)
{
   GdkRectangle   rect;

   if (fdestroy) return;                                                   //  shutdown underway
   if (refresh) Mrefresh++;                                                //  image3 has changed
   
   rect.x = rect.y = 0;
   rect.width = mwinW;
   rect.height = mwinH;
   
   zlock();
   gdk_window_invalidate_rect(drmwin->window,&rect,0);                     //  invalidate whole window
   gdk_window_process_updates(drmwin->window,0);                           //  trigger expose event, repaint
   zunlock();

   return;
}


//  update main window status bar

void update_status_bar()
{
   int         fposn, fcount;
   char        msg[200], msg1[50], msg2[12], msg3[20];
   char        msg4[90], msg5[30], msg6[20], msg7[30];
   const char  *fmt4 = "  aligning: %d %+.1f %+.1f %+.4f %s  match: %.4f";
   const char  *fmt5 = "  stretch: %+.1f %+.1f";
   const char  *fmt6 = "  crop: %dx%d";
   const char  *fmt7 = "  lens mm: %.1f  bow: %.2f";
   
   if (fdestroy) return;
   
   get_image_counts(fposn,fcount);
   
   *msg1 = *msg2 = *msg3 = *msg4 = *msg5 = *msg6 = *msg7 = 0;
   if (modified) strcpy(msg3,"(modified)");
   if (computing) strcpy(msg3,"(computing)");
   if (file1) snprintf(msg1,49,"%s  %dx%d  %.2fMB",msg3,ww3,hh3,f3size/mega);       //  v.41
   if (fposn) snprintf(msg2,11,"  %d/%d",fposn,fcount);
   if (yst1B || yst2B) snprintf(msg5,29,fmt5,yst1B,yst2B);
   if (Nalign) snprintf(msg4,89,fmt4,Nalign,xoffB,yoffB,toffB,msg5,matchB);
   if (crop_stat) snprintf(msg6,19,fmt6,cropw,croph);
   if (Fautolens) snprintf(msg7,29,fmt7,lens_mmB,lens_bowB);
   snprintf(msg,199,"%s%s%s%s%s",msg1,msg2,msg4,msg6,msg7);
   stbar_message(stbar,msg);
   return;
}


//  keyboard event function - some toolbar buttons have KB equivalents     v.17
//  keys are mapped in /usr/include/gtk-2.0/gdk/gdkkeysyms.h

int KBevent(GtkWidget *win, GdkEventKey *event, void *)
{
   KBkey = event->keyval;
   if (thread_busy) return 1;                                              //  let thread handle it

   if (KBkey == GDK_Left) m_prev();                                        //  arrow keys  >>  prev/next image
   if (KBkey == GDK_Right) m_next();
   if (KBkey == GDK_R) m_rotate(+90);                                      //  keys L, R  >>  rotate
   if (KBkey == GDK_r) m_rotate(+90);
   if (KBkey == GDK_L) m_rotate(-90);
   if (KBkey == GDK_l) m_rotate(-90);
   if (KBkey == GDK_Delete) m_trash();                                     //  delete  >>  trash
   if (KBkey == GDK_s) m_slideshow();                                      //  S  >>  slideshow mode in/out
   if (KBkey == GDK_S) m_slideshow();
   if (KBkey == GDK_Escape) m_slideshow();                                 //  escape  >>  slideshow mode out
   return 1;
}


//  mouse event function - capture buttons and drag movements

void mouse_event(GtkWidget *, GdkEventButton *event, void *)
{
   static int     bdtime = 0, butime = 0;
   static int     busy = 0, fmouse = 0, fdrag = 0;
   
   if (busy) return;                                                       //  stop redundant events ***   v.42
   busy++;
   
   if (event->type == GDK_BUTTON_PRESS) {
      bdtime = event->time;
      mpWx = int(event->x);                                                //  track mouse position
      mpWy = int(event->y);
      fmouse++;
      if (event->button == 1) LMdown = 1;                                  //  left mouse pressed
   }

   if (event->type == GDK_MOTION_NOTIFY) {
      mpWx = int(event->x);                                                //  mouse movement
      mpWy = int(event->y);
      fmouse++;
      if (LMdown) fdrag++;                                                 //  drag underway
   }
   
   if (fmouse) {                                                           //  new mouse position
      fmouse = 0;
      mpMx = mpWx - posMx;                                                 //  corresponding pxbM position
      mpMy = mpWy - posMy;
      if (mpMx < 0) mpMx = 0;                                              //  stay inside image
      if (mpMy < 0) mpMy = 0;
      if (mpMx > wwM-1) mpMx = wwM-1;
      if (mpMy > hhM-1) mpMy = hhM-1;
      mp3x = org3x + int(mpMx / Rscale + 0.5);                             //  corresponding pxb3 position
      mp3y = org3y + int(mpMy / Rscale + 0.5);
   }

   if (event->type == GDK_BUTTON_RELEASE) {
      butime = event->time;
      LMdown = 0;                                                          //  cancel left mouse down status   v.28
      if (fdrag) {
         fdrag = 0;                                                        //  if drag underway, end it
         mdx1 = mdy1 = mdx2 = mdy2 = 0;
      }
      else {
         if (butime - bdtime < 300) {                                      //  ignore > 300 ms
            if (event->button == 1) LMclick++;                             //  left mouse click
            if (event->button == 3) RMclick++;                             //  right mouse click
         }
      }
   }

   if (LMclick && ! LMcapture) {                                           //  if not captured by application,
      LMclick = RMclick = 0;                                               //  reset mouse click flags
      mp3xcen = mp3x;                                                      //    set new window center
      mp3ycen = mp3y;
      mwpaint2(1);                                                         //  refresh window
   }

   if (fdrag) {
      if (! mdx1) { mdx1 = mpMx; mdy1 = mpMy; }                            //  new drag start (pxbM relative)
      mdx2 = mpMx; mdy2 = mpMy;                                            //  capture drag extent
   }

   busy = 0;
   return;
}


//  process main window menu and toolbar events

void menufunc(GtkWidget *, const char *menu)
{
   if (strEqu(menu,xtext("kill"))) m_kill();                               //  kill running thread
   if (strEqu(menu,xtext("quit"))) m_quit();                               //  quit fotox
   
   if (strEqu(menu,xtext("1x/2x"))) {
      m_magnify(1);                                                        //  set image size to 1x/2x   v.40
      return;
   }

   if (strEqu(menu,xtext("fit"))) {
      m_magnify(0);                                                        //  fit image to window       v.40
      return;
   }

   if (thread_busy) {
      zmessageACK(xtext("prior function still running (wait or kill)"));
      return;
   }

   if (strEqu(menu,xtext("index"))) m_index();                             //  toolbar functions
   if (strEqu(menu,xtext("open"))) m_open(null);
   if (strEqu(menu,xtext("prev"))) m_prev();
   if (strEqu(menu,xtext("next"))) m_next();
   if (strEqu(menu,xtext("undo"))) m_undo();
   if (strEqu(menu,xtext("redo"))) m_redo();
   if (strEqu(menu,xtext("EXIF"))) m_exif();
   if (strEqu(menu,xtext("save"))) m_save();
   if (strEqu(menu,xtext("trash"))) m_trash();
   if (strEqu(menu,xtext("clone"))) m_clone();

   if (strEqu(menu,xtext("edit parameters"))) m_parms();                   //  non-editing menu functions
   if (strEqu(menu,xtext("check monitor"))) m_montest();
   if (strEqu(menu,xtext("about fotox"))) m_help("about");
   if (strEqu(menu,xtext("fotox user guide"))) m_help("user guide");
   if (strEqu(menu,xtext("README file"))) m_help("README");
   if (strEqu(menu,xtext("change log"))) m_help("changelog");
   if (strEqu(menu,xtext("create thumbnails"))) m_thumbs();

   if (strEqu(menu,xtext("tune"))) m_tune();                               //  image editing menu functions
   if (strEqu(menu,xtext("crop"))) m_crop();
   if (strEqu(menu,xtext("red eye"))) m_redeye();
   if (strEqu(menu,xtext("rotate"))) m_rotate(0);
   if (strEqu(menu,xtext("resize"))) m_resize();
   if (strEqu(menu,xtext("HDR"))) m_HDR();
   if (strEqu(menu,xtext("panorama"))) m_pano();
   if (strEqu(menu,xtext("unbend"))) m_unbend();
   if (strEqu(menu,xtext("stretch"))) m_stretch();
   if (strEqu(menu,xtext("sharpen"))) m_sharpen();
   if (strEqu(menu,xtext("reduce noise"))) m_denoise();
   if (strEqu(menu,xtext("color depth"))) m_colors();
   
   return;
}


//  display a thumbnail index of images in a separate window               v.19

void m_index()
{
   void xfunction(char *file);

   if (! file1 && ! *imagedirk) {                                          //  if nothing available,    v.34
      m_open(0);                                                           //  user must open a file
      if (! file1) return;
   }
   
   if (file1) thumbnail_index(file1,"init",xfunction);                     //  file1 is anchor file
   else thumbnail_index(imagedirk,"init",xfunction);                       //  1st in directory is anchor file
   return;
}

void xfunction(char *filez)                                                //  receive selected file
{
   if (filez) m_open(filez);
   return;
}


//  open a new image file - make images pxb1 (keep) and pxb3 (modified)

void m_open(char *filez)
{
   char           *newfile, *pp, wtitle[100];
   struct stat    f1stat;
   const char     *openmess = xtext("open new image file");
   
   if (mod_keep()) return;                                                 //  keep modifications
   
   if (filez) newfile = filez;                                             //  use passed file
   else {
      if (file1) newfile = zgetfile(openmess,file1,"open");                //  assume same directory as prior 
      else newfile = zgetfile(openmess,imagedirk,"open");                  //  or curr. image directory   v.34
   }
   if (! newfile) return;                                                  //  user cancel

   if (file1) zfree(file1);                                                //  set new file1
   file1 = newfile;
   
   pixbuf_free(pxb1);
   pxb1 = load_pixbuf(file1);                                              //  validate file, load image pxb1  v.33
   if (! pxb1) {
      zmessageACK(xtext("file not fotox compatible: \n %s"),file1);        //  clean missing/non-working file  v.27
      track_image_files(file1,"remove",0);                                 //  remove from image file list
      thumbnail_index(file1,"update");                                     //  update index window (if there)
      zfree(file1);
      file1 = 0;                                                           //  v.34  bugfix
      return;
   }

   pixbuf_poop(1)                                                          //  get pxb1 attributes

   pixbuf_free(pxb3);
   pxb3 = pixbuf_copy(pxb1);                                               //  copy to image3 for editing
   pixbuf_test(pxb3);
   pixbuf_poop(3);
   
   stat(file1,&f1stat);                                                    //  get file size
   f1size = f3size = f1stat.st_size;

   if (! samedirk(file1,imagedirk)) {                                      //  if directory changed,
      track_image_files(file1,"init",0);                                   //    initz. image file list
      thumbnail_index(file1,"update");                                     //      and thumbnail window
   }

   strcpy(imagedirk,file1);                                                //  set new image dirk and file name
   pp = strrchr(imagedirk,'/');
   if (! pp) zappcrash(xtext("invalid directory: %s"),file1);
   strncpy0(fname1,pp+1,99);
   strncpy0(fname3,pp+1,99);
   pp[1] = 0;

   snprintf(wtitle,99,"%s (%s)",fname1,imagedirk);                         //  window title
   if (strlen(wtitle) > 97) strcpy(wtitle+96,"...");                       //  filename (/directory...)
   gtk_window_set_title(GTK_WINDOW(mwin),wtitle);

   clearmem_image3();                                                      //  clear undo stack
   modified = 0;                                                           //  image3 not modified
   rotate_angle = 0;                                                       //  no net rotation
   Mscale = 0;                                                             //  scale to window
   mp3xcen = ww3 / 2;                                                      //  set initial window center    v.40
   mp3ycen = hh3 / 2;                                                      //    for 1x/2x resizing actions
   mwpaint2(1);                                                            //  repaint window
   gtk_window_present(GTK_WINDOW(mwin));                                   //  bring to top

   return;
}


//  opem previous image file in same directory as last file opened         v.13

void m_prev()
{
   char        *fileP;
   
   if (! file1) return;
   fileP = track_image_files(file1,"prev",1);
   if (fileP) m_open(fileP);
   return;
}


//  opem next image file in same directory as last file opened             v.13

void m_next()
{
   char        *fileN;
   
   if (! file1) return;
   fileN = track_image_files(file1,"next",1);
   if (fileN) m_open(fileN);
   return;
}


//  undo last change - restore image3 from undo stack

void m_undo()
{
   pull_image3();                                                          //  restore previous image3
   return;
}


//  redo last undo - advance undo stack if another image is available      v.24

void m_redo()
{
   redo_image3();                                                          //  restore undone image3
   return;
}


//  magnify image to 100% or 200% or fit to window size                    //  v.40

void m_magnify(int mag)                                                    //  buttons instead of mouse clicks
{   

   if (! pxb3) return;

   if (mag == 0) Mscale = 0;                                               //  scale to window

   if (mag == 1) {
      if (Mscale == 0) Mscale = 1;                                         //  initially 1x
      else if (Mscale == 1) Mscale = 2;                                    //  then toggle 1x/2x
      else if (Mscale == 2) Mscale = 1;
   }
   
   mwpaint2(1);                                                            //  repaint main window
   return;
}


//  enter or leave slideshow mode

void m_slideshow()                                                         //  v.41
{
   int            key;
   static int     ww, hh;
   
   key = KBkey;
   KBkey = 0;

   if (! Fslideshow) 
   {
      if (key == GDK_Escape) return;
      gtk_window_get_size(GTK_WINDOW(mwin),&ww,&hh);
      gtk_widget_hide_all(GTK_WIDGET(mmbar));                              //  enter slide show mode
      gtk_widget_hide_all(GTK_WIDGET(mtbar));
      gtk_widget_hide_all(GTK_WIDGET(stbar));
      gtk_window_fullscreen(GTK_WINDOW(mwin));
   }

   if (Fslideshow) 
   {
      gtk_window_unfullscreen(GTK_WINDOW(mwin));                           //  leave slide show mode
      gtk_window_resize(GTK_WINDOW(mwin),ww,hh);
      gtk_widget_show_all(GTK_WIDGET(mmbar));
      gtk_widget_show_all(GTK_WIDGET(mtbar));
      gtk_widget_show_all(GTK_WIDGET(stbar));
   }

   Fslideshow = 1 - Fslideshow;                                            //  toggle mode flag
   return;
}


//  show image EXIF data if available

void m_exif()
{
   if (file1) exif_show(file1);
   return;
}


//  save pxb3 (modified) image to a file

void m_save()
{
   char           *newfile, *pp;
   int            exiferr;
   struct stat    f3stat;
   const char     *title = xtext("save image to a file");
   
   if (! pxb3) return;
   
   newfile = zgetfile(title,file1,"save","quality");                       //  get new file name from user   v.17
   if (! newfile) return;
   
   exiferr = exif_fetch(file1);                                            //  original file EXIF data    v.31

   gdk_pixbuf_save(pxb3,newfile,"jpeg",gerror,"quality",JPGquality,null);  //  save image3 to new or same file

   if (! exiferr) exif_stuff(file1,newfile,ww3,hh3);                       //  EXIF >> new image file with new x/y

   if (strNeq(newfile,file1)) {                                            //  if new file in current directory,
      if (samedirk(file1,newfile)) {                                       //    add to image file index
         track_image_files(newfile,"add",0);
         thumbnail_index(0,"update");
      }
   }

   pp = strrchr(newfile,'/');                                              //  new image3 file name         v.24
   if (! pp) pp = newfile;
   else pp++;
   strncpy0(fname3,pp,99);
   stat(newfile,&f3stat);                                                  //  new size
   f3size = f3stat.st_size;
   update_status_bar();
   
   zfree(newfile);
   modified = 0;
   rotate_angle = 0;
   return;
}


//  delete image file - move file1 to trash                                v.15
//  (not reliable since ~/.Trash is no standard)
//  (g_file_trash() does not work any better)

void m_trash()
{
   int            yn, err;
   char           command[1000];

   if (! file1) return;                                                    //  nothing to trash
   
   yn = zmessageYN(xtext("move image file %s to trash?"),fname1);
   if (! yn) return;

   snprintf(command,999,"mv \"%s\" ~/.Trash/",file1);                      //  move image file to trash directory
   err = system(command);
   if (err) {
      zmessageACK("error: %s",syserrText(err));
      return;
   }
   
   track_image_files(file1,"remove",0);                                    //  remove from image file list
   m_next();                                                               //  step to next file if there
   thumbnail_index(file1,"update");                                        //  update index window if there
   return;
}


//  start a new instance of fotox in parallel                              //  v.39

void m_clone()
{
   char     command[200];
   
   if (is_blank_null(file1)) strcpy(command,"fotox &");
   else snprintf(command,199,"fotox \"%s\" &",file1);                      //  pass current file
   system(command);
   return;
}


//  kill running thread function

void m_kill()
{
   kill_thread++;
   while (thread_busy) zmainloop();
   kill_thread = 0;
   return;
}


//  quit - exit program

void m_quit()
{
   m_kill();
   if (mod_keep()) return;
   fdestroy++;
   gtk_main_quit();   
   return;
}


//  query if modifications should be kept or discarded
//  return:  1 to keep mods,  0 to discard mods

int mod_keep()
{
   if (! modified) return 0;
   if (zmessageYN(xtext("discard modifications?"))) return 0;
   return 1;
}


//  main window destroy signal    v.15b

void destroy()
{
   fdestroy++;
   printf("main window destroyed \n");
   m_kill();
   if (mod_keep()) m_save();
   gtk_main_quit();   
   return;
}


//  edit parameters and get their new values        v.09

void m_parms()
{
   int np = editParms(0,0);                                                //  edit parms
   if (! np) return;

   getparm(pixel_sample_size)                                              //  set new values
   getparm(jpg_save_quality)
   getparm(pano_lens_mm)
   getparm(pano_lens_bow)
   getparm(pano_prealign_size)
   getparm(pano_mouse_leverage)
   getparm(pano_align_size_increase)
   getparm(pano_blend_reduction)
   getparm(pano_minimum_blend)
   getparm(pano_image_stretch)                                             //  v.23
}


//  monitor test function

void m_montest()                                                           //  v.40
{
   pixel       pix3;
   int         red, green, blue;
   int         row, col, row1, row2;
   int         ww = 768, hh = 400;
   
   if (mod_keep()) return;
   clearmem_image3();                                                      //  clear undo stack

   pixbuf_free(pxb1);
   pixbuf_free(pxb3);

   pxb3 = pixbuf_new(colorspace,0,8,ww,hh);
   pixbuf_test(pxb3);
   pixbuf_poop(3);
   nch = 3;

   gtk_window_set_title(GTK_WINDOW(mwin),"monitor check");

   for (red = 0; red <= 1; red++)
   for (green = 0; green <= 1; green++)
   for (blue = 0; blue <= 1; blue++)
   {
      row1 = 4 * red + 2 * green + blue;
      row1 = row1 * hh / 8;
      row2 = row1 + hh / 8;
      
      for (row = row1; row < row2; row++)
      for (col = 0; col < ww; col++)
      {
         pix3 = ppix3 + row * rs3 + col * nch;
         pix3[0] = red * 256 * col / ww;
         pix3[1] = green * 256 * col / ww;
         pix3[2] = blue * 256 * col / ww;
      }
   }

   modified = 0;                                                           //  image3 not modified
   Mscale = 0;                                                             //  scale to window
   mp3xcen = ww3 / 2;                                                      //  set initial window center
   mp3ycen = hh3 / 2;
   mwpaint2(1);                                                            //  repaint window
   
   return;
}


//  thread function to display help file

void m_help(const char *menu)                                              //  menu function
{
   if (strEqu(menu,"about")) 
      zmessageACK(" %s \n %s \n %s",fversion,flicense,fsource);

   if (strEqu(menu,"user guide")) 
      showz_helpfile();                                                    //  launch help file in new process  

   if (strEqu(menu,"README"))                                              //  v.41
      showz_readme();

   if (strEqu(menu,"changelog"))                                           //  v.41
      showz_changelog();

   return;
}


//  create thumbnail images for all image files in current directory

void m_thumbs()
{
   GdkPixbuf   *pixbuf;
   char        *filespec1, *filespec2;
   char        *thumbdir, *pp;
   int         err;

   if (! *imagedirk) m_open(null);
   if (! *imagedirk) return;

   filespec1 = track_image_files(imagedirk,"first",0);
   if (! filespec1) return;

   thumbdir = strdupz(imagedirk,20);
   strcat(thumbdir,"/.thumbnails");
   err = mkdir(thumbdir,0751);
   if (err && errno != EEXIST) {
      zmessageACK("cannot create .thumbnails directory: \n %s", syserrText(errno));
      return;
   }
   
   while (filespec1)
   {
      pixbuf = image_xthumb(filespec1);
      if (pixbuf) g_object_unref(pixbuf);
      pp = strrchr(filespec1,'/');
      stbar_message(stbar,pp+1);
      filespec2 = track_image_files(filespec1,"next",1);
      zfree(filespec1);
      filespec1 = filespec2;
   }
   
   return;
}


/**************************************************************************/

//  adjust image brightness distribution and color 

int    tune_dialog_event(zdialog* zd, const char *event);
int    tune_dialog_compl(zdialog* zd, int zstat);
void * tune_exec_thread(void *);                                           //  tune whole image or selected area
void   tune_exec_1(pixel pix1, pixel pix2, int dist);                      //  tune one pixel in image

double   tuneBR[9], tuneWH[9], tuneRGB[3];                                 //  tune dialog widget values
int      tune_update;                                                      //  tune widget update counter


void m_tune()
{
   int         ii, zstat;
   zdialog     *zd;
   const char  *title = xtext("adjust brightness / whiteness / color");

   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(pxb1);
   pixbuf_poop(1)

   for (ii = 0; ii < 9; ii++) tuneBR[ii] = tuneWH[ii] = 0;                 //  initialize
   tuneRGB[0] = tuneRGB[1] = tuneRGB[2] = 0;
   tune_update = 0;
   sablend = nsapg = sastat = Nsapix = 0;                                  //  no select area

   zd = zdialog_new(title,Bdone,Bcancel,null);                             //  create tune dialog

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","space1","hb1",0,"space=70");
   zdialog_add_widget(zd,"label","labdark","hb1",xtext("darker areas"));
   zdialog_add_widget(zd,"label","expand1","hb1",0,"expand");
   zdialog_add_widget(zd,"label","labbright","hb1",xtext("brighter areas"));
   
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"expand|space=8");        //  brightness buttons
   zdialog_add_widget(zd,"vbox","vb2","hb2",0,"space=8");
   zdialog_add_widget(zd,"label","bright","vb2",Bbrightness);
   zdialog_add_widget(zd,"hbox","hb21","vb2");
   zdialog_add_widget(zd,"vbox","vb22","hb21",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb23","hb21",0,"space=5");
   zdialog_add_widget(zd,"button","br +++","vb22","+++");
   zdialog_add_widget(zd,"button","br ---","vb23"," - - - ");
   zdialog_add_widget(zd,"button","br +-", "vb22"," +  - ");
   zdialog_add_widget(zd,"button","br -+", "vb23"," -  + ");
   zdialog_add_widget(zd,"button","br +-+","vb22"," + - + ");
   zdialog_add_widget(zd,"button","br -+-","vb23"," - + - ");

   zdialog_add_widget(zd,"vscale","brval0","hb2","-10|10|1|0","expand");   //  brightness sliders
   zdialog_add_widget(zd,"vscale","brval1","hb2","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","brval2","hb2","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","brval3","hb2","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","brval4","hb2","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","brval5","hb2","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","brval6","hb2","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","brval7","hb2","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","brval8","hb2","-10|10|1|0","expand");

   zdialog_add_widget(zd,"hbox","hb3","dialog",0,"expand|space=8");        //  whiteness buttons
   zdialog_add_widget(zd,"vbox","vb3","hb3",0,"space=8");
   zdialog_add_widget(zd,"label","white","vb3",Bwhiteness);
   zdialog_add_widget(zd,"hbox","hb31","vb3");
   zdialog_add_widget(zd,"vbox","vb32","hb31",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb33","hb31",0,"space=5");
   zdialog_add_widget(zd,"button","wh +++","vb32","+++");
   zdialog_add_widget(zd,"button","wh ---","vb33"," - - - ");
   zdialog_add_widget(zd,"button","wh +-", "vb32"," +  - ");
   zdialog_add_widget(zd,"button","wh -+", "vb33"," -  + ");
   zdialog_add_widget(zd,"button","wh +-+","vb32"," + - + ");
   zdialog_add_widget(zd,"button","wh -+-","vb33"," - + - ");

   zdialog_add_widget(zd,"vscale","whval0","hb3","-10|10|1|0","expand");   //  whiteness sliders
   zdialog_add_widget(zd,"vscale","whval1","hb3","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","whval2","hb3","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","whval3","hb3","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","whval4","hb3","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","whval5","hb3","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","whval6","hb3","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","whval7","hb3","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","whval8","hb3","-10|10|1|0","expand");

   zdialog_add_widget(zd,"hbox","hb4","dialog",0,"space=10");              //  RGB spin buttons    v.15
   zdialog_add_widget(zd,"label","labred","hb4",xtext("  adjust red"));
   zdialog_add_widget(zd,"spin","spred","hb4","-10|10|0.1|0");
   zdialog_add_widget(zd,"label","labgreen","hb4",xtext("  green"));
   zdialog_add_widget(zd,"spin","spgreen","hb4","-10|10|0.1|0");
   zdialog_add_widget(zd,"label","labblue","hb4",xtext("  blue"));
   zdialog_add_widget(zd,"spin","spblue","hb4","-10|10|0.1|0");

   zdialog_add_widget(zd,"hbox","hb5","dialog",0,"space=10");              //  select area button     v.36
   zdialog_add_widget(zd,"label","space5","hb5",0,"space=10");
   zdialog_add_widget(zd,"button","selarea","hb5",Bselarea);               //  [select area] [select done]
   zdialog_add_widget(zd,"button","seldone","hb5",Bseldone);
   zdialog_add_widget(zd,"label","labblend","hb5",Bselblend);              //  blend width [< >]
   zdialog_add_widget(zd,"spin","sablend","hb5","0|200|1|0");

   zdialog_add_widget(zd,"hbox","hb6","dialog",0,"space=10");              //  undo, redo, reset buttons
   zdialog_add_widget(zd,"label","space6","hb6",0,"space=10");
   zdialog_add_widget(zd,"button","undo","hb6",Bundo);
   zdialog_add_widget(zd,"button","redo","hb6",Bredo);
   zdialog_add_widget(zd,"button","reset","hb6",Breset);
   
   zdialog_resize(zd,0,400);
   zstat = zdialog_run(zd,tune_dialog_event,tune_dialog_compl);            //  run dialog - parallel

   return;
}


int tune_dialog_event(zdialog *zd, const char *event)                      //  tune dialog event function
{
   int         ii;
   char        vsname[20];
   double      wvalue, brinc[9], whinc[9], pi = 3.14159;

   if (strEqu(event,"reset")) {                                            //  reset button
      for (ii = 0; ii < 9; ii++) 
      {
         strcpy(vsname,"brval0");                                          //  set all sliders back home
         vsname[5] = '0' + ii;
         zdialog_stuff(zd,vsname,0);
         strcpy(vsname,"whval0");
         vsname[5] = '0' + ii;
         zdialog_stuff(zd,vsname,0);
      }

      zdialog_stuff(zd,"spred",0);                                         //  reset RGB spin buttons
      zdialog_stuff(zd,"spgreen",0);
      zdialog_stuff(zd,"spblue",0);

      zdialog_stuff(zd,"sablend",0);                                       //  reset select area
      sastat = nsapg = sablend = 0;
   }

   if (strEqu(event,"undo") || strEqu(event,"reset")) {                    //  undo or reset button
      for (ii = 0; ii < 9; ii++) 
         tuneBR[ii] = tuneWH[ii] = 0;                                      //  reset all adjustments
      tuneRGB[0] = tuneRGB[1] = tuneRGB[2] = 0;

      if (modified) prior_image3();                                        //  restore prior image3
      mwpaint2(1);
      return 1;
   }

   for (ii = 0; ii < 9; ii++)                                              //  clear all increments
      brinc[ii] = whinc[ii] = 0;

   if (strEqu(event,"br +++"))                                             //  convenience buttons "+++" etc.
      for (ii = 0; ii < 9; ii++)                                           //  get increments for all sliders
         brinc[ii] = 0.5;
   if (strEqu(event,"br ---")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = -0.5;
   if (strEqu(event,"br +-")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = 0.5 * cos(pi*ii/8.0);
   if (strEqu(event,"br -+")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = -0.5 * cos(pi*ii/8.0);
   if (strEqu(event,"br +-+")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = -0.5 * sin(pi*ii/8.0);
   if (strEqu(event,"br -+-")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = 0.5 * sin(pi*ii/8.0);

   if (strEqu(event,"wh +++")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = 0.5;
   if (strEqu(event,"wh ---")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = -0.5;
   if (strEqu(event,"wh +-")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = 0.5 * cos(pi*ii/8.0);
   if (strEqu(event,"wh -+")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = -0.5 * cos(pi*ii/8.0);
   if (strEqu(event,"wh +-+")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = -0.5 * sin(pi*ii/8.0);
   if (strEqu(event,"wh -+-")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = 0.5 * sin(pi*ii/8.0);

   for (ii = 0; ii < 9; ii++)                                              //  add increments to sliders    v.14
   {
      if (brinc[ii]) {
         strcpy(vsname,"brval0");
         vsname[5] = '0' + ii;
         zdialog_fetch(zd,vsname,wvalue);
         wvalue -= brinc[ii];
         if (wvalue < -10) wvalue = -10;
         if (wvalue > 10) wvalue = 10;
         zdialog_stuff(zd,vsname,wvalue);
      }

      if (whinc[ii]) {
         strcpy(vsname,"whval0");
         vsname[5] = '0' + ii;
         zdialog_fetch(zd,vsname,wvalue);
         wvalue -= whinc[ii];
         if (wvalue < -10) wvalue = -10;
         if (wvalue > 10) wvalue = 10;
         zdialog_stuff(zd,vsname,wvalue);
      }
   }
   
   for (ii = 0; ii < 9; ii++)                                              //  get all slider settings
   {
      strcpy(vsname,"brval0");
      vsname[5] = '0' + ii;
      zdialog_fetch(zd,vsname,wvalue);
      if (wvalue < -10) wvalue = -10;
      if (wvalue > 10) wvalue = 10;
      tuneBR[ii] = -wvalue;                                                //  set bright values, -10 to +10

      strcpy(vsname,"whval0");
      vsname[5] = '0' + ii;
      zdialog_fetch(zd,vsname,wvalue);
      if (wvalue < -10) wvalue = -10;
      if (wvalue > 10) wvalue = 10;
      tuneWH[ii] = -wvalue;                                                //  set white values, -10 to +10
   }

   zdialog_fetch(zd,"spred",tuneRGB[0]);                                   //  red/green/blue, -10 to +10
   zdialog_fetch(zd,"spgreen",tuneRGB[1]);
   zdialog_fetch(zd,"spblue",tuneRGB[2]);
   zdialog_fetch(zd,"sablend",sablend);                                    //  select blend width

   if (strEqu(event,"selarea")) {                                          //  select area for tune      v.36
      if (sastat == 0) {
         sastat = 1;
         pthread_create(&tid,0,select_area_thread,null);                   //  start select area thread
      }
   }

   if (strEqu(event,"seldone"))                                            //  select area completed
      if (sastat == 1) sastat = 2;                                         //  tell thread to finish up

   tune_update++;                                                          //  signal update thread,
   if (thread_busy == 0)                                                   //    new settings available
      pthread_create(&tid,0,tune_exec_thread,null);                        //  restart thread if idle

   return 1;
}


int tune_dialog_compl(zdialog *zd, int zstat)                              //  tune dialog completion function
{
   zdialog_destroy(zd);                                                    //  kill dialog
   sastat = 0;                                                             //  insure thread is stopped
   if (zstat != 1) pull_image3();                                          //  cancel, undo changes

   if (sapix) zfree(sapix);                                                //  free memory for select area pixels
   sapix = 0;
   return 0;
}


void * tune_exec_thread(void *)                                            //  modify image pixels according to
{                                                                          //    brightness & whiteness setpoints
   int         ii, px, py, dist;
   pixel       pix1, pix3;
   
   ++thread_busy;
   zsleep(0.1);                                                            //  kludge, prevent gnome lockups   ****
   
   while (tune_update)                                                     //  image updates are asychronous
   {                                                                       //    to dialog widget updates
      tune_update = 0;

      if (nsapg == 0)                                                      //  process whole image
      {
         for (py = 0; py < hh1; py++)                                      //  loop all pixels
         for (px = 0; px < ww1; px++)
         {
            pix1 = ppix1 + py * rs1 + px * nch;                            //  input pixel
            pix3 = ppix3 + py * rs3 + px * nch;                            //  output pixel
            tune_exec_1(pix1,pix3,0);                                      //  modify
         }
      }
      
      if (nsapg > 0 && sastat == 3)                                        //  process area selection polygon
      {
         for (ii = 0; ii < Nsapix; ii++)                                   //  process all enclosed pixels
         {
            px = sapix[ii].px;
            py = sapix[ii].py;
            dist = sapix[ii].dist;
            pix1 = ppix1 + py * rs1 + px * nch;                            //  input pixel
            pix3 = ppix3 + py * rs3 + px * nch;                            //  output pixel
            tune_exec_1(pix1,pix3,dist);                                   //  modify
         }
      }

      mwpaint2(1);                                                         //  refresh main window
   }

   modified++;                                                             //  image3 modified
   --thread_busy;
   return 0;
}

void tune_exec_1(pixel pix1, pixel pix2, int dist)                         //  modify one pixel according to
{                                                                          //    brightness & whiteness setpoints
   int         ii;
   double      red, green, blue, white, maxbright = 255;
   double      brin, brin8, brvalx, whvalx, brout;
   double      f1, f2;
   
   red = pix1[0];                                                          //  input pixel
   green = pix1[1];
   blue = pix1[2];
   
   brin = red;                                                             //  get brightest color
   if (green > brin) brin = green;
   if (blue > brin) brin = blue;
   
   brin = brin / maxbright;                                                //  normalize 0..1

   brin8 = 8.0 * brin;                                                     //  brightness zone 0--8
   ii = int(brin8);

   if (ii < 8)                                                             //  interpolate brightness value
      brvalx = tuneBR[ii] + (tuneBR[ii+1] - tuneBR[ii]) * (brin8 - ii);
   else brvalx = tuneBR[8];
   brvalx = brvalx * 0.1;                                                  //  normalize -1..+1

   if (ii < 8)                                                             //  interpolate whiteness value
      whvalx = tuneWH[ii] + (tuneWH[ii+1] - tuneWH[ii]) * (brin8 - ii);
   else whvalx = tuneWH[8];
   whvalx = whvalx * 0.1;                                                  //  normalize -1..+1

   white = red;                                                            //  white content
   if (green < white) white = green;                                       //   = min(red,green,blue)
   if (blue < white) white = blue;
   white = white * whvalx;                                                 //  -100% to +100%
   red += white;
   green += white;
   blue += white;
   
   brout = brin + brvalx * sqrt(0.5 - fabs(brin - 0.5));                   //  convert input brightness
   if (brout > 0) brout = brout / brin;                                    //    to output brightness
   else brout = 0;
   if (brout * brin > 1.0) brout = 1.0 / brin;                             //  stop overflows
   red = red * brout + 0.49;                                               //  apply to all colors
   green = green * brout + 0.49;
   blue = blue * brout + 0.49;

   if (tuneRGB[0]) red += red * tuneRGB[0] / 20;                           //  get RGB adjustments          v.15
   if (tuneRGB[1]) green += green * tuneRGB[1] / 20;                       //  -10 to +10 >> -50% to +50%
   if (tuneRGB[2]) blue += blue * tuneRGB[2] / 20;

   if (red > 255) red = 255;                                               //  stop overflows
   if (green > 255) green = 255;
   if (blue > 255) blue = 255;
   
   if (nsapg > 0 && dist < sablend)                                        //  if area selection, blend pixel
   {                                                                       //    changes over distance sablend
      f1 = 1.0 * dist / sablend;
      f2 = 1.0 - f1;
      red = f1 * red + f2 * pix1[0];
      green = f1 * green + f2 * pix1[1];
      blue = f1 * blue + f2 * pix1[2];
   }

   pix2[0] = int(red);                                                     //  output pixel
   pix2[1] = int(green);
   pix2[2] = int(blue);

   return;
}


/**************************************************************************/

//  crop image - use mouse to select image region to retain                v.22  overhauled

void * crop_thread(void *);
int    crop_dialog_compl(zdialog *zd, int zstat);


void m_crop()
{
   if (! pxb3) return;                                                     //  nothing to crop
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(pxb1);
   pixbuf_poop(1)

   pthread_create(&tid,0,crop_thread,null);                                //  thread runs on from here
   return;
}


//  crop thread function

void * crop_thread(void *)   
{
   const char     *crop_message = xtext("click or drag crop margins");
   zdialog        *zd;
   GdkPixbuf      *pxbT = 0;
   GdkCursor      *drag_cursor = 0;
   int            crx1, cry1, crx2, cry2, crww, crhh;                      //  crop rectangle
   
   ++thread_busy;

   zd = zdialog_new(xtext("crop image"),Bcrop,Bcancel,null);               //  start modal dialog to 
   zdialog_add_widget(zd,"label","lab1","dialog",crop_message);            //  get user completion event
   zdialog_run(zd,0,crop_dialog_compl);

   zlock();
   drag_cursor = gdk_cursor_new(GDK_CROSSHAIR);                            //  set drag cursor
   gdk_window_set_cursor(drmwin->window,drag_cursor);
   zunlock();

   crx1 = int(0.2 * wwM);                                                  //  start with 20% crop margins
   cry1 = int(0.2 * hhM);
   crx2 = int(0.8 * wwM);
   cry2 = int(0.8 * hhM);
   crww = crx2 - crx1;
   crhh = cry2 - cry1;

   LMcapture = 1;                                                          //  capture left moust clicks
   crop_stat = 1;                                                          //  status = crop in progress
   Fredraw++;

   while (crop_stat == 1)                                                  //  loop while crop in progress
   {
      zsleep(0.1);                                                         //  mouse sample interval, 0.1 secs.
      
      if (LMclick || LMdown)                                               //  mouse click or drag
      {
         LMclick = 0;
         mwpaint2(0);                                                      //  refresh window (sets Fredraw when done)

         if (abs(mpMx - crx1) < abs(mpMx - crx2)) crx1 = mpMx;             //  new crop rectangle from mouse posn.
         else crx2 = mpMx;
         if (abs(mpMy - cry1) < abs(mpMy - cry2)) cry1 = mpMy;
         else cry2 = mpMy;

         if (crx1 > crx2-10) crx1 = crx2-10;                               //  sanity limits
         if (cry1 > cry2-10) cry1 = cry2-10;
         if (crx1 < 0) crx1 = 0;
         if (cry1 < 0) cry1 = 0;
         if (crx2 > wwM) crx2 = wwM;
         if (cry2 > hhM) cry2 = hhM;
      }

      if (Fredraw)                                                         //  redraw logic revised   v.42
      {
         Fredraw = 0;

         crww = crx2 - crx1;                                               //  draw current crop rectangle
         crhh = cry2 - cry1;
         zlock();
         gdk_draw_rectangle(drmwin->window,gdkgc,0,crx1+posMx,cry1+posMy,crww,crhh);
         zunlock();

         cropw = int(crww / Rscale);                                       //  status bar update   v.23
         croph = int(crhh / Rscale);
         update_status_bar();
      }
   }
   
   if (crop_stat != 2) {                                                   //  crop canceled
      pull_image3();
      goto thread_exit;
   }

   crx1 = int(crx1 / Rscale + org3x + 0.5);                                //  convert to image3 coordinates
   cry1 = int(cry1 / Rscale + org3y + 0.5);
   crx2 = int(crx2 / Rscale + org3x + 0.5);
   cry2 = int(cry2 / Rscale + org3y + 0.5);
   crww = crx2 - crx1;
   crhh = cry2 - cry1;
   if (crx1 + crww > ww3) crww = ww3 - crx1;
   if (cry1 + crhh > hh3) crhh = hh3 - cry1;

   zlock();
   pxbT = pixbuf_new(colorspace,0,8,crww,crhh);                            //  crop image3 in pxb3
   pixbuf_test(pxbT);
   pixbuf_copy_area(pxb3,crx1,cry1,crww,crhh,pxbT,0,0);
   pixbuf_free(pxb3);
   pxb3 = pxbT;
   pixbuf_poop(3)                                                          //  get new pxb3 attributes
   zunlock();

   modified++;                                                             //  image3 modified

thread_exit:

   zlock();
   gdk_window_set_cursor(drmwin->window,0);                                //  restore normal cursor
   if (drag_cursor) gdk_cursor_unref(drag_cursor);                         //  release memory
   zunlock();

   LMcapture = 0;
   crop_stat = 0;
   --thread_busy;
   mwpaint2(1);                                                            //  refresh window
   return 0;
}


//  crop dialog completion callback function

int crop_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat == 1) crop_stat = 2;                                          //  crop done
   else crop_stat = 0;                                                     //  crop canceled
   zdialog_destroy(zd);
   return 0;
}


/**************************************************************************/

//  redeye removal - use mouse to select image area to remove red color    v.27 new

void * redeye_thread(void *);
int    redeye_dialog_event(zdialog *zd, const char *event);
int    redeye_dialog_compl(zdialog *zd, int zstat);

int      redeye_stat = 0;

void m_redeye()
{
   if (! pxb3) return;                                                     //  nothing to redeye
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(pxb1);
   pixbuf_poop(1)

   pthread_create(&tid,0,redeye_thread,null);                              //  thread runs on from here
   return;
}


//  redeye thread function

void * redeye_thread(void *)   
{
   const char     *redeye_message = xtext("drag mouse over red eyes");
   GdkCursor      *drag_cursor = 0;
   zdialog        *zd;
   
   ++thread_busy;

   zd = zdialog_new(xtext("red eye reduction"),Bdone,Bcancel,null);        //  dialog to get user inputs
   zdialog_add_widget(zd,"label","lab1","dialog",redeye_message);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zd,"button","bmore","hb1",xtext("darken"),"space=5");
   zdialog_add_widget(zd,"button","bless","hb1",xtext("go back"),"space=5");
   zdialog_run(zd,redeye_dialog_event,redeye_dialog_compl);

   zlock();
   drag_cursor = gdk_cursor_new(GDK_CROSSHAIR);                            //  set drag cursor
   gdk_window_set_cursor(drmwin->window,drag_cursor);
   zunlock();
   
   dragx = dragy = dragw = dragh = 0;                                      //  no drag rectangle yet
   
   redeye_stat = 1;                                                        //  redeye in progress

   while (redeye_stat == 1)                                                //  loop while redeye in progress
   {
      zsleep(0.1);                                                         //  mouse sample interval, 0.1 secs.

      if (mdx1 || mdy1) {                                                  //  mouse drag underway
         dragw = mdx2 - mdx1;                                              //  drag increment
         dragh = mdy2 - mdy1;
         dragx = mdx1 - dragw/2;                                           //  keep center fixed
         dragy = mdy1 - dragh/2;
         mwpaint2(0);                                                      //  clear old circle
      }

      if (dragx < 0) dragx = 0;                                            //  stay within image area    v.32
      if (dragy < 0) dragy = 0;      
      if (dragx + dragw > wwM) dragw = wwM - dragx;
      if (dragy + dragh > hhM) dragh = hhM - dragy;
      
      zlock();                                                             //  draw new redeye circle
      if (dragw > 0 && dragh > 0)
         gdk_draw_arc(drmwin->window,gdkgc,0,dragx+posMx,dragy+posMy,dragw,dragh,0,64*360);
      zunlock();
   }
   
   if (redeye_stat != 2) pull_image3();                                    //  redeye canceled

   zlock();
   gdk_window_set_cursor(drmwin->window,0);                                //  restore normal cursor
   if (drag_cursor) gdk_cursor_unref(drag_cursor);                         //  release memory
   zunlock();

   --thread_busy;
   mwpaint2(1);                                                            //  refresh window
   return 0;
}


//  redeye dialog event and completion callback functions

int redeye_dialog_event(zdialog * zd, const char *event)
{
   static GdkPixbuf  *pxbT = 0;
   int               wwT, hhT, rsT;
   int               px1, py1, px, py, cx, cy;
   double            r, R, redpart;
   static double     pR, maxred;
   static int        pcx = 0, pcy = 0;
   pixel             ppixT, ppix;

   if (!(dragw > 0 && dragh > 0)) return 0;
   
   px1 = int(dragx / Rscale + org3x);                                      //  convert redeye rectangle 
   py1 = int(dragy / Rscale + org3y);                                      //    to image3 coordinates
   wwT = int(dragw / Rscale);
   hhT = int(dragh / Rscale);
   
   cx = wwT / 2;                                                           //  center
   cy = hhT / 2;
   R = (wwT + hhT) / 4.0;                                                  //  mean radius

   if (cx != pcx || cy != pcy || R != pR) 
   {
      pcx = cx;                                                            //  rectangle has changed
      pcy = cy;
      pR = R;
      pixbuf_free(pxbT);                                 
      pxbT = pixbuf_new(colorspace,0,8,wwT,hhT);                           //  copy redeye rectangle
      pixbuf_copy_area(pxb3,px1,py1,wwT,hhT,pxbT,0,0);

      pixbuf_poop(T);
      maxred = 0;

      for (py = 0; py < hhT; py++)                                         //  scan pixels within circle,
      for (px = 0; px < wwT; px++)                                         //    set initial red threshold
      {
         r = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
         if (r > R) continue;
         ppix = ppixT + py * rsT + px * nch;
         redpart = 1.0 * ppix[0] / (ppix[0] + ppix[1] + ppix[2] + 1);
         if (redpart > maxred) maxred = redpart;
      }
   }

   pixbuf_poop(T);

   if (strEqu(event,"bmore")) 
   {
      maxred = 0.95 * maxred;                                              //  reduce redness threshold
   
      for (py = 0; py < hhT; py++)                                         //  scan pixels
      for (px = 0; px < wwT; px++)
      {
         r = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
         if (r > R) continue;
         ppix = ppixT + py * rsT + px * nch;
         redpart = 1.0 * ppix[0] / (ppix[0] + ppix[1] + ppix[2] + 1);      //  for pixels in pxbT over threshold,
         if (redpart > maxred) {                                           //    darken corresponding pxb3 pixel
            ppix = ppix3 + (py1 + py) * rs3 + (px1 + px) * nch;
            ppix[0] = int(0.90 * ppix[0]);
         }
      }

      modified++;
   }

   if (strEqu(event,"bless")) 
   {
      for (py = 0; py < hhT; py++)                                         //  scan pixels
      for (px = 0; px < wwT; px++)
      {
         r = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
         if (r > R) continue;
         ppix = ppixT + py * rsT + px * nch;
         redpart = 1.0 * ppix[0] / (ppix[0] + ppix[1] + ppix[2] + 1);      //  for pixels in pxbT over threshold,
         if (redpart > maxred) {                                           //    lighten corresponding pxb3 pixel
            ppix = ppix3 + (py1 + py) * rs3 + (px1 + px) * nch;
            ppix[0] = int(1.11 * ppix[0]);
         }
      }

      modified++;
      maxred = 1.053 * maxred;                                             //  increase redness threshold
   }

   mwpaint2(1);
   return 1;
}

int redeye_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat == 1) redeye_stat = 2;                                        //  redeye done
   else redeye_stat = 0;                                                   //  redeye canceled
   zdialog_destroy(zd);
   return 0;
}


/**************************************************************************/

//  rotate image through any arbitrary angle                               v.13

void * rotate_thread(void *);
int    rotate_dialog_event(zdialog *zd, const char * event);
int    rotate_dialog_compl(zdialog *zd, int zstat);

int         rotate_stat = 0;
double      rotate_delta = 0;


void m_rotate(double angle)
{
   if (! pxb3) return;                                                     //  nothing to rotate

   if (angle == 0) {                                                       //  interactive mode                v.41
      push_image3();                                                       //  save image3 undo point
      pixbuf_free(pxb1);                                                   //  image3 >> image1 (with prior mods)
      pxb1 = pixbuf_copy(pxb3);                                            //  (image1 = base image for new mods)
      pixbuf_test(pxb1);
      pixbuf_poop(1)
   }

   rotate_delta = angle;                                                   //  KB mode (no modified/undo)      v.41

   pthread_create(&tid,0,rotate_thread,null);                              //  thread runs on from here
   return;
}


//  rotate thread function

void * rotate_thread(void *)
{
   char        text[20];
   zdialog     *zd;
   
   ++thread_busy;

   if (rotate_delta) {                                                     //  KB mode, +/-90 degrees          v.41
      rotate_angle += rotate_delta;                                        //  new net rotation
      rotate_delta = 0;
      if (rotate_angle >= 360) rotate_angle -=360;
      if (rotate_angle <= -360) rotate_angle +=360;

      zlock();
      pixbuf_free(pxb3);
      pxb3 = pixbuf_rotate(pxb1,rotate_angle);                             //  rotate to new angle
      pixbuf_test(pxb3);
      pixbuf_poop(3);
      zunlock();
      
      mwpaint2(1);                                                         //  update window
      --thread_busy;
      return 0;
   }
   
   zd = zdialog_new(xtext("rotate image"),Bdone,Bcancel,null);             //  dialog: get rotation from user
   zdialog_add_widget(zd,"label","labdeg","dialog","degrees","space=5");
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb3","hb1",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb4","hb1",0,"space=5");
   zdialog_add_widget(zd,"button"," +0.1  ","vb1"," + 0.1 ");              //  button name is increment to use
   zdialog_add_widget(zd,"button"," -0.1  ","vb1"," - 0.1 ");
   zdialog_add_widget(zd,"button"," +1.0  ","vb2"," + 1   ");
   zdialog_add_widget(zd,"button"," -1.0  ","vb2"," - 1   ");
   zdialog_add_widget(zd,"button"," +10.0 ","vb3"," + 10  ");
   zdialog_add_widget(zd,"button"," -10.0 ","vb3"," - 10  ");
   zdialog_add_widget(zd,"button"," +90.0 ","vb4"," + 90  ");
   zdialog_add_widget(zd,"button"," -90.0 ","vb4"," - 90  ");

   zdialog_run(zd,rotate_dialog_event,rotate_dialog_compl);
   
   rotate_stat = 1;

   while (rotate_stat == 1)                                                //  loop while rotate in progress
   {
      zsleep(0.1);                                                         //  sample interval, 0.1 seconds
      
      if (rotate_delta) 
      {
         rotate_angle += rotate_delta;                                     //  new net rotation             v.41
         rotate_delta = 0;
         if (rotate_angle >= 360) rotate_angle -=360;
         if (rotate_angle <= -360) rotate_angle +=360;

         zlock();
         pixbuf_free(pxb3);
         pxb3 = pixbuf_rotate(pxb1,rotate_angle);                          //  rotate to new angle
         pixbuf_test(pxb3);
         pixbuf_poop(3);
         zunlock();

         if (fabs(rotate_angle) < 0.01) rotate_angle = 0;
         if (rotate_angle) modified++;                                     //  image3 modified
         else modified = 0;

         sprintf(text,"degrees: %.1f",rotate_angle);                       //  update dialog angle display
         zdialog_stuff(zd,"labdeg",text);
         mwpaint2(1);                                                      //  update window
      }
   }
   
   if (rotate_stat != 2) pull_image3();                                    //  rotate cancelled

   rotate_angle = 0;
   --thread_busy;
   return 0;
}


//  rotate dialog event and completion callback functions

int rotate_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat == 1) rotate_stat = 2;                                        //  done
   else rotate_stat = 0;                                                   //  canceled
   zdialog_destroy(zd);
   return 0;
}

int rotate_dialog_event(zdialog *zd, const char * event)
{
   int         kk;
   double      incr;
   
   kk = convSD(event,incr);                                                //  button name is increment to use
   if (kk != 0) return 0;
   rotate_delta += incr;
   return 1;
}


/**************************************************************************/

//  resize image                                                           v.15

int      resize_dialog_event(zdialog *zd, const char * event);

int      wpix0, hpix0, wpix1, hpix1;
double   wpct1, hpct1;

void m_resize()
{
   GdkPixbuf   *pxb9;
   zdialog     *zd;
   int         zstat;
   const char  *lockmess = xtext("lock width/height ratio");

   if (! pxb3) return;                                                     //  nothing to resize
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(pxb1);
   pixbuf_poop(1)

   zd = zdialog_new(xtext("resize image"),Bapply,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zd,"vbox","vb11","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb12","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb13","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"label","placeholder","vb11",0);                  //             pixels       percent
   zdialog_add_widget(zd,"label","labw","vb11",Bwidth);                    //    width    [______]     [______]
   zdialog_add_widget(zd,"label","labh","vb11",Bheight);                   //    height   [______]     [______]
   zdialog_add_widget(zd,"label","labpix","vb12","pixels");                //
   zdialog_add_widget(zd,"spin","wpix","vb12","20|9999|1|0");              //    presets  [2/3] [1/2] [1/3] [1/4] 
   zdialog_add_widget(zd,"spin","hpix","vb12","20|9999|1|0");              //
   zdialog_add_widget(zd,"label","labpct","vb13",Bpercent);                //    [_] lock width/height ratio
   zdialog_add_widget(zd,"spin","wpct","vb13","1|500|0.1|100");            //
   zdialog_add_widget(zd,"spin","hpct","vb13","1|500|0.1|100");            //       [ apply ]  [ cancel ]  
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","preset","hb2",Bpreset,"space=5");
   zdialog_add_widget(zd,"button","b 2/3","hb2"," 2/3 ");
   zdialog_add_widget(zd,"button","b 1/2","hb2"," 1/2 ");
   zdialog_add_widget(zd,"button","b 1/3","hb2"," 1/3 ");
   zdialog_add_widget(zd,"button","b 1/4","hb2"," 1/4 ");
   zdialog_add_widget(zd,"check","lock","dialog",lockmess);

   wpix0 = ww3;                                                            //  original image3 width, height
   hpix0 = hh3;
   zdialog_stuff(zd,"wpix",wpix0);
   zdialog_stuff(zd,"hpix",hpix0);
   zdialog_stuff(zd,"lock",1);
   
   zstat = zdialog_run(zd,resize_dialog_event,0);
   zdialog_destroy(zd);

   if (zstat != 1) {                                                       //  resize cancelled
      pull_image3();
      return;
   }

   zlock();
   prior_image3();                                                         //  refresh image3
   pxb9 = pixbuf_scale_simple(pxb3,wpix1,hpix1,interp);                    //  scale to size
   pixbuf_test(pxb9);
   pixbuf_free(pxb3);
   pxb3 = pxb9;
   pixbuf_poop(3);
   zunlock();

   modified++;                                                             //  image3 modified
   mwpaint2(1);                                                            //  update window
}

int resize_dialog_event(zdialog *zd, const char * event)
{
   int         lock;

   zdialog_fetch(zd,"wpix",wpix1);                                         //  get all widget values
   zdialog_fetch(zd,"hpix",hpix1);
   zdialog_fetch(zd,"wpct",wpct1);
   zdialog_fetch(zd,"hpct",hpct1);
   zdialog_fetch(zd,"lock",lock);
   
   if (strEqu(event,"b 2/3")) {
      wpix1 = (2 * wpix0 + 2) / 3;
      hpix1 = (2 * hpix0 + 2) / 3;
   }
   
   if (strEqu(event,"b 1/2")) {
      wpix1 = (wpix0 + 1) / 2;
      hpix1 = (hpix0 + 1) / 2;
   }
   
   if (strEqu(event,"b 1/3")) {
      wpix1 = (wpix0 + 2) / 3;
      hpix1 = (hpix0 + 2) / 3;
   }
   
   if (strEqu(event,"b 1/4")) {
      wpix1 = (wpix0 + 3) / 4;
      hpix1 = (hpix0 + 3) / 4;
   }

   if (strEqu(event,"wpct"))                                               //  width % - set pixel width
      wpix1 = int(wpct1 / 100.0 * wpix0 + 0.5);

   if (strEqu(event,"hpct"))                                               //  height % - set pixel height
      hpix1 = int(hpct1 / 100.0 * hpix0 + 0.5);
   
   if (lock) {                                                             //  if lock, set other dimension
      if (event[0] == 'w')                                                 //    to preserve width/height ratio
         hpix1 = int(wpix1 * (1.0 * hpix0 / wpix0) + 0.5);
      if (event[0] == 'h') 
         wpix1 = int(hpix1 * (1.0 * wpix0 / hpix0) + 0.5);
   }
   
   hpct1 = 100.0 * hpix1 / hpix0;                                          //  set percents to match pixels
   wpct1 = 100.0 * wpix1 / wpix0;
   
   zdialog_stuff(zd,"wpix",wpix1);                                         //  synch. all widget values
   zdialog_stuff(zd,"hpix",hpix1);
   zdialog_stuff(zd,"wpct",wpct1);
   zdialog_stuff(zd,"hpct",hpct1);
   
   return 1;
}


/**************************************************************************/

//  Make an HDR (high dynamic range) image from two images of the same subject 
//  with different exposure levels. The HDR image has expanded gradations of 
//  intensity (visibility of detail) in both the brightest and darkest areas.

void *   HDR_thread(void *);
void     HDR_combine_images(int ww);
void     HDR_level_adjust();
void *   HDR_combine_thread(void *);
int      HDR_level_dialog_event(zdialog *zd, const char *name);

double   hdrweight[9];                                                     //  image weights for 8 zones
int      hdr_weight_update;                                                //  dialog widget update counter

void m_HDR()
{
   if (! pxb3) return;                                                     //  no image

   if (file2) zfree(file2);
   file2 = zgetfile(xtext("select image to combine"),file1,"open");        //  get 2nd HDR image in pxb2
   if (! file2) return;
   pxb2 = load_pixbuf(file2);                                              //  v.33
   if (! pxb2) goto err_nomatch;
   pixbuf_poop(2)

   if (ww2 != ww3) goto err_nomatch;                                       //  validate compatibility
   if (hh2 != hh3) goto err_nomatch;
   if (rs2 != rs3) goto err_nomatch;

   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(pxb1);
   pixbuf_poop(1)

   pthread_create(&tid,0,HDR_thread,null);                                 //  launch thread to combine
   return;                                                                 //    pxb1 + pxb2 >> pxb3

err_nomatch:
   zmessageACK(xtext("2nd file is not compatible with 1st"));
   if (file2) zfree(file2);
   file2 = 0;
   pixbuf_free(pxb2);
   pxb2 = 0;
   return;
}


//  Thread function for combining pxb1 + pxb2 >> pxb3

void * HDR_thread(void *)
{
   double      xfL, xfH, yfL, yfH, tfL, tfH;
   int         firstpass = 1, lastpass = 0;

   ++thread_busy;

   Nalign = 1;                                                             //  alignment in progress
   showRedpix = 1;                                                         //  highlight alignment pixels
   
   fullSize = ww1;                                                         //  full image size
   if (hh1 > ww1) fullSize = hh1;                                          //  (largest dimension)

   alignSize = 100;                                                        //  initial alignment image size
   if (alignSize > fullSize) alignSize = fullSize;

   pixsamp = 1000 * int(parm_pixel_sample_size);                           //  no. pixels to sample

   xoff = yoff = toff = 0;                                                 //  initial offsets = 0
   blend = 0;                                                              //  blend region is 100%

   yst1 = yst2 = yst1B = yst2B = 0;                                        //  disable stuff only for pano
   lens_curve = 0;

   while (true)                                                            //  next alignment stage / image size
   {   
      ww1A = ww1 * alignSize / fullSize;                                   //  align width, height in same ratio
      hh1A = hh1 * alignSize / fullSize;
      ww2A = ww1A;
      hh2A = hh1A;

      if (! lastpass) 
      {
         zlock();
         pixbuf_free(pxb1A);                                               //  create alignment images
         pixbuf_free(pxb2A);
         pxb1A = pixbuf_scale_simple(pxb1,ww1A,hh1A,interp);
         pixbuf_test(pxb1A);
         pxb2A = pixbuf_scale_simple(pxb2,ww2A,hh2A,interp);
         pixbuf_test(pxb2A);
         pixbuf_free(pxb3);
         pxb3 = pixbuf_copy(pxb1A);                                        //  make pxb3 compatible
         pixbuf_test(pxb3);

         pixbuf_poop(1A)                                                   //  get pxb1A attributes
         pixbuf_poop(2A)                                                   //  get pxb2A attributes
         pixbuf_poop(3)                                                    //  get pxb3 attributes
         zunlock();

         get_overlap_region(pxb1A,pxb2A);                                  //  get image overlap area
         get_Bratios(pxb1A,pxb2A);                                         //  get image brightness ratios
         flag_edge_pixels(pxb1A);                                          //  flag high-contrast pixels
      }

      xylim = 2;                                                           //  search range from prior stage:
      xystep = 1;                                                          //    -2 -1 0 +1 +2 pixels

      if (firstpass) xylim = 0.05 * alignSize;                             //  1st stage search range, huge

      if (lastpass) {
         xylim = 1;                                                        //  final stage search range:
         xystep = 0.5;                                                     //    -1.0 -0.5 0.0 +0.5 +1.0
      }

      tlim = xylim / alignSize;                                            //  theta max offset, radians
      tstep = xystep / alignSize;                                          //  theta step size

      HDR_combine_images(0);                                               //  combine images >> pxb3
      mwpaint2(1);                                                         //  update window

      matchlev = match_images(pxb1A,pxb2A);                                //  set base match level
      matchB = matchlev;                                                   //   = best matching offsets

      xfL = xoff - xylim;
      xfH = xoff + xylim + xystep/2;
      yfL = yoff - xylim;
      yfH = yoff + xylim + xystep/2;
      tfL = toff - tlim;
      tfH = toff + tlim + tstep/2;

      xoffB = xoff;
      yoffB = yoff;
      toffB = toff;
      
      for (xoff = xfL; xoff < xfH; xoff += xystep)                         //  test all offset dimensions
      for (yoff = yfL; yoff < yfH; yoff += xystep)                         //    in all combinations
      for (toff = tfL; toff < tfH; toff += tstep)
      {
         matchlev = match_images(pxb1A,pxb2A);
         if (sigdiff(matchlev,matchB,0.00001) > 0) {
            matchB = matchlev;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
            update_status_bar();
         }

         Nalign++;
         update_status_bar();
         if (kill_thread) goto HDR_exit;
      }

      xoff = xoffB;                                                        //  recover best offsets
      yoff = yoffB;
      toff = toffB;

      HDR_combine_images(0);                                               //  combine images >> pxb3
      mwpaint2(1);                                                         //  update window
     
      firstpass = 0;
      if (lastpass) break;                                                 //  done

      if (alignSize == fullSize) {                                         //  full size image was aligned
         lastpass++;                                                       //  one more pass
         continue;
      }

      double R = alignSize;
      alignSize = 2 * alignSize;                                           //  next larger image size
      if (alignSize > 0.8 * fullSize) alignSize = fullSize;                //  if near goal, jump to it now
      R = alignSize / R;                                                   //  ratio of new / old image size
      xoff = R * xoff;                                                     //  adjust offsets for image size
      yoff = R * yoff;
   }

   HDR_level_adjust();                                                     //  dialog, adjust input image weights

HDR_exit:

   Nalign = 0;                                                             //  alignment done
   if (kill_thread) pull_image3();                                         //  killed

   zlock();
   if (file2) zfree(file2);                                                //  free 2nd input file
   file2 = 0;
   pixbuf_free(pxb2);                                                      //  free 2nd input image
   pixbuf_free(pxb1A);                                                     //  free alignment images
   pixbuf_free(pxb2A);
   zunlock();

   mwpaint2(1);                                                            //  repaint window (pxb3)
   --thread_busy;                                                          //  exit thread
   return 0;
}


//  combine images pxb1A and pxb2A using weights in hdrweight[9]
//  hdrweight[0]-[1] for darkest zone, hdrweight[7]-[8] for brightest zone
//  output is in pxb3

void HDR_combine_images(int weight)
{
   int      px1, py1, ii;
   double   px2, py2;
   double   sintf = sin(toff), costf = cos(toff);
   double   br1, br2, brm;
   pixel    pix1, pix2, pix3;

   for (py1 = 0; py1 < hh1A; py1++)                                        //  step through pxb1A pixels
   for (px1 = 0; px1 < ww1A; px1++)
   {
      px2 = (px1 - xoff) * costf + (py1 - yoff) * sintf;                   //  corresponding pxb2A pixels
      py2 = (py1 - yoff) * costf - (px1 - xoff) * sintf;
   
      pix1 = ppix1A + py1 * rs1A + px1 * nch;                              //  input pixel 1
      pix2 = vpixel(ppix2A,px2,py2,ww2A,hh2A,rs2A);                        //  input pixel 2
      pix3 = ppix3 + py1 * rs3 + px1 * nch;                                //  output pxb3 pixel

      if (! pix2) {                                                        //  if non-overlapping pixel,
         pix3[0] = pix3[1] = pix3[2] = 0;                                  //    set output pixel black
         continue;
      }
      
      if (weight) {
         br1 = pix1[0] + pix1[1] + pix1[2];                                //  image 1 pixel brightness
         br2 = pix2[0] + pix2[1] + pix2[2];                                //  image 2
         brm = (br1 + br2) / 2;                                            //  mean, 0 to 765

         ii = int(brm / 96);                                               //  brightness band, 0 .. 7      v.28
         br1 = hdrweight[ii];                                              //  weight at low end of band
         br2 = hdrweight[ii+1];                                            //  weight at high end of band
         brm = br1 + (br2 - br1) * (brm - 96 * ii) / 96;                   //  interpolate within band
         
         br1 = brm / 80.0;                                                 //  image 1 weight: 0 .. 1.0
         br2 = 1.0 - br1;                                                  //  image 2 weight
         
         pix3[0] = int(pix1[0] * br1 + pix2[0] * br2);                     //  build output pixel
         pix3[1] = int(pix1[1] * br1 + pix2[1] * br2);
         pix3[2] = int(pix1[2] * br1 + pix2[2] * br2);
      }

      else {
         pix3[0] = (pix1[0] + pix2[0]) / 2;                                //  output pixel is simple average
         pix3[1] = (pix1[1] + pix2[1]) / 2;
         pix3[2] = (pix1[2] + pix2[2]) / 2;
      }

      if (showRedpix && pix2) {                                            //  highlight alignment pixels   v.25
         ii = py1 * ww1A + px1;
         if (bitmap_get(BMpixels,ii)) {
            pix3[0] = 255;
            pix3[1] = pix3[2] = 0;
         }
      }
   }

   modified++;
   return;
}


//  dialog to adjust image brightness levels
//  pxb1A + pxb2A >> pxb3 under control of scale bars

void HDR_level_adjust()
{
   const char  *weightmess = xtext("\n image weights per brightness band \n");
   int         ii, zstat;
   double      bravg, wval;
   char        vsname[4] = "vs0";
   zdialog     *zd;
   
   zd = zdialog_new(xtext("HDR image weights"),Bdone,Bcancel,null);
   zdialog_add_widget(zd,"label","labt","dialog",weightmess);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"expand");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"expand|space=10");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"expand|space=10");
   zdialog_add_widget(zd,"vbox","vb3","hb1",0,"expand|space=10");
   zdialog_add_widget(zd,"label","lab11","vb1",xtext("input image 2"));
   zdialog_add_widget(zd,"label","lab12","vb1",0,"expand");
   zdialog_add_widget(zd,"label","lab13","vb1",xtext("input image 1"));
   zdialog_add_widget(zd,"label","lab14","vb1",xtext("output image"));
   zdialog_add_widget(zd,"label","lab21","vb2"," 100 % ");
   zdialog_add_widget(zd,"label","lab22","vb2"," 50/50 ","expand");
   zdialog_add_widget(zd,"label","lab23","vb2"," 100 % ");
   zdialog_add_widget(zd,"label","lab24","vb2");
   zdialog_add_widget(zd,"hbox","hb2","vb3",0,"expand|space=5");
   zdialog_add_widget(zd,"vscale","vs0","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs1","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs2","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs3","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs4","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs5","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs6","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs7","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs8","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"hbox","hb3","vb3");
   zdialog_add_widget(zd,"label","lab31","hb3",xtext("darker areas"));
   zdialog_add_widget(zd,"label","lab32","hb3",0,"expand");
   zdialog_add_widget(zd,"label","lab33","hb3",xtext("brighter areas"));
   zdialog_add_widget(zd,"hbox","hb4","dialog",0,"space=15");
   zdialog_add_widget(zd,"label","lab41","hb4",0,"space=50");
   zdialog_add_widget(zd,"button","100/0","hb4","100/0");
   zdialog_add_widget(zd,"button","0/100","hb4","0/100");
   zdialog_add_widget(zd,"button","50/50","hb4","50/50");
   zdialog_add_widget(zd,"button","+/-","hb4","  +/-  ");
   zdialog_add_widget(zd,"button","-/+","hb4","  -/+  ");

   bravg = 0;                                                              //  get mean brightness ratio
   for (ii = 0; ii < 256; ii++)
      bravg = bravg + Bratios[0][ii] + Bratios[1][ii] + Bratios[2][ii];
   bravg = bravg / 256 / 3;
   
   for (ii = 0; ii < 9; ii++)                                              //  initial ramp: use brighter
   {                                                                       //    image for darker bands, etc.
      vsname[2] = '0' + ii;
      if (bravg < 1) wval = 10 * ii;                                       //  ramp up                      v.28
      else  wval = 80 - 10 * ii;                                           //  ramp down
      zdialog_stuff(zd,vsname,wval);
      hdrweight[ii] = wval;
   }

   showRedpix = 0;                                                         //  stop highlights
   HDR_combine_images(1);                                                  //  combine final images >> pxb3
   mwpaint2(1);                                                            //  update window

   hdr_weight_update = 0;

   zdialog_resize(zd,0,400);
   zstat = zdialog_run(zd,HDR_level_dialog_event);                         //  run dialog - blocking
   zdialog_destroy(zd);

   if (zstat != 1) pull_image3();                                          //  cancel, restore image3
   return;
}


//  event callback function for HDR image level adjustment

int HDR_level_dialog_event(zdialog *zd, const char *event)
{
   char        vsname[4] = "vs0";
   double      vsval;
   
   for (int ii = 0; ii < 9; ii++)                                          //  process dialog sliders
   {
      vsname[2] = '0' + ii;
      zdialog_fetch(zd,vsname,vsval);                                      //  get slider ii value

      if (strEqu(event,"100/0")) vsval = 0;                                //  button updates
      if (strEqu(event,"0/100")) vsval = 80;
      if (strEqu(event,"50/50")) vsval = 40;
      if (strEqu(event,"+/-"))   vsval += ii - 4;
      if (strEqu(event,"-/+"))   vsval += 4 - ii;

      if (vsval < 0) vsval = 0;                                            //  stay within limits
      if (vsval > 80) vsval = 80;

      zdialog_stuff(zd,vsname,vsval);                                      //  update dialog slider
      hdrweight[ii] = vsval;                                               //  and corresp. weight
   }
   
   if (hdr_weight_update == 0) {
      hdr_weight_update++;                                                 //  (sequence critical)
      pthread_create(&tid,0,HDR_combine_thread,null);                      //  update image with new settings
   }
   else hdr_weight_update++;

   return 1;
}

void * HDR_combine_thread(void *)                                          //  runs asynchronously   v.36
{
   int      wupdates;

   ++thread_busy;

   while (hdr_weight_update)
   {
      wupdates = hdr_weight_update;
      HDR_combine_images(1);                                               //  combine images >> pxb3
      mwpaint2(1);                                                         //  update window
      hdr_weight_update -= wupdates;
   }

   --thread_busy;
   return 0;
}


/**************************************************************************/

//  panorama function - stitch two images together

void *   pano_thread(void *);                                              //  work function
void     pano_get_align_images();                                          //  rescale images for alignment
void     pano_curve_images(int newf, int strf);                            //  curve/stretch image during alignment
void     pano_interp_curve();                                              //  convert lens mm to % curve
void     pano_prealign();                                                  //  manual pre-align
void     pano_autolens();                                                  //  optimize lens parameters
void     pano_show_images(int fcolor);                                     //  show combined images
void     pano_final_adjust();                                              //  dialog, adjust color and brightness

int      pano_align_stat = 0;                                              //  alignment status
double   Radjust, Gadjust, Badjust;                                        //  manual color adjustments
int      pxM, pxmL, pxmH;                                                  //  pano blend stripe

void m_pano()
{
   if (! pxb3) return;                                                     //  no 1st image

   if (file2) zfree(file2);
   file2 = zgetfile(xtext("select image to combine"),file1,"open");        //  get 2nd pano image in pxb2
   if (! file2) return;
   pxb2 = load_pixbuf(file2);                                              //  v.33
   if (! pxb2) goto err_nomatch;
   pixbuf_poop(2)

   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(pxb1);
   pixbuf_poop(1)

   pthread_create(&tid,0,pano_thread,null);                                //  launch thread to combine
   return;                                                                 //    pxb1 + pxb2 >> pxb3

err_nomatch:
   zmessageACK(xtext("2nd file is not compatible with 1st"));
   if (file2) zfree(file2);
   file2 = 0;
   pixbuf_free(pxb2);
   pxb2 = 0;
   return;
}


//  Thread function for combining pxb1 + pxb2 >> pxb3.

void * pano_thread(void *)
{
   int         firstpass = 1, lastpass = 0;
   double      minblend, alignR;
   double      xfL, xfH, yfL, yfH, tfL, tfH;
   double      yst1L, yst1H, yst2L, yst2H;

   ++thread_busy;

   Nalign = 1;                                                             //  alignment in progress
   showRedpix = 0;                                                         //  no pixel highlight (yet)
   pixsamp = 1000 * int(parm_pixel_sample_size);                           //  no. pixels to sample

   fullSize = hh2;                                                         //  full size = image2 height
   
   alignSize = int(parm_pano_prealign_size);                               //  prealign image size
   pano_get_align_images();                                                //  get prealign images

   xoff = xoffB = 0.8 * ww1A;                                              //  initial x offset (20% overlap)
   yoff = yoffB = toff = toffB = 0;                                        //  initial y and theta offsets
   yst1 = yst1B = yst2 = yst2B = 0;                                        //  initial y-stretch offsets
   lens_mm = parm_pano_lens_mm;                                            //  initial image curvature   v.28
   lens_bow = parm_pano_lens_bow;                                          //  initial image bow (barrel distortion)
   blend = ww1A - xoff;                                                    //  initial blend width = overlap area
   
   pano_curve_images(1,0);                                                 //  make curved alignment images
   get_overlap_region(pxb1C,pxb2C);                                        //  get image overlap area
   pano_show_images(0);                                                    //  show combined images

   pano_prealign();                                                        //  manual pre-align - get new offsets
   if (! pano_align_stat) goto pano_exit;                                  //  user canceled

   blend = ww1A - xoff;                                                    //  new blend from pre-align  v.22
   if (blend < 20) {
      zmessageACK(xtext("too little overlap, cannot align"));
      pano_align_stat = 0;
      goto pano_exit;
   }

   showRedpix = 1;                                                         //  highlight alignment pixels

   while (true)
   {
      alignR = alignSize;
      if (firstpass) alignSize = 100;
      else if (lastpass) alignSize = fullSize;
      else  alignSize = int(parm_pano_align_size_increase * alignSize);    //  next larger image size
      if (alignSize > 0.8 * fullSize) alignSize = fullSize;                //  if near goal, jump to it now
      alignR = alignSize / alignR;                                         //  ratio of new / old image size

      xoff = alignR * xoff;                                                //  adjust offsets for new image size
      yoff = alignR * yoff;
      toff = toff;
      yst1 = alignR * yst1;
      yst2 = alignR * yst2;
      blend = alignR * blend;

      xoffB = xoff;                                                        //  best offsets so far = initial offsets
      yoffB = yoff;
      toffB = toff;
      yst1B = yst1;
      yst2B = yst2;

      if (! firstpass) {
         blend = blend * parm_pano_blend_reduction;                        //  reduce image comparison width
         minblend = 0.01 * parm_pano_minimum_blend * alignSize;
         if (blend < minblend) blend = minblend;                           //  keep above minimum
         if (blend > 0.2 * alignSize) blend = 0.2 * alignSize;             //  keep below 20% of image
      }
      
      xylim = 3;                                                           //  +/- search range, centered on
      xystep = 1;                                                          //    results from prior stage
      
      if (firstpass) xylim = alignSize * 0.05;                             //  tolerate 5 % error in pre-alignment

      if (lastpass) {
         xylim = 1;                                                        //  final stage pixel search steps
         xystep = 0.5;                                                     //   -1.0 -0.5 0.0 +0.5 +1.0
      }

      tlim = xylim / alignSize;                                            //  theta max offset, radians
      tstep = xystep / alignSize;                                          //  theta step size

      if (! lastpass) {
         pano_get_align_images();                                          //  get new alignment images
         pano_curve_images(1,0);                                           //  make new curved alignment images
      }

      get_overlap_region(pxb1C,pxb2C);                                     //  get image overlap area
      get_Bratios(pxb1C,pxb2C);                                            //  get image bright. ratios in overlap 
      flag_edge_pixels(pxb1C);                                             //  flag high-contrast pixels in blend

      if (kill_thread) goto pano_exit;

      zlock();
      gdk_window_freeze_updates(drmwin->window);                           //  inhibit window flashing
      pano_show_images(0);                                                 //  show combined images
      gdk_window_thaw_updates(drmwin->window);
      zunlock();
      
      xfL = xoff - xylim;                                                  //  set x/y/t search ranges, step sizes
      xfH = xoff + xylim + xystep/2;
      yfL = yoff - xylim;
      yfH = yoff + xylim + xystep/2;
      tfL = toff - tlim;
      tfH = toff + tlim + tstep/2;

      matchB = match_images(pxb1C,pxb2C);                                  //  set base match level

      for (xoff = xfL; xoff < xfH; xoff += xystep)                         //  test x, y, theta offsets
      for (yoff = yfL; yoff < yfH; yoff += xystep)                         //    in all possible combinations
      for (toff = tfL; toff < tfH; toff += tstep)
      {
         matchlev = match_images(pxb1C,pxb2C);
         if (sigdiff(matchlev,matchB,0.00001) > 0) {                       //  remember best alignment and offsets
            matchB = matchlev;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
         }

         Nalign++;
         update_status_bar();
         if (kill_thread) goto pano_exit;
      }
      
      xoff = xoffB;                                                        //  recover best offsets
      yoff = yoffB;
      toff = toffB;
      
      pano_show_images(0);                                                 //  refresh window

      if (parm_pano_image_stretch && ! firstpass)                          //  do image2 y-stretch alignment
      {
         ystlim = hh2A * 0.001 * parm_pano_image_stretch;                  //  v.32
         yststep = 0.4;

         yst1L = yst1 - ystlim;                                            //  y-stretch search range, upper half
         yst1H = yst1 + ystlim + yststep/2;

         for (yst1 = yst1L; yst1 < yst1H; yst1 += yststep)                 //  search upper image half
         {
            pano_curve_images(0,1);                                        //  curve and y-stretch
            matchlev = match_images(pxb1C,pxb2C);
            if (sigdiff(matchlev,matchB,0.00001) > 0) {                    //  remember best alignment and y-stretch 
               matchB = matchlev;
               yst1B = yst1;
            }

            Nalign++;
            update_status_bar();
            if (kill_thread) goto pano_exit;
         }
         
         yst1 = yst1B;                                                     //  restore best y-stretch offset
         pano_curve_images(0,1);

         yst2L = yst2 - ystlim;                                            //  lower half search range
         yst2H = yst2 + ystlim + yststep/2;
         
         for (yst2 = yst2L; yst2 < yst2H; yst2 += yststep)                 //  search lower half
         {
            pano_curve_images(0,2);
            matchlev = match_images(pxb1C,pxb2C);
            if (sigdiff(matchlev,matchB,0.00001) > 0) {
               matchB = matchlev;
               yst2B = yst2;
            }

            Nalign++;
            update_status_bar();
            if (kill_thread) goto pano_exit;
         }
         
         yst2 = yst2B;                                                     //  restore best offset
         pano_curve_images(0,0);
         pano_show_images(0);                                              //  refresh window
      }

      firstpass = 0;
      if (lastpass) break;
      if (alignSize == fullSize) lastpass = 1;                             //  one more pass with reduced step size
   }

   pano_final_adjust();                                                    //  user color and brightness adjustments

pano_exit:

   Nalign = 0;                                                             //  alignment done
   if (kill_thread || ! pano_align_stat) pull_image3();                    //  canceled or killed

   zlock();
   if (file2) zfree(file2);                                                //  free 2nd input file
   file2 = 0;
   pixbuf_free(pxb2);                                                      //  free 2nd input image
   pixbuf_free(pxb1A);                                                     //  free alignment images
   pixbuf_free(pxb2A);
   pixbuf_free(pxb1C);
   pixbuf_free(pxb2C);
   zunlock();

   mwpaint2(1);
   --thread_busy;                                                          //  exit thread
   return 0;
}


//  scale images into new pixbufs for alignment
//  scaled images in pxb1A/2A, scaled and curved images in pxb1C/2C

void pano_get_align_images()
{
   ww1A = ww1 * alignSize / fullSize;                                      //  size of alignment images
   hh1A = hh1 * alignSize / fullSize;
   ww2A = ww2 * alignSize / fullSize;
   hh2A = hh2 * alignSize / fullSize;

   zlock();
   pixbuf_free(pxb1A);                                                     //  create alignment images
   pixbuf_free(pxb2A);
   pxb1A = pixbuf_scale_simple(pxb1,ww1A,hh1A,interp);
   pixbuf_test(pxb1A);
   pxb2A = pixbuf_scale_simple(pxb2,ww2A,hh2A,interp);
   pixbuf_test(pxb2A);
   pixbuf_poop(1A)                                                         //  get pxb1A attributes
   pixbuf_poop(2A)                                                         //  get pxb2A attributes
   zunlock();
   return;
}


//  curve and stretch images according to lens_mm, lens_bow, yst1, yst2
//  pxb1A, pxb2A  >>  pxb1C, pxb2C
//  strf:  0: curve both images, y-stretch both halves of image2
//         1: curve image2 only, y-stretch upper half only
//         2: curve image2 only, y-stretch lower half only
//  v.32: curve entire image instead of right or left half

void pano_curve_images(int newf, int strf)
{
   pixel       pix, pixc;
   int         pxc, pyc, pxcL, pxcH, pycL, pycH;
   double      px, py, yst;
   double      xdisp, ydisp, xshrink, yshrink;
   double      costf = cos(toff), sintf = sin(toff);
   double      pcurve, pyadj, pbow, R;

   if (newf)                                                               //  new alignment images are present
   {
      zlock();
      pixbuf_free(pxb1C);                                                  //  make new curved alignment images
      pxb1C = pixbuf_copy(pxb1A);
      pixbuf_test(pxb1C);
      pixbuf_free(pxb2C);
      pxb2C = pixbuf_copy(pxb2A);
      pixbuf_test(pxb2C);
      pixbuf_poop(1C)                                                      //  get attributes
      pixbuf_poop(2C)
      zunlock();
   }
   
   pano_interp_curve();                                                    //  convert lens_mm to curve    v.28
   pcurve = lens_curve * 0.01 * ww2C;                                      //  curve % to pixels
   pyadj = 1.41;                                                           //  y adjustment factor       v.32
   pbow = 0.01 * ww2C * lens_bow;                                          //  lens_bow % to pixels

   R = 1.0 * hh2C / ww2C;
   if (R > 1) pcurve = pcurve / R / R;                                     //  adjust for vertical format     v.28

   if (strf == 0 && ww2C > 0.7 * ww1C)                                     //  no curve if 3rd+ image in pano  v.32
   {   
      for (pyc = 0; pyc < hh1C; pyc++)                                     //  curve image1 (both halves)   v.32
      for (pxc = 0; pxc < ww1C; pxc++)
      {
         xdisp = (pxc - ww1C/2.0) / (ww1C/2.0);                            //  -1 .. 0 .. +1
         ydisp = (pyc - hh1C/2.0) / (ww1C/2.0);                            //  -h/w .. 0 .. +h/w
         xshrink = xdisp * xdisp * xdisp;
         yshrink = pyadj * ydisp * xdisp * xdisp;
         
         px = pxc + pcurve * xshrink;                                      //  x shrink pixels
         py = pyc + pcurve * yshrink;                                      //  y shrink pixels
         px -= pbow * xdisp * ydisp * ydisp;                               //  x bow pixels

         pix = vpixel(ppix1A,px,py,ww1A,hh1A,rs1A);                        //  virtual pixel in uncurved image
         pixc = ppix1C + pyc * rs1C + pxc * nch;                           //  destination pixel in curved image
         if (pix) {  
            pixc[0] = pix[0];
            pixc[1] = pix[1];
            pixc[2] = pix[2];
         }
         else pixc[0] = pixc[1] = pixc[2] = 0;
      }
   }
   
   pycL = 0;
   pycH = hh2C;
   if (strf == 1) pycH = hh2C / 2;                                         //  upper half only
   if (strf == 2) pycL = hh2C / 2;                                         //  lower half only

   pxcL = 0;
   pxcH = ww2C;

   if (strf) {                                                             //  during y-stretch alignment,    v.27
      pxcL = int((pxmL - xoff) * costf + (pyL - yoff) * sintf);            //    stay within blend stripe
      if (pxcL < 0) pxcL = 0;
      pxcH = int((pxmH - xoff) * costf + (pyH - yoff) * sintf);
      if (pxcH > ww2C/2) pxcH = ww2C/2;
   }

   for (pyc = pycL; pyc < pycH; pyc++)                                     //  curve and y-stretch image2
   for (pxc = pxcL; pxc < pxcH; pxc++)
   {
      xdisp = (pxc - ww2C/2.0) / (ww2C/2.0);
      ydisp = (pyc - hh2C/2.0) / (ww2C/2.0);
      xshrink = xdisp * xdisp * xdisp;
      yshrink = pyadj * ydisp * xdisp * xdisp;

      px = pxc + pcurve * xshrink;
      py = pyc + pcurve * yshrink;
      px -= pbow * xdisp * ydisp * ydisp;

      if (pyc < hh2C / 2) yst = yst1;                                      //  do y-stretch
      else yst = yst2;
      if (yst) py += yst * ydisp;

      pix = vpixel(ppix2A,px,py,ww2A,hh2A,rs2A);
      pixc = ppix2C + pyc * rs2C + pxc * nch;
      if (pix) {  
         pixc[0] = pix[0];
         pixc[1] = pix[1];
         pixc[2] = pix[2];
      }
      else pixc[0] = pixc[1] = pixc[2] = 0;
   }

   return;
}


//  convert lens focal length (mm) to image percent curve                  v.28

void pano_interp_curve()
{
   double curvetab[] = { 25, 10.3, 26, 8.4, 28, 6.78, 32, 5.26,            //  25mm = 10.3 % etc.   v.32
                         40, 3.42, 48, 2.32, 60, 1.7, 80, 1.35, 
                         100, 1.2, 200, 1.1, 0 };
   int      ii;
   double   x0, x1, y0, y1;
   
   for (ii = 0; curvetab[ii]; ii += 2)                                     //  find >= table entry
      if (lens_mm <= curvetab[ii]) break;

   if (ii == 0) {                                                          //  1st entry
      lens_curve = curvetab[1];
      return;
   }

   if (curvetab[ii] == 0) {                                                //  EOL
      lens_curve = 1.0;
      return;
   }
   
   x0 = curvetab[ii-2];                                                    //  interpolate
   x1 = curvetab[ii];
   y0 = curvetab[ii-1];
   y1 = curvetab[ii+1];
   lens_curve = y0 + (y1 - y0) * (lens_mm - x0) / (x1 - x0);

   return;
}


//  perform manual pre-align of image2 to image1
//  return offsets: xoff, yoff, toff, lens_mm, lens_bow

void pano_prealign()   
{  
   int pano_prealign_dialog_event(zdialog *zd, const char *event);         //  dialog event function
   int pano_prealign_dialog_compl(zdialog *zd, int zstat);                 //  dialog completion function

   zdialog     *zd;
   int         mx0, my0, mx, my, key;                                      //  mouse drag origin, position
   double      lens_mm0, lens_bow0;
   double      mlever = 1.0 / parm_pano_mouse_leverage;
   double      dtoff;

   const char  *align_message = xtext("drag right image into rough alignment with left \n"
                                      " to rotate, drag right edge up or down");
   const char  *proceed_message = xtext("merge the images together");
   const char  *search_message = xtext("auto-search lens mm and bow");

   zd = zdialog_new("pre-align images",Bcancel,null);                      //  dialog for alignment
   zdialog_add_widget(zd,"label","lab1","dialog",align_message,"space=5");
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");               //  lens mm, 25-200 mm
   zdialog_add_widget(zd,"spin","spmm","hb1","25|200|0.1|35","space=5");
   zdialog_add_widget(zd,"label","labmm","hb1",xtext("lens mm"));
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");               //  lens bow, -9 to +9 %
   zdialog_add_widget(zd,"spin","spbow","hb2","-9|9|0.01|0","space=5");
   zdialog_add_widget(zd,"label","labbow","hb2",xtext("lens bow"));
   zdialog_add_widget(zd,"hbox","hb3","dialog",0,"space=5");               //  next button
   zdialog_add_widget(zd,"button","proceed","hb3",Bproceed,"space=5");
   zdialog_add_widget(zd,"label","labproceed","hb3",proceed_message);
   zdialog_add_widget(zd,"hbox","hb4","dialog",0,"space=5");               //  search button     v.32
   zdialog_add_widget(zd,"button","search","hb4",Bsearch,"space=5");
   zdialog_add_widget(zd,"label","labsearch","hb4",search_message);
   zdialog_stuff(zd,"spmm",lens_mm);                                       //  pre-load current data
   zdialog_stuff(zd,"spbow",lens_bow);
   zdialog_run(zd,pano_prealign_dialog_event,pano_prealign_dialog_compl);  //  run dialog

   lens_mm0 = lens_mm;                                                     //  to detect changes
   lens_bow0 = lens_bow;
   pano_align_stat = 1;                                                    //  status = pre-align in progress
   mx0 = my0 = 0;                                                          //  no drag in progress

   while (pano_align_stat == 1)                                            //  loop while in align status
   {
      if (Fautolens) {
         pano_autolens();                                                  //  get lens parameters   v.28
         if (pano_align_stat != 1) break;
         zdialog_stuff(zd,"spmm",lens_mm);                                 //  update dialog
         zdialog_stuff(zd,"spbow",lens_bow);
      }
         
      if (lens_mm != lens_mm0 || lens_bow != lens_bow0) {                  //  change in lens parameters
         lens_mm0 = lens_mm;
         lens_bow0 = lens_bow;
         pano_curve_images(0,0);
         pano_show_images(0);                                              //  show combined images
      }
      
      if (KBkey) {                                                         //  KB input   v.28
         key = KBkey;
         KBkey = 0;
         if (key == GDK_Left)  xoff -= 0.2;                                //  tweak alignment offsets
         if (key == GDK_Right) xoff += 0.2;
         if (key == GDK_Up)    yoff -= 0.2;
         if (key == GDK_Down)  yoff += 0.2;
         if (key == GDK_r)     toff += 0.0002;
         if (key == GDK_l)     toff -= 0.0002;
         if (key == GDK_p) printf(" %.1f %.2f \n",lens_mm,lens_curve);     //  print curve value

         blend = ww1C - xoff;                                              //  entire overlap
         get_overlap_region(pxb1C,pxb2C);
         pano_show_images(0);                                              //  show combined images
      }

      if (! mdx1 && ! mdy1) {
         mx0 = my0 = 0;                                                    //  no drag in progress
         goto idle;
      }

      mx = mp3x;                                                           //  mouse position in image3
      my = mp3y;

      if (mx < xoff || mx > xoff + ww2C || my > hh3) {                     //  if mouse not in image2 area,
         mx0 = my0 = 0;                                                    //    no drag in progress
         goto idle;
      }

      if (mx0 == 0 && my0 == 0) {                                          //  new drag origin
         mx0 = mx;
         my0 = my;
         goto idle;
      }
      
      if (mx == mx0 && my == my0) goto idle;                               //  no movement

      if (mx > xoff + 0.8 * ww2C) {                                        //  near right edge, theta drag 
         dtoff = mlever * (my - my0) / ww2C;                               //  delta theta, radians
         toff += dtoff;
         xoff += 0.5 * dtoff * hh2C;                                       //  change center of rotation
         yoff -= 0.5 * dtoff * (ww1C - xoff);                              //    to middle of overlap area
      }
      else  {                                                              //  x/y drag
         xoff += mlever * (mx - mx0);                                      //  image2 offsets / mouse leverage
         yoff += mlever * (my - my0);
      }

      mx0 = mx;                                                            //  reset drag origin
      my0 = my;

      if (xoff > ww1C) xoff = ww1C;                                        //  limit nonsense
      if (xoff < 0.3 * ww1C) xoff = 0.3 * ww1C;
      if (yoff < -0.5 * hh2C) yoff = -0.5 * hh2C;
      if (yoff > 0.5 * hh1C) yoff = 0.5 * hh1C;
      if (toff < -0.20) toff = -0.20;
      if (toff > 0.20) toff = 0.20;

      blend = ww1C - xoff;                                                 //  entire overlap
      get_overlap_region(pxb1C,pxb2C);
      pano_show_images(0);                                                 //  show combined images
      zsleep(0.05);
      continue;

idle:                                                                      //  spare time activity
      get_overlap_region(pxb1C,pxb2C);                                     //  get image overlap area
      get_Bratios(pxb1C,pxb2C);                                            //  get image bright. ratios in overlap 
      flag_edge_pixels(pxb1C);                                             //  flag high-contrast pixels in overlap 
      matchB = match_images(pxb1C,pxb2C);                                  //  match images
      xoffB = xoff;
      yoffB = yoff;
      toffB = toff;
      update_status_bar();
      zsleep(0.1);
   }

   return;
}

int pano_prealign_dialog_event(zdialog *zd, const char *event)             //  dialog event callback function
{
   if (strEqu(event,"spmm")) zdialog_fetch(zd,"spmm",lens_mm);             //  get revised lens data
   if (strEqu(event,"spbow")) zdialog_fetch(zd,"spbow",lens_bow);
   if (strEqu(event,"search")) Fautolens = 1;                              //  trigger auto-lens function

   if (strEqu(event,"proceed")) {                                          //  proceed with pano
      pano_align_stat = 2;
      zdialog_destroy(zd);
   }
   return 0;
}

int pano_prealign_dialog_compl(zdialog *zd, int zstat)                     //  dialog completion callback function
{
   zdialog_destroy(zd);                                                    //  (only if cancel button used)
   pano_align_stat = 0;
   return 0;
}


//  optimize lens parameters                                               v.32
//  assumes a good starting point since search ranges are limited
//  inputs and outputs: lens_mm, lens_bow, xoff, yoff, toff

void pano_autolens()
{
   double   mm_range, bow_range, xoff_range, yoff_range, toff_range;
   double   squeeze, xoff_rfinal, rnum;
   int      counter = 0;
   
   mm_range = 0.1 * lens_mm;                                               //  set initial search ranges
   bow_range = 0.5 * lens_bow;
   if (bow_range < 1) bow_range = 1;
   xoff_range = 7;
   yoff_range = 7;
   toff_range = 0.01;

   xoff_rfinal = 0.2;                                                      //  final xoff range - when to quit
   
   Nalign = 0;
   showRedpix = 1;
   pano_curve_images(0,0);
   get_overlap_region(pxb1C,pxb2C);
   get_Bratios(pxb1C,pxb2C);
   flag_edge_pixels(pxb1C);
   pano_show_images(0);

   lens_mmB = lens_mm;                                                     //  initial best fit = current data
   lens_bowB = lens_bow;
   xoffB = xoff;
   yoffB = yoff;
   toffB = toff;
   matchB = match_images(pxb1C,pxb2C);
   
   while (true)
   {
      srand48(time(0) + counter++);
      squeeze = 0.9;                                                       //  search range reduction factor

      lens_mm = lens_mmB + mm_range * (drand48() - 0.5);                   //  new random lens factors
      lens_bow = lens_bowB + bow_range * (drand48() - 0.5);                //     within search range
      pano_curve_images(0,0);
         
      for (int ii = 0; ii < 300; ii++)
      {                                                                    
         rnum = drand48();

         if (rnum < 0.33)                                                  //  random change some alignment offset 
            xoff = xoffB + xoff_range * (drand48() - 0.5);                 //    within search range
         else if (rnum < 0.67)
            yoff = yoffB + yoff_range * (drand48() - 0.5);
         else
            toff = toffB + toff_range * (drand48() - 0.5);
      
         matchlev = match_images(pxb1C,pxb2C);                             //  test quality of image alignment
         if (sigdiff(matchlev,matchB,0.00001) > 0) {
            get_overlap_region(pxb1C,pxb2C);                               //  better: reset alignment data
            get_Bratios(pxb1C,pxb2C);
            flag_edge_pixels(pxb1C);
            matchB = match_images(pxb1C,pxb2C);                            //  save new best fit
            lens_mmB = lens_mm;
            lens_bowB = lens_bow;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
            pano_show_images(0);                                           //  update window
            squeeze = 1;                                                   //  keep same search range as long
            break;                                                         //    as improvements are found
         }

         Nalign++;
         update_status_bar();
         if (kill_thread) goto done;
         if (pano_align_stat != 1) goto done;
      }

      if (xoff_range < xoff_rfinal) goto done;

      mm_range = squeeze * mm_range;                                       //  reduce search range if no 
      if (mm_range < 0.02 * lens_mmB) mm_range = 0.02 * lens_mmB;          //    improvements were found
      bow_range = squeeze * bow_range;
      if (bow_range < 0.1 * lens_bowB) bow_range = 0.1 * lens_bowB;
      if (bow_range < 0.2) bow_range = 0.2;
      xoff_range = squeeze * xoff_range;
      yoff_range = squeeze * yoff_range;
      toff_range = squeeze * toff_range;
   }

done:
   lens_mm = lens_mmB;                                                     //  set best alignment found
   lens_bow = lens_bowB;
   xoff = xoffB;
   yoff = yoffB;
   toff = toffB;
   pano_curve_images(0,0);
   pano_show_images(0);
   update_status_bar();

   zmessageACK("lens mm: %.1f  bow: %.2f",lens_mm,lens_bow);
   Fautolens = 0;
   showRedpix = 0;
   return;
}


//  combine images and update window during alignment                      v.20 overhauled
//    pxb1C + pxb2C  >>  pxb3

void pano_show_images(int fcolor)
{
   int            px3, py3, ii, max;
   int            red1, green1, blue1;
   int            red2, green2, blue2;
   int            red3, green3, blue3;
   pixel          pix1, pix2, pix3;
   double         ww, px2, py2, f1, f2;
   double         costf = cos(toff), sintf = sin(toff);

   ww = xoff + ww2C;                                                       //  combined width
   if (toff < 0) ww -= hh2C * toff;                                        //  adjust for theta
   ww3 = int(ww+1);
   hh3 = hh1A;
   
   zlock();
   pixbuf_free(pxb3);
   pxb3 = pixbuf_new(colorspace,0,8,ww3,hh3);
   pixbuf_test(pxb3);
   pixbuf_poop(3)                                                          //  get pxb3 attributes
   zunlock();

   red1 = green1 = blue1 = 0;                                              //  suppress compiler warnings
   red2 = green2 = blue2 = 0;

   for (py3 = 0; py3 < hh3; py3++)                                         //  step through pxb3 rows
   for (px3 = 0; px3 < ww3; px3++)                                         //  step through pxb3 pixels in row
   {
      pix1 = pix2 = 0;
      red3 = green3 = blue3 = 0;

      if (px3 < pxmH && px3 < ww1C && py3 < hh1C)
         pix1 = ppix1C + py3 * rs1C + px3 * nch;                           //  pxb1C pixel

      if (px3 >= pxmL) {
         px2 = costf * (px3 - xoff) + sintf * (py3 - yoff);                //  pxb2C pixel, after offsets
         py2 = costf * (py3 - yoff) - sintf * (px3 - xoff);
         pix2 = vpixel(ppix2C,px2,py2,ww2C,hh2C,rs2C);
      }
      
      if (pix1) {
         red1 = pix1[0];                                                   //  image1 pixel
         green1 = pix1[1];
         blue1 = pix1[2];
         if (!red1 && !green1 && !blue1) pix1 = 0;
      }

      if (pix2) {                                                          //  image2 pixel
         red2 = pix2[0];
         green2 = pix2[1];
         blue2 = pix2[2];
         if (!red2 && !green2 && !blue2) pix2 = 0;
      }
      
      if (fcolor && pix2) {                                                //  color compensate image2 pixel
         red2 = int(brightfix0[red2] * Radjust);                           //  (auto + manual adjustments)
         green2 = int(brightfix1[green2] * Gadjust);
         blue2 = int(brightfix2[blue2] * Badjust);
         if (red2 > 255 || green2 > 255 || blue2 > 255) {                  //  fix overflow        v.26
            max = red2;
            if (green2 > max) max = green2;
            if (blue2 > max) max = blue2;
            f1 = 255.0 / max;
            red2 = int(red2 * f1);
            green2 = int(green2 * f1);
            blue2 = int(blue2 * f1);
         }
      }

      if (pix1) {
         if (! pix2) {
            red3 = red1;                                                   //  use image1 pixel
            green3 = green1;
            blue3 = blue1; 
         }
         else {                                                            //  use blended pixel
            if (fcolor) {
               if (blend == 0) f1 = 1.0;
               else f1 = (pxmH - px3) / blend;                             //  use progressive blend
               f2 = 1.0 - f1;
               red3 = int(f1 * red1 + f2 * red2);
               green3 = int(f1 * green1 + f2 * green2);
               blue3 = int(f1 * blue1 + f2 * blue2);
            }
            else {                                                         //  use 50/50 mix
               red3 = (red1 + red2) / 2;
               green3 = (green1 + green2) / 2;
               blue3 = (blue1 + blue2) / 2;
            }
         }
      }

      else if (pix2) {
            red3 = red2;                                                   //  use image2 pixel
            green3 = green2;
            blue3 = blue2; 
      }

      pix3 = ppix3 + py3 * rs3 + px3 * nch;                                //  output pixel
      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
      
      if (showRedpix && pix1 && pix2) {                                    //  highlight alignment pixels   v.25
         ii = py3 * ww1C + px3;
         if (bitmap_get(BMpixels,ii)) {
            pix3[0] = 255;
            pix3[1] = pix3[2] = 0;
         }
      }
   }

   modified++;
   mwpaint2(1);                                                            //  update window
   return;
}


//  adjust color, brightness, curvature and recombine images

void pano_final_adjust()
{
   int pano_color_dialog_event(zdialog *zd, const char *event);            //  dialog event function

   zdialog     *zd;
   int         zstat;
   const char  *adjmessage = xtext("\n adjust right image to match left");

   get_Bratios(pxb1C,pxb2C);                                               //  get color ratios for overlap region

   blend = 1;                                                              //  show color diffs at joint
   showRedpix = 0;                                                         //  no alignment pixel highlights
   Radjust = Gadjust = Badjust = 1.0;                                      //  RGB manual adjustments, none yet

   get_overlap_region(pxb1C,pxb2C);
   pano_show_images(1);                                                    //  show final results

   zd = zdialog_new(xtext("match images"),Bdone,Bcancel,null);             //  color adjustment dialog
   zdialog_add_widget(zd,"label","lab0","dialog",adjmessage,"space=10");
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"expand|space=10");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog");                    //  red           [ 100 ]
   zdialog_add_widget(zd,"label","lab1","vb1",Bred,"space=7");             //  green         [ 100 ]
   zdialog_add_widget(zd,"label","lab2","vb1",Bgreen,"space=7");           //  blue          [ 100 ]
   zdialog_add_widget(zd,"label","lab3","vb1",Bblue,"space=7");            //  brightness    [ 100 ]
   zdialog_add_widget(zd,"label","lab4","vb1",Bbrightness,"space=7");      //  blend width   [  0  ]
   zdialog_add_widget(zd,"label","lab5","vb1",Bblendwidth,"space=7");      //
   zdialog_add_widget(zd,"spin","spred","vb2","50|200|1|100");             //     [ apply ]
   zdialog_add_widget(zd,"spin","spgreen","vb2","50|200|1|100");
   zdialog_add_widget(zd,"spin","spblue","vb2","50|200|1|100");
   zdialog_add_widget(zd,"spin","spbright","vb2","50|200|1|100");
   zdialog_add_widget(zd,"spin","spblend","vb2","1|100|1|1");
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"homog|space=10");
   zdialog_add_widget(zd,"button","apply","hb2",Bapply);
   
   zstat = zdialog_run(zd,pano_color_dialog_event);                        //  run dialog
   zdialog_destroy(zd);
   
   if (zstat == 1) return;                                                 //  done
   pull_image3();                                                          //  cancel
   return;
}


//  callback function for adjust color / brightness dialog
//  pxb1C + pxb2C >> pxb3 under control of spin buttons

int pano_color_dialog_event(zdialog *zd, const char *event)                //  v.20 overhauled
{
   double      red, green, blue, bright, bright2;

   if (strNeq(event,"apply")) return 0;                                    //  wait for apply button

   zdialog_fetch(zd,"spred",red);                                          //  get color adjustments
   zdialog_fetch(zd,"spgreen",green);
   zdialog_fetch(zd,"spblue",blue);
   zdialog_fetch(zd,"spbright",bright);                                    //  brightness adjustment
   zdialog_fetch(zd,"spblend",blend);                                      //  blend width

   bright2 = (red + green + blue) / 3;                                     //  RGB brightness
   bright = bright / bright2;                                              //  bright setpoint / RGB brightness
   red = red * bright;                                                     //  adjust RGB brightness
   green = green * bright;
   blue = blue * bright;
   
   bright = (red + green + blue) / 3;
   zdialog_stuff(zd,"spred",red);                                          //  force back into consistency
   zdialog_stuff(zd,"spgreen",green);
   zdialog_stuff(zd,"spblue",blue);
   zdialog_stuff(zd,"spbright",bright);

   Radjust = red / 100;                                                    //  normalize 0.5 ... 2.0
   Gadjust = green / 100;
   Badjust = blue / 100;

   get_overlap_region(pxb1C,pxb2C);
   pano_show_images(1);                                                    //  combine and update window
   return 1;
}


/**************************************************************************
         functions shared by HDR and pano
**************************************************************************/

//  get a virtual pixel at location (px,py) (real values).
//  get the overlapping real pixels and build a composite.

pixel vpixel(pixel ppix, double px, double py, int ww, int hh, int rs)
{
   int            px0, py0;
   pixel          pix0, pix1, pix2, pix3;
   double         f0, f1, f2, f3;
   double         red, green, blue;

   static uchar   rpix[4];

   px0 = int(px);                                                          //  pixel containing (px,py)
   py0 = int(py);

   if (px0 < 1 || py0 < 1) return 0;                                       //  sharpen the border    v.32
   if (px0 > ww-3 || py0 > hh-3) return 0;
   
   pix0 = ppix + py0 * rs + px0 * nch;                                     //  4 pixels based at (px0,py0)
   pix1 = pix0 + rs;
   pix2 = pix0 + nch;
   pix3 = pix0 + rs + nch;

   f0 = (px0+1 - px) * (py0+1 - py);                                       //  overlap of (px,py)
   f1 = (px0+1 - px) * (py - py0);                                         //   in each of the 4 pixels
   f2 = (px - px0) * (py0+1 - py);
   f3 = (px - px0) * (py - py0);
   
   red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];      //  sum the weighted inputs
   green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
   blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
      
   rpix[0] = int(red + 0.5);
   rpix[1] = int(green + 0.5);
   rpix[2] = int(blue + 0.5);

   return rpix;
}


//  compare two doubles for significant difference                         //  v.10
//  return:  0  difference not significant
//          +1  d1 > d2
//          -1  d1 < d2

int sigdiff(double d1, double d2, double signf)
{
   double diff = fabs(d1-d2);
   if (diff == 0.0) return 0;
   diff = diff / (fabs(d1) + fabs(d2));
   if (diff < signf) return 0;
   if (d1 > d2) return 1;
   else return -1;
}


//  Get the rectangle containing the overlap region of two images          v.32 revised

void get_overlap_region(GdkPixbuf *pxb91, GdkPixbuf *pxb92)
{
   int         ww91, hh91, ww92, hh92, rs91, rs92;
   int         pxL2, pyL2;
   pixel       ppix91, ppix92;

   pixbuf_poop(91)
   pixbuf_poop(92)

   pxL = 0;
   if (xoff > 0) pxL = int(xoff);

   pxH = ww91;
   if (pxH > xoff + ww92) pxH = int(xoff + ww92);
   
   pyL = 0;
   if (yoff > 0) pyL = int(yoff);
   
   pyH = hh91;
   if (pyH > yoff + hh92) pyH = int(yoff + hh92);

   if (toff > 0) {
      pyL2 = int(yoff + toff * (pxH - pxL));
      if (pyL2 > pyL) pyL = pyL2;
   }

   if (toff < 0) {   
      pxL2 = int(xoff - toff * (pyH - pyL));
      if (pxL2 > pxL) pxL = pxL2;
   }
   
   if (blend > 0) {
      pxM = int((ww91 + xoff) / 2 - 0.5 * toff * (pyH - pyL));             //  midpoint of blend stripe
      pxmL = pxM - int(blend/2);                                           //  blend stripe pixel x-range
      pxmH = pxM + int(blend/2);
      if (pxmL < pxL) pxmL = pxL;
      if (pxmH > pxH) pxmH = pxH;
   }
   else {                                                                  //  no blend stripe, use whole range
      pxmL = pxL;
      pxmH = pxH;
   }

   return;
}


//  Compute brightness ratio by color for overlapping image areas.         v.23  64 >> 256 bins
//    (image2 is overlayed on image1, offset by xoff, yoff, toff)
//  Output: Bratios[kk][Nth] = image1/image2 brightness ratio for color kk
//          for Nth distribution bin of pixels grouped by kk brightness.

void get_Bratios(GdkPixbuf *pxb91, GdkPixbuf *pxb92)
{
   pixel       ppix91, ppix92, pix1, pix2;
   int         ww91, hh91, ww92, hh92, rs91, rs92;
   int         px1, py1;
   int         ii, jj, kk, npix, npix1, npix2, npix3;
   int         brdist1[3][256], brdist2[3][256];
   double      px2, py2;
   double      brlev1[3][256], brlev2[3][256];
   double      costf = cos(toff), sintf = sin(toff);
   double      a1, a2, b1, b2;

   pixbuf_poop(91)
   pixbuf_poop(92)
   
   for (kk = 0; kk < 3; kk++)                                              //  clear distributions
   for (ii = 0; ii < 256; ii++)
      brdist1[kk][ii] = brdist2[kk][ii] = 0;
      
   npix = 0;

   for (py1 = pyL; py1 < pyH; py1++)                                       //  scan image1/image2 pixels in parallel
   for (px1 = pxL; px1 < pxH; px1++)                                       //  use entire overlap area
   {
      pix1 = ppix91 + py1 * rs91 + px1 * nch;                              //  image1 pixel
      if (!pix1[0] && !pix1[1] && !pix1[2]) continue;                      //  ignore black pixels

      px2 = (px1 - xoff) * costf + (py1 - yoff) * sintf;                   //  corresponding image2 pixel
      py2 = (py1 - yoff) * costf - (px1 - xoff) * sintf;
      pix2 = vpixel(ppix92,px2,py2,ww92,hh92,rs92);
      if (! pix2) continue;                                                //  does not exist
      if (!pix2[0] && !pix2[1] && !pix2[2]) continue;                      //  ignore black pixels

      ++npix;                                                              //  count overlapping pixels
      
      for (kk = 0; kk < 3; kk++)                                           //  accumulate brightness distributions
      {                                                                    //    by color in 256 bins
         ++brdist1[kk][pix1[kk]];
         ++brdist2[kk][pix2[kk]];
      }
   }
   
   npix1 = npix / 256;                                                     //  1/256th of total pixels
   
   for (kk = 0; kk < 3; kk++)                                              //  calc. brlev1[kk][N] = mean brightness
   for (ii = jj = 0; jj < 256; jj++)                                       //    for Nth group of image1 pixels
   {                                                                       //      for color kk
      brlev1[kk][jj] = 0;
      npix2 = npix1;                                                       //  1/256th of total pixels

      while (npix2 > 0 && ii < 256)                                        //  next 1/256th group from distribution
      {
         npix3 = brdist1[kk][ii];
         if (npix3 == 0) { ++ii; continue; }
         if (npix3 > npix2) npix3 = npix2;
         brlev1[kk][jj] += ii * npix3;                                     //  brightness * (pixels with brightness)
         brdist1[kk][ii] -= npix3;
         npix2 -= npix3;
      }

      brlev1[kk][jj] = brlev1[kk][jj] / npix1;                             //  mean brightness for group, 0-255
   }

   for (kk = 0; kk < 3; kk++)                                              //  do same for image2
   for (ii = jj = 0; jj < 256; jj++)
   {
      brlev2[kk][jj] = 0;
      npix2 = npix1;

      while (npix2 > 0 && ii < 256)
      {
         npix3 = brdist2[kk][ii];
         if (npix3 == 0) { ++ii; continue; }
         if (npix3 > npix2) npix3 = npix2;
         brlev2[kk][jj] += ii * npix3;
         brdist2[kk][ii] -= npix3;
         npix2 -= npix3;
      }

      brlev2[kk][jj] = brlev2[kk][jj] / npix1;
   }

   for (kk = 0; kk < 3; kk++)                                              //  color
   for (ii = jj = 0; ii < 256; ii++)                                       //  brlev1 brightness, 0 to 255
   {
      while (ii > brlev2[kk][jj] && jj < 256) ++jj;                        //  find matching brlev2 brightness
      a2 = brlev2[kk][jj];                                                 //  next higher value
      b2 = brlev1[kk][jj];
      if (a2 > 0 && b2 > 0) {
         if (jj > 0) {
            a1 = brlev2[kk][jj-1];                                         //  next lower value
            b1 = brlev1[kk][jj-1];
         }
         else   a1 = b1 = 0;
         if (ii == 0) Bratios[kk][ii] = b2 / a2;
         else Bratios[kk][ii] = (b1 + (ii-a1)/(a2-a1) * (b2-b1)) / ii;     //  interpolate
      }
      else Bratios[kk][ii] = 1;
   }

   for (ii = 0; ii < 256; ii++)                                            //  brightness adjustment factors
   {                                                                       //    for match_pixels() function    v.23
      brightfix0[ii] = Bratios[0][ii] * ii;
      brightfix1[ii] = Bratios[1][ii] * ii;
      brightfix2[ii] = Bratios[2][ii] * ii;
   }

   return;
}


//  find pixels of greatest contrast within overlap area                   v.27 improved
//  flag high-contrast pixels to use in each image compare region

void flag_edge_pixels(GdkPixbuf *pxb9)
{
   void  flag_edge_pixels2(GdkPixbuf *, int pxL, int pxH, int pyL, int pyH, int samp);

   pixel       ppix9;
   int         ww9, hh9, rs9, samp;

   pixbuf_poop(9)

   if (BMpixels) bitmap_delete(BMpixels);                                  //  bitmap to flag alignment pixels
   BMpixels = bitmap_new(ww9*hh9);

   if (blend == 0) {                                                       //  HDR: use 4 regions
      samp = pixsamp / 4;
      flag_edge_pixels2(pxb9, pxL, pxH/2, pyL, pyH/2, samp);
      flag_edge_pixels2(pxb9, pxH/2, pxH, pyL, pyH/2, samp);
      flag_edge_pixels2(pxb9, pxL, pxH/2, pyH/2, pyH, samp);
      flag_edge_pixels2(pxb9, pxH/2, pxH, pyH/2, pyH, samp);
   }
   else {                                                                  //  pano: use 4 regions in blend stripe
      samp = pixsamp / 4;                                                  //  v.32
      flag_edge_pixels2(pxb9, pxmL, pxmH, pyL, pyH/4, samp);
      flag_edge_pixels2(pxb9, pxmL, pxmH, pyH/4, pyH/2, samp);
      flag_edge_pixels2(pxb9, pxmL, pxmH, pyH/2, pyH-pyH/4, samp);
      flag_edge_pixels2(pxb9, pxmL, pxmH, pyH-pyH/4, pyH, samp);
   }

   return;
}


//  Find the highest contrast pixels meeting sample size
//    within the specified sub-region of image overlap.

void flag_edge_pixels2(GdkPixbuf *pxb9, int pxL, int pxH, int pyL, int pyH, int samp)
{
   pixel       ppix9;
   int         ww9, hh9, rs9;
   int         px1, py1, ii, jj, npix;
   int         red1, green1, blue1, red2, green2, blue2, tcon;
   int         Hdist[766], Vdist[766], Hmin, Vmin;
   uchar       *Hcon, *Vcon;
   pixel       pix1, pix2;
   
   pixbuf_poop(9)

   npix = (pxH - pxL) * (pyH - pyL);                                       //  overlapping pixels
   if (npix < 100) return;                                                 //  insignificant
   if (samp > npix / 4) samp = npix / 4;                                   //  use max. 1/4 of pixels       v.32

   Hcon = (uchar *) zmalloc(npix);                                         //  horizontal pixel contrast 0-255
   Vcon = (uchar *) zmalloc(npix);                                         //  vertical pixel contrast 0-255

   for (py1 = pyL; py1 < pyH-2; py1++)                                     //  scan image pixels in sub-region
   for (px1 = pxL; px1 < pxH-2; px1++)
   {
      ii = (py1-pyL) * (pxH-pxL) + (px1-pxL);
      Hcon[ii] = Vcon[ii] = 0;                                             //  horiz. = vert. contrast = 0

      pix1 = ppix9 + py1 * rs9 + px1 * nch;                                //  base pixel
      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];
      if (!red1 && !green1 && !blue1) continue;                            //  ignore if black

      pix2 = pix1 + 2 * nch;                                               //  2 pixels to right
      red2 = pix2[0];
      green2 = pix2[1];
      blue2 = pix2[2];
      if (!red2 && !green2 && !blue2) continue;                            //  ignore if black
      tcon = abs(red1-red2) + abs(green1-green2) + abs(blue1-blue2);       //  horizontal contrast
      if (tcon > 10) Hcon[ii] = tcon / 3;                                  //  ignore low contrast pixels   v.32

      pix2 = pix1 + 2 * rs9;                                               //  2 pixels below
      red2 = pix2[0];
      green2 = pix2[1];
      blue2 = pix2[2];
      if (!red2 && !green2 && !blue2) continue;                            //  ignore if black
      tcon = abs(red1-red2) + abs(green1-green2) + abs(blue1-blue2);       //  vertical contrast
      if (tcon > 10) Vcon[ii] = tcon / 3;
   }

   for (ii = 0; ii < 766; ii++) Hdist[ii] = Vdist[ii] = 0;                 //  clear contrast distributions

   for (py1 = pyL; py1 < pyH-2; py1++)                                     //  scan image pixels
   for (px1 = pxL; px1 < pxH-2; px1++)
   {                                                                       //  build contrast distributions
      ii = (py1-pyL) * (pxH-pxL) + (px1-pxL);
      ++Hdist[Hcon[ii]];
      ++Vdist[Vcon[ii]];
   }
   
   for (npix = 0, ii = 765; ii > 0; ii--)                                  //  find minimum contrast needed to get
   {                                                                       //    enough pixels for sample size
      npix += Hdist[ii];                                                   //      (horizontal contrast pixels)
      if (npix > samp) break; 
   }
   Hmin = ii; 

   for (npix = 0, ii = 765; ii > 0; ii--)                                  //  (verticle contrast pixels)
   {
      npix += Vdist[ii];
      if (npix > samp) break;
   }
   Vmin = ii; 

   for (py1 = pyL; py1 < pyH-2; py1++)                                     //  scan image pixels
   for (px1 = pxL; px1 < pxH-2; px1++)
   {
      ii = (py1-pyL) * (pxH-pxL) + (px1-pxL);
      jj = py1 * ww9 + px1;

      if (Hcon[ii] > Hmin) {
         bitmap_set(BMpixels,jj,1);                                        //  flag horizontal group of 3
         bitmap_set(BMpixels,jj+1,1);
         bitmap_set(BMpixels,jj+2,1);
      }

      if (Vcon[ii] > Vmin) {
         bitmap_set(BMpixels,jj,1);                                        //  flag verticle group of 3
         bitmap_set(BMpixels,jj+ww9,1);
         bitmap_set(BMpixels,jj+2*ww9,1);
      }
   }
   
   zfree(Hcon);
   zfree(Vcon);

   return;
}


//  Compare two images in overlapping areas.
//  (image2 is overlayed on image1, offset by xoff, yoff, toff)
//  If blend > 0, compare verticle stripe in the middle, width = blend.
//  Use pixels with contrast >= minimum needed to reach enough pixels.
//  return: 1 = perfect match, 0 = total mismatch (black/white)

double match_images(GdkPixbuf *pxb91, GdkPixbuf *pxb92)
{
   pixel       ppix91, ppix92, pix1, pix2;
   int         ww91, hh91, ww92, hh92, rs91, rs92;
   int         px1, py1, ii, pixcount, ymid;
   double      px2, py2;
   double      costf = cos(toff), sintf = sin(toff);
   double      match, cmatch, maxcmatch;
   double      yspan, weight;

   pixbuf_poop(91)
   pixbuf_poop(92)
   
   cmatch = maxcmatch = pixcount = 0;

   if (blend > 0) goto match_blend;

   for (py1 = pyL; py1 < pyH; py1++)                                       //  scan image1/image2 pixels in parallel
   for (px1 = pxL; px1 < pxH; px1++)
   {
      ii = py1 * ww91 + px1;                                               //  skip low-contrast pixels   v.25
      if (! bitmap_get(BMpixels,ii)) continue;

      pix1 = ppix91 + py1 * rs91 + px1 * nch;                              //  image1 pixel
      if (!pix1[0] && !pix1[1] && !pix1[2]) continue;                      //  ignore black pixels

      px2 = (px1 - xoff) * costf + (py1 - yoff) * sintf;                   //  corresponding image2 pixel
      py2 = (py1 - yoff) * costf - (px1 - xoff) * sintf;
      pix2 = vpixel(ppix92,px2,py2,ww92,hh92,rs92);
      if (! pix2) continue;                                                //  does not exist
      if (!pix2[0] && !pix2[1] && !pix2[2]) continue;                      //  ignore black pixels

      match = match_pixels(pix1,pix2);                                     //  compare brightness adjusted
      cmatch += match;                                                     //  accumulate total match
      pixcount++;
   }

   return cmatch / pixcount;

match_blend:

   if (pxM > ww91) return 0;                                               //  blend runs off of image, no match

   yspan = 1.0 / (pyH - pyL);                                              //  prepare params for weight calculation
   ymid = (pyL + pyH) / 2;                                                 //  (weight central pixels more)

   for (py1 = pyL; py1 < pyH; py1++)                                       //  step through image1 pixels, y
   {
      weight = 1 - yspan * abs(py1 - ymid);                                //  weight: 0.5 .. 1.0 .. 0.5   v.32

      for (px1 = pxmL; px1 < pxmH; px1++)                                  //  step through image1 pixels, x
      {
         ii = py1 * ww91 + px1;                                            //  skip low-contrast pixels   v.25
         if (! bitmap_get(BMpixels,ii)) continue;

         pix1 = ppix91 + py1 * rs91 + px1 * nch;                           //  image1 pixel
         if (!pix1[0] && !pix1[1] && !pix1[2]) continue;                   //  ignore black pixels

         px2 = (px1 - xoff) * costf + (py1 - yoff) * sintf;                //  corresponding image2 pixel
         py2 = (py1 - yoff) * costf - (px1 - xoff) * sintf;
         pix2 = vpixel(ppix92,px2,py2,ww92,hh92,rs92);
         if (! pix2) continue;                                             //  does not exist
         if (!pix2[0] && !pix2[1] && !pix2[2]) continue;                   //  ignore black pixels

         match = match_pixels(pix1,pix2);                                  //  compare brightness adjusted
         cmatch += match * weight;                                         //  accumulate total match
         maxcmatch += weight;
         pixcount++;
      }
   }

   return cmatch / maxcmatch;
}


//  Compare 2 pixels using precalculated brightness ratios
//  1.0 = perfect match   0 = total mismatch (black/white)

double match_pixels(pixel pix1, pixel pix2)                                //  v.28
{
   double   red1, green1, blue1, red2, green2, blue2;
   double   reddiff, greendiff, bluediff, match;
   
   red1 = pix1[0];
   green1 = pix1[1];
   blue1 = pix1[2];
   
   red2 = brightfix0[pix2[0]];
   green2 = brightfix1[pix2[1]];
   blue2 = brightfix2[pix2[2]];
   
   reddiff = 0.00392 * fabs(red1-red2);                                    //  0 = perfect match
   greendiff = 0.00392 * fabs(green1-green2);                              //  1 = total mismatch
   bluediff = 0.00392 * fabs(blue1-blue2);
   
   match = (1.0 - reddiff) * (1.0 - greendiff) * (1.0 - bluediff);
   return match;
}


/**************************************************************************/

//  unbend a pano image - straighten curvature added by pano

void   * unbend_thread(void *);
int      unbend_dialog_event(zdialog* zd, const char *event);
int      unbend_dialog_compl(zdialog* zd, int zstat);

int      unbend_stat;
int      unbend_horz, unbend_vert;                                         //  unbend values from dialog

void m_unbend()
{
   if (! pxb3) return;                                                     //  nothing to unbend
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(pxb1);
   pixbuf_poop(1)

   pthread_create(&tid,0,unbend_thread,null);                              //  thread runs on from here
   return;
}


//  unbend thread function                                                 //  adjustable axes   v.39

void * unbend_thread(void *)   
{
   const char     *close;
   int            mx1, my1, mx2, my2;
   int            vert, horz, dist1, dist2;
   int            hx1, hy1, hx2, hy2, vx1, vy1, vx2, vy2;
   int            px3, py3, cx3, cy3;
   double         px1, py1, dispx, dispx2, dispy;
   pixel          pix1, pix3;
   zdialog        *zd;

   ++thread_busy;

   zd = zdialog_new(xtext("unbend panorama image"),Bdone,Bcancel,null);    //  create unbend dialog
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=10");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|space=10");
   zdialog_add_widget(zd,"spin","spvert","vb1","0|30|1|0");
   zdialog_add_widget(zd,"spin","sphorz","vb1","-20|20|1|0");
   zdialog_add_widget(zd,"button","apply","vb1",Bapply);
   zdialog_add_widget(zd,"label","labvert","vb2",xtext("vertical unbend"));
   zdialog_add_widget(zd,"label","labhorz","vb2",xtext("horizontal unbend"));
   zdialog_add_widget(zd,"label","space","vb2",0);
   
   zdialog_resize(zd,260,0);
   zdialog_run(zd,unbend_dialog_event,unbend_dialog_compl);                //  run dialog, parallel

   hx1 = 0;                                                                //  initial horz. axis
   hy1 = hh3 / 2;
   hx2 = ww3;
   hy2 = hh3 / 2;
   
   vx1 = ww3 / 2;                                                          //  initial vert. axis
   vy1 = 0;
   vx2 = ww3 / 2;
   vy2 = hh3;
   
   LMcapture = 1;                                                          //  capture left mouse clicks
   Fredraw++;
   unbend_stat = 0;

   while (true)
   {
      zsleep(0.1);

      if (LMclick)                                                         //  mouse click
      {
         LMclick = 0;
         mwpaint2(0);                                                      //  refresh window (sets Fredraw when done)
         
         if (mpMx < 0.2 * wwM || mpMx > 0.8 * wwM) {                       //  check reasonable mouse position
            if (mpMy < 0.1 * hhM || mpMy > 0.9 * hhM) continue;
         }
         else if (mpMy < 0.2 * hhM || mpMy > 0.8 * hhM) {                  //  limits expanded    v.42
            if (mpMx < 0.1 * wwM || mpMx > 0.9 * wwM) continue;
         }
         else continue;

         close = "?";                                                      //  find the closest axis end-point
         dist1 = ww3 * hh3;

         dist2 = (mp3x-hx1)*(mp3x-hx1) + (mp3y-hy1)*(mp3y-hy1);
         if (dist2 < dist1) {
            dist1 = dist2;
            close = "left";
         }

         dist2 = (mp3x-hx2)*(mp3x-hx2) + (mp3y-hy2)*(mp3y-hy2);
         if (dist2 < dist1) {
            dist1 = dist2;
            close = "right";
         }

         dist2 = (mp3x-vx1)*(mp3x-vx1) + (mp3y-vy1)*(mp3y-vy1);
         if (dist2 < dist1) {
            dist1 = dist2;
            close = "top";
         }

         dist2 = (mp3x-vx2)*(mp3x-vx2) + (mp3y-vy2)*(mp3y-vy2);
         if (dist2 < dist1) {
            dist1 = dist2;
            close = "bottom";
         }
         
         if (strEqu(close,"left")) hy1 = mp3y;                             //  set new axis end-point
         if (strEqu(close,"right")) hy2 = mp3y;
         if (strEqu(close,"top")) vx1 = mp3x;
         if (strEqu(close,"bottom")) vx2 = mp3x;
      }

      if (Fredraw)                                                         //  Fredraw logic revised    v.42
      {
         Fredraw = 0;

         zlock();
         mx1 = int((hx1 - org3x) * Rscale + posMx + 0.5);                  //  draw new axes dotted lines
         my1 = int((hy1 - org3y) * Rscale + posMy + 0.5);
         mx2 = int((hx2 - org3x) * Rscale + posMx + 0.5);
         my2 = int((hy2 - org3y) * Rscale + posMy + 0.5);
         gdk_draw_line(drmwin->window,gdkgc,mx1,my1,mx2,my2);

         mx1 = int((vx1 - org3x) * Rscale + posMx + 0.5);
         my1 = int((vy1 - org3y) * Rscale + posMy + 0.5);
         mx2 = int((vx2 - org3x) * Rscale + posMx + 0.5);
         my2 = int((vy2 - org3y) * Rscale + posMy + 0.5);
         gdk_draw_line(drmwin->window,gdkgc,mx1,my1,mx2,my2);
         zunlock();
      }
      
      if (unbend_stat == 0) continue;                                      //  no new user inputs
      if (unbend_stat != 1) break;                                         //  done or cancel
      unbend_stat = 0;

      vert = int(unbend_vert * 0.01 * hh3);                                //  convert % to pixels
      horz = int(unbend_horz * 0.005 * ww3);
      
      for (py3 = 0; py3 < hh3; py3++)                                      //  step through pxb3 pixels
      for (px3 = 0; px3 < ww3; px3++)
      {
         pix3 = ppix3 + py3 * rs3 + px3 * nch;                             //  output pixel

         cx3 = vx1 + (vx2 - vx1) * py3 / hh3;                              //  center of unbend
         cy3 = hy1 + (hy2 - hy1) * px3 / ww3;
         dispx = 2.0 * (px3 - cx3) / ww3;                                  //  -1.0 ..  0.0 .. +1.0 (roughly)
         dispy = 2.0 * (py3 - cy3) / hh3;                                  //  -1.0 ..  0.0 .. +1.0
         dispx2 = dispx * dispx - 0.5;                                     //  +0.5 .. -0.5 .. +0.5  curved

         px1 = px3 + dispx * dispy * horz;                                 //  input virtual pixel, x
         py1 = py3 - dispy * dispx2 * vert;                                //  input virtual pixel, y
         pix1 = vpixel(ppix1,px1,py1,ww1,hh1,rs1);                         //  input virtual pixel

         if (pix1) {
            pix3[0] = pix1[0];                                             //  input pixel >> output pixel
            pix3[1] = pix1[1];
            pix3[2] = pix1[2];
         }
         else pix3[0] = pix3[1] = pix3[2] = 0;
      }
      
      modified++;
      mwpaint2(1);                                                         //  update window
   }

   LMcapture = 0;
   if (unbend_stat == 3) pull_image3();                                    //  cancelled, restore image3
   --thread_busy;
   return 0;
}


//  callback functions for unbend dialog event and completion

int unbend_dialog_event(zdialog *zd, const char *event)
{
   if (strNeq(event,"apply")) return 0;                                    //  wait for apply button
   zdialog_fetch(zd,"spvert",unbend_vert);                                 //  get new unbend values
   zdialog_fetch(zd,"sphorz",unbend_horz);
   unbend_stat = 1;                                                        //  signal new values available
   return 1;
}

int unbend_dialog_compl(zdialog *zd, int zstat)
{
   if (zstat == 1) unbend_stat = 2;                                        //  done
   else unbend_stat = 3;                                                   //  cancelled
   zdialog_destroy(zd);
   return 0;
}


/**************************************************************************/

//  image stretch/distort - select image area and pull with mouse          v.38

void * stretch_thread(void *);                                             //  thread function
int stretch_image(int drx, int dry, int drw, int drh, int accum);          //  stretch image function
int stretch_dialog_event(zdialog *zd, const char *event);                  //  dialog event function
int stretch_dialog_compl(zdialog *zd, int zstat);                          //  dialog completion function

int         stretch_stat;
float       *stretchx, *stretchy;                                          //  memory of all displaced pixels
int         nstrmem, strmem[4][100];                                       //  memory for last 100 stretches

void m_stretch()
{
   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(pxb1);
   pixbuf_poop(1)

   pthread_create(&tid,0,stretch_thread,null);                             //  thread runs on from here
   return;
}


//  stretch thread function

void * stretch_thread(void *)   
{
   const char     *stretch_message = xtext(
                     " Press [select area] to select an area to stretch. \n"
                     " Press [start stretch] and pull area with mouse. \n"
                     " Make multiple mouse pulls until satisfied. \n"
                     " When finished, select another area or press [done]."); 

   GdkCursor      *drag_cursor = 0;
   int            px, py, ii, stretched;
   zdialog        *zd;
   
   ++thread_busy;

   stretchx = (float *) zmalloc(ww3 * hh3 * sizeof(float));                //  get memory for pixel displacements
   stretchy = (float *) zmalloc(ww3 * hh3 * sizeof(float));
   
   for (px = 0; px < ww3; px++)                                            //  set all pixel displacements to zero
   for (py = 0; py < hh3; py++)
   {
      ii = py * ww3 + px;
      stretchx[ii] = stretchy[ii] = 0.0;
   }

   zd = zdialog_new(xtext("select area to stretch"),Bdone,Bcancel,null);
   zdialog_add_widget(zd,"label","lab1","dialog",stretch_message,"space=5");              //  [ select area ]
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");                             //  [ start stretch ]
   zdialog_add_widget(zd,"button","selarea","hb1",Bselarea,"space=5");                    //  [ undo stretch ]
   zdialog_add_widget(zd,"button","sstretch","hb1",xtext("start stretch"),"space=5");
   zdialog_add_widget(zd,"button","ustretch","hb1",xtext("undo stretch"),"space=5");
   zdialog_run(zd,stretch_dialog_event,stretch_dialog_compl);

   sablend = nsapg = sastat = Nsapix = 0;                                  //  no select area
   stretched = nstrmem = stretch_stat = 0;                                 //  no stretch data
   
   while (true)
   {
      zsleep(0.1);

      if (stretch_stat == 1)                                               //  initiate select area
      {
         stretch_stat = 0;                                                 //  reset stretch data
         stretched = 0;                                                    
         nstrmem = 0;

         if (sastat == 0 || sastat == 3) {                                 //  if no select area underway,
            sastat = 1;                                                    //    start new select area
            pthread_create(&tid,0,select_area_thread,null);                //  arbitrary polygon via mouse   v.41
         }
      }
      
      while (stretch_stat == 2)                                            //  start stretch
      {
         zsleep(0.1);

         if (sastat == 1) {                                                //  if select area thread active,
            sastat = 2;                                                    //    tell it to finish now
            while (sastat == 2) zsleep(0.1);
         }

         if (sastat != 3) {                                                //  no valid select area
            stretch_stat = 0;
            break;
         }

         if (! drag_cursor) {
            zlock();
            drag_cursor = gdk_cursor_new(GDK_CROSSHAIR);                   //  set drag cursor
            gdk_window_set_cursor(drmwin->window,drag_cursor);
            zunlock();
         }

         if (mdx1 || mdy1)                                                 //  mouse drag underway
         {
            dragx = int(mdx1 / Rscale + org3x);                            //  drag origin, image3 coordinates
            dragy = int(mdy1 / Rscale + org3y);
            dragw = int((mdx2 - mdx1) / Rscale);                           //  drag increment
            dragh = int((mdy2 - mdy1) / Rscale);
            stretch_image(dragx,dragy,dragw,dragh,0);                      //  stretch image
            stretched = 1;
         }

         else if (stretched)                                               //  drag done, stretch done
         {
            stretched = 0;
            stretch_image(dragx,dragy,dragw,dragh,1);                      //  accumulate sum of all stretches

            if (nstrmem == 100) nstrmem = 99;
            for (ii = nstrmem; ii > 0; ii--)                               //  if full, throw away oldest
            {
               strmem[0][ii] = strmem[0][ii-1];
               strmem[1][ii] = strmem[1][ii-1];
               strmem[2][ii] = strmem[2][ii-1];
               strmem[3][ii] = strmem[3][ii-1];
            }
            strmem[0][0] = dragx;                                          //  save latest stretch
            strmem[1][0] = dragy;
            strmem[2][0] = dragw;
            strmem[3][0] = dragh;
            nstrmem++;
         }
      }

      if (drag_cursor) {
         zlock();
         gdk_window_set_cursor(drmwin->window,0);                          //  restore normal cursor
         gdk_cursor_unref(drag_cursor);
         zunlock();
         drag_cursor = 0;
      }
      
      if (stretch_stat == 3)                                               //  undo stretch
      {
         if (nstrmem)
         {                                                                 //  recover most recent stretch
            dragx = strmem[0][0];
            dragy = strmem[1][0];
            dragw = strmem[2][0];
            dragh = strmem[3][0];
            stretch_image(dragx,dragy,-dragw,-dragh,0);                    //  undo last stretch
            stretch_image(dragx,dragy,-dragw,-dragh,1);

            nstrmem--;
            for (ii = 0; ii < nstrmem; ii++)                               //  pull from undo memory
            {
               strmem[0][ii] = strmem[0][ii+1];
               strmem[1][ii] = strmem[1][ii+1];
               strmem[2][ii] = strmem[2][ii+1];
               strmem[3][ii] = strmem[3][ii+1];
            }
         }
         stretch_stat = 2;                                                 //  continue stretch mode
      }

      if (stretch_stat == 4)                                               //  dialog done
      {
         mwpaint2(0);
         break;
      }

      if (stretch_stat == 5) {
         pull_image3();                                                    //  dialog canceled, undo changes
         break;
      }
   }

   zfree(stretchx);                                                        //  release memory
   zfree(stretchy);

   if (sapix) zfree(sapix);                                                //  free memory for select area pixels
   sapix = 0;

   --thread_busy;
   return 0;
}


//  stretch dialog event and completion callback functions

int stretch_dialog_event(zdialog * zd, const char *event)
{
   if (strEqu(event,"selarea")) stretch_stat = 1;                          //  select area
   if (strEqu(event,"sstretch")) stretch_stat = 2;                         //  start stretch
   if (strEqu(event,"ustretch")) stretch_stat = 3;                         //  undo last stretch
   return 1;
}

int stretch_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat == 1) stretch_stat = 4;                                       //  dialog done
   else stretch_stat = 5;                                                  //  dialog canceled
   zdialog_destroy(zd);
   return 0;
}


//  stretch image within select area according to mouse position and displacement

int stretch_image(int drx, int dry, int drw, int drh, int accum)
{
   int            ii, jj, px, py;
   double         ddx, ddy, dpe, dpm, mag, dispx, dispy;
   pixel          pix1, pix3;
   
   for (ii = 0; ii < Nsapix; ii++)                                         //  process all enclosed pixels
   {
      px = sapix[ii].px;
      py = sapix[ii].py;
      dpe = sapix[ii].dist;                                                //  distance from area edge

      ddx = (px - drx);                                                    //  distance from drag origin
      ddy = (py - dry);
      dpm = sqrt(ddx*ddx + ddy*ddy);

      if (dpm < 1) mag = 1;
      else mag = dpe / (dpe + dpm);                                        //  magnification, 0...1
      mag = mag * mag;

      dispx = -drw * mag;                                                  //  stretch = drag * magnification
      dispy = -drh * mag;
      
      jj = py * ww3 + px;

      if (accum) {                                                         //  mouse drag done, accumulate stretch
         stretchx[jj] += dispx;
         stretchy[jj] += dispy;
         continue;
      }

      dispx += stretchx[jj];                                               //  add this stretch to prior
      dispy += stretchy[jj];

      pix1 = vpixel(ppix1,px+dispx,py+dispy,ww1,hh1,rs1);                  //  input virtual pixel
      if (! pix1) continue;
      pix3 = ppix3 + py * rs3 + px * nch;                                  //  output pixel
      pix3[0] = pix1[0];
      pix3[1] = pix1[1];
      pix3[2] = pix1[2];
   }

   modified++;
   mwpaint2(1);
   return 1;
}


/**************************************************************************/

//  image sharpen                                                          v.38

void * sharpen_thread(void *);
int    sharpen_image();
int    sharpen_dialog_event(zdialog *zd, const char *event);
int    sharpen_dialog_compl(zdialog *zd, int zstat);

int      sharpen_stat;
int      sharpen_grad;
int      sharpen_thresh1 = 100;
double   sharpen_thresh2 = 0.85;


void m_sharpen()
{
   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(pxb1);
   pixbuf_poop(1)

   pthread_create(&tid,0,sharpen_thread,null);                             //  thread runs on from here
   return;
}


//  sharpen image thread function

void * sharpen_thread(void *)   
{
   zdialog     *zd;

   ++thread_busy;

   zd = zdialog_new(xtext("sharpen image"),Bdone,Bcancel,null);            //  dialog for user input
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zd,"label","labsharp","hb1",xtext("sharpen"),"space=5");
   zdialog_add_widget(zd,"spin","sharpen","hb1","0|20|1|0","space=5");
   zdialog_run(zd,sharpen_dialog_event,sharpen_dialog_compl);
   
   sharpen_grad = 0;
   sharpen_stat = 0;

   while (true)
   {
      while (sharpen_stat == 0) zsleep(0.1);

      if (sharpen_stat == 1) {                                             //  sharpen
         sharpen_stat = 0;
         sharpen_image();
      }      

      if (sharpen_stat == 8) break;                                        //  done

      if (sharpen_stat == 9) {                                             //  cancel
         pull_image3();                                                    //  undo changes
         break;
      }
   }

   --thread_busy;
   mwpaint2(1);
   return 0;
}


//  sharpen dialog event and completion callback functions

int sharpen_dialog_event(zdialog * zd, const char *event)
{
   if (strEqu(event,"sharpen")) {
      zdialog_fetch(zd,"sharpen",sharpen_grad);
      sharpen_stat = 1;
   }

   return 1;
}

int sharpen_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat == 1) sharpen_stat = 8;                                       //  dialog done
   else sharpen_stat = 9;                                                  //  dialog canceled
   zdialog_destroy(zd);
   return 0;
}


//  image sharpen function                                                 //  improved   v.39

int sharpen_image()
{
   int      px, py, dd, rgb, grad, thresh;
   int      dx[4] = { -1, 0, 1, 1 };                                       //  4 directions: NW N NE E
   int      dy[4] = { -1, -1, -1, 0 };
   int      vT, v1, v2;
   pixel    pixT, pix1, pix2;
   
   computing++;
   update_status_bar();

   prior_image3();                                                         //  refresh image3
   
   thresh = sharpen_thresh1;
   
   for (grad = 0; grad < sharpen_grad; grad++)
   {
      if (grad > 0) 
         thresh = int(thresh * sharpen_thresh2);

      for (px = 1; px < ww3-1; px++)                                       //  loop all image3 pixels
      for (py = 1; py < hh3-1; py++)
      {
         pixT = ppix3 + py * rs3 + px * nch;                               //  target pixel in image3
         
         for (dd = 0; dd < 4; dd++)                                        //  4 directions   *** bugfix v.39b
         {
            pix1 = pixT + dy[dd] * rs3 + dx[dd] * nch;                     //  upstream pixel
            pix2 = pixT - dy[dd] * rs3 - dx[dd] * nch;                     //  downstream pixel

            for (rgb = 0; rgb < 3; rgb++)                                  //  loop 3 RGB colors
            {
               v1 = pix1[rgb];
               vT = pixT[rgb];
               v2 = pix2[rgb];
               
               if (v2 - v1 > thresh && v1 < vT && vT < v2) {
                  if (v1 > 0) v1--;
                  if (v2 < 255) v2++;
               }
               else if (v1 - v2 > thresh && v2 < vT && vT < v1) {
                  if (v2 > 0) v2--;
                  if (v1 < 255) v1++;
               }

               else continue;

               pix1[rgb] = v1;                                             //  modified brightness values
               pix2[rgb] = v2;                                             //    >> image3 pixel
            }
         }
      }
   }
   
   computing = 0;
   update_status_bar();

   modified++;
   mwpaint2(1);
   return 1;
}


/**************************************************************************/

//  image noise reduction                                                  v.38

void * denoise_thread(void *);                                             //  thread function
int denoise_image();                                                       //  denoise function
int denoise_dialog_event(zdialog *zd, const char *event);                  //  dialog event function
int denoise_dialog_compl(zdialog *zd, int zstat);                          //  dialog completion function

int      denoise_stat;


void m_denoise()
{
   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(pxb1);
   pixbuf_poop(1)

   pthread_create(&tid,0,denoise_thread,null);                             //  thread runs on from here
   return;
}


//  image noise reduction thread function

void * denoise_thread(void *)   
{
   const char  *denoise_message = xtext(" Press the reduce button to \n"
                                        " reduce noise in small steps. \n"
                                        " Use reset to start over.");
   zdialog     *zd;

   ++thread_busy;

   zd = zdialog_new(xtext("noise reduction"),Bdone,Bcancel,null);          //  dialog for user inputs
   zdialog_add_widget(zd,"label","lab1","dialog",denoise_message,"space=5");
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zd,"button","reduce","hb1",xtext("reduce"),"space=5");
   zdialog_add_widget(zd,"button","reset","hb1",Breset,"space=5");
   zdialog_run(zd,denoise_dialog_event,denoise_dialog_compl);
   
   while (true)
   {
      denoise_stat = 0;
      while (denoise_stat == 0) zsleep(0.1);

      if (denoise_stat == 1) {                                             //  denoise
         denoise_image();
      }
      
      if (denoise_stat == 2) {                                             //  reset
         prior_image3();                                                   //  restore prior image3
         mwpaint2(1);
      }

      if (denoise_stat == 8) break;                                        //  done

      if (denoise_stat == 9) {                                             //  cancel
         pull_image3();                                                    //  undo changes
         break;
      }
   }

   --thread_busy;
   mwpaint2(1);
   return 0;
}


//  denoise dialog event and completion callback functions

int denoise_dialog_event(zdialog * zd, const char *event)
{
   if (strEqu(event,"reduce")) denoise_stat = 1;
   if (strEqu(event,"reset")) denoise_stat = 2;
   return 1;
}

int denoise_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat == 1) denoise_stat = 8;                                       //  dialog done
   else denoise_stat = 9;                                                  //  dialog canceled
   zdialog_destroy(zd);
   return 0;
}


//  image noise reduction function

int denoise_image()                                                        //  v.39 improved
{
   int         px, py, iix, iiy;
   int         min0, min1, min2, max0, max1, max2;
   pixel       pix1;

   for (px = 1; px < ww3-1; px++)                                          //  loop all image3 pixels
   for (py = 1; py < hh3-1; py++)
   {
      min0 = min1 = min2 = 255;
      max0 = max1 = max2 = 0;

      for (iix = -1; iix <= 1; iix++)                                      //  loop 8 surrounding pixels
      for (iiy = -1; iiy <= 1; iiy++)
      {
         if (iix == 0 && iiy == 0) continue;                               //  skip self

         pix1 = ppix3 + (py + iiy) * rs3 + (px + iix) * nch;
         if (pix1[0] < min0) min0 = pix1[0];                               //  save min and max per color
         if (pix1[0] > max0) max0 = pix1[0];
         if (pix1[1] < min1) min1 = pix1[1];
         if (pix1[1] > max1) max1 = pix1[1];
         if (pix1[2] < min2) min2 = pix1[2];
         if (pix1[2] > max2) max2 = pix1[2];
      }

      pix1 = ppix3 + py * rs3 + px * nch;                                  //  target pixel in image3
      
      if (pix1[0] <= min0 && min0 < 255) pix1[0] = min0 + 1;               //  if outlier, flatten
      if (pix1[0] >= max0 && max0 > 0) pix1[0] = max0 - 1;
      if (pix1[1] <= min1 && min1 < 255) pix1[1] = min1 + 1;
      if (pix1[1] >= max1 && max1 > 0) pix1[1] = max1 - 1;
      if (pix1[2] <= min2 && min2 < 255) pix1[2] = min2 + 1;
      if (pix1[2] >= max2 && max2 > 0) pix1[2] = max2 - 1;
   }

   modified++;
   mwpaint2(1);
   return 1;
}


/**************************************************************************/

//  image color-depth reduction                                            v.38

void * colors_thread(void *);
int    colors_image();
int    colors_dialog_event(zdialog *zd, const char *event);
int    colors_dialog_compl(zdialog *zd, int zstat);

int      colors_stat;
int      color_depth;


void m_colors()
{
   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(pxb1);
   pixbuf_poop(1)

   pthread_create(&tid,0,colors_thread,null);                              //  thread runs on from here
   return;
}


//  image colors reduction thread function

void * colors_thread(void *)   
{
   const char  *colors_message = xtext(" Set color depth to 1-8 bits per color, \n"
                                       " then press apply to update the image.");
   zdialog     *zd;

   ++thread_busy;

   zd = zdialog_new(xtext("image color depth"),Bdone,Bcancel,null);        //  dialog for user inputs
   zdialog_add_widget(zd,"label","lab1","dialog",colors_message,"space=5");
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zd,"spin","colors","hb1","1|8|1|8","space=5");
   zdialog_add_widget(zd,"button","apply","hb1","apply","space=5");
   zdialog_run(zd,colors_dialog_event,colors_dialog_compl);
   
   colors_stat = 0;
   color_depth = 8;

   while (true)
   {
      while (colors_stat == 0) zsleep(0.1);

      if (colors_stat == 1) {                                              //  colors
         colors_image();
         colors_stat = 0;
      }
      
      if (colors_stat == 3) break;                                         //  done

      if (colors_stat == 4) {                                              //  cancel
         pull_image3();                                                    //  undo changes
         break;
      }
   }

   --thread_busy;
   mwpaint2(1);
   return 0;
}


//  colors dialog event and completion callback functions

int colors_dialog_event(zdialog * zd, const char *event)
{
   if (strEqu(event,"apply")) {
      zdialog_fetch(zd,"colors",color_depth);
      colors_stat = 1;
   }
   return 1;
}

int colors_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat == 1) colors_stat = 3;                                        //  dialog done
   else colors_stat = 4;                                                   //  dialog canceled
   zdialog_destroy(zd);
   return 0;
}


//  image color depth function

int colors_image()
{
   int         px, py, rgb;
   int         mask[8] = { 128, 192, 224, 240, 248, 252, 254, 255 };
   int         mask1, mask2, val;
   pixel       pix1, pix3;
   
   mask1 = mask[color_depth - 1];
   mask2 = 128 >> color_depth;

   for (px = 0; px < ww3; px++)                                            //  loop all image3 pixels
   for (py = 0; py < hh3; py++)
   {
      pix1 = ppix1 + py * rs1 + px * nch;                                  //  source pixel in image1
      pix3 = ppix3 + py * rs3 + px * nch;                                  //  target pixel in image3
      
      for (rgb = 0; rgb < 3; rgb++)
      {
         val = pix1[rgb];
         val = val + mask2;
         if (val > 255) val = 255;
         val = val & mask1;
         pix3[rgb] = val;
      }
   }

   modified++;
   mwpaint2(1);
   return 1;
}


/**************************************************************************/

//  functions to push image3 into undo stack and pull from stack           v.15

GdkPixbuf   *undostack[undomax];                                           //  undo / redo stack of images
int         nust, uftf = 1;

void push_image3()                                                         //  push image3 into stack
{
   if (uftf) {
      uftf = nust = 0;                                                     //  first call, clear undo stack
      for (int ii = 0; ii < undomax; ii++) undostack[ii] = 0;
   }

   if (! pxb3) return;
   zlock();
   if (nust == undomax) {
      pixbuf_free(undostack[0]);                                           //  lose oldest stack entry
      for (int ii = 1; ii < undomax; ii++)
            undostack[ii-1] = undostack[ii];                               //  push other entries down
      nust--;
      undostack[nust] = 0;                                                 //  last entry vacated
   }
   pixbuf_free(undostack[nust]);                                           //  free entry to be overwritten
   undostack[nust] = pixbuf_copy(pxb3);                                    //  image3 >> stack
   pixbuf_test(undostack[nust]);
   nust++;
   zunlock();   
   return;
}

void pull_image3()                                                         //  pull image3 from stack (undo)
{
   GdkPixbuf   *temp;

   if (! pxb3) return;
   if (nust == 0) return;
   zlock();
   temp = undostack[nust-1];                                               //  image3 <==> last stack entry    v.26
   undostack[nust-1] = pxb3;                                               //  fix memory leak   v.26
   pxb3 = temp;
   pixbuf_poop(3);
   nust--;                                                                 //  stack count down
   if (nust == 0) modified = 0;
   zunlock();
   mwpaint2(1);
   return;
}

void prior_image3()                                                        //  get last image3 from stack
{                                                                          //    without removal from stack
   if (! pxb3) return;
   if (nust == 0) {
      modified = 0;
      return;
   }
   zlock();
   pixbuf_free(pxb3);
   pxb3 = pixbuf_copy(undostack[nust-1]);
   pixbuf_test(pxb3);
   pixbuf_poop(3);
   zunlock();
   return;
}

void redo_image3()                                                         //  get last+1 stack entry (redo)
{                                                                          //    and advance stack depth
   GdkPixbuf   *temp;

   if (! pxb3) return;
   if (nust == undomax) return;
   if (! undostack[nust]) return;                                          //  nothing is there
   zlock();
   temp = undostack[nust];                                                 //  image3 <==> last+1 stack entry
   undostack[nust] = pxb3;                                                 //  fix memory leak   v.26
   pxb3 = temp;
   pixbuf_poop(3);
   nust++;                                                                 //  stack count up
   modified++;
   zunlock();
   mwpaint2(1);
   return;
}

void clearmem_image3()                                                     //  clear stack (start new image)
{
   zlock();
   for (int ii = 0; ii < undomax; ii++) 
         pixbuf_free(undostack[ii]);
   nust = 0;
   modified = 0;
   zunlock();
   return;
}


/**************************************************************************/

//  track pixbufs allocated and freed for memory leak detection            v.27

int      pixbuf_count = 0;

void incr_pixbufs()
{
   ++pixbuf_count;
   if (debug) printf("pixbufs: %d \n",pixbuf_count);
   return;
}

void pixbuf_free(GdkPixbuf *&pixbuf)
{
   if (! pixbuf) return;
   zlock();
   g_object_unref(pixbuf);
   pixbuf = 0;
   zunlock();
   --pixbuf_count;
   if (debug) printf("pixbufs: %d \n",pixbuf_count);
   return;
}


//  validate an image file and load pixbuf from file        v.33
//  if an alpha channel is present, remove it

GdkPixbuf * load_pixbuf(const char *file)
{
   GdkPixbuf   *pxb91 = 0, *pxb92 = 0;
   int         ww91, hh91, rs91, ww92, hh92, rs92;
   int         px, py, alfa;
   pixel       ppix91, pix91, ppix92, pix92;

   pxb91 = pixbuf_new_from_file(file,gerror);                              //  validate file and load pixbuf
   if (! pxb91) return 0;

   nch = gdk_pixbuf_get_n_channels(pxb91);
   nbits = gdk_pixbuf_get_bits_per_sample(pxb91);
   alfa = gdk_pixbuf_get_has_alpha(pxb91);

   if (nch < 3 || nbits != 8) {                                            //  must be 3 or 4 channels
      pixbuf_free(pxb91);                                                  //  and 8 bits per channel
      return 0;
   }
   
   if (! alfa) return pxb91;                                               //  no alpha channel, 3 channels

   pixbuf_poop(91);
   pxb92 = pixbuf_new(colorspace,0,8,ww91,hh91);
   pixbuf_test(pxb92);
   pixbuf_poop(92);

   for (py = 0; py < hh91; py++)                                           //  copy without 4th channel (alpha)
   for (px = 0; px < ww91; px++)
   {
      pix91 = ppix91 + rs91 * py + 4 * px;
      pix92 = ppix92 + rs92 * py + 3 * px;
      pix92[0] = pix91[0];
      pix92[1] = pix91[1];
      pix92[2] = pix91[2];
   }

   pixbuf_free(pxb91);
   nch = 3;                                                                //  set channels to 3 
   return pxb92;
}


/**************************************************************************/

//  show EXIF data if available                                            v.31

int exif_show(char *file)
{
   int      err;
   char     command[1000];

   snprintf(command,999,"exiv2 pr \"%s\" 2>&1",file);                      //  command to print exif data
   err = popup_command(command,500,400);
   return err;
}


//  fetch EXIF data from image file

int exif_fetch(char *file)
{
   char        command[1000];
   int         err;

   snprintf(command,999,"exiv2 -f ex \"%s\" 2>&1",file);
   err = system(command);
   return err;
}


//  stuff EXIF data from original image file into new image file
//  (exif_fetch() for file1 must be done beforehand)

int exif_stuff(char *file1, char *file2, int xx, int yy)
{
   void  modsuff(char *ofile, cchar *ifile, cchar *suff);                  //  private function

   char     efile1[1000], efile2[1000];
   char     command[2000];
   int      err = 0;

   modsuff(efile1,file1,".exv");                                           //  get .exv file names from image files
   modsuff(efile2,file2,".exv");

   if (strNeq(efile1,efile2)) {                                            //  if images files are not the same,
      snprintf(command,1999,"mv -f \"%s\" \"%s\" ",efile1,efile2);         //    move and rename the .exv file
      err = system(command);
      if (err) return err;
   }
   
   snprintf(command,999,"exiv2 in \"%s\" ",file2);                         //  load EXIF data into image file
   err = system(command);
   snprintf(command,999,"rm -f \"%s\" ",efile2);                           //  delete EXIF data file
   system(command);
   if (err) return err;

   snprintf(command,999,"exiv2 -M\"set Exif.Photo.PixelXDimension %d\" "   //  modify EXIF data in image file
                             " -M\"set Exif.Photo.PixelYDimension %d\" "   //  (new pixel dimensions)
                             " mo \"%s\" ", xx, yy, file2);
   err = system(command);
   return err;
}

void modsuff(char *outfile, cchar *infile, cchar *suff)                    //  modify file suffix
{
   char     *pp;

   if (strlen(infile) > 990) zappcrash("file name overflow");
   strcpy(outfile,infile);
   pp = strrchr(outfile,'/');
   if (! pp) pp = outfile;
   pp = strrchr(pp,'.');
   if (pp) *pp = 0;
   strcat(outfile,suff);
   return;
}


/**************************************************************************/

//  thread function - select an image area for editing
//  outline an arbitrary polygon using the mouse

void * select_area_thread(void *)
{
   int    sel_area_inside(int px, int py);                                 //  is point inside select area
   int    sel_area_distance(int px, int py);                               //  get distance to nearest polygon edge

   GdkCursor      *drag_cursor = 0;
   int            ii, mdown, x1, y1, x2, y2;
   int            minpgx, minpgy, maxpgx, maxpgy;
   int            inside, px, py, dist, nseg;
   static int     ftf = 1;

   ++thread_busy;

   if (ftf) zmessageACK(xtext("Use mouse drag and click to enclose an area. \n "
                              "Use right click to undo the last line segment"));
   ftf = 0;

   zlock();
   drag_cursor = gdk_cursor_new(GDK_DOTBOX);                               //  set drawing cursor
   gdk_window_set_cursor(drmwin->window,drag_cursor);
   zunlock();

   LMcapture++;                                                            //  capture mouse clicks
   mdown = 0;
   nsapg = 0;
   sastat = 1;

   while (sastat == 1)                                                     //  loop while select in progress
   {
      zsleep(0.1);                                                         //  mouse sample interval, 0.1 secs.

      if (mp3x < 0 || mp3x > ww3-1 || mp3y < 0 || mp3y > hh3-1) {          //  ignore if outside image area   v.42
         Fredraw = 0;
         continue;
      }

      if (LMdown)                                                          //  mouse down
      {
         mdown = 1;
         Fredraw++;

         if (nsapg == 0) {                                                 //  set 1st point if none
            sapgx[0] = mp3x;
            sapgy[0] = mp3y;
            nsapg = 1;
         }
      }

      if (! LMdown && mdown)                                               //  mouse up after down
      {
         mdown = 0;

         if (nsapg > maxsapg-2) {
            zmessageACK("too many points");
            continue;
         }

         if (nsapg > 0) {
            ii = nsapg - 1;                                                //  eliminate coincident point
            if (mp3x == sapgx[ii] && mp3y == sapgy[ii]) continue;
         }
         
         sapgx[nsapg] = mp3x;                                              //  new point = current mouse position
         sapgy[nsapg] = mp3y;
         nsapg++;
      }

      if (LMclick) {                                                       //  left mouse click
         LMclick = 0;
         mdown = 0;
      }

      if (RMclick) {                                                       //  right mouse click
         RMclick = 0;
         if (nsapg > 0) nsapg--;                                           //  undo last point
         Fredraw++;
         mdown = 0;
      }
      
      if (Fredraw) {
         Fredraw = 0;
         mwpaint2(0);                                                      //  refresh window (sets Fredraw when done)
      }

      if (Fredraw)                                                         //  Fredraw logic revised    v.42
      {
         Fredraw = 0;

         if (LMdown) {
            sapgx[nsapg] = mp3x;                                           //  end at current mouse position
            sapgy[nsapg] = mp3y;
            nseg = nsapg;
         }
         else nseg = nsapg - 1;                                            //  only if left mouse down    v.42

         for (ii = 0; ii < nseg; ii++)
         {                                                                 //  draw all lines
            x1 = int(Rscale * (sapgx[ii] - org3x)) + posMx;
            y1 = int(Rscale * (sapgy[ii] - org3y)) + posMy;
            x2 = int(Rscale * (sapgx[ii+1] - org3x)) + posMx;
            y2 = int(Rscale * (sapgy[ii+1] - org3y)) + posMy;

            zlock();
            gdk_draw_line(drmwin->window,gdkgc,x1,y1,x2,y2);
            zunlock();
         }
      }
   }

   zlock();
   gdk_window_set_cursor(drmwin->window,0);                                //  restore normal cursor
   gdk_cursor_unref(drag_cursor);                                          //  release memory
   zunlock();

   if (sastat == 2 && nsapg > 2)                                           //  area select completed
   {
      mwpaint2(0);                                                         //  clear all lines
      sastat = 3;

      sapgx[nsapg] = sapgx[0];                                             //  last point = 1st point
      sapgy[nsapg] = sapgy[0];
      ii = nsapg - 1;                                                      //  eliminate coincident point
      if (sapgx[ii] == sapgx[nsapg] && sapgy[ii] == sapgy[nsapg]) nsapg--;

      for (ii = 0; ii < nsapg; ii++)
      {
         x1 = int(Rscale * (sapgx[ii] - org3x)) + posMx;                   //  draw all lines
         y1 = int(Rscale * (sapgy[ii] - org3y)) + posMy;
         x2 = int(Rscale * (sapgx[ii+1] - org3x)) + posMx;
         y2 = int(Rscale * (sapgy[ii+1] - org3y)) + posMy;
         
         zlock();
         gdk_draw_line(drmwin->window,gdkgc,x1,y1,x2,y2);
         zunlock();
      }

      minpgx = maxpgx = sapgx[0];
      minpgy = maxpgy = sapgy[0];

      for (ii = 0; ii < nsapg; ii++)                                       //  get rectangle enclosing polygon
      {
         if (sapgx[ii] < minpgx) minpgx = sapgx[ii];
         if (sapgx[ii] > maxpgx) maxpgx = sapgx[ii];
         if (sapgy[ii] < minpgy) minpgy = sapgy[ii];
         if (sapgy[ii] > maxpgy) maxpgy = sapgy[ii];
      }
      
      if (sapix) zfree(sapix);                                             //  allocate enough memory for 
      ii = (maxpgx - minpgx) * (maxpgy - minpgy);                          //    all pixels in rectangle
      sapix = (sapix1 *) zmalloc(ii * sizeof(sapix1));

      Nsapix = 0;

      for (px = minpgx; px < maxpgx; px++)
      for (py = minpgy; py < maxpgy; py++)                                 //  loop all pixels in rectangle
      {
         inside = sel_area_inside(px,py);                                  //  check if pixel inside polygon
         if (! inside) continue;
         dist = sel_area_distance(px,py);                                  //  distance to nearest edge
         sapix[Nsapix].px = px;
         sapix[Nsapix].py = py;                                            //  save pixel and distance
         sapix[Nsapix].dist = dist;
         Nsapix++;
      }
   }

   else                                                                    //  area select canceled
   {
      nsapg = 0;                                                           //  clear all lines
      mwpaint2(0);
      sastat = 0;
   }

   LMcapture = 0;
   --thread_busy;
   return 0;
}

int sel_area_inside(int px, int py)                                        //  determine if pixel at (px,py) is
{                                                                          //    inside the area selection polygon
   int sel_area_inside2(int px, int py, int nth);
   int      ii, inside = 0;

   for (ii = 0; ii < nsapg; ii++)                                          //  count times a vertical line up from
      if (sel_area_inside2(px,py,ii) == 2) inside = 1 - inside;            //    (px,py) intersects polygon face

   return inside;
}

int sel_area_inside2(int px, int py, int nth)                              //  determine if point (px,py) is
{                                                                          //    above or below nth polygon face
   int      x1, y1, x2, y2;
   double   yp;
   
   x1 = sapgx[nth];                                                        //  line segment of nth face
   y1 = sapgy[nth];
   x2 = sapgx[nth+1];
   y2 = sapgy[nth+1];
   
   if (x1 == x2) return 0;                                                 //  line is vertical

   if (x1 < x2) {
      if (px <= x1) return 0;                                              //  point is left or right of line
      if (px > x2) return 0;
   }
   else { 
      if (px <= x2) return 0;
      if (px > x1) return 0;
   }

   if (py <= y1 && py <= y2) return 1;                                     //  point is above or below both ends
   if (py >= y1 && py >= y2) return 2;
   
   yp = (px*(y2-y1) + (x2*y1-x1*y2))/(1.0*(x2-x1));                        //  intersect of vertical line x = px
   if (py < yp) return 1;                                                  //  above
   if (py > yp) return 2;                                                  //  below
   return 0;                                                               //  dead on
}

int sel_area_distance(int px, int py)                                      //  get distance from pixel (px,py)
{                                                                          //     to nearest polygon face
   int sel_area_distance2(int px, int py, int nth);
   int      ii, dd, mindd;
   
   mindd = sel_area_distance2(px,py,0);                                    //  distance^2 to 0th polygon face

   for (ii = 1; ii < nsapg; ii++)
   {
      dd = sel_area_distance2(px,py,ii);                                   //  distance^2 to Nth polygon face
      if (dd < mindd) mindd = dd;                                          //  remember smallest distance^2
   }
   
   mindd = int(sqrt(mindd));                                               //  return smallest distance
   return  mindd;
}

int sel_area_distance2(int px, int py, int nth)                            //  get distance^2 from point (px,py)
{                                                                          //    to nth polygon face
   int      x1, y1, x2, y2;
   int      a, b, c, d;
   int      h1, h2, ls, temp;
   int      dist2;

   x1 = sapgx[nth];                                                        //  line segment of nth face
   y1 = sapgy[nth];
   x2 = sapgx[nth+1];
   y2 = sapgy[nth+1];
   
   if (x1 < 5 && x2 < 5) return 1000000;                                   //  v.37
   if (y1 < 5 && y2 < 5) return 1000000;                                   //  if line segment hugs image edge,
   if (ww1 - x1 < 5 && ww1 - x2 < 5) return 1000000;                       //    then it is logically distant
   if (hh1 - y1 < 5 && hh1 - y2 < 5) return 1000000;
   
   a = y1-y2;                                                              //  get min. distance ^2 from point
   b = x2-x1;                                                              //    to entire line of line segment
   c = x1*y2 - y1*x2;                                                      //  (intersect may be outside segment)
   d = (a*px + b*py + c);
   dist2 = int(1.0 * d*d / (a*a + b*b));                                   //  (can overflow 32 bits)
   
   h1 = (x1-px)*(x1-px) + (y1-py)*(y1-py);                                 //  point to (x1,y1) ^2
   h2 = (x2-px)*(x2-px) + (y2-py)*(y2-py);                                 //  point to (x2,y2) ^2
   ls = (x1-x2)*(x1-x2) + (y1-y2)*(y1-y2);                                 //  segment length ^2
   if (h1 > h2) { temp = h2; h2 = h1; h1 = temp; }                         //  re-order, h1 <= h2

   if (h2 - dist2 < ls) return dist2;                                      //  intersect within segment, use dist2
   else return h1;                                                         //  use distance ^2 to nearest end point
}




