/**************************************************************************
     fotox.cpp     image processing program
     Free software licensed under GNU General Public License v.2
     Source URL: http://kornelix.squarespace.com/fotox
***************************************************************************/

#include "zlib.h"
#include <gdk/gdkkeysyms.h>

#define fversion "fotox v.32"                                              //  v.32
#define mega (1048576.0)                                                   //  1024 * 1024
#define undomax 20                                                         //  undo stack size
#define pvwinsize 600                                                      //  preview window pixbuf size

#define nodither GDK_RGB_DITHER_NONE,0,0
#define colorspace GDK_COLORSPACE_RGB
#define interp GDK_INTERP_BILINEAR
#define dottedline GDK_LINE_DOUBLE_DASH,GDK_CAP_BUTT,GDK_JOIN_MITER

#define pixbuf_new(a,b,c,d,e) gdk_pixbuf_new(a,b,c,d,e); incr_pixbufs();
#define pixbuf_scale_simple(a,b,c,d) gdk_pixbuf_scale_simple(a,b,c,d); incr_pixbufs();
#define pixbuf_new_from_file(a,b) gdk_pixbuf_new_from_file(a,b); incr_pixbufs();
#define pixbuf_copy(a) gdk_pixbuf_copy(a); incr_pixbufs();
#define pixbuf_rotate(a,b) gdk_pixbuf_rotate(a,b); incr_pixbufs();
#define pixbuf_copy_area(a,b,c,d,e,f,g,h) gdk_pixbuf_copy_area(a,b,c,d,e,f,g,h);
#define pixbuf_test(p) if (!p) zappcrash("memory allocation failure");

#define pixbuf_poop(ID)                                                    \
   ww##ID = gdk_pixbuf_get_width(pxb##ID);                                 \
   hh##ID = gdk_pixbuf_get_height(pxb##ID);                                \
   rs##ID = gdk_pixbuf_get_rowstride(pxb##ID);                             \
   ppix##ID = gdk_pixbuf_get_pixels(pxb##ID);

#define getparm(pname) parm_##pname = getParm(#pname);
#define setparm(pname,value) parm_##pname = value; setParm(#pname,parm_##pname);

extern char    JPGquality[4];                                              //  quality param for gdk_pixbuf_save()

GtkWidget      *mwin, *drmwin;                                             //  main and drawing window
GtkWidget      *pvwin = 0, *drpvwin;                                       //  preview window (tune function)
GdkGC          *gdkgc = 0;                                                 //  graphics context
GError         **gerror = 0;
GdkColor       black, white;
GdkColormap    *colormap = 0;
uint           maxcolor = 0xffff;

//  pixbufs: pxb1, pxb2 = unchanged input images
//           pxb3 = modified output image
//           pxbM = main/drawing window image
//  pxb3 (from (org3x,org3y)) * Rscale >> pxbM >> main/drawing window at (posMx,posMy)

GdkPixbuf   *pxb1 = 0, *pxb2 = 0;                                          //  input image1 and image2
GdkPixbuf   *pxb3 = 0;                                                     //  modified / output image3
GdkPixbuf   *pxb1A = 0, *pxb2A = 0;                                        //  alignment images
GdkPixbuf   *pxb1C = 0, *pxb2C = 0;                                        //  alignment images, curved
GdkPixbuf   *pxbM = 0;                                                     //  main window drawing image
GdkPixbuf   *pxbP = 0;                                                     //  preview window drawing image
GdkPixbuf   *pxbP1 = 0, *pxbP2 = 0;                                        //  preview input images (tune, HDR)

int      nch, nbits;                                                       //  pixbuf/image attributes
int      ww1, hh1, rs1;                                                    //  pxb1  width, height, rowstride
int      ww2, hh2, rs2;                                                    //  pxb2  width, height, rowstride
int      ww3, hh3, rs3;                                                    //  pxb3  width, height, rowstride
int      ww1A, hh1A, rs1A;                                                 //  pxb1A width, height, rowstride
int      ww2A, hh2A, rs2A;                                                 //  pxb2A width, height, rowstride
int      ww1C, hh1C, rs1C;                                                 //  pxb1C width, height, rowstride
int      ww2C, hh2C, rs2C;                                                 //  pxb2C width, height, rowstride
int      wwM, hhM, rsM;                                                    //  pxbM  width, height, rowstride
int      wwP, hhP, rsP;                                                    //  pxbP  width, height, rowstride
int      wwP1, hhP1, rsP1;                                                 //  pxbP1  width, height, rowstride
int      wwP2, hhP2, rsP2;                                                 //  pxbP2  width, height, rowstride

int      mwinW = 1000, mwinH = 700;                                        //  main window size (= initial size)
int      org3x = 0, org3y = 0;                                             //  image3/pxb3 origin in pxbM 
int      posMx = 0, posMy = 0;                                             //  pxbM position in main/drawing window 
int      Mrescale = 0;                                                     //  flag: refresh/rescale window
int      Mneworg = 0;                                                      //  flag: set new image3 origin
int      Mscale = 0;                                                       //  image3 scale: 0 (fit window), 1x, 2x
double   Rscale = 1;                                                       //  pxb3 to pxbM scaling factor
int      allowmag = 1;                                                     //  enable/disable Mscale

int      mpWx, mpWy;                                                       //  mouse position in main window
int      mpMx, mpMy;                                                       //  corresp. position in pxbM
int      mp3x, mp3y;                                                       //  corresp. position in pxb3
int      Lmclick = 0, Rmclick = 0;                                         //  L/R mouse button clicked
int      Lmdown = 0;                                                       //  L mouse button down (drag in progress)
int      mdx1, mdy1, mdx2, mdy2;                                           //  mouse drag displacement
int      KBkey = 0;                                                        //  keyboard key

typedef  uchar *pixel;                                                     //  3 RGB values, 0-255 each
pixel    ppix1, ppix2, ppix3;                                              //  pxb1, pxb2, pxb3  pixels
pixel    ppix1A, ppix2A, ppix1C, ppix2C;                                   //  pxb1A, pxb2A, pxb1C, pxb2C  pixels
pixel    ppixM;                                                            //  pxbM pixels
pixel    ppixP, ppixP1, ppixP2;                                            //  pxbP, pxbP1, pxbP2 pixels

char     clfile[maxfcc] = "";                                              //  command line image file
char     imagedirk[maxfcc] = "";                                           //  image file directory
char     *file1 = 0, *file2 = 0;                                           //  image1 and image2 pathnames
char     fname1[100] = "", fname3[100] = "";                               //  image1 and image3 file names
int      f1size, f3size;                                                   //  image1 and image3 file sizes

int      debug = 0;                                                        //  set by command line -d parameter
int      thread_busy = 0;                                                  //  thread function active
int      kill_thread = 0;                                                  //  thread kill flag
int      fdestroy = 0;                                                     //  quit / destroy flag
int      fullSize, alignSize;                                              //  full and align image sizes
int      stbar = 0;                                                        //  status bar handle
int      modified = 0;                                                     //  image3 modified (vs image1)
int      crop_stat = 0, cropw, croph;                                      //  image crop status
int      Nalign = 0;                                                       //  alignment progress counter
int      showRedpix = 0;                                                   //  flag, highlight alignment pixels
int      pxL, pxH, pyL, pyH;                                               //  image overlap rectangle
int      pxM, pxmL, pxmH;                                                  //  pano blend stripe
int      Fautolens = 0;                                                    //  flag, autolens function running
int      pixsamp;                                                          //  pixel sample size (HDR, pano)

bitmap   *BMpixels = 0;                                                    //  flags edge pixels for image alignment

double   Bratios[3][256];                                                  //  brightness ratios /color /brightness
double   brightfix0[256], brightfix1[256], brightfix2[256];                //  calculated color adjustment factors 
double   lens_curve, lens_mm, lens_bow, lens_mmB, lens_bowB;               //  lens parameters: focal length, bow
double   xoff, yoff, toff, yst1, yst2;                                     //  alignment offsets: x, y, theta, y-stretch
double   xoffB, yoffB, toffB, yst1B, yst2B;                                //  alignment offsets: current best values
double   xystep, tstep, xylim, tlim, yststep, ystlim;                      //  alignment step size and search range
double   matchlev, matchB;                                                 //  image alignment match level
double   blend;                                                            //  image blend width (pano)

char     parmfile[maxfcc] = "";                                            //  editable parameters
double   parm_pixel_sample_size;
double   parm_jpg_save_quality;                                            //  v.28
double   parm_pano_lens_mm;                                                //  v.28
double   parm_pano_lens_bow;
double   parm_pano_prealign_size;
double   parm_pano_mouse_leverage;
double   parm_pano_align_size_increase;
double   parm_pano_blend_reduction;
double   parm_pano_minimum_blend;
double   parm_pano_image_stretch;                                          //  v.23

int   gtkinitfunc(void *data);                                             //  GTK initz. function
void  mwpaint();                                                           //  window paint function - expose event
void  mwpaint2(int refresh);                                               //  cause window repaint
void  update_status_bar();                                                 //  update status bar

void  pvwin_create();                                                      //  preview window functions
void  pvwin_paint();
void  pvwin_destroy();

int   KBevent(GtkWidget *, GdkEventKey *, void *);                         //  KB event function
void  mouse_event(GtkWidget *, GdkEventButton *, void *);                  //  mouse event function
void  magnify3();                                                          //  magnify image3 in main window
void  menufunc(GtkWidget *, const char *menu);                             //  menu function, main window

void  m_parms();                                                           //  adjust parameters
void  m_index();                                                           //  thumbnail index window  v.19
void  m_open(char *file);                                                  //  open new image file
void  m_prev();                                                            //  open previous file    v.13
void  m_next();                                                            //  open next file        v.13
void  m_undo();                                                            //  undo changes
void  m_redo();                                                            //  redo changes
void  m_save();                                                            //  save modified image
void  m_exif();                                                            //  show EXIF data   v.31
void  m_trash();                                                           //  move image to trash    v.15
void  m_kill();                                                            //  kill running thread
void  m_quit();                                                            //  exit application
void  m_help();                                                            //  display help file

int   mod_keep();                                                          //  query to discard mods
void  destroy();                                                           //  main window destroy signal

void  m_tune();                                                            //  adjust brightness/colors
void  m_crop();                                                            //  crop image
void  m_redeye();                                                          //  redeye removal
void  m_rotate(double angle);                                              //  rotate image
void  m_resize();                                                          //  resize image
void  m_HDR();                                                             //  make HDR image
void  m_pano();                                                            //  make panorama image
void  m_unbend();                                                          //  unbend panorama image

pixel  vpixel(pixel, double px, double py, int, int, int);                 //  get virtual pixel (px,py)
int    sigdiff(double d1, double d2, double signf);                        //  test for significant difference
void   get_overlap_region(GdkPixbuf *pxb1, GdkPixbuf *pxb2);               //  get image overlap rectangle
void   get_Bratios(GdkPixbuf *pxb1, GdkPixbuf *pxb2);                      //  get brightness ratios for overlap 
void   flag_edge_pixels(GdkPixbuf *pixbuf);                                //  flag high-contrast pixels in overlap 
double match_images(GdkPixbuf *pxb1, GdkPixbuf *pxb2);                     //  match images in overlap region
double match_pixels(pixel pix1, pixel pix2);                               //  match two pixels

void  undo_push();                                                         //  push image3 into undo stack
void  undo_pull();                                                         //  pull image3 from undo stack
void  undo_last();                                                         //  get last image3, no stack removal
void  undo_redo();                                                         //  advance undo stack without push
void  undo_clear();                                                        //  clear undo stack

void  incr_pixbufs();                                                      //  track pixbufs allocated
void  pixbuf_free(GdkPixbuf *&pixbuf);                                     //  track pixbufs freed

int   exif_show(char *file);                                               //  display EXIF data                  v.31
int   exif_fetch(char *file);                                              //  extract EXIF data from image file
int   exif_stuff(char *file1, char *file2, int xx, int yy);                //  stuff EXIF data from file1 into file2


//  main program

int main(int argc, char *argv[])
{
   GtkWidget   *vbox, *hbox, *vsep;
   int         ii, tbar;

   gtk_init(&argc,&argv);                                                  //  GTK command line options
   if (! g_thread_supported())                                             //  suddenly required for new gtk   v.30
         g_thread_init(0);                                                 //  initz. GTK for threads
   gdk_threads_init();
   zlockInit();

   initz_appfiles("parameters.txt",0);                                     //  directory and default params file

   for (ii = 1; ii < argc; ii++)                                           //  fotox command line options   v.16
   {
      if (strEqu(argv[ii],"-d")) debug = 1;                                //  -d (debug flag)
      else if (strEqu(argv[ii],"-p") && argc > ii+1)                       //  -p parameterFile
            strcpy(parmfile,argv[++ii]);
      else if (strEqu(argv[ii],"-i") && argc > ii+1)                       //  -i imageDirectory
            strcpy(imagedirk,argv[++ii]);
      else if (strEqu(argv[ii],"-f") && argc > ii+1)                       //  -f imageFile     v.17
            strcpy(clfile,argv[++ii]);
      else strcpy(clfile,argv[ii]);                                        //  imageFile        v.18
   }

   mwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);                             //  create main window
   gtk_window_set_title(GTK_WINDOW(mwin),fversion);
   gtk_window_set_position(GTK_WINDOW(mwin),GTK_WIN_POS_CENTER);
   gtk_window_set_default_size(GTK_WINDOW(mwin),mwinW,mwinH);
   
   vbox = gtk_vbox_new(0,0);                                               //  add vertical packing box
   gtk_container_add(GTK_CONTAINER(mwin),vbox);

   tbar = create_toolbar(vbox,48);                                         //  add horizontal toolbar and buttons  
   add_toolbar_button(tbar,"parms","change parameters","icons/parms.png",menufunc);
   add_toolbar_button(tbar,"index","thumbnail index","icons/index.png",menufunc);
   add_toolbar_button(tbar,"open","open new image file","icons/open.png",menufunc);
   add_toolbar_button(tbar,"prev","open previous image file","icons/prev.png",menufunc);
   add_toolbar_button(tbar,"next","open next image file","icons/next.png",menufunc);
   add_toolbar_button(tbar,"undo","undo image changes","icons/undo.png",menufunc);
   add_toolbar_button(tbar,"redo","redo image changes","icons/redo.png",menufunc);
   add_toolbar_button(tbar,"kill","kill running function","icons/kill.png",menufunc);
   add_toolbar_button(tbar,"save","save image to file","icons/save.png",menufunc);
   add_toolbar_button(tbar,"trash","move image file to trash","icons/trash.png",menufunc);
   add_toolbar_button(tbar,"exif","show EXIF data","icons/exif.png",menufunc);
   add_toolbar_button(tbar,"quit","quit fotox","icons/quit.png",menufunc);
   add_toolbar_button(tbar,"help","view help document","icons/help.png",menufunc);

   hbox = gtk_hbox_new(0,0);
   gtk_box_pack_start(GTK_BOX(vbox),hbox,1,1,0);

   tbar = create_toolbar(hbox,48,1);                                       //  add vertical toolbar and buttons
   add_toolbar_button(tbar,"tune","image brightness/contrast","icons/tune.png",menufunc);
   add_toolbar_button(tbar,"crop","cut-off margins","icons/crop.png",menufunc);
   add_toolbar_button(tbar,"redeye","reduce red-eye effect","icons/redeye.png",menufunc);
   add_toolbar_button(tbar,"rotate","rotate image","icons/rotate.png",menufunc);
   add_toolbar_button(tbar,"resize","resize image","icons/resize.png",menufunc);
   add_toolbar_button(tbar,"HDR","high dynamic range image","icons/HDR.png",menufunc);
   add_toolbar_button(tbar,"pano","panorama image","icons/pano.png",menufunc);
   add_toolbar_button(tbar,"unbend","unbend panorama image","icons/unbend.png",menufunc);

   vsep = gtk_vseparator_new();
   gtk_box_pack_start(GTK_BOX(hbox),vsep,0,0,0);
   
   vbox = gtk_vbox_new(0,0);
   gtk_box_pack_start(GTK_BOX(hbox),vbox,1,1,0);

   drmwin = gtk_drawing_area_new();                                        //  add drawing window
   gtk_box_pack_start(GTK_BOX(vbox),drmwin,1,1,0);

   stbar = create_stbar(vbox);                                             //  add status bar

   G_SIGNAL(mwin,"destroy",destroy,0)                                      //  connect signals to window
   G_SIGNAL(drmwin,"expose-event",mwpaint,0)
   G_SIGNAL(mwin,"key-press-event",KBevent,0)

   gtk_widget_add_events(drmwin,GDK_BUTTON_PRESS_MASK);                    //  connect mouse events
   gtk_widget_add_events(drmwin,GDK_BUTTON_RELEASE_MASK);
   gtk_widget_add_events(drmwin,GDK_BUTTON_MOTION_MASK);
   gtk_widget_add_events(drmwin,GDK_POINTER_MOTION_MASK);
   G_SIGNAL(drmwin,"button-press-event",mouse_event,0)
   G_SIGNAL(drmwin,"button-release-event",mouse_event,0)
   G_SIGNAL(drmwin,"motion-notify-event",mouse_event,0)
   
   gtk_widget_show_all(mwin);                                              //  show all widgets

   gdkgc = gdk_gc_new(drmwin->window);                                     //  initz. graphics context

   black.red = black.green = black.blue = 0;                               //  set up colors black, white
   white.red = white.green = white.blue = maxcolor;
   colormap = gtk_widget_get_colormap(drmwin);
   gdk_rgb_find_color(colormap,&black);
   gdk_rgb_find_color(colormap,&white);

   gtk_init_add((GtkFunction) gtkinitfunc,0);                              //  setup initz. call from gtk_main()

   gdk_threads_enter();
   gtk_main();                                                             //  process window events
   gdk_threads_leave();

   return 0;
}


//  initial function called from gtk_main() at startup

int gtkinitfunc(void * data)
{
   char     *pp;
   int      qual;

   initParmlist(20);                                                       //  set default parameters
   
   setparm(pixel_sample_size,5)
   setparm(jpg_save_quality,80)                                            //  default JPG quality  v.28
   setparm(pano_lens_mm,40)                                                //  default 35mm lens    v.28
   setparm(pano_lens_bow,0)                                                //  default zero bow
   setparm(pano_prealign_size,500)
   setparm(pano_mouse_leverage,2)
   setparm(pano_align_size_increase,1.6)
   setparm(pano_blend_reduction,0.7)
   setparm(pano_minimum_blend,10)
   setparm(pano_image_stretch,0)                                           //  disable by default     v.23

   if (! *parmfile) strcpy(parmfile,"use default");                        //  designated or default parm file
   loadParms(parmfile);

   getparm(pixel_sample_size)
   getparm(jpg_save_quality)
   getparm(pano_lens_mm)
   getparm(pano_lens_bow)
   getparm(pano_prealign_size)
   getparm(pano_mouse_leverage)
   getparm(pano_align_size_increase)
   getparm(pano_blend_reduction)
   getparm(pano_minimum_blend)
   getparm(pano_image_stretch)

   qual = int(parm_jpg_save_quality);                                      //  set JPG file save quality    v.28
   if (qual < 0 || qual > 100) qual = 80;
   snprintf(JPGquality,4,"%d",qual);
   
   if (! *imagedirk) getcwd(imagedirk,maxfcc-1);                           //  if no image dirk, use curr. dirk

   if (*clfile) {                                                          //  command line image file
      if (*clfile != '/') {
         pp = zmalloc(strlen(imagedirk)+strlen(clfile)+5);                 //  clfile is relative to imagedirk
         strcpy(pp,imagedirk);
         strcat(pp,"/");
         strcat(pp,clfile);
      }
      else {
         pp = zmalloc(strlen(clfile)+1);                                   //  use clfile
         strcpy(pp,clfile);
      }
      m_open(pp);                                                          //  open command line file
   }
   
   return 0;
}


//  paint window when created, exposed, or resized       v.27 overhaul
//  uses pxb3 (current modified image3)
//  no zlock(), call only from main thread

void mwpaint()
{
   GdkPixbuf         *pxbT = 0;
   static int        oldW = 0, oldH = 0;
   double            scalew, scaleh;
   int               ww, hh;
   GdkRectangle      rect;
   
   if (fdestroy) return;                                                   //  shutdown underway
   
   if (! pxb3) {                                                           //  no image to show
      gdk_window_clear(drmwin->window);
      return;
   }

   mwinW = drmwin->allocation.width;                                       //  window size
   mwinH = drmwin->allocation.height;
   if (mwinW < 20 || mwinH < 20) return;                                   //  reject tiny window

   if (mwinW != oldW || mwinH != oldH) {
      Mrescale++;                                                          //  if size changed, force rescale
      oldW = mwinW;
      oldH = mwinH;
   }

   if (Mrescale)                                                           //  refresh pxbM 
   {
      pixbuf_free(pxbM);
   
      if (Mscale == 0) {
         scalew = 1.0 * mwinW / ww3;
         scaleh = 1.0 * mwinH / hh3;                                       //  scale entire image3 to fit window
         if (scalew < scaleh) Rscale = scalew;
         else Rscale = scaleh;
         ww = int(Rscale * ww3);
         hh = int(Rscale * hh3);
         pxbM = pixbuf_scale_simple(pxb3,ww,hh,interp);
         pixbuf_test(pxbM);
         org3x = org3y = 0;                                                //  image3 origin = (0,0)
      }
      
      else  {                                                              //  Mscale = 1x or 2x
         if (Mneworg) {                                                    //  new image3 origin in window
            Rscale = Mscale;
            ww = int(mwinW / Rscale);                                      //  image3 pixels fitting in window
            hh = int(mwinH / Rscale);
            org3x = mp3x - ww / 2;                                         //  move image3 origin so that mouse
            org3y = mp3y - hh / 2;                                         //    position = middle of window
         }

         Rscale = Mscale;                                                  //  new scale, 1x or 2x
         ww = int(mwinW / Rscale);                                         //  image3 pixels fitting in window
         hh = int(mwinH / Rscale);
         if (ww > ww3) ww = ww3;
         if (hh > hh3) hh = hh3;

         if (org3x + ww > ww3) org3x = ww3 - ww;                           //  move image3 origin to keep
         if (org3y + hh > hh3) org3y = hh3 - hh;                           //    window full, if needed
         if (org3x < 0) org3x = 0;
         if (org3y < 0) org3y = 0;

         pxbM = pixbuf_new(colorspace,1,8,ww,hh);                          //  get portion of image3    +alpha v.32
         pixbuf_test(pxbM);                                                //    fitting in window
         pixbuf_copy_area(pxb3,org3x,org3y,ww,hh,pxbM,0,0);

         ww = int(ww * Rscale);                                            //  scale to window
         hh = int(hh * Rscale);
         pxbT = pixbuf_scale_simple(pxbM,ww,hh,interp);
         pixbuf_test(pxbT);
         pixbuf_free(pxbM);
         pxbM = pxbT;
      }
   }

   rect.x = rect.y = 0;                                                    //  paint rectangle = all window
   rect.width = mwinW;
   rect.height = mwinH;

   pixbuf_poop(M);                                                         //  pxbM width, height
   posMx = (mwinW - wwM) / 2;                                              //  center pxbM in drawing window   v.27
   posMy = (mwinH - hhM) / 2;

   gdk_window_begin_paint_rect(drmwin->window,&rect);                      //  pxbM >> drawing window
   gdk_draw_pixbuf(drmwin->window,0,pxbM,0,0,posMx,posMy,-1,-1,nodither);
   gdk_window_end_paint(drmwin->window);

   Mrescale = Mneworg = 0;                                                 //  reset flags
   update_status_bar();                                                    //  update status bar
   return;
}


//  invalidate window - cause window to get repainted immediately          //  v.22
//  called only from threads

void mwpaint2(int refresh)
{
   GdkRectangle   rect;

   if (fdestroy) return;                                                   //  shutdown underway
   if (refresh) Mrescale++;                                                //  image3 has changed
   
   rect.x = rect.y = 0;
   rect.width = mwinW;
   rect.height = mwinH;
   
   zlock();
   gdk_window_invalidate_rect(drmwin->window,&rect,0);                     //  invalidate whole window
   gdk_window_process_updates(drmwin->window,0);                           //  trigger expose event, repaint
   zunlock();
   return;
}


//  update main window status bar

void update_status_bar()
{
   char     msg[200], msg1[50], msg2[12], msg3[20];
   char     msg4[90], msg5[30], msg6[20], msg7[30];
   char     *fmt4 = "  aligning: %d %+.1f %+.1f %+.4f %s  match: %.4f";
   char     *fmt5 = "  stretch: %+.1f %+.1f";
   char     *fmt6 = "  crop: %dx%d";
   char     *fmt7 = "  lens mm: %.1f  bow: %.2f";
   int      fposn, fcount;
   
   if (fdestroy) return;
   
   get_image_counts(fposn,fcount);
   
   *msg1 = *msg2 = *msg3 = *msg4 = *msg5 = *msg6 = *msg7 = 0;
   if (modified) strcpy(msg3,"(modified)");
   if (fname3) snprintf(msg1,49,"%s  %dx%d  %.2fMB",msg3,ww3,hh3,f3size/mega);
   if (fposn) snprintf(msg2,11,"  %d/%d",fposn,fcount);
   if (yst1B || yst2B) snprintf(msg5,29,fmt5,yst1B,yst2B);
   if (Nalign) snprintf(msg4,89,fmt4,Nalign,xoffB,yoffB,toffB,msg5,matchB);
   if (crop_stat) snprintf(msg6,19,fmt6,cropw,croph);
   if (Fautolens) snprintf(msg7,29,fmt7,lens_mmB,lens_bowB);
   snprintf(msg,199,"%s%s%s%s%s",msg1,msg2,msg4,msg6,msg7);
   stbar_message(stbar,msg);
   return;
}


//  create preview window from preview pixbuf pxbP

void pvwin_create()
{
   zlock();

   pvwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);                            //  create preview window
   gtk_window_set_title(GTK_WINDOW(pvwin),"preview");
   gtk_window_set_transient_for(GTK_WINDOW(pvwin),GTK_WINDOW(mwin));
   gtk_window_set_default_size(GTK_WINDOW(pvwin),wwP,hhP);
   
   drpvwin = gtk_drawing_area_new();                                       //  add drawing window
   gtk_container_add(GTK_CONTAINER(pvwin),drpvwin);

   G_SIGNAL(pvwin,"destroy",pvwin_destroy,0)                               //  connect events
   G_SIGNAL(drpvwin,"expose-event",pvwin_paint,0)

   gtk_widget_show_all(pvwin);                                             //  show all widgets

   zunlock();
   return;
}


//  update preview window from preview pixbuf pxbP

void pvwin_paint()                                                         //  paint preview window      v.25
{
   GdkPixbuf   *pxbtemp;
   int         x, y, winx, winy;

   if (! pvwin) return;

   winx = drpvwin->allocation.width;                                       //  curr. window size
   winy = drpvwin->allocation.height;
   
   x = int(1.0 * winy * wwP / hhP);                                        //  preserve X/Y ratio
   if (winx > x) winx = x;
   y = int(1.0 * winx * hhP / wwP);
   if (winy > y) winy = y;

   zlock();   
   pxbtemp = pixbuf_scale_simple(pxbP,winx,winy,interp);                   //  scale image to window
   pixbuf_test(pxbtemp);
   gdk_draw_pixbuf(drpvwin->window,0,pxbtemp,0,0,0,0,-1,-1,nodither);      //  paint window
   pixbuf_free(pxbtemp);
   zunlock();

   return;
}


void pvwin_destroy()                                                       //  destroy preview window
{
   zlock();
   if (pvwin) gtk_widget_destroy(pvwin);
   zunlock();
   pvwin = 0;
   return;
}


//  keyboard event function - some toolbar buttons have KB equivalents     v.17
//  keys are mapped in /usr/include/gtk-2.0/gdk/gdkkeysyms.h

int KBevent(GtkWidget *win, GdkEventKey *event, void *)
{
   KBkey = event->keyval;
   if (thread_busy) return 1;

   if (KBkey == GDK_Left) m_prev();                                        //  left and right arrow keys
   if (KBkey == GDK_Right) m_next();
   if (KBkey == GDK_R) m_rotate(90);                                       //  keys L, R   
   if (KBkey == GDK_r) m_rotate(90);
   if (KBkey == GDK_L) m_rotate(-90);
   if (KBkey == GDK_l) m_rotate(-90);
   if (KBkey == GDK_Delete) m_trash();                                     //  delete key
   return 1;
}


//  mouse event function - capture buttons and drag movements

void mouse_event(GtkWidget *, GdkEventButton *event, void *)
{
   static int     buttontime;
   static int     fmouse = 0, fdrag = 0;
   
   if (event->type == GDK_BUTTON_PRESS) {
      if (event->button == 1) Lmdown = 1;                                  //  left mouse pressed
      mpWx = int(event->x);                                                //  track mouse position
      mpWy = int(event->y);
      buttontime = event->time;
      fmouse++;
   }

   if (event->type == GDK_BUTTON_RELEASE) {
      Lmdown = 0;                                                          //  cancel left mouse down status   v.28
      if (fdrag) {
         fdrag = 0;                                                        //  if drag underway, end it
         mdx1 = mdy1 = mdx2 = mdy2 = 0;
      }
      else {                                                               //  no drag underway
         buttontime = event->time - buttontime;
         if (buttontime < 300) {                                           //  300 milliseconds
            if (event->button == 1) Lmclick++;                             //  left mouse click
            if (event->button == 3) Rmclick++;                             //  right mouse click
            if (allowmag) magnify3();                                      //  change magnification if enabled
         }
      }
   }

   if (event->type == GDK_MOTION_NOTIFY && Lmdown) {
      mpWx = int(event->x);                                                //  mouse drag
      mpWy = int(event->y);
      fmouse++;
      fdrag++;
   }
   
   if (fmouse) {                                                           //  new mouse position
      fmouse = 0;
      mpMx = mpWx - posMx;                                                 //  corresponding pxbM position
      mpMy = mpWy - posMy;
      if (mpMx < 0) mpMx = 0;                                              //  stay inside image
      if (mpMy < 0) mpMy = 0;
      if (mpMx > wwM-1) mpMx = wwM-1;
      if (mpMy > hhM-1) mpMy = hhM-1;
      mp3x = org3x + int(mpMx / Rscale + 0.5);                             //  corresponding pxb3 position
      mp3y = org3y + int(mpMy / Rscale + 0.5);
   }
   
   if (fdrag) {
      if (! mdx1) { mdx1 = mpMx; mdy1 = mpMy; }                            //  new drag start (pxbM relative)
      mdx2 = mpMx; mdy2 = mpMy;                                            //  capture drag extent
   }

   return;
}


//  left mouse click: magnify image3 in main window and re-center at mouse position
//  right mouse click: de-magnify image3 (fit entire image in window)

void magnify3()
{
   if (Lmclick) Mscale++;                                                  //  left mouse click: magnify image
   if (Mscale > 2) Mscale = 2;                                             //  limit to 2x
   if (Rmclick) Mscale = 0;                                                //  right mouse click: de-magnify
   if (Lmclick) Mneworg++;                                                 //  re-center image3 in window
   Lmclick = Rmclick = 0;                                                  //  reset mouse click flags
   mwpaint2(1);                                                            //  repaint main window
   return;
}


//  process main window menu and toolbar events

void menufunc(GtkWidget *, const char *menu)
{
   if (strEqu(menu,"kill")) m_kill();                                      //  kill running thread
   if (strEqu(menu,"quit")) m_quit();                                      //  quit fotox
   
   if (thread_busy) {
      zmessageACK("prior function still running (wait or kill)");
      return;
   }

   if (strEqu(menu,"parms")) m_parms();
   if (strEqu(menu,"index")) m_index();
   if (strEqu(menu,"open")) m_open(0);
   if (strEqu(menu,"prev")) m_prev();
   if (strEqu(menu,"next")) m_next();
   if (strEqu(menu,"undo")) m_undo();
   if (strEqu(menu,"redo")) m_redo();
   if (strEqu(menu,"save")) m_save();
   if (strEqu(menu,"trash")) m_trash();
   if (strEqu(menu,"exif")) m_exif();
   if (strEqu(menu,"help")) m_help();

   if (strEqu(menu,"tune")) m_tune();                                      //  image edit functions
   if (strEqu(menu,"crop")) m_crop();
   if (strEqu(menu,"redeye")) m_redeye();
   if (strEqu(menu,"rotate")) m_rotate(0);
   if (strEqu(menu,"resize")) m_resize();
   if (strEqu(menu,"HDR")) m_HDR();
   if (strEqu(menu,"pano")) m_pano();
   if (strEqu(menu,"unbend")) m_unbend();
}


//  edit parameters and get their new values        v.09

void m_parms()
{
   int np = editParms(0,0);                                                //  edit parms
   if (! np) return;

   getparm(pixel_sample_size)                                              //  set new values
   getparm(jpg_save_quality)
   getparm(pano_lens_mm)
   getparm(pano_lens_bow)
   getparm(pano_prealign_size)
   getparm(pano_mouse_leverage)
   getparm(pano_align_size_increase)
   getparm(pano_blend_reduction)
   getparm(pano_minimum_blend)
   getparm(pano_image_stretch)                                             //  v.23
}


//  display a thumbnail index of images in a separate window               v.19

void m_index()
{
   if (! file1) m_open(0);
   if (! file1) return;

   void xfunction(char *file);
   thumbnail_index(file1,"init",xfunction);                                //  file1 is anchor file
   return;
}

void xfunction(char *filez)                                                //  receive selected file
{
   if (filez) m_open(filez);
   return;
}


//  open a new image file - make images pxb1 (keep) and pxb3 (modified)

void m_open(char *filez)
{
   char           *newfile, *pp, wtitle[100];
   GdkPixbuf      *pxbT = 0;
   struct stat    f1stat;
   
   if (mod_keep()) return;                                                 //  keep modifications
   
   if (filez) newfile = filez;                                             //  use passed file
   else {
      if (file1) newfile = zgetfile("open new image file",file1,"open");   //  assume same directory as prior 
      else newfile = zgetfile("open new image file",imagedirk,"open");     //  or use image file directory   v.16
   }
   if (! newfile) return;                                                  //  user cancel

   if (file1) zfree(file1);                                                //  set new file1
   file1 = newfile;
   
   pxbT = pixbuf_new_from_file(file1,gerror);                              //  validate image file
   if (pxbT) {
      nch = gdk_pixbuf_get_n_channels(pxbT);
      nbits = gdk_pixbuf_get_bits_per_sample(pxbT);
   }
   
   if (!pxbT || nch < 3 || nbits != 8) {
      zmessageACK("file is not fotox compatible:\n %s",file1);             //  clean missing/non-working file  v.27
      pixbuf_free(pxb1);
      track_image_files(file1,"remove");                                   //  remove from image file list
      thumbnail_index(file1,"update");                                     //  update index window (if there)
      return;
   }
   
   pixbuf_free(pxb1);                                                      //  set new image1
   pxb1 = pxbT;
   pixbuf_poop(1)                                                          //  get pxb1 attributes

   pixbuf_free(pxb3);
   pxb3 = pixbuf_copy(pxb1);                                               //  copy to image3 for editing
   pixbuf_test(pxb3);
   pixbuf_poop(3);
   
   stat(file1,&f1stat);                                                    //  get file size       v.24
   f1size = f3size = f1stat.st_size;

   if (! samedirk(file1,imagedirk)) {                                      //  if directory changed,     v.19
      track_image_files(file1,"init");                                     //    initz. image file list
      thumbnail_index(file1,"update");                                     //      and thumbnail window
   }

   strcpy(imagedirk,file1);                                                //  set new image dirk and file name
   pp = strrchr(imagedirk,'/');
   if (! pp) zappcrash("invalid directory: %s",file1);
   strncpy0(fname1,pp+1,99);
   strncpy0(fname3,pp+1,99);
   pp[1] = 0;

   snprintf(wtitle,99,"%s (%s)",fname1,imagedirk);                         //  window title                 v.26
   if (strlen(wtitle) > 97) strcpy(wtitle+96,"...");                       //  filename (/directory...)
   gtk_window_set_title(GTK_WINDOW(mwin),wtitle);

   undo_clear();                                                           //  clear undo stack
   modified = 0;                                                           //  image3 not modified
   Mscale = 0;                                                             //  scale to window           v.27
   mwpaint2(1);                                                            //  repaint window
   gtk_window_present(GTK_WINDOW(mwin));                                   //  bring to top

   return;
}


//  opem previous image file in same directory as last file opened         v.13

void m_prev()
{
   char        *fileP;
   
   if (! file1) return;
   fileP = track_image_files(file1,"prev");
   if (fileP) m_open(fileP);
   return;
}


//  opem next image file in same directory as last file opened             v.13

void m_next()
{
   char        *fileN;
   
   if (! file1) return;
   fileN = track_image_files(file1,"next");
   if (fileN) m_open(fileN);
   return;
}


//  undo last change - restore image3 from undo stack

void m_undo()
{
   undo_pull();                                                            //  restore previous image3
   return;
}


//  redo last undo - advance undo stack if another image is available      v.24

void m_redo()
{
   undo_redo();                                                            //  restore undone image3
   return;
}


//  save pxb3 (modified) image to a file

void m_save()
{
   char           *newfile, *pp;
   int            exiferr;
   struct stat    f3stat;
   
   if (! pxb3) return;
   
   newfile = zgetfile("save image to a file",file1,"save","quality");      //  get new file name from user   v.17
   if (! newfile) return;
   
   exiferr = exif_fetch(file1);                                            //  extract original file EXIF data    v.31

   gdk_pixbuf_save(pxb3,newfile,"jpeg",gerror,"quality",JPGquality,null);  //  save image3 to new or same file

   if (! exiferr) exif_stuff(file1,newfile,ww3,hh3);                       //  EXIF data >> new image file with new x/y

   if (strNeq(newfile,file1)) {                                            //  if a new file and in current directory,
      if (samedirk(file1,newfile)) {                                       //    add to image file index
         track_image_files(newfile,"add");
         thumbnail_index(0,"update");
      }
   }

   pp = strrchr(newfile,'/');                                              //  new image3 file name         v.24
   if (! pp) pp = newfile;
   else pp++;
   strncpy0(fname3,pp,99);
   stat(newfile,&f3stat);                                                  //  new size
   f3size = f3stat.st_size;
   update_status_bar();
   
   zfree(newfile);
   modified = 0;
   return;
}


//  delete image file - move file1 to trash                                v.15

void m_trash()
{
   int            yn, err;
   char           command[1000];

   if (! file1) return;                                                    //  nothing to trash
   
   yn = zmessageYN("move image file %s to trash?",fname1);
   if (! yn) return;

   snprintf(command,999,"mv \"%s\" ~/.Trash/",file1);                      //  move image file to trash directory
   err = system(command);
   if (err) {
      zmessageACK("error: %s",syserrText(err));
      return;
   }
   
   track_image_files(file1,"remove");                                      //  remove from image file list
   m_next();                                                               //  step to next file if there
   thumbnail_index(file1,"update");                                        //  update index window if there
   return;
}


//  show image EXIF data if available                                      v.31

void m_exif()
{
   if (file1) exif_show(file1);
   return;
}


//  kill running thread function

void m_kill()
{
   kill_thread++;
   while (thread_busy) zmainloop();
   kill_thread = 0;
   return;
}


//  quit - exit program

void m_quit()
{
   m_kill();
   if (mod_keep()) return;
   fdestroy++;
   gtk_main_quit();   
   return;
}


//  thread function to display help file

void m_help()                                                              //  menu function
{
   showz_helpfile();                                                       //  launch help file in new process  
   return;
}


//  query if modifications should be kept or discarded
//  return:  1 to keep mods,  0 to discard mods

int mod_keep()
{
   if (! modified) return 0;
   if (zmessageYN("discard modifications?")) return 0;
   return 1;
}


//  main window destroy signal    v.15b

void destroy()
{
   fdestroy++;
   printf("main window destroyed \n");
   m_kill();
   if (mod_keep()) m_save();
   gtk_main_quit();   
   return;
}


/**************************************************************************/

//  adjust image brightness, contrast and color levels

int tune_dialog_event(zdialog* zd, const char *event);                     //  tune dialog event function
void tune_exec(GdkPixbuf *pxb1, GdkPixbuf *pxb2);                          //  pxb1 + dialog edits >> pxb2

double      tuneBR[9], tuneWH[9], tuneRGB[3];                              //  tune dialog widget values

void m_tune()
{
   double      ratio;
   int         zstat;
   zdialog     *zd;

   if (! pxb3) return;                                                     //  no image
   undo_push();                                                            //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);
   pixbuf_test(pxb1);
   pixbuf_poop(1)

   ratio = 1.0 * pvwinsize / ww3;                                          //  create preview pixbuf   v.25
   if (hh3 > ww3) ratio = 1.0 * pvwinsize / hh3;
   if (ww3 < pvwinsize && hh3 < pvwinsize) ratio = 1.0;
   wwP = int(ww3 * ratio);
   hhP = int(hh3 * ratio);
   pxbP1 = pixbuf_scale_simple(pxb3,wwP,hhP,interp);                       //  original image
   pixbuf_test(pxbP1);
   pxbP = pixbuf_copy(pxbP1);                                              //  modified image
   pixbuf_test(pxbP);

   pvwin_create();                                                         //  display preview window

   zd = zdialog_new("adjust brightness / whiteness","cancel","done",0);    //  create tune dialog

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","space1","hb1",0,"space=70");
   zdialog_add_widget(zd,"label","labdark","hb1","dark areas");
   zdialog_add_widget(zd,"label","expand1","hb1",0,"expand");
   zdialog_add_widget(zd,"label","labbright","hb1","bright areas ");
   
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"expand|space=10");       //  brightness buttons
   zdialog_add_widget(zd,"vbox","vb2","hb2",0,"space=8");
   zdialog_add_widget(zd,"label","bright","vb2","brightness");
   zdialog_add_widget(zd,"hbox","hb21","vb2");
   zdialog_add_widget(zd,"vbox","vb22","hb21",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb23","hb21",0,"space=5");
   zdialog_add_widget(zd,"button","br +++","vb22","+++");
   zdialog_add_widget(zd,"button","br ---","vb23"," - - - ");
   zdialog_add_widget(zd,"button","br +-", "vb22"," +  - ");
   zdialog_add_widget(zd,"button","br -+", "vb23"," -  + ");
   zdialog_add_widget(zd,"button","br +-+","vb22"," + - + ");
   zdialog_add_widget(zd,"button","br -+-","vb23"," - + - ");

   zdialog_add_widget(zd,"vscale","brval0","hb2","-10|10|1|0","expand");   //  brightness sliders
   zdialog_add_widget(zd,"vscale","brval1","hb2","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","brval2","hb2","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","brval3","hb2","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","brval4","hb2","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","brval5","hb2","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","brval6","hb2","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","brval7","hb2","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","brval8","hb2","-10|10|1|0","expand");

   zdialog_add_widget(zd,"hbox","hb3","dialog",0,0,0,1,10);                //  whiteness buttons
   zdialog_add_widget(zd,"vbox","vb3","hb3",0,0,0,0,8);
   zdialog_add_widget(zd,"label","white","vb3","whiteness");
   zdialog_add_widget(zd,"hbox","hb31","vb3");
   zdialog_add_widget(zd,"vbox","vb32","hb31",0,0,0,0,5);
   zdialog_add_widget(zd,"vbox","vb33","hb31",0,0,0,0,5);
   zdialog_add_widget(zd,"button","wh +++","vb32","+++");
   zdialog_add_widget(zd,"button","wh ---","vb33"," - - - ");
   zdialog_add_widget(zd,"button","wh +-", "vb32"," +  - ");
   zdialog_add_widget(zd,"button","wh -+", "vb33"," -  + ");
   zdialog_add_widget(zd,"button","wh +-+","vb32"," + - + ");
   zdialog_add_widget(zd,"button","wh -+-","vb33"," - + - ");

   zdialog_add_widget(zd,"vscale","whval0","hb3","-10|10|1|0","expand");   //  whiteness sliders
   zdialog_add_widget(zd,"vscale","whval1","hb3","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","whval2","hb3","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","whval3","hb3","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","whval4","hb3","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","whval5","hb3","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","whval6","hb3","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","whval7","hb3","-10|10|1|0","expand");
   zdialog_add_widget(zd,"vscale","whval8","hb3","-10|10|1|0","expand");

   zdialog_add_widget(zd,"hbox","hb4","dialog",0,"space=10");              //  RGB spin buttons    v.15
   zdialog_add_widget(zd,"label","labred","hb4","   adjust red");
   zdialog_add_widget(zd,"spin","spred","hb4","-10|10|1|0");
   zdialog_add_widget(zd,"label","labgreen","hb4","   green");
   zdialog_add_widget(zd,"spin","spgreen","hb4","-10|10|1|0");
   zdialog_add_widget(zd,"label","labblue","hb4","   blue");
   zdialog_add_widget(zd,"spin","spblue","hb4","-10|10|1|0");

   zdialog_add_widget(zd,"hbox","hb5","dialog",0,"space=10");              //  undo, redo, commit, reset buttons
   zdialog_add_widget(zd,"label","space2","hb5",0,"space=30");
   zdialog_add_widget(zd,"button","undo","hb5","  undo  ");
   zdialog_add_widget(zd,"button","redo","hb5","  redo  ");                //  v.26
   zdialog_add_widget(zd,"button","commit","hb5","commit");
   zdialog_add_widget(zd,"button","reset","hb5"," reset ");

   zdialog_resize(zd,0,400);

   zstat = zdialog_run(zd,tune_dialog_event);                              //  run dialog
   zdialog_destroy(zd);

   if (zstat == 2) {                                                       //  done
      tune_exec(pxb1,pxb3);                                                //  image1 + mods >> image3
      modified++;                                                          //  image3 modified
      mwpaint2(1);                                                         //  refresh window
   }
   else undo_pull();                                                       //  cancelled, restore image3
   
   pvwin_destroy();                                                        //  kill preview window
   pixbuf_free(pxbP1);                                                     //  free preview memory
   pixbuf_free(pxbP);
   return;
}

int tune_dialog_event(zdialog *zd, const char *event)                      //  tune dialog event function
{
   int         ii;
   char        vsname[20];
   double      wvalue, brinc[9], whinc[9], pi = 3.14159;

   if (strEqu(event,"reset")) {                                            //  reset button
      for (ii = 0; ii < 9; ii++) 
      {
         strcpy(vsname,"brval0");                                          //  set all sliders back home
         vsname[5] = '0' + ii;
         zdialog_stuff(zd,vsname,0);
         strcpy(vsname,"whval0");
         vsname[5] = '0' + ii;
         zdialog_stuff(zd,vsname,0);
      }

      zdialog_stuff(zd,"spred",0);                                         //  reset RGB spin buttons
      zdialog_stuff(zd,"spgreen",0);
      zdialog_stuff(zd,"spblue",0);
   }

   if (strEqu(event,"undo") || strEqu(event,"reset")) {                    //  undo or reset button
      for (ii = 0; ii < 9; ii++) 
         tuneBR[ii] = tuneWH[ii] = 0;                                      //  reset all adjustments
      tuneRGB[0] = tuneRGB[1] = tuneRGB[2] = 0;

      tune_exec(pxbP1,pxbP);                                               //  update preview window     v.25
      pvwin_paint();

      if (modified) undo_last();                                           //  restore prior image3
      mwpaint2(1);
      return 1;
   }

   for (ii = 0; ii < 9; ii++)                                              //  clear all increments
      brinc[ii] = whinc[ii] = 0;

   if (strEqu(event,"br +++"))                                             //  convenience buttons "+++" etc.
      for (ii = 0; ii < 9; ii++)                                           //  get increments for all sliders
         brinc[ii] = 0.5;
   if (strEqu(event,"br ---")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = -0.5;
   if (strEqu(event,"br +-")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = 0.5 * cos(pi*ii/8.0);
   if (strEqu(event,"br -+")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = -0.5 * cos(pi*ii/8.0);
   if (strEqu(event,"br +-+")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = -0.5 * sin(pi*ii/8.0);
   if (strEqu(event,"br -+-")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = 0.5 * sin(pi*ii/8.0);

   if (strEqu(event,"wh +++")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = 0.5;
   if (strEqu(event,"wh ---")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = -0.5;
   if (strEqu(event,"wh +-")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = 0.5 * cos(pi*ii/8.0);
   if (strEqu(event,"wh -+")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = -0.5 * cos(pi*ii/8.0);
   if (strEqu(event,"wh +-+")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = -0.5 * sin(pi*ii/8.0);
   if (strEqu(event,"wh -+-")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = 0.5 * sin(pi*ii/8.0);

   for (ii = 0; ii < 9; ii++)                                              //  add increments to sliders    v.14
   {
      if (brinc[ii]) {
         strcpy(vsname,"brval0");
         vsname[5] = '0' + ii;
         zdialog_fetch(zd,vsname,wvalue);
         wvalue -= brinc[ii];
         if (wvalue < -10) wvalue = -10;
         if (wvalue > 10) wvalue = 10;
         zdialog_stuff(zd,vsname,wvalue);
      }

      if (whinc[ii]) {
         strcpy(vsname,"whval0");
         vsname[5] = '0' + ii;
         zdialog_fetch(zd,vsname,wvalue);
         wvalue -= whinc[ii];
         if (wvalue < -10) wvalue = -10;
         if (wvalue > 10) wvalue = 10;
         zdialog_stuff(zd,vsname,wvalue);
      }
   }
   
   for (ii = 0; ii < 9; ii++)                                              //  get all slider settings
   {
      strcpy(vsname,"brval0");
      vsname[5] = '0' + ii;
      zdialog_fetch(zd,vsname,wvalue);
      if (wvalue < -10) wvalue = -10;
      if (wvalue > 10) wvalue = 10;
      tuneBR[ii] = -wvalue;                                                //  set bright values, -10 to +10

      strcpy(vsname,"whval0");
      vsname[5] = '0' + ii;
      zdialog_fetch(zd,vsname,wvalue);
      if (wvalue < -10) wvalue = -10;
      if (wvalue > 10) wvalue = 10;
      tuneWH[ii] = -wvalue;                                                //  set white values, -10 to +10
   }

   zdialog_fetch(zd,"spred",tuneRGB[0]);                                   //  red/green/blue, -10 to +10   v.15
   zdialog_fetch(zd,"spgreen",tuneRGB[1]);
   zdialog_fetch(zd,"spblue",tuneRGB[2]);

   tune_exec(pxbP1,pxbP);                                                  //  update preview window     v.25
   pvwin_paint();

   if (strEqu(event,"commit"))                                             //  "commit" button
   {
      tune_exec(pxb1,pxb3);                                                //  image1 + mods >> image3
      modified++;                                                          //  image3 modified
      mwpaint2(1);                                                         //  refresh window
   }

   return 1;
}

void tune_exec(GdkPixbuf *pxb91, GdkPixbuf *pxb92)                         //  modify pixels according to     v.25
{                                                                          //    brightness & whiteness setpoints
   int         ii, px, py;
   int         ww91, hh91, rs91, ww92, hh92, rs92;
   pixel       ppix91, ppix92, pix1, pix2;
   double      red, green, blue, white, maxbright = 255;
   double      brin, brin8, brvalx, whvalx, brout;
   
   pixbuf_poop(91)
   pixbuf_poop(92)

   for (py = 0; py < hh91; py++)                                           //  loop all pixels
   for (px = 0; px < ww91; px++)
   {
      pix1 = ppix91 + py * rs91 + px * nch;                                //  input pixel
      pix2 = ppix92 + py * rs92 + px * nch;                                //  output pixel

      red = pix1[0];                                                       //  input pixel
      green = pix1[1];
      blue = pix1[2];
      
      brin = red;                                                          //  get brightest color
      if (green > brin) brin = green;
      if (blue > brin) brin = blue;
      
      brin = brin / maxbright;                                             //  normalize 0..1

      brin8 = 8.0 * brin;                                                  //  brightness zone 0--8
      ii = int(brin8);

      if (ii < 8)                                                          //  interpolate brightness value
         brvalx = tuneBR[ii] + (tuneBR[ii+1] - tuneBR[ii]) * (brin8 - ii);
      else brvalx = tuneBR[8];
      brvalx = brvalx * 0.1;                                               //  normalize -1..+1

      if (ii < 8)                                                          //  interpolate whiteness value
         whvalx = tuneWH[ii] + (tuneWH[ii+1] - tuneWH[ii]) * (brin8 - ii);
      else whvalx = tuneWH[8];
      whvalx = whvalx * 0.1;                                               //  normalize -1..+1

      //  adjust whiteness

      white = red;                                                         //  white content
      if (green < white) white = green;                                    //   = min(red,green,blue)
      if (blue < white) white = blue;

      white = white * whvalx;                                              //  -100% to +100%
      red += white;
      green += white;
      blue += white;
      
      //  adjust brightness

      brout = brin + brvalx * sqrt(0.5 - fabs(brin - 0.5));                //  convert input brightness
      if (brout > 0) brout = brout / brin;                                 //    to output brightness
      else brout = 0;
      
      if (brout * brin > 1.0) brout = 1.0 / brin;                          //  stop overflows

      red = red * brout + 0.49;                                            //  apply to all colors
      green = green * brout + 0.49;
      blue = blue * brout + 0.49;

      //  adjust RGB colors

      if (tuneRGB[0]) red += red * tuneRGB[0] / 20;                        //  get RGB adjustments          v.15
      if (tuneRGB[1]) green += green * tuneRGB[1] / 20;                    //  -10 to +10 >> -50% to +50%
      if (tuneRGB[2]) blue += blue * tuneRGB[2] / 20;
      
      if (red > 255) red = 255;                                            //  stop overflows
      if (green > 255) green = 255;
      if (blue > 255) blue = 255;

      pix2[0] = int(red);                                                  //  brightness, whiteness, RGB
      pix2[1] = int(green);
      pix2[2] = int(blue);
   }   

   return;
}


/**************************************************************************/

//  crop image - use mouse to select image region to retain                v.22  overhauled

void m_crop()
{
   void * crop_thread(void *);                                             //  thread function
   pthread_t   tid;
   
   if (! pxb3) return;                                                     //  nothing to crop
   undo_push();                                                            //  save image3 undo point

   ++thread_busy;                                                          //  start thread for crop job
   pthread_create(&tid,0,crop_thread,0);
   return;                                                                 //  thread runs on from here
}


//  crop thread function

void * crop_thread(void *)   
{
   int crop_dialog_compl(zdialog *zd, int zstat);                          //  dialog completion function

   char           *crop_message = "click or drag crop margins";

   zdialog        *zd;
   GdkPixbuf      *pxbT = 0;
   GdkCursor      *drag_cursor = 0;
   int8           dashes[2] = { 3, 3 };
   int            crx1, cry1, crx2, cry2, crww, crhh;                      //  crop rectangle
   int            mpMx0, mpMy0, redraw;
   
   Mscale = 0;                                                             //  scale to window
   allowmag = 0;                                                           //  disable magnification

   zd = zdialog_new("crop image","crop","cancel",0);                       //  start modal dialog to 
   zdialog_add_widget(zd,"label","lab1","dialog",crop_message);            //  get user completion event
   zdialog_run(zd,0,crop_dialog_compl);

   zlock();
   gdk_gc_set_foreground(gdkgc,&black);                                    //  set up to draw dotted lines
   gdk_gc_set_background(gdkgc,&white);
   gdk_gc_set_dashes(gdkgc,0,(gint8 *) dashes,2);
   gdk_gc_set_line_attributes(gdkgc,1,dottedline);

   drag_cursor = gdk_cursor_new(GDK_CROSSHAIR);                            //  set drag cursor
   gdk_window_set_cursor(drmwin->window,drag_cursor);
   zunlock();

   crx1 = int(0.2 * wwM);                                                  //  start with 20% crop margins
   cry1 = int(0.2 * hhM);
   crx2 = int(0.8 * wwM);
   cry2 = int(0.8 * hhM);
   crww = crx2 - crx1;
   crhh = cry2 - cry1;

   crop_stat = 1;                                                          //  status = crop in progress
   redraw = 1;
   mpMx0 = mpMx;
   mpMy0 = mpMy;

   while (crop_stat == 1)                                                  //  loop while crop in progress
   {
      zsleep(0.1);                                                         //  mouse sample interval, 0.1 secs.
      if (kill_thread) break;
      
      if (Lmclick || Lmdown)                                               //  mouse click or drag
      {
         Lmclick = 0;
         
         if (mpMx != mpMx0 || mpMy != mpMy0) 
         {
            mpMx0 = mpMx;
            mpMy0 = mpMy;
            redraw = 1;

            if (abs(mpMx - crx1) < abs(mpMx - crx2)) crx1 = mpMx;          //  new crop rectangle from mouse posn.
            else crx2 = mpMx;
            if (abs(mpMy - cry1) < abs(mpMy - cry2)) cry1 = mpMy;
            else cry2 = mpMy;

            if (crx1 > crx2-10) crx1 = crx2-10;                            //  sanity limits
            if (cry1 > cry2-10) cry1 = cry2-10;
            if (crx1 < 0) crx1 = 0;
            if (cry1 < 0) cry1 = 0;
            if (crx2 > wwM) crx2 = wwM;
            if (cry2 > hhM) cry2 = hhM;
         }
      }

      if (redraw)                                                          //  v.28  rework to stop flashing
      {
         redraw = 0;
         mwpaint2(0);                                                      //  clear old crop rectangle

         crww = crx2 - crx1;                                               //  draw current crop rectangle
         crhh = cry2 - cry1;
         zlock();
         gdk_draw_rectangle(drmwin->window,gdkgc,0,crx1+posMx,cry1+posMy,crww,crhh);
         zunlock();

         cropw = int(crww / Rscale);                                       //  status bar update   v.23
         croph = int(crhh / Rscale);
         update_status_bar();
      }
   }
   
   if (crop_stat != 2) {                                                   //  crop canceled
      undo_pull();
      goto thread_exit;
   }

   crx1 = int(crx1 / Rscale + org3x + 0.5);                                //  convert to image3 coordinates
   cry1 = int(cry1 / Rscale + org3y + 0.5);
   crx2 = int(crx2 / Rscale + org3x + 0.5);
   cry2 = int(cry2 / Rscale + org3y + 0.5);
   crww = crx2 - crx1;
   crhh = cry2 - cry1;
   if (crx1 + crww > ww3) crww = ww3 - crx1;
   if (cry1 + crhh > hh3) crhh = hh3 - cry1;

   zlock();
   pxbT = pixbuf_new(colorspace,0,8,crww,crhh);                            //  crop image3 in pxb3
   pixbuf_test(pxbT);
   pixbuf_copy_area(pxb3,crx1,cry1,crww,crhh,pxbT,0,0);
   pixbuf_free(pxb3);
   pxb3 = pxbT;
   pixbuf_poop(3)                                                          //  get new pxb3 attributes
   zunlock();

   modified++;                                                             //  image3 modified

thread_exit:

   zlock();
   gdk_window_set_cursor(drmwin->window,0);                                //  restore normal cursor
   if (drag_cursor) gdk_cursor_unref(drag_cursor);                         //  release memory
   zunlock();

   allowmag = 1;                                                           //  enable magnification 
   crop_stat = 0;
   --thread_busy;
   mwpaint2(1);                                                            //  refresh window
   return 0;
}


//  crop dialog completion callback function

int crop_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat == 1) crop_stat = 2;                                          //  crop done
   else crop_stat = 0;                                                     //  crop canceled
   zdialog_destroy(zd);
   return 0;
}


/**************************************************************************/

//  redeye removal - use mouse to select image area to remove red color    v.27 new

int      redeye_stat = 0;
int      drecx, drecy, drecw, drech;                                       //  drag rectangle

void m_redeye()
{
   void * redeye_thread(void *);                                           //  thread function
   pthread_t   tid;
   
   if (! pxb3) return;                                                     //  nothing to redeye
   undo_push();                                                            //  save image3 undo point

   ++thread_busy;                                                          //  start thread for redeye job
   pthread_create(&tid,0,redeye_thread,0);
   return;                                                                 //  thread runs on from here
}


//  redeye thread function

void * redeye_thread(void *)   
{
   int redeye_dialog_event(zdialog *zd, const char *event);                //  dialog event function
   int redeye_dialog_compl(zdialog *zd, int zstat);                        //  dialog completion function

   char           *redeye_message = "drag mouse over area \n"
                                    "for red-eye reduction";
   GdkCursor      *drag_cursor = 0;
   int8           dashes[2] = { 2, 2 };
   zdialog        *zd;
   
   zd = zdialog_new("redeye reduction","done","cancel",0);                 //  dialog to get user inputs
   zdialog_add_widget(zd,"label","lab1","dialog",redeye_message);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zd,"button","bmore","hb1"," darken ","space=5");
   zdialog_add_widget(zd,"button","bless","hb1"," go back ","space=5");
   zdialog_run(zd,redeye_dialog_event,redeye_dialog_compl);

   zlock();
   gdk_gc_set_foreground(gdkgc,&black);                                    //  set up to draw dotted lines
   gdk_gc_set_background(gdkgc,&white);
   gdk_gc_set_dashes(gdkgc,0,(gint8 *) dashes,2);
   gdk_gc_set_line_attributes(gdkgc,1,dottedline);

   drag_cursor = gdk_cursor_new(GDK_CROSSHAIR);                            //  set drag cursor
   gdk_window_set_cursor(drmwin->window,drag_cursor);
   zunlock();
   
   drecx = drecy = drecw = drech = 0;                                      //  no drag rectangle yet
   
   allowmag = 0;                                                           //  magnification controlled below
   redeye_stat = 1;                                                        //  redeye in progress

   while (redeye_stat == 1)                                                //  loop while redeye in progress
   {
      zsleep(0.1);                                                         //  mouse sample interval, 0.1 secs.

      if (kill_thread) break;

      if (Lmclick || Rmclick) {
         magnify3();                                                       //  mouse click - magnify image
         drecw = drech = 0;
      }

      if (mdx1 || mdy1) {                                                  //  mouse drag underway
         drecw = mdx2 - mdx1;                                              //  drag increment
         drech = mdy2 - mdy1;
         drecx = mdx1 - drecw/2;                                           //  keep center fixed
         drecy = mdy1 - drech/2;
         mwpaint2(0);                                                      //  clear old circle
      }

      if (drecx < 0) drecx = 0;                                            //  stay within image area    v.32
      if (drecy < 0) drecy = 0;      
      if (drecx + drecw > wwM) drecw = wwM - drecx;
      if (drecy + drech > hhM) drech = hhM - drecy;
      
      zlock();                                                             //  draw new redeye circle
      if (drecw > 0 && drech > 0)
         gdk_draw_arc(drmwin->window,gdkgc,0,drecx+posMx,drecy+posMy,drecw,drech,0,64*360);
      zunlock();
   }
   
   if (redeye_stat != 2) undo_pull();                                      //  redeye canceled

   zlock();
   gdk_window_set_cursor(drmwin->window,0);                                //  restore normal cursor
   if (drag_cursor) gdk_cursor_unref(drag_cursor);                         //  release memory
   zunlock();

   allowmag = 1;
   --thread_busy;
   mwpaint2(1);                                                            //  refresh window
   return 0;
}


//  redeye dialog event and completion callback functions

int redeye_dialog_event(zdialog * zd, const char *event)
{
   static GdkPixbuf  *pxbT = 0;
   int               wwT, hhT, rsT;
   int               px1, py1, px, py, cx, cy;
   double            r, R, redpart;
   static double     pR, maxred;
   static int        pcx = 0, pcy = 0;
   pixel             ppixT, ppix;

   if (!(drecw > 0 && drech > 0)) return 0;
   
   px1 = int(drecx / Rscale + org3x);                                      //  convert redeye rectangle 
   py1 = int(drecy / Rscale + org3y);                                      //    to image3 coordinates
   wwT = int(drecw / Rscale);
   hhT = int(drech / Rscale);
   
   cx = wwT / 2;                                                           //  center
   cy = hhT / 2;
   R = (wwT + hhT) / 4.0;                                                  //  mean radius

   if (cx != pcx || cy != pcy || R != pR) 
   {
      pcx = cx;                                                            //  rectangle has changed
      pcy = cy;
      pR = R;
      pixbuf_free(pxbT);                                 
      pxbT = pixbuf_new(colorspace,0,8,wwT,hhT);                           //  copy redeye rectangle
      pixbuf_copy_area(pxb3,px1,py1,wwT,hhT,pxbT,0,0);

      pixbuf_poop(T);
      maxred = 0;

      for (py = 0; py < hhT; py++)                                         //  scan pixels within circle,
      for (px = 0; px < wwT; px++)                                         //    set initial red threshold
      {
         r = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
         if (r > R) continue;
         ppix = ppixT + py * rsT + px * nch;
         redpart = 1.0 * ppix[0] / (ppix[0] + ppix[1] + ppix[2] + 1);
         if (redpart > maxred) maxred = redpart;
      }
   }

   pixbuf_poop(T);

   if (strEqu(event,"bmore")) 
   {
      maxred = 0.95 * maxred;                                              //  reduce redness threshold
   
      for (py = 0; py < hhT; py++)                                         //  scan pixels
      for (px = 0; px < wwT; px++)
      {
         r = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
         if (r > R) continue;
         ppix = ppixT + py * rsT + px * nch;
         redpart = 1.0 * ppix[0] / (ppix[0] + ppix[1] + ppix[2] + 1);      //  for pixels in pxbT over threshold,
         if (redpart > maxred) {                                           //    darken corresponding pxb3 pixel
            ppix = ppix3 + (py1 + py) * rs3 + (px1 + px) * nch;
            ppix[0] = int(0.90 * ppix[0]);
         }
      }

      modified++;
   }

   if (strEqu(event,"bless")) 
   {
      for (py = 0; py < hhT; py++)                                         //  scan pixels
      for (px = 0; px < wwT; px++)
      {
         r = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
         if (r > R) continue;
         ppix = ppixT + py * rsT + px * nch;
         redpart = 1.0 * ppix[0] / (ppix[0] + ppix[1] + ppix[2] + 1);      //  for pixels in pxbT over threshold,
         if (redpart > maxred) {                                           //    lighten corresponding pxb3 pixel
            ppix = ppix3 + (py1 + py) * rs3 + (px1 + px) * nch;
            ppix[0] = int(1.11 * ppix[0]);
         }
      }

      modified++;
      maxred = 1.053 * maxred;                                             //  increase redness threshold
   }

   mwpaint2(1);
   return 1;
}

int redeye_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat == 1) redeye_stat = 2;                                        //  redeye done
   else redeye_stat = 0;                                                   //  redeye canceled
   zdialog_destroy(zd);
   return 0;
}


/**************************************************************************/

//  rotate image through any arbitrary angle                               v.13

void * rotate_thread(void *);                                              //  thread function

int         rotate_stat = 0;
double      rotate_angle = 0;
double      rotate_angle2 = 0;

void m_rotate(double angle)
{
   pthread_t   tid;

   if (! pxb3) return;                                                     //  nothing to rotate
   undo_push();                                                            //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image1 is image3 (with mods)
   pxb1 = pixbuf_copy(pxb3);
   pixbuf_test(pxb1);
   pixbuf_poop(1)
   
   rotate_angle = angle;                                                   //  put passed angle in heap memory

   ++thread_busy;                                                          //  start thread for rotate job
   pthread_create(&tid,0,rotate_thread,0);
   return;                                                                 //  thread runs on from here
}


//  rotate thread function

void * rotate_thread(void *)
{
   int rotate_dialog_event(zdialog *zd, const char * event);               //  dialog event function
   int rotate_dialog_compl(zdialog *zd, int zstat);                        //  dialog completion function

   char        text[20];
   zdialog     *zd;
   
   if (rotate_angle) {                                                     //  angle was passed   v.18
      zlock();
      pixbuf_free(pxb3);
      pxb3 = pixbuf_rotate(pxb1,rotate_angle);                             //  rotate to new angle
      pixbuf_test(pxb3);
      pixbuf_poop(3);
      zunlock();
      modified++;                                                          //  image3 modified
      goto thread_exit;
   }
   
   zd = zdialog_new("rotate image","done","cancel",0);                     //  dialog: get angle from user, rotate
   zdialog_add_widget(zd,"label","labdeg","dialog","degrees","space=5");   //  events and completion status
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb3","hb1",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb4","hb1",0,"space=5");
   zdialog_add_widget(zd,"button"," +0.1  ","vb1"," + 0.1 ");              //  button name is increment to use
   zdialog_add_widget(zd,"button"," -0.1  ","vb1"," - 0.1 ");
   zdialog_add_widget(zd,"button"," +1.0  ","vb2"," + 1   ");
   zdialog_add_widget(zd,"button"," -1.0  ","vb2"," - 1   ");
   zdialog_add_widget(zd,"button"," +10.0 ","vb3"," + 10  ");
   zdialog_add_widget(zd,"button"," -10.0 ","vb3"," - 10  ");
   zdialog_add_widget(zd,"button"," +90.0 ","vb4"," + 90  ");
   zdialog_add_widget(zd,"button"," -90.0 ","vb4"," - 90  ");

   zdialog_run(zd,rotate_dialog_event,rotate_dialog_compl);
   
   rotate_stat = 1;
   rotate_angle = 0;
   rotate_angle2 = 0;

   while (rotate_stat == 1)                                                //  loop while rotate in progress
   {
      zsleep(0.1);                                                         //  sample interval, 0.1 seconds
      
      if (kill_thread) break;

      if (rotate_angle == rotate_angle2) continue;
      rotate_angle2 = rotate_angle;                                        //  angle has changed

      zlock();
      pixbuf_free(pxb3);
      pxb3 = pixbuf_rotate(pxb1,rotate_angle);                             //  rotate to new angle
      pixbuf_test(pxb3);
      pixbuf_poop(3);
      zunlock();

      modified++;                                                          //  image3 modified
      mwpaint2(1);                                                         //  update window

      sprintf(text,"degrees: %.1f",rotate_angle);                          //  update dialog angle display
      zdialog_stuff(zd,"labdeg",text);
   }
   
   if (rotate_stat != 2) undo_pull();                                      //  rotate cancelled

thread_exit:
   --thread_busy;
   mwpaint2(1);                                                            //  refresh window
   return 0;
}


//  rotate dialog event and completion callback functions

int rotate_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat == 1) rotate_stat = 2;                                        //  done
   else rotate_stat = 0;                                                   //  canceled
   zdialog_destroy(zd);
   return 0;
}

int rotate_dialog_event(zdialog *zd, const char * event)
{
   int         kk;
   double      incr;
   
   kk = convSD(event,incr);                                                //  button name is increment to use
   if (kk != 0) return 0;
   rotate_angle += incr;
   if (rotate_angle >= 360) rotate_angle -=360;
   if (rotate_angle <= -360) rotate_angle +=360;
   return 1;
}


/**************************************************************************/

//  resize image                                                           v.15

int      wpix0, hpix0, wpix1, hpix1;
double   wpct1, hpct1;

void m_resize()
{
   int resize_dialog_event(zdialog *zd, const char * event);               //  dialog event function

   zdialog     *zd;
   int         zstat;

   if (! pxb3) return;                                                     //  nothing to resize
   undo_push();                                                            //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);
   pixbuf_test(pxb1);
   pixbuf_poop(1)
   
   zd = zdialog_new("resize image","execute","cancel",0);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zd,"vbox","vb11","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb12","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb13","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"label","placeholder","vb11",0);                  //             pixels       percent
   zdialog_add_widget(zd,"label","labw","vb11","width");                   //    width    [______]     [______]
   zdialog_add_widget(zd,"label","labh","vb11","height");                  //    height   [______]     [______]
   zdialog_add_widget(zd,"label","labpix","vb12","pixels");                //
   zdialog_add_widget(zd,"spin","wpix","vb12","20|9999|1|0");              //    preset [2/3] [1/2] [1/3] [1/4] 
   zdialog_add_widget(zd,"spin","hpix","vb12","20|9999|1|0");              //
   zdialog_add_widget(zd,"label","labpct","vb13","percent");               //    [_] lock width/height ratio
   zdialog_add_widget(zd,"spin","wpct","vb13","1|500|0.1|100");            //
   zdialog_add_widget(zd,"spin","hpct","vb13","1|500|0.1|100");            //       [ execute ]  [ cancel ]  
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","preset","hb2","preset","space=5");
   zdialog_add_widget(zd,"button","b 2/3","hb2"," 2/3 ");
   zdialog_add_widget(zd,"button","b 1/2","hb2"," 1/2 ");
   zdialog_add_widget(zd,"button","b 1/3","hb2"," 1/3 ");
   zdialog_add_widget(zd,"button","b 1/4","hb2"," 1/4 ");
   zdialog_add_widget(zd,"check","lock","dialog","lock width:height ratio");

   wpix0 = ww3;                                                            //  original image3 width, height
   hpix0 = hh3;
   zdialog_stuff(zd,"wpix",wpix0);
   zdialog_stuff(zd,"hpix",hpix0);
   zdialog_stuff(zd,"lock",1);
   
   zstat = zdialog_run(zd,resize_dialog_event,0);
   zdialog_destroy(zd);

   if (zstat != 1) {                                                       //  resize cancelled
      undo_pull();
      return;
   }

   pixbuf_free(pxb3);
   pxb3 = pixbuf_scale_simple(pxb1,wpix1,hpix1,interp);                    //  scale to size
   pixbuf_test(pxb3);
   pixbuf_poop(3);

   modified++;                                                             //  image3 modified
   mwpaint2(1);                                                            //  update window
}

int resize_dialog_event(zdialog *zd, const char * event)
{
   int         lock;

   zdialog_fetch(zd,"wpix",wpix1);                                         //  get all widget values
   zdialog_fetch(zd,"hpix",hpix1);
   zdialog_fetch(zd,"wpct",wpct1);
   zdialog_fetch(zd,"hpct",hpct1);
   zdialog_fetch(zd,"lock",lock);
   
   if (strEqu(event,"b 2/3")) {
      wpix1 = (2 * wpix0 + 2) / 3;
      hpix1 = (2 * hpix0 + 2) / 3;
   }
   
   if (strEqu(event,"b 1/2")) {
      wpix1 = (wpix0 + 1) / 2;
      hpix1 = (hpix0 + 1) / 2;
   }
   
   if (strEqu(event,"b 1/3")) {
      wpix1 = (wpix0 + 2) / 3;
      hpix1 = (hpix0 + 2) / 3;
   }
   
   if (strEqu(event,"b 1/4")) {
      wpix1 = (wpix0 + 3) / 4;
      hpix1 = (hpix0 + 3) / 4;
   }

   if (strEqu(event,"wpct"))                                               //  width % - set pixel width
      wpix1 = int(wpct1 / 100.0 * wpix0 + 0.5);

   if (strEqu(event,"hpct"))                                               //  height % - set pixel height
      hpix1 = int(hpct1 / 100.0 * hpix0 + 0.5);
   
   if (lock) {                                                             //  if lock, set other dimension
      if (event[0] == 'w')                                                 //    to preserve width/height ratio
         hpix1 = int(wpix1 * (1.0 * hpix0 / wpix0) + 0.5);
      if (event[0] == 'h') 
         wpix1 = int(hpix1 * (1.0 * wpix0 / hpix0) + 0.5);
   }
   
   hpct1 = 100.0 * hpix1 / hpix0;                                          //  set percents to match pixels
   wpct1 = 100.0 * wpix1 / wpix0;
   
   zdialog_stuff(zd,"wpix",wpix1);                                         //  synch. all widget values
   zdialog_stuff(zd,"hpix",hpix1);
   zdialog_stuff(zd,"wpct",wpct1);
   zdialog_stuff(zd,"hpct",hpct1);
   
   return 1;
}


/**************************************************************************/

//  Make an HDR (high dynamic range) image from two images of the same subject 
//  with different exposure levels. The HDR image has expanded gradations of 
//  intensity (visibility of detail) in both the brightest and darkest areas.

void *   HDR_thread(void *);
void     HDR_combine_images(int ww);
void     HDR_level_adjust();
void     HDR_prep_preview();
void     HDR_make_preview();

double   hdrw[9];                                                          //  image weights for 8 zones   v.28

void m_HDR()
{
   pthread_t   tid;
   
   if (! pxb3) return;                                                     //  no 1st image

   if (file2) zfree(file2);
   file2 = zgetfile("select image file to combine",file1,"open");          //  get 2nd HDR image in pxb2
   if (! file2) return;
   pxb2 = pixbuf_new_from_file(file2,gerror);
   if (! pxb2) goto err_nomatch;
   pixbuf_poop(2)

   if (ww2 != ww3) goto err_nomatch;                                       //  validate compatibility
   if (hh2 != hh3) goto err_nomatch;
   if (rs2 != rs3) goto err_nomatch;
   if (gdk_pixbuf_get_n_channels(pxb2) != nch) goto err_nomatch;           //  v.25
   if (gdk_pixbuf_get_bits_per_sample(pxb2) != nbits) goto err_nomatch;

   pixbuf_free(pxb1);                                                      //  1st HDR image is image3
   pxb1 = pixbuf_copy(pxb3);                                               //  (keep prior mods)
   pixbuf_test(pxb1);
   pixbuf_poop(1)

   undo_push();                                                            //  save image3 undo point
   ++thread_busy;                                                          //  launch thread to combine
   pthread_create(&tid,0,HDR_thread,0);                                    //    pxb1 + pxb2 >> pxb3
   return;                                                                 //  thread runs on from here

err_nomatch:
   zmessageACK("second image file is incompatible with first");
   if (file2) zfree(file2);
   file2 = 0;
   pixbuf_free(pxb2);
   pxb2 = 0;
   return;
}


//  Thread function for combining pxb1 + pxb2 >> pxb3

void * HDR_thread(void *)
{
   double      xfL, xfH, yfL, yfH, tfL, tfH;
   int         firstpass = 1, lastpass = 0;

   Nalign = 1;                                                             //  alignment in progress
   showRedpix = 1;                                                         //  highlight alignment pixels
   
   fullSize = ww1;                                                         //  full image size
   if (hh1 > ww1) fullSize = hh1;                                          //  (largest dimension)

   alignSize = 100;                                                        //  initial alignment image size
   if (alignSize > fullSize) alignSize = fullSize;

   pixsamp = 1000 * int(parm_pixel_sample_size);                           //  no. pixels to sample

   xoff = yoff = toff = 0;                                                 //  initial offsets = 0
   blend = 0;                                                              //  blend region is 100%

   yst1 = yst2 = yst1B = yst2B = 0;                                        //  disable stuff only for pano
   lens_curve = 0;

   while (true)                                                            //  next alignment stage / image size
   {   
      ww1A = ww1 * alignSize / fullSize;                                   //  align width, height in same ratio
      hh1A = hh1 * alignSize / fullSize;
      ww2A = ww1A;
      hh2A = hh1A;

      if (! lastpass) 
      {
         zlock();
         pixbuf_free(pxb1A);                                               //  create alignment images
         pixbuf_free(pxb2A);
         pxb1A = pixbuf_scale_simple(pxb1,ww1A,hh1A,interp);
         pixbuf_test(pxb1A);
         pxb2A = pixbuf_scale_simple(pxb2,ww2A,hh2A,interp);
         pixbuf_test(pxb2A);
         pixbuf_free(pxb3);
         pxb3 = pixbuf_copy(pxb1A);                                        //  make pxb3 compatible
         pixbuf_test(pxb3);

         pixbuf_poop(1A)                                                   //  get pxb1A attributes
         pixbuf_poop(2A)                                                   //  get pxb2A attributes
         pixbuf_poop(3)                                                    //  get pxb3 attributes
         zunlock();

         get_overlap_region(pxb1A,pxb2A);                                  //  get image overlap area
         get_Bratios(pxb1A,pxb2A);                                         //  get image brightness ratios
         flag_edge_pixels(pxb1A);                                          //  flag high-contrast pixels
      }

      xylim = 2;                                                           //  search range from prior stage:
      xystep = 1;                                                          //    -2 -1 0 +1 +2 pixels

      if (firstpass) xylim = 0.05 * alignSize;                             //  1st stage search range, huge

      if (lastpass) {
         xylim = 1;                                                        //  final stage search range:
         xystep = 0.5;                                                     //    -1.0 -0.5 0.0 +0.5 +1.0
      }

      tlim = xylim / alignSize;                                            //  theta max offset, radians
      tstep = xystep / alignSize;                                          //  theta step size

      HDR_combine_images(0);                                               //  combine images >> pxb3

      matchlev = match_images(pxb1A,pxb2A);                                //  set base match level
      matchB = matchlev;                                                   //   = best matching offsets

      xfL = xoff - xylim;
      xfH = xoff + xylim + xystep/2;
      yfL = yoff - xylim;
      yfH = yoff + xylim + xystep/2;
      tfL = toff - tlim;
      tfH = toff + tlim + tstep/2;

      xoffB = xoff;
      yoffB = yoff;
      toffB = toff;
      
      for (xoff = xfL; xoff < xfH; xoff += xystep)                         //  test all offset dimensions
      for (yoff = yfL; yoff < yfH; yoff += xystep)                         //    in all combinations
      for (toff = tfL; toff < tfH; toff += tstep)
      {
         matchlev = match_images(pxb1A,pxb2A);
         if (sigdiff(matchlev,matchB,0.00001) > 0) {
            matchB = matchlev;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
            update_status_bar();
         }

         Nalign++;
         update_status_bar();
         if (kill_thread) goto HDR_exit;
      }

      xoff = xoffB;                                                        //  recover best offsets
      yoff = yoffB;
      toff = toffB;

      HDR_combine_images(0);
     
      firstpass = 0;
      if (lastpass) break;                                                 //  done

      if (alignSize == fullSize) {                                         //  full size image was aligned
         lastpass++;                                                       //  one more pass
         continue;
      }

      double R = alignSize;
      alignSize = 2 * alignSize;                                           //  next larger image size
      if (alignSize > 0.8 * fullSize) alignSize = fullSize;                //  if near goal, jump to it now
      R = alignSize / R;                                                   //  ratio of new / old image size
      xoff = R * xoff;                                                     //  adjust offsets for image size
      yoff = R * yoff;
   }

   showRedpix = 0;
   HDR_combine_images(0);                                                  //  combine final images >> pxb3
   HDR_level_adjust();                                                     //  dialog, adjust input image weights

HDR_exit:

   Nalign = 0;                                                             //  alignment done
   if (kill_thread) undo_pull();                                           //  killed

   zlock();
   if (file2) zfree(file2);                                                //  free 2nd input file
   file2 = 0;
   pixbuf_free(pxb2);                                                      //  free 2nd input image
   pixbuf_free(pxb1A);                                                     //  free alignment images
   pixbuf_free(pxb2A);
   zunlock();

   mwpaint2(1);                                                            //  repaint window (pxb3)
   --thread_busy;                                                          //  exit thread
   return 0;
}


//  combine images pxb1A and pxb2A using weights in hdrw[9]
//  hdrw[0]-[1] for darkest zone, hdrw[7]-[8] for brightest zone
//  output is in pxb3

void HDR_combine_images(int weight)
{
   int      px1, py1, ii;
   double   px2, py2;
   double   sintf = sin(toff), costf = cos(toff);
   double   br1, br2, brm;
   pixel    pix1, pix2, pix3;

   for (py1 = 0; py1 < hh1A; py1++)                                        //  step through pxb1A pixels
   for (px1 = 0; px1 < ww1A; px1++)
   {
      px2 = (px1 - xoff) * costf + (py1 - yoff) * sintf;                   //  corresponding pxb2A pixels
      py2 = (py1 - yoff) * costf - (px1 - xoff) * sintf;
   
      pix1 = ppix1A + py1 * rs1A + px1 * nch;                              //  input pixel 1
      pix2 = vpixel(ppix2A,px2,py2,ww2A,hh2A,rs2A);                        //  input pixel 2
      pix3 = ppix3 + py1 * rs3 + px1 * nch;                                //  output pxb3 pixel

      if (! pix2) {                                                        //  if non-overlapping pixel,
         pix3[0] = pix3[1] = pix3[2] = 0;                                  //    set output pixel black
         continue;
      }
      
      if (weight) {
         br1 = pix1[0] + pix1[1] + pix1[2];                                //  image 1 pixel brightness
         br2 = pix2[0] + pix2[1] + pix2[2];                                //  image 2
         brm = (br1 + br2) / 2;                                            //  mean, 0 to 765

         ii = int(brm / 96);                                               //  brightness band, 0 .. 7      v.28
         br1 = hdrw[ii];                                                   //  weight at low end of band
         br2 = hdrw[ii+1];                                                 //  weight at high end of band
         brm = br1 + (br2 - br1) * (brm - 96 * ii) / 96;                   //  interpolate within band
         
         br1 = brm / 80.0;                                                 //  image 1 weight: 0 .. 1.0
         br2 = 1.0 - br1;                                                  //  image 2 weight
         
         pix3[0] = int(pix1[0] * br1 + pix2[0] * br2);                     //  build output pixel
         pix3[1] = int(pix1[1] * br1 + pix2[1] * br2);
         pix3[2] = int(pix1[2] * br1 + pix2[2] * br2);
      }

      else {
         pix3[0] = (pix1[0] + pix2[0]) / 2;                                //  output pixel is simple average
         pix3[1] = (pix1[1] + pix2[1]) / 2;
         pix3[2] = (pix1[2] + pix2[2]) / 2;
      }

      if (showRedpix && pix2) {                                            //  highlight alignment pixels   v.25
         ii = py1 * ww1A + px1;
         if (bitmap_get(BMpixels,ii)) {
            pix3[0] = 255;
            pix3[1] = pix3[2] = 0;
         }
      }
   }

   modified++;
   mwpaint2(1);
   return;
}


//  dialog to adjust image brightness levels
//  pxb1A + pxb2A >> pxb3 under control of scale bars

void HDR_level_adjust()
{
   int HDR_level_dialog_event(zdialog *zd, const char *name);              //  dialog event function
   
   int         ii, zstat;
   double      bravg, wval;
   char        vsname[4] = "vs0";
   zdialog     *zd;
   
   zd = zdialog_new("HDR input image weights","done","cancel",0);
   zdialog_add_widget(zd,"label","labt","dialog","\n input image weights per brightness band \n");
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"expand");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"expand|space=10");          //  fix locale bug      v.29
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"expand|space=10");
   zdialog_add_widget(zd,"vbox","vb3","hb1",0,"expand|space=10");
   zdialog_add_widget(zd,"label","lab11","vb1"," input image 2");
   zdialog_add_widget(zd,"label","lab12","vb1",0,"expand");
   zdialog_add_widget(zd,"label","lab13","vb1"," input image 1");
   zdialog_add_widget(zd,"label","lab14","vb1"," output image ");
   zdialog_add_widget(zd,"label","lab21","vb2"," 100 % ");
   zdialog_add_widget(zd,"label","lab22","vb2"," 50/50 ","expand");
   zdialog_add_widget(zd,"label","lab23","vb2"," 100 % ");
   zdialog_add_widget(zd,"label","lab24","vb2");
   zdialog_add_widget(zd,"hbox","hb2","vb3",0,"expand|space=5");
   zdialog_add_widget(zd,"vscale","vs0","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs1","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs2","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs3","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs4","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs5","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs6","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs7","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"vscale","vs8","hb2","0|80|1|0|");
   zdialog_add_widget(zd,"hbox","hb3","vb3");
   zdialog_add_widget(zd,"label","lab31","hb3","darker bands");
   zdialog_add_widget(zd,"label","lab32","hb3",0,"expand");
   zdialog_add_widget(zd,"label","lab33","hb3","brighter bands");
   zdialog_add_widget(zd,"hbox","hb4","dialog",0,"space=15");
   zdialog_add_widget(zd,"label","lab41","hb4",0,"space=50");
   zdialog_add_widget(zd,"button","100/0","hb4","100/0");
   zdialog_add_widget(zd,"button","0/100","hb4","0/100");
   zdialog_add_widget(zd,"button","50/50","hb4","50/50");
   zdialog_add_widget(zd,"button","+/-","hb4","  +/-  ");
   zdialog_add_widget(zd,"button","-/+","hb4","  -/+  ");
   zdialog_add_widget(zd,"hbox","hb5","dialog",0,"space=4");
   zdialog_add_widget(zd,"button","commit","hb5","commit","space=20");

   bravg = 0;                                                              //  get mean brightness ratio
   for (ii = 0; ii < 256; ii++)
      bravg = bravg + Bratios[0][ii] + Bratios[1][ii] + Bratios[2][ii];
   bravg = bravg / 256 / 3;
   
   for (ii = 0; ii < 9; ii++)                                              //  initial ramp: use brighter
   {                                                                       //    image for darker bands, etc.
      vsname[2] = '0' + ii;
      if (bravg < 1) wval = 10 * ii;                                       //  ramp up                      v.28
      else  wval = 80 - 10 * ii;                                           //  ramp down
      zdialog_stuff(zd,vsname,wval);
      hdrw[ii] = wval;
   }

   HDR_prep_preview();                                                     //  prepare preview pixbufs      v.28
   HDR_make_preview();                                                     //  make preview image
   pvwin_create();                                                         //  display preview window

   zdialog_resize(zd,0,400);
   zstat = zdialog_run(zd,HDR_level_dialog_event);                         //  run dialog
   zdialog_destroy(zd);
   
   pvwin_destroy();                                                        //  kill preview window

   pixbuf_free(pxbP);                                                      //  free preview memory
   pixbuf_free(pxbP1);
   pixbuf_free(pxbP2);

   if (zstat == 1) HDR_combine_images(1);                                  //  done
   else undo_pull();                                                       //  cancel, restore image3
   return;
}


//  callback function for HDR image level adjustment

int HDR_level_dialog_event(zdialog *zd, const char *event)
{
   char        vsname[4] = "vs0";
   double      vsval;
   
   for (int ii = 0; ii < 9; ii++)                                          //  process dialog sliders     v.28
   {
      vsname[2] = '0' + ii;
      zdialog_fetch(zd,vsname,vsval);                                      //  get slider ii value

      if (strEqu(event,"100/0")) vsval = 0;                                //  button updates
      if (strEqu(event,"0/100")) vsval = 80;
      if (strEqu(event,"50/50")) vsval = 40;
      if (strEqu(event,"+/-"))   vsval += ii - 4;
      if (strEqu(event,"-/+"))   vsval += 4 - ii;

      if (vsval < 0) vsval = 0;                                            //  stay within limits
      if (vsval > 80) vsval = 80;

      zdialog_stuff(zd,vsname,vsval);                                      //  update dialog slider
      hdrw[ii] = vsval;                                                    //  and corresp. weight
   }
   
   HDR_make_preview();                                                     //  generate new preview image
   pvwin_paint();                                                          //  update window
   
   if (strEqu(event,"commit")) HDR_combine_images(1);                      //  combine images using weights

   return 1;
}


//  Prepare two small preview images which can be rapidly combined
//  with adjustable weights in the HDR interactive preview dialog.
//  inputs: alignment images: pxb1A and pxb2A (with alignment offsets)
//  outputs: aligned preview images: pxbP1, pxbP2

void HDR_prep_preview()                                                    //  v.28
{
   GdkPixbuf   *pxb91, *pxb92;
   int         px1, py1;
   double      ratio, px2, py2;
   double      sintf = sin(toff), costf = cos(toff);
   pixel       pix1, pix2, pix91, pix92;
   int         ww91, hh91, rs91, ww92, hh92, rs92;
   pixel       ppix91, ppix92;

   zlock();   

   ratio = 1.0 * pvwinsize / ww3;                                          //  create preview pixbuf
   if (hh3 > ww3) ratio = 1.0 * pvwinsize / hh3;
   if (ww3 < pvwinsize && hh3 < pvwinsize) ratio = 1.0;
   wwP = int(ww3 * ratio);
   hhP = int(hh3 * ratio);
   pxbP = pixbuf_new(colorspace,0,8,wwP,hhP);
   pixbuf_test(pxbP);
   pixbuf_poop(P);

   pxb91 = pixbuf_new(colorspace,0,8,ww1A,hh1A);                           //  create temp pixbufs
   pxb92 = pixbuf_new(colorspace,0,8,ww2A,hh2A);
   pixbuf_test(pxb91);
   pixbuf_test(pxb92);
   pixbuf_poop(91);
   pixbuf_poop(92);

   zunlock();

   for (py1 = 0; py1 < hh1A; py1++)                                        //  step through pxb1A pixels
   for (px1 = 0; px1 < ww1A; px1++)
   {
      px2 = (px1 - xoff) * costf + (py1 - yoff) * sintf;                   //  corresponding pxb2A pixels
      py2 = (py1 - yoff) * costf - (px1 - xoff) * sintf;
   
      pix1 = ppix1A + py1 * rs1A + px1 * nch;                              //  input pixel 1
      pix2 = vpixel(ppix2A,px2,py2,ww2A,hh2A,rs2A);                        //  input pixel 2
      
      pix91 = ppix91 + py1 * rs91 + px1 * nch;                             //  output pixels
      pix92 = ppix92 + py1 * rs92 + px1 * nch;

      if (! pix2) {                                                        //  if non-overlapping pixel,
         pix91[0] = pix91[1] = pix91[2] = 0;                               //    set output pixels black
         pix92[0] = pix92[1] = pix92[2] = 0;
         continue;
      }
      
      pix91[0] = pix1[0];                                                  //  build output pixels
      pix91[1] = pix1[1];
      pix91[2] = pix1[2];
      
      pix92[0] = pix2[0];
      pix92[1] = pix2[1];
      pix92[2] = pix2[2];
   }

   zlock();
   pxbP1 = pixbuf_scale_simple(pxb91,wwP,hhP,interp);                      //  scale to preview image size
   pixbuf_test(pxbP1);
   pxbP2 = pixbuf_scale_simple(pxb92,wwP,hhP,interp);
   pixbuf_test(pxbP2);
   pixbuf_poop(P1);
   pixbuf_poop(P2);
   
   pixbuf_free(pxb91);                                                     //  free temp pixbufs
   pixbuf_free(pxb92);
   zunlock();
   
   return;
}


//  generate preview image using input image weights in hdrw[9]

void HDR_make_preview()                                                    //  v.28
{
   int      ii, px1, py1;
   pixel    pix1, pix2, pix3;
   double   br1, br2, brm;

   for (py1 = 0; py1 < hhP; py1++)
   for (px1 = 0; px1 < wwP; px1++)
   {
      pix1 = ppixP1 + py1 * rsP1 + px1 * nch;
      pix2 = ppixP2 + py1 * rsP2 + px1 * nch;
      pix3 = ppixP + py1 * rsP + px1 * nch;

      br1 = pix1[0] + pix1[1] + pix1[2];                                   //  image 1 pixel brightness
      br2 = pix2[0] + pix2[1] + pix2[2];                                   //  image 2
      brm = (br1 + br2) / 2;                                               //  mean, 0 to 765

      ii = int(brm / 96);                                                  //  brightness band, 0 .. 7
      br1 = hdrw[ii];                                                      //  weight at low end of band
      br2 = hdrw[ii+1];                                                    //  weight at high end of band
      brm = br1 + (br2 - br1) * (brm - 96 * ii) / 96;                      //  interpolate within band
      
      br1 = brm / 80.0;                                                    //  image 1 weight: 0 .. 1.0
      br2 = 1.0 - br1;                                                     //  image 2 weight

      pix3[0] = int(pix1[0] * br1 + pix2[0] * br2);                        //  build output pixel
      pix3[1] = int(pix1[1] * br1 + pix2[1] * br2);
      pix3[2] = int(pix1[2] * br1 + pix2[2] * br2);
   }

   return;
}


/**************************************************************************/

//  panorama function - stitch two images together

void *   pano_thread(void *);                                              //  work function
void     pano_get_align_images();                                          //  rescale images for alignment
void     pano_curve_images(int newf, int strf);                            //  curve/stretch image during alignment
void     pano_interp_curve();                                              //  convert lens mm to % curve
void     pano_prealign();                                                  //  manual pre-align
void     pano_autolens();                                                  //  optimize lens parameters
void     pano_show_images(int fcolor);                                     //  show combined images
void     pano_final_adjust();                                              //  dialog, adjust color and brightness

int      pano_align_stat = 0;                                              //  alignment status
double   Radjust, Gadjust, Badjust;                                        //  manual color adjustments


void m_pano()
{
   pthread_t   tid;

   if (! pxb3) return;                                                     //  no 1st image

   if (file2) zfree(file2);
   file2 = zgetfile("select image to stitch with first",file1,"open");     //  get 2nd pano image in pxb2
   if (! file2) return;
   pxb2 = pixbuf_new_from_file(file2,gerror);
   if (! pxb2) goto err_nomatch;
   pixbuf_poop(2)

   if (gdk_pixbuf_get_n_channels(pxb2) != nch) goto err_nomatch;           //  validate compatibility
   if (gdk_pixbuf_get_bits_per_sample(pxb2) != nbits) goto err_nomatch;

   pixbuf_free(pxb1);                                                      //  use image3 for 1st pano image
   pxb1 = pixbuf_copy(pxb3);                                               //  (preserve mods)
   pixbuf_test(pxb1);
   pixbuf_poop(1)

   undo_push();                                                            //  save image3 undo point
   ++thread_busy;                                                          //  launch thread to combine
   pthread_create(&tid,0,pano_thread,0);                                   //    pxb1 + pxb2 >> pxb3
   return;                                                                 //  thread runs on from here

err_nomatch:
   zmessageACK("second image file is incompatible with first");
   if (file2) zfree(file2);
   file2 = 0;
   pixbuf_free(pxb2);
   pxb2 = 0;
   return;
}


//  Thread function for combining pxb1 + pxb2 >> pxb3.

void * pano_thread(void *)
{
   int         firstpass = 1, lastpass = 0;
   double      minblend, alignR;
   double      xfL, xfH, yfL, yfH, tfL, tfH;
   double      yst1L, yst1H, yst2L, yst2H;

   Nalign = 1;                                                             //  alignment in progress
   showRedpix = 0;                                                         //  no pixel highlight (yet)
   pixsamp = 1000 * int(parm_pixel_sample_size);                           //  no. pixels to sample

   fullSize = hh2;                                                         //  full size = image2 height
   
   alignSize = int(parm_pano_prealign_size);                               //  prealign image size
   pano_get_align_images();                                                //  get prealign images

   xoff = xoffB = 0.8 * ww1A;                                              //  initial x offset (20% overlap)
   yoff = yoffB = toff = toffB = 0;                                        //  initial y and theta offsets
   yst1 = yst1B = yst2 = yst2B = 0;                                        //  initial y-stretch offsets
   lens_mm = parm_pano_lens_mm;                                            //  initial image curvature   v.28
   lens_bow = parm_pano_lens_bow;                                          //  initial image bow (barrel distortion)
   blend = ww1A - xoff;                                                    //  initial blend width = overlap area
   
   pano_curve_images(1,0);                                                 //  make curved alignment images
   get_overlap_region(pxb1C,pxb2C);                                        //  get image overlap area
   pano_show_images(0);                                                    //  show combined images

   pano_prealign();                                                        //  manual pre-align - get new offsets
   if (! pano_align_stat) goto pano_exit;                                  //  user canceled

   blend = ww1A - xoff;                                                    //  new blend from pre-align  v.22
   if (blend < 20) {
      zmessageACK("too little overlap, cannot align");
      pano_align_stat = 0;
      goto pano_exit;
   }

   showRedpix = 1;                                                         //  highlight alignment pixels

   while (true)
   {
      alignR = alignSize;
      if (firstpass) alignSize = 100;
      else if (lastpass) alignSize = fullSize;
      else  alignSize = int(parm_pano_align_size_increase * alignSize);    //  next larger image size
      if (alignSize > 0.8 * fullSize) alignSize = fullSize;                //  if near goal, jump to it now
      alignR = alignSize / alignR;                                         //  ratio of new / old image size

      xoff = alignR * xoff;                                                //  adjust offsets for new image size
      yoff = alignR * yoff;
      toff = toff;
      yst1 = alignR * yst1;
      yst2 = alignR * yst2;
      blend = alignR * blend;

      xoffB = xoff;                                                        //  best offsets so far = initial offsets
      yoffB = yoff;
      toffB = toff;
      yst1B = yst1;
      yst2B = yst2;

      if (! firstpass) {
         blend = blend * parm_pano_blend_reduction;                        //  reduce image comparison width
         minblend = 0.01 * parm_pano_minimum_blend * alignSize;
         if (blend < minblend) blend = minblend;                           //  keep above minimum
         if (blend > 0.2 * alignSize) blend = 0.2 * alignSize;             //  keep below 20% of image
      }
      
      xylim = 3;                                                           //  +/- search range, centered on   v.27
      xystep = 1;                                                          //    results from prior stage
      
      if (firstpass) xylim = alignSize * 0.05;                             //  tolerate 5 % error in pre-alignment

      if (lastpass) {
         xylim = 1;                                                        //  final stage pixel search steps
         xystep = 0.5;                                                     //   -1.0 -0.5 0.0 +0.5 +1.0
      }

      tlim = xylim / alignSize;                                            //  theta max offset, radians
      tstep = xystep / alignSize;                                          //  theta step size

      if (! lastpass) {
         pano_get_align_images();                                          //  get new alignment images
         pano_curve_images(1,0);                                           //  make new curved alignment images
      }

      get_overlap_region(pxb1C,pxb2C);                                     //  get image overlap area
      get_Bratios(pxb1C,pxb2C);                                            //  get image brightness ratios in overlap 
      flag_edge_pixels(pxb1C);                                             //  flag high-contrast pixels in blend area

      if (kill_thread) goto pano_exit;

      zlock();
      gdk_window_freeze_updates(drmwin->window);                           //  inhibit window flashing
      pano_show_images(0);                                                 //  show combined images
      gdk_window_thaw_updates(drmwin->window);
      zunlock();
      
      xfL = xoff - xylim;                                                  //  set x/y/t search ranges and step sizes
      xfH = xoff + xylim + xystep/2;
      yfL = yoff - xylim;
      yfH = yoff + xylim + xystep/2;
      tfL = toff - tlim;
      tfH = toff + tlim + tstep/2;

      matchB = match_images(pxb1C,pxb2C);                                  //  set base match level

      for (xoff = xfL; xoff < xfH; xoff += xystep)                         //  test x, y, theta offsets
      for (yoff = yfL; yoff < yfH; yoff += xystep)                         //    in all possible combinations
      for (toff = tfL; toff < tfH; toff += tstep)
      {
         matchlev = match_images(pxb1C,pxb2C);
         if (sigdiff(matchlev,matchB,0.00001) > 0) {                       //  remember best alignment and offsets
            matchB = matchlev;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
         }

         Nalign++;
         update_status_bar();
         if (kill_thread) goto pano_exit;
      }
      
      xoff = xoffB;                                                        //  recover best offsets
      yoff = yoffB;
      toff = toffB;
      
      pano_show_images(0);                                                 //  refresh window

      if (parm_pano_image_stretch && ! firstpass)                          //  do image2 y-stretch alignment
      {
         ystlim = hh2A * 0.001 * parm_pano_image_stretch;                  //  v.32
         yststep = 0.4;

         yst1L = yst1 - ystlim;                                            //  y-stretch search range, upper half
         yst1H = yst1 + ystlim + yststep/2;

         for (yst1 = yst1L; yst1 < yst1H; yst1 += yststep)                 //  search upper image half
         {
            pano_curve_images(0,1);                                        //  curve and y-stretch
            matchlev = match_images(pxb1C,pxb2C);
            if (sigdiff(matchlev,matchB,0.00001) > 0) {                    //  remember best alignment and y-stretch 
               matchB = matchlev;
               yst1B = yst1;
            }

            Nalign++;
            update_status_bar();
            if (kill_thread) goto pano_exit;
         }
         
         yst1 = yst1B;                                                     //  restore best y-stretch offset
         pano_curve_images(0,1);

         yst2L = yst2 - ystlim;                                            //  lower half search range
         yst2H = yst2 + ystlim + yststep/2;
         
         for (yst2 = yst2L; yst2 < yst2H; yst2 += yststep)                 //  search lower half
         {
            pano_curve_images(0,2);
            matchlev = match_images(pxb1C,pxb2C);
            if (sigdiff(matchlev,matchB,0.00001) > 0) {
               matchB = matchlev;
               yst2B = yst2;
            }

            Nalign++;
            update_status_bar();
            if (kill_thread) goto pano_exit;
         }
         
         yst2 = yst2B;                                                     //  restore best offset
         pano_curve_images(0,0);
         pano_show_images(0);                                              //  refresh window
      }

      firstpass = 0;
      if (lastpass) break;
      if (alignSize == fullSize) lastpass = 1;                             //  one more pass with reduced step size
   }

   pano_final_adjust();                                                    //  user color and brightness adjustments

pano_exit:

   Nalign = 0;                                                             //  alignment done
   if (kill_thread || ! pano_align_stat) undo_pull();                      //  canceled or killed

   zlock();
   if (file2) zfree(file2);                                                //  free 2nd input file
   file2 = 0;
   pixbuf_free(pxb2);                                                      //  free 2nd input image
   pixbuf_free(pxb1A);                                                     //  free alignment images
   pixbuf_free(pxb2A);
   pixbuf_free(pxb1C);
   pixbuf_free(pxb2C);
   zunlock();

   mwpaint2(1);
   --thread_busy;                                                          //  exit thread
   return 0;
}


//  scale images into new pixbufs for alignment
//  scaled images in pxb1A/2A, scaled and curved images in pxb1C/2C

void pano_get_align_images()
{
   ww1A = ww1 * alignSize / fullSize;                                      //  size of alignment images
   hh1A = hh1 * alignSize / fullSize;
   ww2A = ww2 * alignSize / fullSize;
   hh2A = hh2 * alignSize / fullSize;

   zlock();

   pixbuf_free(pxb1A);                                                     //  create alignment images
   pixbuf_free(pxb2A);
   pxb1A = pixbuf_scale_simple(pxb1,ww1A,hh1A,interp);
   pixbuf_test(pxb1A);
   pxb2A = pixbuf_scale_simple(pxb2,ww2A,hh2A,interp);
   pixbuf_test(pxb2A);

   pixbuf_poop(1A)                                                         //  get pxb1A attributes
   pixbuf_poop(2A)                                                         //  get pxb2A attributes

   zunlock();
   return;
}


//  curve and stretch images according to lens_mm, lens_bow, yst1, yst2
//  pxb1A, pxb2A  >>  pxb1C, pxb2C
//  strf:  0: curve both images, y-stretch both halves of image2
//         1: curve image2 only, y-stretch upper half only
//         2: curve image2 only, y-stretch lower half only
//  v.32: curve entire image instead of right or left half

void pano_curve_images(int newf, int strf)
{
   pixel       pix, pixc;
   int         pxc, pyc, pxcL, pxcH, pycL, pycH;
   double      px, py, yst;
   double      xdisp, ydisp, xshrink, yshrink;
   double      costf = cos(toff), sintf = sin(toff);
   double      pcurve, pyadj, pbow, R;

   if (newf)                                                               //  new alignment images are present
   {
      zlock();
      pixbuf_free(pxb1C);                                                  //  make new curved alignment images
      pxb1C = pixbuf_copy(pxb1A);
      pixbuf_test(pxb1C);
      pixbuf_free(pxb2C);
      pxb2C = pixbuf_copy(pxb2A);
      pixbuf_test(pxb2C);

      pixbuf_poop(1C)                                                      //  get attributes
      pixbuf_poop(2C)
      zunlock();
   }
   
   pano_interp_curve();                                                    //  convert lens_mm to curve    v.28
   pcurve = lens_curve * 0.01 * ww2C;                                      //  curve % to pixels
   pyadj = 1.41;                                                           //  y adjustment factor       v.32
   pbow = 0.01 * ww2C * lens_bow;                                          //  lens_bow % to pixels

   R = 1.0 * hh2C / ww2C;
   if (R > 1) pcurve = pcurve / R / R;                                     //  adjust for vertical format     v.28

   if (strf == 0 && ww2C > 0.7 * ww1C)                                     //  no curve if 3rd+ image in pano    v.32
   {   
      for (pyc = 0; pyc < hh1C; pyc++)                                     //  curve image1 (both halves)   v.32
      for (pxc = 0; pxc < ww1C; pxc++)
      {
         xdisp = (pxc - ww1C/2.0) / (ww1C/2.0);                            //  -1 .. 0 .. +1
         ydisp = (pyc - hh1C/2.0) / (ww1C/2.0);                            //  -h/w .. 0 .. +h/w
         xshrink = xdisp * xdisp * xdisp;
         yshrink = pyadj * ydisp * xdisp * xdisp;
         
         px = pxc + pcurve * xshrink;                                      //  x shrink pixels
         py = pyc + pcurve * yshrink;                                      //  y shrink pixels
         px -= pbow * xdisp * ydisp * ydisp;                               //  x bow pixels

         pix = vpixel(ppix1A,px,py,ww1A,hh1A,rs1A);                        //  source virtual pixel in straight image
         pixc = ppix1C + pyc * rs1C + pxc * nch;                           //  destination pixel in curved image
         if (pix) {  
            pixc[0] = pix[0];
            pixc[1] = pix[1];
            pixc[2] = pix[2];
         }
         else pixc[0] = pixc[1] = pixc[2] = 0;
      }
   }
   
   pycL = 0;
   pycH = hh2C;
   if (strf == 1) pycH = hh2C / 2;                                         //  upper half only
   if (strf == 2) pycL = hh2C / 2;                                         //  lower half only

   pxcL = 0;
   pxcH = ww2C;

   if (strf) {                                                             //  during y-stretch alignment,     v.27
      pxcL = int((pxmL - xoff) * costf + (pyL - yoff) * sintf);            //    stay within blend stripe
      if (pxcL < 0) pxcL = 0;
      pxcH = int((pxmH - xoff) * costf + (pyH - yoff) * sintf);
      if (pxcH > ww2C/2) pxcH = ww2C/2;
   }

   for (pyc = pycL; pyc < pycH; pyc++)                                     //  curve and y-stretch image2
   for (pxc = pxcL; pxc < pxcH; pxc++)
   {
      xdisp = (pxc - ww2C/2.0) / (ww2C/2.0);
      ydisp = (pyc - hh2C/2.0) / (ww2C/2.0);
      xshrink = xdisp * xdisp * xdisp;
      yshrink = pyadj * ydisp * xdisp * xdisp;

      px = pxc + pcurve * xshrink;
      py = pyc + pcurve * yshrink;
      px -= pbow * xdisp * ydisp * ydisp;

      if (pyc < hh2C / 2) yst = yst1;                                      //  do y-stretch
      else yst = yst2;
      if (yst) py += yst * ydisp;

      pix = vpixel(ppix2A,px,py,ww2A,hh2A,rs2A);
      pixc = ppix2C + pyc * rs2C + pxc * nch;
      if (pix) {  
         pixc[0] = pix[0];
         pixc[1] = pix[1];
         pixc[2] = pix[2];
      }
      else pixc[0] = pixc[1] = pixc[2] = 0;
   }

   return;
}


//  convert lens focal length (mm) to image percent curve                  v.28

void pano_interp_curve()
{
   double curvetab[] = { 25, 10.3, 26, 8.4, 28, 6.78, 32, 5.26,            //  25mm = 10.3 % etc.   v.32
                         40, 3.42, 48, 2.32, 60, 1.7, 80, 1.35, 
                         100, 1.2, 200, 1.1, 0 };
   int      ii;
   double   x0, x1, y0, y1;
   
   for (ii = 0; curvetab[ii]; ii += 2)                                     //  find >= table entry
      if (lens_mm <= curvetab[ii]) break;

   if (ii == 0) {                                                          //  1st entry
      lens_curve = curvetab[1];
      return;
   }

   if (curvetab[ii] == 0) {                                                //  EOL
      lens_curve = 1.0;
      return;
   }
   
   x0 = curvetab[ii-2];                                                    //  interpolate
   x1 = curvetab[ii];
   y0 = curvetab[ii-1];
   y1 = curvetab[ii+1];
   lens_curve = y0 + (y1 - y0) * (lens_mm - x0) / (x1 - x0);

   return;
}


//  perform manual pre-align of image2 to image1
//  return offsets: xoff, yoff, toff, lens_mm, lens_bow

void pano_prealign()   
{  
   int pano_prealign_dialog_event(zdialog *zd, const char *event);         //  dialog event function
   int pano_prealign_dialog_compl(zdialog *zd, int zstat);                 //  dialog completion function

   zdialog     *zd;
   int         mx0, my0, mx, my, key;                                      //  mouse drag origin, position
   double      lens_mm0, lens_bow0;
   double      mlever = 1.0 / parm_pano_mouse_leverage;
   double      dtoff;

   char        *align_message = "drag right image into rough alignment with left \n"
                                "(to rotate, drag right edge up or down)";
   char        *next_message = "merge the images together";
   char        *auto_message = "auto-search lens mm and bow";

   zd = zdialog_new("pre-align images","cancel",0);                        //  dialog for alignment
   zdialog_add_widget(zd,"label","lab1","dialog",align_message,"space=5");
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");               //  lens mm, 25-200 mm
   zdialog_add_widget(zd,"spin","spmm","hb1","25|200|0.1|35","space=5");
   zdialog_add_widget(zd,"label","labmm","hb1","lens mm");
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");               //  lens bow, -9 to +9 %
   zdialog_add_widget(zd,"spin","spbow","hb2","-9|9|0.01|0","space=5");
   zdialog_add_widget(zd,"label","labbow","hb2","lens bow");
   zdialog_add_widget(zd,"hbox","hb3","dialog",0,"space=5");               //  next button
   zdialog_add_widget(zd,"button","proceed","hb3","proceed","space=5");
   zdialog_add_widget(zd,"label","labauto","hb3",next_message);
   zdialog_add_widget(zd,"hbox","hb4","dialog",0,"space=5");               //  auto lens button     v.32
   zdialog_add_widget(zd,"button","auto lens","hb4","auto lens","space=5");
   zdialog_add_widget(zd,"label","labauto","hb4",auto_message);
   zdialog_stuff(zd,"spmm",lens_mm);                                       //  pre-load current data
   zdialog_stuff(zd,"spbow",lens_bow);
   zdialog_run(zd,pano_prealign_dialog_event,pano_prealign_dialog_compl);  //  run dialog

   lens_mm0 = lens_mm;                                                     //  to detect changes
   lens_bow0 = lens_bow;
   pano_align_stat = 1;                                                    //  status = pre-align in progress
   mx0 = my0 = 0;                                                          //  no drag in progress

   while (pano_align_stat == 1)                                            //  loop while in align status
   {
      if (Fautolens) {
         pano_autolens();                                                  //  get lens parameters   v.28
         if (pano_align_stat != 1) break;
         zdialog_stuff(zd,"spmm",lens_mm);                                 //  update dialog
         zdialog_stuff(zd,"spbow",lens_bow);
      }
         
      if (lens_mm != lens_mm0 || lens_bow != lens_bow0) {                  //  change in lens parameters
         lens_mm0 = lens_mm;
         lens_bow0 = lens_bow;
         pano_curve_images(0,0);
         pano_show_images(0);                                              //  show combined images
      }
      
      if (KBkey) {                                                         //  KB input   v.28
         key = KBkey;
         KBkey = 0;
         if (key == GDK_Left)  xoff -= 0.2;                                //  tweak alignment offsets
         if (key == GDK_Right) xoff += 0.2;
         if (key == GDK_Up)    yoff -= 0.2;
         if (key == GDK_Down)  yoff += 0.2;
         if (key == GDK_r)     toff += 0.0002;
         if (key == GDK_l)     toff -= 0.0002;
         if (key == GDK_p) printf(" %.1f %.2f \n",lens_mm,lens_curve);     //  print curve value

         blend = ww1C - xoff;                                              //  entire overlap
         get_overlap_region(pxb1C,pxb2C);
         pano_show_images(0);                                              //  show combined images
      }

      if (! mdx1 && ! mdy1) {
         mx0 = my0 = 0;                                                    //  no drag in progress
         goto idle;
      }

      mx = mp3x;                                                           //  mouse position in image3
      my = mp3y;

      if (mx < xoff || mx > xoff + ww2C || my > hh3) {                     //  if mouse not in image2 area,
         mx0 = my0 = 0;                                                    //    no drag in progress
         goto idle;
      }

      if (mx0 == 0 && my0 == 0) {                                          //  new drag origin
         mx0 = mx;
         my0 = my;
         goto idle;
      }
      
      if (mx == mx0 && my == my0) goto idle;                               //  no movement

      if (mx > xoff + 0.8 * ww2C) {                                        //  near right edge, theta drag 
         dtoff = mlever * (my - my0) / ww2C;                               //  delta theta, radians
         toff += dtoff;
         xoff += 0.5 * dtoff * hh2C;                                       //  change center of rotation
         yoff -= 0.5 * dtoff * (ww1C - xoff);                              //    to middle of overlap area
      }
      else  {                                                              //  x/y drag
         xoff += mlever * (mx - mx0);                                      //  image2 offsets / mouse leverage
         yoff += mlever * (my - my0);
      }

      mx0 = mx;                                                            //  reset drag origin
      my0 = my;

      if (xoff > ww1C) xoff = ww1C;                                        //  limit nonsense
      if (xoff < 0.3 * ww1C) xoff = 0.3 * ww1C;
      if (yoff < -0.5 * hh2C) yoff = -0.5 * hh2C;
      if (yoff > 0.5 * hh1C) yoff = 0.5 * hh1C;
      if (toff < -0.20) toff = -0.20;
      if (toff > 0.20) toff = 0.20;

      blend = ww1C - xoff;                                                 //  entire overlap
      get_overlap_region(pxb1C,pxb2C);
      pano_show_images(0);                                                 //  show combined images
      zsleep(0.05);
      continue;

idle:                                                                      //  spare time activity
      get_overlap_region(pxb1C,pxb2C);                                     //  get image overlap area
      get_Bratios(pxb1C,pxb2C);                                            //  get image brightness ratios in overlap 
      flag_edge_pixels(pxb1C);                                             //  flag high-contrast pixels in overlap 
      matchB = match_images(pxb1C,pxb2C);                                  //  match images
      xoffB = xoff;
      yoffB = yoff;
      toffB = toff;
      update_status_bar();
      zsleep(0.1);
      if (kill_thread) break;                                              //  v.32
   }

   Mscale = 0;                                                             //  scale = fit to window
   return;
}

int pano_prealign_dialog_event(zdialog *zd, const char *event)             //  dialog event callback function
{
   if (strEqu(event,"spmm")) zdialog_fetch(zd,"spmm",lens_mm);             //  get revised lens data
   if (strEqu(event,"spbow")) zdialog_fetch(zd,"spbow",lens_bow);
   if (strEqu(event,"auto lens")) Fautolens = 1;                           //  trigger auto-lens function

   if (strEqu(event,"proceed")) {                                          //  proceed with pano
      pano_align_stat = 2;
      zdialog_destroy(zd);
   }
   return 0;
}

int pano_prealign_dialog_compl(zdialog *zd, int zstat)                     //  dialog completion callback function
{
   zdialog_destroy(zd);                                                    //  (only if cancel button used)
   pano_align_stat = 0;
   return 0;
}


//  optimize lens parameters                                               v.32
//  assumes a good starting point since search ranges are limited
//  inputs and outputs: lens_mm, lens_bow, xoff, yoff, toff

void pano_autolens()
{
   double   mm_range, bow_range, xoff_range, yoff_range, toff_range;
   double   squeeze, xoff_rfinal, rnum;
   int      counter = 0;
   
   mm_range = 0.1 * lens_mm;                                               //  set initial search ranges
   bow_range = 0.5 * lens_bow;
   if (bow_range < 1) bow_range = 1;
   xoff_range = 7;
   yoff_range = 7;
   toff_range = 0.01;

   xoff_rfinal = 0.2;                                                      //  final xoff range - when to quit
   
   Nalign = 0;
   showRedpix = 1;
   pano_curve_images(0,0);
   get_overlap_region(pxb1C,pxb2C);
   get_Bratios(pxb1C,pxb2C);
   flag_edge_pixels(pxb1C);
   pano_show_images(0);

   lens_mmB = lens_mm;                                                     //  initial best fit = current data
   lens_bowB = lens_bow;
   xoffB = xoff;
   yoffB = yoff;
   toffB = toff;
   matchB = match_images(pxb1C,pxb2C);
   
   while (true)
   {
      srand48(time(0) + counter++);
      squeeze = 0.9;                                                       //  search range reduction factor

      lens_mm = lens_mmB + mm_range * (drand48() - 0.5);                   //  new random lens factors
      lens_bow = lens_bowB + bow_range * (drand48() - 0.5);                //     within search range
      pano_curve_images(0,0);
         
      for (int ii = 0; ii < 300; ii++)
      {                                                                    
         rnum = drand48();

         if (rnum < 0.33)                                                  //  random change some alignment offset 
            xoff = xoffB + xoff_range * (drand48() - 0.5);                 //    within search range
         else if (rnum < 0.67)
            yoff = yoffB + yoff_range * (drand48() - 0.5);
         else
            toff = toffB + toff_range * (drand48() - 0.5);
      
         matchlev = match_images(pxb1C,pxb2C);                             //  test quality of image alignment
         if (sigdiff(matchlev,matchB,0.00001) > 0) {
            get_overlap_region(pxb1C,pxb2C);                               //  better: reset alignment data
            get_Bratios(pxb1C,pxb2C);
            flag_edge_pixels(pxb1C);
            matchB = match_images(pxb1C,pxb2C);                            //  save new best fit
            lens_mmB = lens_mm;
            lens_bowB = lens_bow;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
            pano_show_images(0);                                           //  update window
            squeeze = 1;                                                   //  keep same search range as long
            break;                                                         //    as improvements are found
         }

         Nalign++;
         update_status_bar();
         if (kill_thread) goto done;
         if (pano_align_stat != 1) goto done;
      }

      if (xoff_range < xoff_rfinal) goto done;

      mm_range = squeeze * mm_range;                                       //  reduce search range if no 
      if (mm_range < 0.02 * lens_mmB) mm_range = 0.02 * lens_mmB;          //    improvements were found
      bow_range = squeeze * bow_range;
      if (bow_range < 0.1 * lens_bowB) bow_range = 0.1 * lens_bowB;
      if (bow_range < 0.2) bow_range = 0.2;
      xoff_range = squeeze * xoff_range;
      yoff_range = squeeze * yoff_range;
      toff_range = squeeze * toff_range;
   }

done:
   lens_mm = lens_mmB;                                                     //  set best alignment found
   lens_bow = lens_bowB;
   xoff = xoffB;
   yoff = yoffB;
   toff = toffB;
   pano_curve_images(0,0);
   pano_show_images(0);
   update_status_bar();

   zmessageACK("lens mm: %.1f  bow: %.2f",lens_mm,lens_bow);
   Fautolens = 0;
   kill_thread = 0;
   showRedpix = 0;
   return;
}


//  combine images and update window during alignment                      v.20 overhauled
//    pxb1C + pxb2C  >>  pxb3

void pano_show_images(int fcolor)
{
   int            px3, py3, ii, max;
   int            red1, green1, blue1;
   int            red2, green2, blue2;
   int            red3, green3, blue3;
   pixel          pix1, pix2, pix3;
   double         ww, px2, py2, f1, f2;
   double         costf = cos(toff), sintf = sin(toff);

   ww = xoff + ww2C;                                                       //  combined width
   if (toff < 0) ww -= hh2C * toff;                                        //  adjust for theta
   ww3 = int(ww+1);
   hh3 = hh1A;
   
   zlock();
   pixbuf_free(pxb3);
   pxb3 = pixbuf_new(colorspace,0,8,ww3,hh3);
   pixbuf_test(pxb3);
   pixbuf_poop(3)                                                          //  get pxb3 attributes
   zunlock();

   red1 = green1 = blue1 = 0;                                              //  suppress compiler warnings
   red2 = green2 = blue2 = 0;

   for (py3 = 0; py3 < hh3; py3++)                                         //  step through pxb3 rows
   for (px3 = 0; px3 < ww3; px3++)                                         //  step through pxb3 pixels in row
   {
      pix1 = pix2 = 0;
      red3 = green3 = blue3 = 0;

      if (px3 < pxmH && px3 < ww1C && py3 < hh1C)
         pix1 = ppix1C + py3 * rs1C + px3 * nch;                           //  pxb1C pixel

      if (px3 >= pxmL) {
         px2 = costf * (px3 - xoff) + sintf * (py3 - yoff);                //  pxb2C pixel, after offsets
         py2 = costf * (py3 - yoff) - sintf * (px3 - xoff);
         pix2 = vpixel(ppix2C,px2,py2,ww2C,hh2C,rs2C);
      }
      
      if (pix1) {
         red1 = pix1[0];                                                   //  image1 pixel
         green1 = pix1[1];
         blue1 = pix1[2];
         if (!red1 && !green1 && !blue1) pix1 = 0;
      }

      if (pix2) {                                                          //  image2 pixel
         red2 = pix2[0];
         green2 = pix2[1];
         blue2 = pix2[2];
         if (!red2 && !green2 && !blue2) pix2 = 0;
      }
      
      if (fcolor && pix2) {                                                //  color compensate image2 pixel
         red2 = int(brightfix0[red2] * Radjust);                           //  (auto + manual adjustments)
         green2 = int(brightfix1[green2] * Gadjust);
         blue2 = int(brightfix2[blue2] * Badjust);
         if (red2 > 255 || green2 > 255 || blue2 > 255) {                  //  fix overflow        v.26
            max = red2;
            if (green2 > max) max = green2;
            if (blue2 > max) max = blue2;
            f1 = 255.0 / max;
            red2 = int(red2 * f1);
            green2 = int(green2 * f1);
            blue2 = int(blue2 * f1);
         }
      }

      if (pix1) {
         if (! pix2) {
            red3 = red1;                                                   //  use image1 pixel
            green3 = green1;
            blue3 = blue1; 
         }
         else {                                                            //  use blended pixel
            if (fcolor) {
               if (blend == 0) f1 = 1.0;
               else f1 = (pxmH - px3) / blend;                             //  use progressive blend
               f2 = 1.0 - f1;
               red3 = int(f1 * red1 + f2 * red2);
               green3 = int(f1 * green1 + f2 * green2);
               blue3 = int(f1 * blue1 + f2 * blue2);
            }
            else {                                                         //  use 50/50 mix
               red3 = (red1 + red2) / 2;
               green3 = (green1 + green2) / 2;
               blue3 = (blue1 + blue2) / 2;
            }
         }
      }

      else if (pix2) {
            red3 = red2;                                                   //  use image2 pixel
            green3 = green2;
            blue3 = blue2; 
      }

      pix3 = ppix3 + py3 * rs3 + px3 * nch;                                //  output pixel
      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
      
      if (showRedpix && pix1 && pix2) {                                    //  highlight alignment pixels   v.25
         ii = py3 * ww1C + px3;
         if (bitmap_get(BMpixels,ii)) {
            pix3[0] = 255;
            pix3[1] = pix3[2] = 0;
         }
      }
   }

   modified++;
   mwpaint2(1);                                                            //  update window
   return;
}


//  adjust color, brightness, curvature and recombine images

void pano_final_adjust()
{
   int pano_color_dialog_event(zdialog *zd, const char *event);            //  dialog event function

   zdialog  *zd;
   int      zstat;
   char     *adjmessage = "\n adjust color / brigntness of right image";

   get_Bratios(pxb1C,pxb2C);                                               //  get color ratios for overlap region

   blend = 1;                                                              //  show color diffs at joint
   showRedpix = 0;                                                         //  no alignment pixel highlights
   Radjust = Gadjust = Badjust = 1.0;                                      //  RGB manual adjustments, none yet

   get_overlap_region(pxb1C,pxb2C);
   pano_show_images(1);                                                    //  show final results

   zd = zdialog_new("match image2 to image1","done","cancel",0);           //  color adjustment dialog
   zdialog_add_widget(zd,"label","lab0","dialog",adjmessage,"space=10");
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"expand|space=10");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog");                    //  red           [ 100 ]
   zdialog_add_widget(zd,"label","lab1","vb1","red","space=7");            //  green         [ 100 ]
   zdialog_add_widget(zd,"label","lab2","vb1","green","space=7");          //  blue          [ 100 ]
   zdialog_add_widget(zd,"label","lab3","vb1","blue","space=7");           //  brightness    [ 100 ]
   zdialog_add_widget(zd,"label","lab4","vb1","brightness","space=7");     //  blend width   [  0  ]
   zdialog_add_widget(zd,"label","lab5","vb1","blend width","space=7");    //
   zdialog_add_widget(zd,"spin","spred","vb2","50|200|1|100");             //     [ execute ]
   zdialog_add_widget(zd,"spin","spgreen","vb2","50|200|1|100");
   zdialog_add_widget(zd,"spin","spblue","vb2","50|200|1|100");
   zdialog_add_widget(zd,"spin","spbright","vb2","50|200|1|100");
   zdialog_add_widget(zd,"spin","spblend","vb2","1|100|1|1");
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"homog|space=10");
   zdialog_add_widget(zd,"button","buttx","hb2","execute");
   
   zstat = zdialog_run(zd,pano_color_dialog_event);                        //  run dialog
   zdialog_destroy(zd);
   
   if (zstat == 1) return;                                                 //  done
   undo_pull();                                                            //  cancel
   return;
}


//  callback function for adjust color / brightness dialog
//  pxb1C + pxb2C >> pxb3 under control of spin buttons

int pano_color_dialog_event(zdialog *zd, const char *event)                //  v.20 overhauled
{
   double      red, green, blue, bright, bright2;

   if (strNeq(event,"buttx")) return 0;                                    //  wait for execute button

   zdialog_fetch(zd,"spred",red);                                          //  get color adjustments
   zdialog_fetch(zd,"spgreen",green);
   zdialog_fetch(zd,"spblue",blue);
   zdialog_fetch(zd,"spbright",bright);                                    //  brightness adjustment
   zdialog_fetch(zd,"spblend",blend);                                      //  blend width

   bright2 = (red + green + blue) / 3;                                     //  RGB brightness
   bright = bright / bright2;                                              //  bright setpoint / RGB brightness
   red = red * bright;                                                     //  adjust RGB brightness
   green = green * bright;
   blue = blue * bright;
   
   bright = (red + green + blue) / 3;
   zdialog_stuff(zd,"spred",red);                                          //  force back into consistency
   zdialog_stuff(zd,"spgreen",green);
   zdialog_stuff(zd,"spblue",blue);
   zdialog_stuff(zd,"spbright",bright);

   Radjust = red / 100;                                                    //  normalize 0.5 ... 2.0
   Gadjust = green / 100;
   Badjust = blue / 100;

   get_overlap_region(pxb1C,pxb2C);
   pano_show_images(1);                                                    //  combine and update window
   return 1;
}


/**************************************************************************/

//  unbend a pano image - straighten curvature added by pano

void m_unbend()
{
   int unbend_dialog_event(zdialog* zd, const char *event);                //  dialog event function

   zdialog     *zd;
   int         zstat;

   if (! pxb3) return;                                                     //  no image
   undo_push();                                                            //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);
   pixbuf_test(pxb1);
   pixbuf_poop(1)
   
   zd = zdialog_new("unbend panorama image","cancel","done",0);            //  create unbend dialog
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=10");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|space=10");
   zdialog_add_widget(zd,"spin","spvert","vb1","0|30|1|0");
   zdialog_add_widget(zd,"spin","sphorz","vb1","-10|10|1|0");
   zdialog_add_widget(zd,"button","buttx","vb1","execute");
   zdialog_add_widget(zd,"label","labvert","vb2","vertical unbend");
   zdialog_add_widget(zd,"label","labhorz","vb2","horizontal unbend");
   zdialog_add_widget(zd,"label","space","vb2",0);
   
   zdialog_resize(zd,260,0);
   zstat = zdialog_run(zd,unbend_dialog_event);                            //  run dialog
   zdialog_destroy(zd);

   if (zstat == 2) {                                                       //  done
      modified++;                                                          //  image3 modified
      mwpaint2(1);                                                         //  refresh window
   }

   else undo_pull();                                                       //  cancelled, restore image3
   return;
}


//  callback function for unbend dialog events

int unbend_dialog_event(zdialog *zd, const char *event)
{
   GdkPixbuf      *pxb9;
   int            horz, vert;
   int            px9, py9, ww9, hh9, rs9;
   double         px3, py3, dispx, dispx2, dispy;
   pixel          pix3, pix9, ppix9;

   if (strNeq(event,"buttx")) return 0;                                    //  wait for execute button

   zdialog_fetch(zd,"spvert",vert);                                        //  get unbend values
   zdialog_fetch(zd,"sphorz",horz);
   vert = int(vert * 0.01 * hh3);                                          //  % to pixels
   horz = int(horz * 0.01 * ww3);
   
   undo_last();                                                            //  refresh pxb3
   
   zlock();
   pxb9 = pixbuf_new(colorspace,0,8,ww3,hh3);                              //  get output pixbuf
   pixbuf_test(pxb9);
   pixbuf_poop(9);
   zunlock();

   for (py9 = 0; py9 < hh9; py9++)                                         //  step through pxb9 pixels
   for (px9 = 0; px9 < ww9; px9++)
   {
      dispx = (2.0 * px9 - ww9) / ww9;                                     //  -1.0 ..  0.0 .. +1.0
      dispy = (2.0 * py9 - hh9) / hh9;                                     //  -1.0 ..  0.0 .. +1.0
      dispx2 = dispx * dispx - 0.5;                                        //  +0.5 .. -0.5 .. +0.5  curved

      px3 = px9 + dispx * dispy * horz;                                    //  input virtual pixel, x
      py3 = py9 - dispy * dispx2 * vert;                                   //  input virtual pixel, y
      pix3 = vpixel(ppix3,px3,py3,ww3,hh3,rs3);                            //  input virtual pixel

      pix9 = ppix9 + py9 * rs9 + px9 * nch;                                //  output pixel
      if (pix3) {
         pix9[0] = pix3[0];                                                //  input pixel >> output pixel
         pix9[1] = pix3[1];
         pix9[2] = pix3[2];
      }
      else pix9[0] = pix9[1] = pix9[2] = 0;
   }
   
   pixbuf_free(pxb3);                                                      //  replace image3 with unbent version
   pxb3 = pxb9;
   pixbuf_poop(3);

   modified++;
   mwpaint2(1);                                                            //  update window
   return 1;
}


/**************************************************************************
         functions shared by HDR and pano
**************************************************************************/

//  get a virtual pixel at location (px,py) (real values).
//  get the overlapping real pixels and build a composite.

pixel vpixel(pixel ppix, double px, double py, int ww, int hh, int rs)
{
   int            px0, py0;
   pixel          pix0, pix1, pix2, pix3;
   double         f0, f1, f2, f3;
   double         red, green, blue;

   static uchar   rpix[4];

   px0 = int(px);                                                          //  pixel containing (px,py)
   py0 = int(py);

   if (px0 < 1 || py0 < 1) return 0;                                       //  sharpen the border    v.32
   if (px0 > ww-3 || py0 > hh-3) return 0;
   
   pix0 = ppix + py0 * rs + px0 * nch;                                     //  4 pixels based at (px0,py0)
   pix1 = pix0 + rs;
   pix2 = pix0 + nch;
   pix3 = pix0 + rs + nch;

   f0 = (px0+1 - px) * (py0+1 - py);                                       //  overlap of (px,py)
   f1 = (px0+1 - px) * (py - py0);                                         //   in each of the 4 pixels
   f2 = (px - px0) * (py0+1 - py);
   f3 = (px - px0) * (py - py0);
   
   red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];      //  sum the weighted inputs
   green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
   blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
      
   rpix[0] = int(red + 0.5);
   rpix[1] = int(green + 0.5);
   rpix[2] = int(blue + 0.5);

   return rpix;
}


//  compare two doubles for significant difference                         //  v.10
//  return:  0  difference not significant
//          +1  d1 > d2
//          -1  d1 < d2

int sigdiff(double d1, double d2, double signf)
{
   double diff = fabs(d1-d2);
   if (diff == 0.0) return 0;
   diff = diff / (fabs(d1) + fabs(d2));
   if (diff < signf) return 0;
   if (d1 > d2) return 1;
   else return -1;
}


//  Get the rectangle containing the overlap region of two images          v.32 revised

void get_overlap_region(GdkPixbuf *pxb91, GdkPixbuf *pxb92)
{
   int         ww91, hh91, ww92, hh92, rs91, rs92;
   int         pxL2, pyL2;
   pixel       ppix91, ppix92;

   pixbuf_poop(91)
   pixbuf_poop(92)

   pxL = 0;
   if (xoff > 0) pxL = int(xoff);

   pxH = ww91;
   if (pxH > xoff + ww92) pxH = int(xoff + ww92);
   
   pyL = 0;
   if (yoff > 0) pyL = int(yoff);
   
   pyH = hh91;
   if (pyH > yoff + hh92) pyH = int(yoff + hh92);

   if (toff > 0) {
      pyL2 = int(yoff + toff * (pxH - pxL));
      if (pyL2 > pyL) pyL = pyL2;
   }

   if (toff < 0) {   
      pxL2 = int(xoff - toff * (pyH - pyL));
      if (pxL2 > pxL) pxL = pxL2;
   }
   
   if (blend > 0) {
      pxM = int((ww91 + xoff) / 2 - 0.5 * toff * (pyH - pyL));             //  midpoint of blend stripe
      pxmL = pxM - int(blend/2);                                           //  blend stripe pixel x-range
      pxmH = pxM + int(blend/2);
      if (pxmL < pxL) pxmL = pxL;
      if (pxmH > pxH) pxmH = pxH;
   }
   else {                                                                  //  no blend stripe, use whole range
      pxmL = pxL;
      pxmH = pxH;
   }

   return;
}


//  Compute brightness ratio by color for overlapping image areas.         v.23  64 >> 256 bins
//    (image2 is overlayed on image1, offset by xoff, yoff, toff)
//  Output: Bratios[kk][Nth] = image1/image2 brightness ratio for color kk
//          for Nth distribution bin of pixels grouped by kk brightness.

void get_Bratios(GdkPixbuf *pxb91, GdkPixbuf *pxb92)
{
   pixel       ppix91, ppix92, pix1, pix2;
   int         ww91, hh91, ww92, hh92, rs91, rs92;
   int         px1, py1;
   int         ii, jj, kk, npix, npix1, npix2, npix3;
   int         brdist1[3][256], brdist2[3][256];
   double      px2, py2;
   double      brlev1[3][256], brlev2[3][256];
   double      costf = cos(toff), sintf = sin(toff);
   double      a1, a2, b1, b2;

   pixbuf_poop(91)
   pixbuf_poop(92)
   
   for (kk = 0; kk < 3; kk++)                                              //  clear distributions
   for (ii = 0; ii < 256; ii++)
      brdist1[kk][ii] = brdist2[kk][ii] = 0;
      
   npix = 0;

   for (py1 = pyL; py1 < pyH; py1++)                                       //  scan image1/image2 pixels in parallel
   for (px1 = pxL; px1 < pxH; px1++)                                       //  use entire overlap area
   {
      pix1 = ppix91 + py1 * rs91 + px1 * nch;                              //  image1 pixel
      if (!pix1[0] && !pix1[1] && !pix1[2]) continue;                      //  ignore black pixels

      px2 = (px1 - xoff) * costf + (py1 - yoff) * sintf;                   //  corresponding image2 pixel
      py2 = (py1 - yoff) * costf - (px1 - xoff) * sintf;
      pix2 = vpixel(ppix92,px2,py2,ww92,hh92,rs92);
      if (! pix2) continue;                                                //  does not exist
      if (!pix2[0] && !pix2[1] && !pix2[2]) continue;                      //  ignore black pixels

      ++npix;                                                              //  count overlapping pixels
      
      for (kk = 0; kk < 3; kk++)                                           //  accumulate brightness distributions
      {                                                                    //    by color in 256 bins
         ++brdist1[kk][pix1[kk]];
         ++brdist2[kk][pix2[kk]];
      }
   }
   
   npix1 = npix / 256;                                                     //  1/256th of total pixels
   
   for (kk = 0; kk < 3; kk++)                                              //  calc. brlev1[kk][N] = mean brightness
   for (ii = jj = 0; jj < 256; jj++)                                       //    for Nth group of image1 pixels
   {                                                                       //      for color kk
      brlev1[kk][jj] = 0;
      npix2 = npix1;                                                       //  1/256th of total pixels

      while (npix2 > 0 && ii < 256)                                        //  next 1/256th group from distribution
      {
         npix3 = brdist1[kk][ii];
         if (npix3 == 0) { ++ii; continue; }
         if (npix3 > npix2) npix3 = npix2;
         brlev1[kk][jj] += ii * npix3;                                     //  brightness * (pixels with brightness)
         brdist1[kk][ii] -= npix3;
         npix2 -= npix3;
      }

      brlev1[kk][jj] = brlev1[kk][jj] / npix1;                             //  mean brightness for group, 0-255
   }

   for (kk = 0; kk < 3; kk++)                                              //  do same for image2
   for (ii = jj = 0; jj < 256; jj++)
   {
      brlev2[kk][jj] = 0;
      npix2 = npix1;

      while (npix2 > 0 && ii < 256)
      {
         npix3 = brdist2[kk][ii];
         if (npix3 == 0) { ++ii; continue; }
         if (npix3 > npix2) npix3 = npix2;
         brlev2[kk][jj] += ii * npix3;
         brdist2[kk][ii] -= npix3;
         npix2 -= npix3;
      }

      brlev2[kk][jj] = brlev2[kk][jj] / npix1;
   }

   for (kk = 0; kk < 3; kk++)                                              //  color
   for (ii = jj = 0; ii < 256; ii++)                                       //  brlev1 brightness, 0 to 255
   {
      while (ii > brlev2[kk][jj] && jj < 256) ++jj;                        //  find matching brlev2 brightness
      a2 = brlev2[kk][jj];                                                 //  next higher value
      b2 = brlev1[kk][jj];
      if (a2 > 0 && b2 > 0) {
         if (jj > 0) {
            a1 = brlev2[kk][jj-1];                                         //  next lower value
            b1 = brlev1[kk][jj-1];
         }
         else   a1 = b1 = 0;
         if (ii == 0) Bratios[kk][ii] = b2 / a2;
         else Bratios[kk][ii] = (b1 + (ii-a1)/(a2-a1) * (b2-b1)) / ii;     //  interpolate
      }
      else Bratios[kk][ii] = 1;
   }

   for (ii = 0; ii < 256; ii++)                                            //  brightness adjustment factors
   {                                                                       //    for match_pixels() function    v.23
      brightfix0[ii] = Bratios[0][ii] * ii;
      brightfix1[ii] = Bratios[1][ii] * ii;
      brightfix2[ii] = Bratios[2][ii] * ii;
   }

   return;
}


//  find pixels of greatest contrast within overlap area                   v.27 improved
//  flag high-contrast pixels to use in each image compare region

void flag_edge_pixels(GdkPixbuf *pxb9)
{
   void  flag_edge_pixels2(GdkPixbuf *, int pxL, int pxH, int pyL, int pyH, int samp);

   pixel       ppix9;
   int         ww9, hh9, rs9, samp;

   pixbuf_poop(9)

   if (BMpixels) bitmap_delete(BMpixels);                                  //  bitmap to flag alignment pixels
   BMpixels = bitmap_new(ww9*hh9);

   if (blend == 0) {                                                       //  HDR: use 4 regions
      samp = pixsamp / 4;
      flag_edge_pixels2(pxb9, pxL, pxH/2, pyL, pyH/2, samp);
      flag_edge_pixels2(pxb9, pxH/2, pxH, pyL, pyH/2, samp);
      flag_edge_pixels2(pxb9, pxL, pxH/2, pyH/2, pyH, samp);
      flag_edge_pixels2(pxb9, pxH/2, pxH, pyH/2, pyH, samp);
   }
   else {                                                                  //  pano: use 4 regions in blend stripe
      samp = pixsamp / 4;                                                  //  v.32
      flag_edge_pixels2(pxb9, pxmL, pxmH, pyL, pyH/4, samp);
      flag_edge_pixels2(pxb9, pxmL, pxmH, pyH/4, pyH/2, samp);
      flag_edge_pixels2(pxb9, pxmL, pxmH, pyH/2, pyH-pyH/4, samp);
      flag_edge_pixels2(pxb9, pxmL, pxmH, pyH-pyH/4, pyH, samp);
   }

   return;
}


//  Find the highest contrast pixels meeting sample size
//    within the specified sub-region of image overlap.

void flag_edge_pixels2(GdkPixbuf *pxb9, int pxL, int pxH, int pyL, int pyH, int samp)
{
   pixel       ppix9;
   int         ww9, hh9, rs9;
   int         px1, py1, ii, jj, npix;
   int         red1, green1, blue1, red2, green2, blue2, tcon;
   int         Hdist[766], Vdist[766], Hmin, Vmin;
   uchar       *Hcon, *Vcon;
   pixel       pix1, pix2;
   
   pixbuf_poop(9)

   npix = (pxH - pxL) * (pyH - pyL);                                       //  overlapping pixels
   if (npix < 100) return;                                                 //  insignificant
   if (samp > npix / 4) samp = npix / 4;                                   //  use max. 1/4 of pixels       v.32

   Hcon = (uchar *) zmalloc(npix);                                         //  horizontal pixel contrast 0-255
   Vcon = (uchar *) zmalloc(npix);                                         //  vertical pixel contrast 0-255

   for (py1 = pyL; py1 < pyH-2; py1++)                                     //  scan image pixels in sub-region
   for (px1 = pxL; px1 < pxH-2; px1++)
   {
      ii = (py1-pyL) * (pxH-pxL) + (px1-pxL);
      Hcon[ii] = Vcon[ii] = 0;                                             //  horiz. = vert. contrast = 0

      pix1 = ppix9 + py1 * rs9 + px1 * nch;                                //  base pixel
      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];
      if (!red1 && !green1 && !blue1) continue;                            //  ignore if black

      pix2 = pix1 + 2 * nch;                                               //  2 pixels to right
      red2 = pix2[0];
      green2 = pix2[1];
      blue2 = pix2[2];
      if (!red2 && !green2 && !blue2) continue;                            //  ignore if black
      tcon = abs(red1-red2) + abs(green1-green2) + abs(blue1-blue2);       //  horizontal contrast
      if (tcon > 10) Hcon[ii] = tcon / 3;                                  //  ignore low contrast pixels   v.32

      pix2 = pix1 + 2 * rs9;                                               //  2 pixels below
      red2 = pix2[0];
      green2 = pix2[1];
      blue2 = pix2[2];
      if (!red2 && !green2 && !blue2) continue;                            //  ignore if black
      tcon = abs(red1-red2) + abs(green1-green2) + abs(blue1-blue2);       //  vertical contrast
      if (tcon > 10) Vcon[ii] = tcon / 3;
   }

   for (ii = 0; ii < 766; ii++) Hdist[ii] = Vdist[ii] = 0;                 //  clear contrast distributions

   for (py1 = pyL; py1 < pyH-2; py1++)                                     //  scan image pixels
   for (px1 = pxL; px1 < pxH-2; px1++)
   {                                                                       //  build contrast distributions
      ii = (py1-pyL) * (pxH-pxL) + (px1-pxL);
      ++Hdist[Hcon[ii]];
      ++Vdist[Vcon[ii]];
   }
   
   for (npix = 0, ii = 765; ii > 0; ii--)                                  //  find minimum contrast needed to get
   {                                                                       //    enough pixels for sample size
      npix += Hdist[ii];                                                   //      (horizontal contrast pixels)
      if (npix > samp) break; 
   }
   Hmin = ii; 

   for (npix = 0, ii = 765; ii > 0; ii--)                                  //  (verticle contrast pixels)
   {
      npix += Vdist[ii];
      if (npix > samp) break;
   }
   Vmin = ii; 

   for (py1 = pyL; py1 < pyH-2; py1++)                                     //  scan image pixels
   for (px1 = pxL; px1 < pxH-2; px1++)
   {
      ii = (py1-pyL) * (pxH-pxL) + (px1-pxL);
      jj = py1 * ww9 + px1;

      if (Hcon[ii] > Hmin) {
         bitmap_set(BMpixels,jj,1);                                        //  flag horizontal group of 3
         bitmap_set(BMpixels,jj+1,1);
         bitmap_set(BMpixels,jj+2,1);
      }

      if (Vcon[ii] > Vmin) {
         bitmap_set(BMpixels,jj,1);                                        //  flag verticle group of 3
         bitmap_set(BMpixels,jj+ww9,1);
         bitmap_set(BMpixels,jj+2*ww9,1);
      }
   }
   
   zfree(Hcon);
   zfree(Vcon);

   return;
}


//  Compare two images in overlapping areas.
//  (image2 is overlayed on image1, offset by xoff, yoff, toff)
//  If blend > 0, compare verticle stripe in the middle, width = blend.
//  Use pixels with contrast >= minimum needed to reach enough pixels.
//  return: 1 = perfect match, 0 = total mismatch (black/white)

double match_images(GdkPixbuf *pxb91, GdkPixbuf *pxb92)
{
   pixel       ppix91, ppix92, pix1, pix2;
   int         ww91, hh91, ww92, hh92, rs91, rs92;
   int         px1, py1, ii, pixcount, ymid;
   double      px2, py2;
   double      costf = cos(toff), sintf = sin(toff);
   double      match, cmatch, maxcmatch;
   double      yspan, weight;

   pixbuf_poop(91)
   pixbuf_poop(92)
   
   cmatch = maxcmatch = pixcount = 0;

   if (blend > 0) goto match_blend;

   for (py1 = pyL; py1 < pyH; py1++)                                       //  scan image1/image2 pixels in parallel
   for (px1 = pxL; px1 < pxH; px1++)
   {
      ii = py1 * ww91 + px1;                                               //  skip low-contrast pixels   v.25
      if (! bitmap_get(BMpixels,ii)) continue;

      pix1 = ppix91 + py1 * rs91 + px1 * nch;                              //  image1 pixel
      if (!pix1[0] && !pix1[1] && !pix1[2]) continue;                      //  ignore black pixels

      px2 = (px1 - xoff) * costf + (py1 - yoff) * sintf;                   //  corresponding image2 pixel
      py2 = (py1 - yoff) * costf - (px1 - xoff) * sintf;
      pix2 = vpixel(ppix92,px2,py2,ww92,hh92,rs92);
      if (! pix2) continue;                                                //  does not exist
      if (!pix2[0] && !pix2[1] && !pix2[2]) continue;                      //  ignore black pixels

      match = match_pixels(pix1,pix2);                                     //  compare brightness adjusted
      cmatch += match;                                                     //  accumulate total match
      pixcount++;
   }

   return cmatch / pixcount;

match_blend:

   if (pxM > ww91) return 0;                                               //  blend runs off of image, no match

   yspan = 1.0 / (pyH - pyL);                                              //  prepare params for weight calculation
   ymid = (pyL + pyH) / 2;                                                 //  (weight central pixels more)

   for (py1 = pyL; py1 < pyH; py1++)                                       //  step through image1 pixels, y
   {
      weight = 1 - yspan * abs(py1 - ymid);                                //  weight: 0.5 .. 1.0 .. 0.5   v.32

      for (px1 = pxmL; px1 < pxmH; px1++)                                  //  step through image1 pixels, x
      {
         ii = py1 * ww91 + px1;                                            //  skip low-contrast pixels   v.25
         if (! bitmap_get(BMpixels,ii)) continue;

         pix1 = ppix91 + py1 * rs91 + px1 * nch;                           //  image1 pixel
         if (!pix1[0] && !pix1[1] && !pix1[2]) continue;                   //  ignore black pixels

         px2 = (px1 - xoff) * costf + (py1 - yoff) * sintf;                //  corresponding image2 pixel
         py2 = (py1 - yoff) * costf - (px1 - xoff) * sintf;
         pix2 = vpixel(ppix92,px2,py2,ww92,hh92,rs92);
         if (! pix2) continue;                                             //  does not exist
         if (!pix2[0] && !pix2[1] && !pix2[2]) continue;                   //  ignore black pixels

         match = match_pixels(pix1,pix2);                                  //  compare brightness adjusted
         cmatch += match * weight;                                         //  accumulate total match
         maxcmatch += weight;
         pixcount++;
      }
   }

   return cmatch / maxcmatch;
}


//  Compare 2 pixels using precalculated brightness ratios
//  1.0 = perfect match   0 = total mismatch (black/white)

double match_pixels(pixel pix1, pixel pix2)                                //  v.28
{
   double   red1, green1, blue1, red2, green2, blue2;
   double   reddiff, greendiff, bluediff, match;
   
   red1 = pix1[0];
   green1 = pix1[1];
   blue1 = pix1[2];
   
   red2 = brightfix0[pix2[0]];
   green2 = brightfix1[pix2[1]];
   blue2 = brightfix2[pix2[2]];
   
   reddiff = 0.00392 * fabs(red1-red2);                                    //  0 = perfect match
   greendiff = 0.00392 * fabs(green1-green2);                              //  1 = total mismatch
   bluediff = 0.00392 * fabs(blue1-blue2);
   
   match = (1.0 - reddiff) * (1.0 - greendiff) * (1.0 - bluediff);
   return match;
}


/**************************************************************************/

//  functions to push image3 into undo stack and pull from stack           v.15

GdkPixbuf   *undostack[undomax];                                           //  undo / redo stack of images
int         nust, uftf = 1;

void undo_push()                                                           //  push image3 into stack
{
   if (uftf) {
      uftf = nust = 0;                                                     //  first call, clear undo stack
      for (int ii = 0; ii < undomax; ii++) undostack[ii] = 0;
   }

   if (! pxb3) return;
   zlock();
   if (nust == undomax) {
      pixbuf_free(undostack[0]);                                           //  lose oldest stack entry
      for (int ii = 1; ii < undomax; ii++)
            undostack[ii-1] = undostack[ii];                               //  push other entries down
      nust--;
      undostack[nust] = 0;                                                 //  last entry vacated
   }
   pixbuf_free(undostack[nust]);                                           //  free entry to be overwritten
   undostack[nust] = pixbuf_copy(pxb3);                                    //  image3 >> stack
   pixbuf_test(undostack[nust]);
   nust++;
   zunlock();   
   return;
}

void undo_pull()                                                           //  pull image3 from stack (undo)
{
   GdkPixbuf   *temp;

   if (! pxb3) return;
   if (nust == 0) return;
   zlock();
   temp = undostack[nust-1];                                               //  image3 <==> last stack entry    v.26
   undostack[nust-1] = pxb3;                                               //  fix memory leak   v.26
   pxb3 = temp;
   pixbuf_poop(3);
   nust--;                                                                 //  stack count down
   if (nust == 0) modified = 0;
   zunlock();
   mwpaint2(1);
   return;
}

void undo_last()                                                           //  get last image3 from stack
{                                                                          //    without removal from stack
   if (! pxb3) return;
   if (nust == 0) {
      modified = 0;
      return;
   }
   zlock();
   pixbuf_free(pxb3);
   pxb3 = pixbuf_copy(undostack[nust-1]);
   pixbuf_test(pxb3);
   pixbuf_poop(3);
   zunlock();
   return;
}

void undo_redo()                                                           //  get last+1 stack entry    v.24
{                                                                          //    and advance stack depth
   GdkPixbuf   *temp;

   if (! pxb3) return;
   if (nust == undomax) return;
   if (! undostack[nust]) return;                                          //  nothing is there
   zlock();
   temp = undostack[nust];                                                 //  image3 <==> last+1 stack entry
   undostack[nust] = pxb3;                                                 //  fix memory leak   v.26
   pxb3 = temp;
   pixbuf_poop(3);
   nust++;                                                                 //  stack count up
   modified++;
   zunlock();
   mwpaint2(1);
   return;
}

void undo_clear()                                                          //  clear stack (start new image)
{
   zlock();
   for (int ii = 0; ii < undomax; ii++) 
         pixbuf_free(undostack[ii]);
   nust = 0;
   modified = 0;
   zunlock();
   return;
}


/**************************************************************************/

//  track pixbufs allocated and freed for memory leak detection            v.27

int      pixbuf_count = 0;

void incr_pixbufs()
{
   ++pixbuf_count;
   if (debug) printf("pixbufs: %d \n",pixbuf_count);
   return;
}

void pixbuf_free(GdkPixbuf *&pixbuf)
{
   if (! pixbuf) return;
   zlock();
   g_object_unref(pixbuf);
   pixbuf = 0;
   zunlock();
   --pixbuf_count;
   if (debug) printf("pixbufs: %d \n",pixbuf_count);
   return;
}


/**************************************************************************/

//  show EXIF data if available                                            v.31

int exif_show(char *file)
{
   int         err, eof;
   char        command[1000], buff[100];
   zdialog     *zd;
   GtkWidget   *wedit;

   snprintf(command,999,"exiv2 pr \"%s\" 2>&1",file);                      //  command to print exif data

   err = createProc(command);                                              //  execute command
   if (err) {
      zmessageACK(" %s \n %s",command,syserrText(err));
      return err;
   }

   zd = zdialog_new("EXIF data","OK",0);                                   //  create dialog to show outputs
   zdialog_add_widget(zd,"edit","edit","dialog",0,"expand");
   wedit = zdialog_widget(zd,"edit");

   while (true)
   {   
      eof = getProcOutput(buff,99,err);                                    //  get outputs, stuff dialog
      if (eof) break;
      if (strlen(buff) < 3) continue;
      wprintf(wedit," %s \n",buff);
   }
   
   if (err) wprintf(wedit," %s \n",syserrText(err));
   zdialog_run(zd);                                                        //  show dialog with command outputs
   zdialog_destroy(zd);
   return err;
}


//  fetch EXIF data from image file

int exif_fetch(char *file)
{
   char     command[1000];
   int      err;

   snprintf(command,999,"exiv2 -f ex \"%s\" 2>&1",file);
   err = system(command);
   return err;
}


//  stuff EXIF data from original image file into new image file
//  (exif_fetch() for file1 must be done beforehand)

int exif_stuff(char *file1, char *file2, int xx, int yy)
{
   void  modsuff(char *ofile, cchar *ifile, cchar *suff);                  //  private function

   char     efile1[1000], efile2[1000];
   char     command[2000];
   int      err = 0;

   modsuff(efile1,file1,".exv");                                           //  get .exv file names from image file names
   modsuff(efile2,file2,".exv");

   if (strNeq(efile1,efile2)) {                                            //  if images files are not the same,
      snprintf(command,1999,"mv -f \"%s\" \"%s\" ",efile1,efile2);         //    move and rename the .exv file
      err = system(command);
      if (err) return err;
   }
   
   snprintf(command,999,"exiv2 in \"%s\" ",file2);                         //  load EXIF data into image file
   err = system(command);
   snprintf(command,999,"rm -f \"%s\" ",efile2);                           //  delete EXIF data file
   system(command);
   if (err) return err;

   snprintf(command,999,"exiv2 -M\"set Exif.Photo.PixelXDimension %d\" "   //  modify EXIF data in image file
                             " -M\"set Exif.Photo.PixelYDimension %d\" "   //  (new pixel dimensions)
                             " mo \"%s\" ", xx, yy, file2);
   err = system(command);
   return err;
}

void modsuff(char *outfile, cchar *infile, cchar *suff)
{
   char     *pp;

   if (strlen(infile) > 990) zappcrash("file name overflow");
   strcpy(outfile,infile);
   pp = strrchr(outfile,'/');
   if (! pp) pp = outfile;
   pp = strrchr(pp,'.');
   if (pp) *pp = 0;
   strcat(outfile,suff);
   return;
}



