/* -*- linux-c -*- ------------------------------------------------------- *
 *   
 *   Copyright 2002 H. Peter Anvin - All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, Inc., 53 Temple Place Ste 330,
 *   Bostom MA 02111-1307, USA; either version 2 of the License, or
 *   (at your option) any later version; incorporated herein by reference.
 *
 * ----------------------------------------------------------------------- */

/*
 * raid6sse2.c
 *
 * SSE-2 implementation of RAID-6 syndrome functions
 *
 */

#if defined(__i386__) || defined(__x86_64__)

#include "raid6.h"
#include "raid6x86.h"

static const struct raid6_sse_constants {
	u64 x1d[2];
	u64 xfe[2];
} raid6_sse_constants = {
	{ 0x1d1d1d1d1d1d1d1dULL, 0x1d1d1d1d1d1d1d1dULL },
	{ 0xfefefefefefefefeULL, 0xfefefefefefefefeULL }
};

static inline int cpuid_features(void)
{
	u32 eax = 1;
	u32 ebx, ecx, edx;

	asm volatile("cpuid" :
		     "+a" (eax), "=b" (ebx), "=c" (ecx), "=d" (edx));

	return edx;
}

static int raid6_have_sse2(void)
{
	/*** FIX TO USE KERNEL PRIMITIVES ***/
	u32 features = cpuid_features();
	return ( (features & (15<<23)) == (15<<23) ); 
}

/*
 * Plain SSE2 implementation
 */
static void raid6_sse21_gen_syndrome(int disks, size_t bytes, void **ptrs)
{
	u8 **dptr = (u8 **)ptrs;
	u8 *p, *q;
	int d, z, z0;
	raid6_sse_save_t sa;

	z0 = disks - 3;		/* Highest data disk */
	p = dptr[z0+1];		/* XOR parity */
	q = dptr[z0+2];		/* RS syndrome */

	raid6_before_sse(&sa);

	asm volatile("movdqa %0,%%xmm0" :: "m" (raid6_sse_constants.x1d[0]));
	asm volatile("movdqa %0,%%xmm1" :: "m" (raid6_sse_constants.xfe[0]));

	asm volatile("pxor %xmm2,%xmm2");	/* P[0] */
	asm volatile("pxor %xmm4,%xmm4"); /* Q[0] */
	asm volatile("pxor %xmm5,%xmm5");	/* Zero temp */

	for ( d = 0 ; d < bytes ; d += 16 ) {
		asm volatile("prefetchnta %0" :: "m" (dptr[z0][d]));
		asm volatile("pxor %xmm2,%xmm2");	/* P[0] */
		asm volatile("pxor %xmm4,%xmm4"); /* Q[0] */
		asm volatile("movdqa %0,%%xmm6" :: "m" (dptr[z0][d]));
		for ( z = z0-1 ; z >= 0 ; z-- ) {
			asm volatile("prefetchnta %0" :: "m" (dptr[z][d]));
			asm volatile("pcmpgtb %xmm4,%xmm5");
			asm volatile("psllw $1,%xmm4");
			asm volatile("pand %xmm0,%xmm5");
			asm volatile("pand %xmm1,%xmm4");
			asm volatile("pxor %xmm5,%xmm4");
			asm volatile("pxor %xmm5,%xmm5");
			asm volatile("pxor %xmm6,%xmm2");
			asm volatile("pxor %xmm6,%xmm4");
			asm volatile("movdqa %0,%%xmm6" :: "m" (dptr[z][d]));
		}
		asm volatile("pcmpgtb %xmm4,%xmm5");
		asm volatile("psllw $1,%xmm4");
		asm volatile("pand %xmm0,%xmm5");
		asm volatile("pand %xmm1,%xmm4");
		asm volatile("pxor %xmm5,%xmm4");
		asm volatile("pxor %xmm5,%xmm5");
		asm volatile("pxor %xmm6,%xmm2");
		asm volatile("pxor %xmm6,%xmm4");

		asm volatile("movntdq %%xmm2,%0" : "=m" (p[d]));
		asm volatile("pxor %xmm2,%xmm2");
		asm volatile("movntdq %%xmm4,%0" : "=m" (q[d]));
		asm volatile("pxor %xmm4,%xmm4");
	}
	asm volatile("sfence" : : : "memory");
	raid6_after_sse(&sa);
}

static int raid6_sse21_chk_syndrome(int disks, size_t bytes, void **ptrs)
{
	u8 **dptr = (u8 **)ptrs;
	u8 *p, *q;
	int d, z, z0;
	u32 perr, qerr;
	raid6_sse_save_t sa;

	z0 = disks - 3;		/* Highest data disk */
	p = dptr[z0+1];		/* XOR parity */
	q = dptr[z0+2];		/* RS syndrome */

	raid6_before_sse(&sa);

	asm volatile("movdqa %0,%%xmm0" :: "m" (raid6_sse_constants.x1d[0]));
	asm volatile("movdqa %0,%%xmm1" :: "m" (raid6_sse_constants.xfe[0]));

	asm volatile("pxor %xmm3,%xmm3");	/* Collects P inequalities */
	asm volatile("pxor %xmm7,%xmm7");	/* Collects Q inequalities */

	for ( d = 0 ; d < bytes ; d += 16 ) {
		asm volatile("prefetchnta %0" :: "m" (dptr[z0][d]));
		asm volatile("pxor %xmm2,%xmm2");	/* P[0] */
		asm volatile("pxor %xmm4,%xmm4"); /* Q[0] */
		asm volatile("pxor %xmm5,%xmm5");	/* Zero temp */
		asm volatile("movdqa %0,%%xmm6" :: "m" (dptr[z0][d]));
		for ( z = z0-1 ; ; z-- ) {
			asm volatile("prefetchnta %0" :: "m" (dptr[z][d]));
			asm volatile("pcmpgtb %xmm4,%xmm5");
			asm volatile("psllw $1,%xmm4");
			asm volatile("pand %xmm0,%xmm5");
			asm volatile("pand %xmm1,%xmm4");
			asm volatile("pxor %xmm5,%xmm4");
			asm volatile("pxor %xmm5,%xmm5");
			asm volatile("pxor %xmm6,%xmm2");
			asm volatile("pxor %xmm6,%xmm4");
			asm volatile("movdqa %0,%%xmm6" :: "m" (dptr[z][d]));
		}
		asm volatile("pcmpgtb %xmm4,%xmm5");
		asm volatile("psllw $1,%xmm4");
		asm volatile("pand %xmm0,%xmm5");
		asm volatile("pand %xmm1,%xmm4");
		asm volatile("pxor %xmm5,%xmm4");
		asm volatile("pxor %xmm5,%xmm5");
		asm volatile("pxor %xmm6,%xmm2");
		asm volatile("pxor %xmm6,%xmm4");

		asm volatile("pcmpeqb %0,%%xmm2" :: "m" (p[d]));
		asm volatile("pcmpeqb %0,%%xmm4" :: "m" (q[d]));
		asm volatile("por %xmm2,%xmm3");
		asm volatile("por %xmm4,%xmm7");
	}
	asm volatile("pmovmskb %%xmm3,%0" : "=r" (perr));
	asm volatile("pmovmskb %%xmm7,%0" : "=r" (qerr));

	raid6_after_sse(&sa);

	return (perr ? RAID6_P_BAD : 0)|(qerr ? RAID6_Q_BAD : 0);
}

const struct raid6_calls raid6_sse2x1 = {
	raid6_sse21_gen_syndrome,
	raid6_sse21_chk_syndrome,
	raid6_have_sse2,
	"sse2x1",
	1			/* Has cache hints */
};

/*
 * Unrolled-by-2 SSE2 implementation
 */
static void raid6_sse22_gen_syndrome(int disks, size_t bytes, void **ptrs)
{
	u8 **dptr = (u8 **)ptrs;
	u8 *p, *q;
	int d, z, z0;
	raid6_sse_save_t sa;

	z0 = disks - 3;		/* Highest data disk */
	p = dptr[z0+1];		/* XOR parity */
	q = dptr[z0+2];		/* RS syndrome */

	raid6_before_sse(&sa);

	asm volatile("movdqa %0,%%xmm0" :: "m" (raid6_sse_constants.x1d[0]));
	asm volatile("movdqa %0,%%xmm1" :: "m" (raid6_sse_constants.xfe[0]));
	
	asm volatile("pxor %xmm2,%xmm2");	/* P[0] */
	asm volatile("pxor %xmm3,%xmm3");	/* P[1] */
	asm volatile("pxor %xmm4,%xmm4"); /* Q[0] */
	asm volatile("pxor %xmm5,%xmm5");	/* Zero temp */
	asm volatile("pxor %xmm6,%xmm6"); /* Q[1] */
	asm volatile("pxor %xmm7,%xmm7"); /* Zero temp */

	for ( d = 0 ; d < bytes ; d += 32 ) {
		for ( z = z0 ; z >= 0 ; z-- ) {
			/* We assume this prefetch covers both entries */
			asm volatile("prefetchnta %0" :: "m" (dptr[z][d]));
			asm volatile("pcmpgtb %xmm4,%xmm5");
			asm volatile("pcmpgtb %xmm6,%xmm7");
			asm volatile("psllw $1,%xmm4");
			asm volatile("psllw $1,%xmm6");
			asm volatile("pand %xmm0,%xmm5");
			asm volatile("pand %xmm0,%xmm7");
			asm volatile("pand %xmm1,%xmm4");
			asm volatile("pand %xmm1,%xmm6");
			asm volatile("pxor %xmm5,%xmm4");
			asm volatile("pxor %xmm7,%xmm6");
			asm volatile("movdqa %0,%%xmm5" :: "m" (dptr[z][d]));
			asm volatile("movdqa %0,%%xmm7" :: "m" (dptr[z][d+16]));
			asm volatile("pxor %xmm5,%xmm2");
			asm volatile("pxor %xmm7,%xmm3");
			asm volatile("pxor %xmm5,%xmm4");
			asm volatile("pxor %xmm7,%xmm6");
			asm volatile("pxor %xmm5,%xmm5");
			asm volatile("pxor %xmm7,%xmm7");
		}
		asm volatile("movntdq %%xmm2,%0" : "=m" (p[d]));
		asm volatile("pxor %xmm2,%xmm2");
		asm volatile("movntdq %%xmm3,%0" : "=m" (p[d+16]));
		asm volatile("pxor %xmm3,%xmm3");
		asm volatile("movntdq %%xmm4,%0" : "=m" (q[d]));
		asm volatile("pxor %xmm4,%xmm4");
		asm volatile("movntdq %%xmm6,%0" : "=m" (q[d+16]));
		asm volatile("pxor %xmm6,%xmm6");
	}
	asm volatile("sfence" : : : "memory");
	raid6_after_sse(&sa);
}

static int raid6_sse22_chk_syndrome(int disks, size_t bytes, void **ptrs)
{
	u8 **dptr = (u8 **)ptrs;
	u8 *p, *q;
	int d, z, z0;
	u32 perr = 0, qerr = 0, pxerr, qxerr;
	raid6_sse_save_t sa;

	z0 = disks - 3;		/* Highest data disk */
	p = dptr[z0+1];		/* XOR parity */
	q = dptr[z0+2];		/* RS syndrome */

	raid6_before_sse(&sa);

	asm volatile("movdqa %0,%%xmm0" :: "m" (raid6_sse_constants.x1d[0]));
	asm volatile("movdqa %0,%%xmm1" :: "m" (raid6_sse_constants.xfe[0]));
	
	asm volatile("pxor %xmm2,%xmm2");	/* P[0] */
	asm volatile("pxor %xmm3,%xmm3");	/* P[1] */
	asm volatile("pxor %xmm4,%xmm4"); /* Q[0] */
	asm volatile("pxor %xmm5,%xmm5");	/* Zero temp */
	asm volatile("pxor %xmm6,%xmm6"); /* Q[1] */
	asm volatile("pxor %xmm7,%xmm7"); /* Zero temp */

	asm volatile("movdqa %%xmm2,%0" : "=m" (perr));
	asm volatile("movdqa %%xmm2,%0" : "=m" (qerr));

	for ( d = 0 ; d < bytes ; d += 32 ) {
		for ( z = z0 ; z >= 0 ; z-- ) {
			/* We assume this prefetch covers both entries */
			asm volatile("prefetchnta %0" :: "m" (dptr[z][d]));
			asm volatile("pcmpgtb %xmm4,%xmm5");
			asm volatile("pcmpgtb %xmm6,%xmm7");
			asm volatile("psllw $1,%xmm4");
			asm volatile("psllw $1,%xmm6");
			asm volatile("pand %xmm0,%xmm5");
			asm volatile("pand %xmm0,%xmm7");
			asm volatile("pand %xmm1,%xmm4");
			asm volatile("pand %xmm1,%xmm6");
			asm volatile("pxor %xmm5,%xmm4");
			asm volatile("pxor %xmm7,%xmm6");
			asm volatile("movdqa %0,%%xmm5" :: "m" (dptr[z][d]));
			asm volatile("movdqa %0,%%xmm7" :: "m" (dptr[z][d+16]));
			asm volatile("pxor %xmm5,%xmm2");
			asm volatile("pxor %xmm7,%xmm3");
			asm volatile("pxor %xmm5,%xmm4");
			asm volatile("pxor %xmm7,%xmm6");
			asm volatile("pxor %xmm5,%xmm5");
			asm volatile("pxor %xmm7,%xmm7");
		}
		asm volatile("pcmpeqb %0,%%xmm2" :: "m" (p[d]));
		asm volatile("pcmpeqb %0,%%xmm3" :: "m" (p[d+16]));
		asm volatile("pcmpeqb %0,%%xmm4" :: "m" (q[d]));
		asm volatile("pcmpeqb %0,%%xmm6" :: "m" (q[d+16]));
		asm volatile("por  %xmm2,%xmm3");
		asm volatile("por  %xmm4,%xmm6");
		asm volatile("pxor %xmm2,%xmm2");
		asm volatile("pxor %xmm4,%xmm4");
		asm volatile("pmovmskb %%xmm3,%0" : "=r" (pxerr));
		asm volatile("pmovmskb %%xmm6,%0" : "=r" (qxerr));
		asm volatile("pxor %xmm3,%xmm3");
		asm volatile("pxor %xmm6,%xmm6");
		perr |= pxerr;
		qerr |= qxerr;
	}

	raid6_after_sse(&sa);

	return (perr ? RAID6_P_BAD : 0)|(qerr ? RAID6_Q_BAD : 0);
}

const struct raid6_calls raid6_sse2x2 = {
	raid6_sse22_gen_syndrome,
	raid6_sse22_chk_syndrome,
	raid6_have_sse2,
	"sse2x2",
	1			/* Has cache hints */
};

#endif
