/*
 * Copyright 2008 Sony Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the names of the copyright holders nor the names of their
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "e_cell.h"

#include "libspe2_runtime.h"

#define GET_QUEUE_ELM(q, n) ((q)->buffer + (q)->aligned_elm_size * (n))

static int cell_queue_push_begin(uint64_t q_ea, uint64_t *elm_ea, int *size)
	{
	ALIGNED(static cell_queue_t q, SHARED_ATOMIC_DATA_ALIGN);

	uint32_t status;
	uint32_t r_status, w_status;
	int cont = 1;

	/* mark 'writing' */
	do
		{
		spu_write_event_mask(MFC_LLR_LOST_EVENT);
		spu_write_event_ack(MFC_LLR_LOST_EVENT);

		mfc_getllar(&q, q_ea, 0, 0);
		mfc_read_atomic_status();

		status = q.status;
		r_status = CELL_QUEUE_UNPACK_R_STATUS(status);
		w_status = CELL_QUEUE_UNPACK_W_STATUS(status);

		if (!CELL_QUEUE_IS_LOCKED(w_status) /* no other thread writing to this queue */ &&
			CELL_QUEUE_POINTER(w_status) < CELL_QUEUE_POINTER(r_status) + q.depth)
			{
			w_status = CELL_QUEUE_SET_LOCKED(w_status); /* lock write status */
			q.status = CELL_QUEUE_PACK_STATUS(r_status, w_status);
			spu_dsync(); /* make sure 'q' has been updated */
			mfc_putllc(&q, q_ea, 0, 0);
			cont = (mfc_read_atomic_status() & MFC_PUTLLC_STATUS);
			}
		else
			{
			spu_read_event_status(); /* wait for reservation lost */
			}
		} while (cont);

	/* pointer mod q.depth */
	w_status = CELL_QUEUE_POINTER(w_status);
	if (w_status >= q.depth) w_status -= q.depth;
	ASSERT(w_status < q.depth);

	*elm_ea = GET_QUEUE_ELM(&q, w_status);
	*size = q.elm_size;

	return 1;
	}

static int cell_queue_push_end(uint64_t q_ea)
	{
	ALIGNED(static cell_queue_t q, SHARED_ATOMIC_DATA_ALIGN);

	uint32_t status;
	uint32_t r_status, w_status;
	int notify ;

	mfc_sync(0); /* sync with the system memory */

	/* finish */
	do
		{
		mfc_getllar(&q, q_ea, 0, 0);
		mfc_read_atomic_status();

		status = q.status;
		r_status = CELL_QUEUE_UNPACK_R_STATUS(status);
		w_status = CELL_QUEUE_UNPACK_W_STATUS(status);

		notify = (CELL_QUEUE_POINTER(w_status) == CELL_QUEUE_POINTER(r_status));

		/* unlock write status and increment write pointer */
		w_status = CELL_QUEUE_POINTER(w_status) + 1;
		if (CELL_QUEUE_POINTER(w_status) >= q.depth &&
			CELL_QUEUE_POINTER(r_status) >= q.depth)
			{
			w_status -= q.depth;
			r_status -= q.depth;
			}

		q.status = CELL_QUEUE_PACK_STATUS(r_status, w_status);

		spu_dsync(); /* make sure 'q' has been updated */
		mfc_putllc(&q, q_ea, 0, 0);
		} while (mfc_read_atomic_status());

	if (notify)
		{
		sync_notify(q_ea);
		}

	return 1;
	}

int cell_queue_push(uint64_t q_ea, const void *buffer)
	{
	uint32_t tag = 0;
	uint64_t elm_ea;
	int size;

	if (!cell_queue_push_begin(q_ea, &elm_ea, &size))
		return 0;

	/* copy data */
	mfc_put((void*)buffer, elm_ea, ALIGN_CEIL(size, SHARED_DATA_ALIGN), tag, 0, 0);

	/* wait for finishing DMA */
	mfc_write_tag_mask(1 << tag);
	mfc_read_tag_status_all();

	return cell_queue_push_end(q_ea);
	}

static int cell_queue_pop_begin(uint64_t q_ea, uint64_t *elm_ea, int *size)
	{
	ALIGNED(static cell_queue_t q, SHARED_ATOMIC_DATA_ALIGN);

	uint32_t status;
	uint32_t r_status, w_status;
	int cont = 1;

	do
		{
		spu_write_event_mask(MFC_LLR_LOST_EVENT);
		spu_write_event_ack(MFC_LLR_LOST_EVENT);

		mfc_getllar(&q, q_ea, 0, 0);
		mfc_read_atomic_status();

		status = q.status;
		r_status = CELL_QUEUE_UNPACK_R_STATUS(status);
		w_status = CELL_QUEUE_UNPACK_W_STATUS(status);

		if (!CELL_QUEUE_IS_LOCKED(r_status) /* no other thread reading from this queue */ &&
			CELL_QUEUE_POINTER(r_status) < CELL_QUEUE_POINTER(w_status))
			{
			r_status = CELL_QUEUE_SET_LOCKED(r_status); /* lock read status */
			q.status = CELL_QUEUE_PACK_STATUS(r_status, w_status);
			spu_dsync(); /* make sure 'q' has been updated */
			mfc_putllc(&q, q_ea, 0, 0);
			cont = (mfc_read_atomic_status() & MFC_PUTLLC_STATUS);
			}
		else
			{
			spu_read_event_status(); /* wait for reservation lost */
			}
		} while (cont);

	/* pointer mod q.depth */
	r_status = CELL_QUEUE_POINTER(r_status);
	if (r_status >= q.depth) r_status -= q.depth;
	ASSERT(r_status < q.depth);

	*elm_ea = GET_QUEUE_ELM(&q, r_status);
	*size = q.elm_size;

	return 1;
	}

static int cell_queue_pop_end(uint64_t q_ea)
	{
	ALIGNED(static cell_queue_t q, SHARED_ATOMIC_DATA_ALIGN);

	uint32_t status;
	uint32_t r_status, w_status;
	int notify;

	mfc_sync(0); /* sync with the system memory */

	/* finish */
	do
		{
		mfc_getllar(&q, q_ea, 0, 0);
		mfc_read_atomic_status();

		status = q.status;
		r_status = CELL_QUEUE_UNPACK_R_STATUS(status);
		w_status = CELL_QUEUE_UNPACK_W_STATUS(status);

		notify = (CELL_QUEUE_POINTER(w_status) == CELL_QUEUE_POINTER(r_status) + q.depth);

		/* unlock read status and increment read pointer */
		r_status = CELL_QUEUE_POINTER(r_status) + 1;
		if (CELL_QUEUE_POINTER(w_status) >= q.depth &&
			CELL_QUEUE_POINTER(r_status) >= q.depth)
			{
			w_status -= q.depth;
			r_status -= q.depth;
			}

		q. status = CELL_QUEUE_PACK_STATUS(r_status, w_status);

		spu_dsync(); /* make sure 'q' has been updated */
		mfc_putllc(&q, q_ea, 0, 0);
		} while (mfc_read_atomic_status());

	if (notify)
		{
		sync_notify(q_ea);
		}

	return 1;
	}

int cell_queue_pop(uint64_t q_ea, void *buffer)
	{
	uint32_t tag = 0;
	uint64_t elm_ea;
	int size;

	if (!cell_queue_pop_begin(q_ea, &elm_ea, &size))
		return 0;

	/* copy data */
	mfc_get(buffer, elm_ea, ALIGN_CEIL(size, SHARED_DATA_ALIGN), tag, 0, 0);

	/* wait for finishing DMA */
	mfc_write_tag_mask(1 << tag);
	mfc_read_tag_status_all();

	return cell_queue_pop_end(q_ea);
	}
