package org.eclipse.webdav.dom;

/*
 * (c) Copyright IBM Corp. 2000, 2001.
 * All Rights Reserved.
 */

import org.eclipse.webdav.Policy;
import org.w3c.dom.*;

/**
 * An element editor for the WebDAV propfind element. See RFC2518
 * section 12.14 for the element's definition.
 * <p>
 * <b>Note:</b> This class/interface is part of an interim API that is still under 
 * development and expected to change significantly before reaching stability. 
 * It is being made available at this early stage to solicit feedback from pioneering 
 * adopters on the understanding that any code that uses this API will almost 
 * certainly be broken (repeatedly) as the API evolves.
 * </p>
 *
 * @see Prop
 */
public class PropFind extends ElementEditor {
	/**
	 * An ordered collection of the element names of the propfind
	 * element's children.
	 */
	protected static final String[] childNames = 
		new String[] {"allprop", "propname", "prop"}; 
/**
 * Creates a new editor on the given WebDAV propfind element. The
 * element is assumed to be well formed.
 *
 * @param root a propfind element
 * @throws        MalformedElementException if there is reason to
 *                believe that the element is not well formed
 */
public PropFind(Element root) throws MalformedElementException {

	super(root, "propfind");
}
/**
 * Creates a new WebDAV propfind element and sets it as the root of the
 * given document. Returns an editor on the new propfind element. The
 * document must not be <code>null</code>, and must not already have a
 * root element.
 *
 * @param document the document that will become the root of a new
 *                 propfind element
 * @return         an element editor on a propfind element
 */
public static PropFind create(Document document) {

	Assert.isNotNull(document);
	Assert.isTrue(document.getDocumentElement() == null);

	Element element = create(document, "propfind");

	try {
		return new PropFind(element);
	} catch (MalformedElementException e) {
		Assert.isTrue(false, Policy.bind("assert.internalError"));
		return null;
	}
}
/**
 * Returns this propfind's prop.
 *
 * @return this propfind's prop
 * @throws MalformedElementException if there is reason to believe that
 *         this editor's underlying element is not well formed, or if
 *         this propfind is not in the "prop" form
 */
public Prop getProp() throws MalformedElementException {

	Element prop = getFirstChild(root, "prop");
	ensureNotNull(Policy.bind("ensure.missingPropElmt"), prop);

	return new Prop(prop);
}
/**
 * Returns <code>true</code> iff this propfind is in the "all prop"
 * form.
 *
 * @return a boolean indicating whether this propfind is in the "all
 *         prop" form
 * @throws MalformedElementException if there is reason to believe that
 *         this editor's underlying element is not well formed
 */
public boolean isAllProp() throws MalformedElementException {

	Element child = getFirstChild(root, childNames);
	ensureNotNull(Policy.bind("ensure.missingAllpropOrPropnameOrPropElmt"), child);

	boolean isAllProp = isDAVElement(child, "allprop");

	child = getNextSibling(child, childNames);
	ensureNull(Policy.bind("ensure.conflictingAllpropOrPropnameOrPropElmt"), child); 

	return isAllProp;
}
/**
 * Returns <code>true</code> iff this propfind is in the "prop" form.
 *
 * @return a boolean indicating whether this propfind is in the "prop"
 *         form
 * @throws MalformedElementException if there is reason to believe that
 *         this editor's underlying element is not well formed
 */
public boolean isProp() throws MalformedElementException {

	Element child = getFirstChild(root, childNames);
	ensureNotNull(Policy.bind("ensure.missingAllpropOrPropnameOrPropElmt"), child);

	boolean isProp = isDAVElement(child, "prop");

	child = getNextSibling(child, childNames);
	ensureNull(Policy.bind("ensure.conflictingAllpropOrPropnameOrPropElmt"), child); 

	return isProp;
}
/**
 * Returns <code>true</code> iff this propfind is in the "prop name"
 * form.
 *
 * @return a boolean indicating whether this propfind is in the
 *         "propname" form
 * @throws MalformedElementException if there is reason to believe that
 *         this editor's underlying element is not well formed
 */
public boolean isPropName() throws MalformedElementException {

	Element child = getFirstChild(root, childNames);
	ensureNotNull(Policy.bind("ensure.missingAllpropOrPropnameOrPropElmt"), child);

	boolean isPropName = isDAVElement(child, "propname");

	child = getNextSibling(child, childNames);
	ensureNull(Policy.bind("ensure.conflictingAllpropOrPropnameOrPropElmt"), child); 

	return isPropName;
}
/**
 * Sets whether this propfind is in the "all prop" form.
 *
 * @param a boolean indicating whether this propfind will be in the
 *        "all prop" form
 */
public void setIsAllProp(boolean isAllProp) {

	Element child = getFirstChild(root, childNames);
	boolean isAlreadyAllProp = isDAVElement(child, "allprop");

	if (isAllProp) {
		if (!isAlreadyAllProp) {
			if (child != null)
				root.removeChild(child);
			appendChild(root, "allprop");
		}
	} else if (isAlreadyAllProp)
		root.removeChild(child);
}
/**
 * Sets whether this propfind is in the "prop name" form.
 *
 * @param a boolean indicating whether this propfind will be in the
 *        "prop name" form
 */
public void setIsPropName(boolean isPropName) {

	Element child = getFirstChild(root, childNames);
	boolean isAlreadyPropName = isDAVElement(child, "propname");

	if (isPropName) {
		if (!isAlreadyPropName) {
			if (child != null)
				root.removeChild(child);
			appendChild(root, "propname");
		}
	} else if (isAlreadyPropName)
		root.removeChild(child);
}
/**
 * Creates and sets a new prop on this propfind and returns an editor on
 * it. This propfind must not already be in the "all prop" or "prop
 * name" form.
 *
 * @return an editor on a new prop element
 */
public Prop setProp() {

	Assert.isTrue(
		getFirstChild(root, new String[] {"allprop", "propname"}) == null); 

	Element prop = setChild(root, "prop", new String[] {"prop"}, true);

	try {
		return new Prop(prop);
	} catch (MalformedElementException e) {
		Assert.isTrue(false, Policy.bind("assert.internalError"));
		return null; // Never reached.
	}
}
}
