<?php
// $Id: update.php,v 1.252.2.2 2008/12/10 22:30:13 goba Exp $
// 日本語メッセージ版
//         2008/02/11 -- v 1.252     初版リリース
//         2008/12/11 -- v 1.252.2.2 オリジナルの変更に伴う修正
// translated by Takafumi (Drupal Japan)
/**
 * @file
 * Administrative page for handling updates from one Drupal version to another.
 *
 * Point your browser to "http://www.example.com/update.php" and follow the
 * instructions.
 *
 * If you are not logged in as administrator, you will need to modify the access
 * check statement inside your settings.php file. After finishing the upgrade,
 * be sure to open settings.php again, and change it back to its original state!
 */

/**
 * Global flag to identify update.php run, and so avoid various unwanted
 * operations, such as hook_init() and hook_exit() invokes, css/js preprocessing
 * and translation, and solve some theming issues. This flag is checked on several
 * places in Drupal code (not just update.php).
 */
define('MAINTENANCE_MODE', 'update');

/**
 * Add a column to a database using syntax appropriate for PostgreSQL.
 * Save result of SQL commands in $ret array.
 *
 * Note: when you add a column with NOT NULL and you are not sure if there are
 * already rows in the table, you MUST also add DEFAULT. Otherwise PostgreSQL
 * won't work when the table is not empty, and db_add_column() will fail.
 * To have an empty string as the default, you must use: 'default' => "''"
 * in the $attributes array. If NOT NULL and DEFAULT are set the PostgreSQL
 * version will set values of the added column in old rows to the
 * DEFAULT value.
 *
 * @param $ret
 *   Array to which results will be added.
 * @param $table
 *   Name of the table, without {}
 * @param $column
 *   Name of the column
 * @param $type
 *   Type of column
 * @param $attributes
 *   Additional optional attributes. Recognized attributes:
 *     not null => TRUE|FALSE
 *     default  => NULL|FALSE|value (the value must be enclosed in '' marks)
 * @return
 *   nothing, but modifies $ret parameter.
 */
function db_add_column(&$ret, $table, $column, $type, $attributes = array()) {
  if (array_key_exists('not null', $attributes) and $attributes['not null']) {
    $not_null = 'NOT NULL';
  }
  if (array_key_exists('default', $attributes)) {
    if (is_null($attributes['default'])) {
      $default_val = 'NULL';
      $default = 'default NULL';
    }
    elseif ($attributes['default'] === FALSE) {
      $default = '';
    }
    else {
      $default_val = "$attributes[default]";
      $default = "default $attributes[default]";
    }
  }

  $ret[] = update_sql("ALTER TABLE {". $table ."} ADD $column $type");
  if (!empty($default)) {
    $ret[] = update_sql("ALTER TABLE {". $table ."} ALTER $column SET $default");
  }
  if (!empty($not_null)) {
    if (!empty($default)) {
      $ret[] = update_sql("UPDATE {". $table ."} SET $column = $default_val");
    }
    $ret[] = update_sql("ALTER TABLE {". $table ."} ALTER $column SET NOT NULL");
  }
}

/**
 * Change a column definition using syntax appropriate for PostgreSQL.
 * Save result of SQL commands in $ret array.
 *
 * Remember that changing a column definition involves adding a new column
 * and dropping an old one. This means that any indices, primary keys and
 * sequences from serial-type columns are dropped and might need to be
 * recreated.
 *
 * @param $ret
 *   Array to which results will be added.
 * @param $table
 *   Name of the table, without {}
 * @param $column
 *   Name of the column to change
 * @param $column_new
 *   New name for the column (set to the same as $column if you don't want to change the name)
 * @param $type
 *   Type of column
 * @param $attributes
 *   Additional optional attributes. Recognized attributes:
 *     not null => TRUE|FALSE
 *     default  => NULL|FALSE|value (with or without '', it won't be added)
 * @return
 *   nothing, but modifies $ret parameter.
 */
function db_change_column(&$ret, $table, $column, $column_new, $type, $attributes = array()) {
  if (array_key_exists('not null', $attributes) and $attributes['not null']) {
    $not_null = 'NOT NULL';
  }
  if (array_key_exists('default', $attributes)) {
    if (is_null($attributes['default'])) {
      $default_val = 'NULL';
      $default = 'default NULL';
    }
    elseif ($attributes['default'] === FALSE) {
      $default = '';
    }
    else {
      $default_val = "$attributes[default]";
      $default = "default $attributes[default]";
    }
  }

  $ret[] = update_sql("ALTER TABLE {". $table ."} RENAME $column TO ". $column ."_old");
  $ret[] = update_sql("ALTER TABLE {". $table ."} ADD $column_new $type");
  $ret[] = update_sql("UPDATE {". $table ."} SET $column_new = ". $column ."_old");
  if ($default) { $ret[] = update_sql("ALTER TABLE {". $table ."} ALTER $column_new SET $default"); }
  if ($not_null) { $ret[] = update_sql("ALTER TABLE {". $table ."} ALTER $column_new SET NOT NULL"); }
  $ret[] = update_sql("ALTER TABLE {". $table ."} DROP ". $column ."_old");
}

/**
 * Perform one update and store the results which will later be displayed on
 * the finished page.
 *
 * An update function can force the current and all later updates for this
 * module to abort by returning a $ret array with an element like:
 * $ret['#abort'] = array('success' => FALSE, 'query' => 'What went wrong');
 * The schema version will not be updated in this case, and all the
 * aborted updates will continue to appear on update.php as updates that
 * have not yet been run.
 *
 * @param $module
 *   The module whose update will be run.
 * @param $number
 *   The update number to run.
 * @param $context
 *   The batch context array
 */
function update_do_one($module, $number, &$context) {
  // If updates for this module have been aborted
  // in a previous step, go no further.
  if (!empty($context['results'][$module]['#abort'])) {
    return;
  }

  $function = $module .'_update_'. $number;
  if (function_exists($function)) {
    $ret = $function($context['sandbox']);
  }

  if (isset($ret['#finished'])) {
    $context['finished'] = $ret['#finished'];
    unset($ret['#finished']);
  }

  if (!isset($context['results'][$module])) {
    $context['results'][$module] = array();
  }
  if (!isset($context['results'][$module][$number])) {
    $context['results'][$module][$number] = array();
  }
  $context['results'][$module][$number] = array_merge($context['results'][$module][$number], $ret);

  if (!empty($ret['#abort'])) {
    $context['results'][$module]['#abort'] = TRUE;
  }
  // Record the schema update if it was completed successfully.
  if ($context['finished'] == 1 && empty($context['results'][$module]['#abort'])) {
    drupal_set_installed_schema_version($module, $number);
  }

  $context['message'] = check_plain($module) .' モジュールの更新中';
}

function update_selection_page() {
  $output = '<p>更新中のDrupalのバージョンは自動的に検出されました。異なるバージョンを選択することもできますが、その必要はありません。</p>';
  $output .= '<p>更新処理を開始するには「更新」をクリックしてください。</p>';

  drupal_set_title('Drupal データベースの更新');
  $output .= drupal_get_form('update_script_selection_form');

  update_task_list('select');

  return $output;
}

function update_script_selection_form() {
  $form = array();
  $form['start'] = array(
    '#tree' => TRUE,
    '#type' => 'fieldset',
    '#title' => 'バージョンの選択',
    '#collapsible' => TRUE,
    '#collapsed' => TRUE,
  );

  // Ensure system.module's updates appear first
  $form['start']['system'] = array();

  $modules = drupal_get_installed_schema_version(NULL, FALSE, TRUE);
  foreach ($modules as $module => $schema_version) {
    $updates = drupal_get_schema_versions($module);
    // Skip incompatible module updates completely, otherwise test schema versions.
    if (!update_check_incompatibility($module) && $updates !== FALSE && $schema_version >= 0) {
      // module_invoke returns NULL for nonexisting hooks, so if no updates
      // are removed, it will == 0.
      $last_removed = module_invoke($module, 'update_last_removed');
      if ($schema_version < $last_removed) {
        $form['start'][$module] = array(
          '#value'  => '<em>'. $module .'</em> モジュールは更新できません。 スキーマバージョンは '. $schema_version .'です。 '. $last_removed .' までの更新はこのリリースで削除されました。 <em>'. $module .'</em> モジュールを更新するにはまず、これらの更新が利用可能であった最終バージョンに<a href="http://drupal.org/upgrade">アップグレードする必要があります</a>。',
          '#prefix' => '<div class="warning">',
          '#suffix' => '</div>',
        );
        $form['start']['#collapsed'] = FALSE;
        continue;
      }
      $updates = drupal_map_assoc($updates);
      $updates[] = '利用できる更新はありません';
      $default = $schema_version;
      foreach (array_keys($updates) as $update) {
        if ($update > $schema_version) {
          $default = $update;
          break;
        }
      }
      $form['start'][$module] = array(
        '#type' => 'select',
        '#title' => $module .' モジュール',
        '#default_value' => $default,
        '#options' => $updates,
      );
    }
  }

  $form['has_js'] = array(
    '#type' => 'hidden',
    '#default_value' => FALSE,
    '#attributes' => array('id' => 'edit-has_js'),
  );
  $form['submit'] = array(
    '#type' => 'submit',
    '#value' => '更新',
  );
  return $form;
}

function update_batch() {
  global $base_url;

  $operations = array();
  // Set the installed version so updates start at the correct place.
  foreach ($_POST['start'] as $module => $version) {
    drupal_set_installed_schema_version($module, $version - 1);
    $updates = drupal_get_schema_versions($module);
    $max_version = max($updates);
    if ($version <= $max_version) {
      foreach ($updates as $update) {
        if ($update >= $version) {
          $operations[] = array('update_do_one', array($module, $update));
        }
      }
    }
  }
  $batch = array(
    'operations' => $operations,
    'title' => '更新中',
    'init_message' => '更新開始',
    'error_message' => '回復不能エラーが発生しました。 以下でエラーメッセージを見つけることができます。 参考のためにそれをクリップボードにコピーすることをお勧めします。',
    'finished' => 'update_finished',
  );
  batch_set($batch);
  batch_process($base_url .'/update.php?op=results', $base_url .'/update.php');
}

function update_finished($success, $results, $operations) {
  // clear the caches in case the data has been updated.
  drupal_flush_all_caches();

  $_SESSION['update_results'] = $results;
  $_SESSION['update_success'] = $success;
  $_SESSION['updates_remaining'] = $operations;
}

function update_results_page() {
  drupal_set_title('Drupal データベースの更新');
  // NOTE: we can't use l() here because the URL would point to 'update.php?q=admin'.
  $links[] = '<a href="'. base_path() .'">メインページ</a>';
  $links[] = '<a href="'. base_path() .'?q=admin">管理ページ</a>';

  update_task_list();
  // Report end result
  if (module_exists('dblog')) {
    $log_message = ' すべてのエラーは<a href="'. base_path() .'?q=admin/reports/dblog">記録されました</a>。';
  }
  else {
    $log_message = ' すべてのエラーは記録されました。';
  }

  if ($_SESSION['update_success']) {
    $output = '<p>更新が試行されました。 以下に失敗が表示されていない場合は、そのまま<a href="'. base_path() .'?q=admin">管理ページ</a>に進むことができます。 そうでない場合は、手作業でデータベースを更新する必要があるかもしれません。'. $log_message .'</p>';
  }
  else {
    list($module, $version) = array_pop(reset($_SESSION['updates_remaining']));
    $output = '<p class="error">更新処理は<strong>'. $module .'.module の更新#'. $version .' の実行中に早期に中断されました</strong>。'. $log_message;
    if (module_exists('dblog')) {
      $output .= ' <code>watchdog</code> データベーステーブルを手作業で確認する必要があるかもしれません。';
    }
    $output .= '</p>';
  }

  if (!empty($GLOBALS['update_free_access'])) {
    $output .= "<p><strong>注意： 忘れずに <code>settings.php</code> 内の <code>\$update_free_access</code> の値を <code>FALSE</code> に戻してください。</strong></p>";
  }

  $output .= theme('item_list', $links);

  // Output a list of queries executed
  if (!empty($_SESSION['update_results'])) {
    $output .= '<div id="update-results">';
    $output .= '<h2>次のクエリが実行されました</h2>';
    foreach ($_SESSION['update_results'] as $module => $updates) {
      $output .= '<h3>'. $module .' モジュール</h3>';
      foreach ($updates as $number => $queries) {
        if ($number != '#abort') {
          $output .= '<h4>更新#'. $number .'</h4>';
          $output .= '<ul>';
          foreach ($queries as $query) {
            if ($query['success']) {
              $output .= '<li class="success">'. $query['query'] .'</li>';
            }
            else {
              $output .= '<li class="failure"><strong>失敗：</strong> '. $query['query'] .'</li>';
            }
          }
          if (!count($queries)) {
            $output .= '<li class="none">クエリなし</li>';
          }
        }
        $output .= '</ul>';
      }
    }
    $output .= '</div>';
  }
  unset($_SESSION['update_results']);
  unset($_SESSION['update_success']);

  return $output;
}

function update_info_page() {
  // Change query-strings on css/js files to enforce reload for all users.
  _drupal_flush_css_js();
  // Flush the cache of all data for the update status module.
  if (db_table_exists('cache_update')) {
    cache_clear_all('*', 'cache_update', TRUE);
  }

  update_task_list('info');
  drupal_set_title('Drupal データベースの更新');
  $token = drupal_get_token('update');
  $output = '<p>Drupalやモジュールの新しいリリースをインストールした際には常に、データベースを更新するために、このユーティリティを使用してください。</p><p>より詳細な情報は、<a href="http://drupal.org/node/258">Installation and upgrading handbook（英文）</a>を参照してください。 もし、これらが何を意味するのかがわからない場合は、サーバ管理者にお問い合わせください。</p>';
  $output .= "<ol>\n";
  $output .= "<li><strong>データベースをバックアップしてください</strong>。 このプロセスはデータベースの値を変更するでしょう。 また、緊急事態の場合にはバックアップに戻す必要があるかもしれません。</li>\n";
  $output .= "<li><strong>コードをバックアップしてください</strong>。 ヒント： モジュールコードのバックアップをする際、Drupalの自動発見メカニズムを混乱させる可能性があるため、'modules' や 'sites/*/modules' にそのバックアップを残さないでください。</li>\n";
  $output .= '<li>サイトを<a href="'. base_path() .'?q=admin/settings/site-maintenance">メンテナンスモード</a>にしてください。</li>'."\n";
  $output .= "<li>ハンドブックで説明されているように、適切な場所に新しいファイルをインストールしてください。</li>\n";
  $output .= "</ol>\n";
  $output .= "<p>上記のステップを実行後に続行してください。</p>\n";
  $output .= '<form method="post" action="update.php?op=selection&token='. $token .'"><input type="submit" value="続行" /></form>';
  $output .= "\n";
  return $output;
}

function update_access_denied_page() {
  drupal_set_title('アクセスが拒否されました');
  return '<p>このページにアクセスする権限がないため、アクセスは拒否されました。 管理ユーザ（最初に作成したユーザ）としてログインしてください。 ログインできない場合、このアクセスチェックを回避するために <code>settings.php</code> を編集する必要があります。</p>編集方法：
<ol>
 <li>システムの settings.php ファイルをテキストエディタで開いてください。 このファイルは、すべてのファイルがインストールされたDrupalルートディレクトリ下の、<code>sites/your_site_name</code> のようなディレクトリが存在する場合はそこに、あるいは <code>sites/default</code> にあります。</li>
 <li>settings.php ファイル内に <code>$update_free_access = FALSE;</code> という行がありますので、そこを <code>$update_free_access = TRUE;</code> に変更してください。</li>
 <li>update.php スクリプトの実行後、ただちに settings.php ファイルを元の状態（ <code>$update_free_access = FALSE;</code> ）に戻す必要があります。</li>
 <li>今後この問題に遭遇することを避けるために、更新処理の始めにデータベースをバックアップする前に、忘れずに管理ユーザとしてウェブサイトにログインしてください。</li>
</ol>';
}

/**
 * Create the batch table.
 *
 * This is part of the Drupal 5.x to 6.x migration.
 */
function update_create_batch_table() {

  // If batch table exists, update is not necessary
  if (db_table_exists('batch')) {
    return;
  }

  $schema['batch'] = array(
    'fields' => array(
      'bid'       => array('type' => 'serial', 'unsigned' => TRUE, 'not null' => TRUE),
      'token'     => array('type' => 'varchar', 'length' => 64, 'not null' => TRUE),
      'timestamp' => array('type' => 'int', 'not null' => TRUE),
      'batch'     => array('type' => 'text', 'not null' => FALSE, 'size' => 'big')
    ),
    'primary key' => array('bid'),
    'indexes' => array('token' => array('token')),
  );

  $ret = array();
  db_create_table($ret, 'batch', $schema['batch']);
  return $ret;
}

/**
 * Disable anything in the {system} table that is not compatible with the
 * current version of Drupal core.
 */
function update_fix_compatibility() {
  $ret = array();
  $incompatible = array();
  $query = db_query("SELECT name, type, status FROM {system} WHERE status = 1 AND type IN ('module','theme')");
  while ($result = db_fetch_object($query)) {
    if (update_check_incompatibility($result->name, $result->type)) {
      $incompatible[] = $result->name;
    }
  }
  if (!empty($incompatible)) {
    $ret[] = update_sql("UPDATE {system} SET status = 0 WHERE name IN ('". implode("','", $incompatible) ."')");
  }
  return $ret;
}

/**
 * Helper function to test compatibility of a module or theme.
 */
function update_check_incompatibility($name, $type = 'module') {
  static $themes, $modules;

  // Store values of expensive functions for future use.
  if (empty($themes) || empty($modules)) {
    $themes = _system_theme_data();
    $modules = module_rebuild_cache();
  }

  if ($type == 'module' && isset($modules[$name])) {
    $file = $modules[$name];
  }
  else if ($type == 'theme' && isset($themes[$name])) {
    $file = $themes[$name];
  }
  if (!isset($file)
      || !isset($file->info['core'])
      || $file->info['core'] != DRUPAL_CORE_COMPATIBILITY
      || version_compare(phpversion(), $file->info['php']) < 0) {
    return TRUE;
  }
  return FALSE;
}

/**
 * Perform Drupal 5.x to 6.x updates that are required for update.php
 * to function properly.
 *
 * This function runs when update.php is run the first time for 6.x,
 * even before updates are selected or performed.  It is important
 * that if updates are not ultimately performed that no changes are
 * made which make it impossible to continue using the prior version.
 * Just adding columns is safe.  However, renaming the
 * system.description column to owner is not.  Therefore, we add the
 * system.owner column and leave it to system_update_6008() to copy
 * the data from description and remove description. The same for
 * renaming locales_target.locale to locales_target.language, which
 * will be finished by locale_update_6002().
 */
function update_fix_d6_requirements() {
  $ret = array();

  if (drupal_get_installed_schema_version('system') < 6000 && !variable_get('update_d6_requirements', FALSE)) {
    $spec = array('type' => 'int', 'size' => 'small', 'default' => 0, 'not null' => TRUE);
    db_add_field($ret, 'cache', 'serialized', $spec);
    db_add_field($ret, 'cache_filter', 'serialized', $spec);
    db_add_field($ret, 'cache_page', 'serialized', $spec);
    db_add_field($ret, 'cache_menu', 'serialized', $spec);

    db_add_field($ret, 'system', 'info', array('type' => 'text'));
    db_add_field($ret, 'system', 'owner', array('type' => 'varchar', 'length' => 255, 'not null' => TRUE, 'default' => ''));
    if (db_table_exists('locales_target')) {
      db_add_field($ret, 'locales_target', 'language', array('type' => 'varchar', 'length' => 12, 'not null' => TRUE, 'default' => ''));
    }
    if (db_table_exists('locales_source')) {
      db_add_field($ret, 'locales_source', 'textgroup', array('type' => 'varchar', 'length' => 255, 'not null' => TRUE, 'default' => 'default'));
      db_add_field($ret, 'locales_source', 'version', array('type' => 'varchar', 'length' => 20, 'not null' => TRUE, 'default' => 'none'));
    }
    variable_set('update_d6_requirements', TRUE);

    // Create the cache_block table. See system_update_6027() for more details.
    $schema['cache_block'] = array(
      'fields' => array(
        'cid'        => array('type' => 'varchar', 'length' => 255, 'not null' => TRUE, 'default' => ''),
        'data'       => array('type' => 'blob', 'not null' => FALSE, 'size' => 'big'),
        'expire'     => array('type' => 'int', 'not null' => TRUE, 'default' => 0),
        'created'    => array('type' => 'int', 'not null' => TRUE, 'default' => 0),
        'headers'    => array('type' => 'text', 'not null' => FALSE),
        'serialized' => array('type' => 'int', 'size' => 'small', 'not null' => TRUE, 'default' => 0)
      ),
      'indexes' => array('expire' => array('expire')),
      'primary key' => array('cid'),
    );
    db_create_table($ret, 'cache_block', $schema['cache_block']);
  }

  return $ret;
}

/**
 * Add the update task list to the current page.
 */
function update_task_list($active = NULL) {
  // Default list of tasks.
  $tasks = array(
    'info' => '概要',
    'select' => '更新の選択',
    'run' => '更新の実行',
    'finished' => 'ログのレビュー',
  );

  drupal_set_content('left', theme('task_list', $tasks, $active));
}

/**
 * Check update requirements and report any errors.
 */
function update_check_requirements() {
  // Check the system module requirements only.
  $requirements = module_invoke('system', 'requirements', 'update');
  $severity = drupal_requirements_severity($requirements);

  // If there are issues, report them.
  if ($severity != REQUIREMENT_OK) {
    foreach ($requirements as $requirement) {
      if (isset($requirement['severity']) && $requirement['severity'] != REQUIREMENT_OK) {
        $message = isset($requirement['description']) ? $requirement['description'] : '';
        if (isset($requirement['value']) && $requirement['value']) {
          $message .= ' （現在使用中 '. $requirement['title'] .' '. $requirement['value'] .'）';
        }
        drupal_set_message($message, 'warning');
      }
    }
  }
}

// Some unavoidable errors happen because the database is not yet up-to-date.
// Our custom error handler is not yet installed, so we just suppress them.
ini_set('display_errors', FALSE);

require_once './includes/bootstrap.inc';

// We only load DRUPAL_BOOTSTRAP_CONFIGURATION for the update requirements
// check to avoid reaching the PHP memory limit.
$op = isset($_REQUEST['op']) ? $_REQUEST['op'] : '';
if (empty($op)) {
  // Minimum load of components.
  drupal_bootstrap(DRUPAL_BOOTSTRAP_CONFIGURATION);

  require_once './includes/install.inc';
  require_once './includes/file.inc';
  require_once './modules/system/system.install';

  // Load module basics.
  include_once './includes/module.inc';
  $module_list['system']['filename'] = 'modules/system/system.module';
  $module_list['filter']['filename'] = 'modules/filter/filter.module';
  module_list(TRUE, FALSE, FALSE, $module_list);
  drupal_load('module', 'system');
  drupal_load('module', 'filter');

  // Set up $language, since the installer components require it.
  drupal_init_language();

  // Set up theme system for the maintenance page.
  drupal_maintenance_theme();

  // Check the update requirements for Drupal.
  update_check_requirements();

  // Display the warning messages (if any) in a dedicated maintenance page,
  // or redirect to the update information page if no message.
  $messages = drupal_set_message();
  if (!empty($messages['warning'])) {
    drupal_maintenance_theme();
    print theme('update_page', '<form method="post" action="update.php?op=info"><input type="submit" value="続行" /></form>', FALSE);
    exit;
  }
  install_goto('update.php?op=info');
}

drupal_bootstrap(DRUPAL_BOOTSTRAP_FULL);
drupal_maintenance_theme();

// This must happen *after* drupal_bootstrap(), since it calls
// variable_(get|set), which only works after a full bootstrap.
update_create_batch_table();

// Turn error reporting back on. From now on, only fatal errors (which are
// not passed through the error handler) will cause a message to be printed.
ini_set('display_errors', TRUE);

// Access check:
if (!empty($update_free_access) || $user->uid == 1) {

  include_once './includes/install.inc';
  include_once './includes/batch.inc';
  drupal_load_updates();

  update_fix_d6_requirements();
  update_fix_compatibility();

  $op = isset($_REQUEST['op']) ? $_REQUEST['op'] : '';
  switch ($op) {
    case 'selection':
      if (isset($_GET['token']) && $_GET['token'] == drupal_get_token('update')) {
      $output = update_selection_page();
      break;
      }

    case '更新':
      if (isset($_GET['token']) && $_GET['token'] == drupal_get_token('update')) {
      update_batch();
      break;
      }

    // update.php ops
    case 'info':
      $output = update_info_page();
      break;

    case 'results':
      $output = update_results_page();
      break;

    // Regular batch ops : defer to batch processing API
    default:
      update_task_list('run');
      $output = _batch_page();
      break;
  }
}
else {
  $output = update_access_denied_page();
}
if (isset($output) && $output) {
  // We defer the display of messages until all updates are done.
  $progress_page = ($batch = batch_get()) && isset($batch['running']);
  print theme('update_page', $output, !$progress_page);
}
