# -*- coding: EUC-JP -*-
#
#  aya.py - an "Aya" compatible Shiori module for ninix
#  Copyright (C) 2002, 2003 by Shyouzou Sugitani <shy@debian.or.jp>
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
#  $Id: aya.py,v 1.24 2003/12/15 10:12:10 shy Exp $
#

import os
import string
import sys
import random
import time
from types import *
import math
from stat import *
import shutil
import StringIO
import fcntl
import re

import kanjilib

class AyaError(ValueError):
    pass

def encrypt_char(char):
    c = ord(char)
    j = 0
    while j < 3:
        msb = c & 0x80
        c = c << 1
        c = c & 0xff
        if msb:
            c = c | 0x01
        else:
            c = c & 0xfe
        j = j + 1
    c = c ^ 0xd2
    return chr(c)

def decrypt_char(char):
    c = ord(char)
    c = c ^ 0xd2
    j = 0
    while j < 3:
        lsb = c & 0x01
        c = c >> 1
        if lsb:
            c = c | 0x80
        else:
            c = c & 0x7f
        j = j + 1
    return chr(c)

def decrypt_readline(file):
    line = ''
    while 1:
        c = file.read(1)
        if c == '':
            break
        line = string.join((line, decrypt_char(c)), '')
        if line[-1:] == chr(10) or \
           line[-2:] == string.join((chr(13), chr(10)), ''):
            break
    return line

def line_strip(line):
    line = string.strip(line)
    while len(line) > 0:
        if line[:2] == '':
            line = string.strip(line[2:])
        else:
            break
    while len(line) > 0:
        if line[-2:] == '':
            line = string.strip(line[:-2])
        else:
            break
    return line

def find_not_quoted(line, token):
    position = 0
    while 1:
        pos_new = string.find(line, token, position)
        if pos_new < 0:
            break
        elif pos_new == 0:
            break
        position = string.find(line, '"', position)
        if position >= 0 and position < pos_new:
            position = position + 1
            while position < len(line)-1:
                if line[position] == '"':
                    position = position + 1
                    break
                else:
                    position = position + 1
                    continue
        else:
            break
    return pos_new

def find_comment(line):
    if line[:2] == '//':
        return 0, len(line)
    start = len(line) # not len(line)-1
    end = -1
    for token in [' //', '\t//', '//', '/*']:
        pos_new = find_not_quoted(line, token)
        if pos_new >= 0 and pos_new < start:
            start = pos_new
            if token == '/*':
                end = find_not_quoted(line, '*/')
                if end >= 0:
                    end = end + 2
            else:
                end = len(line)
    if start == len(line):
        start = -1
    return start, end

class Shiori:
    __AYA_TXT = 'aya.txt'
    __DBNAME = 'aya_variable.cfg'
    def __init__(self, dll_name):
        self.dll_name = dll_name
        if dll_name is not None:
            self.__AYA_TXT = dll_name[:-3] + 'txt'
            self.__DBNAME = dll_name[:-4] + '_variable.cfg'
        self.saori = None
    def use_saori(self, saori):
        self.saori = saori
    def find(self, dir, dll_name):
        result = 0
        if os.path.isfile(os.path.join(dir, 'aya.txt')):
            result = 200
        elif dll_name is not None and os.path.isfile(os.path.join(dir, dll_name[:-3]+'txt')):
            result = 100
        return result
    def show_description(self):
        sys.stdout.write('Shiori: AYA compatible module for ninix\n'
                         '        Copyright (C) 2002, 2003 by Shyouzou Sugitani\n'
                         '        Copyright (C) 2002, 2003 by MATSUMURA Namihiko\n')
    def reset(self):
        self.boot_time = time.time()
        self.aitalk = 0
        self.first_boot = 0
        self.dic_files = []
        self.dic = AyaDictionary(self)
        self.global_namespace = AyaGlobalNamespace(self)
        self.system_functions = AyaSystemFunctions(self)
        self.logfile = None
        self.filelist = {}
        self.reset_request()
        self.ver_3 = 0 # Ver.3
    def reset_request(self):
        self.req_command = ''
        self.req_protocol = ''
        self.req_key = []
        self.req_header = {}
        self.global_namespace.reset_res_reference()
    def load(self, aya_dir=None):
        self.aya_dir = aya_dir
        self.dbpath = os.path.join(self.aya_dir, self.__DBNAME)
        self.saori_library = AyaSaoriLibrary(self.saori, self.aya_dir)
        self.reset()
        self.first_boot = self.global_namespace.load_database(self)
        try:
            path = os.path.join(self.aya_dir, self.__AYA_TXT)
            aya_txt = open(path)
            self.load_aya_txt(aya_txt)
        except IOError:
            print 'cannot read aya.txt'
            return 0
        except AyaError, error:
            print error
            return 0
        # default setting
        if not self.global_namespace.exists('log'):
            self.global_namespace.put('log', '')
        if not self.global_namespace.exists('logmode'):
            self.global_namespace.put('logmode', 'simple')
        for path in self.dic_files:
            try:
                dicfile = open(path, 'r')
            except:
                print 'cannnot read %s' % path
                continue
            if path[-4:] == '.ayc':
                encrypted = 1
            else:
                encrypted = 0
            self.dic.load(dicfile, encrypted)
        if not self.global_namespace.exists('aitalkinterval'): # Ver.3
            self.global_namespace.put('aitalkinterval', 180)
        if not self.global_namespace.exists('securitylevel'): # Ver.3
            self.global_namespace.put('securitylevel', 'high')
        if not self.dic.get_function('OnRequest'): # Ver.3
            self.ver_3 = 1
        self.request('NOTIFY SHIORI/3.0\r\n' \
                     'ID: OnLoad\r\n' \
                     'Sender: AYA\r\n' \
                     'SecurityLevel: local\r\n' \
                     'Path: %s\r\n\r\n' % kanjilib.euc2sjis(string.replace(self.aya_dir, '/', '\\')))
        return 1
    def load_aya_txt(self, file):
        comment = 0
        while 1:
            line = kanjilib.sjis2euc(file.readline(), 'ignore')
            if not line:
                break # EOF
            if comment:
                end = find_not_quoted(line, '*/')
                if end < 0:
                    continue
                else:
                    line = line[end+2:]
                    comment = 0
            while 1:
                start, end = find_comment(line)
                if start < 0:
                    break
                if end < 0:
                    comment = 1
                    line = line[:start]
                    break
                line = string.join((line[:start], line[end:]))
            line = line_strip(line)
            if not line:
                continue
            comma = string.find(line, ',')
            if comma >= 0:
                key = line_strip(line[:comma])
                value = line_strip(line[comma+1:])
            else:
                continue
            self.evaluate_config(key, value)
    def evaluate_config(self, key, value):
        if key == 'dic':
            filename = string.lower(string.replace(value, '\\', '/'))
            path = os.path.join(self.aya_dir, filename)
            self.dic_files.append(path)
        elif key == 'log':
            path = os.path.join(self.aya_dir, str(value))
            try:
                file = open(path, 'w')
            except:
                print 'cannnot open %s' % path
            else:
                if self.logfile:
                    self.logfile.close()
                self.logfile = file
                self.global_namespace.put('log', str(value))
        elif key == 'logmode':
            pass # FIXME
        elif key == 'aitalkinterval': # Ver.3
            if not self.global_namespace.exists('aitalkinterval'):
                try:
                    int(value)
                except ValueError:
                    print 'Could not convert %s to an integer' % value
                else:
                    self.global_namespace.put('aitalkinterval', int(value))
        elif key != None and key != '':
            try:
                value = int(value)
            except:
                value = str(value)
            self.global_namespace.put(key, value)
    def get_dictionary(self):
        return self.dic
    def get_ghost_dir(self):
        return self.aya_dir
    def get_global_namespace(self):
        return self.global_namespace
    def get_system_functions(self):
        return self.system_functions
    def get_boot_time(self):
        return self.boot_time
    def unload(self):
        self.request('NOTIFY SHIORI/3.0\r\n' \
                     'ID: OnUnload\r\n' \
                     'Sender: AYA\r\n' \
                     'SecurityLevel: local\r\n\r\n')
        self.global_namespace.save_database()
        self.saori_library.unload()
        if self.logfile != None:
            self.logfile.close()
        for key in self.filelist.keys():
            self.filelist[key].close()
    # SHIORI API
    def request(self, req_string):
        header = StringIO.StringIO(req_string)
        line = header.readline()
        if line:
            line = string.strip(line)
            list = string.split(line)
            if len(list) >= 2:
                self.req_command = line_strip(list[0])
                self.req_protocol = line_strip(list[1])
            while 1:
                line = header.readline()
                if not line:
                    break # EOF
                line = line_strip(line)
                if not line:
                    continue
                colon = string.find(line, ':')
                if colon >= 0:
                    key = line_strip(line[:colon])
                    value = line_strip(line[colon+1:])
                    try:
                        value = int(value)
                    except:
                        value = str(value)
                    self.req_key.append(key)
                    self.req_header[key] = value
                else:
                    continue
            for key in self.req_header.keys():
                if type(self.req_header[key]) is StringType:
                    self.req_header[key] = kanjilib.sjis2euc(self.req_header[key], 'ignore')
        if self.first_boot:
            if self.req_header['ID'] in ['OnBoot', 'OnVanished', 'OnGhostChanged']:
                self.first_boot = 0
                print "We lost the %s. Initializing...." % self.__DBNAME
                self.request('NOTIFY SHIORI/3.0\r\n'
                             'ID: OnFirstBoot\r\n'
                             'Sender: ninix\r\n'
                             'SecurityLevel: local\r\n'
                             'Reference0: 0\r\n\r\n')
            elif self.req_header['ID'] == 'OnFirstBoot':
                self.first_boot = 0
        result = ''
        func = self.dic.get_function('OnRequest')
        if not func and self.req_header.has_key('ID'): # Ver.3
            for i in range(9):
                self.global_namespace.remove(string.join(('reference', str(i)), ''))
            for i in range(9):
                key = string.join(('Reference', str(i)), '')
                if self.req_header.has_key(key):
                    self.global_namespace.put(string.join(('reference', str(i)), ''),
                                              self.req_header[key])
            if self.req_header['ID'][:2] != 'On':
                prefix = 'On_'
            else:
                prefix = ''
            func = self.dic.get_function(string.join((prefix, self.req_header['ID']), ''))
        if func:
            result = func.call()
        if self.ver_3 and self.req_header.has_key('ID') and \
           self.req_header['ID'] == 'OnSecondChange': # Ver.3
            aitalkinterval = self.global_namespace.get('aitalkinterval')
            if aitalkinterval > 0:
                self.aitalk = self.aitalk + 1
                if self.aitalk > aitalkinterval:
                    self.aitalk = 0
                    result = self.request('GET SHIORI/3.0\r\n' \
                                          'ID: OnAiTalk\r\n' \
                                          'Sender: ninix\r\n' \
                                          'SecurityLevel: local\r\n\r\n')
                    self.reset_request()
                    return result
        self.reset_request()
        if self.ver_3: # Ver.3
            result = 'SHIORI/3.0 200 OK\r\n' \
                     'Sender: AYA\r\n' \
                     'Value: %s\r\n\r\n' % kanjilib.euc2sjis(result, 'ignore')
            return result
        else:
            return kanjilib.euc2sjis(result, 'ignore')

class AyaSecurity:
    __DENY = 0
    __ACCEPT = 1
    __CONFIG = 'aya_security.cfg'
    def __init__(self, aya):
        self.__aya = aya
        self.__cfg = ''
        self.__aya_dir = os.path.abspath(self.__aya.aya_dir)
        self.__fwrite = [[self.__DENY, '*'], [self.__ACCEPT, self.__aya_dir]]
        self.__fread = [[self.__ACCEPT, '*']]
        self.__loadlib = [[self.__ACCEPT, '*']]
        self.__logfile = None
        self.load_cfg()
        self.__fwrite.reverse()
        self.__fread.reverse()
    def load_cfg(self):
        path = []
        head, tail = os.path.split(self.__aya_dir)
        current = head
        while tail:
            path.append(tail)
            head, tail = os.path.split(current)
            current = head
        else:
            path.append(current)
        dir = ''
        while path:
            dir = os.path.join(dir, path.pop())
            if os.path.exists(os.path.join(dir, self.__CONFIG)):
                self.__cfg = os.path.join(dir, self.__CONFIG)
                break
        else: # default setting
            print '*WARNING : aya_security.cfg - file not found.'
            return
        try:
            file = open(self.__cfg)
        except IOError:
            print 'cannot read aya.txt'
            return
        except AyaError, error:
            print error
            return
        name = ''
        data = {}
        self.comment = 0
        line = self.readline(file)
        while line:
            start = string.find(line, '[')
            if start >= 0:
                if name:
                    name = self.expand_path(name)
                    if name == '*':
                        break
                    if self.__aya_dir[:len(name)] == name:
                        break
                data = {}
                end = string.find(line, ']')
                if end < 0:
                    end = len(line)
                name = line[start+1:end]
            else:
                comma = string.find(line, ',')
                if comma >= 0:
                    key = string.strip(line[:comma])
                    value = string.strip(line[comma+1:])
                    if key[:5] == 'deny.':
                        key = key[5:]
                        list = data.get(key, [])
                        list.append([self.__DENY, value])
                        data[key] = list
                    elif key[:7] == 'accept.':
                        key = key[7:]
                        list = data.get(key, [])
                        list.append([self.__ACCEPT, value])
                        data[key] = list
                    elif key == 'log':
                        head, tail = os.path.split(self.__cfg)
                        value = string.join(('./', value), '')
                        value = os.path.join(head, value)
                        value = os.path.abspath(value)
                        data[key] = value
                    else:
                        pass # error
            line = self.readline(file)
        else:
            if not name:
                print '*WARNING : aya_security.cfg - no entry found for %s.' % os.path.join(self.__aya_dir, 'aya.dll')
                return
        self.__fwrite.extend(data.get('fwrite', []))
        for i in range(len(self.__fwrite)):
            self.__fwrite[i][1] = self.expand_path(self.__fwrite[i][1])
        self.__fread.extend(data.get('fread', []))
        for i in range(len(self.__fread)):
            self.__fread[i][1] = self.expand_path(self.__fread[i][1])
        self.__loadlib.extend(data.get('loadlib', []))
        if data.has_key('log'):
            self.__logfile = data['log']
    def expand_path(self, path):
        head, tail = os.path.split(self.__cfg)
        if path == '*':
            return path
        if path == '':
            return head
        meta = string.rfind(path, '%CFGDIR')
        if meta >= 0:
            path = string.lower(string.replace(path[meta+7:], '\\', '/'))
            path = string.join(('./', path), '')
            path = os.path.join(head, path)
        else:
            path = string.lower(string.replace(path, '\\', '/'))
        path = os.path.abspath(path)
        return path
    def readline(self, file):
        while 1:
            line = kanjilib.sjis2euc(file.readline(), 'ignore')
            if not line:
                break # EOF
            if self.comment:
                end = find_not_quoted(line, '*/')
                if end < 0:
                    continue
                else:
                    line = line[end+2:]
                    self.comment = 0
            while 1:
                start, end = find_comment(line)
                if start < 0:
                    break
                if end < 0:
                    self.comment = 1
                    line = line[:start]
                    break
                line = string.join((line[:start] + ' ' + line[end:]), '')
            line = line_strip(line)
            if not line:
                continue
            break
        return line
    def check_path(self, path, flag='w'):
        result = 0
        abspath = os.path.abspath(path)
        head, tail = os.path.split(abspath)
        if tail != "aya_security.cfg":
            if flag in ['w', 'w+', 'r+', 'a', 'a+']:
                for perm, name in self.__fwrite:
                    if name == '*' or abspath[:len(name)] == name:
                        if perm == self.__ACCEPT:
                            result = 1
                        elif perm == self.__DENY:
                            result = 0
                        else:
                            continue
                        break
            elif flag == 'r':
                result = 1 # default
                for perm, name in self.__fread:
                    if name == '*' or abspath[:len(name)] == name:
                        if perm == self.__ACCEPT:
                            result = 1
                        elif perm == self.__DENY:
                            result = 0
                        else:
                            continue
                        break
        if self.__logfile and result == 0:
            if flag == 'r':
                self.logging('ĤƤʤեޤϥǥ쥯ȥ곬ؤɤ߼֥åޤ.', 'file', abspath)
            else:
                self.logging('ĤƤʤեޤϥǥ쥯ȥ곬ؤؤν񤭹ߤ֥åޤ.', 'file', abspath)
        return result
    def check_lib(self, dll):
        result = 1 # default
        head, tail = os.path.split(string.lower(string.replace(dll, '\\', '/')))
        dll_name = tail
        for perm, name in self.__loadlib:
            if name == '*' or dll_name == name:
                if perm == self.__ACCEPT:
                    result = 1
                elif perm == self.__DENY:
                    result = 0
        if self.__logfile and result == 0:
            self.logging('ĤƤʤ DLL Υɤ֥åޤ.', 'dll ', dll_name)
        return result
    def logging(self, message, target_type, target):
        if self.__logfile == None:
            return None
        else:
            try:
                file = open(self.__logfile, 'a')
            except:
                print 'cannnot open %s' % self.__logfile
                return None
        fcntl.lockf(file.fileno(), fcntl.LOCK_EX)
        aya_dll = os.path.join(self.__aya_dir, 'aya.dll')
        line = string.join(('*WARNING : ', str(message), '\n'), '')
        line = string.join((line, 'AYA  : ', aya_dll, '\n'), '')
        line = string.join((line, 'date : ', time.strftime('%Y/%m/%d(%a) %H:%M:%S'), '\n'), '')
        line = string.join((line, str(target_type), ' : ', str(target), '\n'), '')
        file.write(line)
        file.write('\n')
        fcntl.lockf(file.fileno(), fcntl.LOCK_UN)
        file.close()
        return None

class AyaDictionary:
    def __init__(self, aya):
        self.aya = aya
        self.functions = {}
        self.global_macro = {}
    def get_function(self, name):
        return self.functions.get(name, None)
    def load(self, file, encrypted):
        all_lines = []
        local_macro = {}
        logical_line = ''
        comment = 0
        while 1:
            if encrypted:
                line = decrypt_readline(file)
            else:
                line = file.readline()
            line = kanjilib.sjis2euc(line, 'ignore')
            if not line:
                break # EOF
            if comment:
                end = find_not_quoted(line, '*/')
                if end < 0:
                    continue
                else:
                    line = line[end+2:]
                    comment = 0
            while 1:
                start, end = find_comment(line)
                if start < 0:
                    break
                if end < 0:
                    comment = 1
                    line = line[:start]
                    break
                line = string.join((line[:start] + ' ' + line[end:]), '')
            line = line_strip(line)
            if not line:
                continue
            if line[-1:] == '/':
                logical_line = logical_line + line[:-1]
            else:
                logical_line = logical_line + line
                buffer = line
                # preprosess
                if buffer[0] == '#':
                    buffer = line_strip(buffer[1:])
                    for (tag, target) in [('define', local_macro),
                                          ('globaldefine', self.global_macro)]:
                        if buffer[:len(tag)] == tag:
                            buffer = line_strip(buffer[len(tag):])
                            i = 0
                            while i < len(buffer):
                                if buffer[i] == ' ' or buffer[i] == '\t' or \
                                   buffer[i:i+2] == '':
                                    key = line_strip(buffer[:i])
                                    target[key] = line_strip(buffer[i:])
                                    break
                                i = i + 1
                            break
                    logical_line = '' # reset
                    continue
                for macro in [local_macro, self.global_macro]:
                    logical_line = self.preprosess(macro, logical_line)
                # multi statement
                list_lines = self.split_line(line_strip(logical_line))
                if len(list_lines) > 0:
                    all_lines.extend(list_lines)
                logical_line = '' # reset
        for line in all_lines:
            while 1:
                pos = find_not_quoted(line, '')
                if pos >= 0:
                    line = string.join((line[:pos], ' ', line[pos+2:]), '')
                else:
                    break
        self.evaluate_lines(all_lines, os.path.split(file.name)[1])
    def split_line(self, line):
        lines = []
        while 1:
            if not line:
                break
            pos = len(line) # not len(line)-1
            token = ''
            for x in ['{', '}']:
                pos_new = find_not_quoted(line, x)
                if pos_new >= 0 and pos_new < pos:
                    pos = pos_new
                    token = x
            new = line_strip(line[:pos])
            line = line_strip(line[pos+len(token):])
            if new:
                lines.append(new)
            if token != '':
                lines.append(token)
        return lines
    def preprosess(self, macro, line):
        for key in macro.keys():
            line = string.replace(line, key, macro[key])
        return line
    __SPECIAL_CHARS = [r']', r'(', r')', r'[', r'+', r'-', r'*', r'/', r'=', r':', r';', r'!', r'{', r'}', r'%', r'&', r'#', r'"', r'<', r'>', r',', r'?']
    def evaluate_lines(self, lines, file_name):
        prev = None
        name = None
        function = None
        option = None
        block_nest = 0
        for i in range(len(lines)):
            line = lines[i]
            if line == '{':
                if name != None:
                    if block_nest > 0:
                        function.append(line)
                    block_nest = block_nest + 1
                else:
                    if prev is None:
                        print 'syntax error in %s: unbalanced "{" at the top of file' % file_name
                    else:
                        print 'syntax error in %s: unbalanced "{" at the bottom of function "%s"' % (file_name, prev)
            elif line == '}':
                if name != None:
                    block_nest = block_nest - 1
                    if block_nest > 0:
                        function.append(line)
                    elif block_nest == 0:
                        self.functions[name] = AyaFunction(self, name, function,
                                                           option)
                        # reset
                        prev = name
                        name = None
                        function = None
                        option = None
                else:
                    if prev is None:
                        print 'syntax error in %s: unbalanced "}" at the top of file' % file_name                    
                    else:
                        print 'syntax error in %s: unbalanced "}" at the bottom of function "%s"' % (file_name, prev)
                    block_nest = 0
            elif name == None:
                colon = string.find(line, ':')
                if colon > 0:
                    option = string.strip(line[colon+1:])
                    name = string.strip(line[:colon])
                else:
                    name = line
                for char in self.__SPECIAL_CHARS:
                    if string.find(name, char) >= 0:
                        print 'illegal function name "%s" in %s' % (name, file_name)
                function = []
            else:
                if name != None and block_nest > 0:
                    function.append(line)
                else:
                    print 'syntax error in %s: %s' % (file_name, line)

class AyaFunction:
    __TYPE_INT = 10
    __TYPE_FLOAT = 11
    __TYPE_DECISION = 12
    __TYPE_RETURN = 13
    __TYPE_BLOCK = 14
    __TYPE_SUBSTITUTION = 15
    __TYPE_INC = 16
    __TYPE_DEC = 17
    __TYPE_IF = 18
    __TYPE_WHILE = 19
    __TYPE_FOR = 20
    __TYPE_BREAK = 21
    __TYPE_CONTINUE = 22
    __TYPE_SWITCH = 23
    __TYPE_CASE = 24
    __TYPE_STRING_LITERAL = 25
    __TYPE_STRING = 26
    __TYPE_OPERATOR = 27
    __TYPE_STATEMENT = 28
    __TYPE_CONDITION = 29
    __TYPE_SYSTEM_FUNCTION = 30
    __TYPE_FUNCTION = 31
    __TYPE_ARRAY_POINTER = 32
    __TYPE_ARRAY = 33
    __TYPE_VARIABLE_POINTER = 34
    __TYPE_VARIABLE = 35
    __TYPE_TOKEN = 36
    __CODE_NONE = 40
    __CODE_RETURN = 41
    __CODE_BREAK = 42
    __CODE_CONTINUE = 43
    __re_f = re.compile('[-+]?\d+(\.\d*)$')
    __re_d = re.compile('[-+]?\d+$')
    __re_b = re.compile('[-+]?0[bB][01]+$')
    __re_x = re.compile('[-+]?0[xX][\dA-Fa-f]+$')
    __re_if = re.compile('if\s')
    __re_elseif = re.compile('elseif\s')
    __re_while = re.compile('while\s')
    __re_for = re.compile('for\s')
    __re_switch = re.compile('switch\s')
    __re_case = re.compile('case\s')
    __re_when = re.compile('when\s')
    __SPECIAL_CHARS = [r']', r'(', r')', r'[', r'+', r'-', r'*', r'/', r'=', r':', r';', r'!', r'{', r'}', r'%', r'&', r'#', r'"', r'<', r'>', r',', r'?']
    def __init__(self, dic, name, lines, option):
        self.dic = dic
        self.name = name
        self.status = self.__CODE_NONE
        self.lines = self.parse(lines)
        if option == 'nonoverlap':
            self.nonoverlap = [[], [], []]
        else:
            self.nonoverlap = None
        if option == 'sequential':
            self.sequential = [[], []]
        else:
            self.sequential = None
    def parse(self, lines):
        result = []
        n_lines = len(lines)
        i = 0
        while i < len(lines):
            line = lines[i]
            if line == '--':
                result.append([self.__TYPE_DECISION, []])
            elif line == 'return':
                result.append([self.__TYPE_RETURN, []])
            elif line == 'break':
                result.append([self.__TYPE_BREAK, []])
            elif line == 'continue':
                result.append([self.__TYPE_CONTINUE, []])
            elif line == '{':
                inner_func = []
                i, inner_func = self.get_block(lines, i)
                result.append([self.__TYPE_BLOCK, self.parse(inner_func)])
            elif self.__re_if.match(line):
                inner_blocks = []
                while 1:
                    current_line = lines[i]
                    if self.__re_if.match(current_line):
                        condition_tokens = AyaStatement(string.strip(current_line[2:])).tokens
                        condition = self.parse_condition(condition_tokens)
                    elif self.__re_elseif.match(current_line):
                        condition_tokens = AyaStatement(string.strip(current_line[6:])).tokens
                        condition = self.parse_condition(condition_tokens)
                    else:
                        condition = [self.__TYPE_CONDITION, None]
                    inner_block = []
                    i, inner_block = self.get_block(lines, i+1)
                    if condition is None:
                        inner_blocks = []
                        break
                    entry = []
                    entry.append(condition)
                    entry.append(self.parse(inner_block))
                    inner_blocks.append(entry)
                    if i+1 >= len(lines):
                        break
                    next_line = lines[i+1]
                    if not self.__re_elseif.match(next_line) and \
                       next_line != 'else':
                        break
                    i = i + 1
                if inner_blocks:
                    result.append([self.__TYPE_IF, inner_blocks])
            elif self.__re_while.match(line):
                condition_tokens = AyaStatement(string.strip(line[5:])).tokens
                condition = self.parse_condition(condition_tokens)
                inner_block = []
                i, inner_block = self.get_block(lines, i+1)
                result.append([self.__TYPE_WHILE, [condition, self.parse(inner_block)]])
            elif self.__re_for.match(line):
                inner_block = []
                i, inner_block = self.get_block(lines, i+1)
                end = find_not_quoted(line, ';')
                if end < 0:
                    print 'syntax error in function "%s": illegal for statement "%s"' % (self.name, line)
                else:
                    init = self.parse([string.strip(line[3:end])])
                    condition = string.strip(line[end+1:])
                    end = find_not_quoted(condition, ';')
                    if end < 0:
                        print 'syntax error in function "%s": illegal for statement "%s"' % (self.name, line)
                    else:
                        reset = self.parse([string.strip(condition[end+1:])])
                        condition_tokens = AyaStatement(string.strip(condition[:end])).tokens
                        condition = self.parse_condition(condition_tokens)
                        if condition is not None:
                            result.append([self.__TYPE_FOR, [[init, condition, reset], self.parse(inner_block)]])
            elif self.__re_switch.match(line):
                index = self.parse_token(string.strip(line[6:]))
                ##assert index[0] in [] # FIXME
                inner_block = []
                i, inner_block = self.get_block(lines, i+1)
                result.append([self.__TYPE_SWITCH, [index, self.parse(inner_block)]])
            elif self.__re_case.match(line):
                left = self.parse_token(string.strip(line[4:]))
                ## assert left[0] in [] # FIXME
                i, block = self.get_block(lines, i+1)
                inner_blocks = []
                j = 0
                while 1:
                    current_line = block[j]
                    if self.__re_when.match(current_line):
                        right = string.strip(current_line[4:])
                    else: # 'others'
                        right = None
                    inner_block = []
                    j, inner_block = self.get_block(block, j+1)
                    if right is not None:
                        argument = AyaArgument(right)
                        while argument.has_more_tokens():
                            entry = []
                            right = argument.next_token()
                            tokens = AyaStatement(right).tokens
                            if tokens[0] in ['-', '+']:
                                min = self.parse_statement([tokens.pop(0), tokens.pop(0)])
                            else:
                                min = self.parse_statement([tokens.pop(0)])
                            max = min
                            if tokens:
                                if tokens[0] != '-':
                                    print 'syntax error in function "%s": when %s' % (self.name, right)
                                    continue
                                else:
                                    tokens.pop(0)
                                if len(tokens) > 2 or \
                                   (len(tokens) == 2 and not tokens[0] in ['-', '+']):
                                    print 'syntax error in function "%s": when %s' % (self.name, right)
                                    continue
                                else:
                                    max = self.parse_statement(tokens)
                            entry.append([min, max])
                            entry.append(self.parse(inner_block))
                            inner_blocks.append(entry)
                    else:
                        entry = []
                        entry.append(right)
                        entry.append(self.parse(inner_block))
                        inner_blocks.append(entry)
                    if j+1 == len(block):
                        break
                    next_line = block[j+1]
                    if not self.__re_when.match(next_line) and \
                       next_line != 'others':
                        break
                    j = j + 1
                result.append([self.__TYPE_CASE, [left, inner_blocks]])
            elif find_not_quoted(line, ';') >= 0:
                end = find_not_quoted(line, ';')
                new_line = string.strip(line[end+1:])
                line = string.strip(line[:end])
                new_lines = lines[:i]
                if line:
                    new_lines.append(line)
                if new_line:
                    new_lines.append(new_line)
                new_lines.extend(lines[i+1:])
                lines = new_lines
                continue
            elif self.is_substitution(line):
                tokens = AyaStatement(line).tokens
                left = self.parse_token(tokens[0])
                if left[0] not in [self.__TYPE_ARRAY,
                                   self.__TYPE_VARIABLE,
                                   self.__TYPE_TOKEN]:
                    print 'syntax error in function "%s": illegall substitution "%s"' % (self.name, line)
                else:
                    if left[0] == self.__TYPE_TOKEN: # this cannot be FUNCTION
                        left[0] = self.__TYPE_VARIABLE
                        left[1] = [left[1], None]
                    ope = [self.__TYPE_OPERATOR, tokens[1]]
                    right = self.parse_statement(tokens[2:])
                    result.append([self.__TYPE_SUBSTITUTION, [left, ope, right]])
            elif self.is_inc_or_dec(line): # ++/--
                ope = line[-2:]
                var = self.parse_token(line[:-2])
                if var[0] not in [self.__TYPE_ARRAY,
                                  self.__TYPE_VARIABLE,
                                  self.__TYPE_TOKEN]:
                    print 'syntax error in function "%s": illegall increment/decrement "%s"' % (self.name, line)
                else:
                    if var[0] == self.__TYPE_TOKEN:
                        var[0] = self.__TYPE_VARIABLE
                        var[1] = [var[1], None]
                    if ope == '++':
                        result.append([self.__TYPE_INC, var])
                    elif ope == '--':
                        result.append([self.__TYPE_DEC, var])
                    else:
                        return None # should not reach here
            else:
                tokens = AyaStatement(line).tokens
                if tokens[-1] == '"': # This is kluge.
                    print 'syntax error in function "%s": unbalanced \'"\' or \'"\' in string %s' % (self.name, string.join(tokens, ''))
                    token = string.join(tokens[:-1], '')
                    if len(token) > 0 and token[0] == '"':
                        token = token[1:]
                    if len(token) > 0:
                        if string.find(token, '%') < 0:
                            result.append([self.__TYPE_STRING_LITERAL, token])
                        else:
                            result.append([self.__TYPE_STRING, token])
                elif len(tokens) == 1:
                    result.append(self.parse_token(tokens[0]))
                else:
                    result.append(self.parse_statement(tokens))
            i = i + 1
        result.append([self.__TYPE_DECISION, []])
        return result
    def parse_statement(self, statement_tokens):
        n_tokens = len(statement_tokens)
        statement = []
        if n_tokens == 1:
            statement = [self.__TYPE_STATEMENT, self.parse_token(statement_tokens[0])]
        elif statement_tokens[0] in ['+', '-']:
            tokens = ['0']
            tokens.extend(statement_tokens)
            statement = self.parse_statement(tokens)
        else:
            ope_index = None
            for ope in ['+', '-']:
                if ope in statement_tokens:
                    new_index = statement_tokens.index(ope)
                    if ope_index == None or new_index < ope_index:
                        ope_index = new_index
            if ope_index == None:
                statement_tokens.reverse()
                try:
                    for ope in ['*', '/', '%']:
                        if ope in statement_tokens:
                            new_index = statement_tokens.index(ope)
                            if ope_index == None or new_index < ope_index:
                                ope_index = new_index
                    if ope_index != None:
                        ope_index = -1 - ope_index
                finally:
                    statement_tokens.reverse()
            if ope_index in [None, -1, 0, n_tokens-1]:
                if statement_tokens[0][0] == '"' and \
                   statement_tokens[0][-1] == '"' and \
                   statement_tokens[-1][0] == '"' and \
                   statement_tokens[-1][-1] == '"':
                    print 'syntax error in function "%s": \'"\' in string %s' % (self.name, string.join(statement_tokens))
                    return self.parse_token(string.join(statement_tokens))
                else:
                    print 'syntax error in function "%s": illegal statement "%s"' % (self.name, string.join(statement_tokens))
                    return []
            else:
                ope = [self.__TYPE_OPERATOR, statement_tokens[ope_index]]
                if len(statement_tokens[:ope_index]) == 1:
                    if statement_tokens[0][0] == '(':
                        tokens = AyaStatement(statement_tokens[0][1:-1]).tokens
                        left = self.parse_statement(tokens)
                    else:
                        left = self.parse_token(statement_tokens[:ope_index][0])
                else:
                    left = self.parse_statement(statement_tokens[:ope_index])
                if len(statement_tokens[ope_index+1:]) == 1:
                    if statement_tokens[-1][0] == '(':
                        tokens = AyaStatement(statement_tokens[ope_index+1][1:-1]).tokens
                        right = self.parse_statement(tokens)
                    else:
                        right = self.parse_token(statement_tokens[ope_index+1:][0])
                else:
                    right = self.parse_statement(statement_tokens[ope_index+1:])
                statement = [self.__TYPE_STATEMENT, left, ope, right]
        return statement
    def parse_condition(self, condition_tokens):
        n_tokens = len(condition_tokens)
        condition = None
        ope_index = None
        condition_tokens.reverse()
        try:
            for ope in ['&&', '||']:
                if ope in condition_tokens:
                    new_index = condition_tokens.index(ope)
                    if ope_index == None or new_index < ope_index:
                        ope_index = new_index
            if ope_index != None:
                ope_index = -1 - ope_index
        finally:
            condition_tokens.reverse()
        if ope_index == None:
            for ope in ['==', '!=', '>', '<', '>=', '<=', '_in_', '!_in_']:
                if ope in condition_tokens:
                    new_index = condition_tokens.index(ope)
                    if ope_index == None or new_index < ope_index:
                        ope_index = new_index
            if ope_index in [None, -1, 0, n_tokens-1]:
                print 'syntax error in function "%s": illegal condition "%s"' % (self.name, string.join(condition_tokens))
                return None
            ope = [self.__TYPE_OPERATOR, condition_tokens[ope_index]]
            if len(condition_tokens[:ope_index]) == 1:
                left = self.parse_token(condition_tokens[:ope_index][0])
            else:
                left = self.parse_statement(condition_tokens[:ope_index])
            if len(condition_tokens[ope_index+1:]) == 1:
                right = self.parse_token(condition_tokens[ope_index+1:][0])
            else:
                right = self.parse_statement(condition_tokens[ope_index+1:])
            condition = [self.__TYPE_CONDITION, [left, ope, right]]
        else:
            ope = [self.__TYPE_OPERATOR, condition_tokens[ope_index]]
            left = self.parse_condition(condition_tokens[:ope_index])
            right = self.parse_condition(condition_tokens[ope_index+1:])
            if left is not None and right is not None:
                condition = [self.__TYPE_CONDITION, [left, ope, right]]
        return condition
    def parse_argument(self, args):
        argument = AyaArgument(args)
        arguments = []
        while argument.has_more_tokens():
            token = argument.next_token()
            if token[0] == '&':
                result = self.parse_token(token[1:])
                if result[0] == self.__TYPE_ARRAY:
                    arguments.append([self.__TYPE_ARRAY_POINTER, result[1]])
                elif result[0] == self.__TYPE_VARIABLE:
                    arguments.append([self.__TYPE_VARIABLE_POINTER, result[1]])
                elif result[0] == self.__TYPE_TOKEN:
                    arguments.append([self.__TYPE_VARIABLE_POINTER, [result[1], None]])
                else:
                    print 'syntax error in function "%s": illegal argument "%s"' % (self.name, token)
            elif token[0] == '(':
                if token[-1] != ')':
                    print 'syntax error in function "%s": unbalanced "(" in the string(%s)' % (self.name, line)
                    return None
                else:
                    statement = AyaStatement(token[1:-1])
                    arguments.append(self.parse_statement(statement.tokens))
            else:
                arguments.append(self.parse_statement([token]))
        return arguments
    def parse_token(self, token):
        result = []
        if self.__re_f.match(token):
            result = [self.__TYPE_FLOAT, token]
        elif self.__re_d.match(token):
            result = [self.__TYPE_INT, token]
        elif token[0] == '"':
            text = token[1:]
            if text[-1] == '"':
                text = text[:-1]
            if string.count(text, '"') > 0:
                print 'syntax error in function "%s": \'"\' in string "%s"' % (self.name, text)
            if string.find(text, '%') < 0:
                result = [self.__TYPE_STRING_LITERAL, text]
            else:
                result = [self.__TYPE_STRING, text]
        else:
            pos_parenthesis_open = string.find(token, '(')
            pos_block_open = string.find(token, '[')
            if pos_parenthesis_open != -1 and \
               (pos_block_open == -1 or \
                pos_parenthesis_open < pos_block_open): # function
                if token[-1] != ')':
                    print 'syntax error: unbalnced "(" in "%s"' % token
                else:
                    func_name = token[:pos_parenthesis_open]
                    arguments = self.parse_argument(token[pos_parenthesis_open+1:-1])
                    for char in self.__SPECIAL_CHARS:
                        if string.find(func_name, char) >=0:
                            print 'illegal character "%s" in the name of function "%s"' % (char, token)
                            break
                    else:
                        if self.dic.aya.get_system_functions().exists(func_name):
                            if func_name == 'LOGGING':
                                result = [self.__TYPE_SYSTEM_FUNCTION, [func_name, arguments, token[pos_parenthesis_open+1:-1]]]
                            else:
                                result = [self.__TYPE_SYSTEM_FUNCTION, [func_name, arguments]]
                        else:
                            result = [self.__TYPE_FUNCTION, [func_name, arguments]]
            elif pos_block_open != -1: # array
                if token[-1] != ']':
                    print 'syntax error: unbalnced "[" in "%s"' % token
                else:
                    array_name = token[:pos_block_open]
                    index = self.parse_token(token[pos_block_open+1:-1])
                    for char in self.__SPECIAL_CHARS:
                        if string.find(array_name, char) >=0:
                            print 'illegal character "%s" in the name of array "%s"' % (char, token)
                            break
                    else:
                        result = [self.__TYPE_ARRAY, [array_name, index]]
            else: # variable or function
                for char in self.__SPECIAL_CHARS:
                    if string.find(token, char) >=0:
                        print 'syntax error in function "%s": illegal character "%s" in the name of function/variable "%s"' % (self.name, char, token)
                        break
                else:
                    result = [self.__TYPE_TOKEN, token]
        return result
    def call(self, argv=None):
        namespace = AyaNamespace(self.dic.aya)
        _argv = []
        if not argv:
            namespace.put('_argc', 0)
        else:
            namespace.put('_argc', len(argv))
            for i in range(len(argv)):
                if type(argv[i]) is DictType:
                    _argv.append(argv[i]['value'])
                else:
                    _argv.append(argv[i])
        namespace.put('_argv', _argv)
        self.status = self.__CODE_NONE
        result = self.evaluate(namespace, self.lines, -1, 0)
        if result == None:
            result = ''
        if argv:
            for i in range(len(argv)):
                if type(argv[i]) is DictType:
                    value = _argv[i]
                    name = argv[i]['name']
                    namespace = argv[i]['namespace']
                    index = argv[i]['index']
                    namespace.put(name, value, index)
        return result
    def evaluate(self, namespace, lines, index_to_return, is_inner_block):
        result = []
        alternatives = []
        n_lines = len(lines)
        i = 0
        while i < n_lines:
            line = lines[i]
            if len(line) < 1:
                i = i + 1
                continue
            if line[0] in [self.__TYPE_DECISION, self.__TYPE_RETURN, self.__TYPE_BREAK, self.__TYPE_CONTINUE] or \
               self.status in [self.__CODE_RETURN, self.__CODE_BREAK, self.__CODE_CONTINUE]:
                if len(alternatives) > 0:
                    if is_inner_block:
                        if index_to_return < 0:
                            result.append(random.choice(alternatives))
                        elif index_to_return <= len(alternatives)-1:
                            result.append(alternatives[index_to_return])
                        else: # out of range
                            result.append('')
                    else:
                        result.append(alternatives)
                    alternatives = []
                if line[0] == self.__TYPE_RETURN or \
                   self.status == self.__CODE_RETURN:
                    self.status = self.__CODE_RETURN
                    break
                elif line[0] == self.__TYPE_BREAK or \
                     self.status == self.__CODE_BREAK:
                    self.status = self.__CODE_BREAK
                    break
                elif line[0] == self.__TYPE_CONTINUE or \
                     self.status == self.__CODE_CONTINUE:
                    self.status = self.__CODE_CONTINUE
                    break
            elif line[0] == self.__TYPE_BLOCK:
                inner_func = line[1]
                local_namespace = AyaNamespace(self.dic.aya, namespace)
                result_of_inner_func = self.evaluate(local_namespace,
                                                     inner_func, -1, 1)
                if result_of_inner_func:
                    alternatives.append(result_of_inner_func)
            elif line[0] == self.__TYPE_SUBSTITUTION:
                left, ope, right = line[1]
                ##assert left[0] in [self.__TYPE_ARRAY, self.__TYPE_VARIABLE]
                ##assert ope[0] == self.__TYPE_OPERATOR
                ope = ope[1]
                if ope in [':=', '+:=', '-:=', '*:=', '/:=', '%:=']:
                    type_float = 1
                else:
                    type_float = 0
                ##assert right[0] == self.__TYPE_STATEMENT
                right_result = self.evaluate_statement(namespace, right, type_float)
                if ope != '=' and ope != ':=':
                    left_result = self.evaluate_token(namespace, left) 
                    right_result = self.operation(left_result, ope[0], right_result, type_float)
                    ope = ope[1:]
                self.substitute(namespace, left, ope, right_result)
            elif line[0] == self.__TYPE_INC or \
                 line[0] == self.__TYPE_DEC: # ++/--
                if line[0] == self.__TYPE_INC:
                    ope = '++'
                elif line[0] == self.__TYPE_DEC:
                    ope = '--'
                else:
                    return None # should not reach here
                var = line[1]
                ## assert var[0] in [self.__TYPE_ARRAY, self.__TYPE_VARIABLE]
                var_name = var[1][0]
                if var_name[0] == '_':
                    target_namespace = namespace
                else:
                    target_namespace = self.dic.aya.get_global_namespace()
                value = self.evaluate_token(namespace, var)
                if var[0] == self.__TYPE_ARRAY: # _argv[n] only
                    index = self.evaluate_token(namespace, var[1][1])
                    try:
                        index = int(index)
                    except:
                        print 'index of array has to be integer: %s[%s]' % (var_name, var[1][1][0])
                        return None
                else:
                    index = None
                if type(value) in [IntType, FloatType]:
                    if ope == '++':
                        target_namespace.put(var_name, int(value)+1, index)
                    elif ope == '--':
                        target_namespace.put(var_name, int(value)-1, index)
                    else:
                        return None # should not reach here
                else:
                    print 'illegal increment/decrement:' \
                          'type of variable %s is not number' % var_name
            elif line[0] == self.__TYPE_IF:
                inner_blocks = line[1]
                n_blocks = len(inner_blocks)
                for j in range(n_blocks):
                    entry = inner_blocks[j]
                    condition = entry[0]
                    inner_block = entry[1]
                    assert condition[0] == self.__TYPE_CONDITION
                    if condition == None or \
                       self.evaluate_condition(namespace, condition) == 1:
                        local_namespace = AyaNamespace(self.dic.aya, namespace)
                        result_of_inner_block = self.evaluate(local_namespace,
                                                              inner_block, -1, 1)
                        if result_of_inner_block:
                            alternatives.append(result_of_inner_block)
                        break
            elif line[0] == self.__TYPE_WHILE:
                condition = line[1][0]
                inner_block = line[1][1]
                assert condition[0] == self.__TYPE_CONDITION
                while self.evaluate_condition(namespace, condition):
                    local_namespace = AyaNamespace(self.dic.aya, namespace)
                    result_of_inner_block = self.evaluate(local_namespace,
                                                          inner_block, -1, 1)
                    if result_of_inner_block != None:
                        alternatives.append(result_of_inner_block)
                    if self.status == self.__CODE_RETURN:
                        break
                    if self.status == self.__CODE_BREAK:
                        self.status = self.__CODE_NONE
                        break
                    if self.status == self.__CODE_CONTINUE:
                        self.status = self.__CODE_NONE
            elif line[0] == self.__TYPE_FOR:
                init = line[1][0][0]
                condition = line[1][0][1]
                reset = line[1][0][2]
                inner_block = line[1][1]
                self.evaluate(namespace, init, -1, 1)
                assert condition[0] == self.__TYPE_CONDITION
                while self.evaluate_condition(namespace, condition):
                    local_namespace = AyaNamespace(self.dic.aya, namespace)
                    result_of_inner_block = self.evaluate(local_namespace,
                                                          inner_block, -1, 1)
                    if result_of_inner_block != None:
                        alternatives.append(result_of_inner_block)
                    if self.status == self.__CODE_RETURN:
                        break
                    if self.status == self.__CODE_BREAK:
                        self.status = self.__CODE_NONE
                        break
                    if self.status == self.__CODE_CONTINUE:
                        self.status = self.__CODE_NONE
                    self.evaluate(namespace, reset, -1, 1)
            elif line[0] == self.__TYPE_SWITCH:
                index = self.evaluate_token(namespace, line[1][0])
                inner_block = line[1][1]
                try:
                    index = int(index)
                except ValueError:
                    index = 0
                local_namespace = AyaNamespace(self.dic.aya, namespace)
                result_of_inner_block = self.evaluate(local_namespace,
                                                      inner_block, index, 1)
                if result_of_inner_block:
                    alternatives.append(result_of_inner_block)
            elif line[0] == self.__TYPE_CASE:
                left = self.evaluate_token(namespace, line[1][0])
                inner_blocks = line[1][1]
                n_blocks = len(inner_blocks)
                default_result = None
                for j in range(n_blocks):
                    entry = inner_blocks[j]
                    inner_block = entry[1]
                    local_namespace = AyaNamespace(self.dic.aya, namespace)
                    if entry[0] != None:
                        min, max = entry[0]
                        min = self.evaluate_statement(namespace, min, 1)
                        max = self.evaluate_statement(namespace, max, 1)
                        if left >= min and left <= max:
                            result_of_inner_block = self.evaluate(local_namespace,
                                                                  inner_block, -1, 1)
                            if result_of_inner_block:
                                alternatives.append(result_of_inner_block)
                                break
                    else:
                        default_result = self.evaluate(local_namespace,
                                                       inner_block, -1, 1)
                else:
                    if default_result:
                        alternatives.append(default_result)
            elif line[0] == self.__TYPE_STATEMENT:
                result_of_func = self.evaluate_statement(namespace, line, 0)
                if result_of_func:
                    alternatives.append(result_of_func)
            else:
                result_of_eval = self.evaluate_token(namespace, line)
                if result_of_eval:
                    alternatives.append(result_of_eval)
            i = i + 1
        if not is_inner_block:
            if self.sequential != None:
                list = []
                for alt in result:
                    list.append(len(alt))
                if self.sequential[0] != list:
                    self.sequential[0] = list
                    self.sequential[1] = [0] * len(result)
                else:
                    for index in range(len(result)):
                        current = self.sequential[1][index]
                        if current < len(result[index])-1:
                            self.sequential[1][index] = current + 1
                            break
                        else:
                            self.sequential[1][index] = 0
            if self.nonoverlap != None:
                list = []
                for alt in result:
                    list.append(len(alt))
                if self.nonoverlap[0] != list:
                    self.nonoverlap[0] = list
                    self.nonoverlap[2] = []
                if len(self.nonoverlap[2]) == 0:
                    self.nonoverlap[2].append([0] * len(result))
                    while 1:
                        new = []
                        new.extend(self.nonoverlap[2][-1])
                        for index in range(len(result)):
                            if new[index] < len(result[index])-1:
                                new[index] = new[index] + 1
                                self.nonoverlap[2].append(new)
                                break
                            else:
                                new[index] = 0
                        else:
                            break
                next = random.choice(range(len(self.nonoverlap[2])))
                self.nonoverlap[1] = self.nonoverlap[2][next]
                del self.nonoverlap[2][next]
            for index in range(len(result)):
                if self.sequential != None:
                    result[index] = result[index][self.sequential[1][index]]
                elif self.nonoverlap != None:
                    result[index] = result[index][self.nonoverlap[1][index]]
                else:
                    result[index] = random.choice(result[index])
        if len(result) == 0:
            return None
        elif len(result) == 1:
            return result[0]
        else:
            return string.join(tuple(map(str, result)), '')
    def substitute(self, namespace, left, ope, right):
        var_name = left[1][0]
        if var_name[0] == '_':
            target_namespace = namespace
        else:
            target_namespace = self.dic.aya.get_global_namespace()
        if left[0] is not self.__TYPE_ARRAY:
            target_namespace.put(var_name, right)
        else:
            index = self.evaluate_token(namespace, left[1][1])
            try:
                index = int(index)
            except ValueError:
                print 'Could not convert %s to an integer' % index
            else:
                if ope == '=':
                    elem = right
                elif ope == ':=':
                    if type(right) is IntType:
                        elem = float(right)
                    else:
                        elem = right
                else:
                    return None # should not reach here
                target_namespace.put(var_name, elem, index)
    def evaluate_token(self, namespace, token):
        result = '' # default
        if token[0] == self.__TYPE_TOKEN:
            if self.__re_b.match(token[1]):
                pos = self.__re_d.search(token[1]).start()
                result = int(token[1][pos:], 2)
            elif self.__re_x.match(token[1]):
                result = int(token[1], 16)
            else:
                func = self.dic.get_function(token[1])
                system_functions = self.dic.aya.get_system_functions()
                if func:
                    result = func.call()
                elif system_functions.exists(token[1]):
                    result = system_functions.call(namespace, token[1], [])
                elif token[1][:6] == 'random': # ver.3
                    result = int(random.randrange(0, 100, 1))
                else:
                    if token[1][0] == '_':
                        target_namespace = namespace
                    else:
                        target_namespace = self.dic.aya.get_global_namespace()
                    if target_namespace.exists(token[1]):
                        result = target_namespace.get(token[1])
        elif token[0] == self.__TYPE_STRING_LITERAL:
            result = token[1]
        elif token[0] == self.__TYPE_STRING:
            result = self.evaluate_string(namespace, token[1])
        elif token[0] == self.__TYPE_INT:
            result = int(token[1])
        elif token[0] == self.__TYPE_FLOAT:
            result = float(token[1])
        elif token[0] == self.__TYPE_SYSTEM_FUNCTION:
            system_functions = self.dic.aya.get_system_functions()
            func_name = token[1][0]
            ##assert system_functions.exists(func_name)
            ##raise Exception(string.join(('function ', func_name, ' not found.'), ''))
            arguments = self.evaluate_argument(namespace, func_name, token[1][1], 1)
            if func_name == 'CALLBYNAME':
                func = self.dic.get_function(arguments[0])
                system_functions = self.dic.aya.get_system_functions()
                if func:
                    result = func.call()
                elif system_functions.exists(arguments[0]):
                    result = system_functions.call(namespace, arguments[0], [])
            elif func_name == 'LOGGING':
                arguments.insert(0, token[1][2])
                arguments.insert(0, self.name)
                arguments.insert(0, self.dic.aya.logfile)
                result = system_functions.call(namespace, func_name, arguments)
            else:
                result = system_functions.call(namespace, func_name, arguments)
        elif token[0] == self.__TYPE_FUNCTION:
            func_name = token[1][0]
            func = self.dic.get_function(func_name)
            ##assert func != None:
            ##raise Exception(string.join(('function ', func_name, ' not found.'), ''))
            arguments = self.evaluate_argument(namespace, func_name, token[1][1], 0)
            result = func.call(arguments)
        elif token[0] == self.__TYPE_ARRAY:
            var_name = token[1][0]
            if var_name[0] == '_':
                target_namespace = namespace
            else:
                target_namespace = self.dic.aya.get_global_namespace()
            index = self.evaluate_token(namespace, token[1][1])
            try:
                index = int(index)
            except:
                print 'index of array has to be integer: %s[%s]' % (var_name, token[1][1])
            else:
                if var_name == 'random': # Ver.3
                    result = int(random.randrange(0, index, 1))
                elif var_name == 'ascii': # Ver.3
                    if index >= 0 and index < 0x80:
                        result = chr(index)
                    else:
                        result = ' '
                elif target_namespace.exists(var_name):
                    result = target_namespace.get(var_name, index)
        elif token[0] == self.__TYPE_VARIABLE:
            var_name = token[1][0]
            if var_name[0] == '_':
                target_namespace = namespace
            else:
                target_namespace = self.dic.aya.get_global_namespace()
            if target_namespace.exists(var_name):
                result = target_namespace.get(var_name)
        elif token[0] == self.__TYPE_ARRAY_POINTER:
            var_name = token[1][0]
            if var_name[0] == '_':
                target_namespace = namespace
            else:
                target_namespace = self.dic.aya.get_global_namespace()
            index = self.evaluate_token(namespace, token[1][1])
            try:
                index = int(index)
            except:
                print 'index of array has to be integer: %s[%s]' % (var_name, token[1][1])
            else:
                if var_name == 'random': # Ver.3
                    result = int(random.randrange(0, index, 1))
                elif var_name == 'ascii': # Ver.3
                    if index >= 0 and index < 0x80:
                        result = chr(index)
                    else:
                        result = ' '
                else:
                    value = target_namespace.get(var_name, index)
                    result = {'name': var_name,
                              'index': index,
                              'namespace': target_namespace,
                              'value': value}
        elif token[0] == self.__TYPE_VARIABLE_POINTER:
            var_name = token[1][0]
            if var_name[0] == '_':
                target_namespace = namespace
            else:
                target_namespace = self.dic.aya.get_global_namespace()
            value = target_namespace.get(var_name)
            result = {'name': var_name,
                      'index': None,
                      'namespace': target_namespace,
                      'value': value}
        else:
            print 'error in evaluate_token:', token
        return result
    def evaluate_condition(self, namespace, condition):
        result = 0
        if condition[1] is None:
            return 1
        left = condition[1][0]
        ope = condition[1][1]
        right = condition[1][2]
        assert ope[0] == self.__TYPE_OPERATOR
        if left[0] == self.__TYPE_CONDITION:
            left_result = self.evaluate_condition(namespace, left)
        elif left[0] == self.__TYPE_STATEMENT:
            left_result = self.evaluate_statement(namespace, left, 1)
        else:
            left_result = self.evaluate_token(namespace, left)
        if right[0] == self.__TYPE_CONDITION:
            right_result = self.evaluate_condition(namespace, right)
        elif right[0] == self.__TYPE_STATEMENT:
            right_result = self.evaluate_statement(namespace, right, 1)
        else:
            right_result = self.evaluate_token(namespace, right)
        if ope[1] == '==':
            result = (left_result == right_result)
        elif ope[1] == '!=':
            result = (left_result != right_result)
        elif ope[1] == '_in_':
            if type(right_result) is StringType and type(left_result) is StringType:
                result = (string.find(right_result, left_result) >= 0)
            else:
                result = 0
        elif ope[1] == '!_in_':
            if type(right_result) is StringType and type(left_result) is StringType:
                result = (string.find(right_result, left_result) == -1)
            else:
                result = 0
        elif ope[1] == '<':
            result = left_result < right_result
        elif ope[1] == '>':
            result = left_result > right_result
        elif ope[1] == '<=':
            result = left_result <= right_result
        elif ope[1] == '>=':
            result = left_result >= right_result
        elif ope[1] == '||':
            result = left_result or right_result
        elif ope[1] == '&&':
            result = left_result and right_result
        else:
            pass
        return result
    def evaluate_statement(self, namespace, statement, type_float):
        num = len(statement[1:])
        if num == 0:
            return ''
        type = statement[0]
        token = statement[1]
        if type == self.__TYPE_STATEMENT:
            left = self.evaluate_statement(namespace, token, type_float)
        else:
            left = self.evaluate_token(namespace, statement)
        ##else:
        ##    print 'illegal statement: %s' % string.join(statement[1])
        ##    return ''
        if num == 3:
            ##assert statement[2][0] == self.__TYPE_OPERATOR
            ope = statement[2][1]
            type = statement[3][0]
            if type == self.__TYPE_INT:
                token = statement[3][1]
                if type_float:
                    right = float(token)
                else:
                    right = int(token)
            elif type == self.__TYPE_FLOAT:
                token = statement[3][1]
                if type_float:
                    right = float(token)
                else:
                    right = int(float(token))
            elif type == self.__TYPE_STATEMENT:
                right = self.evaluate_statement(namespace, statement[3], type_float)
            else:
                right = self.evaluate_token(namespace, statement[3])
            ##else:
            ##    print 'illegal statement: %s' % string.join(statement[0][1], statement[1][1], statement[2][1])
            ##    return ''            
            result = self.operation(left, ope, right, type_float)
        else:
            result = left
        return result
    def operation(self, left, ope, right, type_float):
        try:
            if type_float:
                left = float(left)
                right = float(right)
            elif ope != '+' or (type(left) is not StringType and \
                                type(right) is not StringType):
                left = int(left)
                right = int(right)
            else:
                left = str(left)
                right = str(right)
        except:
            left = str(left)
            right = str(right)
        try:
            if ope == '+':
                return left + right
            elif ope == '-':
                return left - right
            elif ope == '*':
                return left * right
            elif ope == '/':
                if right == 0:
                    return 0
                else:
                    return left / right
            elif ope == '%':
                return left % right
        except:
            print 'illegal operation: %s' % string.join([str(left), str(ope), str(right)])
            return ''
    def get_block(self, parent, startpoint):
        result = []
        n_lines = len(parent)
        inner_nest_level = 0
        for i in range(startpoint, n_lines):
            inner_content = parent[i]
            if inner_content == '{':
                if inner_nest_level > 0:
                    result.append(inner_content)
                inner_nest_level = inner_nest_level + 1
            elif inner_content == '}':
                inner_nest_level = inner_nest_level - 1
                if inner_nest_level > 0:
                    result.append(inner_content)
            else:
                result.append(inner_content)
            if inner_nest_level == 0:
                return i, result
        return startpoint, result
    def evaluate_string(self, namespace, line):
        history = [] # %[n]
        buffer = ''
        startpoint = 0
        system_functions = self.dic.aya.get_system_functions()
        while startpoint < len(line):
            pos = string.find(line, '%', startpoint)
            if pos < 0:
                buffer = string.join((buffer, line[startpoint:]), '')
                startpoint = len(line)
                continue
            else:
                buffer = string.join((buffer, line[startpoint:pos]), '')
                startpoint = pos
            endpoint = len(line)
            for char in self.__SPECIAL_CHARS:
                pos = string.find(line, char, startpoint+2, endpoint)
                if pos > 0 and pos < endpoint:
                    endpoint = pos
            if  line[startpoint+1] == '[': # history
                if line[endpoint] != ']':
                    print 'unbalanced "%[" or illegal index in the string(%s)' % line
                    buffer = ''
                    break
                index_token = self.parse_token(line[startpoint+2:endpoint])
                index = self.evaluate_token(namespace, index_token)
                try:
                    index = int(index)
                except:
                    print 'illegal history index in the string(%s)' % line
                else:
                    if index >= 0 and index < len(history):
                        buffer = string.join((buffer,
                                              self.format(history[index])), '')
                startpoint = endpoint + 1
                continue
            replaced = 0
            while endpoint > startpoint+1:
                token = line[startpoint+1:endpoint]
                if token == 'random' or token == 'ascii': # Ver.3
                    if endpoint < len(line) and \
                       line[endpoint] == '[':
                        end_of_block = string.find(line, ']', endpoint+1)
                        if end_of_block < 0:
                            print 'unbalanced "[" or illegal index in the string(%s)' % line
                            startpoint = len(line)
                            buffer = ''
                            break
                        index = self.parse_token(line[endpoint+1:end_of_block])
                        content_of_var = self.evaluate_token(namespace, [self.__TYPE_ARRAY, [token, index]])
                        if content_of_var == None:
                            content_of_var = ''
                        history.append(content_of_var)
                        buffer = string.join((buffer,
                                              self.format(content_of_var)), '')
                        startpoint = end_of_block+1
                        replaced = 1
                        break
                func = self.dic.get_function(token)
                is_system_func = system_functions.exists(token)
                if func != None or is_system_func:
                    if endpoint < len(line) and \
                       line[endpoint] == '(':
                        end_of_parenthesis = string.find(line, ')', endpoint+1)
                        if end_of_parenthesis < 0:
                            print 'unbalanced "(" in the string(%s)' % line
                            startpoint = len(line)
                            buffer = ''
                            break
                        func_name = token
                        arguments = self.parse_argument(line[endpoint+1:end_of_parenthesis])
                        arguments = self.evaluate_argument(namespace, func_name, arguments, is_system_func)
                        if is_system_func:
                            if func_name == 'CALLBYNAME':
                                func = self.dic.get_function(arguments[0])
                                if func:
                                    result_of_func = func.call()
                                elif system_functions.exists(arguments[0]):
                                    result_of_func = system_functions.call(namespace, arguments[0], [])
                            elif func_name == 'LOGGING':
                                arguments.insert(0, line[endpoint+1:end_of_parenthesis])
                                arguments.insert(0, self.name)
                                arguments.insert(0, self.dic.aya.logfile)
                                result_of_func = system_functions.call(namespace, func_name, arguments)
                            else:
                                result_of_func = system_functions.call(namespace, func_name, arguments)
                        else:
                            result_of_func = func.call(arguments)
                        if result_of_func == None:
                            result_of_func = ''
                        history.append(result_of_func)
                        buffer = string.join((buffer,
                                              self.format(result_of_func)), '')
                        startpoint = end_of_parenthesis+1
                        replaced = 1
                        break
                    elif func != None:
                        result_of_func = func.call()
                        history.append(result_of_func)
                        buffer = string.join((buffer,
                                              self.format(result_of_func)), '')
                        startpoint = endpoint
                        replaced = 1
                        break
                    else:
                        result_of_func = system_functions.call(namespace, token, [])
                        if result_of_func == None:
                            result_of_func = ''
                        history.append(result_of_func)
                        buffer = string.join((buffer,
                                              self.format(result_of_func)), '')
                        startpoint = endpoint
                        replaced = 1
                        break
                else:
                    if token[0] == '_':
                        target_namespace = namespace
                    else:
                        target_namespace = self.dic.aya.get_global_namespace()
                    if target_namespace.exists(token):
                        have_index = 0
                        index = None
                        if endpoint < len(line) and line[endpoint] == '[':
                            end_of_block = string.find(line, ']', endpoint+1)
                            if end_of_block < 0:
                                print 'unbalanced "[" or illegal index in the string(%s)' % line
                                startpoint = len(line)
                                buffer = ''
                                break
                            have_index = 1
                            index_token = self.parse_token(line[endpoint+1:end_of_block])
                            index = self.evaluate_token(namespace, index_token)
                            try:
                                index = int(index)
                            except ValueError:
                                have_index = 0
                                index = None
                        value = target_namespace.get(token, index)
                        if value != None:
                            content_of_var = value
                            history.append(content_of_var)
                            buffer = string.join((buffer,
                                                  self.format(content_of_var)), '')
                            if have_index:
                                startpoint = end_of_block+1
                            else:
                                startpoint = endpoint
                            replaced = 1
                            break
                endpoint = endpoint - 1
            if not replaced:
                buffer = string.join((buffer, line[startpoint]), '')
                startpoint = startpoint + 1
        return buffer
    def format(self, input):
        if type(input) is FloatType:
            result = str(round(input, 6))
        else:
            result = str(input)
        return result
    def evaluate_argument(self, namespace, name, argument, is_system_func):
        arguments = []
        for i in range(len(argument)):
            if is_system_func and \
               self.dic.aya.get_system_functions().not_to_evaluate(name, i):
                ## assert argument[i] in [] ## FIXME
                arguments.append(argument[i][1][1])
            else:
                arguments.append(self.evaluate_statement(namespace, argument[i], 1))
        if is_system_func:
            if name == 'NAMETOVALUE' and \
               len(arguments) == 1: # this is kluge
                arguments[0] = self.evaluate_statement(namespace, argument[0], 1)
        return arguments
    def is_substitution(self, line):
        statement = AyaStatement(line)
        if statement.countTokens() >= 3:
            statement.next_token() # left
            ope = statement.next_token()
            ope_list = ['=', ':=',
                        '+=', '-=', '*=', '/=', '%=',
                        '+:=', '-:=', '*:=', '/:=', '%:=']
            if ope in ope_list:
                return 1
        return 0
    def is_inc_or_dec(self, line):
        if len(line) <= 2:
            return 0
        if line[-2:] in ['++', '--']:
            return 1
        else:
            return 0

class AyaSystemFunctions:
    def __init__(self, aya):
        self.aya = aya
        self.saori_statuscode = ''
        self.saori_header = []
        self.saori_value = {}
        self.saori_protocol = ''
        self.errno = 0
        self.security = AyaSecurity(self.aya)        
        self.functions = {
            'TONUMBER': [self.TONUMBER, [0], [1], None],
            'TOSTRING': [self.TOSTRING, [0], [1], None],
            'TONUMBER2': [self.TONUMBER2, [None], [1], None],
            'TOSTRING2': [self.TOSTRING2, [None], [1], None],
            'TOUPPER': [self.TOUPPER, [None], [1], None],
            'TOLOWER': [self.TOLOWER, [None], [1], None],
            'TOBINSTR': [self.TOBINSTR, [None], [1], None],
            'TOHEXSTR': [self.TOHEXSTR, [None], [1], None],
            'BINSTRTONUM': [self.BINSTRTONUM, [None], [1], None],
            'HEXSTRTONUM': [self.HEXSTRTONUM, [None], [1], None],
            'ERASEVARIABLE': [self.ERASEVARIABLE, [None], [1], None],
            'STRLEN': [self.STRLEN, [1], [1, 2], None],
            'STRSTR': [self.STRSTR, [3], [3, 4], None],
            'SUBSTR': [self.SUBSTR, [None], [3], None],
            'REPLACE': [self.REPLACE, [None], [3], None],
            'ERASE': [self.ERASE, [None], [3], None],
            'INSERT': [self.INSERT, [None], [3], None],
            'CUTSPACE': [self.CUTSPACE, [None], [1], None],
            'MSTRLEN': [self.MSTRLEN, [None], [1], None],
            'MSTRSTR': [self.MSTRSTR, [None], [3], None],
            'MSUBSTR': [self.MSUBSTR, [None], [3], None],
            'MERASE': [self.MERASE, [None], [3], None],
            'MINSERT': [self.MINSERT, [None], [3], None],
            'NAMETOVALUE': [self.NAMETOVALUE, [0], [1, 2], None],
            'LETTONAME': [self.LETTONAME, [None], [2], None],
            'ARRAYSIZE': [self.ARRAYSIZE, [0, 1], [1, 2], None],
            'CALLBYNAME': [self.CALLBYNAME, [None], [1], None],
            ##'FUNCTIONEX': [self.FUNCTIONEX, [None], [None], None], # FIXME # Ver.3
            ##'SAORI': [self.SAORI, [None], [None], None], # FIXME # Ver.3
            'RAND': [self.RAND, [None], [0, 1], None],
            'ASC': [self.ASC, [None], [1], None],
            'IASC': [self.IASC, [None], [1], None],
            'FLOOR': [self.FLOOR, [None], [1], None],
            'CEIL': [self.CEIL, [None], [1], None],
            'ROUND': [self.ROUND, [None], [1], None],
            'ISINSIDE': [self.ISINSIDE, [None], [3], None],
            'ISINTEGER': [self.ISINTEGER, [None], [1], None],
            'ISREAL': [self.ISREAL, [None], [1], None],
            'ISFUNCTION': [self.ISFUNCTION, [None], [1], None],
            'SIN': [self.SIN, [None], [1], None],
            'COS': [self.COS, [None], [1], None],
            'TAN': [self.TAN, [None], [1], None],
            'LOG': [self.LOG, [None], [1], None],
            'LOG10': [self.LOG10, [None], [1], None],
            'POW': [self.POW, [None], [2], None],
            'SQRT': [self.SQRT, [None], [1], None],
            'SETSEPARATOR': [self.SETSEPARATOR, [0], [2], None],
            'REQ.COMMAND': [self.REQ_COMMAND, [None], [0], None],
            'REQ.HEADER': [self.REQ_HEADER, [None], [1], None],
            'REQ.KEY': [self.REQ_HEADER, [None], [1], None], # alias
            'REQ.VALUE': [self.REQ_VALUE, [None], [1], None],
            'REQ.PROTOCOL': [self.REQ_PROTOCOL, [None], [0], None],
            'LOADLIB': [self.LOADLIB, [None], [1], 16],
            'UNLOADLIB': [self.UNLOADLIB, [None], [1], None],
            'REQUESTLIB': [self.REQUESTLIB, [None], [2], None],
            'LIB.STATUSCODE': [self.LIB_STATUSCODE, [None], [0], None],
            'LIB.HEADER': [self.LIB_HEADER, [None], [1], None],
            'LIB.KEY': [self.LIB_HEADER, [None], [1], None], # alias
            'LIB.VALUE': [self.LIB_VALUE, [None], [1], None],
            'LIB.PROTOCOL': [self.LIB_PROTOCOL, [None], [0], None],
            'FOPEN': [self.FOPEN, [None], [2], 256],
            'FCLOSE': [self.FCLOSE, [None], [1], None],
            'FREAD': [self.FREAD, [None], [1], None],
            'FWRITE': [self.FWRITE, [None], [2], None],
            'FWRITE2': [self.FWRITE2, [None], [2], None],
            'FCOPY': [self.FCOPY, [None], [2], 259],
            'FMOVE': [self.FMOVE, [None], [2], 264],
            'FDELETE': [self.FDELETE, [None], [1], 269],
            'FRENAME': [self.FRENAME, [None], [2], 273],
            'FSIZE': [self.FSIZE, [None], [1], 278],
            'MKDIR': [self.MKDIR, [None], [1], 282],
            'RMDIR': [self.RMDIR, [None], [1], 286],
            'FENUM': [self.FENUM, [None], [1, 2], 290],
            'GETLASTERROR': [self.GETLASTERROR, [None], [None], None],
            'LOGGING': [self.LOGGING, [None], [4], None]
        }
    def exists(self, name):
        return self.functions.has_key(name)
    def call(self, namespace, name, argv):
        self.errno = 0
        if self.functions.has_key(name) and \
           self.check_num_args(name, argv):
            return self.functions[name][0](namespace, argv)
        else:
            return ''
    def not_to_evaluate(self, name, index):
        if index in self.functions[name][1]:
            return 1
        else:
            return 0
    def check_num_args(self, name, argv):
        list_num = self.functions[name][2]
        if list_num == [None]:
            return 1
        else:
            if len(argv) in list_num:
                return 1
            list_num.sort()
            if len(argv) < list_num[0]:
                errno = self.functions[name][3]
                if errno != None:
                    self.errno = errno
                print string.join((str(name), ': called with too few argument(s)'), '')
                return 0
            return 1
    def TONUMBER(self, namespace, argv):
        var = str(argv[0])
        target_namespace = self.select_namespace(namespace, var)
        token = target_namespace.get(var)
        try:
            if string.find(token, '.') >= 0:
                result = float(token)
            else:
                result = int(token)
        except:
            result = 0
        target_namespace.put(var, result)
        return None
    def TOSTRING(self, namespace, argv):
        name = str(argv[0])
        target_namespace = self.select_namespace(namespace, name)
        value = str(target_namespace.get(name))
        target_namespace.put(name, value)
    def TONUMBER2(self, namespace, argv):
        token = str(argv[0])
        try:
            if string.find(token, '.') >= 0:
                value = float(token)
            else:
                value = int(token)
        except:
            return 0
        else:
            return value
    def TOSTRING2(self, namespace, argv):
        return str(argv[0])
    def TOUPPER(self, namespace, argv):
        return string.upper(str(argv[0]))
    def TOLOWER(self, namespace, argv):
        return string.lower(str(argv[0]))
    def TOBINSTR(self, namespace, argv):
        try:
            i = int(argv[0])
        except:
            return ''
        if i < 0:
            i = abs(i)
            numsin = '-'
        else:
            numsin = ''
        line = ''
        while i:
            mod = i % 2
            i = i / 2
            line = string.join((str(mod), line), '')
        line = string.join((numsin, line), '')
        return line
    def TOHEXSTR(self, namespace, argv):
        try:
            return "%x" % int(argv[0])
        except:
            return ''
    def BINSTRTONUM(self, namespace, argv):
        try:
            return int(str(argv[0]), 2)
        except:
            return -1
    def HEXSTRTONUM(self, namespace, argv):
        try:
            return int(str(argv[0]), 16)
        except:
            return -1
    def ERASEVARIABLE(self, namespace, argv):
        var = str(argv[0])
        target_namespace = self.select_namespace(namespace, var)
        target_namespace.remove(var)
    def STRLEN(self, namespace, argv):
        line = str(argv[0])
        if len(argv) == 2:
            var = str(argv[1])
            target_namespace = self.select_namespace(namespace, var)
            target_namespace.put(var, len(line))
            return None
        else:
            return len(line)
    def STRSTR(self, namespace, argv):
        line = str(argv[0])
        to_find = str(argv[1])
        try:
            start = int(argv[2])
        except:
            return -1
        if start >= len(line):
            result = -1
        else:
            result = string.find(line, to_find, start)
            if result != -1:
                result = result
        if len(argv) == 4:
            var = str(argv[3])
            target_namespace = self.select_namespace(namespace, var)
            target_namespace.put(var, result)
            return None
        else:
            return result
    def SUBSTR(self, namespace, argv):
        line = str(argv[0])
        try:
            start = int(argv[1])
            bytes = int(argv[2])
        except:
            return ''
        return line[start:start+bytes]
    def REPLACE(self, namespace, argv):
        line = str(argv[0])
        old = str(argv[1])
        new = str(argv[2])
        return string.replace(line, old, new)
    def ERASE(self, namespace, argv):
        line = str(argv[0])
        try:
            start = int(argv[1])
            bytes = int(argv[2])
        except:
            return ''
        return string.join((line[:start], line[start+bytes:]), '')
    def INSERT(self, namespace, argv):
        line = str(argv[0])
        try:
            start = int(argv[1])
        except:
            return ''
        to_insert = str(argv[2])
        if start < 0:
            start = 0
        return string.join((line[:start], to_insert, line[start:]), '')
    def MSTRLEN(self, namespace, argv):
        line = str(argv[0])
        i = 0
        count = 0
        while i < len(line):
            if ord(line[i]) < 0x80:
                n = 1
            else:
                n = 2
            i = i + n
            count = count + 1
        return count
    def MSTRSTR(self, namespace, argv):
        line = str(argv[0])
        to_find = str(argv[1])
        try:
            int(argv[2])
        except:
            return -1
        start = len(line)
        i = 0
        count = 0
        while i < len(line):
            if ord(line[i]) < 0x80:
                n = 1
            else:
                n = 2
            i = i + n
            count = count + 1
            if count == int(argv[2]):
                start = i
                break
        result = string.find(line, to_find, start)
        if result != -1:
            result = self.MSTRLEN(namespace, [line[:result]])
        return result
    def MSUBSTR(self, namespace, argv):
        line = str(argv[0])
        try:
            int(argv[1])
            int(argv[2])
        except:
            return ''
        start = len(line)
        end = len(line)
        i = 0
        count = 0
        while i < len(line):
            if count == int(argv[1]):
                start = i
            if count == int(argv[1]) + int(argv[2]):
                end = i
                break
            if ord(line[i]) < 0x80:
                n = 1
            else:
                n = 2
            i = i + n
            count = count + 1
        return line[start:end]
    def MERASE(self, namespace, argv):
        line = str(argv[0])
        try:
            int(argv[1])
            int(argv[2])
        except:
            return ''
        start = len(line)
        end = len(line)
        i = 0
        count = 0
        while i < len(line):
            if count == int(argv[1]):
                start = i
            if count == int(argv[1]) + int(argv[2]):
                end = i
                break
            if ord(line[i]) < 0x80:
                n = 1
            else:
                n = 2
            i = i + n
            count = count + 1
        return string.join((line[:start], line[end:]), '')
    def MINSERT(self, namespace, argv):
        line = str(argv[0])
        try:
            int(argv[1])
        except:
            return ''
        to_insert = str(argv[2])
        if int(argv[1]) < 0:
            start = 0
        else:
            start = len(line)
            i = 0
            count = 0
            while i < len(line):
                if count == int(argv[1]):
                    start = i
                    break
                if ord(line[i]) < 0x80:
                    n = 1
                else:
                    n = 2
                i = i + n
                count = count + 1
        return string.join((line[:start], to_insert, line[start:]), '')
    def CUTSPACE(self, namespace, argv):
        return line_strip(str(argv[0]))
    def NAMETOVALUE(self, namespace, argv):
        if len(argv) == 2:
            var = str(argv[0])
            name = str(argv[1])
        else:
            name = str(argv[0])
        target_namespace = self.select_namespace(namespace, name)
        value = target_namespace.get(name)
        if len(argv) == 2:
            target_namespace = self.select_namespace(namespace, var)
            target_namespace.put(var, value)
            return None
        else:
            return value
    def LETTONAME(self, namespace, argv):
        var = str(argv[0])
        value = argv[1]
        if not var:
            return None
        target_namespace = self.select_namespace(namespace, var)
        target_namespace.put(var, value)
        return None
    def ARRAYSIZE(self, namespace, argv):
        if type(argv[0]) is StringType:
            line = argv[0]
            if not line or line == '':
                value = 0
            elif line[0] == '"' and line[-1] == '"':
                value = string.count(line, ',') + 1
            else:
                target_namespace = self.select_namespace(namespace, line)
                value = target_namespace.get_size(line)
        else:
            value = 0
        if len(argv) == 2:
            var = str(argv[1])
            target_namespace = self.select_namespace(namespace, var)
            target_namespace.put(var, value)
            return None
        else:
            return value
    def CALLBYNAME(self, namespace, argv): # dummy
        return None
    def FUNCTIONEX(self, namespace, argv): # FIXME # Ver.3
        return None
    def SAORI(self, namespace, argv): # FIXME # Ver.3
        return None
    def GETLASTERROR(self, namespace, argv):
        return self.errno
    def LOGGING(self, namespace, argv):
        if argv[0] == None:
            return None
        logfile = argv[0]
        line = string.join(('> function ', str(argv[1]), '  ', str(argv[2])), '')
        if argv[3] != None:
            line = string.join((line, ' = '), '')
            if type(argv[3]) is IntType or \
               type(argv[3]) is FloatType:
                line = string.join((line, str(argv[3])), '')
            else:
                line = string.join((line, '"', str(argv[3]), '"'), '')
        line = string.join((line, '\n'), '')
        logfile.write(line)
        logfile.write('\n')
        return None
    def RAND(self, namespace, argv):
        if len(argv) == 0:
            return int(random.randrange(0, 100, 1))
        else:
            try:
                int(argv[0])
            except:
                return -1
            return int(random.randrange(0, int(argv[0]), 1))
    def ASC(self, namespace, argv):
        try:
            int(argv[0])
        except:
            return ''
        index = int(argv[0])
        if index >= 0 and index < 0x80:
            return chr(index)
        else:
            return ' '
    def IASC(self, namespace, argv):
        if type(argv[0]) is not StringType:
            return -1
        try:
            char = argv[0][0]
        except:
            return -1
        return ord(char)
    def FLOOR(self, namespace, argv):
        try:
            return int(math.floor(float(argv[0])))
        except:
            return -1
    def CEIL(self, namespace, argv):
        try:
            return int(math.ceil(float(argv[0])))
        except:
            return -1
    def ROUND(self, namespace, argv):
        try:
            value = math.floor(float(argv[0])+0.5)
        except:
            return -1
        return int(value)
    def ISINSIDE(self, namespace, argv):
        if argv[0] >= argv[1] and argv[0] <= argv[2]:
            return 1
        else:
            return 0
    def ISINTEGER(self, namespace, argv):
        if type(argv[0]) is IntType:
            return 1
        else:
            return 0
    def ISREAL(self, namespace, argv):
        if type(argv[0]) is FloatType or \
           type(argv[0]) is IntType:
            return 1
        else:
            return 0
    def ISFUNCTION(self, namespace, argv):
        if type(argv[0]) is not StringType:
            return 0
        elif self.aya.dic.get_function(argv[0]) != None:
            return 1
        elif self.aya.get_system_functions().exists(argv[0]):
            return 2
        else:
            return 0
    def SIN(self, namespace, argv):
        try:
            result = math.sin(float(argv[0]))
        except:
            return -1
        return self.select_math_type(result)
    def COS(self, namespace, argv):
        try:
            result = math.cos(float(argv[0]))
        except:
            return -1
        return self.select_math_type(result)
    def TAN(self, namespace, argv):
        try:
            result = math.tan(float(argv[0]))
        except:
            return -1
        return self.select_math_type(result)
    def LOG(self, namespace, argv):
        try:
            float(argv[0])
        except:
            return -1
        if float(argv[0]) == 0:
            return 0
        result = math.log(float(argv[0]))
        return self.select_math_type(result)
    def LOG10(self, namespace, argv):
        try:
            float(argv[0])
        except:
            return -1
        if float(argv[0]) == 0:
            return 0
        result = math.log10(float(argv[0]))
        return self.select_math_type(result)
    def POW(self, namespace, argv):
        try:
            result = math.pow(float(argv[0]), float(argv[1]))
        except:
            return -1
        return self.select_math_type(result)
    def SQRT(self, namespace, argv):
        try:
            float(argv[0])
        except:
            return -1
        if float(argv[0]) < 0.:
            return -1
        else:
            result = math.sqrt(float(argv[0]))
            return self.select_math_type(result)
    def SETSEPARATOR(self, namespace, argv):
        name = str(argv[0])
        separator = str(argv[1])
        target_namespace = self.select_namespace(namespace, name)
        target_namespace.set_separator(name, separator)
        return None
    def REQ_COMMAND(self, namespace, argv):
        return self.aya.req_command
    def REQ_HEADER(self, namespace, argv):
        try:
            int(argv[0])
        except:
            return ''
        if len(self.aya.req_key) > int(argv[0]):
            return self.aya.req_key[int(argv[0])]
        else:
            return ''
    def REQ_VALUE(self, namespace, argv):
        if type(argv[0]) is IntType:
            name = self.REQ_HEADER(namespace, [argv[0]])
        else:
            name = str(argv[0])
        if self.aya.req_header.has_key(name):
            return self.aya.req_header[name]
        else:
            return ''
    def REQ_PROTOCOL(self, namespace, argv):
        return self.aya.req_protocol
    def LOADLIB(self, namespace, argv):
        dll = str(argv[0])
        result = 0
        if dll:
            if self.security.check_lib(dll):
                result = self.aya.saori_library.load(dll, self.aya.aya_dir)
                if result == 0:
                    self.errno = 17
            else:
                self.errno = 18
        return result
    def UNLOADLIB(self, namespace, argv):
        if str(argv[0]):
            self.aya.saori_library.unload(str(argv[0]))
        return None
    def REQUESTLIB(self, namespace, argv):
        response = self.aya.saori_library.request(str(argv[0]), kanjilib.euc2sjis(str(argv[1])))
        header = StringIO.StringIO(response)
        line = header.readline()
        self.saori_statuscode = ''
        self.saori_header = []
        self.saori_value = {}
        self.saori_protocol = ''
        if line:
            line = line_strip(line)
            pos_space = string.find(line, ' ')
            if pos_space >= 0:
                self.saori_protocol = line_strip(line[:pos_space])
                self.saori_statuscode = line_strip(line[pos_space:])
            while 1:
                line = header.readline()
                if not line:
                    break # EOF
                colon = string.find(line, ':')
                if colon >= 0:
                    key = line_strip(line[:colon])
                    value = line_strip(line[colon+1:])
                    if key:
                        self.saori_header.append(key)
                        self.saori_value[key] = value
        return None
    def LIB_STATUSCODE(self, namespace, argv):
        return self.saori_statuscode
    def LIB_HEADER(self, namespace, argv):
        try:
            int(argv[0])
        except:
            return ''
        result = ''
        header_list = self.saori_header
        if header_list and int(argv[0]) < len(header_list):
            result = header_list[int(argv[0])]
        return result
    def LIB_VALUE(self, namespace, argv):
        result = ''
        if type(argv[0]) is IntType:
            header_list = self.saori_header
            if header_list and int(argv[0]) < len(header_list):
                key = header_list[int(argv[0])]
        else:
            key = str(argv[0])
        if self.saori_value.has_key(key):
            result = self.saori_value[key]
        return result
    def LIB_PROTOCOL(self, namespace, argv):
        return self.aya.saori_protocol
    def FOPEN(self, namespace, argv):
        filename = string.lower(string.replace(str(argv[0]), '\\', '/'))
        accessmode = str(argv[1])
        result = 0
        path = os.path.join(self.aya.aya_dir, filename)
        if self.security.check_path(path, accessmode[0]):
            norm_path = os.path.normpath(path)
            if self.aya.filelist.has_key(norm_path):
                result = 2
            else:
                try:
                    self.aya.filelist[norm_path] = open(path, accessmode[0])
                except:
                    self.errno = 257
                else:
                    result = 1
        else:
            self.errno = 258
        return result
    def FCLOSE(self, namespace, argv):
        filename = string.lower(string.replace(str(argv[0]), '\\', '/'))
        path = os.path.join(self.aya.aya_dir, filename)
        norm_path = os.path.normpath(path)
        if self.aya.filelist.has_key(norm_path):
            self.aya.filelist[norm_path].close()
            del self.aya.filelist[norm_path]
        return None
    def FREAD(self, namespace, argv):
        filename = string.lower(string.replace(str(argv[0]), '\\', '/'))
        path = os.path.join(self.aya.aya_dir, filename)
        norm_path = os.path.normpath(path)
        result = -1
        if self.aya.filelist.has_key(norm_path):
            file = self.aya.filelist[norm_path]
            result = kanjilib.sjis2euc(file.readline(), 'ignore')
            if not result:
                result = -1
            elif result[-2:] == '\r\n':
                result = result[:-2]
            elif result[-1] == '\n':
                result = result[:-1]
        return result
    def FWRITE(self, namespace, argv):
        filename = string.lower(string.replace(str(argv[0]), '\\', '/'))
        path = os.path.join(self.aya.aya_dir, filename)
        norm_path = os.path.normpath(path)
        data = string.join((str(argv[1]), '\n'), '')
        if self.aya.filelist.has_key(norm_path):
            file = self.aya.filelist[norm_path]
            data = kanjilib.euc2sjis(data, 'ignore')
            file.write(data)
        return None
    def FWRITE2(self, namespace, argv):
        filename = string.lower(string.replace(str(argv[0]), '\\', '/'))
        path = os.path.join(self.aya.aya_dir, filename)
        norm_path = os.path.normpath(path)
        data = str(argv[1])
        if self.aya.filelist.has_key(norm_path):
            file = self.aya.filelist[norm_path]
            data = kanjilib.euc2sjis(data, 'ignore')
            file.write(data)
        return None
    def FCOPY(self, namespace, argv):
        src = string.lower(string.replace(str(argv[0]), '\\', '/'))
        head, tail = os.path.split(src)
        dst = string.join((string.lower(string.replace(str(argv[1]), '\\', '/')), '/', tail), '')
        src_path = os.path.join(self.aya.aya_dir, src)
        dst_path = os.path.join(self.aya.aya_dir, dst)
        result = 0
        if not os.path.isfile(src_path):
            self.errno = 260
        elif not os.path.isdir(dst_path):
            self.errno = 261
        elif self.security.check_path(src_path, 'r') and \
             self.security.check_path(dst_path):
            try:
                shutil.copyfile(src_path, dst_path)
            except:
                self.errno = 262
            else:
                result = 1
        else:
            self.errno = 263
        return result
    def FMOVE(self, namespace, argv):
        src = string.lower(string.replace(str(argv[0]), '\\', '/'))
        head, tail = os.path.split(src)
        dst = string.join((string.lower(string.replace(str(argv[1]), '\\', '/')), '/', tail), '')
        src_path = os.path.join(self.aya.aya_dir, src)
        dst_path = os.path.join(self.aya.aya_dir, dst)
        result = 0
        head, tail = os.path.split(dst_path)
        if not os.path.isfile(src_path):
            self.errno = 265
        elif not os.path.isdir(head):
            self.errno = 266
        elif self.security.check_path(src_path) and \
             self.security.check_path(dst_path):
            try:
                os.rename(src_path, dst_path)
            except:
                self.errno = 267
            else:
                result = 1
        else:
            self.errno = 268
        return result
    def FDELETE(self, namespace, argv):
        filename = string.lower(string.replace(str(argv[0]), '\\', '/'))
        path = os.path.join(self.aya.aya_dir, filename)
        result = 0
        if not os.path.isfile(path):
            self.errno = 270
        elif self.security.check_path(path):
            try:
                os.remove(path)
            except:
                self.errno = 271
            else:
                result = 1
        else:
            self.errno = 272
        return result
    def FRENAME(self, namespace, argv):
        src = string.lower(string.replace(str(argv[0]), '\\', '/'))
        dst = string.lower(string.replace(str(argv[1]), '\\', '/'))
        src_path = os.path.join(self.aya.aya_dir, src)
        dst_path = os.path.join(self.aya.aya_dir, dst)
        result = 0
        head, tail = os.path.split(dst_path)
        if not os.path.exists(src_path):
            self.errno = 274
        elif not os.path.isdir(head):
            self.errno = 275
        elif self.security.check_path(dst_path):
            try:
                os.rename(src_path, dst_path)
            except:
                self.errno = 276
            else:
                result = 1
        else:
            self.errno = 277
        return result
    def FSIZE(self, namespace, argv):
        filename = string.lower(string.replace(str(argv[0]), '\\', '/'))
        path = os.path.join(self.aya.aya_dir, filename)
        size = -1
        if not os.path.exists(path):
            self.errno = 279
        elif self.security.check_path(path, 'r'):
            try:
                size = os.stat(path)[ST_SIZE]
            except:
                self.errno = 280
        else:
            self.errno = 281
        return size
    def MKDIR(self, namespace, argv):
        dirname = string.lower(string.replace(str(argv[0]), '\\', '/'))
        path = os.path.join(self.aya.aya_dir, dirname)
        result = 0
        head, tail = os.path.split(path)
        if not os.path.isdir(head):
          self.errno = 283
        elif self.security.check_path(path):
            try:
                os.mkdir(path, 0755)
            except:
                self.errno = 284
            else:
                result = 1
        else:
            self.errno = 285
        return result
    def RMDIR(self, namespace, argv):
        dirname = string.lower(string.replace(str(argv[0]), '\\', '/'))
        path = os.path.join(self.aya.aya_dir, dirname)
        result = 0
        if not os.path.isdir(path):
            self.errno = 287
        elif self.security.check_path(path):
            try:
                os.rmdir(path)
            except:
                self.errno = 288
            else:
                result = 1
        else:
            self.errno = 289
        return result
    def FENUM(self, namespace, argv):
        if len(argv) >= 2:
            separator = str(argv[1])
        else:
            separator = ','
        dirname = string.lower(string.replace(str(argv[0]), '\\', '/'))
        path = os.path.join(self.aya.aya_dir, dirname)
        list = ''
        if self.security.check_path(path, 'r'):
            try:
                list = os.listdir(path)
            except:
                self.errno = 291
        else:
            self.errno = 292
        result = ''
        for index in range(len(list)):
            path = os.path.join(self.aya.aya_dir, dirname, list[index])
            mode = os.stat(path)[ST_MODE]
            if S_ISDIR(mode):
                result = string.join((result, '\\'), '')
            result = result + list[index]
            if index != len(list)-1:
                result = string.join((result, separator), '')
        return result
    def select_math_type(self, value):
        if math.floor(value) == value:
            return int(value)
        else:
            return value
    def select_namespace(self, namespace, name):
        if name[0] == '_':
            return namespace
        else:
            return self.aya.get_global_namespace()

class AyaNamespace:
    def __init__(self, aya, parent=None):
        self.aya = aya
        self.parent = parent
        self.table = {}
    def put(self, name, content, index=None):
        if self.parent != None and self.parent.exists(name):
            self.parent.put(name, content, index)
        elif index == None:
            if not self.exists(name):
                self.table[name] = AyaVariable(name)
            self.table[name].put(content)
        elif self.exists(name) and index >=0:
            self.table[name].put(content, index)
        else:
            pass # ERROR
    def get(self, name, index=None):
        if self.table.has_key(name):
            return self.table[name].get(index)
        elif self.parent != None and self.parent.exists(name):
            return self.parent.get(name, index)
        else:
            return None
    def set_separator(self, name, separator):
        if self.parent != None and self.parent.exists(name):
            self.parent.set_separator(name, separator)
        elif self.table.has_key(name):
            self.table[name].set_separator(separator)
        else:
            pass # ERROR
    def get_size(self, name):
        if self.table.has_key(name):
            return self.table[name].get_size()
        elif self.parent != None and self.parent.exists(name):
            return self.parent.get_size(name)
        else:
            return 0
    def remove(self, name): # only works with local table
        if self.table.has_key(name):
            del self.table[name]
    def exists(self, name):
        result = self.table.has_key(name) or \
                 (self.parent != None and self.parent.exists(name))
        return result

class AyaGlobalNamespace(AyaNamespace):
    __SYS_VARIABLES = ['year', 'month', 'day', 'weekday',
                       'hour', '12hour', 'ampm', 'minute', 'second',
                       'systemuptickcount', 'systemuptime',
                       'systemuphour', 'systemupminute', 'systemupsecond',
                       'memoryload',
                       'memorytotalphys', 'memoryavailphys',
                       'memorytotalvirtual', 'memoryavailvirtual',
                       'random', 'ascii' # Ver.3
                      ] # except for 'aitalkinterval', etc.
    __re_res = re.compile('res_reference\d+$')
    def reset_res_reference(self):
        for key in self.table.keys():
            if self.__re_res.match(key):
                del self.table[key]
    def get(self, name, index=None):
        t = time.localtime(time.time())
        past = time.time() - self.aya.get_boot_time()
        if name == 'year':
            return t[0]
        elif name == 'month':
            return t[1]
        elif name == 'day':
            return t[2]
        elif name == 'weekday':
            return (t[6]+1) % 7 
        elif name == 'hour':
            return t[3]
        elif name == '12hour':
            return t[3] % 12
        elif name == 'ampm':
            if t[3] >= 12:
                return 1 # pm
            else:
                return 0 # am
        elif name == 'minute':
            return t[4]
        elif name == 'second':
            return t[5]
        elif name == 'systemuptickcount':
            return int(past*1000.0)
        elif name == 'systemuptime':
            return int(past)
        elif name == 'systemuphour':
            return int(past/60.0/60.0)
        elif name == 'systemupminute':
            return int(past/60.0) % 60
        elif name == 'systemupsecond':
            return int(past) % 60
        elif name == 'memoryload' or \
             name == 'memorytotalphys' or \
             name == 'memoryavailphys' or \
             name == 'memorytotalvirtual' or \
             name == 'memoryavailvirtual':
            return 0 # FIXME
        else:
            return AyaNamespace.get(self, name, index)
    def exists(self, name):
        if name in self.__SYS_VARIABLES:
            return 1
        else:
            return AyaNamespace.exists(self, name)
    def load_database(self, aya):
        try:
            file = open(aya.dbpath)
        except:
            return 1
        line = file.readline()
        if line[:14] != '# Format: v1.0' and \
           line[:14] != '# Format: v1.1':
            file.close()
            return 1
        while 1:
            line = file.readline()
            if not line:
                break
            if len(line) == 0:
                break
            comma = string.find(line, ',')
            if comma >= 0:
                key = line[:comma]
            else:
                continue
            value = string.strip(line[comma+1:])
            comma = find_not_quoted(value, ',')
            if comma >= 0:
                separator = string.strip(value[comma+1:])
                separator = separator[1:-1]
                value = string.strip(value[:comma])
                value = value[1:-1]
                self.put(key, str(value))
                self.table[key].set_separator(str(separator))
            elif value[0] == '"': # Format: v1.0
                value = value[1:-1]
                self.put(key, str(value))
            elif value != 'None':
                if string.find(value, '.') >= 0:
                    self.put(key, float(value))
                else:
                    self.put(key, int(value))
            else:
                pass
        file.close()
        return 0
    def save_database(self):
        try:
            file = open(self.aya.dbpath, 'w')
        except IOError:
            print 'aya.py: cannot write database (ignored)'
            return
        file.write('# Format: v1.1\n')
        for key in self.table.keys():
            line = self.table[key].dump()
            if line != None:
                file.write(string.join((line, '\n'), ''))
        file.close()

class AyaStatement:
    __SPECIAL_CHARS = '=+-*/<>|&!:'
    def __init__(self, line):
        self.n_tokens = 0
        self.tokens = []
        self.position_of_next_token = 0
        self.tokenize(line)
    def tokenize(self, line):
        token_startpoint = 0
        block_nest_level = 0
        length = len(line)
        i = 0
        while i < length:
            c = line[i]
            if c == '(':
                block_nest_level = block_nest_level + 1
                i = i + 1
            elif c == ')':
                block_nest_level = block_nest_level - 1
                i = i + 1
            elif c == '"':
                if block_nest_level == 0:
                    self.append_unless_empty(string.strip(line[token_startpoint:i]))
                    token_startpoint = i
                position = i
                while position < length-1:
                    position = position + 1
                    if line[position] == '"':
                        break
                i = position
                if block_nest_level == 0:
                    self.tokens.append(line[token_startpoint:position+1])
                    token_startpoint = position + 1
                i = i + 1
            elif block_nest_level == 0 and (c == ' ' or c == '\t'):
                self.append_unless_empty(string.strip(line[token_startpoint:i]))
                token_startpoint = i + 1
                i = i + 1
            elif block_nest_level == 0 and line[i:i+2] == '':
                self.append_unless_empty(string.strip(line[token_startpoint:i]))
                token_startpoint = i + 2
                i = i + 2
            elif block_nest_level == 0 and \
                 string.strip(line[i:])[:4] == '_in_':
                self.append_unless_empty(string.strip(line[token_startpoint:i]))
                self.tokens.append('_in_')
                token_startpoint = i + 4
                i = i + 4
            elif block_nest_level == 0 and \
                 string.strip(line[i:])[:5] == '!_in_':
                self.append_unless_empty(string.strip(line[token_startpoint:i]))
                self.tokens.append('!_in_')
                token_startpoint = i + 5
                i = i + 5
            elif block_nest_level == 0 and string.find(self.__SPECIAL_CHARS, c) != -1:
                self.append_unless_empty(string.strip(line[token_startpoint:i]))
                ope_list = [':=', '+=', '-=', '*=', '/=', '%=',
                            '<=', '>=', '==', '!=', '&&', '||',
                            '+:=', '-:=', '*:=', '/:=', '%:=']
                if line[i:i+2] in ope_list:
                    self.tokens.append(line[i:i+2])
                    token_startpoint = i + 2
                    i = i + 2
                elif line[i:i+3] in ope_list:
                    self.tokens.append(line[i:i+3])
                    token_startpoint = i + 3
                    i = i + 3
                else:
                    self.tokens.append(line[i:i+1])
                    token_startpoint = i + 1
                    i = i + 1
            else:
                i = i + 1
        self.append_unless_empty(string.strip(line[token_startpoint:]))
        self.n_tokens = len(self.tokens)
    def append_unless_empty(self, token):
        if len(token) > 0:
            self.tokens.append(token)
    def has_more_tokens(self):
        return (self.position_of_next_token < self.n_tokens)
    def countTokens(self):
        return self.n_tokens
    def next_token(self):
        if not self.has_more_tokens():
            return None
        result = self.tokens[self.position_of_next_token]
        self.position_of_next_token = self.position_of_next_token + 1
        return result

class AyaVariable:
    __TYPE_STRING = 0
    __TYPE_INT = 1
    __TYPE_REAL = 2
    __TYPE_ARRAY = 3
    def __init__(self, name):
        self.name = name
        self.line = ''
        self.separator = ','
        self.type = None
        self.array = []
    def set_separator(self, separator):
        if self.type != self.__TYPE_STRING:
            return
        self.separator = separator
        self.reset()
    def reset(self):
        if self.type != self.__TYPE_STRING:
            return
        self.position = 0
        self.is_empty = 0
        self.array = []
        while not self.is_empty:
            separator_position = string.find(self.line, self.separator, self.position)
            if separator_position == -1:
                token = self.line[self.position:]
                self.is_empty = 1
            else:
                token = self.line[self.position:separator_position]
                self.position = separator_position + len(self.separator)
            self.array.append(token)
    def get_size(self):
        return len(self.array)
    def get(self, index=None):
        if index >= 0 and index < len(self.array):
            value = self.array[index]
            if self.type == self.__TYPE_STRING:
                return str(value)
            elif self.type == self.__TYPE_INT:
                return int(value)
            elif self.type == self.__TYPE_REAL:
                return float(value)
            elif self.type == self.__TYPE_ARRAY:
                return value
            else:
                return None # should not reach here
        elif index == None:
            if self.type == self.__TYPE_STRING:
                return str(self.line)
            elif self.type == self.__TYPE_INT:
                return int(self.line)
            elif self.type == self.__TYPE_REAL:
                return float(self.line)
            else:
                return ''
        else:
            return ''
    def put(self, value, index=None):
        if index == None:
            self.line = str(value)
            if type(value) == StringType:
                self.type = self.__TYPE_STRING
            elif type(value) == IntType:
                self.type = self.__TYPE_INT
            elif type(value) == FloatType:
                self.type = self.__TYPE_REAL
            elif type(value) == ListType:
                self.type = self.__TYPE_ARRAY
                self.array = value
            self.reset()
        elif index < 0:
            pass
        else:
            if self.type == self.__TYPE_STRING:
                self.line = ''
                for i in range(len(self.array)):
                    if i == index:
                        self.line = string.join((self.line, str(value)), '')
                    else:
                        self.line = string.join((self.line, self.array[i]), '')
                    if i != len(self.array)-1:
                        self.line = string.join((self.line, self.separator), '')
                if index >= len(self.array):
                    for i in range(len(self.array), index+1):
                        if i == index:
                            self.line = string.join((self.line, self.separator, str(value)), '')
                        else:
                            self.line = string.join((self.line, self.separator, ''), '')
                self.reset()
            elif self.type == self.__TYPE_ARRAY:
                if index >= 0 and index < len(self.array):
                    self.array[index] = value
            else:
                pass # ERROR
    def dump(self):
        line = None
        if self.type == self.__TYPE_STRING:
            line = '%s, "%s", "%s"' % (self.name, self.line, self.separator)
        elif self.type != self.__TYPE_ARRAY:
            line = '%s, %s' % (self.name, self.line)
        else:
            pass
        return line

class AyaArgument:
    def __init__(self, line):
        self.line = string.strip(line)
        self.length = len(self.line)
        self.current_position = 0
    def has_more_tokens(self):
        return (self.current_position != -1 and \
                self.current_position < self.length)
    def next_token(self):
        if not self.has_more_tokens():
            return None
        startpoint = self.current_position
        self.current_position = self.position_of_next_token()
        if self.current_position == -1:
            token = self.line[startpoint:]
        else:
            token = self.line[startpoint:self.current_position-1]
        return string.strip(token)
    def position_of_next_token(self):
        locked = 1
        position = self.current_position
        parenthesis_nest_level = 0
        while position < self.length:
            c = self.line[position]
            if c == '"':
                if not locked:
                    return position
                while position < self.length-1:
                    position = position + 1
                    if self.line[position] == '"':
                        break
            elif c == '(':
                parenthesis_nest_level = parenthesis_nest_level + 1
            elif c == ')':
                parenthesis_nest_level = parenthesis_nest_level - 1
            elif c == ',':
                if parenthesis_nest_level == 0:
                    locked = 0
            else:
                if not locked:
                    return position
            position = position + 1
        return -1


class AyaSaoriLibrary:
    def __init__(self, saori, dir):
        self.saori_list = {}
        self.saori = saori
    def load(self, name, dir):
        result = 0
        if self.saori and not self.saori_list.has_key(name):
            module = self.saori.request(name)
            if module:
                self.saori_list[name] = module
        if self.saori_list.has_key(name):
            result = self.saori_list[name].load(dir)
        return result
    def unload(self, name=None):
        if name:
            if self.saori_list.has_key(name):
                self.saori_list[name].unload()
                del self.saori_list[name]
        else:
            for key in self.saori_list.keys():
                self.saori_list[key].unload()
        return None
    def request(self, name, req):
        result = '' # FIXME
        if name and self.saori_list.has_key(name):
            result = self.saori_list[name].request(req)
        return result

def test(dir, function, argv):
    aya = Shiori('aya.dll')
    aya.load(dir)
    result = aya.dic.get_function(function).call(argv)
    print str(result)

if __name__ == "__main__":
    USAGE= "Usage(1): aya.py <dir> <function> [<arguments>]\n" \
           "Usage(2): aya.py encrypt <in_file> <out_file>"
    if len(sys.argv) < 3:
        print USAGE
    elif sys.argv[1] == 'encrypt':
        if len(sys.argv) != 4:
            print USAGE
        else:
            path = os.path.join(os.curdir, sys.argv[2])
            input = open(path)
            path = os.path.join(os.curdir, sys.argv[3])
            output = open(path, 'w')
            while 1:
                c = input.read(1)
                if c == '':
                    break
                output.write(encrypt_char(c))
            input.close()
            output.close()
    else:
        test(sys.argv[1], sys.argv[2], sys.argv[3:])
