# -*- coding: ascii -*-
#
#  Copyright (C) 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2005 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: seriko.py,v 1.15 2005/01/25 05:16:25 shy Exp $
#

import re
import sys
import random

def print_error(message):
    sys.stderr.write(message + "\n")

class Actor:

    def __init__(self, id, interval):
        self.id = id
        self.interval = interval
        self.patterns = []
        self.last_method = None
        self.exclusive = 0

    def set_exclusive(self):
        self.exclusive = 1

    def get_id(self):
        return self.id

    def get_interval(self):
        return self.interval

    def get_patterns(self):
        return self.patterns

    def add_pattern(self, surface, interval, method, args):
        self.patterns.append((surface, interval, method, args))

    def invoke(self):
        pass

    def update(self, window):
        pass

    def terminate(self):
        pass

    def show_pattern(self, window, surface, method, args):
        if self.last_method in ["overlay", "overlayfast"]:
            window.remove_overlay(self)
        if method == "move":
            window.move_surface(args[0], args[1])
        elif method in ["overlay", "overlayfast"]:
            window.add_overlay(self, surface, args[0], args[1])
        elif method == "base":
            window.surface_id = self.surface_id # This is kluge.
            window.set_surface(surface, restart=0)
        elif method == "start":
            window.invoke(args[0], update=1)
        elif method == "alternativestart":
            window.invoke(random.choice(args), update=1)
        else:
            raise RuntimeError, "should not reach here"
        self.last_method = method

class ActiveActor(Actor): # always

    def __init__(self, id, interval):
        Actor.__init__(self, id, interval)
        self.wait = 0
        self.pattern = 0

    def update(self, window):
        self.wait -= 1
        if self.wait > 0:
            return
        if self.pattern == 0:
            self.surface_id = window.get_surface()
        surface, interval, method, args = self.patterns[self.pattern]
        self.wait = interval
        self.pattern += 1
        if self.pattern == len(self.patterns):
            self.pattern = 0
        self.show_pattern(window, surface, method, args)

class RandomActor(Actor): # sometimes, rarely, randome

    def __init__(self, id, interval, wait_min, wait_max):
        Actor.__init__(self, id, interval)
        self.wait_min = wait_min
        self.wait_max = wait_max
        self.reset()

    def reset(self):
        self.wait = random.randint(self.wait_min, self.wait_max)
        self.pattern = 0

    def invoke(self):
        self.wait = 0
        self.pattern = 0

    def update(self, window):
        self.wait -= 1
        if self.wait > 0:
            return
        if self.pattern == 0:
            self.surface_id = window.get_surface()
        surface, interval, method, args = self.patterns[self.pattern]
        self.pattern += 1
        if self.pattern < len(self.patterns):
            self.wait = interval
        else:
            self.reset()
        self.show_pattern(window, surface, method, args)

    def terminate(self):
        self.reset()

class OneTimeActor(Actor): # runone

    def __init__(self, id, interval):
        Actor.__init__(self, id, interval)
        self.wait = -1
        self.pattern = 0

    def invoke(self):
        self.wait = 0
        self.pattern = 0

    def update(self, window):
        if self.wait < 0:
            return
        self.wait -= 1
        if self.wait > 0:
            return
        if self.pattern == 0:
            self.surface_id = window.get_surface()
        surface, interval, method, args = self.patterns[self.pattern]
        self.pattern += 1
        if self.pattern < len(self.patterns):
            self.wait = interval
        else:
            self.wait = -1 # done
        self.show_pattern(window, surface, method, args)

    def terminate(self):
        self.wait = -1

class PassiveActor(Actor): # never, yen-e, talk

    def __init__(self, id, interval):
        Actor.__init__(self, id, interval)
        self.wait = -1

    def invoke(self):
        self.wait = 0
        self.pattern = 0

    def update(self, window):
        if self.wait < 0:
            return
        self.wait -= 1
        if self.wait > 0:
            return
        if self.pattern == 0:
            self.surface_id = window.get_surface()
        surface, interval, method, args = self.patterns[self.pattern]
        self.pattern += 1
        if self.pattern < len(self.patterns):
            self.wait = interval
        else:
            self.wait = -1 # done
        self.show_pattern(window, surface, method, args)

    def terminate(self):
        self.wait = -1

class Mayuna(Actor):

    def __init__(self, id, interval):
        self.id = id
        self.interval = interval
        self.patterns = []

    def set_exclusive(self):
        pass

    def show_pattern(self, window, surface, method, args):
        pass

re_seriko_interval = re.compile("^([0-9]+)interval$")
re_seriko_interval_value = re.compile("^(sometimes|rarely|random,[0-9]+|always|runonce|yesn-e|talk,[0-9]+|never)$")
re_seriko_pattern = re.compile(r"^([0-9]+|-[12])\s*,\s*([+-]?[0-9]+)\s*,\s*(overlay|overlayfast|base|move|start|alternativestart|)\s*,?\s*([+-]?[0-9]+)?\s*,?\s*([+-]?[0-9]+)?\s*,?\s*(\[[0-9]+(\.[0-9]+)*\])?$")

re_seriko2_interval = re.compile("^animation([0-9]+)\.interval$")
re_seriko2_interval_value = re.compile("^(sometimes|rarely|random,[0-9]+|always|runonce|yesn-e|talk,[0-9]+|never)$")
re_seriko2_pattern = re.compile(r"^(overlay|overlayfast|base|move|start|alternativestart|)\s*,\s*([0-9]+|-[12])\s*,\s*([+-]?[0-9]+)\s*,?\s*([+-]?[0-9]+)?\s*,?\s*([+-]?[0-9]+)?\s*,?\s*(\([0-9]+(\.[0-9]+)*\))?$")

def get_actors(config):
    version = None
    buffer = []
    for key, value in config.items():
        if version == 1:
            match = re_seriko_interval.match(key)
        elif version == 2:
            match = re_seriko2_interval.match(key)
        else:
            match1 = re_seriko_interval.match(key)
            match2 = re_seriko2_interval.match(key)
            if match1:
                version = 1
                match = match1
            elif match2:
                version = 2
                match = match2
            else:
                continue
        if not match:
            continue
        if version == 1 and not re_seriko_interval_value.match(value):
            continue
        if version == 2 and not re_seriko2_interval_value.match(value):
            continue
        buffer.append((int(match.group(1)), value))
    actors = []
    for id, interval in buffer:
        if interval == "always":
            actor = ActiveActor(id, interval)
        elif interval == "sometimes":
            actor = RandomActor(id, interval, 0, 1000) # 0 to 10 seconds
        elif interval == "rarely":
            actor = RandomActor(id, interval, 2000, 6000)
        elif interval[:6] == "random":
            actor = RandomActor(id, interval, 0, 100 * int(interval[7]))
        elif interval == "runonce":
            actor = OneTimeActor(id, interval)
        elif interval == "yen-e":
            actor = PassiveActor(id, interval)
        elif interval[:4] == "talk":
            actor = PassiveActor(id, interval)
        elif interval == "never":
            actor = PassiveActor(id, interval)
        if version == 1:
            key = str(id) + 'option'
        else:
            key = 'animation' + str(id) + '.option'
        if config.has_key(key) and config[key] == 'exclusive':
            actor.set_exclusive()
        try:
            for n in range(128): # up to 128 patterns (0 - 127)
                if version == 1:
                    key = str(id) + "pattern" + str(n)
                else:
                    key = 'animation' + str(id) + ".pattern" + str(n)
                if not config.has_key(key):
                    key = str(id) + "patturn" + str(n) # only for version 1
                    if not config.has_key(key):
                        break
                pattern = config[key]
                if version == 1:
                    match = re_seriko_pattern.match(pattern)
                else:
                    match = re_seriko2_pattern.match(pattern)
                if not match:
                    raise ValueError, "unsupported pattern: %s" % pattern
                if version == 1:
                    surface = str(int(match.group(1)))
                    interval = abs(int(match.group(2)))
                    method = match.group(3)
                else:
                    method = match.group(1)
                    surface = str(int(match.group(2)))
                    interval = abs(int(match.group(3))) / 10 ## FIXME
                if method == "":
                    method = "base"
                if method == "start":
                    group = match.group(4)
                    if group is None:
                        raise ValueError, "syntax error: %s" % pattern
                    args = [int(group)]
                elif method == "alternativestart":
                    list = match.group(6)
                    if list is None:
                        raise ValueError, "syntax error: %s" % pattern
                    args = [int(s) for s in list[1:-1].split(".")]
                else:
                    if surface in ["-1", "-2"]:
                        x = 0
                        y = 0
                    else:
                        x = int(match.group(4) or 0)
                        y = int(match.group(5) or 0)
                    args = [x, y]
                actor.add_pattern(surface, interval, method, args)
        except ValueError, error:
            print_error("seriko.py: " + str(error))
            continue
        if not actor.get_patterns():
            print_error("seriko.py: animation group #%d has no pattern (ignored)" % id)
            continue
        actors.append(actor)
    actors.sort(lambda a1, a2: cmp(a1.get_id(), a2.get_id()))
    return actors

re_mayuna_interval = re.compile("^([0-9]+)interval$")
re_mayuna_interval_value = re.compile("^(bind)$")
re_mayuna_pattern = re.compile(r"^([0-9]+|-[12])\s*,\s*([0-9]+)\s*,\s*(bind|add|reduce|insert)\s*,?\s*([+-]?[0-9]+)?\s*,?\s*([+-]?[0-9]+)?\s*,?\s*(\[[0-9]+(\.[0-9]+)*\])?$")

re_mayuna2_interval = re.compile("^animation([0-9]+)\.interval$")
re_mayuna2_interval_value = re.compile("^(bind)$")
re_mayuna2_pattern = re.compile(r"^(bind|add|reduce|insert)\s*,\s*([0-9]+|-[12])\s*,\s*([0-9]+)\s*,?\s*([+-]?[0-9]+)?\s*,?\s*([+-]?[0-9]+)?\s*,?\s*(\([0-9]+(\.[0-9]+)*\))?$")

def get_mayuna(config):
    version = None
    buffer = []
    for key, value in config.items():
        if version == 1:
            match = re_mayuna_interval.match(key)
        elif version == 2:
            match = re_mayuna2_interval.match(key)
        else:
            match1 = re_mayuna_interval.match(key)
            match2 = re_mayuna2_interval.match(key)
            if match1:
                version = 1
                match = match1
            elif match2:
                version = 2
                match = match2
            else:
                continue
        if not match:
            continue
        if version == 1 and not re_mayuna_interval_value.match(value):
            continue
        if version == 2 and not re_mayuna2_interval_value.match(value):
            continue
        buffer.append((int(match.group(1)), value))
    mayuna = []
    for id, interval in buffer:
        ##assert interval == "bind"
        actor = Mayuna(id, interval)
        try:
            for n in range(128): # up to 128 patterns (0 - 127)
                if version == 1:
                    key = str(id) + "pattern" + str(n)
                else:
                    key = 'animation' + str(id) + ".pattern" + str(n)
                if not config.has_key(key):
                    key = str(id) + "patturn" + str(n) # only for version 1
                    if not config.has_key(key):
                        break
                pattern = config[key]
                if version == 1:
                    match = re_mayuna_pattern.match(pattern)
                else:
                    match = re_mayuna2_pattern.match(pattern)
                if not match:
                    raise ValueError, "unsupported pattern: %s" % pattern
                if version == 1:
                    surface = str(int(match.group(1)))
                    interval = abs(int(match.group(2)))
                    method = match.group(3)
                else:
                    method = match.group(1)
                    surface = str(int(match.group(2)))
                    interval = abs(int(match.group(3))) / 10 ## FIXME
                if method not in ["bind", "add", "reduce", "insert"]:
                    continue
                else:
                    if surface in ["-1", "-2"]:
                        x = 0
                        y = 0
                    else:
                        x = int(match.group(4) or 0)
                        y = int(match.group(5) or 0)
                    args = [x, y]
                actor.add_pattern(surface, interval, method, args)
        except ValueError, error:
            print_error("seriko.py: " + str(error))
            continue
        if not actor.get_patterns():
            print_error("seriko.py: animation group #%d has no pattern (ignored)" % id)
            continue
        mayuna.append(actor)
    mayuna.sort(lambda a1, a2: cmp(a1.get_id(), a2.get_id()))
    return mayuna

# find ~/.ninix -name 'surface*a.txt' | xargs python seriko.py
def test():
    import sys
    import ninix.config
    if len(sys.argv) == 1:
        print "Usage:", sys.argv[0], "[surface??a.txt ...]"
    for file in sys.argv[1:]:
        print "Reading", file, "..."
        for actor in get_actors(ninix.config.open(file)):
            print "#%d" % actor.get_id(),
            print actor.__class__.__name__,
            print "(%s)" % actor.get_interval()
            print "number of patterns =", len(actor.get_patterns())
            for pattern in actor.get_patterns():
                print "surface=%s, interval=%d, method=%s, args=%s" % pattern
        for actor in get_mayuna(ninix.config.open(file)):
            print "#%d" % actor.get_id(),
            print actor.__class__.__name__,
            print "(%s)" % actor.get_interval()
            print "number of patterns =", len(actor.get_patterns())
            for pattern in actor.get_patterns():
                print "surface=%s, interval=%d, method=%s, args=%s" % pattern

if __name__ == "__main__":
    test()
