// SPDX-License-Identifier: GPL-2.0
/*
 * R8A7740 processor support
 *
 * Copyright (C) 2011  Renesas Solutions Corp.
 * Copyright (C) 2011  Kuninori Morimoto <kuninori.morimoto.gx@renesas.com>
 */
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/pinctrl/pinconf-generic.h>

#include "sh_pfc.h"

#define CPU_ALL_PORT(fn, pfx, sfx)					\
	PORT_10(0,  fn, pfx, sfx),	PORT_90(0,   fn, pfx, sfx),	\
	PORT_10(100, fn, pfx##10, sfx),	PORT_90(100, fn, pfx##1, sfx),	\
	PORT_10(200, fn, pfx##20, sfx),					\
	PORT_1(210, fn, pfx##210, sfx),	PORT_1(211, fn, pfx##211, sfx)

#define IRQC_PIN_MUX(irq, pin)						\
static const unsigned int intc_irq##irq##_pins[] = {			\
	pin,								\
};									\
static const unsigned int intc_irq##irq##_mux[] = {			\
	IRQ##irq##_MARK,						\
}

#define IRQC_PINS_MUX(irq, idx, pin)					\
static const unsigned int intc_irq##irq##_##idx##_pins[] = {		\
	pin,								\
};									\
static const unsigned int intc_irq##irq##_##idx##_mux[] = {		\
	IRQ##irq##_PORT##pin##_MARK,					\
}

enum {
	PINMUX_RESERVED = 0,

	/* PORT0_DATA -> PORT211_DATA */
	PINMUX_DATA_BEGIN,
	PORT_ALL(DATA),
	PINMUX_DATA_END,

	/* PORT0_IN -> PORT211_IN */
	PINMUX_INPUT_BEGIN,
	PORT_ALL(IN),
	PINMUX_INPUT_END,

	/* PORT0_OUT -> PORT211_OUT */
	PINMUX_OUTPUT_BEGIN,
	PORT_ALL(OUT),
	PINMUX_OUTPUT_END,

	PINMUX_FUNCTION_BEGIN,
	PORT_ALL(FN_IN),	/* PORT0_FN_IN -> PORT211_FN_IN */
	PORT_ALL(FN_OUT),	/* PORT0_FN_OUT -> PORT211_FN_OUT */
	PORT_ALL(FN0),		/* PORT0_FN0 -> PORT211_FN0 */
	PORT_ALL(FN1),		/* PORT0_FN1 -> PORT211_FN1 */
	PORT_ALL(FN2),		/* PORT0_FN2 -> PORT211_FN2 */
	PORT_ALL(FN3),		/* PORT0_FN3 -> PORT211_FN3 */
	PORT_ALL(FN4),		/* PORT0_FN4 -> PORT211_FN4 */
	PORT_ALL(FN5),		/* PORT0_FN5 -> PORT211_FN5 */
	PORT_ALL(FN6),		/* PORT0_FN6 -> PORT211_FN6 */
	PORT_ALL(FN7),		/* PORT0_FN7 -> PORT211_FN7 */

	MSEL1CR_31_0,	MSEL1CR_31_1,
	MSEL1CR_30_0,	MSEL1CR_30_1,
	MSEL1CR_29_0,	MSEL1CR_29_1,
	MSEL1CR_28_0,	MSEL1CR_28_1,
	MSEL1CR_27_0,	MSEL1CR_27_1,
	MSEL1CR_26_0,	MSEL1CR_26_1,
	MSEL1CR_16_0,	MSEL1CR_16_1,
	MSEL1CR_15_0,	MSEL1CR_15_1,
	MSEL1CR_14_0,	MSEL1CR_14_1,
	MSEL1CR_13_0,	MSEL1CR_13_1,
	MSEL1CR_12_0,	MSEL1CR_12_1,
	MSEL1CR_9_0,	MSEL1CR_9_1,
	MSEL1CR_7_0,	MSEL1CR_7_1,
	MSEL1CR_6_0,	MSEL1CR_6_1,
	MSEL1CR_5_0,	MSEL1CR_5_1,
	MSEL1CR_4_0,	MSEL1CR_4_1,
	MSEL1CR_3_0,	MSEL1CR_3_1,
	MSEL1CR_2_0,	MSEL1CR_2_1,
	MSEL1CR_0_0,	MSEL1CR_0_1,

	MSEL3CR_15_0,	MSEL3CR_15_1, /* Trace / Debug ? */
	MSEL3CR_6_0,	MSEL3CR_6_1,

	MSEL4CR_19_0,	MSEL4CR_19_1,
	MSEL4CR_18_0,	MSEL4CR_18_1,
	MSEL4CR_15_0,	MSEL4CR_15_1,
	MSEL4CR_10_0,	MSEL4CR_10_1,
	MSEL4CR_6_0,	MSEL4CR_6_1,
	MSEL4CR_4_0,	MSEL4CR_4_1,
	MSEL4CR_1_0,	MSEL4CR_1_1,

	MSEL5CR_31_0,	MSEL5CR_31_1, /* irq/fiq output */
	MSEL5CR_30_0,	MSEL5CR_30_1,
	MSEL5CR_29_0,	MSEL5CR_29_1,
	MSEL5CR_27_0,	MSEL5CR_27_1,
	MSEL5CR_25_0,	MSEL5CR_25_1,
	MSEL5CR_23_0,	MSEL5CR_23_1,
	MSEL5CR_21_0,	MSEL5CR_21_1,
	MSEL5CR_19_0,	MSEL5CR_19_1,
	MSEL5CR_17_0,	MSEL5CR_17_1,
	MSEL5CR_15_0,	MSEL5CR_15_1,
	MSEL5CR_14_0,	MSEL5CR_14_1,
	MSEL5CR_13_0,	MSEL5CR_13_1,
	MSEL5CR_12_0,	MSEL5CR_12_1,
	MSEL5CR_11_0,	MSEL5CR_11_1,
	MSEL5CR_10_0,	MSEL5CR_10_1,
	MSEL5CR_8_0,	MSEL5CR_8_1,
	MSEL5CR_7_0,	MSEL5CR_7_1,
	MSEL5CR_6_0,	MSEL5CR_6_1,
	MSEL5CR_5_0,	MSEL5CR_5_1,
	MSEL5CR_4_0,	MSEL5CR_4_1,
	MSEL5CR_3_0,	MSEL5CR_3_1,
	MSEL5CR_2_0,	MSEL5CR_2_1,
	MSEL5CR_0_0,	MSEL5CR_0_1,
	PINMUX_FUNCTION_END,

	PINMUX_MARK_BEGIN,

	/* IRQ */
	IRQ0_PORT2_MARK,	IRQ0_PORT13_MARK,
	IRQ1_MARK,
	IRQ2_PORT11_MARK,	IRQ2_PORT12_MARK,
	IRQ3_PORT10_MARK,	IRQ3_PORT14_MARK,
	IRQ4_PORT15_MARK,	IRQ4_PORT172_MARK,
	IRQ5_PORT0_MARK,	IRQ5_PORT1_MARK,
	IRQ6_PORT121_MARK,	IRQ6_PORT173_MARK,
	IRQ7_PORT120_MARK,	IRQ7_PORT209_MARK,
	IRQ8_MARK,
	IRQ9_PORT118_MARK,	IRQ9_PORT210_MARK,
	IRQ10_MARK,
	IRQ11_MARK,
	IRQ12_PORT42_MARK,	IRQ12_PORT97_MARK,
	IRQ13_PORT64_MARK,	IRQ13_PORT98_MARK,
	IRQ14_PORT63_MARK,	IRQ14_PORT99_MARK,
	IRQ15_PORT62_MARK,	IRQ15_PORT100_MARK,
	IRQ16_PORT68_MARK,	IRQ16_PORT211_MARK,
	IRQ17_MARK,
	IRQ18_MARK,
	IRQ19_MARK,
	IRQ20_MARK,
	IRQ21_MARK,
	IRQ22_MARK,
	IRQ23_MARK,
	IRQ24_MARK,
	IRQ25_MARK,
	IRQ26_PORT58_MARK,	IRQ26_PORT81_MARK,
	IRQ27_PORT57_MARK,	IRQ27_PORT168_MARK,
	IRQ28_PORT56_MARK,	IRQ28_PORT169_MARK,
	IRQ29_PORT50_MARK,	IRQ29_PORT170_MARK,
	IRQ30_PORT49_MARK,	IRQ30_PORT171_MARK,
	IRQ31_PORT41_MARK,	IRQ31_PORT167_MARK,

	/* Function */

	/* DBGT */
	DBGMDT2_MARK,	DBGMDT1_MARK,	DBGMDT0_MARK,
	DBGMD10_MARK,	DBGMD11_MARK,	DBGMD20_MARK,
	DBGMD21_MARK,

	/* FSI-A */
	FSIAISLD_PORT0_MARK,	/* FSIAISLD Port 0/5 */
	FSIAISLD_PORT5_MARK,
	FSIASPDIF_PORT9_MARK,	/* FSIASPDIF Port 9/18 */
	FSIASPDIF_PORT18_MARK,
	FSIAOSLD1_MARK,	FSIAOSLD2_MARK,	FSIAOLR_MARK,
	FSIAOBT_MARK,	FSIAOSLD_MARK,	FSIAOMC_MARK,
	FSIACK_MARK,	FSIAILR_MARK,	FSIAIBT_MARK,

	/* FSI-B */
	FSIBCK_MARK,

	/* FMSI */
	FMSISLD_PORT1_MARK, /* FMSISLD Port 1/6 */
	FMSISLD_PORT6_MARK,
	FMSIILR_MARK,	FMSIIBT_MARK,	FMSIOLR_MARK,	FMSIOBT_MARK,
	FMSICK_MARK,	FMSOILR_MARK,	FMSOIBT_MARK,	FMSOOLR_MARK,
	FMSOOBT_MARK,	FMSOSLD_MARK,	FMSOCK_MARK,

	/* SCIFA0 */
	SCIFA0_SCK_MARK,	SCIFA0_CTS_MARK,	SCIFA0_RTS_MARK,
	SCIFA0_RXD_MARK,	SCIFA0_TXD_MARK,

	/* SCIFA1 */
	SCIFA1_CTS_MARK,	SCIFA1_SCK_MARK,	SCIFA1_RXD_MARK,
	SCIFA1_TXD_MARK,	SCIFA1_RTS_MARK,

	/* SCIFA2 */
	SCIFA2_SCK_PORT22_MARK, /* SCIFA2_SCK Port 22/199 */
	SCIFA2_SCK_PORT199_MARK,
	SCIFA2_RXD_MARK,	SCIFA2_TXD_MARK,
	SCIFA2_CTS_MARK,	SCIFA2_RTS_MARK,

	/* SCIFA3 */
	SCIFA3_RTS_PORT105_MARK, /* MSEL5CR_8_0 */
	SCIFA3_SCK_PORT116_MARK,
	SCIFA3_CTS_PORT117_MARK,
	SCIFA3_RXD_PORT174_MARK,
	SCIFA3_TXD_PORT175_MARK,

	SCIFA3_RTS_PORT161_MARK, /* MSEL5CR_8_1 */
	SCIFA3_SCK_PORT158_MARK,
	SCIFA3_CTS_PORT162_MARK,
	SCIFA3_RXD_PORT159_MARK,
	SCIFA3_TXD_PORT160_MARK,

	/* SCIFA4 */
	SCIFA4_RXD_PORT12_MARK, /* MSEL5CR[12:11] = 00 */
	SCIFA4_TXD_PORT13_MARK,

	SCIFA4_RXD_PORT204_MARK, /* MSEL5CR[12:11] = 01 */
	SCIFA4_TXD_PORT203_MARK,

	SCIFA4_RXD_PORT94_MARK, /* MSEL5CR[12:11] = 10 */
	SCIFA4_TXD_PORT93_MARK,

	SCIFA4_SCK_PORT21_MARK, /* SCIFA4_SCK Port 21/205 */
	SCIFA4_SCK_PORT205_MARK,

	/* SCIFA5 */
	SCIFA5_TXD_PORT20_MARK, /* MSEL5CR[15:14] = 00 */
	SCIFA5_RXD_PORT10_MARK,

	SCIFA5_RXD_PORT207_MARK, /* MSEL5CR[15:14] = 01 */
	SCIFA5_TXD_PORT208_MARK,

	SCIFA5_TXD_PORT91_MARK, /* MSEL5CR[15:14] = 10 */
	SCIFA5_RXD_PORT92_MARK,

	SCIFA5_SCK_PORT23_MARK, /* SCIFA5_SCK Port 23/206 */
	SCIFA5_SCK_PORT206_MARK,

	/* SCIFA6 */
	SCIFA6_SCK_MARK,	SCIFA6_RXD_MARK,	SCIFA6_TXD_MARK,

	/* SCIFA7 */
	SCIFA7_TXD_MARK,	SCIFA7_RXD_MARK,

	/* SCIFB */
	SCIFB_SCK_PORT190_MARK, /* MSEL5CR_17_0 */
	SCIFB_RXD_PORT191_MARK,
	SCIFB_TXD_PORT192_MARK,
	SCIFB_RTS_PORT186_MARK,
	SCIFB_CTS_PORT187_MARK,

	SCIFB_SCK_PORT2_MARK, /* MSEL5CR_17_1 */
	SCIFB_RXD_PORT3_MARK,
	SCIFB_TXD_PORT4_MARK,
	SCIFB_RTS_PORT172_MARK,
	SCIFB_CTS_PORT173_MARK,

	/* LCD0 */
	LCD0_D0_MARK,	LCD0_D1_MARK,	LCD0_D2_MARK,	LCD0_D3_MARK,
	LCD0_D4_MARK,	LCD0_D5_MARK,	LCD0_D6_MARK,	LCD0_D7_MARK,
	LCD0_D8_MARK,	LCD0_D9_MARK,	LCD0_D10_MARK,	LCD0_D11_MARK,
	LCD0_D12_MARK,	LCD0_D13_MARK,	LCD0_D14_MARK,	LCD0_D15_MARK,
	LCD0_D16_MARK,	LCD0_D17_MARK,
	LCD0_DON_MARK,	LCD0_VCPWC_MARK,	LCD0_VEPWC_MARK,
	LCD0_DCK_MARK,	LCD0_VSYN_MARK,	/* for RGB */
	LCD0_HSYN_MARK,	LCD0_DISP_MARK,	/* for RGB */
	LCD0_WR_MARK,	LCD0_RD_MARK,	/* for SYS */
	LCD0_CS_MARK,	LCD0_RS_MARK,	/* for SYS */

	LCD0_D21_PORT158_MARK,	LCD0_D23_PORT159_MARK, /* MSEL5CR_6_1 */
	LCD0_D22_PORT160_MARK,	LCD0_D20_PORT161_MARK,
	LCD0_D19_PORT162_MARK,	LCD0_D18_PORT163_MARK,
	LCD0_LCLK_PORT165_MARK,

	LCD0_D18_PORT40_MARK,	LCD0_D22_PORT0_MARK, /* MSEL5CR_6_0 */
	LCD0_D23_PORT1_MARK,	LCD0_D21_PORT2_MARK,
	LCD0_D20_PORT3_MARK,	LCD0_D19_PORT4_MARK,
	LCD0_LCLK_PORT102_MARK,

	/* LCD1 */
	LCD1_D0_MARK,	LCD1_D1_MARK,	LCD1_D2_MARK,	LCD1_D3_MARK,
	LCD1_D4_MARK,	LCD1_D5_MARK,	LCD1_D6_MARK,	LCD1_D7_MARK,
	LCD1_D8_MARK,	LCD1_D9_MARK,	LCD1_D10_MARK,	LCD1_D11_MARK,
	LCD1_D12_MARK,	LCD1_D13_MARK,	LCD1_D14_MARK,	LCD1_D15_MARK,
	LCD1_D16_MARK,	LCD1_D17_MARK,	LCD1_D18_MARK,	LCD1_D19_MARK,
	LCD1_D20_MARK,	LCD1_D21_MARK,	LCD1_D22_MARK,	LCD1_D23_MARK,
	LCD1_DON_MARK,	LCD1_VCPWC_MARK,
	LCD1_LCLK_MARK,	LCD1_VEPWC_MARK,

	LCD1_DCK_MARK,	LCD1_VSYN_MARK,	/* for RGB */
	LCD1_HSYN_MARK,	LCD1_DISP_MARK,	/* for RGB */
	LCD1_RS_MARK,	LCD1_CS_MARK,	/* for SYS */
	LCD1_RD_MARK,	LCD1_WR_MARK,	/* for SYS */

	/* RSPI */
	RSPI_SSL0_A_MARK,	RSPI_SSL1_A_MARK,	RSPI_SSL2_A_MARK,
	RSPI_SSL3_A_MARK,	RSPI_CK_A_MARK,		RSPI_MOSI_A_MARK,
	RSPI_MISO_A_MARK,

	/* VIO CKO */
	VIO_CKO1_MARK, /* needs fixup */
	VIO_CKO2_MARK,
	VIO_CKO_1_MARK,
	VIO_CKO_MARK,

	/* VIO0 */
	VIO0_D0_MARK,	VIO0_D1_MARK,	VIO0_D2_MARK,	VIO0_D3_MARK,
	VIO0_D4_MARK,	VIO0_D5_MARK,	VIO0_D6_MARK,	VIO0_D7_MARK,
	VIO0_D8_MARK,	VIO0_D9_MARK,	VIO0_D10_MARK,	VIO0_D11_MARK,
	VIO0_D12_MARK,	VIO0_VD_MARK,	VIO0_HD_MARK,	VIO0_CLK_MARK,
	VIO0_FIELD_MARK,

	VIO0_D13_PORT26_MARK, /* MSEL5CR_27_0 */
	VIO0_D14_PORT25_MARK,
	VIO0_D15_PORT24_MARK,

	VIO0_D13_PORT22_MARK, /* MSEL5CR_27_1 */
	VIO0_D14_PORT95_MARK,
	VIO0_D15_PORT96_MARK,

	/* VIO1 */
	VIO1_D0_MARK,	VIO1_D1_MARK,	VIO1_D2_MARK,	VIO1_D3_MARK,
	VIO1_D4_MARK,	VIO1_D5_MARK,	VIO1_D6_MARK,	VIO1_D7_MARK,
	VIO1_VD_MARK,	VIO1_HD_MARK,	VIO1_CLK_MARK,	VIO1_FIELD_MARK,

	/* TPU0 */
	TPU0TO0_MARK,	TPU0TO1_MARK,	TPU0TO3_MARK,
	TPU0TO2_PORT66_MARK, /* TPU0TO2 Port 66/202 */
	TPU0TO2_PORT202_MARK,

	/* SSP1 0 */
	STP0_IPD0_MARK,	STP0_IPD1_MARK,	STP0_IPD2_MARK,	STP0_IPD3_MARK,
	STP0_IPD4_MARK,	STP0_IPD5_MARK,	STP0_IPD6_MARK,	STP0_IPD7_MARK,
	STP0_IPEN_MARK,	STP0_IPCLK_MARK,	STP0_IPSYNC_MARK,

	/* SSP1 1 */
	STP1_IPD1_MARK,	STP1_IPD2_MARK,	STP1_IPD3_MARK,	STP1_IPD4_MARK,
	STP1_IPD5_MARK,	STP1_IPD6_MARK,	STP1_IPD7_MARK,	STP1_IPCLK_MARK,
	STP1_IPSYNC_MARK,

	STP1_IPD0_PORT186_MARK, /* MSEL5CR_23_0 */
	STP1_IPEN_PORT187_MARK,

	STP1_IPD0_PORT194_MARK, /* MSEL5CR_23_1 */
	STP1_IPEN_PORT193_MARK,

	/* SIM */
	SIM_RST_MARK,	SIM_CLK_MARK,
	SIM_D_PORT22_MARK, /* SIM_D  Port 22/199 */
	SIM_D_PORT199_MARK,

	/* SDHI0 */
	SDHI0_D0_MARK,	SDHI0_D1_MARK,	SDHI0_D2_MARK,	SDHI0_D3_MARK,
	SDHI0_CD_MARK,	SDHI0_WP_MARK,	SDHI0_CMD_MARK,	SDHI0_CLK_MARK,

	/* SDHI1 */
	SDHI1_D0_MARK,	SDHI1_D1_MARK,	SDHI1_D2_MARK,	SDHI1_D3_MARK,
	SDHI1_CD_MARK,	SDHI1_WP_MARK,	SDHI1_CMD_MARK,	SDHI1_CLK_MARK,

	/* SDHI2 */
	SDHI2_D0_MARK,	SDHI2_D1_MARK,	SDHI2_D2_MARK,	SDHI2_D3_MARK,
	SDHI2_CLK_MARK,	SDHI2_CMD_MARK,

	SDHI2_CD_PORT24_MARK, /* MSEL5CR_19_0 */
	SDHI2_WP_PORT25_MARK,

	SDHI2_WP_PORT177_MARK, /* MSEL5CR_19_1 */
	SDHI2_CD_PORT202_MARK,

	/* MSIOF2 */
	MSIOF2_TXD_MARK,	MSIOF2_RXD_MARK,	MSIOF2_TSCK_MARK,
	MSIOF2_SS2_MARK,	MSIOF2_TSYNC_MARK,	MSIOF2_SS1_MARK,
	MSIOF2_MCK1_MARK,	MSIOF2_MCK0_MARK,	MSIOF2_RSYNC_MARK,
	MSIOF2_RSCK_MARK,

	/* KEYSC */
	KEYIN4_MARK,	KEYIN5_MARK,	KEYIN6_MARK,	KEYIN7_MARK,
	KEYOUT0_MARK,	KEYOUT1_MARK,	KEYOUT2_MARK,	KEYOUT3_MARK,
	KEYOUT4_MARK,	KEYOUT5_MARK,	KEYOUT6_MARK,	KEYOUT7_MARK,

	KEYIN0_PORT43_MARK, /* MSEL4CR_18_0 */
	KEYIN1_PORT44_MARK,
	KEYIN2_PORT45_MARK,
	KEYIN3_PORT46_MARK,

	KEYIN0_PORT58_MARK, /* MSEL4CR_18_1 */
	KEYIN1_PORT57_MARK,
	KEYIN2_PORT56_MARK,
	KEYIN3_PORT55_MARK,

	/* VOU */
	DV_D0_MARK,	DV_D1_MARK,	DV_D2_MARK,	DV_D3_MARK,
	DV_D4_MARK,	DV_D5_MARK,	DV_D6_MARK,	DV_D7_MARK,
	DV_D8_MARK,	DV_D9_MARK,	DV_D10_MARK,	DV_D11_MARK,
	DV_D12_MARK,	DV_D13_MARK,	DV_D14_MARK,	DV_D15_MARK,
	DV_CLK_MARK,	DV_VSYNC_MARK,	DV_HSYNC_MARK,

	/* MEMC */
	MEMC_AD0_MARK,	MEMC_AD1_MARK,	MEMC_AD2_MARK,	MEMC_AD3_MARK,
	MEMC_AD4_MARK,	MEMC_AD5_MARK,	MEMC_AD6_MARK,	MEMC_AD7_MARK,
	MEMC_AD8_MARK,	MEMC_AD9_MARK,	MEMC_AD10_MARK,	MEMC_AD11_MARK,
	MEMC_AD12_MARK,	MEMC_AD13_MARK,	MEMC_AD14_MARK,	MEMC_AD15_MARK,
	MEMC_CS0_MARK,	MEMC_INT_MARK,	MEMC_NWE_MARK,	MEMC_NOE_MARK,

	MEMC_CS1_MARK, /* MSEL4CR_6_0 */
	MEMC_ADV_MARK,
	MEMC_WAIT_MARK,
	MEMC_BUSCLK_MARK,

	MEMC_A1_MARK, /* MSEL4CR_6_1 */
	MEMC_DREQ0_MARK,
	MEMC_DREQ1_MARK,
	MEMC_A0_MARK,

	/* MMC */
	MMC0_D0_PORT68_MARK,	MMC0_D1_PORT69_MARK,	MMC0_D2_PORT70_MARK,
	MMC0_D3_PORT71_MARK,	MMC0_D4_PORT72_MARK,	MMC0_D5_PORT73_MARK,
	MMC0_D6_PORT74_MARK,	MMC0_D7_PORT75_MARK,	MMC0_CLK_PORT66_MARK,
	MMC0_CMD_PORT67_MARK,	/* MSEL4CR_15_0 */

	MMC1_D0_PORT149_MARK,	MMC1_D1_PORT148_MARK,	MMC1_D2_PORT147_MARK,
	MMC1_D3_PORT146_MARK,	MMC1_D4_PORT145_MARK,	MMC1_D5_PORT144_MARK,
	MMC1_D6_PORT143_MARK,	MMC1_D7_PORT142_MARK,	MMC1_CLK_PORT103_MARK,
	MMC1_CMD_PORT104_MARK,	/* MSEL4CR_15_1 */

	/* MSIOF0 */
	MSIOF0_SS1_MARK,	MSIOF0_SS2_MARK,	MSIOF0_RXD_MARK,
	MSIOF0_TXD_MARK,	MSIOF0_MCK0_MARK,	MSIOF0_MCK1_MARK,
	MSIOF0_RSYNC_MARK,	MSIOF0_RSCK_MARK,	MSIOF0_TSCK_MARK,
	MSIOF0_TSYNC_MARK,

	/* MSIOF1 */
	MSIOF1_RSCK_MARK,	MSIOF1_RSYNC_MARK,
	MSIOF1_MCK0_MARK,	MSIOF1_MCK1_MARK,

	MSIOF1_SS2_PORT116_MARK,	MSIOF1_SS1_PORT117_MARK,
	MSIOF1_RXD_PORT118_MARK,	MSIOF1_TXD_PORT119_MARK,
	MSIOF1_TSYNC_PORT120_MARK,
	MSIOF1_TSCK_PORT121_MARK,	/* MSEL4CR_10_0 */

	MSIOF1_SS1_PORT67_MARK,		MSIOF1_TSCK_PORT72_MARK,
	MSIOF1_TSYNC_PORT73_MARK,	MSIOF1_TXD_PORT74_MARK,
	MSIOF1_RXD_PORT75_MARK,
	MSIOF1_SS2_PORT202_MARK,	/* MSEL4CR_10_1 */

	/* GPIO */
	GPO0_MARK,	GPI0_MARK,	GPO1_MARK,	GPI1_MARK,

	/* USB0 */
	USB0_OCI_MARK,	USB0_PPON_MARK,	VBUS_MARK,

	/* USB1 */
	USB1_OCI_MARK,	USB1_PPON_MARK,

	/* BBIF1 */
	BBIF1_RXD_MARK,		BBIF1_TXD_MARK,		BBIF1_TSYNC_MARK,
	BBIF1_TSCK_MARK,	BBIF1_RSCK_MARK,	BBIF1_RSYNC_MARK,
	BBIF1_FLOW_MARK,	BBIF1_RX_FLOW_N_MARK,

	/* BBIF2 */
	BBIF2_TXD2_PORT5_MARK, /* MSEL5CR_0_0 */
	BBIF2_RXD2_PORT60_MARK,
	BBIF2_TSYNC2_PORT6_MARK,
	BBIF2_TSCK2_PORT59_MARK,

	BBIF2_RXD2_PORT90_MARK, /* MSEL5CR_0_1 */
	BBIF2_TXD2_PORT183_MARK,
	BBIF2_TSCK2_PORT89_MARK,
	BBIF2_TSYNC2_PORT184_MARK,

	/* BSC / FLCTL / PCMCIA */
	CS0_MARK,	CS2_MARK,	CS4_MARK,
	CS5B_MARK,	CS6A_MARK,
	CS5A_PORT105_MARK, /* CS5A PORT 19/105 */
	CS5A_PORT19_MARK,
	IOIS16_MARK, /* ? */

	A0_MARK,	A1_MARK,	A2_MARK,	A3_MARK,
	A4_FOE_MARK,	/* share with FLCTL */
	A5_FCDE_MARK,	/* share with FLCTL */
	A6_MARK,	A7_MARK,	A8_MARK,	A9_MARK,
	A10_MARK,	A11_MARK,	A12_MARK,	A13_MARK,
	A14_MARK,	A15_MARK,	A16_MARK,	A17_MARK,
	A18_MARK,	A19_MARK,	A20_MARK,	A21_MARK,
	A22_MARK,	A23_MARK,	A24_MARK,	A25_MARK,
	A26_MARK,

	D0_NAF0_MARK,	D1_NAF1_MARK,	D2_NAF2_MARK,	/* share with FLCTL */
	D3_NAF3_MARK,	D4_NAF4_MARK,	D5_NAF5_MARK,	/* share with FLCTL */
	D6_NAF6_MARK,	D7_NAF7_MARK,	D8_NAF8_MARK,	/* share with FLCTL */
	D9_NAF9_MARK,	D10_NAF10_MARK,	D11_NAF11_MARK,	/* share with FLCTL */
	D12_NAF12_MARK,	D13_NAF13_MARK,	D14_NAF14_MARK,	/* share with FLCTL */
	D15_NAF15_MARK,					/* share with FLCTL */
	D16_MARK,	D17_MARK,	D18_MARK,	D19_MARK,
	D20_MARK,	D21_MARK,	D22_MARK,	D23_MARK,
	D24_MARK,	D25_MARK,	D26_MARK,	D27_MARK,
	D28_MARK,	D29_MARK,	D30_MARK,	D31_MARK,

	WE0_FWE_MARK,	/* share with FLCTL */
	WE1_MARK,
	WE2_ICIORD_MARK,	/* share with PCMCIA */
	WE3_ICIOWR_MARK,	/* share with PCMCIA */
	CKO_MARK,	BS_MARK,	RDWR_MARK,
	RD_FSC_MARK,	/* share with FLCTL */
	WAIT_PORT177_MARK, /* WAIT Port 90/177 */
	WAIT_PORT90_MARK,

	FCE0_MARK,	FCE1_MARK,	FRB_MARK, /* FLCTL */

	/* IRDA */
	IRDA_FIRSEL_MARK,	IRDA_IN_MARK,	IRDA_OUT_MARK,

	/* ATAPI */
	IDE_D0_MARK,	IDE_D1_MARK,	IDE_D2_MARK,	IDE_D3_MARK,
	IDE_D4_MARK,	IDE_D5_MARK,	IDE_D6_MARK,	IDE_D7_MARK,
	IDE_D8_MARK,	IDE_D9_MARK,	IDE_D10_MARK,	IDE_D11_MARK,
	IDE_D12_MARK,	IDE_D13_MARK,	IDE_D14_MARK,	IDE_D15_MARK,
	IDE_A0_MARK,	IDE_A1_MARK,	IDE_A2_MARK,	IDE_CS0_MARK,
	IDE_CS1_MARK,	IDE_IOWR_MARK,	IDE_IORD_MARK,	IDE_IORDY_MARK,
	IDE_INT_MARK,		IDE_RST_MARK,		IDE_DIRECTION_MARK,
	IDE_EXBUF_ENB_MARK,	IDE_IODACK_MARK,	IDE_IODREQ_MARK,

	/* RMII */
	RMII_CRS_DV_MARK,	RMII_RX_ER_MARK,	RMII_RXD0_MARK,
	RMII_RXD1_MARK,		RMII_TX_EN_MARK,	RMII_TXD0_MARK,
	RMII_MDC_MARK,		RMII_TXD1_MARK,		RMII_MDIO_MARK,
	RMII_REF50CK_MARK,	/* for RMII */
	RMII_REF125CK_MARK,	/* for GMII */

	/* GEther */
	ET_TX_CLK_MARK,	ET_TX_EN_MARK,	ET_ETXD0_MARK,	ET_ETXD1_MARK,
	ET_ETXD2_MARK,	ET_ETXD3_MARK,
	ET_ETXD4_MARK,	ET_ETXD5_MARK, /* for GEther */
	ET_ETXD6_MARK,	ET_ETXD7_MARK, /* for GEther */
	ET_COL_MARK,	ET_TX_ER_MARK,	ET_RX_CLK_MARK,	ET_RX_DV_MARK,
	ET_ERXD0_MARK,	ET_ERXD1_MARK,	ET_ERXD2_MARK,	ET_ERXD3_MARK,
	ET_ERXD4_MARK,	ET_ERXD5_MARK, /* for GEther */
	ET_ERXD6_MARK,	ET_ERXD7_MARK, /* for GEther */
	ET_RX_ER_MARK,	ET_CRS_MARK,		ET_MDC_MARK,	ET_MDIO_MARK,
	ET_LINK_MARK,	ET_PHY_INT_MARK,	ET_WOL_MARK,	ET_GTX_CLK_MARK,

	/* DMA0 */
	DREQ0_MARK,	DACK0_MARK,

	/* DMA1 */
	DREQ1_MARK,	DACK1_MARK,

	/* SYSC */
	RESETOUTS_MARK,		RESETP_PULLUP_MARK,	RESETP_PLAIN_MARK,

	/* IRREM */
	IROUT_MARK,

	/* SDENC */
	SDENC_CPG_MARK,		SDENC_DV_CLKI_MARK,

	/* HDMI */
	HDMI_HPD_MARK, HDMI_CEC_MARK,

	/* DEBUG */
	EDEBGREQ_PULLUP_MARK,	/* for JTAG */
	EDEBGREQ_PULLDOWN_MARK,

	TRACEAUD_FROM_VIO_MARK,	/* for TRACE/AUD */
	TRACEAUD_FROM_LCDC0_MARK,
	TRACEAUD_FROM_MEMC_MARK,

	PINMUX_MARK_END,
};

static const u16 pinmux_data[] = {
	PINMUX_DATA_ALL(),

	/* Port0 */
	PINMUX_DATA(DBGMDT2_MARK,		PORT0_FN1),
	PINMUX_DATA(FSIAISLD_PORT0_MARK,	PORT0_FN2,	MSEL5CR_3_0),
	PINMUX_DATA(FSIAOSLD1_MARK,		PORT0_FN3),
	PINMUX_DATA(LCD0_D22_PORT0_MARK,	PORT0_FN4,	MSEL5CR_6_0),
	PINMUX_DATA(SCIFA7_RXD_MARK,		PORT0_FN6),
	PINMUX_DATA(LCD1_D4_MARK,		PORT0_FN7),
	PINMUX_DATA(IRQ5_PORT0_MARK,		PORT0_FN0,	MSEL1CR_5_0),

	/* Port1 */
	PINMUX_DATA(DBGMDT1_MARK,		PORT1_FN1),
	PINMUX_DATA(FMSISLD_PORT1_MARK,		PORT1_FN2,	MSEL5CR_5_0),
	PINMUX_DATA(FSIAOSLD2_MARK,		PORT1_FN3),
	PINMUX_DATA(LCD0_D23_PORT1_MARK,	PORT1_FN4,	MSEL5CR_6_0),
	PINMUX_DATA(SCIFA7_TXD_MARK,		PORT1_FN6),
	PINMUX_DATA(LCD1_D3_MARK,		PORT1_FN7),
	PINMUX_DATA(IRQ5_PORT1_MARK,		PORT1_FN0,	MSEL1CR_5_1),

	/* Port2 */
	PINMUX_DATA(DBGMDT0_MARK,		PORT2_FN1),
	PINMUX_DATA(SCIFB_SCK_PORT2_MARK,	PORT2_FN2,	MSEL5CR_17_1),
	PINMUX_DATA(LCD0_D21_PORT2_MARK,	PORT2_FN4,	MSEL5CR_6_0),
	PINMUX_DATA(LCD1_D2_MARK,		PORT2_FN7),
	PINMUX_DATA(IRQ0_PORT2_MARK,		PORT2_FN0,	MSEL1CR_0_1),

	/* Port3 */
	PINMUX_DATA(DBGMD21_MARK,		PORT3_FN1),
	PINMUX_DATA(SCIFB_RXD_PORT3_MARK,	PORT3_FN2,	MSEL5CR_17_1),
	PINMUX_DATA(LCD0_D20_PORT3_MARK,	PORT3_FN4,	MSEL5CR_6_0),
	PINMUX_DATA(LCD1_D1_MARK,		PORT3_FN7),

	/* Port4 */
	PINMUX_DATA(DBGMD20_MARK,		PORT4_FN1),
	PINMUX_DATA(SCIFB_TXD_PORT4_MARK,	PORT4_FN2,	MSEL5CR_17_1),
	PINMUX_DATA(LCD0_D19_PORT4_MARK,	PORT4_FN4,	MSEL5CR_6_0),
	PINMUX_DATA(LCD1_D0_MARK,		PORT4_FN7),

	/* Port5 */
	PINMUX_DATA(DBGMD11_MARK,		PORT5_FN1),
	PINMUX_DATA(BBIF2_TXD2_PORT5_MARK,	PORT5_FN2,	MSEL5CR_0_0),
	PINMUX_DATA(FSIAISLD_PORT5_MARK,	PORT5_FN4,	MSEL5CR_3_1),
	PINMUX_DATA(RSPI_SSL0_A_MARK,		PORT5_FN6),
	PINMUX_DATA(LCD1_VCPWC_MARK,		PORT5_FN7),

	/* Port6 */
	PINMUX_DATA(DBGMD10_MARK,		PORT6_FN1),
	PINMUX_DATA(BBIF2_TSYNC2_PORT6_MARK,	PORT6_FN2,	MSEL5CR_0_0),
	PINMUX_DATA(FMSISLD_PORT6_MARK,		PORT6_FN4,	MSEL5CR_5_1),
	PINMUX_DATA(RSPI_SSL1_A_MARK,		PORT6_FN6),
	PINMUX_DATA(LCD1_VEPWC_MARK,		PORT6_FN7),

	/* Port7 */
	PINMUX_DATA(FSIAOLR_MARK,		PORT7_FN1),

	/* Port8 */
	PINMUX_DATA(FSIAOBT_MARK,		PORT8_FN1),

	/* Port9 */
	PINMUX_DATA(FSIAOSLD_MARK,		PORT9_FN1),
	PINMUX_DATA(FSIASPDIF_PORT9_MARK,	PORT9_FN2,	MSEL5CR_4_0),

	/* Port10 */
	PINMUX_DATA(FSIAOMC_MARK,		PORT10_FN1),
	PINMUX_DATA(SCIFA5_RXD_PORT10_MARK,	PORT10_FN3,	MSEL5CR_14_0,	MSEL5CR_15_0),
	PINMUX_DATA(IRQ3_PORT10_MARK,		PORT10_FN0,	MSEL1CR_3_0),

	/* Port11 */
	PINMUX_DATA(FSIACK_MARK,		PORT11_FN1),
	PINMUX_DATA(FSIBCK_MARK,		PORT11_FN2),
	PINMUX_DATA(IRQ2_PORT11_MARK,		PORT11_FN0,	MSEL1CR_2_0),

	/* Port12 */
	PINMUX_DATA(FSIAILR_MARK,		PORT12_FN1),
	PINMUX_DATA(SCIFA4_RXD_PORT12_MARK,	PORT12_FN2,	MSEL5CR_12_0,	MSEL5CR_11_0),
	PINMUX_DATA(LCD1_RS_MARK,		PORT12_FN6),
	PINMUX_DATA(LCD1_DISP_MARK,		PORT12_FN7),
	PINMUX_DATA(IRQ2_PORT12_MARK,		PORT12_FN0,	MSEL1CR_2_1),

	/* Port13 */
	PINMUX_DATA(FSIAIBT_MARK,		PORT13_FN1),
	PINMUX_DATA(SCIFA4_TXD_PORT13_MARK,	PORT13_FN2,	MSEL5CR_12_0,	MSEL5CR_11_0),
	PINMUX_DATA(LCD1_RD_MARK,		PORT13_FN7),
	PINMUX_DATA(IRQ0_PORT13_MARK,		PORT13_FN0,	MSEL1CR_0_0),

	/* Port14 */
	PINMUX_DATA(FMSOILR_MARK,		PORT14_FN1),
	PINMUX_DATA(FMSIILR_MARK,		PORT14_FN2),
	PINMUX_DATA(VIO_CKO1_MARK,		PORT14_FN3),
	PINMUX_DATA(LCD1_D23_MARK,		PORT14_FN7),
	PINMUX_DATA(IRQ3_PORT14_MARK,		PORT14_FN0,	MSEL1CR_3_1),

	/* Port15 */
	PINMUX_DATA(FMSOIBT_MARK,		PORT15_FN1),
	PINMUX_DATA(FMSIIBT_MARK,		PORT15_FN2),
	PINMUX_DATA(VIO_CKO2_MARK,		PORT15_FN3),
	PINMUX_DATA(LCD1_D22_MARK,		PORT15_FN7),
	PINMUX_DATA(IRQ4_PORT15_MARK,		PORT15_FN0,	MSEL1CR_4_0),

	/* Port16 */
	PINMUX_DATA(FMSOOLR_MARK,		PORT16_FN1),
	PINMUX_DATA(FMSIOLR_MARK,		PORT16_FN2),

	/* Port17 */
	PINMUX_DATA(FMSOOBT_MARK,		PORT17_FN1),
	PINMUX_DATA(FMSIOBT_MARK,		PORT17_FN2),

	/* Port18 */
	PINMUX_DATA(FMSOSLD_MARK,		PORT18_FN1),
	PINMUX_DATA(FSIASPDIF_PORT18_MARK,	PORT18_FN2,	MSEL5CR_4_1),

	/* Port19 */
	PINMUX_DATA(FMSICK_MARK,		PORT19_FN1),
	PINMUX_DATA(CS5A_PORT19_MARK,		PORT19_FN7,	MSEL5CR_2_1),
	PINMUX_DATA(IRQ10_MARK,			PORT19_FN0),

	/* Port20 */
	PINMUX_DATA(FMSOCK_MARK,		PORT20_FN1),
	PINMUX_DATA(SCIFA5_TXD_PORT20_MARK,	PORT20_FN3,	MSEL5CR_15_0,	MSEL5CR_14_0),
	PINMUX_DATA(IRQ1_MARK,			PORT20_FN0),

	/* Port21 */
	PINMUX_DATA(SCIFA1_CTS_MARK,		PORT21_FN1),
	PINMUX_DATA(SCIFA4_SCK_PORT21_MARK,	PORT21_FN2,	MSEL5CR_10_0),
	PINMUX_DATA(TPU0TO1_MARK,		PORT21_FN4),
	PINMUX_DATA(VIO1_FIELD_MARK,		PORT21_FN5),
	PINMUX_DATA(STP0_IPD5_MARK,		PORT21_FN6),
	PINMUX_DATA(LCD1_D10_MARK,		PORT21_FN7),

	/* Port22 */
	PINMUX_DATA(SCIFA2_SCK_PORT22_MARK,	PORT22_FN1,	MSEL5CR_7_0),
	PINMUX_DATA(SIM_D_PORT22_MARK,		PORT22_FN4,	MSEL5CR_21_0),
	PINMUX_DATA(VIO0_D13_PORT22_MARK,	PORT22_FN7,	MSEL5CR_27_1),

	/* Port23 */
	PINMUX_DATA(SCIFA1_RTS_MARK,		PORT23_FN1),
	PINMUX_DATA(SCIFA5_SCK_PORT23_MARK,	PORT23_FN3,	MSEL5CR_13_0),
	PINMUX_DATA(TPU0TO0_MARK,		PORT23_FN4),
	PINMUX_DATA(VIO_CKO_1_MARK,		PORT23_FN5),
	PINMUX_DATA(STP0_IPD2_MARK,		PORT23_FN6),
	PINMUX_DATA(LCD1_D7_MARK,		PORT23_FN7),

	/* Port24 */
	PINMUX_DATA(VIO0_D15_PORT24_MARK,	PORT24_FN1,	MSEL5CR_27_0),
	PINMUX_DATA(VIO1_D7_MARK,		PORT24_FN5),
	PINMUX_DATA(SCIFA6_SCK_MARK,		PORT24_FN6),
	PINMUX_DATA(SDHI2_CD_PORT24_MARK,	PORT24_FN7,	MSEL5CR_19_0),

	/* Port25 */
	PINMUX_DATA(VIO0_D14_PORT25_MARK,	PORT25_FN1,	MSEL5CR_27_0),
	PINMUX_DATA(VIO1_D6_MARK,		PORT25_FN5),
	PINMUX_DATA(SCIFA6_RXD_MARK,		PORT25_FN6),
	PINMUX_DATA(SDHI2_WP_PORT25_MARK,	PORT25_FN7,	MSEL5CR_19_0),

	/* Port26 */
	PINMUX_DATA(VIO0_D13_PORT26_MARK,	PORT26_FN1,	MSEL5CR_27_0),
	PINMUX_DATA(VIO1_D5_MARK,		PORT26_FN5),
	PINMUX_DATA(SCIFA6_TXD_MARK,		PORT26_FN6),

	/* Port27 - Port39 Function */
	PINMUX_DATA(VIO0_D7_MARK,		PORT27_FN1),
	PINMUX_DATA(VIO0_D6_MARK,		PORT28_FN1),
	PINMUX_DATA(VIO0_D5_MARK,		PORT29_FN1),
	PINMUX_DATA(VIO0_D4_MARK,		PORT30_FN1),
	PINMUX_DATA(VIO0_D3_MARK,		PORT31_FN1),
	PINMUX_DATA(VIO0_D2_MARK,		PORT32_FN1),
	PINMUX_DATA(VIO0_D1_MARK,		PORT33_FN1),
	PINMUX_DATA(VIO0_D0_MARK,		PORT34_FN1),
	PINMUX_DATA(VIO0_CLK_MARK,		PORT35_FN1),
	PINMUX_DATA(VIO_CKO_MARK,		PORT36_FN1),
	PINMUX_DATA(VIO0_HD_MARK,		PORT37_FN1),
	PINMUX_DATA(VIO0_FIELD_MARK,		PORT38_FN1),
	PINMUX_DATA(VIO0_VD_MARK,		PORT39_FN1),

	/* Port38 IRQ */
	PINMUX_DATA(IRQ25_MARK,			PORT38_FN0),

	/* Port40 */
	PINMUX_DATA(LCD0_D18_PORT40_MARK,	PORT40_FN4,	MSEL5CR_6_0),
	PINMUX_DATA(RSPI_CK_A_MARK,		PORT40_FN6),
	PINMUX_DATA(LCD1_LCLK_MARK,		PORT40_FN7),

	/* Port41 */
	PINMUX_DATA(LCD0_D17_MARK,		PORT41_FN1),
	PINMUX_DATA(MSIOF2_SS1_MARK,		PORT41_FN2),
	PINMUX_DATA(IRQ31_PORT41_MARK,		PORT41_FN0,	MSEL1CR_31_1),

	/* Port42 */
	PINMUX_DATA(LCD0_D16_MARK,		PORT42_FN1),
	PINMUX_DATA(MSIOF2_MCK1_MARK,		PORT42_FN2),
	PINMUX_DATA(IRQ12_PORT42_MARK,		PORT42_FN0,	MSEL1CR_12_1),

	/* Port43 */
	PINMUX_DATA(LCD0_D15_MARK,		PORT43_FN1),
	PINMUX_DATA(MSIOF2_MCK0_MARK,		PORT43_FN2),
	PINMUX_DATA(KEYIN0_PORT43_MARK,		PORT43_FN3,	MSEL4CR_18_0),
	PINMUX_DATA(DV_D15_MARK,		PORT43_FN6),

	/* Port44 */
	PINMUX_DATA(LCD0_D14_MARK,		PORT44_FN1),
	PINMUX_DATA(MSIOF2_RSYNC_MARK,		PORT44_FN2),
	PINMUX_DATA(KEYIN1_PORT44_MARK,		PORT44_FN3,	MSEL4CR_18_0),
	PINMUX_DATA(DV_D14_MARK,		PORT44_FN6),

	/* Port45 */
	PINMUX_DATA(LCD0_D13_MARK,		PORT45_FN1),
	PINMUX_DATA(MSIOF2_RSCK_MARK,		PORT45_FN2),
	PINMUX_DATA(KEYIN2_PORT45_MARK,		PORT45_FN3,	MSEL4CR_18_0),
	PINMUX_DATA(DV_D13_MARK,		PORT45_FN6),

	/* Port46 */
	PINMUX_DATA(LCD0_D12_MARK,		PORT46_FN1),
	PINMUX_DATA(KEYIN3_PORT46_MARK,		PORT46_FN3,	MSEL4CR_18_0),
	PINMUX_DATA(DV_D12_MARK,		PORT46_FN6),

	/* Port47 */
	PINMUX_DATA(LCD0_D11_MARK,		PORT47_FN1),
	PINMUX_DATA(KEYIN4_MARK,		PORT47_FN3),
	PINMUX_DATA(DV_D11_MARK,		PORT47_FN6),

	/* Port48 */
	PINMUX_DATA(LCD0_D10_MARK,		PORT48_FN1),
	PINMUX_DATA(KEYIN5_MARK,		PORT48_FN3),
	PINMUX_DATA(DV_D10_MARK,		PORT48_FN6),

	/* Port49 */
	PINMUX_DATA(LCD0_D9_MARK,		PORT49_FN1),
	PINMUX_DATA(KEYIN6_MARK,		PORT49_FN3),
	PINMUX_DATA(DV_D9_MARK,			PORT49_FN6),
	PINMUX_DATA(IRQ30_PORT49_MARK,		PORT49_FN0,	MSEL1CR_30_1),

	/* Port50 */
	PINMUX_DATA(LCD0_D8_MARK,		PORT50_FN1),
	PINMUX_DATA(KEYIN7_MARK,		PORT50_FN3),
	PINMUX_DATA(DV_D8_MARK,			PORT50_FN6),
	PINMUX_DATA(IRQ29_PORT50_MARK,		PORT50_FN0,	MSEL1CR_29_1),

	/* Port51 */
	PINMUX_DATA(LCD0_D7_MARK,		PORT51_FN1),
	PINMUX_DATA(KEYOUT0_MARK,		PORT51_FN3),
	PINMUX_DATA(DV_D7_MARK,			PORT51_FN6),

	/* Port52 */
	PINMUX_DATA(LCD0_D6_MARK,		PORT52_FN1),
	PINMUX_DATA(KEYOUT1_MARK,		PORT52_FN3),
	PINMUX_DATA(DV_D6_MARK,			PORT52_FN6),

	/* Port53 */
	PINMUX_DATA(LCD0_D5_MARK,		PORT53_FN1),
	PINMUX_DATA(KEYOUT2_MARK,		PORT53_FN3),
	PINMUX_DATA(DV_D5_MARK,			PORT53_FN6),

	/* Port54 */
	PINMUX_DATA(LCD0_D4_MARK,		PORT54_FN1),
	PINMUX_DATA(KEYOUT3_MARK,		PORT54_FN3),
	PINMUX_DATA(DV_D4_MARK,			PORT54_FN6),

	/* Port55 */
	PINMUX_DATA(LCD0_D3_MARK,		PORT55_FN1),
	PINMUX_DATA(KEYOUT4_MARK,		PORT55_FN3),
	PINMUX_DATA(KEYIN3_PORT55_MARK,		PORT55_FN4,	MSEL4CR_18_1),
	PINMUX_DATA(DV_D3_MARK,			PORT55_FN6),

	/* Port56 */
	PINMUX_DATA(LCD0_D2_MARK,		PORT56_FN1),
	PINMUX_DATA(KEYOUT5_MARK,		PORT56_FN3),
	PINMUX_DATA(KEYIN2_PORT56_MARK,		PORT56_FN4,	MSEL4CR_18_1),
	PINMUX_DATA(DV_D2_MARK,			PORT56_FN6),
	PINMUX_DATA(IRQ28_PORT56_MARK,		PORT56_FN0,	MSEL1CR_28_1),

	/* Port57 */
	PINMUX_DATA(LCD0_D1_MARK,		PORT57_FN1),
	PINMUX_DATA(KEYOUT6_MARK,		PORT57_FN3),
	PINMUX_DATA(KEYIN1_PORT57_MARK,		PORT57_FN4,	MSEL4CR_18_1),
	PINMUX_DATA(DV_D1_MARK,			PORT57_FN6),
	PINMUX_DATA(IRQ27_PORT57_MARK,		PORT57_FN0,	MSEL1CR_27_1),

	/* Port58 */
	PINMUX_DATA(LCD0_D0_MARK,		PORT58_FN1,	MSEL3CR_6_0),
	PINMUX_DATA(KEYOUT7_MARK,		PORT58_FN3),
	PINMUX_DATA(KEYIN0_PORT58_MARK,		PORT58_FN4,	MSEL4CR_18_1),
	PINMUX_DATA(DV_D0_MARK,			PORT58_FN6),
	PINMUX_DATA(IRQ26_PORT58_MARK,		PORT58_FN0,	MSEL1CR_26_1),

	/* Port59 */
	PINMUX_DATA(LCD0_VCPWC_MARK,		PORT59_FN1),
	PINMUX_DATA(BBIF2_TSCK2_PORT59_MARK,	PORT59_FN2,	MSEL5CR_0_0),
	PINMUX_DATA(RSPI_MOSI_A_MARK,		PORT59_FN6),

	/* Port60 */
	PINMUX_DATA(LCD0_VEPWC_MARK,		PORT60_FN1),
	PINMUX_DATA(BBIF2_RXD2_PORT60_MARK,	PORT60_FN2,	MSEL5CR_0_0),
	PINMUX_DATA(RSPI_MISO_A_MARK,		PORT60_FN6),

	/* Port61 */
	PINMUX_DATA(LCD0_DON_MARK,		PORT61_FN1),
	PINMUX_DATA(MSIOF2_TXD_MARK,		PORT61_FN2),

	/* Port62 */
	PINMUX_DATA(LCD0_DCK_MARK,		PORT62_FN1),
	PINMUX_DATA(LCD0_WR_MARK,		PORT62_FN4),
	PINMUX_DATA(DV_CLK_MARK,		PORT62_FN6),
	PINMUX_DATA(IRQ15_PORT62_MARK,		PORT62_FN0,	MSEL1CR_15_1),

	/* Port63 */
	PINMUX_DATA(LCD0_VSYN_MARK,		PORT63_FN1),
	PINMUX_DATA(DV_VSYNC_MARK,		PORT63_FN6),
	PINMUX_DATA(IRQ14_PORT63_MARK,		PORT63_FN0,	MSEL1CR_14_1),

	/* Port64 */
	PINMUX_DATA(LCD0_HSYN_MARK,		PORT64_FN1),
	PINMUX_DATA(LCD0_CS_MARK,		PORT64_FN4),
	PINMUX_DATA(DV_HSYNC_MARK,		PORT64_FN6),
	PINMUX_DATA(IRQ13_PORT64_MARK,		PORT64_FN0,	MSEL1CR_13_1),

	/* Port65 */
	PINMUX_DATA(LCD0_DISP_MARK,		PORT65_FN1),
	PINMUX_DATA(MSIOF2_TSCK_MARK,		PORT65_FN2),
	PINMUX_DATA(LCD0_RS_MARK,		PORT65_FN4),

	/* Port66 */
	PINMUX_DATA(MEMC_INT_MARK,		PORT66_FN1),
	PINMUX_DATA(TPU0TO2_PORT66_MARK,	PORT66_FN3,	MSEL5CR_25_0),
	PINMUX_DATA(MMC0_CLK_PORT66_MARK,	PORT66_FN4,	MSEL4CR_15_0),
	PINMUX_DATA(SDHI1_CLK_MARK,		PORT66_FN6),

	/* Port67 - Port73 Function1 */
	PINMUX_DATA(MEMC_CS0_MARK,		PORT67_FN1),
	PINMUX_DATA(MEMC_AD8_MARK,		PORT68_FN1),
	PINMUX_DATA(MEMC_AD9_MARK,		PORT69_FN1),
	PINMUX_DATA(MEMC_AD10_MARK,		PORT70_FN1),
	PINMUX_DATA(MEMC_AD11_MARK,		PORT71_FN1),
	PINMUX_DATA(MEMC_AD12_MARK,		PORT72_FN1),
	PINMUX_DATA(MEMC_AD13_MARK,		PORT73_FN1),

	/* Port67 - Port73 Function2 */
	PINMUX_DATA(MSIOF1_SS1_PORT67_MARK,	PORT67_FN2,	MSEL4CR_10_1),
	PINMUX_DATA(MSIOF1_RSCK_MARK,		PORT68_FN2),
	PINMUX_DATA(MSIOF1_RSYNC_MARK,		PORT69_FN2),
	PINMUX_DATA(MSIOF1_MCK0_MARK,		PORT70_FN2),
	PINMUX_DATA(MSIOF1_MCK1_MARK,		PORT71_FN2),
	PINMUX_DATA(MSIOF1_TSCK_PORT72_MARK,	PORT72_FN2,	MSEL4CR_10_1),
	PINMUX_DATA(MSIOF1_TSYNC_PORT73_MARK,	PORT73_FN2,	MSEL4CR_10_1),

	/* Port67 - Port73 Function4 */
	PINMUX_DATA(MMC0_CMD_PORT67_MARK,	PORT67_FN4,	MSEL4CR_15_0),
	PINMUX_DATA(MMC0_D0_PORT68_MARK,	PORT68_FN4,	MSEL4CR_15_0),
	PINMUX_DATA(MMC0_D1_PORT69_MARK,	PORT69_FN4,	MSEL4CR_15_0),
	PINMUX_DATA(MMC0_D2_PORT70_MARK,	PORT70_FN4,	MSEL4CR_15_0),
	PINMUX_DATA(MMC0_D3_PORT71_MARK,	PORT71_FN4,	MSEL4CR_15_0),
	PINMUX_DATA(MMC0_D4_PORT72_MARK,	PORT72_FN4,	MSEL4CR_15_0),
	PINMUX_DATA(MMC0_D5_PORT73_MARK,	PORT73_FN4,	MSEL4CR_15_0),

	/* Port67 - Port73 Function6 */
	PINMUX_DATA(SDHI1_CMD_MARK,		PORT67_FN6),
	PINMUX_DATA(SDHI1_D0_MARK,		PORT68_FN6),
	PINMUX_DATA(SDHI1_D1_MARK,		PORT69_FN6),
	PINMUX_DATA(SDHI1_D2_MARK,		PORT70_FN6),
	PINMUX_DATA(SDHI1_D3_MARK,		PORT71_FN6),
	PINMUX_DATA(SDHI1_CD_MARK,		PORT72_FN6),
	PINMUX_DATA(SDHI1_WP_MARK,		PORT73_FN6),

	/* Port67 - Port71 IRQ */
	PINMUX_DATA(IRQ20_MARK,			PORT67_FN0),
	PINMUX_DATA(IRQ16_PORT68_MARK,		PORT68_FN0,	MSEL1CR_16_0),
	PINMUX_DATA(IRQ17_MARK,			PORT69_FN0),
	PINMUX_DATA(IRQ18_MARK,			PORT70_FN0),
	PINMUX_DATA(IRQ19_MARK,			PORT71_FN0),

	/* Port74 */
	PINMUX_DATA(MEMC_AD14_MARK,		PORT74_FN1),
	PINMUX_DATA(MSIOF1_TXD_PORT74_MARK,	PORT74_FN2,	MSEL4CR_10_1),
	PINMUX_DATA(MMC0_D6_PORT74_MARK,	PORT74_FN4,	MSEL4CR_15_0),
	PINMUX_DATA(STP1_IPD7_MARK,		PORT74_FN6),
	PINMUX_DATA(LCD1_D21_MARK,		PORT74_FN7),

	/* Port75 */
	PINMUX_DATA(MEMC_AD15_MARK,		PORT75_FN1),
	PINMUX_DATA(MSIOF1_RXD_PORT75_MARK,	PORT75_FN2,	MSEL4CR_10_1),
	PINMUX_DATA(MMC0_D7_PORT75_MARK,	PORT75_FN4,	MSEL4CR_15_0),
	PINMUX_DATA(STP1_IPD6_MARK,		PORT75_FN6),
	PINMUX_DATA(LCD1_D20_MARK,		PORT75_FN7),

	/* Port76 - Port80 Function */
	PINMUX_DATA(SDHI0_CMD_MARK,		PORT76_FN1),
	PINMUX_DATA(SDHI0_D0_MARK,		PORT77_FN1),
	PINMUX_DATA(SDHI0_D1_MARK,		PORT78_FN1),
	PINMUX_DATA(SDHI0_D2_MARK,		PORT79_FN1),
	PINMUX_DATA(SDHI0_D3_MARK,		PORT80_FN1),

	/* Port81 */
	PINMUX_DATA(SDHI0_CD_MARK,		PORT81_FN1),
	PINMUX_DATA(IRQ26_PORT81_MARK,		PORT81_FN0,	MSEL1CR_26_0),

	/* Port82 - Port88 Function */
	PINMUX_DATA(SDHI0_CLK_MARK,		PORT82_FN1),
	PINMUX_DATA(SDHI0_WP_MARK,		PORT83_FN1),
	PINMUX_DATA(RESETOUTS_MARK,		PORT84_FN1),
	PINMUX_DATA(USB0_PPON_MARK,		PORT85_FN1),
	PINMUX_DATA(USB0_OCI_MARK,		PORT86_FN1),
	PINMUX_DATA(USB1_PPON_MARK,		PORT87_FN1),
	PINMUX_DATA(USB1_OCI_MARK,		PORT88_FN1),

	/* Port89 */
	PINMUX_DATA(DREQ0_MARK,			PORT89_FN1),
	PINMUX_DATA(BBIF2_TSCK2_PORT89_MARK,	PORT89_FN2,	MSEL5CR_0_1),
	PINMUX_DATA(RSPI_SSL3_A_MARK,		PORT89_FN6),

	/* Port90 */
	PINMUX_DATA(DACK0_MARK,			PORT90_FN1),
	PINMUX_DATA(BBIF2_RXD2_PORT90_MARK,	PORT90_FN2,	MSEL5CR_0_1),
	PINMUX_DATA(RSPI_SSL2_A_MARK,		PORT90_FN6),
	PINMUX_DATA(WAIT_PORT90_MARK,		PORT90_FN7,	MSEL5CR_2_1),

	/* Port91 */
	PINMUX_DATA(MEMC_AD0_MARK,		PORT91_FN1),
	PINMUX_DATA(BBIF1_RXD_MARK,		PORT91_FN2),
	PINMUX_DATA(SCIFA5_TXD_PORT91_MARK,	PORT91_FN3,	MSEL5CR_15_1,	MSEL5CR_14_0),
	PINMUX_DATA(LCD1_D5_MARK,		PORT91_FN7),

	/* Port92 */
	PINMUX_DATA(MEMC_AD1_MARK,		PORT92_FN1),
	PINMUX_DATA(BBIF1_TSYNC_MARK,		PORT92_FN2),
	PINMUX_DATA(SCIFA5_RXD_PORT92_MARK,	PORT92_FN3,	MSEL5CR_15_1,	MSEL5CR_14_0),
	PINMUX_DATA(STP0_IPD1_MARK,		PORT92_FN6),
	PINMUX_DATA(LCD1_D6_MARK,		PORT92_FN7),

	/* Port93 */
	PINMUX_DATA(MEMC_AD2_MARK,		PORT93_FN1),
	PINMUX_DATA(BBIF1_TSCK_MARK,		PORT93_FN2),
	PINMUX_DATA(SCIFA4_TXD_PORT93_MARK,	PORT93_FN3,	MSEL5CR_12_1,	MSEL5CR_11_0),
	PINMUX_DATA(STP0_IPD3_MARK,		PORT93_FN6),
	PINMUX_DATA(LCD1_D8_MARK,		PORT93_FN7),

	/* Port94 */
	PINMUX_DATA(MEMC_AD3_MARK,		PORT94_FN1),
	PINMUX_DATA(BBIF1_TXD_MARK,		PORT94_FN2),
	PINMUX_DATA(SCIFA4_RXD_PORT94_MARK,	PORT94_FN3,	MSEL5CR_12_1,	MSEL5CR_11_0),
	PINMUX_DATA(STP0_IPD4_MARK,		PORT94_FN6),
	PINMUX_DATA(LCD1_D9_MARK,		PORT94_FN7),

	/* Port95 */
	PINMUX_DATA(MEMC_CS1_MARK,		PORT95_FN1,	MSEL4CR_6_0),
	PINMUX_DATA(MEMC_A1_MARK,		PORT95_FN1,	MSEL4CR_6_1),

	PINMUX_DATA(SCIFA2_CTS_MARK,		PORT95_FN2),
	PINMUX_DATA(SIM_RST_MARK,		PORT95_FN4),
	PINMUX_DATA(VIO0_D14_PORT95_MARK,	PORT95_FN7,	MSEL5CR_27_1),
	PINMUX_DATA(IRQ22_MARK,			PORT95_FN0),

	/* Port96 */
	PINMUX_DATA(MEMC_ADV_MARK,		PORT96_FN1,	MSEL4CR_6_0),
	PINMUX_DATA(MEMC_DREQ0_MARK,		PORT96_FN1,	MSEL4CR_6_1),

	PINMUX_DATA(SCIFA2_RTS_MARK,		PORT96_FN2),
	PINMUX_DATA(SIM_CLK_MARK,		PORT96_FN4),
	PINMUX_DATA(VIO0_D15_PORT96_MARK,	PORT96_FN7,	MSEL5CR_27_1),
	PINMUX_DATA(IRQ23_MARK,			PORT96_FN0),

	/* Port97 */
	PINMUX_DATA(MEMC_AD4_MARK,		PORT97_FN1),
	PINMUX_DATA(BBIF1_RSCK_MARK,		PORT97_FN2),
	PINMUX_DATA(LCD1_CS_MARK,		PORT97_FN6),
	PINMUX_DATA(LCD1_HSYN_MARK,		PORT97_FN7),
	PINMUX_DATA(IRQ12_PORT97_MARK,		PORT97_FN0,	MSEL1CR_12_0),

	/* Port98 */
	PINMUX_DATA(MEMC_AD5_MARK,		PORT98_FN1),
	PINMUX_DATA(BBIF1_RSYNC_MARK,		PORT98_FN2),
	PINMUX_DATA(LCD1_VSYN_MARK,		PORT98_FN7),
	PINMUX_DATA(IRQ13_PORT98_MARK,		PORT98_FN0,	MSEL1CR_13_0),

	/* Port99 */
	PINMUX_DATA(MEMC_AD6_MARK,		PORT99_FN1),
	PINMUX_DATA(BBIF1_FLOW_MARK,		PORT99_FN2),
	PINMUX_DATA(LCD1_WR_MARK,		PORT99_FN6),
	PINMUX_DATA(LCD1_DCK_MARK,		PORT99_FN7),
	PINMUX_DATA(IRQ14_PORT99_MARK,		PORT99_FN0,	MSEL1CR_14_0),

	/* Port100 */
	PINMUX_DATA(MEMC_AD7_MARK,		PORT100_FN1),
	PINMUX_DATA(BBIF1_RX_FLOW_N_MARK,	PORT100_FN2),
	PINMUX_DATA(LCD1_DON_MARK,		PORT100_FN7),
	PINMUX_DATA(IRQ15_PORT100_MARK,		PORT100_FN0,	MSEL1CR_15_0),

	/* Port101 */
	PINMUX_DATA(FCE0_MARK,			PORT101_FN1),

	/* Port102 */
	PINMUX_DATA(FRB_MARK,			PORT102_FN1),
	PINMUX_DATA(LCD0_LCLK_PORT102_MARK,	PORT102_FN4,	MSEL5CR_6_0),

	/* Port103 */
	PINMUX_DATA(CS5B_MARK,			PORT103_FN1),
	PINMUX_DATA(FCE1_MARK,			PORT103_FN2),
	PINMUX_DATA(MMC1_CLK_PORT103_MARK,	PORT103_FN3,	MSEL4CR_15_1),

	/* Port104 */
	PINMUX_DATA(CS6A_MARK,			PORT104_FN1),
	PINMUX_DATA(MMC1_CMD_PORT104_MARK,	PORT104_FN3,	MSEL4CR_15_1),
	PINMUX_DATA(IRQ11_MARK,			PORT104_FN0),

	/* Port105 */
	PINMUX_DATA(CS5A_PORT105_MARK,		PORT105_FN1,	MSEL5CR_2_0),
	PINMUX_DATA(SCIFA3_RTS_PORT105_MARK,	PORT105_FN4,	MSEL5CR_8_0),

	/* Port106 */
	PINMUX_DATA(IOIS16_MARK,		PORT106_FN1),
	PINMUX_DATA(IDE_EXBUF_ENB_MARK,		PORT106_FN6),

	/* Port107 - Port115 Function */
	PINMUX_DATA(WE3_ICIOWR_MARK,		PORT107_FN1),
	PINMUX_DATA(WE2_ICIORD_MARK,		PORT108_FN1),
	PINMUX_DATA(CS0_MARK,			PORT109_FN1),
	PINMUX_DATA(CS2_MARK,			PORT110_FN1),
	PINMUX_DATA(CS4_MARK,			PORT111_FN1),
	PINMUX_DATA(WE1_MARK,			PORT112_FN1),
	PINMUX_DATA(WE0_FWE_MARK,		PORT113_FN1),
	PINMUX_DATA(RDWR_MARK,			PORT114_FN1),
	PINMUX_DATA(RD_FSC_MARK,		PORT115_FN1),

	/* Port116 */
	PINMUX_DATA(A25_MARK,			PORT116_FN1),
	PINMUX_DATA(MSIOF0_SS2_MARK,		PORT116_FN2),
	PINMUX_DATA(MSIOF1_SS2_PORT116_MARK,	PORT116_FN3,	MSEL4CR_10_0),
	PINMUX_DATA(SCIFA3_SCK_PORT116_MARK,	PORT116_FN4,	MSEL5CR_8_0),
	PINMUX_DATA(GPO1_MARK,			PORT116_FN5),

	/* Port117 */
	PINMUX_DATA(A24_MARK,			PORT117_FN1),
	PINMUX_DATA(MSIOF0_SS1_MARK,		PORT117_FN2),
	PINMUX_DATA(MSIOF1_SS1_PORT117_MARK,	PORT117_FN3,	MSEL4CR_10_0),
	PINMUX_DATA(SCIFA3_CTS_PORT117_MARK,	PORT117_FN4,	MSEL5CR_8_0),
	PINMUX_DATA(GPO0_MARK,			PORT117_FN5),

	/* Port118 */
	PINMUX_DATA(A23_MARK,			PORT118_FN1),
	PINMUX_DATA(MSIOF0_MCK1_MARK,		PORT118_FN2),
	PINMUX_DATA(MSIOF1_RXD_PORT118_MARK,	PORT118_FN3,	MSEL4CR_10_0),
	PINMUX_DATA(GPI1_MARK,			PORT118_FN5),
	PINMUX_DATA(IRQ9_PORT118_MARK,		PORT118_FN0,	MSEL1CR_9_0),

	/* Port119 */
	PINMUX_DATA(A22_MARK,			PORT119_FN1),
	PINMUX_DATA(MSIOF0_MCK0_MARK,		PORT119_FN2),
	PINMUX_DATA(MSIOF1_TXD_PORT119_MARK,	PORT119_FN3,	MSEL4CR_10_0),
	PINMUX_DATA(GPI0_MARK,			PORT119_FN5),
	PINMUX_DATA(IRQ8_MARK,			PORT119_FN0),

	/* Port120 */
	PINMUX_DATA(A21_MARK,			PORT120_FN1),
	PINMUX_DATA(MSIOF0_RSYNC_MARK,		PORT120_FN2),
	PINMUX_DATA(MSIOF1_TSYNC_PORT120_MARK,	PORT120_FN3,	MSEL4CR_10_0),
	PINMUX_DATA(IRQ7_PORT120_MARK,		PORT120_FN0,	MSEL1CR_7_1),

	/* Port121 */
	PINMUX_DATA(A20_MARK,			PORT121_FN1),
	PINMUX_DATA(MSIOF0_RSCK_MARK,		PORT121_FN2),
	PINMUX_DATA(MSIOF1_TSCK_PORT121_MARK,	PORT121_FN3,	MSEL4CR_10_0),
	PINMUX_DATA(IRQ6_PORT121_MARK,		PORT121_FN0,	MSEL1CR_6_0),

	/* Port122 */
	PINMUX_DATA(A19_MARK,			PORT122_FN1),
	PINMUX_DATA(MSIOF0_RXD_MARK,		PORT122_FN2),

	/* Port123 */
	PINMUX_DATA(A18_MARK,			PORT123_FN1),
	PINMUX_DATA(MSIOF0_TSCK_MARK,		PORT123_FN2),

	/* Port124 */
	PINMUX_DATA(A17_MARK,			PORT124_FN1),
	PINMUX_DATA(MSIOF0_TSYNC_MARK,		PORT124_FN2),

	/* Port125 - Port141 Function */
	PINMUX_DATA(A16_MARK,			PORT125_FN1),
	PINMUX_DATA(A15_MARK,			PORT126_FN1),
	PINMUX_DATA(A14_MARK,			PORT127_FN1),
	PINMUX_DATA(A13_MARK,			PORT128_FN1),
	PINMUX_DATA(A12_MARK,			PORT129_FN1),
	PINMUX_DATA(A11_MARK,			PORT130_FN1),
	PINMUX_DATA(A10_MARK,			PORT131_FN1),
	PINMUX_DATA(A9_MARK,			PORT132_FN1),
	PINMUX_DATA(A8_MARK,			PORT133_FN1),
	PINMUX_DATA(A7_MARK,			PORT134_FN1),
	PINMUX_DATA(A6_MARK,			PORT135_FN1),
	PINMUX_DATA(A5_FCDE_MARK,		PORT136_FN1),
	PINMUX_DATA(A4_FOE_MARK,		PORT137_FN1),
	PINMUX_DATA(A3_MARK,			PORT138_FN1),
	PINMUX_DATA(A2_MARK,			PORT139_FN1),
	PINMUX_DATA(A1_MARK,			PORT140_FN1),
	PINMUX_DATA(CKO_MARK,			PORT141_FN1),

	/* Port142 - Port157 Function1 */
	PINMUX_DATA(D15_NAF15_MARK,		PORT142_FN1),
	PINMUX_DATA(D14_NAF14_MARK,		PORT143_FN1),
	PINMUX_DATA(D13_NAF13_MARK,		PORT144_FN1),
	PINMUX_DATA(D12_NAF12_MARK,		PORT145_FN1),
	PINMUX_DATA(D11_NAF11_MARK,		PORT146_FN1),
	PINMUX_DATA(D10_NAF10_MARK,		PORT147_FN1),
	PINMUX_DATA(D9_NAF9_MARK,		PORT148_FN1),
	PINMUX_DATA(D8_NAF8_MARK,		PORT149_FN1),
	PINMUX_DATA(D7_NAF7_MARK,		PORT150_FN1),
	PINMUX_DATA(D6_NAF6_MARK,		PORT151_FN1),
	PINMUX_DATA(D5_NAF5_MARK,		PORT152_FN1),
	PINMUX_DATA(D4_NAF4_MARK,		PORT153_FN1),
	PINMUX_DATA(D3_NAF3_MARK,		PORT154_FN1),
	PINMUX_DATA(D2_NAF2_MARK,		PORT155_FN1),
	PINMUX_DATA(D1_NAF1_MARK,		PORT156_FN1),
	PINMUX_DATA(D0_NAF0_MARK,		PORT157_FN1),

	/* Port142 - Port149 Function3 */
	PINMUX_DATA(MMC1_D7_PORT142_MARK,	PORT142_FN3,	MSEL4CR_15_1),
	PINMUX_DATA(MMC1_D6_PORT143_MARK,	PORT143_FN3,	MSEL4CR_15_1),
	PINMUX_DATA(MMC1_D5_PORT144_MARK,	PORT144_FN3,	MSEL4CR_15_1),
	PINMUX_DATA(MMC1_D4_PORT145_MARK,	PORT145_FN3,	MSEL4CR_15_1),
	PINMUX_DATA(MMC1_D3_PORT146_MARK,	PORT146_FN3,	MSEL4CR_15_1),
	PINMUX_DATA(MMC1_D2_PORT147_MARK,	PORT147_FN3,	MSEL4CR_15_1),
	PINMUX_DATA(MMC1_D1_PORT148_MARK,	PORT148_FN3,	MSEL4CR_15_1),
	PINMUX_DATA(MMC1_D0_PORT149_MARK,	PORT149_FN3,	MSEL4CR_15_1),

	/* Port158 */
	PINMUX_DATA(D31_MARK,			PORT158_FN1),
	PINMUX_DATA(SCIFA3_SCK_PORT158_MARK,	PORT158_FN2,	MSEL5CR_8_1),
	PINMUX_DATA(RMII_REF125CK_MARK,		PORT158_FN3),
	PINMUX_DATA(LCD0_D21_PORT158_MARK,	PORT158_FN4,	MSEL5CR_6_1),
	PINMUX_DATA(IRDA_FIRSEL_MARK,		PORT158_FN5),
	PINMUX_DATA(IDE_D15_MARK,		PORT158_FN6),

	/* Port159 */
	PINMUX_DATA(D30_MARK,			PORT159_FN1),
	PINMUX_DATA(SCIFA3_RXD_PORT159_MARK,	PORT159_FN2,	MSEL5CR_8_1),
	PINMUX_DATA(RMII_REF50CK_MARK,		PORT159_FN3),
	PINMUX_DATA(LCD0_D23_PORT159_MARK,	PORT159_FN4,	MSEL5CR_6_1),
	PINMUX_DATA(IDE_D14_MARK,		PORT159_FN6),

	/* Port160 */
	PINMUX_DATA(D29_MARK,			PORT160_FN1),
	PINMUX_DATA(SCIFA3_TXD_PORT160_MARK,	PORT160_FN2,	MSEL5CR_8_1),
	PINMUX_DATA(LCD0_D22_PORT160_MARK,	PORT160_FN4,	MSEL5CR_6_1),
	PINMUX_DATA(VIO1_HD_MARK,		PORT160_FN5),
	PINMUX_DATA(IDE_D13_MARK,		PORT160_FN6),

	/* Port161 */
	PINMUX_DATA(D28_MARK,			PORT161_FN1),
	PINMUX_DATA(SCIFA3_RTS_PORT161_MARK,	PORT161_FN2,	MSEL5CR_8_1),
	PINMUX_DATA(ET_RX_DV_MARK,		PORT161_FN3),
	PINMUX_DATA(LCD0_D20_PORT161_MARK,	PORT161_FN4,	MSEL5CR_6_1),
	PINMUX_DATA(IRDA_IN_MARK,		PORT161_FN5),
	PINMUX_DATA(IDE_D12_MARK,		PORT161_FN6),

	/* Port162 */
	PINMUX_DATA(D27_MARK,			PORT162_FN1),
	PINMUX_DATA(SCIFA3_CTS_PORT162_MARK,	PORT162_FN2,	MSEL5CR_8_1),
	PINMUX_DATA(LCD0_D19_PORT162_MARK,	PORT162_FN4,	MSEL5CR_6_1),
	PINMUX_DATA(IRDA_OUT_MARK,		PORT162_FN5),
	PINMUX_DATA(IDE_D11_MARK,		PORT162_FN6),

	/* Port163 */
	PINMUX_DATA(D26_MARK,			PORT163_FN1),
	PINMUX_DATA(MSIOF2_SS2_MARK,		PORT163_FN2),
	PINMUX_DATA(ET_COL_MARK,		PORT163_FN3),
	PINMUX_DATA(LCD0_D18_PORT163_MARK,	PORT163_FN4,	MSEL5CR_6_1),
	PINMUX_DATA(IROUT_MARK,			PORT163_FN5),
	PINMUX_DATA(IDE_D10_MARK,		PORT163_FN6),

	/* Port164 */
	PINMUX_DATA(D25_MARK,			PORT164_FN1),
	PINMUX_DATA(MSIOF2_TSYNC_MARK,		PORT164_FN2),
	PINMUX_DATA(ET_PHY_INT_MARK,		PORT164_FN3),
	PINMUX_DATA(LCD0_RD_MARK,		PORT164_FN4),
	PINMUX_DATA(IDE_D9_MARK,		PORT164_FN6),

	/* Port165 */
	PINMUX_DATA(D24_MARK,			PORT165_FN1),
	PINMUX_DATA(MSIOF2_RXD_MARK,		PORT165_FN2),
	PINMUX_DATA(LCD0_LCLK_PORT165_MARK,	PORT165_FN4,	MSEL5CR_6_1),
	PINMUX_DATA(IDE_D8_MARK,		PORT165_FN6),

	/* Port166 - Port171 Function1 */
	PINMUX_DATA(D21_MARK,			PORT166_FN1),
	PINMUX_DATA(D20_MARK,			PORT167_FN1),
	PINMUX_DATA(D19_MARK,			PORT168_FN1),
	PINMUX_DATA(D18_MARK,			PORT169_FN1),
	PINMUX_DATA(D17_MARK,			PORT170_FN1),
	PINMUX_DATA(D16_MARK,			PORT171_FN1),

	/* Port166 - Port171 Function3 */
	PINMUX_DATA(ET_ETXD5_MARK,		PORT166_FN3),
	PINMUX_DATA(ET_ETXD4_MARK,		PORT167_FN3),
	PINMUX_DATA(ET_ETXD3_MARK,		PORT168_FN3),
	PINMUX_DATA(ET_ETXD2_MARK,		PORT169_FN3),
	PINMUX_DATA(ET_ETXD1_MARK,		PORT170_FN3),
	PINMUX_DATA(ET_ETXD0_MARK,		PORT171_FN3),

	/* Port166 - Port171 Function6 */
	PINMUX_DATA(IDE_D5_MARK,		PORT166_FN6),
	PINMUX_DATA(IDE_D4_MARK,		PORT167_FN6),
	PINMUX_DATA(IDE_D3_MARK,		PORT168_FN6),
	PINMUX_DATA(IDE_D2_MARK,		PORT169_FN6),
	PINMUX_DATA(IDE_D1_MARK,		PORT170_FN6),
	PINMUX_DATA(IDE_D0_MARK,		PORT171_FN6),

	/* Port167 - Port171 IRQ */
	PINMUX_DATA(IRQ31_PORT167_MARK,		PORT167_FN0,	MSEL1CR_31_0),
	PINMUX_DATA(IRQ27_PORT168_MARK,		PORT168_FN0,	MSEL1CR_27_0),
	PINMUX_DATA(IRQ28_PORT169_MARK,		PORT169_FN0,	MSEL1CR_28_0),
	PINMUX_DATA(IRQ29_PORT170_MARK,		PORT170_FN0,	MSEL1CR_29_0),
	PINMUX_DATA(IRQ30_PORT171_MARK,		PORT171_FN0,	MSEL1CR_30_0),

	/* Port172 */
	PINMUX_DATA(D23_MARK,			PORT172_FN1),
	PINMUX_DATA(SCIFB_RTS_PORT172_MARK,	PORT172_FN2,	MSEL5CR_17_1),
	PINMUX_DATA(ET_ETXD7_MARK,		PORT172_FN3),
	PINMUX_DATA(IDE_D7_MARK,		PORT172_FN6),
	PINMUX_DATA(IRQ4_PORT172_MARK,		PORT172_FN0,	MSEL1CR_4_1),

	/* Port173 */
	PINMUX_DATA(D22_MARK,			PORT173_FN1),
	PINMUX_DATA(SCIFB_CTS_PORT173_MARK,	PORT173_FN2,	MSEL5CR_17_1),
	PINMUX_DATA(ET_ETXD6_MARK,		PORT173_FN3),
	PINMUX_DATA(IDE_D6_MARK,		PORT173_FN6),
	PINMUX_DATA(IRQ6_PORT173_MARK,		PORT173_FN0,	MSEL1CR_6_1),

	/* Port174 */
	PINMUX_DATA(A26_MARK,			PORT174_FN1),
	PINMUX_DATA(MSIOF0_TXD_MARK,		PORT174_FN2),
	PINMUX_DATA(ET_RX_CLK_MARK,		PORT174_FN3),
	PINMUX_DATA(SCIFA3_RXD_PORT174_MARK,	PORT174_FN4,	MSEL5CR_8_0),

	/* Port175 */
	PINMUX_DATA(A0_MARK,			PORT175_FN1),
	PINMUX_DATA(BS_MARK,			PORT175_FN2),
	PINMUX_DATA(ET_WOL_MARK,		PORT175_FN3),
	PINMUX_DATA(SCIFA3_TXD_PORT175_MARK,	PORT175_FN4,	MSEL5CR_8_0),

	/* Port176 */
	PINMUX_DATA(ET_GTX_CLK_MARK,		PORT176_FN3),

	/* Port177 */
	PINMUX_DATA(WAIT_PORT177_MARK,		PORT177_FN1,	MSEL5CR_2_0),
	PINMUX_DATA(ET_LINK_MARK,		PORT177_FN3),
	PINMUX_DATA(IDE_IOWR_MARK,		PORT177_FN6),
	PINMUX_DATA(SDHI2_WP_PORT177_MARK,	PORT177_FN7,	MSEL5CR_19_1),

	/* Port178 */
	PINMUX_DATA(VIO0_D12_MARK,		PORT178_FN1),
	PINMUX_DATA(VIO1_D4_MARK,		PORT178_FN5),
	PINMUX_DATA(IDE_IORD_MARK,		PORT178_FN6),

	/* Port179 */
	PINMUX_DATA(VIO0_D11_MARK,		PORT179_FN1),
	PINMUX_DATA(VIO1_D3_MARK,		PORT179_FN5),
	PINMUX_DATA(IDE_IORDY_MARK,		PORT179_FN6),

	/* Port180 */
	PINMUX_DATA(VIO0_D10_MARK,		PORT180_FN1),
	PINMUX_DATA(TPU0TO3_MARK,		PORT180_FN4),
	PINMUX_DATA(VIO1_D2_MARK,		PORT180_FN5),
	PINMUX_DATA(IDE_INT_MARK,		PORT180_FN6),
	PINMUX_DATA(IRQ24_MARK,			PORT180_FN0),

	/* Port181 */
	PINMUX_DATA(VIO0_D9_MARK,		PORT181_FN1),
	PINMUX_DATA(VIO1_D1_MARK,		PORT181_FN5),
	PINMUX_DATA(IDE_RST_MARK,		PORT181_FN6),

	/* Port182 */
	PINMUX_DATA(VIO0_D8_MARK,		PORT182_FN1),
	PINMUX_DATA(VIO1_D0_MARK,		PORT182_FN5),
	PINMUX_DATA(IDE_DIRECTION_MARK,		PORT182_FN6),

	/* Port183 */
	PINMUX_DATA(DREQ1_MARK,			PORT183_FN1),
	PINMUX_DATA(BBIF2_TXD2_PORT183_MARK,	PORT183_FN2,	MSEL5CR_0_1),
	PINMUX_DATA(ET_TX_EN_MARK,		PORT183_FN3),

	/* Port184 */
	PINMUX_DATA(DACK1_MARK,			PORT184_FN1),
	PINMUX_DATA(BBIF2_TSYNC2_PORT184_MARK,	PORT184_FN2,	MSEL5CR_0_1),
	PINMUX_DATA(ET_TX_CLK_MARK,		PORT184_FN3),

	/* Port185 - Port192 Function1 */
	PINMUX_DATA(SCIFA1_SCK_MARK,		PORT185_FN1),
	PINMUX_DATA(SCIFB_RTS_PORT186_MARK,	PORT186_FN1,	MSEL5CR_17_0),
	PINMUX_DATA(SCIFB_CTS_PORT187_MARK,	PORT187_FN1,	MSEL5CR_17_0),
	PINMUX_DATA(SCIFA0_SCK_MARK,		PORT188_FN1),
	PINMUX_DATA(SCIFB_SCK_PORT190_MARK,	PORT190_FN1,	MSEL5CR_17_0),
	PINMUX_DATA(SCIFB_RXD_PORT191_MARK,	PORT191_FN1,	MSEL5CR_17_0),
	PINMUX_DATA(SCIFB_TXD_PORT192_MARK,	PORT192_FN1,	MSEL5CR_17_0),

	/* Port185 - Port192 Function3 */
	PINMUX_DATA(ET_ERXD0_MARK,		PORT185_FN3),
	PINMUX_DATA(ET_ERXD1_MARK,		PORT186_FN3),
	PINMUX_DATA(ET_ERXD2_MARK,		PORT187_FN3),
	PINMUX_DATA(ET_ERXD3_MARK,		PORT188_FN3),
	PINMUX_DATA(ET_ERXD4_MARK,		PORT189_FN3),
	PINMUX_DATA(ET_ERXD5_MARK,		PORT190_FN3),
	PINMUX_DATA(ET_ERXD6_MARK,		PORT191_FN3),
	PINMUX_DATA(ET_ERXD7_MARK,		PORT192_FN3),

	/* Port185 - Port192 Function6 */
	PINMUX_DATA(STP1_IPCLK_MARK,		PORT185_FN6),
	PINMUX_DATA(STP1_IPD0_PORT186_MARK,	PORT186_FN6,	MSEL5CR_23_0),
	PINMUX_DATA(STP1_IPEN_PORT187_MARK,	PORT187_FN6,	MSEL5CR_23_0),
	PINMUX_DATA(STP1_IPSYNC_MARK,		PORT188_FN6),
	PINMUX_DATA(STP0_IPCLK_MARK,		PORT189_FN6),
	PINMUX_DATA(STP0_IPD0_MARK,		PORT190_FN6),
	PINMUX_DATA(STP0_IPEN_MARK,		PORT191_FN6),
	PINMUX_DATA(STP0_IPSYNC_MARK,		PORT192_FN6),

	/* Port193 */
	PINMUX_DATA(SCIFA0_CTS_MARK,		PORT193_FN1),
	PINMUX_DATA(RMII_CRS_DV_MARK,		PORT193_FN3),
	PINMUX_DATA(STP1_IPEN_PORT193_MARK,	PORT193_FN6,	MSEL5CR_23_1), /* ? */
	PINMUX_DATA(LCD1_D17_MARK,		PORT193_FN7),

	/* Port194 */
	PINMUX_DATA(SCIFA0_RTS_MARK,		PORT194_FN1),
	PINMUX_DATA(RMII_RX_ER_MARK,		PORT194_FN3),
	PINMUX_DATA(STP1_IPD0_PORT194_MARK,	PORT194_FN6,	MSEL5CR_23_1), /* ? */
	PINMUX_DATA(LCD1_D16_MARK,		PORT194_FN7),

	/* Port195 */
	PINMUX_DATA(SCIFA1_RXD_MARK,		PORT195_FN1),
	PINMUX_DATA(RMII_RXD0_MARK,		PORT195_FN3),
	PINMUX_DATA(STP1_IPD3_MARK,		PORT195_FN6),
	PINMUX_DATA(LCD1_D15_MARK,		PORT195_FN7),

	/* Port196 */
	PINMUX_DATA(SCIFA1_TXD_MARK,		PORT196_FN1),
	PINMUX_DATA(RMII_RXD1_MARK,		PORT196_FN3),
	PINMUX_DATA(STP1_IPD2_MARK,		PORT196_FN6),
	PINMUX_DATA(LCD1_D14_MARK,		PORT196_FN7),

	/* Port197 */
	PINMUX_DATA(SCIFA0_RXD_MARK,		PORT197_FN1),
	PINMUX_DATA(VIO1_CLK_MARK,		PORT197_FN5),
	PINMUX_DATA(STP1_IPD5_MARK,		PORT197_FN6),
	PINMUX_DATA(LCD1_D19_MARK,		PORT197_FN7),

	/* Port198 */
	PINMUX_DATA(SCIFA0_TXD_MARK,		PORT198_FN1),
	PINMUX_DATA(VIO1_VD_MARK,		PORT198_FN5),
	PINMUX_DATA(STP1_IPD4_MARK,		PORT198_FN6),
	PINMUX_DATA(LCD1_D18_MARK,		PORT198_FN7),

	/* Port199 */
	PINMUX_DATA(MEMC_NWE_MARK,		PORT199_FN1),
	PINMUX_DATA(SCIFA2_SCK_PORT199_MARK,	PORT199_FN2,	MSEL5CR_7_1),
	PINMUX_DATA(RMII_TX_EN_MARK,		PORT199_FN3),
	PINMUX_DATA(SIM_D_PORT199_MARK,		PORT199_FN4,	MSEL5CR_21_1),
	PINMUX_DATA(STP1_IPD1_MARK,		PORT199_FN6),
	PINMUX_DATA(LCD1_D13_MARK,		PORT199_FN7),

	/* Port200 */
	PINMUX_DATA(MEMC_NOE_MARK,		PORT200_FN1),
	PINMUX_DATA(SCIFA2_RXD_MARK,		PORT200_FN2),
	PINMUX_DATA(RMII_TXD0_MARK,		PORT200_FN3),
	PINMUX_DATA(STP0_IPD7_MARK,		PORT200_FN6),
	PINMUX_DATA(LCD1_D12_MARK,		PORT200_FN7),

	/* Port201 */
	PINMUX_DATA(MEMC_WAIT_MARK,		PORT201_FN1,	MSEL4CR_6_0),
	PINMUX_DATA(MEMC_DREQ1_MARK,		PORT201_FN1,	MSEL4CR_6_1),

	PINMUX_DATA(SCIFA2_TXD_MARK,		PORT201_FN2),
	PINMUX_DATA(RMII_TXD1_MARK,		PORT201_FN3),
	PINMUX_DATA(STP0_IPD6_MARK,		PORT201_FN6),
	PINMUX_DATA(LCD1_D11_MARK,		PORT201_FN7),

	/* Port202 */
	PINMUX_DATA(MEMC_BUSCLK_MARK,		PORT202_FN1,	MSEL4CR_6_0),
	PINMUX_DATA(MEMC_A0_MARK,		PORT202_FN1,	MSEL4CR_6_1),

	PINMUX_DATA(MSIOF1_SS2_PORT202_MARK,	PORT202_FN2,	MSEL4CR_10_1),
	PINMUX_DATA(RMII_MDC_MARK,		PORT202_FN3),
	PINMUX_DATA(TPU0TO2_PORT202_MARK,	PORT202_FN4,	MSEL5CR_25_1),
	PINMUX_DATA(IDE_CS0_MARK,		PORT202_FN6),
	PINMUX_DATA(SDHI2_CD_PORT202_MARK,	PORT202_FN7,	MSEL5CR_19_1),
	PINMUX_DATA(IRQ21_MARK,			PORT202_FN0),

	/* Port203 - Port208 Function1 */
	PINMUX_DATA(SDHI2_CLK_MARK,		PORT203_FN1),
	PINMUX_DATA(SDHI2_CMD_MARK,		PORT204_FN1),
	PINMUX_DATA(SDHI2_D0_MARK,		PORT205_FN1),
	PINMUX_DATA(SDHI2_D1_MARK,		PORT206_FN1),
	PINMUX_DATA(SDHI2_D2_MARK,		PORT207_FN1),
	PINMUX_DATA(SDHI2_D3_MARK,		PORT208_FN1),

	/* Port203 - Port208 Function3 */
	PINMUX_DATA(ET_TX_ER_MARK,		PORT203_FN3),
	PINMUX_DATA(ET_RX_ER_MARK,		PORT204_FN3),
	PINMUX_DATA(ET_CRS_MARK,		PORT205_FN3),
	PINMUX_DATA(ET_MDC_MARK,		PORT206_FN3),
	PINMUX_DATA(ET_MDIO_MARK,		PORT207_FN3),
	PINMUX_DATA(RMII_MDIO_MARK,		PORT208_FN3),

	/* Port203 - Port208 Function6 */
	PINMUX_DATA(IDE_A2_MARK,		PORT203_FN6),
	PINMUX_DATA(IDE_A1_MARK,		PORT204_FN6),
	PINMUX_DATA(IDE_A0_MARK,		PORT205_FN6),
	PINMUX_DATA(IDE_IODACK_MARK,		PORT206_FN6),
	PINMUX_DATA(IDE_IODREQ_MARK,		PORT207_FN6),
	PINMUX_DATA(IDE_CS1_MARK,		PORT208_FN6),

	/* Port203 - Port208 Function7 */
	PINMUX_DATA(SCIFA4_TXD_PORT203_MARK,	PORT203_FN7,	MSEL5CR_12_0,	MSEL5CR_11_1),
	PINMUX_DATA(SCIFA4_RXD_PORT204_MARK,	PORT204_FN7,	MSEL5CR_12_0,	MSEL5CR_11_1),
	PINMUX_DATA(SCIFA4_SCK_PORT205_MARK,	PORT205_FN7,	MSEL5CR_10_1),
	PINMUX_DATA(SCIFA5_SCK_PORT206_MARK,	PORT206_FN7,	MSEL5CR_13_1),
	PINMUX_DATA(SCIFA5_RXD_PORT207_MARK,	PORT207_FN7,	MSEL5CR_15_0,	MSEL5CR_14_1),
	PINMUX_DATA(SCIFA5_TXD_PORT208_MARK,	PORT208_FN7,	MSEL5CR_15_0,	MSEL5CR_14_1),

	/* Port209 */
	PINMUX_DATA(VBUS_MARK,			PORT209_FN1),
	PINMUX_DATA(IRQ7_PORT209_MARK,		PORT209_FN0,	MSEL1CR_7_0),

	/* Port210 */
	PINMUX_DATA(IRQ9_PORT210_MARK,		PORT210_FN0,	MSEL1CR_9_1),
	PINMUX_DATA(HDMI_HPD_MARK,		PORT210_FN1),

	/* Port211 */
	PINMUX_DATA(IRQ16_PORT211_MARK,		PORT211_FN0,	MSEL1CR_16_1),
	PINMUX_DATA(HDMI_CEC_MARK,		PORT211_FN1),

	/* SDENC */
	PINMUX_DATA(SDENC_CPG_MARK,				MSEL4CR_19_0),
	PINMUX_DATA(SDENC_DV_CLKI_MARK,				MSEL4CR_19_1),

	/* SYSC */
	PINMUX_DATA(RESETP_PULLUP_MARK,				MSEL4CR_4_0),
	PINMUX_DATA(RESETP_PLAIN_MARK,				MSEL4CR_4_1),

	/* DEBUG */
	PINMUX_DATA(EDEBGREQ_PULLDOWN_MARK,			MSEL4CR_1_0),
	PINMUX_DATA(EDEBGREQ_PULLUP_MARK,			MSEL4CR_1_1),

	PINMUX_DATA(TRACEAUD_FROM_VIO_MARK,			MSEL5CR_30_0,	MSEL5CR_29_0),
	PINMUX_DATA(TRACEAUD_FROM_LCDC0_MARK,			MSEL5CR_30_0,	MSEL5CR_29_1),
	PINMUX_DATA(TRACEAUD_FROM_MEMC_MARK,			MSEL5CR_30_1,	MSEL5CR_29_0),
};

#define __I		(SH_PFC_PIN_CFG_INPUT)
#define __O		(SH_PFC_PIN_CFG_OUTPUT)
#define __IO		(SH_PFC_PIN_CFG_INPUT | SH_PFC_PIN_CFG_OUTPUT)
#define __PD		(SH_PFC_PIN_CFG_PULL_DOWN)
#define __PU		(SH_PFC_PIN_CFG_PULL_UP)
#define __PUD		(SH_PFC_PIN_CFG_PULL_DOWN | SH_PFC_PIN_CFG_PULL_UP)

#define R8A7740_PIN_I_PD(pin)		SH_PFC_PIN_CFG(pin, __I | __PD)
#define R8A7740_PIN_I_PU(pin)		SH_PFC_PIN_CFG(pin, __I | __PU)
#define R8A7740_PIN_I_PU_PD(pin)	SH_PFC_PIN_CFG(pin, __I | __PUD)
#define R8A7740_PIN_IO(pin)		SH_PFC_PIN_CFG(pin, __IO)
#define R8A7740_PIN_IO_PD(pin)		SH_PFC_PIN_CFG(pin, __IO | __PD)
#define R8A7740_PIN_IO_PU(pin)		SH_PFC_PIN_CFG(pin, __IO | __PU)
#define R8A7740_PIN_IO_PU_PD(pin)	SH_PFC_PIN_CFG(pin, __IO | __PUD)
#define R8A7740_PIN_O(pin)		SH_PFC_PIN_CFG(pin, __O)
#define R8A7740_PIN_O_PU_PD(pin)	SH_PFC_PIN_CFG(pin, __O | __PUD)

static const struct sh_pfc_pin pinmux_pins[] = {
	/* Table 56-1 (I/O and Pull U/D) */
	R8A7740_PIN_IO_PD(0),		R8A7740_PIN_IO_PD(1),
	R8A7740_PIN_IO_PD(2),		R8A7740_PIN_IO_PD(3),
	R8A7740_PIN_IO_PD(4),		R8A7740_PIN_IO_PD(5),
	R8A7740_PIN_IO_PD(6),		R8A7740_PIN_IO(7),
	R8A7740_PIN_IO(8),		R8A7740_PIN_IO(9),
	R8A7740_PIN_IO_PD(10),		R8A7740_PIN_IO_PD(11),
	R8A7740_PIN_IO_PD(12),		R8A7740_PIN_IO_PU_PD(13),
	R8A7740_PIN_IO_PD(14),		R8A7740_PIN_IO_PD(15),
	R8A7740_PIN_IO_PD(16),		R8A7740_PIN_IO_PD(17),
	R8A7740_PIN_IO(18),		R8A7740_PIN_IO_PU(19),
	R8A7740_PIN_IO_PU_PD(20),	R8A7740_PIN_IO_PD(21),
	R8A7740_PIN_IO_PU_PD(22),	R8A7740_PIN_IO(23),
	R8A7740_PIN_IO_PU(24),		R8A7740_PIN_IO_PU(25),
	R8A7740_PIN_IO_PU(26),		R8A7740_PIN_IO_PU(27),
	R8A7740_PIN_IO_PU(28),		R8A7740_PIN_IO_PU(29),
	R8A7740_PIN_IO_PU(30),		R8A7740_PIN_IO_PD(31),
	R8A7740_PIN_IO_PD(32),		R8A7740_PIN_IO_PD(33),
	R8A7740_PIN_IO_PD(34),		R8A7740_PIN_IO_PU(35),
	R8A7740_PIN_IO_PU(36),		R8A7740_PIN_IO_PD(37),
	R8A7740_PIN_IO_PU(38),		R8A7740_PIN_IO_PD(39),
	R8A7740_PIN_IO_PU_PD(40),	R8A7740_PIN_IO_PD(41),
	R8A7740_PIN_IO_PD(42),		R8A7740_PIN_IO_PU_PD(43),
	R8A7740_PIN_IO_PU_PD(44),	R8A7740_PIN_IO_PU_PD(45),
	R8A7740_PIN_IO_PU_PD(46),	R8A7740_PIN_IO_PU_PD(47),
	R8A7740_PIN_IO_PU_PD(48),	R8A7740_PIN_IO_PU_PD(49),
	R8A7740_PIN_IO_PU_PD(50),	R8A7740_PIN_IO_PD(51),
	R8A7740_PIN_IO_PD(52),		R8A7740_PIN_IO_PD(53),
	R8A7740_PIN_IO_PD(54),		R8A7740_PIN_IO_PU_PD(55),
	R8A7740_PIN_IO_PU_PD(56),	R8A7740_PIN_IO_PU_PD(57),
	R8A7740_PIN_IO_PU_PD(58),	R8A7740_PIN_IO_PU_PD(59),
	R8A7740_PIN_IO_PU_PD(60),	R8A7740_PIN_IO_PD(61),
	R8A7740_PIN_IO_PD(62),		R8A7740_PIN_IO_PD(63),
	R8A7740_PIN_IO_PD(64),		R8A7740_PIN_IO_PD(65),
	R8A7740_PIN_IO_PU_PD(66),	R8A7740_PIN_IO_PU_PD(67),
	R8A7740_PIN_IO_PU_PD(68),	R8A7740_PIN_IO_PU_PD(69),
	R8A7740_PIN_IO_PU_PD(70),	R8A7740_PIN_IO_PU_PD(71),
	R8A7740_PIN_IO_PU_PD(72),	R8A7740_PIN_IO_PU_PD(73),
	R8A7740_PIN_IO_PU_PD(74),	R8A7740_PIN_IO_PU_PD(75),
	R8A7740_PIN_IO_PU_PD(76),	R8A7740_PIN_IO_PU_PD(77),
	R8A7740_PIN_IO_PU_PD(78),	R8A7740_PIN_IO_PU_PD(79),
	R8A7740_PIN_IO_PU_PD(80),	R8A7740_PIN_IO_PU_PD(81),
	R8A7740_PIN_IO(82),		R8A7740_PIN_IO_PU_PD(83),
	R8A7740_PIN_IO(84),		R8A7740_PIN_IO_PD(85),
	R8A7740_PIN_IO_PD(86),		R8A7740_PIN_IO_PD(87),
	R8A7740_PIN_IO_PD(88),		R8A7740_PIN_IO_PD(89),
	R8A7740_PIN_IO_PD(90),		R8A7740_PIN_IO_PU_PD(91),
	R8A7740_PIN_IO_PU_PD(92),	R8A7740_PIN_IO_PU_PD(93),
	R8A7740_PIN_IO_PU_PD(94),	R8A7740_PIN_IO_PU_PD(95),
	R8A7740_PIN_IO_PU_PD(96),	R8A7740_PIN_IO_PU_PD(97),
	R8A7740_PIN_IO_PU_PD(98),	R8A7740_PIN_IO_PU_PD(99),
	R8A7740_PIN_IO_PU_PD(100),	R8A7740_PIN_IO(101),
	R8A7740_PIN_IO_PU(102),		R8A7740_PIN_IO_PU_PD(103),
	R8A7740_PIN_IO_PU(104),		R8A7740_PIN_IO_PU(105),
	R8A7740_PIN_IO_PU_PD(106),	R8A7740_PIN_IO(107),
	R8A7740_PIN_IO(108),		R8A7740_PIN_IO(109),
	R8A7740_PIN_IO(110),		R8A7740_PIN_IO(111),
	R8A7740_PIN_IO(112),		R8A7740_PIN_IO(113),
	R8A7740_PIN_IO_PU_PD(114),	R8A7740_PIN_IO(115),
	R8A7740_PIN_IO_PD(116),		R8A7740_PIN_IO_PD(117),
	R8A7740_PIN_IO_PD(118),		R8A7740_PIN_IO_PD(119),
	R8A7740_PIN_IO_PD(120),		R8A7740_PIN_IO_PD(121),
	R8A7740_PIN_IO_PD(122),		R8A7740_PIN_IO_PD(123),
	R8A7740_PIN_IO_PD(124),		R8A7740_PIN_IO(125),
	R8A7740_PIN_IO(126),		R8A7740_PIN_IO(127),
	R8A7740_PIN_IO(128),		R8A7740_PIN_IO(129),
	R8A7740_PIN_IO(130),		R8A7740_PIN_IO(131),
	R8A7740_PIN_IO(132),		R8A7740_PIN_IO(133),
	R8A7740_PIN_IO(134),		R8A7740_PIN_IO(135),
	R8A7740_PIN_IO(136),		R8A7740_PIN_IO(137),
	R8A7740_PIN_IO(138),		R8A7740_PIN_IO(139),
	R8A7740_PIN_IO(140),		R8A7740_PIN_IO(141),
	R8A7740_PIN_IO_PU(142),		R8A7740_PIN_IO_PU(143),
	R8A7740_PIN_IO_PU(144),		R8A7740_PIN_IO_PU(145),
	R8A7740_PIN_IO_PU(146),		R8A7740_PIN_IO_PU(147),
	R8A7740_PIN_IO_PU(148),		R8A7740_PIN_IO_PU(149),
	R8A7740_PIN_IO_PU(150),		R8A7740_PIN_IO_PU(151),
	R8A7740_PIN_IO_PU(152),		R8A7740_PIN_IO_PU(153),
	R8A7740_PIN_IO_PU(154),		R8A7740_PIN_IO_PU(155),
	R8A7740_PIN_IO_PU(156),		R8A7740_PIN_IO_PU(157),
	R8A7740_PIN_IO_PD(158),		R8A7740_PIN_IO_PD(159),
	R8A7740_PIN_IO_PU_PD(160),	R8A7740_PIN_IO_PD(161),
	R8A7740_PIN_IO_PD(162),		R8A7740_PIN_IO_PD(163),
	R8A7740_PIN_IO_PD(164),		R8A7740_PIN_IO_PD(165),
	R8A7740_PIN_IO_PU(166),		R8A7740_PIN_IO_PU(167),
	R8A7740_PIN_IO_PU(168),		R8A7740_PIN_IO_PU(169),
	R8A7740_PIN_IO_PU(170),		R8A7740_PIN_IO_PU(171),
	R8A7740_PIN_IO_PD(172),		R8A7740_PIN_IO_PD(173),
	R8A7740_PIN_IO_PD(174),		R8A7740_PIN_IO_PD(175),
	R8A7740_PIN_IO_PU(176),		R8A7740_PIN_IO_PU_PD(177),
	R8A7740_PIN_IO_PU(178),		R8A7740_PIN_IO_PD(179),
	R8A7740_PIN_IO_PD(180),		R8A7740_PIN_IO_PU(181),
	R8A7740_PIN_IO_PU(182),		R8A7740_PIN_IO(183),
	R8A7740_PIN_IO_PD(184),		R8A7740_PIN_IO_PD(185),
	R8A7740_PIN_IO_PD(186),		R8A7740_PIN_IO_PD(187),
	R8A7740_PIN_IO_PD(188),		R8A7740_PIN_IO_PD(189),
	R8A7740_PIN_IO_PD(190),		R8A7740_PIN_IO_PD(191),
	R8A7740_PIN_IO_PD(192),		R8A7740_PIN_IO_PU_PD(193),
	R8A7740_PIN_IO_PU_PD(194),	R8A7740_PIN_IO_PD(195),
	R8A7740_PIN_IO_PU_PD(196),	R8A7740_PIN_IO_PD(197),
	R8A7740_PIN_IO_PU_PD(198),	R8A7740_PIN_IO_PU_PD(199),
	R8A7740_PIN_IO_PU_PD(200),	R8A7740_PIN_IO_PU(201),
	R8A7740_PIN_IO_PU_PD(202),	R8A7740_PIN_IO(203),
	R8A7740_PIN_IO_PU_PD(204),	R8A7740_PIN_IO_PU_PD(205),
	R8A7740_PIN_IO_PU_PD(206),	R8A7740_PIN_IO_PU_PD(207),
	R8A7740_PIN_IO_PU_PD(208),	R8A7740_PIN_IO_PD(209),
	R8A7740_PIN_IO_PD(210),		R8A7740_PIN_IO_PD(211),
};

/* - BSC -------------------------------------------------------------------- */
static const unsigned int bsc_data8_pins[] = {
	/* D[0:7] */
	157, 156, 155, 154, 153, 152, 151, 150,
};
static const unsigned int bsc_data8_mux[] = {
	D0_NAF0_MARK, D1_NAF1_MARK, D2_NAF2_MARK, D3_NAF3_MARK,
	D4_NAF4_MARK, D5_NAF5_MARK, D6_NAF6_MARK, D7_NAF7_MARK,
};
static const unsigned int bsc_data16_pins[] = {
	/* D[0:15] */
	157, 156, 155, 154, 153, 152, 151, 150,
	149, 148, 147, 146, 145, 144, 143, 142,
};
static const unsigned int bsc_data16_mux[] = {
	D0_NAF0_MARK, D1_NAF1_MARK, D2_NAF2_MARK, D3_NAF3_MARK,
	D4_NAF4_MARK, D5_NAF5_MARK, D6_NAF6_MARK, D7_NAF7_MARK,
	D8_NAF8_MARK, D9_NAF9_MARK, D10_NAF10_MARK, D11_NAF11_MARK,
	D12_NAF12_MARK, D13_NAF13_MARK, D14_NAF14_MARK, D15_NAF15_MARK,
};
static const unsigned int bsc_data32_pins[] = {
	/* D[0:31] */
	157, 156, 155, 154, 153, 152, 151, 150,
	149, 148, 147, 146, 145, 144, 143, 142,
	171, 170, 169, 168, 167, 166, 173, 172,
	165, 164, 163, 162, 161, 160, 159, 158,
};
static const unsigned int bsc_data32_mux[] = {
	D0_NAF0_MARK, D1_NAF1_MARK, D2_NAF2_MARK, D3_NAF3_MARK,
	D4_NAF4_MARK, D5_NAF5_MARK, D6_NAF6_MARK, D7_NAF7_MARK,
	D8_NAF8_MARK, D9_NAF9_MARK, D10_NAF10_MARK, D11_NAF11_MARK,
	D12_NAF12_MARK, D13_NAF13_MARK, D14_NAF14_MARK, D15_NAF15_MARK,
	D16_MARK, D17_MARK, D18_MARK, D19_MARK,
	D20_MARK, D21_MARK, D22_MARK, D23_MARK,
	D24_MARK, D25_MARK, D26_MARK, D27_MARK,
	D28_MARK, D29_MARK, D30_MARK, D31_MARK,
};
static const unsigned int bsc_cs0_pins[] = {
	/* CS */
	109,
};
static const unsigned int bsc_cs0_mux[] = {
	CS0_MARK,
};
static const unsigned int bsc_cs2_pins[] = {
	/* CS */
	110,
};
static const unsigned int bsc_cs2_mux[] = {
	CS2_MARK,
};
static const unsigned int bsc_cs4_pins[] = {
	/* CS */
	111,
};
static const unsigned int bsc_cs4_mux[] = {
	CS4_MARK,
};
static const unsigned int bsc_cs5a_0_pins[] = {
	/* CS */
	105,
};
static const unsigned int bsc_cs5a_0_mux[] = {
	CS5A_PORT105_MARK,
};
static const unsigned int bsc_cs5a_1_pins[] = {
	/* CS */
	19,
};
static const unsigned int bsc_cs5a_1_mux[] = {
	CS5A_PORT19_MARK,
};
static const unsigned int bsc_cs5b_pins[] = {
	/* CS */
	103,
};
static const unsigned int bsc_cs5b_mux[] = {
	CS5B_MARK,
};
static const unsigned int bsc_cs6a_pins[] = {
	/* CS */
	104,
};
static const unsigned int bsc_cs6a_mux[] = {
	CS6A_MARK,
};
static const unsigned int bsc_rd_we8_pins[] = {
	/* RD, WE[0] */
	115, 113,
};
static const unsigned int bsc_rd_we8_mux[] = {
	RD_FSC_MARK, WE0_FWE_MARK,
};
static const unsigned int bsc_rd_we16_pins[] = {
	/* RD, WE[0:1] */
	115, 113, 112,
};
static const unsigned int bsc_rd_we16_mux[] = {
	RD_FSC_MARK, WE0_FWE_MARK, WE1_MARK,
};
static const unsigned int bsc_rd_we32_pins[] = {
	/* RD, WE[0:3] */
	115, 113, 112, 108, 107,
};
static const unsigned int bsc_rd_we32_mux[] = {
	RD_FSC_MARK, WE0_FWE_MARK, WE1_MARK, WE2_ICIORD_MARK, WE3_ICIOWR_MARK,
};
static const unsigned int bsc_bs_pins[] = {
	/* BS */
	175,
};
static const unsigned int bsc_bs_mux[] = {
	BS_MARK,
};
static const unsigned int bsc_rdwr_pins[] = {
	/* RDWR */
	114,
};
static const unsigned int bsc_rdwr_mux[] = {
	RDWR_MARK,
};
/* - CEU0 ------------------------------------------------------------------- */
static const unsigned int ceu0_data_0_7_pins[] = {
	/* D[0:7] */
	34, 33, 32, 31, 30, 29, 28, 27,
};
static const unsigned int ceu0_data_0_7_mux[] = {
	VIO0_D0_MARK, VIO0_D1_MARK, VIO0_D2_MARK, VIO0_D3_MARK,
	VIO0_D4_MARK, VIO0_D5_MARK, VIO0_D6_MARK, VIO0_D7_MARK,
};
static const unsigned int ceu0_data_8_15_0_pins[] = {
	/* D[8:15] */
	182, 181, 180, 179, 178, 26, 25, 24,
};
static const unsigned int ceu0_data_8_15_0_mux[] = {
	VIO0_D8_MARK, VIO0_D9_MARK, VIO0_D10_MARK, VIO0_D11_MARK,
	VIO0_D12_MARK, VIO0_D13_PORT26_MARK, VIO0_D14_PORT25_MARK,
	VIO0_D15_PORT24_MARK,
};
static const unsigned int ceu0_data_8_15_1_pins[] = {
	/* D[8:15] */
	182, 181, 180, 179, 178, 22, 95, 96,
};
static const unsigned int ceu0_data_8_15_1_mux[] = {
	VIO0_D8_MARK, VIO0_D9_MARK, VIO0_D10_MARK, VIO0_D11_MARK,
	VIO0_D12_MARK, VIO0_D13_PORT22_MARK, VIO0_D14_PORT95_MARK,
	VIO0_D15_PORT96_MARK,
};
static const unsigned int ceu0_clk_0_pins[] = {
	/* CKO */
	36,
};
static const unsigned int ceu0_clk_0_mux[] = {
	VIO_CKO_MARK,
};
static const unsigned int ceu0_clk_1_pins[] = {
	/* CKO */
	14,
};
static const unsigned int ceu0_clk_1_mux[] = {
	VIO_CKO1_MARK,
};
static const unsigned int ceu0_clk_2_pins[] = {
	/* CKO */
	15,
};
static const unsigned int ceu0_clk_2_mux[] = {
	VIO_CKO2_MARK,
};
static const unsigned int ceu0_sync_pins[] = {
	/* CLK, VD, HD */
	35, 39, 37,
};
static const unsigned int ceu0_sync_mux[] = {
	VIO0_CLK_MARK, VIO0_VD_MARK, VIO0_HD_MARK,
};
static const unsigned int ceu0_field_pins[] = {
	/* FIELD */
	38,
};
static const unsigned int ceu0_field_mux[] = {
	VIO0_FIELD_MARK,
};
/* - CEU1 ------------------------------------------------------------------- */
static const unsigned int ceu1_data_pins[] = {
	/* D[0:7] */
	182, 181, 180, 179, 178, 26, 25, 24,
};
static const unsigned int ceu1_data_mux[] = {
	VIO1_D0_MARK, VIO1_D1_MARK, VIO1_D2_MARK, VIO1_D3_MARK,
	VIO1_D4_MARK, VIO1_D5_MARK, VIO1_D6_MARK, VIO1_D7_MARK,
};
static const unsigned int ceu1_clk_pins[] = {
	/* CKO */
	23,
};
static const unsigned int ceu1_clk_mux[] = {
	VIO_CKO_1_MARK,
};
static const unsigned int ceu1_sync_pins[] = {
	/* CLK, VD, HD */
	197, 198, 160,
};
static const unsigned int ceu1_sync_mux[] = {
	VIO1_CLK_MARK, VIO1_VD_MARK, VIO1_HD_MARK,
};
static const unsigned int ceu1_field_pins[] = {
	/* FIELD */
	21,
};
static const unsigned int ceu1_field_mux[] = {
	VIO1_FIELD_MARK,
};
/* - FSIA ------------------------------------------------------------------- */
static const unsigned int fsia_mclk_in_pins[] = {
	/* CK */
	11,
};
static const unsigned int fsia_mclk_in_mux[] = {
	FSIACK_MARK,
};
static const unsigned int fsia_mclk_out_pins[] = {
	/* OMC */
	10,
};
static const unsigned int fsia_mclk_out_mux[] = {
	FSIAOMC_MARK,
};
static const unsigned int fsia_sclk_in_pins[] = {
	/* ILR, IBT */
	12, 13,
};
static const unsigned int fsia_sclk_in_mux[] = {
	FSIAILR_MARK, FSIAIBT_MARK,
};
static const unsigned int fsia_sclk_out_pins[] = {
	/* OLR, OBT */
	7, 8,
};
static const unsigned int fsia_sclk_out_mux[] = {
	FSIAOLR_MARK, FSIAOBT_MARK,
};
static const unsigned int fsia_data_in_0_pins[] = {
	/* ISLD */
	0,
};
static const unsigned int fsia_data_in_0_mux[] = {
	FSIAISLD_PORT0_MARK,
};
static const unsigned int fsia_data_in_1_pins[] = {
	/* ISLD */
	5,
};
static const unsigned int fsia_data_in_1_mux[] = {
	FSIAISLD_PORT5_MARK,
};
static const unsigned int fsia_data_out_0_pins[] = {
	/* OSLD */
	9,
};
static const unsigned int fsia_data_out_0_mux[] = {
	FSIAOSLD_MARK,
};
static const unsigned int fsia_data_out_1_pins[] = {
	/* OSLD */
	0,
};
static const unsigned int fsia_data_out_1_mux[] = {
	FSIAOSLD1_MARK,
};
static const unsigned int fsia_data_out_2_pins[] = {
	/* OSLD */
	1,
};
static const unsigned int fsia_data_out_2_mux[] = {
	FSIAOSLD2_MARK,
};
static const unsigned int fsia_spdif_0_pins[] = {
	/* SPDIF */
	9,
};
static const unsigned int fsia_spdif_0_mux[] = {
	FSIASPDIF_PORT9_MARK,
};
static const unsigned int fsia_spdif_1_pins[] = {
	/* SPDIF */
	18,
};
static const unsigned int fsia_spdif_1_mux[] = {
	FSIASPDIF_PORT18_MARK,
};
/* - FSIB ------------------------------------------------------------------- */
static const unsigned int fsib_mclk_in_pins[] = {
	/* CK */
	11,
};
static const unsigned int fsib_mclk_in_mux[] = {
	FSIBCK_MARK,
};
/* - GETHER ----------------------------------------------------------------- */
static const unsigned int gether_rmii_pins[] = {
	/* RXD[0:1], RX_ER, CRS_DV, TXD[0:1], TX_EN, REF_CLK, MDC, MDIO */
	195, 196, 194, 193, 200, 201, 199, 159, 202, 208,
};
static const unsigned int gether_rmii_mux[] = {
	RMII_RXD0_MARK, RMII_RXD1_MARK, RMII_RX_ER_MARK, RMII_CRS_DV_MARK,
	RMII_TXD0_MARK, RMII_TXD1_MARK, RMII_TX_EN_MARK, RMII_REF50CK_MARK,
	RMII_MDC_MARK, RMII_MDIO_MARK,
};
static const unsigned int gether_mii_pins[] = {
	/* RXD[0:3], RX_CLK, RX_DV, RX_ER
	 * TXD[0:3], TX_CLK, TX_EN, TX_ER
	 * CRS, COL, MDC, MDIO,
	 */
	185, 186, 187, 188, 174, 161, 204,
	171, 170, 169, 168, 184, 183, 203,
	205, 163, 206, 207,
};
static const unsigned int gether_mii_mux[] = {
	ET_ERXD0_MARK, ET_ERXD1_MARK, ET_ERXD2_MARK, ET_ERXD3_MARK,
	ET_RX_CLK_MARK, ET_RX_DV_MARK, ET_RX_ER_MARK,
	ET_ETXD0_MARK, ET_ETXD1_MARK, ET_ETXD2_MARK, ET_ETXD3_MARK,
	ET_TX_CLK_MARK, ET_TX_EN_MARK, ET_TX_ER_MARK,
	ET_CRS_MARK, ET_COL_MARK, ET_MDC_MARK, ET_MDIO_MARK,
};
static const unsigned int gether_gmii_pins[] = {
	/* RXD[0:7], RX_CLK, RX_DV, RX_ER
	 * TXD[0:7], GTX_CLK, TX_CLK, TX_EN, TX_ER
	 * CRS, COL, MDC, MDIO, REF125CK_MARK,
	 */
	185, 186, 187, 188, 189, 190, 191, 192, 174, 161, 204,
	171, 170, 169, 168, 167, 166, 173, 172, 176, 184, 183, 203,
	205, 163, 206, 207, 158,
};
static const unsigned int gether_gmii_mux[] = {
	ET_ERXD0_MARK, ET_ERXD1_MARK, ET_ERXD2_MARK, ET_ERXD3_MARK,
	ET_ERXD4_MARK, ET_ERXD5_MARK, ET_ERXD6_MARK, ET_ERXD7_MARK,
	ET_RX_CLK_MARK, ET_RX_DV_MARK, ET_RX_ER_MARK,
	ET_ETXD0_MARK, ET_ETXD1_MARK, ET_ETXD2_MARK, ET_ETXD3_MARK,
	ET_ETXD4_MARK, ET_ETXD5_MARK, ET_ETXD6_MARK, ET_ETXD7_MARK,
	ET_GTX_CLK_MARK, ET_TX_CLK_MARK, ET_TX_EN_MARK, ET_TX_ER_MARK,
	ET_CRS_MARK, ET_COL_MARK, ET_MDC_MARK, ET_MDIO_MARK,
	RMII_REF125CK_MARK,
};
static const unsigned int gether_int_pins[] = {
	/* PHY_INT */
	164,
};
static const unsigned int gether_int_mux[] = {
	ET_PHY_INT_MARK,
};
static const unsigned int gether_link_pins[] = {
	/* LINK */
	177,
};
static const unsigned int gether_link_mux[] = {
	ET_LINK_MARK,
};
static const unsigned int gether_wol_pins[] = {
	/* WOL */
	175,
};
static const unsigned int gether_wol_mux[] = {
	ET_WOL_MARK,
};
/* - HDMI ------------------------------------------------------------------- */
static const unsigned int hdmi_pins[] = {
	/* HPD, CEC */
	210, 211,
};
static const unsigned int hdmi_mux[] = {
	HDMI_HPD_MARK, HDMI_CEC_MARK,
};
/* - INTC ------------------------------------------------------------------- */
IRQC_PINS_MUX(0, 0, 2);
IRQC_PINS_MUX(0, 1, 13);
IRQC_PIN_MUX(1, 20);
IRQC_PINS_MUX(2, 0, 11);
IRQC_PINS_MUX(2, 1, 12);
IRQC_PINS_MUX(3, 0, 10);
IRQC_PINS_MUX(3, 1, 14);
IRQC_PINS_MUX(4, 0, 15);
IRQC_PINS_MUX(4, 1, 172);
IRQC_PINS_MUX(5, 0, 0);
IRQC_PINS_MUX(5, 1, 1);
IRQC_PINS_MUX(6, 0, 121);
IRQC_PINS_MUX(6, 1, 173);
IRQC_PINS_MUX(7, 0, 120);
IRQC_PINS_MUX(7, 1, 209);
IRQC_PIN_MUX(8, 119);
IRQC_PINS_MUX(9, 0, 118);
IRQC_PINS_MUX(9, 1, 210);
IRQC_PIN_MUX(10, 19);
IRQC_PIN_MUX(11, 104);
IRQC_PINS_MUX(12, 0, 42);
IRQC_PINS_MUX(12, 1, 97);
IRQC_PINS_MUX(13, 0, 64);
IRQC_PINS_MUX(13, 1, 98);
IRQC_PINS_MUX(14, 0, 63);
IRQC_PINS_MUX(14, 1, 99);
IRQC_PINS_MUX(15, 0, 62);
IRQC_PINS_MUX(15, 1, 100);
IRQC_PINS_MUX(16, 0, 68);
IRQC_PINS_MUX(16, 1, 211);
IRQC_PIN_MUX(17, 69);
IRQC_PIN_MUX(18, 70);
IRQC_PIN_MUX(19, 71);
IRQC_PIN_MUX(20, 67);
IRQC_PIN_MUX(21, 202);
IRQC_PIN_MUX(22, 95);
IRQC_PIN_MUX(23, 96);
IRQC_PIN_MUX(24, 180);
IRQC_PIN_MUX(25, 38);
IRQC_PINS_MUX(26, 0, 58);
IRQC_PINS_MUX(26, 1, 81);
IRQC_PINS_MUX(27, 0, 57);
IRQC_PINS_MUX(27, 1, 168);
IRQC_PINS_MUX(28, 0, 56);
IRQC_PINS_MUX(28, 1, 169);
IRQC_PINS_MUX(29, 0, 50);
IRQC_PINS_MUX(29, 1, 170);
IRQC_PINS_MUX(30, 0, 49);
IRQC_PINS_MUX(30, 1, 171);
IRQC_PINS_MUX(31, 0, 41);
IRQC_PINS_MUX(31, 1, 167);

/* - LCD0 ------------------------------------------------------------------- */
static const unsigned int lcd0_data8_pins[] = {
	/* D[0:7] */
	58, 57, 56, 55, 54, 53, 52, 51,
};
static const unsigned int lcd0_data8_mux[] = {
	LCD0_D0_MARK, LCD0_D1_MARK, LCD0_D2_MARK, LCD0_D3_MARK,
	LCD0_D4_MARK, LCD0_D5_MARK, LCD0_D6_MARK, LCD0_D7_MARK,
};
static const unsigned int lcd0_data9_pins[] = {
	/* D[0:8] */
	58, 57, 56, 55, 54, 53, 52, 51,
	50,
};
static const unsigned int lcd0_data9_mux[] = {
	LCD0_D0_MARK, LCD0_D1_MARK, LCD0_D2_MARK, LCD0_D3_MARK,
	LCD0_D4_MARK, LCD0_D5_MARK, LCD0_D6_MARK, LCD0_D7_MARK,
	LCD0_D8_MARK,
};
static const unsigned int lcd0_data12_pins[] = {
	/* D[0:11] */
	58, 57, 56, 55, 54, 53, 52, 51,
	50, 49, 48, 47,
};
static const unsigned int lcd0_data12_mux[] = {
	LCD0_D0_MARK, LCD0_D1_MARK, LCD0_D2_MARK, LCD0_D3_MARK,
	LCD0_D4_MARK, LCD0_D5_MARK, LCD0_D6_MARK, LCD0_D7_MARK,
	LCD0_D8_MARK, LCD0_D9_MARK, LCD0_D10_MARK, LCD0_D11_MARK,
};
static const unsigned int lcd0_data16_pins[] = {
	/* D[0:15] */
	58, 57, 56, 55, 54, 53, 52, 51,
	50, 49, 48, 47, 46, 45, 44, 43,
};
static const unsigned int lcd0_data16_mux[] = {
	LCD0_D0_MARK, LCD0_D1_MARK, LCD0_D2_MARK, LCD0_D3_MARK,
	LCD0_D4_MARK, LCD0_D5_MARK, LCD0_D6_MARK, LCD0_D7_MARK,
	LCD0_D8_MARK, LCD0_D9_MARK, LCD0_D10_MARK, LCD0_D11_MARK,
	LCD0_D12_MARK, LCD0_D13_MARK, LCD0_D14_MARK, LCD0_D15_MARK,
};
static const unsigned int lcd0_data18_pins[] = {
	/* D[0:17] */
	58, 57, 56, 55, 54, 53, 52, 51,
	50, 49, 48, 47, 46, 45, 44, 43,
	42, 41,
};
static const unsigned int lcd0_data18_mux[] = {
	LCD0_D0_MARK, LCD0_D1_MARK, LCD0_D2_MARK, LCD0_D3_MARK,
	LCD0_D4_MARK, LCD0_D5_MARK, LCD0_D6_MARK, LCD0_D7_MARK,
	LCD0_D8_MARK, LCD0_D9_MARK, LCD0_D10_MARK, LCD0_D11_MARK,
	LCD0_D12_MARK, LCD0_D13_MARK, LCD0_D14_MARK, LCD0_D15_MARK,
	LCD0_D16_MARK, LCD0_D17_MARK,
};
static const unsigned int lcd0_data24_0_pins[] = {
	/* D[0:23] */
	58, 57, 56, 55, 54, 53, 52, 51,
	50, 49, 48, 47, 46, 45, 44, 43,
	42, 41, 40, 4, 3, 2, 0, 1,
};
static const unsigned int lcd0_data24_0_mux[] = {
	LCD0_D0_MARK, LCD0_D1_MARK, LCD0_D2_MARK, LCD0_D3_MARK,
	LCD0_D4_MARK, LCD0_D5_MARK, LCD0_D6_MARK, LCD0_D7_MARK,
	LCD0_D8_MARK, LCD0_D9_MARK, LCD0_D10_MARK, LCD0_D11_MARK,
	LCD0_D12_MARK, LCD0_D13_MARK, LCD0_D14_MARK, LCD0_D15_MARK,
	LCD0_D16_MARK, LCD0_D17_MARK, LCD0_D18_PORT40_MARK, LCD0_D19_PORT4_MARK,
	LCD0_D20_PORT3_MARK, LCD0_D21_PORT2_MARK, LCD0_D22_PORT0_MARK,
	LCD0_D23_PORT1_MARK,
};
static const unsigned int lcd0_data24_1_pins[] = {
	/* D[0:23] */
	58, 57, 56, 55, 54, 53, 52, 51,
	50, 49, 48, 47, 46, 45, 44, 43,
	42, 41, 163, 162, 161, 158, 160, 159,
};
static const unsigned int lcd0_data24_1_mux[] = {
	LCD0_D0_MARK, LCD0_D1_MARK, LCD0_D2_MARK, LCD0_D3_MARK,
	LCD0_D4_MARK, LCD0_D5_MARK, LCD0_D6_MARK, LCD0_D7_MARK,
	LCD0_D8_MARK, LCD0_D9_MARK, LCD0_D10_MARK, LCD0_D11_MARK,
	LCD0_D12_MARK, LCD0_D13_MARK, LCD0_D14_MARK, LCD0_D15_MARK,
	LCD0_D16_MARK, LCD0_D17_MARK, LCD0_D18_PORT163_MARK,
	LCD0_D19_PORT162_MARK, LCD0_D20_PORT161_MARK, LCD0_D21_PORT158_MARK,
	LCD0_D22_PORT160_MARK, LCD0_D23_PORT159_MARK,
};
static const unsigned int lcd0_display_pins[] = {
	/* DON, VCPWC, VEPWC */
	61, 59, 60,
};
static const unsigned int lcd0_display_mux[] = {
	LCD0_DON_MARK, LCD0_VCPWC_MARK, LCD0_VEPWC_MARK,
};
static const unsigned int lcd0_lclk_0_pins[] = {
	/* LCLK */
	102,
};
static const unsigned int lcd0_lclk_0_mux[] = {
	LCD0_LCLK_PORT102_MARK,
};
static const unsigned int lcd0_lclk_1_pins[] = {
	/* LCLK */
	165,
};
static const unsigned int lcd0_lclk_1_mux[] = {
	LCD0_LCLK_PORT165_MARK,
};
static const unsigned int lcd0_sync_pins[] = {
	/* VSYN, HSYN, DCK, DISP */
	63, 64, 62, 65,
};
static const unsigned int lcd0_sync_mux[] = {
	LCD0_VSYN_MARK, LCD0_HSYN_MARK, LCD0_DCK_MARK, LCD0_DISP_MARK,
};
static const unsigned int lcd0_sys_pins[] = {
	/* CS, WR, RD, RS */
	64, 62, 164, 65,
};
static const unsigned int lcd0_sys_mux[] = {
	LCD0_CS_MARK, LCD0_WR_MARK, LCD0_RD_MARK, LCD0_RS_MARK,
};
/* - LCD1 ------------------------------------------------------------------- */
static const unsigned int lcd1_data8_pins[] = {
	/* D[0:7] */
	4, 3, 2, 1, 0, 91, 92, 23,
};
static const unsigned int lcd1_data8_mux[] = {
	LCD1_D0_MARK, LCD1_D1_MARK, LCD1_D2_MARK, LCD1_D3_MARK,
	LCD1_D4_MARK, LCD1_D5_MARK, LCD1_D6_MARK, LCD1_D7_MARK,
};
static const unsigned int lcd1_data9_pins[] = {
	/* D[0:8] */
	4, 3, 2, 1, 0, 91, 92, 23,
	93,
};
static const unsigned int lcd1_data9_mux[] = {
	LCD1_D0_MARK, LCD1_D1_MARK, LCD1_D2_MARK, LCD1_D3_MARK,
	LCD1_D4_MARK, LCD1_D5_MARK, LCD1_D6_MARK, LCD1_D7_MARK,
	LCD1_D8_MARK,
};
static const unsigned int lcd1_data12_pins[] = {
	/* D[0:11] */
	4, 3, 2, 1, 0, 91, 92, 23,
	93, 94, 21, 201,
};
static const unsigned int lcd1_data12_mux[] = {
	LCD1_D0_MARK, LCD1_D1_MARK, LCD1_D2_MARK, LCD1_D3_MARK,
	LCD1_D4_MARK, LCD1_D5_MARK, LCD1_D6_MARK, LCD1_D7_MARK,
	LCD1_D8_MARK, LCD1_D9_MARK, LCD1_D10_MARK, LCD1_D11_MARK,
};
static const unsigned int lcd1_data16_pins[] = {
	/* D[0:15] */
	4, 3, 2, 1, 0, 91, 92, 23,
	93, 94, 21, 201, 200, 199, 196, 195,
};
static const unsigned int lcd1_data16_mux[] = {
	LCD1_D0_MARK, LCD1_D1_MARK, LCD1_D2_MARK, LCD1_D3_MARK,
	LCD1_D4_MARK, LCD1_D5_MARK, LCD1_D6_MARK, LCD1_D7_MARK,
	LCD1_D8_MARK, LCD1_D9_MARK, LCD1_D10_MARK, LCD1_D11_MARK,
	LCD1_D12_MARK, LCD1_D13_MARK, LCD1_D14_MARK, LCD1_D15_MARK,
};
static const unsigned int lcd1_data18_pins[] = {
	/* D[0:17] */
	4, 3, 2, 1, 0, 91, 92, 23,
	93, 94, 21, 201, 200, 199, 196, 195,
	194, 193,
};
static const unsigned int lcd1_data18_mux[] = {
	LCD1_D0_MARK, LCD1_D1_MARK, LCD1_D2_MARK, LCD1_D3_MARK,
	LCD1_D4_MARK, LCD1_D5_MARK, LCD1_D6_MARK, LCD1_D7_MARK,
	LCD1_D8_MARK, LCD1_D9_MARK, LCD1_D10_MARK, LCD1_D11_MARK,
	LCD1_D12_MARK, LCD1_D13_MARK, LCD1_D14_MARK, LCD1_D15_MARK,
	LCD1_D16_MARK, LCD1_D17_MARK,
};
static const unsigned int lcd1_data24_pins[] = {
	/* D[0:23] */
	4, 3, 2, 1, 0, 91, 92, 23,
	93, 94, 21, 201, 200, 199, 196, 195,
	194, 193, 198, 197, 75, 74, 15, 14,
};
static const unsigned int lcd1_data24_mux[] = {
	LCD1_D0_MARK, LCD1_D1_MARK, LCD1_D2_MARK, LCD1_D3_MARK,
	LCD1_D4_MARK, LCD1_D5_MARK, LCD1_D6_MARK, LCD1_D7_MARK,
	LCD1_D8_MARK, LCD1_D9_MARK, LCD1_D10_MARK, LCD1_D11_MARK,
	LCD1_D12_MARK, LCD1_D13_MARK, LCD1_D14_MARK, LCD1_D15_MARK,
	LCD1_D16_MARK, LCD1_D17_MARK, LCD1_D18_MARK, LCD1_D19_MARK,
	LCD1_D20_MARK, LCD1_D21_MARK, LCD1_D22_MARK, LCD1_D23_MARK,
};
static const unsigned int lcd1_display_pins[] = {
	/* DON, VCPWC, VEPWC */
	100, 5, 6,
};
static const unsigned int lcd1_display_mux[] = {
	LCD1_DON_MARK, LCD1_VCPWC_MARK, LCD1_VEPWC_MARK,
};
static const unsigned int lcd1_lclk_pins[] = {
	/* LCLK */
	40,
};
static const unsigned int lcd1_lclk_mux[] = {
	LCD1_LCLK_MARK,
};
static const unsigned int lcd1_sync_pins[] = {
	/* VSYN, HSYN, DCK, DISP */
	98, 97, 99, 12,
};
static const unsigned int lcd1_sync_mux[] = {
	LCD1_VSYN_MARK, LCD1_HSYN_MARK, LCD1_DCK_MARK, LCD1_DISP_MARK,
};
static const unsigned int lcd1_sys_pins[] = {
	/* CS, WR, RD, RS */
	97, 99, 13, 12,
};
static const unsigned int lcd1_sys_mux[] = {
	LCD1_CS_MARK, LCD1_WR_MARK, LCD1_RD_MARK, LCD1_RS_MARK,
};
/* - MMCIF ------------------------------------------------------------------ */
static const unsigned int mmc0_data1_0_pins[] = {
	/* D[0] */
	68,
};
static const unsigned int mmc0_data1_0_mux[] = {
	MMC0_D0_PORT68_MARK,
};
static const unsigned int mmc0_data4_0_pins[] = {
	/* D[0:3] */
	68, 69, 70, 71,
};
static const unsigned int mmc0_data4_0_mux[] = {
	MMC0_D0_PORT68_MARK, MMC0_D1_PORT69_MARK, MMC0_D2_PORT70_MARK, MMC0_D3_PORT71_MARK,
};
static const unsigned int mmc0_data8_0_pins[] = {
	/* D[0:7] */
	68, 69, 70, 71, 72, 73, 74, 75,
};
static const unsigned int mmc0_data8_0_mux[] = {
	MMC0_D0_PORT68_MARK, MMC0_D1_PORT69_MARK, MMC0_D2_PORT70_MARK, MMC0_D3_PORT71_MARK,
	MMC0_D4_PORT72_MARK, MMC0_D5_PORT73_MARK, MMC0_D6_PORT74_MARK, MMC0_D7_PORT75_MARK,
};
static const unsigned int mmc0_ctrl_0_pins[] = {
	/* CMD, CLK */
	67, 66,
};
static const unsigned int mmc0_ctrl_0_mux[] = {
	MMC0_CMD_PORT67_MARK, MMC0_CLK_PORT66_MARK,
};

static const unsigned int mmc0_data1_1_pins[] = {
	/* D[0] */
	149,
};
static const unsigned int mmc0_data1_1_mux[] = {
	MMC1_D0_PORT149_MARK,
};
static const unsigned int mmc0_data4_1_pins[] = {
	/* D[0:3] */
	149, 148, 147, 146,
};
static const unsigned int mmc0_data4_1_mux[] = {
	MMC1_D0_PORT149_MARK, MMC1_D1_PORT148_MARK, MMC1_D2_PORT147_MARK, MMC1_D3_PORT146_MARK,
};
static const unsigned int mmc0_data8_1_pins[] = {
	/* D[0:7] */
	149, 148, 147, 146, 145, 144, 143, 142,
};
static const unsigned int mmc0_data8_1_mux[] = {
	MMC1_D0_PORT149_MARK, MMC1_D1_PORT148_MARK, MMC1_D2_PORT147_MARK, MMC1_D3_PORT146_MARK,
	MMC1_D4_PORT145_MARK, MMC1_D5_PORT144_MARK, MMC1_D6_PORT143_MARK, MMC1_D7_PORT142_MARK,
};
static const unsigned int mmc0_ctrl_1_pins[] = {
	/* CMD, CLK */
	104, 103,
};
static const unsigned int mmc0_ctrl_1_mux[] = {
	MMC1_CMD_PORT104_MARK, MMC1_CLK_PORT103_MARK,
};
/* - SCIFA0 ----------------------------------------------------------------- */
static const unsigned int scifa0_data_pins[] = {
	/* RXD, TXD */
	197, 198,
};
static const unsigned int scifa0_data_mux[] = {
	SCIFA0_RXD_MARK, SCIFA0_TXD_MARK,
};
static const unsigned int scifa0_clk_pins[] = {
	/* SCK */
	188,
};
static const unsigned int scifa0_clk_mux[] = {
	SCIFA0_SCK_MARK,
};
static const unsigned int scifa0_ctrl_pins[] = {
	/* RTS, CTS */
	194, 193,
};
static const unsigned int scifa0_ctrl_mux[] = {
	SCIFA0_RTS_MARK, SCIFA0_CTS_MARK,
};
/* - SCIFA1 ----------------------------------------------------------------- */
static const unsigned int scifa1_data_pins[] = {
	/* RXD, TXD */
	195, 196,
};
static const unsigned int scifa1_data_mux[] = {
	SCIFA1_RXD_MARK, SCIFA1_TXD_MARK,
};
static const unsigned int scifa1_clk_pins[] = {
	/* SCK */
	185,
};
static const unsigned int scifa1_clk_mux[] = {
	SCIFA1_SCK_MARK,
};
static const unsigned int scifa1_ctrl_pins[] = {
	/* RTS, CTS */
	23, 21,
};
static const unsigned int scifa1_ctrl_mux[] = {
	SCIFA1_RTS_MARK, SCIFA1_CTS_MARK,
};
/* - SCIFA2 ----------------------------------------------------------------- */
static const unsigned int scifa2_data_pins[] = {
	/* RXD, TXD */
	200, 201,
};
static const unsigned int scifa2_data_mux[] = {
	SCIFA2_RXD_MARK, SCIFA2_TXD_MARK,
};
static const unsigned int scifa2_clk_0_pins[] = {
	/* SCK */
	22,
};
static const unsigned int scifa2_clk_0_mux[] = {
	SCIFA2_SCK_PORT22_MARK,
};
static const unsigned int scifa2_clk_1_pins[] = {
	/* SCK */
	199,
};
static const unsigned int scifa2_clk_1_mux[] = {
	SCIFA2_SCK_PORT199_MARK,
};
static const unsigned int scifa2_ctrl_pins[] = {
	/* RTS, CTS */
	96, 95,
};
static const unsigned int scifa2_ctrl_mux[] = {
	SCIFA2_RTS_MARK, SCIFA2_CTS_MARK,
};
/* - SCIFA3 ----------------------------------------------------------------- */
static const unsigned int scifa3_data_0_pins[] = {
	/* RXD, TXD */
	174, 175,
};
static const unsigned int scifa3_data_0_mux[] = {
	SCIFA3_RXD_PORT174_MARK, SCIFA3_TXD_PORT175_MARK,
};
static const unsigned int scifa3_clk_0_pins[] = {
	/* SCK */
	116,
};
static const unsigned int scifa3_clk_0_mux[] = {
	SCIFA3_SCK_PORT116_MARK,
};
static const unsigned int scifa3_ctrl_0_pins[] = {
	/* RTS, CTS */
	105, 117,
};
static const unsigned int scifa3_ctrl_0_mux[] = {
	SCIFA3_RTS_PORT105_MARK, SCIFA3_CTS_PORT117_MARK,
};
static const unsigned int scifa3_data_1_pins[] = {
	/* RXD, TXD */
	159, 160,
};
static const unsigned int scifa3_data_1_mux[] = {
	SCIFA3_RXD_PORT159_MARK, SCIFA3_TXD_PORT160_MARK,
};
static const unsigned int scifa3_clk_1_pins[] = {
	/* SCK */
	158,
};
static const unsigned int scifa3_clk_1_mux[] = {
	SCIFA3_SCK_PORT158_MARK,
};
static const unsigned int scifa3_ctrl_1_pins[] = {
	/* RTS, CTS */
	161, 162,
};
static const unsigned int scifa3_ctrl_1_mux[] = {
	SCIFA3_RTS_PORT161_MARK, SCIFA3_CTS_PORT162_MARK,
};
/* - SCIFA4 ----------------------------------------------------------------- */
static const unsigned int scifa4_data_0_pins[] = {
	/* RXD, TXD */
	12, 13,
};
static const unsigned int scifa4_data_0_mux[] = {
	SCIFA4_RXD_PORT12_MARK, SCIFA4_TXD_PORT13_MARK,
};
static const unsigned int scifa4_data_1_pins[] = {
	/* RXD, TXD */
	204, 203,
};
static const unsigned int scifa4_data_1_mux[] = {
	SCIFA4_RXD_PORT204_MARK, SCIFA4_TXD_PORT203_MARK,
};
static const unsigned int scifa4_data_2_pins[] = {
	/* RXD, TXD */
	94, 93,
};
static const unsigned int scifa4_data_2_mux[] = {
	SCIFA4_RXD_PORT94_MARK, SCIFA4_TXD_PORT93_MARK,
};
static const unsigned int scifa4_clk_0_pins[] = {
	/* SCK */
	21,
};
static const unsigned int scifa4_clk_0_mux[] = {
	SCIFA4_SCK_PORT21_MARK,
};
static const unsigned int scifa4_clk_1_pins[] = {
	/* SCK */
	205,
};
static const unsigned int scifa4_clk_1_mux[] = {
	SCIFA4_SCK_PORT205_MARK,
};
/* - SCIFA5 ----------------------------------------------------------------- */
static const unsigned int scifa5_data_0_pins[] = {
	/* RXD, TXD */
	10, 20,
};
static const unsigned int scifa5_data_0_mux[] = {
	SCIFA5_RXD_PORT10_MARK, SCIFA5_TXD_PORT20_MARK,
};
static const unsigned int scifa5_data_1_pins[] = {
	/* RXD, TXD */
	207, 208,
};
static const unsigned int scifa5_data_1_mux[] = {
	SCIFA5_RXD_PORT207_MARK, SCIFA5_TXD_PORT208_MARK,
};
static const unsigned int scifa5_data_2_pins[] = {
	/* RXD, TXD */
	92, 91,
};
static const unsigned int scifa5_data_2_mux[] = {
	SCIFA5_RXD_PORT92_MARK, SCIFA5_TXD_PORT91_MARK,
};
static const unsigned int scifa5_clk_0_pins[] = {
	/* SCK */
	23,
};
static const unsigned int scifa5_clk_0_mux[] = {
	SCIFA5_SCK_PORT23_MARK,
};
static const unsigned int scifa5_clk_1_pins[] = {
	/* SCK */
	206,
};
static const unsigned int scifa5_clk_1_mux[] = {
	SCIFA5_SCK_PORT206_MARK,
};
/* - SCIFA6 ----------------------------------------------------------------- */
static const unsigned int scifa6_data_pins[] = {
	/* RXD, TXD */
	25, 26,
};
static const unsigned int scifa6_data_mux[] = {
	SCIFA6_RXD_MARK, SCIFA6_TXD_MARK,
};
static const unsigned int scifa6_clk_pins[] = {
	/* SCK */
	24,
};
static const unsigned int scifa6_clk_mux[] = {
	SCIFA6_SCK_MARK,
};
/* - SCIFA7 ----------------------------------------------------------------- */
static const unsigned int scifa7_data_pins[] = {
	/* RXD, TXD */
	0, 1,
};
static const unsigned int scifa7_data_mux[] = {
	SCIFA7_RXD_MARK, SCIFA7_TXD_MARK,
};
/* - SCIFB ------------------------------------------------------------------ */
static const unsigned int scifb_data_0_pins[] = {
	/* RXD, TXD */
	191, 192,
};
static const unsigned int scifb_data_0_mux[] = {
	SCIFB_RXD_PORT191_MARK, SCIFB_TXD_PORT192_MARK,
};
static const unsigned int scifb_clk_0_pins[] = {
	/* SCK */
	190,
};
static const unsigned int scifb_clk_0_mux[] = {
	SCIFB_SCK_PORT190_MARK,
};
static const unsigned int scifb_ctrl_0_pins[] = {
	/* RTS, CTS */
	186, 187,
};
static const unsigned int scifb_ctrl_0_mux[] = {
	SCIFB_RTS_PORT186_MARK, SCIFB_CTS_PORT187_MARK,
};
static const unsigned int scifb_data_1_pins[] = {
	/* RXD, TXD */
	3, 4,
};
static const unsigned int scifb_data_1_mux[] = {
	SCIFB_RXD_PORT3_MARK, SCIFB_TXD_PORT4_MARK,
};
static const unsigned int scifb_clk_1_pins[] = {
	/* SCK */
	2,
};
static const unsigned int scifb_clk_1_mux[] = {
	SCIFB_SCK_PORT2_MARK,
};
static const unsigned int scifb_ctrl_1_pins[] = {
	/* RTS, CTS */
	172, 173,
};
static const unsigned int scifb_ctrl_1_mux[] = {
	SCIFB_RTS_PORT172_MARK, SCIFB_CTS_PORT173_MARK,
};
/* - SDHI0 ------------------------------------------------------------------ */
static const unsigned int sdhi0_data1_pins[] = {
	/* D0 */
	77,
};
static const unsigned int sdhi0_data1_mux[] = {
	SDHI0_D0_MARK,
};
static const unsigned int sdhi0_data4_pins[] = {
	/* D[0:3] */
	77, 78, 79, 80,
};
static const unsigned int sdhi0_data4_mux[] = {
	SDHI0_D0_MARK, SDHI0_D1_MARK, SDHI0_D2_MARK, SDHI0_D3_MARK,
};
static const unsigned int sdhi0_ctrl_pins[] = {
	/* CMD, CLK */
	76, 82,
};
static const unsigned int sdhi0_ctrl_mux[] = {
	SDHI0_CMD_MARK, SDHI0_CLK_MARK,
};
static const unsigned int sdhi0_cd_pins[] = {
	/* CD */
	81,
};
static const unsigned int sdhi0_cd_mux[] = {
	SDHI0_CD_MARK,
};
static const unsigned int sdhi0_wp_pins[] = {
	/* WP */
	83,
};
static const unsigned int sdhi0_wp_mux[] = {
	SDHI0_WP_MARK,
};
/* - SDHI1 ------------------------------------------------------------------ */
static const unsigned int sdhi1_data1_pins[] = {
	/* D0 */
	68,
};
static const unsigned int sdhi1_data1_mux[] = {
	SDHI1_D0_MARK,
};
static const unsigned int sdhi1_data4_pins[] = {
	/* D[0:3] */
	68, 69, 70, 71,
};
static const unsigned int sdhi1_data4_mux[] = {
	SDHI1_D0_MARK, SDHI1_D1_MARK, SDHI1_D2_MARK, SDHI1_D3_MARK,
};
static const unsigned int sdhi1_ctrl_pins[] = {
	/* CMD, CLK */
	67, 66,
};
static const unsigned int sdhi1_ctrl_mux[] = {
	SDHI1_CMD_MARK, SDHI1_CLK_MARK,
};
static const unsigned int sdhi1_cd_pins[] = {
	/* CD */
	72,
};
static const unsigned int sdhi1_cd_mux[] = {
	SDHI1_CD_MARK,
};
static const unsigned int sdhi1_wp_pins[] = {
	/* WP */
	73,
};
static const unsigned int sdhi1_wp_mux[] = {
	SDHI1_WP_MARK,
};
/* - SDHI2 ------------------------------------------------------------------ */
static const unsigned int sdhi2_data1_pins[] = {
	/* D0 */
	205,
};
static const unsigned int sdhi2_data1_mux[] = {
	SDHI2_D0_MARK,
};
static const unsigned int sdhi2_data4_pins[] = {
	/* D[0:3] */
	205, 206, 207, 208,
};
static const unsigned int sdhi2_data4_mux[] = {
	SDHI2_D0_MARK, SDHI2_D1_MARK, SDHI2_D2_MARK, SDHI2_D3_MARK,
};
static const unsigned int sdhi2_ctrl_pins[] = {
	/* CMD, CLK */
	204, 203,
};
static const unsigned int sdhi2_ctrl_mux[] = {
	SDHI2_CMD_MARK, SDHI2_CLK_MARK,
};
static const unsigned int sdhi2_cd_0_pins[] = {
	/* CD */
	202,
};
static const unsigned int sdhi2_cd_0_mux[] = {
	SDHI2_CD_PORT202_MARK,
};
static const unsigned int sdhi2_wp_0_pins[] = {
	/* WP */
	177,
};
static const unsigned int sdhi2_wp_0_mux[] = {
	SDHI2_WP_PORT177_MARK,
};
static const unsigned int sdhi2_cd_1_pins[] = {
	/* CD */
	24,
};
static const unsigned int sdhi2_cd_1_mux[] = {
	SDHI2_CD_PORT24_MARK,
};
static const unsigned int sdhi2_wp_1_pins[] = {
	/* WP */
	25,
};
static const unsigned int sdhi2_wp_1_mux[] = {
	SDHI2_WP_PORT25_MARK,
};
/* - TPU0 ------------------------------------------------------------------- */
static const unsigned int tpu0_to0_pins[] = {
	/* TO */
	23,
};
static const unsigned int tpu0_to0_mux[] = {
	TPU0TO0_MARK,
};
static const unsigned int tpu0_to1_pins[] = {
	/* TO */
	21,
};
static const unsigned int tpu0_to1_mux[] = {
	TPU0TO1_MARK,
};
static const unsigned int tpu0_to2_0_pins[] = {
	/* TO */
	66,
};
static const unsigned int tpu0_to2_0_mux[] = {
	TPU0TO2_PORT66_MARK,
};
static const unsigned int tpu0_to2_1_pins[] = {
	/* TO */
	202,
};
static const unsigned int tpu0_to2_1_mux[] = {
	TPU0TO2_PORT202_MARK,
};
static const unsigned int tpu0_to3_pins[] = {
	/* TO */
	180,
};
static const unsigned int tpu0_to3_mux[] = {
	TPU0TO3_MARK,
};

static const struct sh_pfc_pin_group pinmux_groups[] = {
	SH_PFC_PIN_GROUP(bsc_data8),
	SH_PFC_PIN_GROUP(bsc_data16),
	SH_PFC_PIN_GROUP(bsc_data32),
	SH_PFC_PIN_GROUP(bsc_cs0),
	SH_PFC_PIN_GROUP(bsc_cs2),
	SH_PFC_PIN_GROUP(bsc_cs4),
	SH_PFC_PIN_GROUP(bsc_cs5a_0),
	SH_PFC_PIN_GROUP(bsc_cs5a_1),
	SH_PFC_PIN_GROUP(bsc_cs5b),
	SH_PFC_PIN_GROUP(bsc_cs6a),
	SH_PFC_PIN_GROUP(bsc_rd_we8),
	SH_PFC_PIN_GROUP(bsc_rd_we16),
	SH_PFC_PIN_GROUP(bsc_rd_we32),
	SH_PFC_PIN_GROUP(bsc_bs),
	SH_PFC_PIN_GROUP(bsc_rdwr),
	SH_PFC_PIN_GROUP(ceu0_data_0_7),
	SH_PFC_PIN_GROUP(ceu0_data_8_15_0),
	SH_PFC_PIN_GROUP(ceu0_data_8_15_1),
	SH_PFC_PIN_GROUP(ceu0_clk_0),
	SH_PFC_PIN_GROUP(ceu0_clk_1),
	SH_PFC_PIN_GROUP(ceu0_clk_2),
	SH_PFC_PIN_GROUP(ceu0_sync),
	SH_PFC_PIN_GROUP(ceu0_field),
	SH_PFC_PIN_GROUP(ceu1_data),
	SH_PFC_PIN_GROUP(ceu1_clk),
	SH_PFC_PIN_GROUP(ceu1_sync),
	SH_PFC_PIN_GROUP(ceu1_field),
	SH_PFC_PIN_GROUP(fsia_mclk_in),
	SH_PFC_PIN_GROUP(fsia_mclk_out),
	SH_PFC_PIN_GROUP(fsia_sclk_in),
	SH_PFC_PIN_GROUP(fsia_sclk_out),
	SH_PFC_PIN_GROUP(fsia_data_in_0),
	SH_PFC_PIN_GROUP(fsia_data_in_1),
	SH_PFC_PIN_GROUP(fsia_data_out_0),
	SH_PFC_PIN_GROUP(fsia_data_out_1),
	SH_PFC_PIN_GROUP(fsia_data_out_2),
	SH_PFC_PIN_GROUP(fsia_spdif_0),
	SH_PFC_PIN_GROUP(fsia_spdif_1),
	SH_PFC_PIN_GROUP(fsib_mclk_in),
	SH_PFC_PIN_GROUP(gether_rmii),
	SH_PFC_PIN_GROUP(gether_mii),
	SH_PFC_PIN_GROUP(gether_gmii),
	SH_PFC_PIN_GROUP(gether_int),
	SH_PFC_PIN_GROUP(gether_link),
	SH_PFC_PIN_GROUP(gether_wol),
	SH_PFC_PIN_GROUP(hdmi),
	SH_PFC_PIN_GROUP(intc_irq0_0),
	SH_PFC_PIN_GROUP(intc_irq0_1),
	SH_PFC_PIN_GROUP(intc_irq1),
	SH_PFC_PIN_GROUP(intc_irq2_0),
	SH_PFC_PIN_GROUP(intc_irq2_1),
	SH_PFC_PIN_GROUP(intc_irq3_0),
	SH_PFC_PIN_GROUP(intc_irq3_1),
	SH_PFC_PIN_GROUP(intc_irq4_0),
	SH_PFC_PIN_GROUP(intc_irq4_1),
	SH_PFC_PIN_GROUP(intc_irq5_0),
	SH_PFC_PIN_GROUP(intc_irq5_1),
	SH_PFC_PIN_GROUP(intc_irq6_0),
	SH_PFC_PIN_GROUP(intc_irq6_1),
	SH_PFC_PIN_GROUP(intc_irq7_0),
	SH_PFC_PIN_GROUP(intc_irq7_1),
	SH_PFC_PIN_GROUP(intc_irq8),
	SH_PFC_PIN_GROUP(intc_irq9_0),
	SH_PFC_PIN_GROUP(intc_irq9_1),
	SH_PFC_PIN_GROUP(intc_irq10),
	SH_PFC_PIN_GROUP(intc_irq11),
	SH_PFC_PIN_GROUP(intc_irq12_0),
	SH_PFC_PIN_GROUP(intc_irq12_1),
	SH_PFC_PIN_GROUP(intc_irq13_0),
	SH_PFC_PIN_GROUP(intc_irq13_1),
	SH_PFC_PIN_GROUP(intc_irq14_0),
	SH_PFC_PIN_GROUP(intc_irq14_1),
	SH_PFC_PIN_GROUP(intc_irq15_0),
	SH_PFC_PIN_GROUP(intc_irq15_1),
	SH_PFC_PIN_GROUP(intc_irq16_0),
	SH_PFC_PIN_GROUP(intc_irq16_1),
	SH_PFC_PIN_GROUP(intc_irq17),
	SH_PFC_PIN_GROUP(intc_irq18),
	SH_PFC_PIN_GROUP(intc_irq19),
	SH_PFC_PIN_GROUP(intc_irq20),
	SH_PFC_PIN_GROUP(intc_irq21),
	SH_PFC_PIN_GROUP(intc_irq22),
	SH_PFC_PIN_GROUP(intc_irq23),
	SH_PFC_PIN_GROUP(intc_irq24),
	SH_PFC_PIN_GROUP(intc_irq25),
	SH_PFC_PIN_GROUP(intc_irq26_0),
	SH_PFC_PIN_GROUP(intc_irq26_1),
	SH_PFC_PIN_GROUP(intc_irq27_0),
	SH_PFC_PIN_GROUP(intc_irq27_1),
	SH_PFC_PIN_GROUP(intc_irq28_0),
	SH_PFC_PIN_GROUP(intc_irq28_1),
	SH_PFC_PIN_GROUP(intc_irq29_0),
	SH_PFC_PIN_GROUP(intc_irq29_1),
	SH_PFC_PIN_GROUP(intc_irq30_0),
	SH_PFC_PIN_GROUP(intc_irq30_1),
	SH_PFC_PIN_GROUP(intc_irq31_0),
	SH_PFC_PIN_GROUP(intc_irq31_1),
	SH_PFC_PIN_GROUP(lcd0_data8),
	SH_PFC_PIN_GROUP(lcd0_data9),
	SH_PFC_PIN_GROUP(lcd0_data12),
	SH_PFC_PIN_GROUP(lcd0_data16),
	SH_PFC_PIN_GROUP(lcd0_data18),
	SH_PFC_PIN_GROUP(lcd0_data24_0),
	SH_PFC_PIN_GROUP(lcd0_data24_1),
	SH_PFC_PIN_GROUP(lcd0_display),
	SH_PFC_PIN_GROUP(lcd0_lclk_0),
	SH_PFC_PIN_GROUP(lcd0_lclk_1),
	SH_PFC_PIN_GROUP(lcd0_sync),
	SH_PFC_PIN_GROUP(lcd0_sys),
	SH_PFC_PIN_GROUP(lcd1_data8),
	SH_PFC_PIN_GROUP(lcd1_data9),
	SH_PFC_PIN_GROUP(lcd1_data12),
	SH_PFC_PIN_GROUP(lcd1_data16),
	SH_PFC_PIN_GROUP(lcd1_data18),
	SH_PFC_PIN_GROUP(lcd1_data24),
	SH_PFC_PIN_GROUP(lcd1_display),
	SH_PFC_PIN_GROUP(lcd1_lclk),
	SH_PFC_PIN_GROUP(lcd1_sync),
	SH_PFC_PIN_GROUP(lcd1_sys),
	SH_PFC_PIN_GROUP(mmc0_data1_0),
	SH_PFC_PIN_GROUP(mmc0_data4_0),
	SH_PFC_PIN_GROUP(mmc0_data8_0),
	SH_PFC_PIN_GROUP(mmc0_ctrl_0),
	SH_PFC_PIN_GROUP(mmc0_data1_1),
	SH_PFC_PIN_GROUP(mmc0_data4_1),
	SH_PFC_PIN_GROUP(mmc0_data8_1),
	SH_PFC_PIN_GROUP(mmc0_ctrl_1),
	SH_PFC_PIN_GROUP(scifa0_data),
	SH_PFC_PIN_GROUP(scifa0_clk),
	SH_PFC_PIN_GROUP(scifa0_ctrl),
	SH_PFC_PIN_GROUP(scifa1_data),
	SH_PFC_PIN_GROUP(scifa1_clk),
	SH_PFC_PIN_GROUP(scifa1_ctrl),
	SH_PFC_PIN_GROUP(scifa2_data),
	SH_PFC_PIN_GROUP(scifa2_clk_0),
	SH_PFC_PIN_GROUP(scifa2_clk_1),
	SH_PFC_PIN_GROUP(scifa2_ctrl),
	SH_PFC_PIN_GROUP(scifa3_data_0),
	SH_PFC_PIN_GROUP(scifa3_clk_0),
	SH_PFC_PIN_GROUP(scifa3_ctrl_0),
	SH_PFC_PIN_GROUP(scifa3_data_1),
	SH_PFC_PIN_GROUP(scifa3_clk_1),
	SH_PFC_PIN_GROUP(scifa3_ctrl_1),
	SH_PFC_PIN_GROUP(scifa4_data_0),
	SH_PFC_PIN_GROUP(scifa4_data_1),
	SH_PFC_PIN_GROUP(scifa4_data_2),
	SH_PFC_PIN_GROUP(scifa4_clk_0),
	SH_PFC_PIN_GROUP(scifa4_clk_1),
	SH_PFC_PIN_GROUP(scifa5_data_0),
	SH_PFC_PIN_GROUP(scifa5_data_1),
	SH_PFC_PIN_GROUP(scifa5_data_2),
	SH_PFC_PIN_GROUP(scifa5_clk_0),
	SH_PFC_PIN_GROUP(scifa5_clk_1),
	SH_PFC_PIN_GROUP(scifa6_data),
	SH_PFC_PIN_GROUP(scifa6_clk),
	SH_PFC_PIN_GROUP(scifa7_data),
	SH_PFC_PIN_GROUP(scifb_data_0),
	SH_PFC_PIN_GROUP(scifb_clk_0),
	SH_PFC_PIN_GROUP(scifb_ctrl_0),
	SH_PFC_PIN_GROUP(scifb_data_1),
	SH_PFC_PIN_GROUP(scifb_clk_1),
	SH_PFC_PIN_GROUP(scifb_ctrl_1),
	SH_PFC_PIN_GROUP(sdhi0_data1),
	SH_PFC_PIN_GROUP(sdhi0_data4),
	SH_PFC_PIN_GROUP(sdhi0_ctrl),
	SH_PFC_PIN_GROUP(sdhi0_cd),
	SH_PFC_PIN_GROUP(sdhi0_wp),
	SH_PFC_PIN_GROUP(sdhi1_data1),
	SH_PFC_PIN_GROUP(sdhi1_data4),
	SH_PFC_PIN_GROUP(sdhi1_ctrl),
	SH_PFC_PIN_GROUP(sdhi1_cd),
	SH_PFC_PIN_GROUP(sdhi1_wp),
	SH_PFC_PIN_GROUP(sdhi2_data1),
	SH_PFC_PIN_GROUP(sdhi2_data4),
	SH_PFC_PIN_GROUP(sdhi2_ctrl),
	SH_PFC_PIN_GROUP(sdhi2_cd_0),
	SH_PFC_PIN_GROUP(sdhi2_wp_0),
	SH_PFC_PIN_GROUP(sdhi2_cd_1),
	SH_PFC_PIN_GROUP(sdhi2_wp_1),
	SH_PFC_PIN_GROUP(tpu0_to0),
	SH_PFC_PIN_GROUP(tpu0_to1),
	SH_PFC_PIN_GROUP(tpu0_to2_0),
	SH_PFC_PIN_GROUP(tpu0_to2_1),
	SH_PFC_PIN_GROUP(tpu0_to3),
};

static const char * const bsc_groups[] = {
	"bsc_data8",
	"bsc_data16",
	"bsc_data32",
	"bsc_cs0",
	"bsc_cs2",
	"bsc_cs4",
	"bsc_cs5a_0",
	"bsc_cs5a_1",
	"bsc_cs5b",
	"bsc_cs6a",
	"bsc_rd_we8",
	"bsc_rd_we16",
	"bsc_rd_we32",
	"bsc_bs",
	"bsc_rdwr",
};

static const char * const ceu0_groups[] = {
	"ceu0_data_0_7",
	"ceu0_data_8_15_0",
	"ceu0_data_8_15_1",
	"ceu0_clk_0",
	"ceu0_clk_1",
	"ceu0_clk_2",
	"ceu0_sync",
	"ceu0_field",
};

static const char * const ceu1_groups[] = {
	"ceu1_data",
	"ceu1_clk",
	"ceu1_sync",
	"ceu1_field",
};

static const char * const fsia_groups[] = {
	"fsia_mclk_in",
	"fsia_mclk_out",
	"fsia_sclk_in",
	"fsia_sclk_out",
	"fsia_data_in_0",
	"fsia_data_in_1",
	"fsia_data_out_0",
	"fsia_data_out_1",
	"fsia_data_out_2",
	"fsia_spdif_0",
	"fsia_spdif_1",
};

static const char * const fsib_groups[] = {
	"fsib_mclk_in",
};

static const char * const gether_groups[] = {
	"gether_rmii",
	"gether_mii",
	"gether_gmii",
	"gether_int",
	"gether_link",
	"gether_wol",
};

static const char * const hdmi_groups[] = {
	"hdmi",
};

static const char * const intc_groups[] = {
	"intc_irq0_0",
	"intc_irq0_1",
	"intc_irq1",
	"intc_irq2_0",
	"intc_irq2_1",
	"intc_irq3_0",
	"intc_irq3_1",
	"intc_irq4_0",
	"intc_irq4_1",
	"intc_irq5_0",
	"intc_irq5_1",
	"intc_irq6_0",
	"intc_irq6_1",
	"intc_irq7_0",
	"intc_irq7_1",
	"intc_irq8",
	"intc_irq9_0",
	"intc_irq9_1",
	"intc_irq10",
	"intc_irq11",
	"intc_irq12_0",
	"intc_irq12_1",
	"intc_irq13_0",
	"intc_irq13_1",
	"intc_irq14_0",
	"intc_irq14_1",
	"intc_irq15_0",
	"intc_irq15_1",
	"intc_irq16_0",
	"intc_irq16_1",
	"intc_irq17",
	"intc_irq18",
	"intc_irq19",
	"intc_irq20",
	"intc_irq21",
	"intc_irq22",
	"intc_irq23",
	"intc_irq24",
	"intc_irq25",
	"intc_irq26_0",
	"intc_irq26_1",
	"intc_irq27_0",
	"intc_irq27_1",
	"intc_irq28_0",
	"intc_irq28_1",
	"intc_irq29_0",
	"intc_irq29_1",
	"intc_irq30_0",
	"intc_irq30_1",
	"intc_irq31_0",
	"intc_irq31_1",
};

static const char * const lcd0_groups[] = {
	"lcd0_data8",
	"lcd0_data9",
	"lcd0_data12",
	"lcd0_data16",
	"lcd0_data18",
	"lcd0_data24_0",
	"lcd0_data24_1",
	"lcd0_display",
	"lcd0_lclk_0",
	"lcd0_lclk_1",
	"lcd0_sync",
	"lcd0_sys",
};

static const char * const lcd1_groups[] = {
	"lcd1_data8",
	"lcd1_data9",
	"lcd1_data12",
	"lcd1_data16",
	"lcd1_data18",
	"lcd1_data24",
	"lcd1_display",
	"lcd1_lclk",
	"lcd1_sync",
	"lcd1_sys",
};

static const char * const mmc0_groups[] = {
	"mmc0_data1_0",
	"mmc0_data4_0",
	"mmc0_data8_0",
	"mmc0_ctrl_0",
	"mmc0_data1_1",
	"mmc0_data4_1",
	"mmc0_data8_1",
	"mmc0_ctrl_1",
};

static const char * const scifa0_groups[] = {
	"scifa0_data",
	"scifa0_clk",
	"scifa0_ctrl",
};

static const char * const scifa1_groups[] = {
	"scifa1_data",
	"scifa1_clk",
	"scifa1_ctrl",
};

static const char * const scifa2_groups[] = {
	"scifa2_data",
	"scifa2_clk_0",
	"scifa2_clk_1",
	"scifa2_ctrl",
};

static const char * const scifa3_groups[] = {
	"scifa3_data_0",
	"scifa3_clk_0",
	"scifa3_ctrl_0",
	"scifa3_data_1",
	"scifa3_clk_1",
	"scifa3_ctrl_1",
};

static const char * const scifa4_groups[] = {
	"scifa4_data_0",
	"scifa4_data_1",
	"scifa4_data_2",
	"scifa4_clk_0",
	"scifa4_clk_1",
};

static const char * const scifa5_groups[] = {
	"scifa5_data_0",
	"scifa5_data_1",
	"scifa5_data_2",
	"scifa5_clk_0",
	"scifa5_clk_1",
};

static const char * const scifa6_groups[] = {
	"scifa6_data",
	"scifa6_clk",
};

static const char * const scifa7_groups[] = {
	"scifa7_data",
};

static const char * const scifb_groups[] = {
	"scifb_data_0",
	"scifb_clk_0",
	"scifb_ctrl_0",
	"scifb_data_1",
	"scifb_clk_1",
	"scifb_ctrl_1",
};

static const char * const sdhi0_groups[] = {
	"sdhi0_data1",
	"sdhi0_data4",
	"sdhi0_ctrl",
	"sdhi0_cd",
	"sdhi0_wp",
};

static const char * const sdhi1_groups[] = {
	"sdhi1_data1",
	"sdhi1_data4",
	"sdhi1_ctrl",
	"sdhi1_cd",
	"sdhi1_wp",
};

static const char * const sdhi2_groups[] = {
	"sdhi2_data1",
	"sdhi2_data4",
	"sdhi2_ctrl",
	"sdhi2_cd_0",
	"sdhi2_wp_0",
	"sdhi2_cd_1",
	"sdhi2_wp_1",
};

static const char * const tpu0_groups[] = {
	"tpu0_to0",
	"tpu0_to1",
	"tpu0_to2_0",
	"tpu0_to2_1",
	"tpu0_to3",
};

static const struct sh_pfc_function pinmux_functions[] = {
	SH_PFC_FUNCTION(bsc),
	SH_PFC_FUNCTION(ceu0),
	SH_PFC_FUNCTION(ceu1),
	SH_PFC_FUNCTION(fsia),
	SH_PFC_FUNCTION(fsib),
	SH_PFC_FUNCTION(gether),
	SH_PFC_FUNCTION(hdmi),
	SH_PFC_FUNCTION(intc),
	SH_PFC_FUNCTION(lcd0),
	SH_PFC_FUNCTION(lcd1),
	SH_PFC_FUNCTION(mmc0),
	SH_PFC_FUNCTION(scifa0),
	SH_PFC_FUNCTION(scifa1),
	SH_PFC_FUNCTION(scifa2),
	SH_PFC_FUNCTION(scifa3),
	SH_PFC_FUNCTION(scifa4),
	SH_PFC_FUNCTION(scifa5),
	SH_PFC_FUNCTION(scifa6),
	SH_PFC_FUNCTION(scifa7),
	SH_PFC_FUNCTION(scifb),
	SH_PFC_FUNCTION(sdhi0),
	SH_PFC_FUNCTION(sdhi1),
	SH_PFC_FUNCTION(sdhi2),
	SH_PFC_FUNCTION(tpu0),
};

static const struct pinmux_cfg_reg pinmux_config_regs[] = {
	PORTCR(0,	0xe6050000), /* PORT0CR */
	PORTCR(1,	0xe6050001), /* PORT1CR */
	PORTCR(2,	0xe6050002), /* PORT2CR */
	PORTCR(3,	0xe6050003), /* PORT3CR */
	PORTCR(4,	0xe6050004), /* PORT4CR */
	PORTCR(5,	0xe6050005), /* PORT5CR */
	PORTCR(6,	0xe6050006), /* PORT6CR */
	PORTCR(7,	0xe6050007), /* PORT7CR */
	PORTCR(8,	0xe6050008), /* PORT8CR */
	PORTCR(9,	0xe6050009), /* PORT9CR */
	PORTCR(10,	0xe605000a), /* PORT10CR */
	PORTCR(11,	0xe605000b), /* PORT11CR */
	PORTCR(12,	0xe605000c), /* PORT12CR */
	PORTCR(13,	0xe605000d), /* PORT13CR */
	PORTCR(14,	0xe605000e), /* PORT14CR */
	PORTCR(15,	0xe605000f), /* PORT15CR */
	PORTCR(16,	0xe6050010), /* PORT16CR */
	PORTCR(17,	0xe6050011), /* PORT17CR */
	PORTCR(18,	0xe6050012), /* PORT18CR */
	PORTCR(19,	0xe6050013), /* PORT19CR */
	PORTCR(20,	0xe6050014), /* PORT20CR */
	PORTCR(21,	0xe6050015), /* PORT21CR */
	PORTCR(22,	0xe6050016), /* PORT22CR */
	PORTCR(23,	0xe6050017), /* PORT23CR */
	PORTCR(24,	0xe6050018), /* PORT24CR */
	PORTCR(25,	0xe6050019), /* PORT25CR */
	PORTCR(26,	0xe605001a), /* PORT26CR */
	PORTCR(27,	0xe605001b), /* PORT27CR */
	PORTCR(28,	0xe605001c), /* PORT28CR */
	PORTCR(29,	0xe605001d), /* PORT29CR */
	PORTCR(30,	0xe605001e), /* PORT30CR */
	PORTCR(31,	0xe605001f), /* PORT31CR */
	PORTCR(32,	0xe6050020), /* PORT32CR */
	PORTCR(33,	0xe6050021), /* PORT33CR */
	PORTCR(34,	0xe6050022), /* PORT34CR */
	PORTCR(35,	0xe6050023), /* PORT35CR */
	PORTCR(36,	0xe6050024), /* PORT36CR */
	PORTCR(37,	0xe6050025), /* PORT37CR */
	PORTCR(38,	0xe6050026), /* PORT38CR */
	PORTCR(39,	0xe6050027), /* PORT39CR */
	PORTCR(40,	0xe6050028), /* PORT40CR */
	PORTCR(41,	0xe6050029), /* PORT41CR */
	PORTCR(42,	0xe605002a), /* PORT42CR */
	PORTCR(43,	0xe605002b), /* PORT43CR */
	PORTCR(44,	0xe605002c), /* PORT44CR */
	PORTCR(45,	0xe605002d), /* PORT45CR */
	PORTCR(46,	0xe605002e), /* PORT46CR */
	PORTCR(47,	0xe605002f), /* PORT47CR */
	PORTCR(48,	0xe6050030), /* PORT48CR */
	PORTCR(49,	0xe6050031), /* PORT49CR */
	PORTCR(50,	0xe6050032), /* PORT50CR */
	PORTCR(51,	0xe6050033), /* PORT51CR */
	PORTCR(52,	0xe6050034), /* PORT52CR */
	PORTCR(53,	0xe6050035), /* PORT53CR */
	PORTCR(54,	0xe6050036), /* PORT54CR */
	PORTCR(55,	0xe6050037), /* PORT55CR */
	PORTCR(56,	0xe6050038), /* PORT56CR */
	PORTCR(57,	0xe6050039), /* PORT57CR */
	PORTCR(58,	0xe605003a), /* PORT58CR */
	PORTCR(59,	0xe605003b), /* PORT59CR */
	PORTCR(60,	0xe605003c), /* PORT60CR */
	PORTCR(61,	0xe605003d), /* PORT61CR */
	PORTCR(62,	0xe605003e), /* PORT62CR */
	PORTCR(63,	0xe605003f), /* PORT63CR */
	PORTCR(64,	0xe6050040), /* PORT64CR */
	PORTCR(65,	0xe6050041), /* PORT65CR */
	PORTCR(66,	0xe6050042), /* PORT66CR */
	PORTCR(67,	0xe6050043), /* PORT67CR */
	PORTCR(68,	0xe6050044), /* PORT68CR */
	PORTCR(69,	0xe6050045), /* PORT69CR */
	PORTCR(70,	0xe6050046), /* PORT70CR */
	PORTCR(71,	0xe6050047), /* PORT71CR */
	PORTCR(72,	0xe6050048), /* PORT72CR */
	PORTCR(73,	0xe6050049), /* PORT73CR */
	PORTCR(74,	0xe605004a), /* PORT74CR */
	PORTCR(75,	0xe605004b), /* PORT75CR */
	PORTCR(76,	0xe605004c), /* PORT76CR */
	PORTCR(77,	0xe605004d), /* PORT77CR */
	PORTCR(78,	0xe605004e), /* PORT78CR */
	PORTCR(79,	0xe605004f), /* PORT79CR */
	PORTCR(80,	0xe6050050), /* PORT80CR */
	PORTCR(81,	0xe6050051), /* PORT81CR */
	PORTCR(82,	0xe6050052), /* PORT82CR */
	PORTCR(83,	0xe6050053), /* PORT83CR */

	PORTCR(84,	0xe6051054), /* PORT84CR */
	PORTCR(85,	0xe6051055), /* PORT85CR */
	PORTCR(86,	0xe6051056), /* PORT86CR */
	PORTCR(87,	0xe6051057), /* PORT87CR */
	PORTCR(88,	0xe6051058), /* PORT88CR */
	PORTCR(89,	0xe6051059), /* PORT89CR */
	PORTCR(90,	0xe605105a), /* PORT90CR */
	PORTCR(91,	0xe605105b), /* PORT91CR */
	PORTCR(92,	0xe605105c), /* PORT92CR */
	PORTCR(93,	0xe605105d), /* PORT93CR */
	PORTCR(94,	0xe605105e), /* PORT94CR */
	PORTCR(95,	0xe605105f), /* PORT95CR */
	PORTCR(96,	0xe6051060), /* PORT96CR */
	PORTCR(97,	0xe6051061), /* PORT97CR */
	PORTCR(98,	0xe6051062), /* PORT98CR */
	PORTCR(99,	0xe6051063), /* PORT99CR */
	PORTCR(100,	0xe6051064), /* PORT100CR */
	PORTCR(101,	0xe6051065), /* PORT101CR */
	PORTCR(102,	0xe6051066), /* PORT102CR */
	PORTCR(103,	0xe6051067), /* PORT103CR */
	PORTCR(104,	0xe6051068), /* PORT104CR */
	PORTCR(105,	0xe6051069), /* PORT105CR */
	PORTCR(106,	0xe605106a), /* PORT106CR */
	PORTCR(107,	0xe605106b), /* PORT107CR */
	PORTCR(108,	0xe605106c), /* PORT108CR */
	PORTCR(109,	0xe605106d), /* PORT109CR */
	PORTCR(110,	0xe605106e), /* PORT110CR */
	PORTCR(111,	0xe605106f), /* PORT111CR */
	PORTCR(112,	0xe6051070), /* PORT112CR */
	PORTCR(113,	0xe6051071), /* PORT113CR */
	PORTCR(114,	0xe6051072), /* PORT114CR */

	PORTCR(115,	0xe6052073), /* PORT115CR */
	PORTCR(116,	0xe6052074), /* PORT116CR */
	PORTCR(117,	0xe6052075), /* PORT117CR */
	PORTCR(118,	0xe6052076), /* PORT118CR */
	PORTCR(119,	0xe6052077), /* PORT119CR */
	PORTCR(120,	0xe6052078), /* PORT120CR */
	PORTCR(121,	0xe6052079), /* PORT121CR */
	PORTCR(122,	0xe605207a), /* PORT122CR */
	PORTCR(123,	0xe605207b), /* PORT123CR */
	PORTCR(124,	0xe605207c), /* PORT124CR */
	PORTCR(125,	0xe605207d), /* PORT125CR */
	PORTCR(126,	0xe605207e), /* PORT126CR */
	PORTCR(127,	0xe605207f), /* PORT127CR */
	PORTCR(128,	0xe6052080), /* PORT128CR */
	PORTCR(129,	0xe6052081), /* PORT129CR */
	PORTCR(130,	0xe6052082), /* PORT130CR */
	PORTCR(131,	0xe6052083), /* PORT131CR */
	PORTCR(132,	0xe6052084), /* PORT132CR */
	PORTCR(133,	0xe6052085), /* PORT133CR */
	PORTCR(134,	0xe6052086), /* PORT134CR */
	PORTCR(135,	0xe6052087), /* PORT135CR */
	PORTCR(136,	0xe6052088), /* PORT136CR */
	PORTCR(137,	0xe6052089), /* PORT137CR */
	PORTCR(138,	0xe605208a), /* PORT138CR */
	PORTCR(139,	0xe605208b), /* PORT139CR */
	PORTCR(140,	0xe605208c), /* PORT140CR */
	PORTCR(141,	0xe605208d), /* PORT141CR */
	PORTCR(142,	0xe605208e), /* PORT142CR */
	PORTCR(143,	0xe605208f), /* PORT143CR */
	PORTCR(144,	0xe6052090), /* PORT144CR */
	PORTCR(145,	0xe6052091), /* PORT145CR */
	PORTCR(146,	0xe6052092), /* PORT146CR */
	PORTCR(147,	0xe6052093), /* PORT147CR */
	PORTCR(148,	0xe6052094), /* PORT148CR */
	PORTCR(149,	0xe6052095), /* PORT149CR */
	PORTCR(150,	0xe6052096), /* PORT150CR */
	PORTCR(151,	0xe6052097), /* PORT151CR */
	PORTCR(152,	0xe6052098), /* PORT152CR */
	PORTCR(153,	0xe6052099), /* PORT153CR */
	PORTCR(154,	0xe605209a), /* PORT154CR */
	PORTCR(155,	0xe605209b), /* PORT155CR */
	PORTCR(156,	0xe605209c), /* PORT156CR */
	PORTCR(157,	0xe605209d), /* PORT157CR */
	PORTCR(158,	0xe605209e), /* PORT158CR */
	PORTCR(159,	0xe605209f), /* PORT159CR */
	PORTCR(160,	0xe60520a0), /* PORT160CR */
	PORTCR(161,	0xe60520a1), /* PORT161CR */
	PORTCR(162,	0xe60520a2), /* PORT162CR */
	PORTCR(163,	0xe60520a3), /* PORT163CR */
	PORTCR(164,	0xe60520a4), /* PORT164CR */
	PORTCR(165,	0xe60520a5), /* PORT165CR */
	PORTCR(166,	0xe60520a6), /* PORT166CR */
	PORTCR(167,	0xe60520a7), /* PORT167CR */
	PORTCR(168,	0xe60520a8), /* PORT168CR */
	PORTCR(169,	0xe60520a9), /* PORT169CR */
	PORTCR(170,	0xe60520aa), /* PORT170CR */
	PORTCR(171,	0xe60520ab), /* PORT171CR */
	PORTCR(172,	0xe60520ac), /* PORT172CR */
	PORTCR(173,	0xe60520ad), /* PORT173CR */
	PORTCR(174,	0xe60520ae), /* PORT174CR */
	PORTCR(175,	0xe60520af), /* PORT175CR */
	PORTCR(176,	0xe60520b0), /* PORT176CR */
	PORTCR(177,	0xe60520b1), /* PORT177CR */
	PORTCR(178,	0xe60520b2), /* PORT178CR */
	PORTCR(179,	0xe60520b3), /* PORT179CR */
	PORTCR(180,	0xe60520b4), /* PORT180CR */
	PORTCR(181,	0xe60520b5), /* PORT181CR */
	PORTCR(182,	0xe60520b6), /* PORT182CR */
	PORTCR(183,	0xe60520b7), /* PORT183CR */
	PORTCR(184,	0xe60520b8), /* PORT184CR */
	PORTCR(185,	0xe60520b9), /* PORT185CR */
	PORTCR(186,	0xe60520ba), /* PORT186CR */
	PORTCR(187,	0xe60520bb), /* PORT187CR */
	PORTCR(188,	0xe60520bc), /* PORT188CR */
	PORTCR(189,	0xe60520bd), /* PORT189CR */
	PORTCR(190,	0xe60520be), /* PORT190CR */
	PORTCR(191,	0xe60520bf), /* PORT191CR */
	PORTCR(192,	0xe60520c0), /* PORT192CR */
	PORTCR(193,	0xe60520c1), /* PORT193CR */
	PORTCR(194,	0xe60520c2), /* PORT194CR */
	PORTCR(195,	0xe60520c3), /* PORT195CR */
	PORTCR(196,	0xe60520c4), /* PORT196CR */
	PORTCR(197,	0xe60520c5), /* PORT197CR */
	PORTCR(198,	0xe60520c6), /* PORT198CR */
	PORTCR(199,	0xe60520c7), /* PORT199CR */
	PORTCR(200,	0xe60520c8), /* PORT200CR */
	PORTCR(201,	0xe60520c9), /* PORT201CR */
	PORTCR(202,	0xe60520ca), /* PORT202CR */
	PORTCR(203,	0xe60520cb), /* PORT203CR */
	PORTCR(204,	0xe60520cc), /* PORT204CR */
	PORTCR(205,	0xe60520cd), /* PORT205CR */
	PORTCR(206,	0xe60520ce), /* PORT206CR */
	PORTCR(207,	0xe60520cf), /* PORT207CR */
	PORTCR(208,	0xe60520d0), /* PORT208CR */
	PORTCR(209,	0xe60520d1), /* PORT209CR */

	PORTCR(210,	0xe60530d2), /* PORT210CR */
	PORTCR(211,	0xe60530d3), /* PORT211CR */

	{ PINMUX_CFG_REG("MSEL1CR", 0xe605800c, 32, 1, GROUP(
			MSEL1CR_31_0,	MSEL1CR_31_1,
			MSEL1CR_30_0,	MSEL1CR_30_1,
			MSEL1CR_29_0,	MSEL1CR_29_1,
			MSEL1CR_28_0,	MSEL1CR_28_1,
			MSEL1CR_27_0,	MSEL1CR_27_1,
			MSEL1CR_26_0,	MSEL1CR_26_1,
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0,
			MSEL1CR_16_0,	MSEL1CR_16_1,
			MSEL1CR_15_0,	MSEL1CR_15_1,
			MSEL1CR_14_0,	MSEL1CR_14_1,
			MSEL1CR_13_0,	MSEL1CR_13_1,
			MSEL1CR_12_0,	MSEL1CR_12_1,
			0, 0, 0, 0,
			MSEL1CR_9_0,	MSEL1CR_9_1,
			0, 0,
			MSEL1CR_7_0,	MSEL1CR_7_1,
			MSEL1CR_6_0,	MSEL1CR_6_1,
			MSEL1CR_5_0,	MSEL1CR_5_1,
			MSEL1CR_4_0,	MSEL1CR_4_1,
			MSEL1CR_3_0,	MSEL1CR_3_1,
			MSEL1CR_2_0,	MSEL1CR_2_1,
			0, 0,
			MSEL1CR_0_0,	MSEL1CR_0_1,
		))
	},
	{ PINMUX_CFG_REG("MSEL3CR", 0xE6058020, 32, 1, GROUP(
			0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0,
			MSEL3CR_15_0,	MSEL3CR_15_1,
			0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0,
			MSEL3CR_6_0,	MSEL3CR_6_1,
			0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0,
			))
	},
	{ PINMUX_CFG_REG("MSEL4CR", 0xE6058024, 32, 1, GROUP(
			0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0,
			MSEL4CR_19_0,	MSEL4CR_19_1,
			MSEL4CR_18_0,	MSEL4CR_18_1,
			0, 0, 0, 0,
			MSEL4CR_15_0,	MSEL4CR_15_1,
			0, 0, 0, 0, 0, 0, 0, 0,
			MSEL4CR_10_0,	MSEL4CR_10_1,
			0, 0, 0, 0, 0, 0,
			MSEL4CR_6_0,	MSEL4CR_6_1,
			0, 0,
			MSEL4CR_4_0,	MSEL4CR_4_1,
			0, 0, 0, 0,
			MSEL4CR_1_0,	MSEL4CR_1_1,
			0, 0,
		))
	},
	{ PINMUX_CFG_REG("MSEL5CR", 0xE6058028, 32, 1, GROUP(
			MSEL5CR_31_0,	MSEL5CR_31_1,
			MSEL5CR_30_0,	MSEL5CR_30_1,
			MSEL5CR_29_0,	MSEL5CR_29_1,
			0, 0,
			MSEL5CR_27_0,	MSEL5CR_27_1,
			0, 0,
			MSEL5CR_25_0,	MSEL5CR_25_1,
			0, 0,
			MSEL5CR_23_0,	MSEL5CR_23_1,
			0, 0,
			MSEL5CR_21_0,	MSEL5CR_21_1,
			0, 0,
			MSEL5CR_19_0,	MSEL5CR_19_1,
			0, 0,
			MSEL5CR_17_0,	MSEL5CR_17_1,
			0, 0,
			MSEL5CR_15_0,	MSEL5CR_15_1,
			MSEL5CR_14_0,	MSEL5CR_14_1,
			MSEL5CR_13_0,	MSEL5CR_13_1,
			MSEL5CR_12_0,	MSEL5CR_12_1,
			MSEL5CR_11_0,	MSEL5CR_11_1,
			MSEL5CR_10_0,	MSEL5CR_10_1,
			0, 0,
			MSEL5CR_8_0,	MSEL5CR_8_1,
			MSEL5CR_7_0,	MSEL5CR_7_1,
			MSEL5CR_6_0,	MSEL5CR_6_1,
			MSEL5CR_5_0,	MSEL5CR_5_1,
			MSEL5CR_4_0,	MSEL5CR_4_1,
			MSEL5CR_3_0,	MSEL5CR_3_1,
			MSEL5CR_2_0,	MSEL5CR_2_1,
			0, 0,
			MSEL5CR_0_0,	MSEL5CR_0_1,
		))
	},
	{ },
};

static const struct pinmux_data_reg pinmux_data_regs[] = {
	{ PINMUX_DATA_REG("PORTL031_000DR", 0xe6054800, 32, GROUP(
		PORT31_DATA,	PORT30_DATA,	PORT29_DATA,	PORT28_DATA,
		PORT27_DATA,	PORT26_DATA,	PORT25_DATA,	PORT24_DATA,
		PORT23_DATA,	PORT22_DATA,	PORT21_DATA,	PORT20_DATA,
		PORT19_DATA,	PORT18_DATA,	PORT17_DATA,	PORT16_DATA,
		PORT15_DATA,	PORT14_DATA,	PORT13_DATA,	PORT12_DATA,
		PORT11_DATA,	PORT10_DATA,	PORT9_DATA,	PORT8_DATA,
		PORT7_DATA,	PORT6_DATA,	PORT5_DATA,	PORT4_DATA,
		PORT3_DATA,	PORT2_DATA,	PORT1_DATA,	PORT0_DATA ))
	},
	{ PINMUX_DATA_REG("PORTL063_032DR", 0xe6054804, 32, GROUP(
		PORT63_DATA,	PORT62_DATA,	PORT61_DATA,	PORT60_DATA,
		PORT59_DATA,	PORT58_DATA,	PORT57_DATA,	PORT56_DATA,
		PORT55_DATA,	PORT54_DATA,	PORT53_DATA,	PORT52_DATA,
		PORT51_DATA,	PORT50_DATA,	PORT49_DATA,	PORT48_DATA,
		PORT47_DATA,	PORT46_DATA,	PORT45_DATA,	PORT44_DATA,
		PORT43_DATA,	PORT42_DATA,	PORT41_DATA,	PORT40_DATA,
		PORT39_DATA,	PORT38_DATA,	PORT37_DATA,	PORT36_DATA,
		PORT35_DATA,	PORT34_DATA,	PORT33_DATA,	PORT32_DATA ))
	},
	{ PINMUX_DATA_REG("PORTL095_064DR", 0xe6054808, 32, GROUP(
		0, 0, 0, 0,
		0, 0, 0, 0,
		0, 0, 0, 0,
		PORT83_DATA,	PORT82_DATA,	PORT81_DATA,	PORT80_DATA,
		PORT79_DATA,	PORT78_DATA,	PORT77_DATA,	PORT76_DATA,
		PORT75_DATA,	PORT74_DATA,	PORT73_DATA,	PORT72_DATA,
		PORT71_DATA,	PORT70_DATA,	PORT69_DATA,	PORT68_DATA,
		PORT67_DATA,	PORT66_DATA,	PORT65_DATA,	PORT64_DATA ))
	},
	{ PINMUX_DATA_REG("PORTD095_064DR", 0xe6055808, 32, GROUP(
		PORT95_DATA,	PORT94_DATA,	PORT93_DATA,	PORT92_DATA,
		PORT91_DATA,	PORT90_DATA,	PORT89_DATA,	PORT88_DATA,
		PORT87_DATA,	PORT86_DATA,	PORT85_DATA,	PORT84_DATA,
		0, 0, 0, 0,
		0, 0, 0, 0,
		0, 0, 0, 0,
		0, 0, 0, 0,
		0, 0, 0, 0 ))
	},
	{ PINMUX_DATA_REG("PORTD127_096DR", 0xe605580c, 32, GROUP(
		0, 0, 0, 0,
		0, 0, 0, 0,
		0, 0, 0, 0,
		0,		PORT114_DATA,	PORT113_DATA,	PORT112_DATA,
		PORT111_DATA,	PORT110_DATA,	PORT109_DATA,	PORT108_DATA,
		PORT107_DATA,	PORT106_DATA,	PORT105_DATA,	PORT104_DATA,
		PORT103_DATA,	PORT102_DATA,	PORT101_DATA,	PORT100_DATA,
		PORT99_DATA,	PORT98_DATA,	PORT97_DATA,	PORT96_DATA ))
	},
	{ PINMUX_DATA_REG("PORTR127_096DR", 0xe605680C, 32, GROUP(
		PORT127_DATA,	PORT126_DATA,	PORT125_DATA,	PORT124_DATA,
		PORT123_DATA,	PORT122_DATA,	PORT121_DATA,	PORT120_DATA,
		PORT119_DATA,	PORT118_DATA,	PORT117_DATA,	PORT116_DATA,
		PORT115_DATA,	0, 0, 0,
		0, 0, 0, 0,
		0, 0, 0, 0,
		0, 0, 0, 0,
		0, 0, 0, 0 ))
	},
	{ PINMUX_DATA_REG("PORTR159_128DR", 0xe6056810, 32, GROUP(
		PORT159_DATA,	PORT158_DATA,	PORT157_DATA,	PORT156_DATA,
		PORT155_DATA,	PORT154_DATA,	PORT153_DATA,	PORT152_DATA,
		PORT151_DATA,	PORT150_DATA,	PORT149_DATA,	PORT148_DATA,
		PORT147_DATA,	PORT146_DATA,	PORT145_DATA,	PORT144_DATA,
		PORT143_DATA,	PORT142_DATA,	PORT141_DATA,	PORT140_DATA,
		PORT139_DATA,	PORT138_DATA,	PORT137_DATA,	PORT136_DATA,
		PORT135_DATA,	PORT134_DATA,	PORT133_DATA,	PORT132_DATA,
		PORT131_DATA,	PORT130_DATA,	PORT129_DATA,	PORT128_DATA ))
	},
	{ PINMUX_DATA_REG("PORTR191_160DR", 0xe6056814, 32, GROUP(
		PORT191_DATA,	PORT190_DATA,	PORT189_DATA,	PORT188_DATA,
		PORT187_DATA,	PORT186_DATA,	PORT185_DATA,	PORT184_DATA,
		PORT183_DATA,	PORT182_DATA,	PORT181_DATA,	PORT180_DATA,
		PORT179_DATA,	PORT178_DATA,	PORT177_DATA,	PORT176_DATA,
		PORT175_DATA,	PORT174_DATA,	PORT173_DATA,	PORT172_DATA,
		PORT171_DATA,	PORT170_DATA,	PORT169_DATA,	PORT168_DATA,
		PORT167_DATA,	PORT166_DATA,	PORT165_DATA,	PORT164_DATA,
		PORT163_DATA,	PORT162_DATA,	PORT161_DATA,	PORT160_DATA ))
	},
	{ PINMUX_DATA_REG("PORTR223_192DR", 0xe6056818, 32, GROUP(
		0, 0, 0, 0,
		0, 0, 0, 0,
		0, 0, 0, 0,
		0, 0,				PORT209_DATA,	PORT208_DATA,
		PORT207_DATA,	PORT206_DATA,	PORT205_DATA,	PORT204_DATA,
		PORT203_DATA,	PORT202_DATA,	PORT201_DATA,	PORT200_DATA,
		PORT199_DATA,	PORT198_DATA,	PORT197_DATA,	PORT196_DATA,
		PORT195_DATA,	PORT194_DATA,	PORT193_DATA,	PORT192_DATA ))
	},
	{ PINMUX_DATA_REG("PORTU223_192DR", 0xe6057818, 32, GROUP(
		0, 0, 0, 0,
		0, 0, 0, 0,
		0, 0, 0, 0,
		PORT211_DATA,	PORT210_DATA, 0, 0,
		0, 0, 0, 0,
		0, 0, 0, 0,
		0, 0, 0, 0,
		0, 0, 0, 0 ))
	},
	{ },
};

static const struct pinmux_irq pinmux_irqs[] = {
	PINMUX_IRQ(2,   13),	/* IRQ0A */
	PINMUX_IRQ(20),		/* IRQ1A */
	PINMUX_IRQ(11,  12),	/* IRQ2A */
	PINMUX_IRQ(10,  14),	/* IRQ3A */
	PINMUX_IRQ(15,  172),	/* IRQ4A */
	PINMUX_IRQ(0,   1),	/* IRQ5A */
	PINMUX_IRQ(121, 173),	/* IRQ6A */
	PINMUX_IRQ(120, 209),	/* IRQ7A */
	PINMUX_IRQ(119),	/* IRQ8A */
	PINMUX_IRQ(118, 210),	/* IRQ9A */
	PINMUX_IRQ(19),		/* IRQ10A */
	PINMUX_IRQ(104),	/* IRQ11A */
	PINMUX_IRQ(42,  97),	/* IRQ12A */
	PINMUX_IRQ(64,  98),	/* IRQ13A */
	PINMUX_IRQ(63,  99),	/* IRQ14A */
	PINMUX_IRQ(62,  100),	/* IRQ15A */
	PINMUX_IRQ(68,  211),	/* IRQ16A */
	PINMUX_IRQ(69),		/* IRQ17A */
	PINMUX_IRQ(70),		/* IRQ18A */
	PINMUX_IRQ(71),		/* IRQ19A */
	PINMUX_IRQ(67),		/* IRQ20A */
	PINMUX_IRQ(202),	/* IRQ21A */
	PINMUX_IRQ(95),		/* IRQ22A */
	PINMUX_IRQ(96),		/* IRQ23A */
	PINMUX_IRQ(180),	/* IRQ24A */
	PINMUX_IRQ(38),		/* IRQ25A */
	PINMUX_IRQ(58,  81),	/* IRQ26A */
	PINMUX_IRQ(57,  168),	/* IRQ27A */
	PINMUX_IRQ(56,  169),	/* IRQ28A */
	PINMUX_IRQ(50,  170),	/* IRQ29A */
	PINMUX_IRQ(49,  171),	/* IRQ30A */
	PINMUX_IRQ(41,  167),	/* IRQ31A */
};

#define PORTnCR_PULMD_OFF	(0 << 6)
#define PORTnCR_PULMD_DOWN	(2 << 6)
#define PORTnCR_PULMD_UP	(3 << 6)
#define PORTnCR_PULMD_MASK	(3 << 6)

struct r8a7740_portcr_group {
	unsigned int end_pin;
	unsigned int offset;
};

static const struct r8a7740_portcr_group r8a7740_portcr_offsets[] = {
	{ 83, 0x0000 }, { 114, 0x1000 }, { 209, 0x2000 }, { 211, 0x3000 },
};

static void __iomem *r8a7740_pinmux_portcr(struct sh_pfc *pfc, unsigned int pin)
{
	unsigned int i;

	for (i = 0; i < ARRAY_SIZE(r8a7740_portcr_offsets); ++i) {
		const struct r8a7740_portcr_group *group =
			&r8a7740_portcr_offsets[i];

		if (pin <= group->end_pin)
			return pfc->windows->virt + group->offset + pin;
	}

	return NULL;
}

static unsigned int r8a7740_pinmux_get_bias(struct sh_pfc *pfc, unsigned int pin)
{
	void __iomem *addr = r8a7740_pinmux_portcr(pfc, pin);
	u32 value = ioread8(addr) & PORTnCR_PULMD_MASK;

	switch (value) {
	case PORTnCR_PULMD_UP:
		return PIN_CONFIG_BIAS_PULL_UP;
	case PORTnCR_PULMD_DOWN:
		return PIN_CONFIG_BIAS_PULL_DOWN;
	case PORTnCR_PULMD_OFF:
	default:
		return PIN_CONFIG_BIAS_DISABLE;
	}
}

static void r8a7740_pinmux_set_bias(struct sh_pfc *pfc, unsigned int pin,
				   unsigned int bias)
{
	void __iomem *addr = r8a7740_pinmux_portcr(pfc, pin);
	u32 value = ioread8(addr) & ~PORTnCR_PULMD_MASK;

	switch (bias) {
	case PIN_CONFIG_BIAS_PULL_UP:
		value |= PORTnCR_PULMD_UP;
		break;
	case PIN_CONFIG_BIAS_PULL_DOWN:
		value |= PORTnCR_PULMD_DOWN;
		break;
	}

	iowrite8(value, addr);
}

static const struct sh_pfc_soc_operations r8a7740_pfc_ops = {
	.get_bias = r8a7740_pinmux_get_bias,
	.set_bias = r8a7740_pinmux_set_bias,
};

const struct sh_pfc_soc_info r8a7740_pinmux_info = {
	.name		= "r8a7740_pfc",
	.ops		= &r8a7740_pfc_ops,

	.input		= { PINMUX_INPUT_BEGIN,
			    PINMUX_INPUT_END },
	.output		= { PINMUX_OUTPUT_BEGIN,
			    PINMUX_OUTPUT_END },
	.function	= { PINMUX_FUNCTION_BEGIN,
			    PINMUX_FUNCTION_END },

	.pins		= pinmux_pins,
	.nr_pins	= ARRAY_SIZE(pinmux_pins),
	.groups		= pinmux_groups,
	.nr_groups	= ARRAY_SIZE(pinmux_groups),
	.functions	= pinmux_functions,
	.nr_functions	= ARRAY_SIZE(pinmux_functions),

	.cfg_regs	= pinmux_config_regs,
	.data_regs	= pinmux_data_regs,

	.pinmux_data	= pinmux_data,
	.pinmux_data_size = ARRAY_SIZE(pinmux_data),

	.gpio_irq	= pinmux_irqs,
	.gpio_irq_size	= ARRAY_SIZE(pinmux_irqs),
};
