/*
 * qvbanner.cc
 *
 * Copyright (C) 1995-1999 Kenichi Kourai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with qvbanner; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/xpm.h>
#include "cursor.h"

Display* display;
Window w;
long expiry = 5;           // expiry time of logo
char* logo_file = NULL;

void AnalizeOptions(int argc, char** argv);
void Usage();
Window CreateScreenWindow();
void CreateNullCursor();
Bool CreateLogoPixmap(char* logo_file, Pixmap* pix, XpmAttributes* attr);
void SetExpiryTimer(long sec);
void EventLoop(Pixmap pix, const XpmAttributes& attr);
void ExpireBanner(int sig);

int main(int argc, char **argv)
{
  /*
   * Analize command line options.
   */
  AnalizeOptions(argc, argv);

  /*
   * Open display.
   */
  if ((display = XOpenDisplay(NULL)) == NULL) {
    fprintf(stderr, "qvbanner: Can't open display\n");
    exit(1);
  }

  /*
   * Grab X server so that other windows aren't mapped on this.
   */
  XGrabServer(display);

  /*
   * Create whole screen window.
   */
  w = CreateScreenWindow();

  /*
   * Create transparent cursor.
   */
  CreateNullCursor();

  /*
   * Map window on top.
   */
  XMapRaised(display, w);
  XFlush(display);

  /*
   * Create logo pixmap.
   */
  XpmAttributes attr;
  Pixmap pix;

  if (!CreateLogoPixmap(logo_file, &pix, &attr)) {
    XCloseDisplay(display);
    exit(1);
  }

  SetExpiryTimer(expiry);

  EventLoop(pix, attr);
}

void AnalizeOptions(int argc, char** argv)
{
  if (argc == 1)
    Usage();

  for (int i = 1; i < argc; i++) {
    if (*argv[i] == '-') {
      if (strcmp(argv[i] + 1, "v") == 0) {
	fprintf(stderr, "qvbanner version %s\n", VERSION);
	fprintf(stderr, "Copyright (C) 1996-1999 Kenichi Kourai.\n");
	exit(1);
      }
      else if (strcmp(argv[i] + 1, "expiry") == 0)
	expiry = atoi(argv[++i]);
      else
	Usage();
    }
    else {
      if (i == argc - 1)
	logo_file = argv[i];
      else
	Usage();
    }
  }
}

void Usage()
{
  fprintf(stderr, "Usage: qvbanner [-v] [-expiry time(sec)] logo_file\n");
  exit(1);
}

Window CreateScreenWindow()
{
  XSetWindowAttributes attributes;
  unsigned long valueMask;
  Window w;

  attributes.event_mask = ExposureMask;
  attributes.background_pixel = BlackPixel(display, 0);
  attributes.override_redirect = True;
  valueMask = CWBackPixel | CWEventMask | CWOverrideRedirect;

  w = XCreateWindow(display, RootWindow(display, 0),
		    0, 0, DisplayWidth(display, 0), DisplayHeight(display, 0),
		    0, CopyFromParent, InputOutput, CopyFromParent,
		    valueMask, &attributes);

  return w;
}

/*
 * CreateNullCursor --
 *   Create transparent cursor.
 */
void CreateNullCursor()
{
  Pixmap pixCursor, pixMask;
  XColor black, dummyColor;
  Cursor cursor;

  pixCursor = XCreateBitmapFromData(display, w, curbm_bits,
				    cursor_width, cursor_height);
  pixMask = XCreateBitmapFromData(display, w, maskbm_bits,
				  cursor_width, cursor_height);

  XAllocNamedColor(display, DefaultColormap(display, 0), "black",
		   &black, &dummyColor);
  cursor = XCreatePixmapCursor(display, pixCursor, pixMask, &black, &black,
			       0, 0);
  XDefineCursor(display, w, cursor);
}

/*
 * CreateLogoPixmap --
 *   Create logo pixmap from file.
 */
Bool CreateLogoPixmap(char* logo_file, Pixmap* pix, XpmAttributes* attr)
{
  attr->valuemask = XpmColormap | XpmDepth | XpmCloseness;
  attr->colormap = DefaultColormap(display, 0);
  attr->depth = DefaultDepth(display, 0);
  attr->closeness = 65535;
  
  int error;

  if ((error = XpmReadFileToPixmap(display, w, logo_file, pix, NULL, attr))
      == XpmSuccess) {
    return True;
  }

  switch (error) {
  case XpmOpenFailed:
    fprintf(stderr, "qvbanner: can't find pixmap '%s'\n", logo_file);
    break;

  case XpmFileInvalid:
    fprintf(stderr, "qvbanner: '%s' isn't valid pixmap file\n", logo_file);
    break;

  case XpmColorFailed:
    fprintf(stderr, "qvbanner: too many colors in '%s'\n", logo_file);
    break;

  default:
    fprintf(stderr, "qvbanner: pixmap error in '%s'", logo_file);
  }

  return False;
}

/*
 * SetExpiryTimer --
 *   Set expiry timer.
 */
void SetExpiryTimer(long sec)
{
  struct itimerval itv;

  itv.it_value.tv_sec = sec;
  itv.it_value.tv_usec = 0;
  itv.it_interval.tv_sec = 0;
  itv.it_interval.tv_usec = 0;
  setitimer(ITIMER_REAL, &itv, NULL);

  signal(SIGALRM, ExpireBanner);
}

/*
 * EventLoop --
 *   Event loop.
 */
void EventLoop(Pixmap pix, const XpmAttributes& attr)
{
  XEvent ev;
  GC gc = XCreateGC(display, w, 0, 0);
  unsigned int dWidth = DisplayWidth(display, 0);
  unsigned int dHeight = DisplayHeight(display, 0);

  while (1) {
    XNextEvent(display, &ev);
    switch (ev.type) {
    case Expose:
      XCopyArea(display, pix, w, gc, 0, 0, attr.width, attr.height,
		(dWidth-attr.width)/2, (dHeight-attr.height)/2);
      break;
    }
  }
}

/*
 * ExpireBanner --
 *   Exit banner.
 */
void ExpireBanner(int sig)
{
  XUngrabServer(display);

  XCloseDisplay(display);
  exit(0);
}
