/*
 *  Copyright (c) by Shuu Yamaguchi <shuu@dotaster.com>
 *
 *  $Id: init.c,v 1.10 2003/09/21 04:11:23 shuu Exp shuu $
 *
 *  Can be freely distributed and used under the terms of the GNU GPL.
 */
#include	<stdio.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<fcntl.h>
#include	<unistd.h>
#include	<string.h>
#include	<stdlib.h>
#include	<sys/wait.h>
#define __USE_XOPEN_EXTENDED 1
#include	<ftw.h>
#include	<limits.h>

#include	"murasaki.h"

#define	USB_PROCDIR		"/proc/bus/usb"
#ifdef DEBUG_WITH_DEVICES
#define	DEVICE_FILE		"/tmp/devices"
#else
#define	DEVICE_FILE		"/proc/bus/usb/devices"
#endif
#define	dprintf(arg...)	if(debug) printf(arg)
#define	MU_BLOCK_SIZE	64
#define	DEVICE_LINE_SIZE	256

#define	ST_NONE			0
#define	ST_NEW			1
#define	ST_PRODUCT		(1<<1)
#define	ST_TYPE			(1<<2)
#define	ST_INTERFACE		(1<<3)
#define	ST_ENDPOINT		(1<<4)
#define	ST_TOPOLOGY		(1<<5)
#define	ST_END			-1

#define	DEV_BUS(d)		(d)->bus
#define	DEV_DEVNO(d)		(d)->devno
#define	DEV_VENDOR(d)		(d)->product.vendor
#define	DEV_PRODUCT(d)		(d)->product.product
#define	DEV_DEVICE(d)		(d)->product.device
#define	DEV_TYPE_CLASS(d)	(d)->type.class
#define	DEV_TYPE_SUBCLASS(d)	(d)->type.subclass
#define	DEV_TYPE_PROTOCOL(d)	(d)->type.protocol
#define	DEV_INT_CLASS(d,n)	(d)->interface[(n)].class
#define	DEV_INT_SUBCLASS(d,n)	(d)->interface[(n)].subclass
#define	DEV_INT_PROTOCOL(d,n)	(d)->interface[(n)].protocol

#define INTERFACE_MAX		16
#define	SCRATCH_BUFSIZE		256

enum mount_action {
	USBFS_MOUNT, 
	USBFS_UNMOUNT 
};

struct mu_usb_product {
	u_int16_t	vendor;
	u_int16_t	product;
	u_int16_t	device;	/* BCD */
};

struct mu_usb_class {
	u_int8_t	class;
	u_int8_t	subclass;
	u_int8_t	protocol;
};

struct usb_init {
	int						bus;
	int						devno;
	struct mu_usb_product	product;
	struct mu_usb_class	type;
	struct mu_usb_class	interface[INTERFACE_MAX];
};

/* global */
int debug = 0;	/* debug flag */
char *sysfs = NULL;	/* SYSFS dir */
	
static void
execute_hotplug(struct usb_init *dev)
{
	char buf[SCRATCH_BUFSIZE],*scratch;
	char *argv[3],*envp[16];
	int i;
	char *ptr;

	switch(fork()) {
	case -1:
		syslog(LOG_LEVEL,"fork error\n");
		exit(1);
	case 0:
		init_argument(argv,MU_ARG_USB);

		i = 0;
		scratch = buf;
		if ((ptr = getenv("MURASAKI_BEEP")) != NULL) {
			envp[i++] = ptr-14;	/* uum */
		}
		envp[i++] = "HOME=/";
		envp[i++] = "PATH=/sbin:/bin:/usr/sbin:/usr/bin";
		envp[i++] = "ACTION=add";
		/* When /proc/bus/usb is mounted */
		envp[i++] = "DEVFS=/proc/bus/usb";
		envp[i++] = scratch;
		scratch += sprintf(scratch,"DEVICE=/proc/bus/usb/%03d/%03d",
			DEV_BUS(dev),DEV_DEVNO(dev));

		envp[i++] = scratch;
		scratch += sprintf(scratch,"PRODUCT=%x/%x/%x",
			DEV_VENDOR(dev),DEV_PRODUCT(dev),DEV_DEVICE(dev));
		envp[i++] = scratch;
		scratch += sprintf(scratch,"TYPE=%x/%x/%x",
			DEV_TYPE_CLASS(dev),
			DEV_TYPE_SUBCLASS(dev),
			DEV_TYPE_PROTOCOL(dev));
		/* INTERFACE-0, INTERFACE-1, ... */
		envp[i++] = scratch;
		scratch += sprintf(scratch,"INTERFACE=%x/%x/%x",
			DEV_INT_CLASS(dev,0),
			DEV_INT_SUBCLASS(dev,0),
			DEV_INT_PROTOCOL(dev,0));
		envp[i] = 0;
		dprintf("execve [0]:%s [1]:%s\n",argv[0],argv[1]);
		execve(argv[0],argv,envp);
	default:
		if (debug) {
			wait(NULL);
		}
		break;
	}
}
/* 
 * using I(interface descriptor) instead of D(Device descriptor)
 */

static void
parse_line(char *linep,int status,struct usb_init *dev)
{
	int i;
	static int int_no;	/* interface number */

	switch(status) {
	case ST_PRODUCT:
		/* P:  Vendor=xxxx ProdID=xxxx Rev=xx.xx */
		linep = strchr(linep,'=');
		linep++;
		DEV_VENDOR(dev) = strtoul(linep,NULL,16);
		linep = strchr(linep,'=');
		linep++;
		DEV_PRODUCT(dev) = strtoul(linep,NULL,16);
		DEV_DEVICE(dev) = 0;	/* set up later */
		dprintf("Vender=0x%x Product=0x%x\n",
			DEV_VENDOR(dev),DEV_PRODUCT(dev));
		break;
	case ST_TYPE:
		/* Ver=x.xx Cls=xx(sssss) Sub=xx Prot=xx MxPS=dd #Cfgs=dd */
		for(i=0;i < 2;i++) {
			linep = strchr(linep,'=');
			linep++;
		}
		DEV_TYPE_CLASS(dev) = strtoul(linep,NULL,16);
		linep = strchr(linep,'=');
		linep++;
		DEV_TYPE_SUBCLASS(dev) = strtoul(linep,NULL,16);
		linep = strchr(linep,'=');
		linep++;
		DEV_TYPE_PROTOCOL(dev) = strtoul(linep,NULL,16);
		dprintf("Type Class=0x%x Subclass=0x%x Protocol=0x%x\n",
			DEV_TYPE_CLASS(dev),DEV_TYPE_SUBCLASS(dev),DEV_TYPE_PROTOCOL(dev));
		break;
	case ST_INTERFACE:	
		/* I:  If#=dd Alt=dd #EPs=dd Cls=xx(sssss) Sub=xx Prot=xx Driver=ssss
	 	 */
		if (int_no >= INTERFACE_MAX-1) {
			syslog(LOG_LEVEL,"interface overflow\n");
			break;
		}
		for(i=0;i < 4;i++) {
			linep = strchr(linep,'=');
			linep++;
		}
		DEV_INT_CLASS(dev,int_no) = strtoul(linep,NULL,16);
		linep = strchr(linep,'=');
		linep++;
		DEV_INT_SUBCLASS(dev,int_no) = strtoul(linep,NULL,16);
		linep = strchr(linep,'=');
		linep++;
		DEV_INT_PROTOCOL(dev,int_no) = strtoul(linep,NULL,16);
		dprintf("Interface Class=0x%x Subclass=0x%x Protocol=0x%x\n",
			DEV_INT_CLASS(dev,int_no),
			DEV_INT_SUBCLASS(dev,int_no),
			DEV_INT_PROTOCOL(dev,int_no));
		int_no++;
		break;
	case ST_TOPOLOGY:
		/* T:  Bus=dd Lev=dd Prnt=dd Port=dd Cnt=dd Dev#=ddd Spd=ddd MxCh=dd
		 */
		int_no = 0;	/* clear interface number */
		linep = strchr(linep,'=');
		linep++;
		DEV_BUS(dev) = strtoul(linep,NULL,10);
		dprintf("Bus=%03d\n", DEV_BUS(dev));
		linep = strchr(linep,'#');
		linep += 2;
		DEV_DEVNO(dev) = strtoul(linep,NULL,10);
		dprintf("Dev#=%03d\n", DEV_DEVNO(dev));
		break;
	}
}

static int
parse_devices(char **curp,char *endp)
{
	struct usb_init dev;
	char line[DEVICE_LINE_SIZE];
	int status = ST_NEW;
	int i = 0;	/* verbose */
	char *ptr;
	int first = 1;

	ptr = *curp;

	if (*ptr == '\n') {	/* new devices format */
		ptr++;
	}
	if (*ptr != 'T') {
		syslog(LOG_LEVEL,"invalid buffer line\n");
		return ST_END;
	}
	for(;ptr < endp;ptr++) {
		if (status == ST_NEW) {
			i=0;
			switch(*ptr) {
			case 'P':
				status = ST_PRODUCT;
				break;
			case 'D':
				status = ST_TYPE;
				break;
			case 'I':
				status = ST_INTERFACE;
				break;
			case 'T':
				status = ST_TOPOLOGY;
				if (!first) {
					/*
					 * A next USB device info starts.
					 * I got enough infos of USB device
					 */
					goto loop_end;
				}
				first = 0;
				break;
			default:
				status = ST_NONE;
				break;
			}
		} else if (status & (ST_PRODUCT|ST_TYPE|ST_INTERFACE|ST_TOPOLOGY) ) {
			line[i++] = *ptr;
		}
		/* parse the line */
		if (*ptr == '\n') {
			line[i] = '\0';
			if (status & (ST_PRODUCT|ST_TYPE|ST_INTERFACE|ST_TOPOLOGY)) {
				parse_line(line,status,&dev);
			}
			status = ST_NEW;
		}
	}
loop_end:
	*curp = ptr;	/* ptr point to next character */
	dprintf("left %d\n",endp - ptr);
	/* call Hotplug */
	if (debug == 0) {
		execute_hotplug(&dev);
	}
	/*
	 * ptr equals to endp which the last character and
	 *  '\n'.
	 */
	if (ptr == endp)
		return ST_END;

	return ST_NEW;	/* ! ST_END */
}

static int
observe(void)
{
	char *buf,*curp;
	int sum;

	buf = alloc_read(DEVICE_FILE,&sum);
	if (buf == NULL)
		return -1;
	curp = buf;
	while(parse_devices(&curp,buf+sum-1) != ST_END)
		;

#ifdef KERNEL_JOB
	free(buf);
#endif

	return 0;
}

/*
 * mount -t usbdevfs /proc/bus/usb /proc/bus/usb
 *                    and 
 * umount /proc/bus/usb
 *
 * Don't use mount(2) and umount(2),because /etc/mtab is unnecessary.
 */
static void
do_mount(enum mount_action action)
{
	char *mount_arg[] = {"mount","-t","usbdevfs",USB_PROCDIR,USB_PROCDIR,NULL};
	char *unmount_arg[] = {"umount",USB_PROCDIR,NULL};
	char **mnt_arg;
	static int mnt_flag = 0;	/* verbose */

	dprintf("do_mount action %d\n",action);
	if (action == USBFS_MOUNT) {
		if (access(DEVICE_FILE,R_OK) == 0)
			return;
	} else if (mnt_flag == 0) { /* usbmgr didn't mount /proc/bus/usb */
		dprintf("do_mount need not to unmount\n");
		return;
	}

	switch(fork()){
	case 0:		/* child */
		if (action == USBFS_MOUNT)
			mnt_arg = mount_arg;
		else
			mnt_arg = unmount_arg;
		dprintf("%s %s\n",mnt_arg[0],DEVICE_FILE);
		execvp(mnt_arg[0],mnt_arg);
		dprintf("exec error %s %s\n",mnt_arg[0],DEVICE_FILE);
		exit(1);
		break;
	case -1:	/* error */
		dprintf("do_mount fork error\n");
		break;
	default:	/* parent */
		dprintf("do_mount mnt_flag %d\n",mnt_flag);
		if (action == USBFS_MOUNT)
			mnt_flag = 1;
		else
			mnt_flag = 0;
		dprintf("do_mount mnt_flag -> %d\n",mnt_flag);
		wait(NULL);
		break;
	}
}


/*
 * sysfs 
 */
static int
sysfs_sub(const char *path,const struct stat *st,int flag,struct FTW *s)
{
	char dname[PATH_MAX],*ptr;

	if (flag == FTW_F) {
		/* need not waiting for files */
		if (strcmp(path + s->base,"bNumConfigurations") == 0 ||
			strcmp(path + s->base,"iInterface") == 0) {
			strcpy(dname,path+strlen(sysfs));
			ptr = strrchr(dname,'/');
			*ptr = '\0';
			dprintf("devpath=%s\n",dname);
			if (debug == 0) {
				execute_hotplug_sysfs(dname,MU_ARG_USB);
			}
		}
	}
	
	return 0;
}


/*
 * global: config_list
 */
int
main(int argc,char **argv)
{
	char sysfs_devices[PATH_MAX];

	if (check_config(MU_INIT_USB) == ARG_OFF) {
		syslog(LOG_LEVEL,"%s is defined as \"%s\"",
			config_list[0],config_list[1]);
		exit(0);
	}
	if (getenv("MURASAKI_DEBUG"))
		debug = 1;
		
	sysfs = check_sysfs(NULL);
	if (sysfs != NULL) {
		strcpy(sysfs_devices,sysfs);
		strcat(sysfs_devices,"/devices");
		nftw(sysfs_devices,sysfs_sub,16,FTW_PHYS);
	} else {
#ifdef DEBUG_WITH_DEVICES
		observe();
#else
		do_mount(USBFS_MOUNT);

		observe();

		do_mount(USBFS_UNMOUNT);
#endif
	}

	return 0;
}
