/*
 *  Copyright (c) by Shuu Yamaguchi <shuu@dotaster.com>
 *
 *  $Id: init.c,v 1.4 2003/09/21 04:17:58 shuu Exp shuu $
 *
 *  Can be freely distributed and used under the terms of the GNU GPL.
 */
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<sys/types.h>
#include	<unistd.h>
#include	<wait.h>

#include	"murasaki.h"
#include	"murasaki_ieee1394.h"

#ifdef	DEBUG
#define	PROC_FILE	"./devices"
#else
#define	PROC_FILE	"/proc/bus/ieee1394/devices"
#endif

#define	STR_VENDOR	"Vendor/Model ID:"
#define	STR_SPEC	"Software Specifier ID:"
#define	STR_VER		"Software Version:"

#define	MATCH(A,B)	strncmp((A),(B),sizeof((A))-1) == 0

#define	ST_NEWLINE	1
#define	ST_DUMB		2

/* global */
char *hotplug_path = NULL;
int debug = 0;

static char *
get_vendor_model(MU_ieee1394_config_t *config,char *line)
{
	char *ptr,*nextp;

	ptr = strchr(line,'[');
	if (ptr == NULL)
		return NULL;
	config->vendor = (unsigned int)strtoul(ptr+1,&nextp,16);
	ptr = strchr(nextp,'[');
	if (ptr == NULL)
		return NULL;
	config->model = (unsigned int)strtoul(ptr+1,&nextp,16);

	return nextp;
}

static void
execute_hotplug(MU_ieee1394_config_t *config)
{
	char buf[256],*scratch;
	char *argv[3],*envp[16];
	int i;
	char *ptr;

	switch(fork()) {
	case -1:
		syslog(LOG_LEVEL,"fork error\n");
		exit(1);
	case 0:
		if (hotplug_path == NULL && (hotplug_path = get_hotplug()) == NULL)
			hotplug_path = DEFAULT_HOTPLUG;
		i = 0;
		argv[i++] = hotplug_path;
		argv[i++] = MU_ARG_IEEE1394;
		argv[i] = 0;

		i = 0;
		scratch = buf;
		if ((ptr = getenv("MURASAKI_BEEP")) != NULL) {
			envp[i++] = ptr-14;
		}
		/*
		 * ignore "GUID_ID"
		 */
		envp[i++] = "HOME=/";
		envp[i++] = "PATH=/sbin:/bin:/usr/sbin:/usr/bin";
		envp[i++] = "ACTION=add";
		envp[i++] = scratch;
		scratch += sprintf(scratch,"VENDOR_ID=%06x",config->vendor);
		envp[i++] = scratch;
		scratch += sprintf(scratch,"SPECIFIER_ID=%06x",config->specifier);
		envp[i++] = scratch;
		scratch += sprintf(scratch,"VERSION=%06x",config->version);
		envp[i] = 0;

		execve(argv[0], argv, envp);
		break;
	default:
		if (debug) {
			wait(0);
		}
		break;
	}
}

static int
parse_devices(char *buf,int sum)
{
	char *ptr,*nextp;
	int status = ST_NEWLINE;
	MU_ieee1394_config_t config;

	for(ptr = buf; ptr <= buf+sum;ptr++) {
		if (status == ST_NEWLINE) {
			if (*ptr == ' ')
				continue;
			if (MATCH("Node",ptr)) {
				;
			} else if (MATCH(STR_VENDOR,ptr)) {
				ptr = get_vendor_model(&config,ptr);
				if (ptr == NULL)
					return -1;
			} else if (MATCH(STR_SPEC,ptr)) {
				config.specifier = (unsigned int)strtoul(ptr+sizeof(STR_SPEC)-1,&nextp,16);
				ptr = nextp;
			} else if (MATCH(STR_VER,ptr)) {
				config.version = (unsigned int)strtoul(ptr+sizeof(STR_VER)-1,&nextp,16);
				ptr = nextp;
				DPRINTF(LOG_LEVEL, "vendor:0x%x model:0x%x specifier:0x%x version:0x%x\n",config.vendor, config.model, config.specifier, config.version);
				/* call hotplug */
				if (debug == 0) {
					execute_hotplug(&config);
				}
			}
			status = ST_DUMB;
		}
		if (*ptr == '\n') {
			status = ST_NEWLINE;
		}
	}

	return 0;
}

int
main(int argc,char **argv)
{
	char *buf;
	int sum;

	if (check_config(MU_INIT_IEEE1394) == ARG_OFF) {
		syslog(LOG_LEVEL,"%s is defined as \"%s\"",
			config_list[0],config_list[1]);
		exit(0);
	}
	if (getenv("MURASAKI_DEBUG"))
		debug = 1;

	if (access(PROC_FILE,R_OK) != 0)	/* no IEEE1394 devices */
		return 0;

	buf = alloc_read(PROC_FILE,&sum);
	if (buf == NULL)
		return -1;

	parse_devices(buf,sum);
	
#ifdef	KERNEL_JOB
	free(buf);
#endif

	return 0;
}

