/*  xfce4
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifndef PATH_MAX
#define PATH_MAX 1024
#endif

#include <libxfce4util/libxfce4util.h>

#include "icons.h"
#include "xfce-icontheme.h"

static XfceIconTheme *icon_theme = NULL;


/*
 * create a GdkPixbuf from inline data and scale it to a given size
 */
GdkPixbuf *
xfce_inline_icon_at_size (const guint8 * data, int width, int height)
{

    GdkPixbuf *base;

    base = gdk_pixbuf_new_from_inline (-1, data, FALSE, NULL);

    g_assert (base);

    if ((width < 0 && height < 0)
	|| (gdk_pixbuf_get_width (base) == width
	    && gdk_pixbuf_get_height (base) == height))
    {
	return base;
    }
    else
    {
	GdkPixbuf *scaled;

	scaled =
	    gdk_pixbuf_scale_simple (base,
				     width >
				     0 ? width : gdk_pixbuf_get_width (base),
				     height >
				     0 ? height :
				     gdk_pixbuf_get_height (base),
				     GDK_INTERP_BILINEAR);

	g_object_unref (G_OBJECT (base));

	return scaled;
    }
}

/* deprecated */
GdkPixbuf *
inline_icon_at_size (const guint8 * data, int width, int height)
{
	return xfce_inline_icon_at_size(data, width, height);
}

/**
 * xfce_set_icon_theme:
 * @theme_name: The name of the icon theme.
 *
 * Sets the icon theme that xfce_load_themed_icon() will use when looking up
 * icons.  Note: this function amounts to a no-op if libxfcegui4 is compiled
 * against Gtk+ 2.4.0 or higher, as Gtk+ 2.4.0 has built-in support for icon
 * theming.
 *
 * Since 4.1
 **/
void
xfce_set_icon_theme(const gchar *theme_name)
{
	/* noop */
}

/*< deprecated name; use xfce_themed_icon_load() instead >*/
GdkPixbuf *
xfce_load_themed_icon(const gchar *name, gint size)
{
	return xfce_themed_icon_load(name, size);
}

static void
ensure_xfce_icon_theme()
{	
	if(icon_theme)
		return;
	icon_theme = xfce_icon_theme_get_for_screen(NULL);
}


/**
 * xfce_themed_icon_load:
 * @name: The name of the icon.
 * @size: The requested size of the icon.
 *
 * Uses the current icon theme to find an icon matching @name and @size.  For
 * themed icons, @name should be the root name of the icon, without extension.
 * #xfce_load_themed_icon() will also attempt to directly load icon files if an
 * absoulte or relative path is provided.  If an icon is found, it is resized
 * to the specified size if needed.
 *
 * Return value: A pointer to a #GdkPixbuf, or %NULL on failure.  This data must
 *               be freed with g_object_unref() after use.
 *
 * Since 4.1
 **/
GdkPixbuf *
xfce_themed_icon_load(const gchar *name, gint size)
{
	if(!icon_theme)
		ensure_xfce_icon_theme();
	return xfce_icon_theme_load(icon_theme, name, size);
}

/**
 * xfce_themed_icon_lookup:
 * @name: The name of the icon.
 * @size: The requested size of the icon.
 *
 * Uses the current icon theme to find an icon matching @name and @size.  For
 * themed icons, @name should be the root name of the icon, without extension.
 * #xfce_themed_icon_lookup() will also attempt to directly load icon files if
 * an absoulte or relative path is provided.  
 *
 * Return value: An icon filename (should be freed with #g_free()), or %NULL.
 *
 * Since 4.1
 **/
gchar *
xfce_themed_icon_lookup(const gchar *name, gint size)
{	
	if(!icon_theme)
		ensure_xfce_icon_theme();
	return xfce_icon_theme_lookup(icon_theme, name, size);
}

void
xfce_themed_icon_add_search_path(const gchar *path)
{
	if(!icon_theme)
		ensure_xfce_icon_theme();
	xfce_icon_theme_append_search_path(icon_theme, path);
}

/**
 * xfce_pixbuf_new_from_file_at_size:
 * @filename: An image file name.
 * @width: The target image width.
 * @height: The target image height.
 * @error: a pointer to a GError, or NULL.
 *
 * Convenience function to avoid GTK_CHECK_VERSION(), as GdkPixbuf 2.2 and below
 * do not have gdk_pixbuf_new_from_file_at_size().
 *
 * Return value: A new GdkPixbuf, sized @width by @height, or NULL if the file
 *               cannot be opened.  If @error is not NULL, it will be set upon
 *               error.
 *
 * Since 4.1
 **/
GdkPixbuf *
xfce_pixbuf_new_from_file_at_size(const gchar *filename, gint width, gint height,
		GError **error)
{
#if GTK_CHECK_VERSION(2, 4, 0)
	return gdk_pixbuf_new_from_file_at_size(filename, width, height, error);
#else
	GdkPixbuf *pix, *tmp;
	gint w, h;
	
	pix = gdk_pixbuf_new_from_file(filename, error);
	if(pix) {
		w = gdk_pixbuf_get_width(pix);
		h = gdk_pixbuf_get_height(pix);
		if(w != width || h != height) {
			tmp = gdk_pixbuf_scale_simple(pix, width, height,
					GDK_INTERP_BILINEAR);
			g_object_unref(G_OBJECT(pix));
			pix = tmp;
		}
	}
	
	return pix;
#endif
}
