/*
 * Copyright (C) 2001, 2002 Hiroshi Takekawa <sian@big.or.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *                
 */

#if defined(HAVE_CONFIG_H)
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "converter.h"
#include "../xmms/i18n.h"

#ifdef HAVE_ICONV
# define ICONV_OUTPUT_SIZE 65536
# include <iconv.h>
#else
# ifdef KANJI
#  include "kanji.h"
# endif
#endif

#ifdef HAVE_ICONV
int xmms_converter_convert(gchar *s, gchar **d_r, char *from, char *to)
{
  iconv_t cd;
  gchar *d, *inptr, *outptr;
  size_t avail, insize, nconv;

  if (!s) {
    *d_r = NULL;
    return 0;
  }
  if ((insize = strlen(s)) == 0) {
    *d_r = g_strdup("");
    return 0;
  }
  if (strcasecmp(from, "noconv") == 0) {
    *d_r = g_strdup(s);
    return strlen(s);
  }

  if ((d = malloc(ICONV_OUTPUT_SIZE)) == NULL) {
    fprintf(stderr, "No enough memory.\n");
    if ((*d_r = g_strdup(s)) == NULL)
      return 0;
    return strlen(s);
  }

  if ((cd = iconv_open(to, from)) == (iconv_t)-1) {
    free(d);
    if (errno == EINVAL) {
      *d_r = g_strdup(s);
      return strlen(s);
    }
    perror("xmms_converter_convert");
    return errno;
  }

  inptr = s;
  outptr = d;
  avail = ICONV_OUTPUT_SIZE - 1;

  if ((nconv = iconv(cd, (char **)&inptr, &insize, &outptr, &avail)) != (size_t)-1) {
    *outptr = '\0';
    *d_r = g_strdup(d);
  } else {
    switch (errno) {
    case E2BIG:
      fprintf(stderr, _("Increase ICONV_OUTPUT_SIZE and recompile.\n"));
      break;
    case EILSEQ:
      fprintf(stderr, _("Invalid sequence passed.\n"));
      break;
    case EINVAL:
      fprintf(stderr, _("Incomplete multi-byte sequence passed.\n"));
      break;
    default:
      perror("xmms_converter_convert");
      break;
    }
    *d_r = g_strdup(s);
  }

  iconv_close(cd);
  free(d);

  return strlen(*d_r);
}
#else
# ifdef KANJI
static int kanjicode(char *codename)
{
  if (strcasecmp(codename, "EUC-JP") == 0 || strcasecmp(codename, "EUC") == 0)
    return EUC;
  if (strcasecmp(codename, "SJIS") == 0 || strcasecmp(codename, "Shift_JIS") == 0)
    return SJIS;
  if (strcasecmp(codename, "ISO-2022-JP") == 0 || strcasecmp(codename, "JIS") == 0)
    return JIS;
  fprintf(stderr, __FUNCTION__ ": A codename %s is unknown.\n", codename);

  return NOTDETERMINED;
}

int xmms_converter_convert(gchar *s, gchar **d_r, char *fromname, char *toname)
{
  int from, to, now;

  if (!s) {
    *d_r = NULL;
    return 0;
  }
  if (strlen(s) == 0) {
    *d_r = g_strdup("");
    return 0;
  }

  from = kanjicode(fromname);
  to   = kanjicode(toname);
  kanji_initialize();
  (void)kanji_checkcode_string(s, &now);

  if (now == to || now == ASCII) {
    if ((*d_r = g_strdup(s)) == NULL)
      return 0;
    return strlen(s);
  }

  if (now != from) {
    fprintf(stderr, "now != from(%d != %d)... from = now.\n", now, from);
    from = now;
  }

  /* Usually, I don't write like this... */ 
  switch (from) {
  case EUC:
    switch (to) {
    case SJIS: *d_r = kanji_euc2sjis_string(s); break;
    case JIS:  *d_r = kanji_euc2jis_string(s); break;
    case EUC:
    default:    *d_r = g_strdup(s); break;
    }
    break;
  case SJIS:
    switch (to) {
    case EUC:  *d_r = kanji_sjis2euc_string(s); break;
    case JIS:  *d_r = kanji_sjis2jis_string(s); break;
    case SJIS:
    default:    *d_r = g_strdup(s); break;
    }
    break;
  case JIS:
    switch (to) {
    case EUC:  *d_r = kanji_jis2euc_string(s); break;
    case SJIS: *d_r = kanji_jis2sjis_string(s); break;
    case JIS:
    default:    *d_r = g_strdup(s); break;
    }
    break;
  }

  return strlen(*d_r);
}
# else
int xmms_converter_convert(gchar *s, gchar **d_r, char *from, char *to)
{
  if (!s) {
    *d_r = NULL;
    return 0;
  }

  *d_r = g_strdup(s);
  return strlen(s);
}
# endif
#endif
