/*  XMMS - ALSA output plugin
 *  Copyright (C) 2001-2003 Matthieu Sozeau <mattam@altern.org>
 *  Copyright (C) 1998-2003  Peter Alm, Mikael Alm, Olle Hallnas,
 *                           Thomas Nilsson and 4Front Technologies
 *  Copyright (C) 1999-2003  Haavard Kvaalen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "alsa.h"
#include <ctype.h>
#include <libxmms/xconvert.h>

static snd_pcm_t		*alsa_pcm 	    = NULL;
static snd_pcm_status_t		*alsa_status 	    = NULL;
static snd_pcm_channel_area_t   *areas              = NULL;

static snd_output_t		*logs 		    = NULL;

static snd_pcm_uframes_t         alsa_period_size;
/* Hmm, this is currently bytes per ms */
static int 			 alsa_bps 	    = 0;
static guint64			 alsa_total_written = 0;

/* Set/Get volume */
static snd_mixer_elem_t 	*pcm_element 	    = NULL;
static snd_mixer_t 		*mixer 		    = NULL;

static int                       mmap = 1;
/* Loop */
static int 			 going              = 0;
static gboolean 		 paused 	    = FALSE;
static int                       first_write        = 1;

static gpointer 		 buffer;

static int alsa_can_pause;

struct snd_format {
	unsigned int rate;
	unsigned int channels;
	snd_pcm_format_t format;
	AFormat xmms_format;
};

static struct snd_format *inputf = NULL;
static struct snd_format *effectf = NULL;
static struct snd_format *outputf = NULL;


static int alsa_setup(struct snd_format *f);
static void alsa_mmap_audio(char *data, int length);
static void alsa_write_audio(gpointer data, int length);

static struct snd_format * snd_format_from_xmms(AFormat fmt, int rate, int channels);

static struct xmms_convert_buffers *convertb;

static convert_func_t alsa_convert_func;
static convert_channel_func_t alsa_stereo_convert_func;
static convert_freq_func_t alsa_frequency_convert_func;

static struct {
	AFormat xmms;
	snd_pcm_format_t alsa;
} format_table[] =
{{FMT_S16_LE, SND_PCM_FORMAT_S16_LE},
 {FMT_S16_BE, SND_PCM_FORMAT_S16_BE},
 {FMT_S16_NE,
#ifdef WORDS_BIGENDIAN
  SND_PCM_FORMAT_S16_BE
#else
  SND_PCM_FORMAT_S16_LE
#endif
 },
 {FMT_U16_LE, SND_PCM_FORMAT_U16_LE},
 {FMT_U16_BE, SND_PCM_FORMAT_U16_BE},
 {FMT_U16_NE,
#ifdef WORDS_BIGENDIAN
  SND_PCM_FORMAT_U16_BE
#else
  SND_PCM_FORMAT_U16_LE
#endif
 },
 {FMT_U8, SND_PCM_FORMAT_U8},
 {FMT_S8, SND_PCM_FORMAT_S8},
};


static void debug(char *str, ...)
{
	va_list args;

	if (alsa_cfg.debug)
	{
		va_start(args, str);
		g_logv(NULL, G_LOG_LEVEL_MESSAGE, str, args);
		va_end(args);
	}
}

int alsa_playing(void)
{
	if (!going || paused)
		return FALSE;

	return(snd_pcm_state(alsa_pcm) == SND_PCM_STATE_RUNNING);
}

static void xrun_recover(void)
{
	int err;

	if (alsa_cfg.debug)
	{
		snd_pcm_status_alloca(&alsa_status);
		if ((err = snd_pcm_status(alsa_pcm, alsa_status)) < 0)
			g_warning("xrun_recover(): snd_pcm_status() failed");
		else
		{
			printf("Status:\n");
			snd_pcm_status_dump(alsa_status, logs);
		}
	}

	if (snd_pcm_state(alsa_pcm) == SND_PCM_STATE_XRUN)
	{
		if ((err = snd_pcm_prepare(alsa_pcm)) < 0)
			g_warning("xrun_recover(): snd_pcm_prepare() failed.");
		if (mmap)
			first_write = 1;
	}
}

static snd_pcm_sframes_t alsa_get_avail(void)
{
	snd_pcm_sframes_t ret;
	if ((ret = snd_pcm_avail_update(alsa_pcm)) == -EPIPE)
		xrun_recover();
	else if (ret < 0)
	{
		g_warning("alsa_get_avail(): snd_pcm_avail_update() failed: %s",
			  snd_strerror(-ret));
		return 0;
	}
	else
		return ret;
	if ((ret = snd_pcm_avail_update(alsa_pcm)) < 0)
	{
		g_warning("alsa_get_avail(): snd_pcm_avail_update() failed: %s",
			  snd_strerror(-ret));
		return 0;
	}
	return ret;
}

int alsa_free(void)
{
	snd_pcm_sframes_t avail;

	if (paused)
		return 0;
	else
	{
		avail = alsa_get_avail();
		return snd_pcm_frames_to_bytes(alsa_pcm, avail);
	}
}

void alsa_pause(short p)
{
	debug("alsa_pause");
	paused = p;

	if (alsa_can_pause && !mmap)
		snd_pcm_pause(alsa_pcm, p);
	else
	{
	    if (p)
		    snd_pcm_drop(alsa_pcm);
	    snd_pcm_prepare(alsa_pcm);
	}

	first_write = 1;
}


void alsa_close(void)
{
	int err, started;
	debug("Closing device");

	started = going;
	going = 0;

	pcm_element = NULL;

	if (mixer)
	{
		snd_mixer_close(mixer);
		mixer = NULL;
	}

	if (alsa_pcm != NULL)
	{
		if (started)
			if ((err = snd_pcm_drop(alsa_pcm)) < 0)
				g_warning ("alsa_pcm_drop() failed: %s",
					   snd_strerror(-err));

		if ((err = snd_pcm_close(alsa_pcm)) < 0)
			g_warning ("alsa_pcm_close() failed: %s",
				   snd_strerror(-err));
		alsa_pcm = NULL;
	}

	if (mmap) {
		g_free(buffer);
		buffer = NULL;

		g_free(areas);
		areas = NULL;
	}

	xmms_convert_buffers_destroy(convertb);
	convertb = NULL;
	g_free(inputf);
	inputf = NULL;
	g_free(effectf);
	effectf = NULL;

	debug("Device closed");
}

static void alsa_reopen(struct snd_format *f)
{
	unsigned int tmp = alsa_get_written_time();
	first_write = 1;

	if (alsa_pcm != NULL)
	{
		snd_pcm_close(alsa_pcm);
		alsa_pcm = NULL;
	}

	if (mmap) {
		g_free(buffer);
		buffer = NULL;
		
		g_free(areas);
		areas = NULL;
	}

	if (alsa_setup(f) < 0)
		g_warning("Failed to reopen the audio device");

	alsa_total_written = tmp;
	snd_pcm_prepare(alsa_pcm);
}

void alsa_flush(int time)
{
	alsa_total_written = (guint64) time * alsa_bps;
}

static void parse_mixer_name(char *str, char **name, int *index)
{
	char *end;

	while (isspace(*str))
		str++;

	if ((end = strchr(str, ',')) != NULL)
	{
		*name = g_strndup(str, end - str);
		end++;
		*index = atoi(end);
	}
	else
	{
		*name = g_strdup(str);
		*index = 0;
	}
}

int alsa_setup_mixer(void)
{
	char *dev, *name;
	long int a, b;
	snd_mixer_selem_id_t *selem_id;
	long alsa_min_vol, alsa_max_vol;
	int err, index;

	debug("alsa_setup_mixer");

	dev = g_strdup_printf("hw:%i", alsa_cfg.mixer_card);

	if ((err = snd_mixer_open(&mixer, 0)) < 0)
	{
		g_warning("alsa_setup_mixer(): Failed to open empty mixer: %s",
			  snd_strerror(-err));
		mixer = NULL;
		return -1;
	}
	if ((err = snd_mixer_attach(mixer, dev)) < 0)
	{
		g_warning("alsa_setup_mixer(): Attaching to mixer %s failed: %s",
			  dev, snd_strerror(-err));
		return -1;
	}
	if ((err = snd_mixer_selem_register(mixer, NULL, NULL)) < 0)
	{
		g_warning("alsa_setup_mixer(): Failed to register mixer: %s",
			  snd_strerror(-err));
		return -1;
	}
	if ((err = snd_mixer_load(mixer)) < 0)
	{
		g_warning("alsa_setup_mixer(): Failed to load mixer: %s",
			  snd_strerror(-err));
		return -1;
	}

	snd_mixer_selem_id_alloca(&selem_id);

	parse_mixer_name(alsa_cfg.mixer_device, &name, &index);

	snd_mixer_selem_id_set_index(selem_id, index);
	snd_mixer_selem_id_set_name(selem_id, name);
	g_free(name);

	pcm_element = snd_mixer_find_selem(mixer, selem_id);
	
	if (!pcm_element)
	{
		g_warning("alsa_setup_mixer(): Failed to find mixer element: %s",
			  alsa_cfg.mixer_device);
		return -1;
	}

	snd_mixer_selem_get_playback_volume_range(pcm_element,
						  &alsa_min_vol, &alsa_max_vol);
	snd_mixer_selem_set_playback_volume_range(pcm_element, 0, 100);

	if (alsa_max_vol == 0)
	{
		pcm_element = NULL;
		return -1;
	}

	snd_mixer_selem_get_playback_volume(pcm_element,
					    SND_MIXER_SCHN_FRONT_LEFT, &a);
	snd_mixer_selem_get_playback_volume(pcm_element,
					    SND_MIXER_SCHN_FRONT_RIGHT, &b);

	alsa_set_volume(a * 100 / alsa_max_vol, b * 100 / alsa_max_vol);

	g_free(dev);
	debug("alsa_setup_mixer: end");

	return 0;
}

void alsa_get_volume(int *l, int *r)
{
	static gboolean first = TRUE;

	if (first)
	{
		alsa_setup_mixer();
		first = !first;
	}

	if (!pcm_element)
		return;

	snd_mixer_handle_events (mixer);

	snd_mixer_selem_get_playback_volume(pcm_element,
					    SND_MIXER_SCHN_FRONT_LEFT,
					    (long int *) l);
	snd_mixer_selem_get_playback_volume(pcm_element,
					    SND_MIXER_SCHN_FRONT_RIGHT,
					    (long int *) r);

}



void alsa_set_volume(int l, int r)
{
	if (!pcm_element)
		return;

	snd_mixer_selem_set_playback_volume(pcm_element,
					    SND_MIXER_SCHN_FRONT_LEFT, l);
	snd_mixer_selem_set_playback_volume(pcm_element,
					    SND_MIXER_SCHN_FRONT_RIGHT, r);
}


int alsa_get_output_time(void)
{
	snd_pcm_sframes_t delay;
	ssize_t db = 0;

	if (!going)
		return 0;

	if (!snd_pcm_delay(alsa_pcm, &delay))
		db = snd_pcm_frames_to_bytes(alsa_pcm, delay);
	
	if (db < alsa_total_written)
		return ((alsa_total_written - db) / alsa_bps);
	return 0;
}

int alsa_get_written_time(void)
{
	return (alsa_total_written / alsa_bps);
}

void alsa_write(gpointer data, int length)
{
	EffectPlugin *ep;

	if (paused)
		return;

	if (effects_enabled() && (ep = get_current_effect_plugin()))
	{
		int new_freq = inputf->rate;
		int new_chn = inputf->channels;
		AFormat f = inputf->xmms_format;
		
		if (ep->query_format)
		{
			ep->query_format(&f, &new_freq, &new_chn);
			
			if (f != effectf->xmms_format ||
			    new_freq != effectf->rate ||
			    new_chn != effectf->channels)
			{
				debug("Changing audio format for effect plugin");

				g_free(effectf);
				effectf = snd_format_from_xmms(f, new_freq,
							       new_chn);
				alsa_reopen(effectf);
			}
			
		}
	
		length = ep->mod_samples(&data, length,
					 inputf->xmms_format,
					 inputf->rate,
					 inputf->channels);
	}
	else if (effectf)
	{
		g_free(effectf);
		effectf = NULL;
		effectf = snd_format_from_xmms(inputf->xmms_format,
					       inputf->rate,
					       inputf->channels);
		alsa_reopen(inputf);
	}

	if (alsa_convert_func != NULL)
		length = alsa_convert_func(convertb, &data, length);
	if (alsa_stereo_convert_func != NULL)
		length = alsa_stereo_convert_func(convertb, &data, length);
	if (alsa_frequency_convert_func != NULL)
		length = alsa_frequency_convert_func(convertb, &data, length,
						     effectf->rate,
						     outputf->rate);

	if (mmap)
		alsa_mmap_audio(data, length);
	else
		alsa_write_audio(data, length);
}

static void alsa_write_audio(gpointer data, int length)
{
	snd_pcm_sframes_t written_frames;

	while (length > 0)
	{
		int frames = snd_pcm_bytes_to_frames(alsa_pcm, length);
		written_frames = snd_pcm_writei(alsa_pcm, data, frames);
		
		if (written_frames > 0)
		{
			int written = snd_pcm_frames_to_bytes(alsa_pcm,
							      written_frames);
			alsa_total_written += written;
			length -= written;
			data = (char*) data + written;
		}
		else if (written_frames == -EPIPE)
			xrun_recover();
		else
		{
			g_warning("alsa_write_audio(): write error: %s",
				  snd_strerror(-written_frames));
			break;
		}
	}
}

static void alsa_mmap_audio(char *data, int length)
{
	int cnt = 0, err;
	snd_pcm_uframes_t offset, frames, frame;
	snd_pcm_sframes_t avail;
	const snd_pcm_channel_area_t *chan_areas = areas;
	int channel_offset = 0, channel;
	ssize_t sample_size, offset_bytes, step;

	while (length > 0)
	{
		avail = alsa_get_avail();
			
		if (avail < alsa_period_size)
		{
			if ((err = snd_pcm_wait(alsa_pcm, 1)) < 0)
				g_warning("alsa_mmap_audio(): snd_pcm_wait() "
					  "failed: %s", snd_strerror(-err));
			continue;
		}
			
		frames = snd_pcm_bytes_to_frames(alsa_pcm, length);
		if ((err = snd_pcm_mmap_begin(alsa_pcm, &chan_areas, &offset, &frames) < 0))
			g_warning("alsa_mmap_audio(): snd_pcm_mmap_begin() "
				  "failed: %s", snd_strerror(-err));

		cnt = snd_pcm_frames_to_bytes(alsa_pcm, frames);
					
		sample_size = snd_pcm_samples_to_bytes(alsa_pcm, 1);
		step = chan_areas[0].step / 8;
		offset_bytes = offset * step;

		for(frame = 0; frame < frames; frame++) {
		    for (channel = 0; channel < outputf->channels; channel++) {
			memcpy((char *) chan_areas[channel].addr + chan_areas[channel].first / 8 + offset_bytes,
			       data + channel_offset, sample_size);
			channel_offset += sample_size;
		    }
		    offset_bytes += step;
		}
			
		err = snd_pcm_mmap_commit(alsa_pcm, offset, frames);
		if (err < 0)
			xrun_recover();
		else if (err != frames)
			g_warning("alsa_mmap_audio(): snd_pcm_mmap_commit "
				  "returned %d, expected %d", err, (int)frames);
		
		alsa_total_written += cnt;
		
		length -= cnt;
		
		if (first_write)
		{
			first_write = 0;
			if ((err = snd_pcm_start(alsa_pcm)) < 0)
				g_warning("alsa_mmap_audio(): snd_pcm_start "
					  "failed: %s\n", snd_strerror(-err));
			else
				debug("Stream started\n");
		}
	}
}

int alsa_open(AFormat fmt, int rate, int nch)
{
	debug("Opening device");
	inputf = snd_format_from_xmms(fmt, rate, nch);
	effectf = snd_format_from_xmms(fmt, rate, nch);
	
	if (alsa_cfg.debug)
		snd_output_stdio_attach(&logs, stdout, 0);

	mmap = alsa_cfg.mmap;
	
	if (alsa_setup(inputf) < 0)
	{
		alsa_close();
		return 0;
	}

	if (alsa_setup_mixer() < 0)
	{
		alsa_close();
		return 0;
	}

	convertb = xmms_convert_buffers_new();
	
	alsa_total_written = 0;
	going = first_write = 1;
	paused = FALSE;
	
	snd_pcm_prepare(alsa_pcm);
	
	return 1;
}

static struct snd_format * snd_format_from_xmms(AFormat fmt, int rate, int channels)
{
	struct snd_format *f = g_malloc(sizeof(struct snd_format));
	int i;

	f->xmms_format = fmt;
	f->format = SND_PCM_FORMAT_UNKNOWN;

	for (i = 0; i < sizeof(format_table) / sizeof(format_table[0]); i++)
		if (format_table[i].xmms == fmt)
		{
			f->format = format_table[i].alsa;
			break;
		}
			

	f->rate = rate;
	f->channels = channels;

	return f;
}

static int format_from_alsa(snd_pcm_format_t fmt)
{
	int i;
	for (i = 0; i < sizeof(format_table) / sizeof(format_table[0]); i++)
		if (format_table[i].alsa == fmt)
			return format_table[i].xmms;
	g_warning("Unsupported format: %s", snd_pcm_format_name(fmt));
	return -1;
}

static int alsa_setup(struct snd_format *f)
{
	int err;
	snd_pcm_hw_params_t *hwparams;
	snd_pcm_sw_params_t *swparams;
	int alsa_buffer_time, bits_per_sample;
	unsigned int alsa_period_time;
	snd_pcm_uframes_t alsa_buffer_size;
	char *device;

	debug("alsa_setup");

	alsa_convert_func = NULL;
	alsa_stereo_convert_func = NULL;
	alsa_frequency_convert_func = NULL;

	outputf = snd_format_from_xmms(effectf->xmms_format,
				       effectf->rate,
				       effectf->channels);
	if (alsa_cfg.use_user_device)
		device = g_strdup(alsa_cfg.user_device);
	else
		device = g_strdup_printf("hw:%d,%d", alsa_cfg.audio_card,
					 alsa_cfg.audio_device);

	debug("Opening device: %s\n", device);
	if ((err = snd_pcm_open(&alsa_pcm, device, SND_PCM_STREAM_PLAYBACK, 0)) < 0)
	{
		g_warning("alsa_setup(): Failed to open pcm device (%s): %s",
			  device, snd_strerror(-err));
		g_free(device);
		alsa_pcm = NULL;
		return -1;
	}
	g_free(device);

	if (alsa_cfg.debug)
	{
		snd_pcm_info_t *info;
		int alsa_card, alsa_device, alsa_subdevice;

		snd_pcm_info_alloca(&info);
		snd_pcm_info(alsa_pcm, info);
		alsa_card =  snd_pcm_info_get_card(info);
		alsa_device = snd_pcm_info_get_device(info);
		alsa_subdevice = snd_pcm_info_get_subdevice(info);
		printf("Card %i, Device %i, Subdevice %i\n",
		       alsa_card, alsa_device, alsa_subdevice);
	}

	snd_pcm_hw_params_alloca(&hwparams);

	if ((err = snd_pcm_hw_params_any(alsa_pcm, hwparams)) < 0)
	{
		g_warning("alsa_setup(): No configuration available for "
			  "playback: %s", snd_strerror(-err));
		return -1;
	}
	
	if (mmap &&
	    (err = snd_pcm_hw_params_set_access(alsa_pcm, hwparams,
						SND_PCM_ACCESS_MMAP_INTERLEAVED)) < 0)
	{
		g_message("alsa_setup(): Cannot set mmap'ed mode: %s. "
			  "falling back to direct write", snd_strerror(-err));
		mmap = 0;
	}

	if (!mmap &&
	    (err = snd_pcm_hw_params_set_access(alsa_pcm, hwparams,
						SND_PCM_ACCESS_RW_INTERLEAVED)) < 0)
	{
		g_warning("alsa_setup(): Cannot set direct write mode: %s",
			  snd_strerror(-err));
		return -1;
	}

	if ((err = snd_pcm_hw_params_set_format(alsa_pcm, hwparams, outputf->format)) < 0)
	{
		/*
		 * Try if one of these format work (one of them should work
		 * on almost all soundcards)
		 */
		snd_pcm_format_t formats[] = {SND_PCM_FORMAT_S16_LE,
					      SND_PCM_FORMAT_S16_BE,
					      SND_PCM_FORMAT_U8};
		int i;

		for (i = 0; i < sizeof(formats) / sizeof(formats[0]); i++)
		{
			if (snd_pcm_hw_params_set_format(alsa_pcm, hwparams,
							 formats[i]) == 0)
			{
				outputf->format = formats[i];
				break;
			}
		}
		if (outputf->format != effectf->format)
		{
			outputf->xmms_format =
				format_from_alsa(outputf->format);
			debug("Converting format from %d to %d",
			      effectf->xmms_format, outputf->xmms_format);
			if (outputf->xmms_format < 0)
				return -1;
			alsa_convert_func =
				xmms_convert_get_func(outputf->xmms_format,
						      effectf->xmms_format);
			if (alsa_convert_func == NULL)
				return -1;
		}
		else
		{
			g_warning("alsa_setup(): Sample format not "
				  "available for playback: %s",
				  snd_strerror(-err));
			return -1;
		}
	}

	snd_pcm_hw_params_set_channels_near(alsa_pcm, hwparams, &outputf->channels);
	if (outputf->channels != effectf->channels)
	{
		debug("Converting channels from %d to %d",
		      effectf->channels, outputf->channels);
		alsa_stereo_convert_func =
			xmms_convert_get_channel_func(outputf->xmms_format,
						      outputf->channels,
						      effectf->channels);
		if (alsa_stereo_convert_func == NULL)
			return -1;
	}

	snd_pcm_hw_params_set_rate_near(alsa_pcm, hwparams, &outputf->rate, 0);
	if (outputf->rate == 0)
	{
		g_warning("alsa_setup(): No usable samplerate available.");
		return -1;
	}
	if (outputf->rate != effectf->rate)
	{
		debug("Converting samplerate from %d to %d",
		      effectf->rate, outputf->rate);
		alsa_frequency_convert_func =
			xmms_convert_get_frequency_func(outputf->xmms_format,
							outputf->channels);
		if (alsa_frequency_convert_func == NULL)
			return -1;
	}
	
	alsa_buffer_time = alsa_cfg.buffer_time * 1000;
	if ((err = snd_pcm_hw_params_set_buffer_time_near(alsa_pcm, hwparams,
							  &alsa_buffer_time, 0)) < 0)
	{
		g_warning("alsa_setup(): Set buffer time failed: %s.",
			  snd_strerror(-err));
		return -1;
	}      
	
	alsa_period_time = alsa_cfg.period_time * 1000;
	if ((err = snd_pcm_hw_params_set_period_time_near(alsa_pcm, hwparams,
							  &alsa_period_time, 0)) < 0)
	{
		g_warning("alsa_setup(): Set period time failed: %s.",
			  snd_strerror(-err));
		return -1;
	}

	if (snd_pcm_hw_params(alsa_pcm, hwparams) < 0)
	{
		if (alsa_cfg.debug)
			snd_pcm_hw_params_dump(hwparams, logs);
		g_warning("alsa_setup(): Unable to install hw params");
		return -1;
	}

	if ((err = snd_pcm_hw_params_get_buffer_size(hwparams, &alsa_buffer_size)) < 0)
	{
	    g_warning ("alsa_setup(): snd_pcm_hw_params_get_buffer_size() failed: %s",
		       snd_strerror(-err));
	    return -1;
	}
	
	if ((err = snd_pcm_hw_params_get_period_size(hwparams, &alsa_period_size, 0)) < 0)
	{
	    g_warning ("alsa_setup(): snd_pcm_hw_params_get_period_size() failed: %s",
		       snd_strerror(-err));
	    return -1;
	}
	
	alsa_can_pause = snd_pcm_hw_params_can_pause(hwparams);

	snd_pcm_sw_params_alloca(&swparams);
	snd_pcm_sw_params_current(alsa_pcm, swparams);
	
	if (snd_pcm_sw_params(alsa_pcm, swparams) < 0)
	{
		g_warning("alsa_setup(): Unable to install sw params");
		return -1;
	}

	if (alsa_cfg.debug)
	{
		snd_pcm_sw_params_dump(swparams, logs);
		snd_pcm_dump(alsa_pcm, logs);
	}
	
	bits_per_sample = snd_pcm_format_physical_width(outputf->format);
	alsa_bps = (outputf->rate * bits_per_sample * outputf->channels) / 8000;

	if (mmap)
	{
		int chn;
		buffer = g_malloc(alsa_period_size * bits_per_sample / 8 * outputf->channels);
		
		areas = g_malloc0(outputf->channels * sizeof(snd_pcm_channel_area_t));
		
		for (chn = 0; chn < outputf->channels; chn++)
		{
			areas[chn].addr = buffer;
			areas[chn].first = chn * bits_per_sample;
			areas[chn].step = outputf->channels * bits_per_sample;
		}
	}

	debug("Device setup: buffer time: %i, size: %i frames.", alsa_buffer_time,
	      snd_pcm_frames_to_bytes(alsa_pcm, alsa_buffer_size));
	debug("bits per sample: %i; frame size: %i; Bps: %i",
	      bits_per_sample, snd_pcm_frames_to_bytes(alsa_pcm, 1), alsa_bps);

	return 0;
}
