;;; -*- mode: lisp; coding: utf-8 -*-
;;; 
;;; Copyright (c) 2008 Masayuki Onjo <onjo@lispuser.net>
;;; 
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 
;;;   * Redistributions of source code must retain the above copyright
;;;     notice, this list of conditions and the following disclaimer.
;;; 
;;;   * Redistributions in binary form must reproduce the above
;;;     copyright notice, this list of conditions and the following
;;;     disclaimer in the documentation and/or other materials
;;;     provided with the distribution.
;;; 
;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR 'AS IS' AND ANY EXPRESSED
;;; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

(in-package :jp)

(declaim (inline make-encoding raw-encode raw-decode encode decode))

(defun make-encoding (charset &key (eol-style :lf))
  #+sbcl (declare (ignore eol-style))
  (let ((cs (ecase charset (:ascii  #+(or allegro lispworks sbcl) :ascii
				    #+clisp "ISO-8859-1")
		           (:utf8   #+allegro :utf8
         			    #+(or lispworks sbcl) :utf-8
         			    #+clisp "UTF-8")
         		   (:sjis   #+allegro :shiftjis
         			    #+lispworks :sjis
         			    #+sbcl :cp932
         			    #+clisp "Shift_JIS")
         		   (:euc-jp #+(or allegro lispworks) :euc-jp
         			    #+sbcl :eucjp
         			    #+clisp "EUC-JP")
         		   (:jis    #+(or allegro lispworks)  :jis
         			    #+clisp "ISO-2022-JP")))
 	#-sbcl
	(eol (ecase eol-style (:cr #+allegro #'identity
				   #+lispworks :cr
				   #+clisp :mac)
                              (:lf #+allegro #'identity
				   #+lispworks :lf
				   #+clisp :dos)
			      (:crlf #+allegro #'excl::crlf-base-ef
				     #+lispworks :crlf
				     #+clisp :dos))))
    #+allegro (funcall eol cs)
    #+lispworks `(,cs :eol-style ,eol)
    #+sbcl cs
    #+clisp (ext:make-encoding :charset cs :line-terminator eol)))

(defun raw-encode (string external-format start end null-terminate)
  #+allegro
  (excl:string-to-octets string :external-format external-format :start start :end end :null-terminate null-terminate)
  #+lispworks
  (if null-terminate
      (concatenate '(vector (unsigned-byte 8)) (external-format:encode-lisp-string string external-format :start start :end end) #(0))
      (external-format:encode-lisp-string string external-format :start start :end end))
  #+sbcl
  (sb-ext:string-to-octets string :external-format external-format :start start :end end :null-terminate null-terminate)
  #+clisp
  (if null-terminate
      (concatenate '(vector (unsigned-byte 8)) (ext:convert-string-to-bytes string external-format :start start :end end) #(0))
      (ext:convert-string-to-bytes string external-format :start start :end end)))

(defun raw-decode (usb8-array external-format start end)
  #+allegro
  (excl:octets-to-string usb8-array :external-format external-format :start start :end end)
  #+lispworks
  (external-format:decode-external-string usb8-array external-format :start start :end end)
  #+sbcl
  (sb-ext:octets-to-string usb8-array :external-format external-format :start start :end end)
  #+clisp
  (ext:convert-string-from-bytes usb8-array external-format :start start :end end))

(defun encode (string external-format &key (start 0) end (null-terminate nil))
  "convert string to bytes"
  (raw-encode string external-format start end null-terminate))

(defun decode (vector external-format &key (start 0) end
	       &aux (usb8-array (coerce vector '(vector (unsigned-byte 8)))))
  "convert string from bytes"
  (if (eql external-format :guess)
      (let ((ef (guess usb8-array)))
	(raw-decode usb8-array (make-encoding ef) start end))
      (raw-decode usb8-array external-format start end)))
