/*
 * kernel/power/suspend_swap.c
 *
 * Copyright 2004-2006 Nigel Cunningham <nigel@suspend2.net>
 *
 * Distributed under GPLv2.
 * 
 * This file encapsulates functions for usage of swap space as a
 * backing store.
 */

#include <linux/suspend.h>
#include <linux/module.h>
#include <linux/blkdev.h>
#include <linux/swapops.h>
#include <linux/swap.h>

#include "suspend.h"
#include "sysfs.h"
#include "modules.h"
#include "io.h"
#include "ui.h"
#include "extent.h"
#include "block_io.h"

static struct suspend_module_ops swapwriterops;

#define SIGNATURE_VER 6

/* --- Struct of pages stored on disk */

union diskpage {
	union swap_header swh;	/* swh.magic is the only member used */
};

union p_diskpage {
	union diskpage *pointer;
	char *ptr;
        unsigned long address;
};

/* Devices used for swap */
static struct suspend_bdev_info devinfo[MAX_SWAPFILES];

/* Extent chains for swap & blocks */
struct extent_chain swapextents;
struct extent_chain block_chain[MAX_SWAPFILES];

static dev_t header_dev_t;
static struct block_device *header_block_device;
static unsigned long headerblock;

/* For swapfile automatically swapon/off'd. */
static char swapfilename[SWAP_FILENAME_MAXLENGTH] = "";
extern asmlinkage long sys_swapon(const char *specialfile, int swap_flags);
extern asmlinkage long sys_swapoff(const char *specialfile);
static int suspend_swapon_status;

/* Header Page Information */
static int header_pages_allocated;

/* User Specified Parameters. */

static unsigned long resume_firstblock;
static int resume_blocksize;
static dev_t resume_dev_t;
static struct block_device *resume_block_device;

struct sysinfo swapinfo;
static int swapwriter_invalidate_image(void);

/* Block devices open. */
struct bdev_opened
{
	dev_t device;
	struct block_device *bdev;
	int claimed;
};

/* 
 * Entry MAX_SWAPFILES is the resume block device, which may
 * not be a swap device enabled when we suspend.
 * Entry MAX_SWAPFILES + 1 is the header block device, which
 * is needed before we find out which slot it occupies.
 */
static struct bdev_opened *bdev_info_list[MAX_SWAPFILES + 2];
       
static void close_bdev(int i)
{
	struct bdev_opened *this = bdev_info_list[i];

	if (this->claimed)
		bd_release(this->bdev);

	/* Release our reference. */
	blkdev_put(this->bdev);

	/* Free our info. */
	kfree(this);

	bdev_info_list[i] = NULL;
}

static void close_bdevs(void)
{
	int i;

	for (i = 0; i < MAX_SWAPFILES; i++)
		if (bdev_info_list[i])
			close_bdev(i);

	resume_block_device = header_block_device = NULL;
}

static struct block_device *open_bdev(int index, dev_t device)
{
	struct bdev_opened *this;
	struct block_device *bdev;

	if (bdev_info_list[index] && (bdev_info_list[index]->device == device)){
		bdev = bdev_info_list[index]->bdev;
		return bdev;
	}
	
	if (bdev_info_list[index] && bdev_info_list[index]->device != device)
		close_bdev(index);

	bdev = open_by_devnum(device, FMODE_READ);

	if (IS_ERR(bdev) || !bdev) {
		suspend_early_boot_message(1,SUSPEND_CONTINUE_REQ,  
				"Failed to get access to block device "
				"\"%s\" (error %d).\n Maybe you need "
				"to run mknod and/or lvmsetup in an "
				"initrd/ramfs?", device, bdev);
		return ERR_PTR(-EINVAL);
	}

	this = kmalloc(sizeof(struct bdev_opened), GFP_KERNEL);
	BUG_ON(!this);

	bdev_info_list[index] = this;
	this->device = device;
	this->bdev = bdev;

	if (index < MAX_SWAPFILES)
		devinfo[index].bdev = bdev;

	return bdev;
}

/* Must be silent - might be called from cat /sys/power/suspend2/debug_info
 * Returns 0 if was off, -EBUSY if was on, error value otherwise.
 */
static int enable_swapfile(void)
{
	int activateswapresult = -EINVAL;

	if (suspend_swapon_status)
		return 0;

	if (swapfilename[0]) {
		/* Attempt to swap on with maximum priority */
		activateswapresult = sys_swapon(swapfilename, 0xFFFF);
		if ((activateswapresult) && (activateswapresult != -EBUSY))
			printk(name_suspend
				"The swapfile/partition specified by "
				"/sys/power/suspend2/swapwriter/swapfile (%s) could not"
				" be turned on (error %d). Attempting "
				"to continue.\n",
				swapfilename, activateswapresult);
		if (!activateswapresult)
			suspend_swapon_status = 1;
	}
	return activateswapresult;
}

/* Returns 0 if was on, -EINVAL if was off, error value otherwise */
static int disable_swapfile(void)
{
	int result = -EINVAL;
	
	if (!suspend_swapon_status)
		return 0;

	if (swapfilename[0]) {
		result = sys_swapoff(swapfilename);
		if (result == -EINVAL)
	 		return 0;	/* Wasn't on */
		if (!result)
			suspend_swapon_status = 0;
	}

	return result;
}

static int try_to_parse_resume_device(char *commandline)
{
	struct kstat stat;
	int error;

	resume_dev_t = name_to_dev_t(commandline);

	if (!resume_dev_t) {
		error = vfs_stat(commandline, &stat);
		if (!error)
			resume_dev_t = stat.rdev;
	}

	if (!resume_dev_t) {
		if (test_suspend_state(SUSPEND_TRYING_TO_RESUME))
			suspend_early_boot_message(1, SUSPEND_CONTINUE_REQ,
			  "Failed to translate \"%s\" into a device id.\n",
			  commandline);
		else
			printk(name_suspend
			  "Can't translate \"%s\" into a device id yet.\n",
			  commandline);
		return 1;
	}

	if (IS_ERR(resume_block_device =
	     open_bdev(MAX_SWAPFILES, resume_dev_t))) {
		suspend_early_boot_message(1, SUSPEND_CONTINUE_REQ,
			"Failed to get access to \"%s\", where"
			" the swap header should be found.",
			commandline);
		return 1;
	}

	return 0;
}

/* 
 * If we have read part of the image, we might have filled  memory with
 * data that should be zeroed out.
 */
static void swapwriter_noresume_reset(void)
{
	memset((char *) &devinfo, 0, sizeof(devinfo));
	close_bdevs();
}

static int parse_signature(char *header, int restore)
{
	int type = -1;

	if (!memcmp("SWAP-SPACE",header,10))
		return 0;
	else if (!memcmp("SWAPSPACE2",header,10))
		return 1;

	else if (!memcmp("S1SUSP",header,6))
		type = 2;
	else if (!memcmp("S2SUSP",header,6))
		type = 3;
	else if (!memcmp("S1SUSPEND",header,9))
		type = 4;
	
	else if (!memcmp("z",header,1))
		type = 12;
	else if (!memcmp("Z",header,1))
		type = 13;
	
	/* 
	 * Put bdev of suspend header in last byte of swap header
	 * (unsigned short)
	 */
	if (type > 11) {
		dev_t *header_ptr = (dev_t *) &header[1];
		unsigned char *headerblocksize_ptr =
			(unsigned char *) &header[5];
		u32 *headerblock_ptr = (u32 *) &header[6];
		header_dev_t = *header_ptr;
		/* 
		 * We are now using the highest bit of the char to indicate
		 * whether we have attempted to resume from this image before.
		 */
		clear_suspend_state(SUSPEND_RESUMED_BEFORE);
		if (((int) *headerblocksize_ptr) & 0x80)
			set_suspend_state(SUSPEND_RESUMED_BEFORE);
		headerblock = (unsigned long) *headerblock_ptr;
	}

	if ((restore) && (type > 5)) {
		/* We only reset our own signatures */
		if (type & 1)
			memcpy(header,"SWAPSPACE2",10);
		else
			memcpy(header,"SWAP-SPACE",10);
	}

	return type;
}

/*
 * prepare_signature
 */
static int prepare_signature(dev_t bdev, unsigned long block,
		char *current_header)
{
	int current_type = parse_signature(current_header, 0);
	dev_t *header_ptr = (dev_t *) (&current_header[1]);
	unsigned long *headerblock_ptr =
		(unsigned long *) (&current_header[6]);

	if ((current_type > 1) && (current_type < 6))
		return 1;

	/* At the moment, I don't have a way to handle the block being
	 * > 32 bits. Not enough room in the signature and no way to
	 * safely put the data elsewhere. */

	if (BITS_PER_LONG == 64 && ffs(block) > 31) {
		suspend_prepare_status(DONT_CLEAR_BAR,
			"Header sector requires 33+ bits. "
			"Would not be able to resume.");
		return 1;
	}

	if (current_type & 1)
		current_header[0] = 'Z';
	else
		current_header[0] = 'z';
	*header_ptr = bdev;
	/* prev is the first/last swap page of the resume area */
	*headerblock_ptr = (unsigned long) block; 
	return 0;
}

static int swapwriter_allocate_storage(int space_requested);

static int swapwriter_allocate_header_space(int space_requested)
{
	int i;

	if (!swapextents.size)
		swapwriter_allocate_storage(space_requested);

	suspend_extent_state_goto_start(&suspend_writer_posn);
	suspend_bio_ops.forward_one_page(); /* To first page */
	
	for (i = 0; i < space_requested; i++) {
		if (suspend_bio_ops.forward_one_page()) {
			printk("Out of space while seeking to allocate "
					"header pages,\n");
			header_pages_allocated = i;
			return -ENOSPC;
		}

	}

	header_pages_allocated = space_requested;

	/* The end of header pages will be the start of pageset 2;
	 * we are now sitting on the first pageset2 page. */
	suspend_extent_state_save(&suspend_writer_posn,
			&suspend_writer_posn_save[2]);
	return 0;
}

static void get_main_pool_phys_params(void)
{
	struct extent *extentpointer = NULL;
	unsigned long address;
	int i, extent_min = -1, extent_max = -1, last_chain = -1;

	for (i = 0; i < MAX_SWAPFILES; i++)
		if (block_chain[i].first)
			suspend_put_extent_chain(&block_chain[i]);

	suspend_extent_for_each(&swapextents, extentpointer, address) {
		swp_entry_t swap_address = extent_val_to_swap_entry(address);
		pgoff_t offset = swp_offset(swap_address);
		unsigned swapfilenum = swp_type(swap_address);
		struct swap_info_struct *sis = get_swap_info_struct(swapfilenum);
		sector_t new_sector = map_swap_page(sis, offset);

		if ((new_sector == extent_max + 1) &&
		    (last_chain == swapfilenum))
			extent_max++;
		else {
			if (extent_min > -1) {
				if (test_action_state(SUSPEND_TEST_BIO))
					printk("Adding extent chain %d %d-%d.\n",
						swapfilenum,
						extent_min <<
						 devinfo[last_chain].bmap_shift,
						extent_max <<
						 devinfo[last_chain].bmap_shift);
						
				suspend_add_to_extent_chain(
					&block_chain[last_chain],
					extent_min, extent_max);
			}
			extent_min = extent_max = new_sector;
			last_chain = swapfilenum;
		}
	}

	if (extent_min > -1) {
		if (test_action_state(SUSPEND_TEST_BIO))
			printk("Adding extent chain %d %d-%d.\n",
				last_chain,
				extent_min <<
					devinfo[last_chain].bmap_shift,
				extent_max <<
					devinfo[last_chain].bmap_shift);
		suspend_add_to_extent_chain(
			&block_chain[last_chain],
			extent_min, extent_max);
	}

	swapwriter_allocate_header_space(header_pages_allocated);
}

static int swapwriter_storage_allocated(void)
{
	return swapextents.size;
}

static int swapwriter_storage_available(void)
{
	si_swapinfo(&swapinfo);
	return swapinfo.freeswap + swapwriter_storage_allocated();
}

static int swapwriter_initialise(int starting_cycle)
{
	if (!starting_cycle)
		return 0;

	enable_swapfile();

	if (resume_dev_t && !resume_block_device &&
	    IS_ERR(resume_block_device =
	    		open_bdev(MAX_SWAPFILES, resume_dev_t)))
		return 1;
	
	return 0;
}

static void swapwriter_cleanup(int ending_cycle)
{
	if (ending_cycle)
		disable_swapfile();
	
	close_bdevs();
}

static int swapwriter_release_storage(void)
{
	int i = 0;

	if (test_action_state(SUSPEND_KEEP_IMAGE) &&
	    test_suspend_state(SUSPEND_NOW_RESUMING))
		return 0;

	header_pages_allocated = 0;

	if (swapextents.first) {
		/* Free swap entries */
		struct extent *extentpointer;
		unsigned long extentvalue;
		swp_entry_t entry;
		suspend_extent_for_each(&swapextents, extentpointer, 
				extentvalue) {
			entry = extent_val_to_swap_entry(extentvalue);
			swap_free(entry);
		}

		suspend_put_extent_chain(&swapextents);

		for (i = 0; i < MAX_SWAPFILES; i++)
			if (block_chain[i].first)
				suspend_put_extent_chain(&block_chain[i]);
	}

	return 0;
}

/* 
 * Round robin allocation (where swap storage has the same priority).
 * could make this very inefficient, so we track extents allocated on
 * a per-swapfiles basis.
 */
static int swapwriter_allocate_storage(int space_requested)
{
	int i, result = 0, first[MAX_SWAPFILES];
	int pages_to_get = space_requested - swapextents.size;
	unsigned long extent_min[MAX_SWAPFILES], extent_max[MAX_SWAPFILES];

	if (pages_to_get < 1)
		return 0;

	for (i=0; i < MAX_SWAPFILES; i++) {
		struct swap_info_struct *si = get_swap_info_struct(i);
		if ((devinfo[i].bdev = si->bdev))
			devinfo[i].dev_t = si->bdev->bd_dev;
		devinfo[i].bmap_shift = 3;
		devinfo[i].blocks_per_page = 1;
		first[i] = 1;
	}

	for(i=0; i < pages_to_get; i++) {
		swp_entry_t entry;
		unsigned long new_value;
		unsigned swapfilenum;

		entry = get_swap_page();
		if (!entry.val) {
			printk("Failed to get a swap page.\n");
			result = -ENOSPC;
			break;
		}

		swapfilenum = swp_type(entry);
		new_value = swap_entry_to_extent_val(entry);
		if (first[swapfilenum]) {
			first[swapfilenum] = 0;
			extent_min[swapfilenum] = extent_max[swapfilenum] =
				new_value;
		} else {
			if (new_value == extent_max[swapfilenum] + 1)
				extent_max[swapfilenum]++;
			else {
				suspend_add_to_extent_chain(
					&swapextents,
					extent_min[swapfilenum],
					extent_max[swapfilenum]);
				extent_min[swapfilenum] =
					extent_max[swapfilenum] = new_value;
			}
		}
	}

	for (i = 0; i < MAX_SWAPFILES; i++)
		if (!first[i])
			suspend_add_to_extent_chain(
				&swapextents,
				extent_min[i], extent_max[i]);

	get_main_pool_phys_params();
	return result;
}

static int swapwriter_write_header_init(void)
{
	int i, result;
	struct swap_info_struct *si;

	suspend_header_bytes_used = 0;

	suspend_extent_state_goto_start(&suspend_writer_posn);
	/* Forward one page will be done prior to the read */

	for (i = 0; i < MAX_SWAPFILES; i++) {
		si = get_swap_info_struct(i);
		if (si->swap_file)
			devinfo[i].dev_t = si->bdev->bd_dev;
		else
			devinfo[i].dev_t = (dev_t) 0;
	}

	suspend_writer_buffer = (char *) get_zeroed_page(GFP_ATOMIC);
	if (!suspend_writer_buffer) {
		printk("Failed to get swapwriter buffer.\n");
		return -ENOMEM;
	}

	suspend_writer_buffer_posn = 0;

	/* Info needed to bootstrap goes at the start of the header.
	 * First we save the positions and devinfo, including the number
	 * of header pages. Then we save the structs containing data needed
	 * for reading the header pages back.
	 * Note that even if header pages take more than one page, when we
	 * read back the info, we will have restored the location of the
	 * next header page by the time we go to use it.
	 */

	if ((result = suspend_bio_ops.rw_header_chunk(WRITE,
			&swapwriterops,
			(char *) &suspend_writer_posn_save, 
			sizeof(suspend_writer_posn_save))))
		return result;

	if ((result = suspend_bio_ops.rw_header_chunk(WRITE,
			&swapwriterops,
			(char *) &devinfo, sizeof(devinfo))))
		return result;

	for (i=0; i < MAX_SWAPFILES; i++)
		suspend_serialise_extent_chain(&swapwriterops, &block_chain[i]);

	return 0;
}

static int swapwriter_write_header_cleanup(void)
{
	int result;
	struct swap_info_struct *si;

	/* Write any unsaved data */
	if (suspend_writer_buffer_posn)
		suspend_bio_ops.write_header_chunk_finish();

	suspend_bio_ops.finish_all_io();

	suspend_extent_state_goto_start(&suspend_writer_posn);
	suspend_bio_ops.forward_one_page();

	/* Adjust swap header */
	suspend_bio_ops.bdev_page_io(READ, resume_block_device,
			resume_firstblock,
			virt_to_page(suspend_writer_buffer));

	si = get_swap_info_struct(suspend_writer_posn.current_chain);
	result = prepare_signature(si->bdev->bd_dev,
			suspend_writer_posn.current_offset,
		((union swap_header *) suspend_writer_buffer)->magic.magic);
		
	if (!result)
		suspend_bio_ops.bdev_page_io(WRITE, resume_block_device,
			resume_firstblock,
			virt_to_page(suspend_writer_buffer));

	free_page((unsigned long) suspend_writer_buffer);
	suspend_writer_buffer = NULL;
	
	suspend_bio_ops.finish_all_io();

	return result;
}

/* ------------------------- HEADER READING ------------------------- */

/*
 * read_header_init()
 * 
 * Description:
 * 1. Attempt to read the device specified with resume2=.
 * 2. Check the contents of the swap header for our signature.
 * 3. Warn, ignore, reset and/or continue as appropriate.
 * 4. If continuing, read the swapwriter configuration section
 *    of the header and set up block device info so we can read
 *    the rest of the header & image.
 *
 * Returns:
 * May not return if user choose to reboot at a warning.
 * -EINVAL if cannot resume at this time. Booting should continue
 * normally.
 */

static int swapwriter_read_header_init(void)
{
	int i;

	BUG_ON(!resume_block_device);
	BUG_ON(!resume_dev_t);

	suspend_header_bytes_used = 0;

	suspend_writer_buffer = (char *) get_zeroed_page(GFP_ATOMIC);

	BUG_ON(!suspend_writer_buffer);

	if (!header_dev_t) {
		printk("read_header_init called when we haven't "
				"verified there is an image!\n");
		return -EINVAL;
	}

	/* 
	 * If the header is not on the resume_dev_t, get the resume device first.
	 */
	if (header_dev_t != resume_dev_t) {
		header_block_device = open_bdev(MAX_SWAPFILES + 1,
				header_dev_t);

		if (IS_ERR(header_block_device))
			return PTR_ERR(header_block_device);
	} else
		header_block_device = resume_block_device;

	/* 
	 * Read swapwriter configuration.
	 * Headerblock size taken into account already.
	 */
	suspend_bio_ops.bdev_page_io(READ, header_block_device,
			headerblock << 3,
			virt_to_page((unsigned long) suspend_writer_buffer));

	memcpy(&suspend_writer_posn_save, suspend_writer_buffer, 3 * sizeof(struct extent_iterate_saved_state));

	suspend_writer_buffer_posn = 3 * sizeof(struct extent_iterate_saved_state);
	suspend_header_bytes_used += 3 * sizeof(struct extent_iterate_saved_state);

	memcpy(&devinfo, suspend_writer_buffer + suspend_writer_buffer_posn, sizeof(devinfo));

	suspend_writer_buffer_posn += sizeof(devinfo);
	suspend_header_bytes_used += sizeof(devinfo);

	/* Restore device info */
	for (i = 0; i < MAX_SWAPFILES; i++) {
		dev_t thisdevice = devinfo[i].dev_t;
		struct block_device *result;

		devinfo[i].bdev = NULL;

		if (!thisdevice)
			continue;

		if (thisdevice == resume_dev_t) {
			devinfo[i].bdev = resume_block_device;
			bdev_info_list[i] = bdev_info_list[MAX_SWAPFILES];
			BUG_ON(!bdev_info_list[i]);
			bdev_info_list[MAX_SWAPFILES] = NULL;
			continue;
		}

		if (thisdevice == header_dev_t) {
			devinfo[i].bdev = header_block_device;
			bdev_info_list[i] = bdev_info_list[MAX_SWAPFILES + 1];
			BUG_ON(!bdev_info_list[i]);
			bdev_info_list[MAX_SWAPFILES + 1] = NULL;
			continue;
		}

		result = open_bdev(i, thisdevice);
		if (IS_ERR(result)) {
			close_bdevs();
			return PTR_ERR(result);
		}
	}

	suspend_extent_state_goto_start(&suspend_writer_posn);
	suspend_bio_ops.set_extra_page_forward();

	for (i = 0; i < MAX_SWAPFILES; i++)
		suspend_load_extent_chain(&block_chain[i]);

	return 0;
}

static int swapwriter_read_header_cleanup(void)
{
	free_page((unsigned long) suspend_writer_buffer);
	return 0;
}

/* swapwriter_invalidate_image
 * 
 */
static int swapwriter_invalidate_image(void)
{
	union p_diskpage cur;
	int result = 0;
	char newsig[11];
	
	cur.address = get_zeroed_page(GFP_ATOMIC);
	if (!cur.address) {
		printk("Unable to allocate a page for restoring the swap signature.\n");
		return -ENOMEM;
	}

	/*
	 * If nr_suspends == 0, we must be booting, so no swap pages
	 * will be recorded as used yet.
	 */

	if (nr_suspends > 0)
		swapwriter_release_storage();

	/* 
	 * We don't do a sanity check here: we want to restore the swap 
	 * whatever version of kernel made the suspend image.
	 * 
	 * We need to write swap, but swap may not be enabled so
	 * we write the device directly
	 */
	
	suspend_bio_ops.bdev_page_io(READ, resume_block_device,
			resume_firstblock,
			virt_to_page(cur.pointer));

	result = parse_signature(cur.pointer->swh.magic.magic, 1);
		
	if (result < 5)
		goto out;

	strncpy(newsig, cur.pointer->swh.magic.magic, 10);
	newsig[10] = 0;

	suspend_bio_ops.bdev_page_io(WRITE, resume_block_device,
			resume_firstblock,
			virt_to_page(cur.pointer));

	if (!nr_suspends)
		printk(KERN_WARNING name_suspend "Image invalidated.\n");
out:
	suspend_bio_ops.finish_all_io();
	free_page(cur.address);
	return 0;
}

/*
 * workspace_size
 *
 * Description:
 * Returns the number of bytes of RAM needed for this
 * code to do its work. (Used when calculating whether
 * we have enough memory to be able to suspend & resume).
 *
 */
static unsigned long swapwriter_memory_needed(void)
{
	return 1;
}

/*
 * Print debug info
 *
 * Description:
 */
static int swapwriter_print_debug_stats(char *buffer, int size)
{
	int len = 0;
	struct sysinfo sysinfo;
	
	if (suspend_active_writer != &swapwriterops) {
		len = snprintf_used(buffer, size, "- Swapwriter inactive.\n");
		return len;
	}

	len = snprintf_used(buffer, size, "- Swapwriter active.\n");
	if (swapfilename[0])
		len+= snprintf_used(buffer+len, size-len,
			"  Attempting to automatically swapon: %s.\n", swapfilename);

	si_swapinfo(&sysinfo);
	
	len+= snprintf_used(buffer+len, size-len, "  Swap available for image: %ld pages.\n",
			sysinfo.freeswap + swapwriter_storage_allocated());

	return len;
}

/*
 * Storage needed
 *
 * Returns amount of space in the swap header required
 * for the swapwriter's data. This ignores the links between
 * pages, which we factor in when allocating the space.
 *
 * We ensure the space is allocated, but actually save the
 * data from write_header_init and therefore don't also define a
 * save_config_info routine.
 */
static unsigned long swapwriter_storage_needed(void)
{
	int i, result;
	result = sizeof(suspend_writer_posn_save) + sizeof(devinfo);

	for (i = 0; i < MAX_SWAPFILES; i++) {
		result += 3 * sizeof(int);
		result += (2 * sizeof(unsigned long) * 
			(block_chain[i].allocs - block_chain[i].frees));
	}

	return result;
}

/*
 * Image_exists
 */
static int swapwriter_image_exists(void)
{
	int signature_found;
	union p_diskpage diskpage;
	
	if (!resume_dev_t) {
		printk("Not even trying to read header "
				"because resume_dev_t is not set.\n");
		return 0;
	}
	
	if (!resume_block_device &&
	    IS_ERR(resume_block_device = open_bdev(MAX_SWAPFILES, resume_dev_t))) {
		printk("Failed to open resume dev_t (%x).\n", resume_dev_t);
		return 0;
	}

	diskpage.address = get_zeroed_page(GFP_ATOMIC);

	suspend_bio_ops.bdev_page_io(READ, resume_block_device,
			resume_firstblock,
			virt_to_page(diskpage.ptr));
	suspend_bio_ops.finish_all_io();

	signature_found = parse_signature(diskpage.pointer->swh.magic.magic, 0);
	free_page(diskpage.address);

	if (signature_found < 2) {
		printk(name_suspend "Normal swapspace found.\n");
		return 0;	/* Normal swap space */
	} else if (signature_found == -1) {
		printk(KERN_ERR name_suspend
			"Unable to find a signature. Could you have moved "
			"a swap file?\n");
		return 0;
	} else if (signature_found < 6) {
		printk(name_suspend "Dectected another implementation's signature.\n");
		return 0;
	} else if ((signature_found >> 1) != SIGNATURE_VER) {
		if ((!(test_suspend_state(SUSPEND_NORESUME_SPECIFIED))) &&
			suspend_early_boot_message(1, SUSPEND_CONTINUE_REQ,
			 "Found a different style suspend image signature.")) {
			set_suspend_state(SUSPEND_NORESUME_SPECIFIED);
			printk(name_suspend "Dectected another implementation's signature.\n");
		}
	}

	return 1;
}

/*
 * Mark resume attempted.
 *
 * Record that we tried to resume from this image.
 */
static void swapwriter_mark_resume_attempted(void)
{
	union p_diskpage diskpage;
	int signature_found;
	
	if (!resume_dev_t) {
		printk("Not even trying to record attempt at resuming"
				" because resume_dev_t is not set.\n");
		return;
	}
	
	diskpage.address = get_zeroed_page(GFP_ATOMIC);

	suspend_bio_ops.bdev_page_io(READ, resume_block_device,
			resume_firstblock,
			virt_to_page(diskpage.ptr));
	signature_found = parse_signature(diskpage.pointer->swh.magic.magic, 0);

	switch (signature_found) {
		case 12:
		case 13:
			diskpage.pointer->swh.magic.magic[5] |= 0x80;
			break;
	}
	
	suspend_bio_ops.bdev_page_io(WRITE, resume_block_device,
			resume_firstblock,
			virt_to_page(diskpage.ptr));
	suspend_bio_ops.finish_all_io();
	free_page(diskpage.address);
	
	close_bdevs();
	return;
}

/*
 * Parse Image Location
 *
 * Attempt to parse a resume2= parameter.
 * Swap Writer accepts:
 * resume2=swap:DEVNAME[:FIRSTBLOCK][@BLOCKSIZE]
 *
 * Where:
 * DEVNAME is convertable to a dev_t by name_to_dev_t
 * FIRSTBLOCK is the location of the first block in the swap file
 * (specifying for a swap partition is nonsensical but not prohibited).
 * Data is validated by attempting to read a swap header from the
 * location given. Failure will result in swapwriter refusing to
 * save an image, and a reboot with correct parameters will be
 * necessary.
 */
static int swapwriter_parse_sig_location(char *commandline, int only_writer)
{
	char *thischar, *devstart, *colon = NULL, *at_symbol = NULL;
	union p_diskpage diskpage;
	int signature_found, result = -EINVAL, temp_result;

	if (strncmp(commandline, "swap:", 5)) {
		/* 
		 * Failing swap:, we'll take a simple
		 * resume2=/dev/hda2, but fall through to
		 * other writers if /dev/ isn't matched.
		 */
		if (strncmp(commandline, "/dev/", 5))
			return 1;
	} else
		commandline += 5;

	devstart = thischar = commandline;
	while ((*thischar != ':') && (*thischar != '@') &&
		((thischar - commandline) < 250) && (*thischar))
		thischar++;

	if (*thischar == ':') {
		colon = thischar;
		*colon = 0;
		thischar++;
	}

	while ((*thischar != '@') && ((thischar - commandline) < 250) && (*thischar))
		thischar++;

	if (*thischar == '@') {
		at_symbol = thischar;
		*at_symbol = 0;
	}
	
	if (colon)
		resume_firstblock = (int) simple_strtoul(colon + 1, NULL, 0);
	else
		resume_firstblock = 0;

	clear_suspend_state(SUSPEND_CAN_SUSPEND);
	clear_suspend_state(SUSPEND_CAN_RESUME);
	
	/* Legacy */
	if (at_symbol) {
		resume_blocksize = (int) simple_strtoul(at_symbol + 1, NULL, 0);
		if (resume_blocksize & (SECTOR_SIZE - 1)) {
			printk("Swapwriter: Blocksizes are multiples of %d!\n", SECTOR_SIZE);
			return -EINVAL;
		}
		resume_firstblock = resume_firstblock * (resume_blocksize / SECTOR_SIZE);
	}
	
	temp_result = try_to_parse_resume_device(devstart);

	if (colon)
		*colon = ':';
	if (at_symbol)
		*at_symbol = '@';

	if (temp_result)
		return -EINVAL;

	diskpage.address = get_zeroed_page(GFP_ATOMIC);
	if (!diskpage.address) {
		printk(KERN_ERR name_suspend "Swapwriter: Failed to allocate a diskpage for I/O.\n");
		return -ENOMEM;
	}

	temp_result = suspend_bio_ops.bdev_page_io(READ,
			resume_block_device,
			resume_firstblock,
			virt_to_page(diskpage.ptr));

	suspend_bio_ops.finish_all_io();
	
	if (temp_result) {
		printk(KERN_ERR name_suspend "Swapwriter: Failed to submit I/O.\n");
		goto invalid;
	}
	
	signature_found = parse_signature(diskpage.pointer->swh.magic.magic, 0);

	if (signature_found != -1) {
		printk(name_suspend "Swapwriter: Signature found.\n");
		result = 0;

		suspend_bio_ops.set_devinfo(devinfo);
		suspend_writer_posn.chains = &block_chain[0];
		suspend_writer_posn.num_chains = MAX_SWAPFILES;
		set_suspend_state(SUSPEND_CAN_SUSPEND);
		set_suspend_state(SUSPEND_CAN_RESUME);
	} else
		printk(KERN_ERR name_suspend "Swapwriter: No swap signature found at specified location.\n");
invalid:
	free_page((unsigned long) diskpage.address);
	return result;

}

static int header_locations_read_sysfs(const char *page, int count)
{
	int i, printedpartitionsmessage = 0, len = 0, haveswap = 0;
	struct inode *swapf = 0;
	int zone;
	char *path_page = (char *) __get_free_page(GFP_KERNEL);
	char *path, *output = (char *) page;
	int path_len;
	
	if (!page)
		return 0;

	for (i = 0; i < MAX_SWAPFILES; i++) {
		struct swap_info_struct *si =  get_swap_info_struct(i);

		if (!si->swap_file)
			continue;
		
		if (S_ISBLK(si->swap_file->f_mapping->host->i_mode)) {
			haveswap = 1;
			if (!printedpartitionsmessage) {
				len += sprintf(output + len, 
					"For swap partitions, simply use the format: resume2=swap:/dev/hda1.\n");
				printedpartitionsmessage = 1;
			}
		} else {
			path_len = 0;
			
			path = d_path(si->swap_file->f_dentry,
				si->swap_file->f_vfsmnt,
				path_page,
				PAGE_SIZE);
			path_len = snprintf(path_page, 31, "%s", path);
			
			haveswap = 1;
			swapf = si->swap_file->f_mapping->host;
			if (!(zone = bmap(swapf,0))) {
				len+= sprintf(output + len, 
					"Swapfile %s has been corrupted. Reuse mkswap on it and try again.\n",
					path_page);
			} else {
				char name_buffer[255];
				len+= sprintf(output + len, "For swapfile `%s`, use resume2=swap:/dev/%s:0x%x.\n",
						path_page,
						bdevname(si->bdev, name_buffer),
						zone << (swapf->i_blkbits - 9));
			}

		}
	}
	
	if (!haveswap)
		len = sprintf(output, "You need to turn on swap partitions before examining this file.\n");

	free_page((unsigned long) path_page);
	return len;
}

static struct suspend_sysfs_data swapwriter_sysfs_data[] = {
	{
	 SUSPEND2_ATTR("swapfilename", SYSFS_RW),
	 SYSFS_STRING(swapfilename, 255, SYSFS_SM_NOT_NEEDED)
	},

	{
	 SUSPEND2_ATTR("headerlocations", SYSFS_READONLY),
	 SYSFS_CUSTOM(header_locations_read_sysfs, NULL, SYSFS_SM_NOT_NEEDED)
	},

	{ SUSPEND2_ATTR("enabled", SYSFS_RW),
	  SYSFS_INT(&swapwriterops.enabled, 0, 1),
	  .write_side_effect		= attempt_to_parse_resume_device2,
	}
};

static struct suspend_module_ops swapwriterops = {
	.type					= WRITER_MODULE,
	.name					= "Swap Writer",
	.module					= THIS_MODULE,
	.memory_needed				= swapwriter_memory_needed,
	.print_debug_info			= swapwriter_print_debug_stats,
	.storage_needed				= swapwriter_storage_needed,
	.initialise				= swapwriter_initialise,
	.cleanup				= swapwriter_cleanup,

	.noresume_reset	= swapwriter_noresume_reset,
	.storage_available 	= swapwriter_storage_available,
	.storage_allocated	= swapwriter_storage_allocated,
	.release_storage	= swapwriter_release_storage,
	.allocate_header_space	= swapwriter_allocate_header_space,
	.allocate_storage	= swapwriter_allocate_storage,
	.image_exists		= swapwriter_image_exists,
	.mark_resume_attempted	= swapwriter_mark_resume_attempted,
	.write_header_init	= swapwriter_write_header_init,
	.write_header_cleanup	= swapwriter_write_header_cleanup,
	.read_header_init	= swapwriter_read_header_init,
	.read_header_cleanup	= swapwriter_read_header_cleanup,
	.invalidate_image	= swapwriter_invalidate_image,
	.parse_sig_location	= swapwriter_parse_sig_location,
};

/* ---- Registration ---- */
static __init int swapwriter_load(void)
{
	int result;
	int i, numfiles = sizeof(swapwriter_sysfs_data) / sizeof(struct suspend_sysfs_data);
	
	printk("Suspend2 Swap Writer loading.\n");

	swapwriterops.rw_init = suspend_bio_ops.rw_init;
	swapwriterops.rw_cleanup = suspend_bio_ops.rw_cleanup;
	swapwriterops.read_chunk = suspend_bio_ops.read_chunk;
	swapwriterops.write_chunk = suspend_bio_ops.write_chunk;
	swapwriterops.rw_header_chunk = suspend_bio_ops.rw_header_chunk;

	if (!(result = suspend_register_module(&swapwriterops))) {
		struct kobject *kobj = make_suspend2_sysdir("swapwriter");
		for (i=0; i< numfiles; i++)
			suspend_register_sysfs_file(kobj,
					&swapwriter_sysfs_data[i]);
	} else
		printk("Suspend2 Swap Writer unable to register!\n");
	return result;
}

#ifdef MODULE
static __exit void swapwriter_unload(void)
{
	int i, numfiles = sizeof(swapwriter_sysfs_data) /
		    sizeof(struct suspend_sysfs_data);

	printk("Suspend2 Swap Writer unloading.\n");

	for (i=0; i< numfiles; i++)
		suspend_unregister_sysfs_file(&swapwriter_sysfs_data[i]);
	suspend_unregister_module(&swapwriterops);
}

module_init(swapwriter_load);
module_exit(swapwriter_unload);
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Nigel Cunningham");
MODULE_DESCRIPTION("Suspend2 swap writer");
#else
late_initcall(swapwriter_load);
#endif
