/*
 * drivers/power/process.c - Functions for starting/stopping processes on 
 *                           suspend transitions.
 *
 * Originally from swsusp.
 */

#undef DEBUG

#include <linux/smp_lock.h>
#include <linux/interrupt.h>
#include <linux/suspend.h>
#include <linux/module.h>
#include <linux/syscalls.h>
#include <linux/buffer_head.h>
#include <linux/freezer.h>

unsigned long freezer_state = 0;

#ifdef CONFIG_PM_DEBUG
#define freezer_message(msg, a...) do { printk(msg, ##a); } while(0)
#else
#define freezer_message(msg, a...) do { } while(0)
#endif

/* 
 * Timeout for stopping processes
 */
#define TIMEOUT	(20 * HZ)

struct frozen_fs
{
	struct list_head fsb_list;
	struct super_block *sb;
};

LIST_HEAD(frozen_fs_list);

void freezer_make_fses_rw(void)
{
	struct frozen_fs *fs, *next_fs;

	list_for_each_entry_safe(fs, next_fs, &frozen_fs_list, fsb_list) {
		thaw_bdev(fs->sb->s_bdev, fs->sb);

		list_del(&fs->fsb_list);
		kfree(fs);
	}
}

/* 
 * Done after userspace is frozen, so there should be no danger of
 * fses being unmounted while we're in here.
 */
int freezer_make_fses_ro(void)
{
	struct frozen_fs *fs;
	struct super_block *sb;

	/* Generate the list */
	list_for_each_entry(sb, &super_blocks, s_list) {
		if (!sb->s_root || !sb->s_bdev ||
		    (sb->s_frozen == SB_FREEZE_TRANS) ||
		    (sb->s_flags & MS_RDONLY))
			continue;

		fs = kmalloc(sizeof(struct frozen_fs), GFP_ATOMIC);
		if (!fs)
			return 1;
		fs->sb = sb;
		list_add_tail(&fs->fsb_list, &frozen_fs_list);
	};

	/* Do the freezing in reverse order so filesystems dependant
	 * upon others are frozen in the right order. (Eg loopback
	 * on ext3). */
	list_for_each_entry_reverse(fs, &frozen_fs_list, fsb_list)
		freeze_bdev(fs->sb->s_bdev);

	return 0;
}


static inline int freezeable(struct task_struct * p)
{
	if ((p == current) || 
	    (p->flags & PF_NOFREEZE) ||
	    (p->exit_state == EXIT_ZOMBIE) ||
	    (p->exit_state == EXIT_DEAD) ||
	    (p->state == TASK_STOPPED))
		return 0;
	return 1;
}

/* Refrigerator is place where frozen processes are stored :-). */
void refrigerator(void)
{
	/* Hmm, should we be allowed to suspend when there are realtime
	   processes around? */
	long save;
	save = current->state;
	pr_debug("%s entered refrigerator\n", current->comm);
	freezer_message("=");

	frozen_process(current);
	spin_lock_irq(&current->sighand->siglock);
	recalc_sigpending(); /* We sent fake signal, clean it up */
	spin_unlock_irq(&current->sighand->siglock);

	while (frozen(current)) {
		current->state = TASK_UNINTERRUPTIBLE;
		schedule();
	}
	pr_debug("%s left refrigerator\n", current->comm);
	current->state = save;
}

static inline void freeze_process(struct task_struct *p)
{
	unsigned long flags;

	if (!freezing(p)) {
		freeze(p);
		spin_lock_irqsave(&p->sighand->siglock, flags);
		signal_wake_up(p, 0);
		spin_unlock_irqrestore(&p->sighand->siglock, flags);
	}
}

static void cancel_freezing(struct task_struct *p)
{
	unsigned long flags;

	if (freezing(p)) {
		pr_debug("  clean up: %s\n", p->comm);
		do_not_freeze(p);
		spin_lock_irqsave(&p->sighand->siglock, flags);
		recalc_sigpending_tsk(p);
		spin_unlock_irqrestore(&p->sighand->siglock, flags);
	}
}

/* 0 = success, else # of processes that we failed to stop */
int freeze_processes(void)
{
	int todo, nr_user, user_frozen;
	unsigned long start_time;
	struct task_struct *g, *p;

	user_frozen = test_freezer_state(FREEZER_ON);
	
	if (!user_frozen)
		set_freezer_state(FREEZER_ON);

	freezer_message( "Stopping tasks: " );
	start_time = jiffies;
	do {
		nr_user = todo = 0;
		read_lock(&tasklist_lock);
		do_each_thread(g, p) {
			if (!freezeable(p))
				continue;
			if (frozen(p))
				continue;
			if (p->state == TASK_TRACED && frozen(p->parent)) {
				cancel_freezing(p);
				continue;
			}
			if (p->mm && !(p->flags & PF_BORROWED_MM)) {
				/* The task is a user-space one.
				 * Freeze it unless there's a vfork completion
				 * pending
				 */
				if (!p->vfork_done)
					freeze_process(p);
				nr_user++;
			} else {
				/* Freeze only if the user space is frozen */
				if (user_frozen)
					freeze_process(p);
				todo++;
			}
		} while_each_thread(g, p);
		read_unlock(&tasklist_lock);
		todo += nr_user;
		if (!user_frozen && !nr_user) {
			freezer_message("Freezing bdevs... ");
			freezer_make_fses_ro();
			start_time = jiffies;
			freezer_message("Freezing kernel threads... ");
		}
		user_frozen = !nr_user;
		yield();			/* Yield is okay here */
		if (todo && time_after(jiffies, start_time + TIMEOUT))
			break;
	} while(todo);

	/* This does not unfreeze processes that are already frozen
	 * (we have slightly ugly calling convention in that respect,
	 * and caller must call thaw_processes() if something fails),
	 * but it cleans up leftover PF_FREEZE requests.
	 */
	if (todo) {
		freezer_message( "\n" );
		freezer_message(KERN_ERR " stopping tasks timed out "
			"after %d seconds (%d tasks remaining):\n",
			TIMEOUT / HZ, todo);
		read_lock(&tasklist_lock);
		do_each_thread(g, p) {
			if (freezeable(p) && !frozen(p))
				freezer_message(KERN_ERR "  %s\n", p->comm);
			cancel_freezing(p);
		} while_each_thread(g, p);
		read_unlock(&tasklist_lock);
		return todo;
	}

	freezer_message( "|\n" );
	BUG_ON(in_atomic());
	set_freezer_state(FREEZING_COMPLETE);
	return 0;
}

void thaw_processes(int all)
{
	struct task_struct *g, *p;
	int pass = 0; /* Start on kernel space */

	if (!test_freezer_state(FREEZER_ON))
		return;

	if (!test_freezer_state(FREEZING_COMPLETE))
		pass++;

	clear_freezer_state(FREEZING_COMPLETE);

	freezer_message( "Restarting tasks... " );
	read_lock(&tasklist_lock);
	do {
		do_each_thread(g, p) {
			/* 
			 * is_user = 0 if kernel thread or borrowed mm,
			 * 1 otherwise.
			 */
			int is_user = !!(p->mm && !(p->flags & PF_BORROWED_MM));
			if (!freezeable(p) || (is_user != pass))
				continue;
			if (!thaw_process(p))
				freezer_message(KERN_INFO " Strange, %s not stopped\n", p->comm );
		} while_each_thread(g, p);

		if (!pass) {
			read_unlock(&tasklist_lock);
			freezer_message("Thawing bdevs... ");
			freezer_make_fses_rw();
			read_lock(&tasklist_lock);
		}

		pass++;
	} while(pass < 2 && all);

	read_unlock(&tasklist_lock);
	schedule();
	freezer_message( "Done\n" );

	if (all)
		clear_freezer_state(FREEZER_ON);
}

EXPORT_SYMBOL(refrigerator);
