/*
 * kernel/power/pageflags.c
 *
 * Copyright (C) 2004-2006 Nigel Cunningham <nigel@suspend2.net>
 * 
 * This file is released under the GPLv2.
 *
 * Routines for serialising and relocating pageflags in which we
 * store our image metadata.
 *
 */

#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/bitops.h>
#include <linux/list.h>
#include <linux/suspend.h>
#include "pageflags.h"
#include "modules.h"
#include "pagedir.h"

/* Maps used in copying the image back are in builtin.c */
dyn_pageflags_t pageset1_map;
dyn_pageflags_t pageset1_copy_map;
dyn_pageflags_t pageset2_map;
dyn_pageflags_t pageset2_rw_map;
dyn_pageflags_t in_use_map;

static int pages_for_zone(struct zone *zone)
{
	return (zone->spanned_pages + (PAGE_SIZE << 3) - 1) /
			(PAGE_SIZE << 3);
}

int suspend_pageflags_space_needed(void)
{
	int total = 0;
	struct zone *zone;

	for_each_zone(zone)
		if (populated_zone(zone))
			total += sizeof(int) * 2 + pages_for_zone(zone) * PAGE_SIZE;

	total += sizeof(int);

	return total;
}

/* save_dyn_pageflags
 *
 * Description: Save a set of pageflags.
 * Arguments:   dyn_pageflags_t *: Pointer to the bitmap being saved.
 */

void save_dyn_pageflags(dyn_pageflags_t pagemap)
{
	int i, zone_num, size;
	struct zone *zone;

	if (!*pagemap)
		return;

	for_each_zone(zone) {
		if (!populated_zone(zone))
			continue;

		size = pages_for_zone(zone);
		zone_num = page_zone_id(pfn_to_page(zone->zone_start_pfn));

		suspend_active_writer->rw_header_chunk(WRITE, NULL,
				(char *) &zone_num, sizeof(int));
		suspend_active_writer->rw_header_chunk(WRITE, NULL,
				(char *) &size, sizeof(int));

		for (i = 0; i < size; i++)
			suspend_active_writer->rw_header_chunk(WRITE, NULL,
				(char *) pagemap[zone_num][i], PAGE_SIZE);
	}
	zone_num = -1;
	suspend_active_writer->rw_header_chunk(WRITE, NULL,
			(char *) &zone_num, sizeof(int));
}

/* load_dyn_pageflags
 *
 * Description: Load a set of pageflags.
 * Arguments:   dyn_pageflags_t *: Pointer to the bitmap being loaded.
 *              (It must be allocated before calling this routine).
 */

void load_dyn_pageflags(dyn_pageflags_t pagemap)
{
	int i, zone_num, zone_check = 0, size;
	struct zone *zone;

	if (!pagemap)
		return;

	for_each_zone(zone) {
		if (!populated_zone(zone))
			continue;

		zone_num = page_zone_id(pfn_to_page(zone->zone_start_pfn));

		suspend_active_writer->rw_header_chunk(READ, NULL,
				(char *) &zone_check, sizeof(int));
		if (zone_check != zone_num) {
			printk("Zone check (%d) != zone_num (%d).\n",
					zone_check, zone_num);
			BUG();
		}
		suspend_active_writer->rw_header_chunk(READ, NULL,
				(char *) &size, sizeof(int));

		for (i = 0; i < size; i++)
			suspend_active_writer->rw_header_chunk(READ, NULL,
					(char *) pagemap[zone_num][i],
					PAGE_SIZE);
	}
	suspend_active_writer->rw_header_chunk(READ, NULL, (char *) &zone_check,
			sizeof(int));
	if (zone_check != -1) {
		printk("Didn't read end of dyn pageflag data marker.(%x)\n",
				zone_check);
		BUG();
	}
}

/* relocate_dyn_pageflags
 *
 * Description: Relocate a set of pageflags to ensure they don't collide with
 *              pageset 1 data which will get overwritten on copyback.
 * Arguments:   dyn_pageflags_t *: Pointer to the bitmap being relocated.
 */

void relocate_dyn_pageflags(dyn_pageflags_t *pagemap)
{
	int i, zone_num;
	struct zone *zone;

	if (!*pagemap)
		return;

	suspend_relocate_if_required((unsigned long *) pagemap,
			(unsigned int) (sizeof(void *) * (1 << ZONETABLE_SHIFT)));
	BUG_ON(PagePageset1(virt_to_page(*pagemap)));

	for_each_zone(zone) {
		int pages = pages_for_zone(zone);

		if (!populated_zone(zone))
			continue;

		zone_num = page_zone_id(pfn_to_page(zone->zone_start_pfn));

		suspend_relocate_if_required((void *) &((*pagemap)[zone_num]),
			       sizeof(void *) * pages);
		BUG_ON(PagePageset1(virt_to_page((*pagemap)[zone_num])));

		for (i = 0; i < pages; i++) {
			suspend_relocate_if_required(
				(void *) &((*pagemap)[zone_num][i]),
				PAGE_SIZE);
			BUG_ON(PagePageset1(virt_to_page(
						(*pagemap)[zone_num][i])));
		}
	}
}
