/*
 * kernel/power/io.c
 *
 * Copyright (C) 1998-2001 Gabor Kuti <seasons@fornax.hu>
 * Copyright (C) 1998,2001,2002 Pavel Machek <pavel@suse.cz>
 * Copyright (C) 2002-2003 Florent Chabaud <fchabaud@free.fr>
 * Copyright (C) 2002-2006 Nigel Cunningham <nigel@suspend2.net>
 *
 * This file is released under the GPLv2.
 *
 * It contains high level IO routines for suspending.
 *
 */

#include <linux/suspend.h>
#include <linux/version.h>
#include <linux/utsname.h>
#include <linux/mount.h>
#include <linux/suspend2.h>
#include <linux/highmem.h>
#include <asm/tlbflush.h>

#include "suspend.h"
#include "modules.h"
#include "pageflags.h"
#include "io.h"
#include "ui.h"
#include "storage.h"
#include "prepare_image.h"
#include "extent.h"

/* suspend_attempt_to_parse_resume_device
 *
 * Can we suspend, using the current resume2= parameter?
 */
int suspend_attempt_to_parse_resume_device(void)
{
	struct list_head *writer;
	struct suspend_module_ops *this_writer;
	int result, returning = 0;

	if (suspend_activate_storage(0))
		return 0;

	suspend_active_writer = NULL;
	clear_suspend_state(SUSPEND_RESUME_DEVICE_OK);
	clear_suspend_state(SUSPEND_CAN_RESUME);
	clear_result_state(SUSPEND_ABORTED);

	if (!suspend_num_writers) {
		printk(name_suspend "No writers have been registered. Suspending will be disabled.\n");
		goto cleanup;
	}
	
	if (!resume2_file[0]) {
		printk(name_suspend "Resume2 parameter is empty. Suspending will be disabled.\n");
		goto cleanup;
	}

	list_for_each(writer, &suspend_writers) {
		this_writer = list_entry(writer, struct suspend_module_ops, type_list);

		/* 
		 * Not sure why you'd want to disable a writer, but
		 * we should honour the flag if we're providing it
		 */
		if (!this_writer->enabled)
			continue;

		result = this_writer->parse_sig_location(
				resume2_file, (suspend_num_writers == 1));

		switch (result) {
			case -EINVAL:
				/* 
				 * For this writer, but not a valid 
				 * configuration. Error already printed.
				 */

				goto cleanup;

			case 0:
				/*
				 * For this writer and valid.
				 */

				suspend_active_writer = this_writer;

				set_suspend_state(SUSPEND_RESUME_DEVICE_OK);
				set_suspend_state(SUSPEND_CAN_RESUME);
				printk(name_suspend "Resuming enabled.\n");

				returning = 1;
				goto cleanup;
		}
	}
	printk(name_suspend "No matching enabled writer found. Resuming disabled.\n");
cleanup:
	suspend_deactivate_storage(0);
	return returning;
}

void attempt_to_parse_resume_device2(void)
{
	suspend_prepare_usm();
	suspend_attempt_to_parse_resume_device();
	suspend_cleanup_usm();
}

/* noresume_reset_modules
 *
 * Description:	When we read the start of an image, modules (and especially the
 * 		active writer) might need to reset data structures if we decide
 * 		to invalidate the image rather than resuming from it.
 */

static void noresume_reset_modules(void)
{
	struct suspend_module_ops *this_filter;
	
	list_for_each_entry(this_filter, &suspend_filters, type_list) {
		if (this_filter->noresume_reset)
			this_filter->noresume_reset();
	}

	if (suspend_active_writer && suspend_active_writer->noresume_reset)
		suspend_active_writer->noresume_reset();
}

/* fill_suspend_header()
 * 
 * Description:	Fill the suspend header structure.
 * Arguments:	struct suspend_header: Header data structure to be filled.
 */

static void fill_suspend_header(struct suspend_header *sh)
{
	int i;
	
	memset((char *)sh, 0, sizeof(*sh));

	sh->version_code = LINUX_VERSION_CODE;
	sh->num_physpages = num_physpages;
	sh->orig_mem_free = suspend_orig_mem_free;
	strncpy(sh->machine, system_utsname.machine, 65);
	strncpy(sh->version, system_utsname.version, 65);
	sh->page_size = PAGE_SIZE;
	sh->pagedir = pagedir1;
	sh->pageset_2_size = pagedir2.pageset_size;
	sh->param0 = suspend_result;
	sh->param1 = suspend_action;
	sh->param2 = suspend_debug_state;
	sh->param3 = console_loglevel;
	sh->root_fs = current->fs->rootmnt->mnt_sb->s_dev;
	for (i = 0; i < 4; i++)
		sh->io_time[i/2][i%2] =
		       suspend_io_time[i/2][i%2];
}

/*
 * rw_init_modules
 *
 * Iterate over modules, preparing the ones that will be used to read or write
 * data.
 */
static int rw_init_modules(int rw, int which)
{
	struct suspend_module_ops *this_module;
	/* Initialise page transformers */
	list_for_each_entry(this_module, &suspend_filters, type_list) {
		if (!this_module->enabled)
			continue;
		if (this_module->rw_init &&
		     	this_module->rw_init(rw, which)) {
			abort_suspend("Failed to initialise the %s filter.",
				this_module->name);
				return 1;
		}
	}

	/* Initialise writer */
	if (suspend_active_writer->rw_init(rw, which)) {
		abort_suspend("Failed to initialise the writer."); 
		if (!rw)
			suspend_active_writer->invalidate_image();
		return 1;
	}

	/* Initialise other modules */
	list_for_each_entry(this_module, &suspend_modules, module_list) {
		if (!this_module->enabled)
			continue;
		if ((this_module->type == FILTER_MODULE) ||
		    (this_module->type == WRITER_MODULE))
			continue;
		if (this_module->rw_init && this_module->rw_init(rw, which)) {
				set_result_state(SUSPEND_ABORTED);
				return 1;
			}
	}

	return 0;
}

/*
 * rw_cleanup_modules
 *
 * Cleanup components after reading or writing a set of pages.
 * Only the writer may fail.
 */
static int rw_cleanup_modules(int rw)
{
	struct suspend_module_ops *this_module;
	int result = 0;

	/* Cleanup other modules */
	list_for_each_entry(this_module, &suspend_modules, module_list) {
		if (!this_module->enabled)
			continue;
		if ((this_module->type == FILTER_MODULE) ||
		    (this_module->type == WRITER_MODULE))
			continue;
		if (this_module->rw_cleanup)
			result |= this_module->rw_cleanup(rw);
	}

	/* Flush data and cleanup */
	list_for_each_entry(this_module, &suspend_filters, type_list) {
		if (!this_module->enabled)
			continue;
		if (this_module->rw_cleanup)
			result |= this_module->rw_cleanup(rw);
	}

	result |= suspend_active_writer->rw_cleanup(rw);

	return result;
}

/*
 * do_rw_loop
 *
 * The main I/O loop for reading or writing pages.
 */
static int do_rw_loop(int write, int finish_at, dyn_pageflags_t *pageflags,
		int base, int barmax)
{
	int pfn, pc, step = 1, nextupdate = 0, i;
	int result;
	struct suspend_module_ops *first_filter = suspend_get_next_filter(NULL);
	char *buffer = (char *) get_zeroed_page(GFP_ATOMIC);

	pfn = get_next_bit_on(*pageflags, max_pfn);

	pc = finish_at / 5;

	/* Read the pages */
	for (i=0; i< finish_at; i++) {
		struct page *page = pfn_to_page(pfn);

		/* Status */
		if ((i+base) >= nextupdate)
			nextupdate = suspend_update_status(i+base, barmax,
				" %d/%d MB ", MB(base+i+1), MB(barmax));

		if ((i + 1) == pc) {
			printk("%d%%...", 20 * step);
			step++;
			pc = finish_at * step / 5;
		}
		
		if (write)
			result = first_filter->write_chunk(page);
		else
			result = first_filter->read_chunk(page, SUSPEND_ASYNC);

		if (result) {
			if (write) {
				printk("Write chunk returned %d.\n", result);
				abort_suspend("Failed to write a chunk of the "
					"image.");
				free_page((unsigned long) buffer);
				return result;
			} else
				panic("Failed to read chunk %d/%d of the image. (%d)",
					i, finish_at, result);
		}

		/* Interactivity*/
		suspend_cond_pause(0, NULL);

		if (test_result_state(SUSPEND_ABORTED) && write) {
			free_page((unsigned long) buffer);
			return 1;
		}

		/* Prepare next */
		pfn = get_next_bit_on(*pageflags, pfn);
	}

	printk("done.\n");

	suspend_update_status(base + finish_at, barmax, " %d/%d MB ",
			MB(base + finish_at), MB(barmax));
	free_page((unsigned long) buffer);
	return 0;
}

/* write_pageset()
 *
 * Description:	Write a pageset to disk.
 * Arguments:	pagedir:	Pointer to the pagedir to be saved.
 * 		whichtowrite:	Controls what debugging output is printed.
 * Returns:	Zero on success or -1 on failure.
 */

int write_pageset(struct pagedir *pagedir, int whichtowrite)
{
	int finish_at, base = 0, start_time, end_time;
	int barmax = pagedir1.pageset_size + pagedir2.pageset_size;
	long error = 0;
	dyn_pageflags_t *pageflags;

	/* 
	 * Even if there is nothing to read or write, the writer
	 * may need the init/cleanup for it's housekeeping.  (eg:
	 * Pageset1 may start where pageset2 ends when writing).
	 */
	finish_at = pagedir->pageset_size;

	if (whichtowrite == 1) {
		suspend_prepare_status(DONT_CLEAR_BAR,
				"Writing kernel & process data...");
		base = pagedir2.pageset_size;
		if (test_action_state(SUSPEND_TEST_FILTER_SPEED) ||
		    test_action_state(SUSPEND_TEST_BIO))
			pageflags = &pageset1_map;
		else
			pageflags = &pageset1_copy_map;
	} else {
		suspend_prepare_status(CLEAR_BAR, "Writing caches...");
		pageflags = &pageset2_map;
		bytes_in = bytes_out = 0;
	}	
	
	start_time = jiffies;

	if (!rw_init_modules(1, whichtowrite))
		error = do_rw_loop(1, finish_at, pageflags, base, barmax);

	if (rw_cleanup_modules(WRITE)) {
		abort_suspend("Failed to cleanup after writing.");
		error = 1;
	}

	/* Statistics */
	end_time = jiffies;
	
	if ((end_time - start_time) && (!test_result_state(SUSPEND_ABORTED))) {
		suspend_io_time[0][0] += finish_at,
		suspend_io_time[0][1] += (end_time - start_time);
	}

	return error;
}

/* read_pageset()
 *
 * Description:	Read a pageset from disk.
 * Arguments:	pagedir:	Pointer to the pagedir to be saved.
 * 		whichtowrite:	Controls what debugging output is printed.
 * 		overwrittenpagesonly: Whether to read the whole pageset or
 * 		only part.
 * Returns:	Zero on success or -1 on failure.
 */

static int read_pageset(struct pagedir *pagedir, int whichtoread,
		int overwrittenpagesonly)
{
	int result = 0, base = 0, start_time, end_time;
	int finish_at = pagedir->pageset_size;
	int barmax = pagedir1.pageset_size + pagedir2.pageset_size;
	dyn_pageflags_t *pageflags;

	if (whichtoread == 1) {
		suspend_prepare_status(CLEAR_BAR,
				"Reading kernel & process data...");
		pageflags = &pageset1_copy_map;
	} else {
		suspend_prepare_status(DONT_CLEAR_BAR, "Reading caches...");
		if (overwrittenpagesonly)
			barmax = finish_at = min(pagedir1.pageset_size, 
						 pagedir2.pageset_size);
		else {
			base = pagedir1.pageset_size;
		}
		pageflags = &pageset2_map;
	}	
	
	start_time = jiffies;

	if (rw_init_modules(0, whichtoread)) {
		suspend_active_writer->invalidate_image();
		result = 1;
	} else
		result = do_rw_loop(0, finish_at, pageflags, base, barmax);

	if (rw_cleanup_modules(READ)) {
		abort_suspend("Failed to cleanup after reading.");
		result = 1;
	}

	/* Statistics */
	end_time=jiffies;

	if ((end_time - start_time) && (!test_result_state(SUSPEND_ABORTED))) {
		suspend_io_time[1][0] += finish_at,
		suspend_io_time[1][1] += (end_time - start_time);
	}

	return result;
}

/* write_module_configs()
 *
 * Description:	Store the configuration for each module in the image header.
 * Returns:	Int: Zero on success, Error value otherwise.
 */
static int write_module_configs(void)
{
	struct suspend_module_ops *this_module;
	char *buffer = (char *) get_zeroed_page(GFP_ATOMIC);
	int len, index = 1;
	struct suspend_module_header suspend_module_header;

	if (!buffer) {
		printk("Failed to allocate a buffer for saving "
				"module configuration info.\n");
		return -ENOMEM;
	}
		
	/* 
	 * We have to know which data goes with which module, so we at
	 * least write a length of zero for a module. Note that we are
	 * also assuming every module's config data takes <= PAGE_SIZE.
	 */

	/* For each module (in registration order) */
	list_for_each_entry(this_module, &suspend_modules, module_list) {
		if (!this_module->enabled || !this_module->storage_needed ||
		    (this_module->type == WRITER_MODULE &&
		     suspend_active_writer != this_module))
			continue;

		/* Get the data from the module */
		len = 0;
		if (this_module->save_config_info)
			len = this_module->save_config_info(buffer);

		/* Save the details of the module */
		suspend_module_header.enabled = this_module->enabled;
		suspend_module_header.type = this_module->type;
		suspend_module_header.index = index++;
		strncpy(suspend_module_header.name, this_module->name, 
					sizeof(suspend_module_header.name));
		suspend_active_writer->rw_header_chunk(WRITE,
				this_module,
				(char *) &suspend_module_header,
				sizeof(suspend_module_header));

		/* Save the size of the data and any data returned */
		suspend_active_writer->rw_header_chunk(WRITE,
				this_module,
				(char *) &len, sizeof(int));
		if (len)
			suspend_active_writer->rw_header_chunk(
				WRITE, this_module, buffer, len);
	}

	/* Write a blank header to terminate the list */
	suspend_module_header.name[0] = '\0';
	suspend_active_writer->rw_header_chunk(WRITE, 
			NULL,
			(char *) &suspend_module_header,
			sizeof(suspend_module_header));

	free_page((unsigned long) buffer);
	return 0;
}

/* read_module_configs()
 *
 * Description:	Reload module configurations from the image header.
 * Returns:	Int. Zero on success, error value otherwise.
 */

static int read_module_configs(void)
{
	struct suspend_module_ops *this_module;
	char *buffer = (char *) get_zeroed_page(GFP_ATOMIC);
	int len, result = 0;
	struct suspend_module_header suspend_module_header;

	if (!buffer) {
		printk("Failed to allocate a buffer for reloading module "
				"configuration info.\n");
		return -ENOMEM;
	}
		
	/* All modules are initially disabled. That way, if we have a module
	 * loaded now that wasn't loaded when we suspended, it won't be used
	 * in trying to read the data.
	 */
	list_for_each_entry(this_module, &suspend_modules, module_list)
		this_module->enabled = 0;
	
	/* Get the first module header */
	result = suspend_active_writer->rw_header_chunk(READ, NULL,
			(char *) &suspend_module_header, sizeof(suspend_module_header));
	if (!result) {
		printk("Failed to read the next module header.\n");
		free_page((unsigned long) buffer);
		return -EINVAL;
	}

	/* For each module (in registration order) */
	while (suspend_module_header.name[0]) {

		/* Find the module */
		this_module = suspend_find_module_given_name(suspend_module_header.name);

		if (!this_module) {
			/* 
			 * Is it used? Only need to worry about filters. The active
			 * writer must be loaded!
			 */
			if (suspend_module_header.enabled) {
				suspend_early_boot_message(1, SUSPEND_CONTINUE_REQ,
					"It looks like we need module %s for "
					"reading the image but it hasn't been "
					"registered.\n",
					suspend_module_header.name);
				if (!(test_suspend_state(SUSPEND_CONTINUE_REQ))) {
					suspend_active_writer->invalidate_image();
					free_page((unsigned long) buffer);
					return -EINVAL;
				}
			} else
				printk("Module %s configuration data found, but the module "
					"hasn't registered. Looks like it was disabled, so "
					"we're ignoring it's data.",
					suspend_module_header.name);
		}
		
		/* Get the length of the data (if any) */
		result = suspend_active_writer->rw_header_chunk(READ, NULL,
				(char *) &len, sizeof(int));
		if (!result) {
			printk("Failed to read the length of the module %s's"
					" configuration data.\n",
					suspend_module_header.name);
			free_page((unsigned long) buffer);
			return -EINVAL;
		}

		/* Read any data and pass to the module (if we found one) */
		if (len) {
			suspend_active_writer->rw_header_chunk(READ, NULL,
					buffer, len);
			if (this_module) {
				if (!this_module->save_config_info) {
					printk("Huh? Module %s appears to have "
						"a save_config_info, but not a "
						"load_config_info function!\n",
						this_module->name);
				} else
					this_module->load_config_info(buffer, len);
			}
		}

		if (this_module) {
			/* Now move this module to the tail of its lists. This
			 * will put it in order. Any new modules will end up at
			 * the top of the lists. They should have been set to
			 * disabled when loaded (people will normally not edit
			 * an initrd to load a new module and then suspend
			 * without using it!).
			 */

			suspend_move_module_tail(this_module);

			/* 
			 * We apply the disabled state; modules don't need to
			 * save whether they were disabled and if they do, we
			 * override them anyway.
			 */
			this_module->enabled = suspend_module_header.enabled;
		}

		/* Get the next module header */
		result = suspend_active_writer->rw_header_chunk(READ, NULL,
				(char *) &suspend_module_header,
				sizeof(suspend_module_header));

		if (!result) {
			printk("Failed to read the next module header.\n");
			free_page((unsigned long) buffer);
			return -EINVAL;
		}

	}

	free_page((unsigned long) buffer);
	return 0;
}

/* write_image_header()
 *
 * Description:	Write the image header after write the image proper.
 * Returns:	Int. Zero on success or -1 on failure.
 */

int write_image_header(void)
{
	int ret;
	int total = pagedir1.pageset_size + pagedir2.pageset_size+2;
	char *header_buffer = NULL;

	/* Now prepare to write the header */
	if ((ret = suspend_active_writer->write_header_init())) {
		abort_suspend("Active writer's write_header_init"
				" function failed.");
		goto write_image_header_abort;
	}

	/* Get a buffer */
	header_buffer = (char *) get_zeroed_page(GFP_ATOMIC);
	if (!header_buffer) {
		abort_suspend("Out of memory when trying to get page "
				"for header!");
		goto write_image_header_abort;
	}

	/* Write suspend header */
	fill_suspend_header((struct suspend_header *) header_buffer);
	suspend_active_writer->rw_header_chunk(WRITE, NULL,
			header_buffer, sizeof(struct suspend_header));

	free_page((unsigned long) header_buffer);

	/* Write module configurations */
	if ((ret = write_module_configs())) {
		abort_suspend("Failed to write module configs.");
		goto write_image_header_abort;
	}

	save_dyn_pageflags(pageset1_map);

	/* Flush data and let writer cleanup */
	if (suspend_active_writer->write_header_cleanup()) {
		abort_suspend("Failed to cleanup writing header.");
		goto write_image_header_abort_no_cleanup;
	}

	if (test_result_state(SUSPEND_ABORTED))
		goto write_image_header_abort_no_cleanup;

	suspend_message(SUSPEND_IO, SUSPEND_VERBOSE, 1, "|\n");
	suspend_update_status(total, total, NULL);

	return 0;

write_image_header_abort:
	suspend_active_writer->write_header_cleanup();
write_image_header_abort_no_cleanup:
	return -1;
}

/* sanity_check()
 *
 * Description:	Perform a few checks, seeking to ensure that the kernel being
 * 		booted matches the one suspended. They need to match so we can
 * 		be _sure_ things will work. It is not absolutely impossible for
 * 		resuming from a different kernel to work, just not assured.
 * Arguments:	Struct suspend_header. The header which was saved at suspend
 * 		time.
 */
static char *sanity_check(struct suspend_header *sh)
{
	if (sh->version_code != LINUX_VERSION_CODE)
		return "Incorrect kernel version.";
	
	if (sh->num_physpages != num_physpages)
		return "Incorrect memory size.";

	if (strncmp(sh->machine, system_utsname.machine, 65))
		return "Incorrect machine type.";

	if (strncmp(sh->version, system_utsname.version, 65))
		return "Right kernel version but wrong build number.";

	if (sh->page_size != PAGE_SIZE)
		return "Incorrect PAGE_SIZE.";

	if (!test_action_state(SUSPEND_IGNORE_ROOTFS)) {
		const struct super_block *sb;
		list_for_each_entry(sb, &super_blocks, s_list) {
			if ((!(sb->s_flags & MS_RDONLY)) &&
			    (sb->s_type->fs_flags & FS_REQUIRES_DEV))
				return "Device backed fs has been mounted "
					"rw prior to resume."; 
		}
	}

	return 0;
}

/* __read_pageset1
 *
 * Description:	Test for the existence of an image and attempt to load it.
 * Returns:	Int. Zero if image found and pageset1 successfully loaded.
 * 		Error if no image found or loaded.
 */
static int __read_pageset1(void)
{			
	int i, result = 0;
	char *header_buffer = (char *) get_zeroed_page(GFP_ATOMIC),
	     *sanity_error = NULL;
	struct suspend_header *suspend_header;

	if (!header_buffer) {
		printk("Unable to allocate a page for reading the signature.\n");
		return -ENOMEM;
	}
	
	/* Check for an image */
	if (!(result = suspend_active_writer->image_exists())) {
		result = -ENODATA;
		noresume_reset_modules();
		printk(name_suspend "No image found.\n");
		goto out;
	}

	/* Check for noresume command line option */
	if (test_suspend_state(SUSPEND_NORESUME_SPECIFIED)) {
		suspend_active_writer->invalidate_image();
		result = -EINVAL;
		noresume_reset_modules();
		printk(name_suspend "Noresume: Invalidated image.\n");
		goto out;
	}

	/* Check whether we've resumed before */
	if (test_suspend_state(SUSPEND_RESUMED_BEFORE)) {
		int resumed_before_default = 0;
		if (test_suspend_state(SUSPEND_RETRY_RESUME))
			resumed_before_default = SUSPEND_CONTINUE_REQ;
		suspend_early_boot_message(1, resumed_before_default, NULL);
		clear_suspend_state(SUSPEND_RETRY_RESUME);
		if (!(test_suspend_state(SUSPEND_CONTINUE_REQ))) {
			suspend_active_writer->invalidate_image();
			result = -EINVAL;
			noresume_reset_modules();
			printk(name_suspend "Tried to resume before: Invalidated image.\n");
			goto out;
		}
	}

	clear_suspend_state(SUSPEND_CONTINUE_REQ);

	/* 
	 * Prepare the active writer for reading the image header. The
	 * activate writer might read its own configuration.
	 * 
	 * NB: This call may never return because there might be a signature
	 * for a different image such that we warn the user and they choose
	 * to reboot. (If the device ids look erroneous (2.4 vs 2.6) or the
	 * location of the image might be unavailable if it was stored on a
	 * network connection.
	 */

	if ((result = suspend_active_writer->read_header_init())) {
		noresume_reset_modules();
		printk(name_suspend "Failed to initialise, reading the image header.\n");
		goto out;
	}
	
	/* Read suspend header */
	if ((result = suspend_active_writer->rw_header_chunk(READ, NULL,
			header_buffer, sizeof(struct suspend_header))) < 0) {
		noresume_reset_modules();
		printk(name_suspend "Failed to read the image signature.\n");
		goto out;
	}
	
	suspend_header = (struct suspend_header *) header_buffer;

	/*
	 * NB: This call may also result in a reboot rather than returning.
	 */

	if ((sanity_error = sanity_check(suspend_header)) &&
	    suspend_early_boot_message(1, SUSPEND_CONTINUE_REQ, sanity_error)) {
		suspend_active_writer->invalidate_image();
		result = -EINVAL;
		noresume_reset_modules();
		printk(name_suspend "Sanity check failed.\n");
		goto out;
	}

	/*
	 * We have an image and it looks like it will load okay.
	 */

	/* Get metadata from header. Don't override commandline parameters.
	 *
	 * We don't need to save the image size limit because it's not used
	 * during resume and will be restored with the image anyway.
	 */
	
	suspend_orig_mem_free = suspend_header->orig_mem_free;
	memcpy((char *) &pagedir1,
		(char *) &suspend_header->pagedir, sizeof(pagedir1));
	suspend_result = suspend_header->param0;
	suspend_action = suspend_header->param1;
	suspend_debug_state = suspend_header->param2;
	console_loglevel = suspend_header->param3;
	clear_suspend_state(SUSPEND_IGNORE_LOGLEVEL);
	pagedir2.pageset_size = suspend_header->pageset_2_size;
	for (i = 0; i < 4; i++)
		suspend_io_time[i/2][i%2] =
			suspend_header->io_time[i/2][i%2];

	/* Read module configurations */
	if ((result = read_module_configs())) {
		noresume_reset_modules();
		pagedir1.pageset_size =
			pagedir2.pageset_size = 0;
		printk(name_suspend "Failed to read Suspend module configurations.\n");
		goto out;
	}

	suspend_prepare_console();

	suspend_cond_pause(1, "About to read original pageset1 locations.");

	/*
	 * Read original pageset1 locations. These are the addresses we can't
	 * use for the data to be restored.
	 */

	allocate_dyn_pageflags(&pageset1_map);
	load_dyn_pageflags(pageset1_map);

	set_suspend_state(SUSPEND_NOW_RESUMING);

	/* Relocate it so that it's not overwritten while we're using it to
	 * copy the original contents back */
	relocate_dyn_pageflags(&pageset1_map);
	
	allocate_dyn_pageflags(&pageset1_copy_map);
	relocate_dyn_pageflags(&pageset1_copy_map);

	/* Clean up after reading the header */
	if ((result = suspend_active_writer->read_header_cleanup())) {
		noresume_reset_modules();
		printk(name_suspend "Failed to cleanup after reading the image header.\n");
		goto out_reset_console;
	}

	suspend_cond_pause(1, "About to read pagedir.");

	/* 
	 * Get the addresses of pages into which we will load the kernel to
	 * be copied back
	 */
	if (suspend_get_pageset1_load_addresses()) {
		result = -ENOMEM;
		noresume_reset_modules();
		printk(name_suspend "Failed to get load addresses for pageset1.\n");
		goto out_reset_console;
	}

	/* Read the original kernel back */
	suspend_cond_pause(1, "About to read pageset 1.");

	if (read_pageset(&pagedir1, 1, 0)) {
		suspend_prepare_status(CLEAR_BAR, "Failed to read pageset 1.");
		result = -EPERM;
		noresume_reset_modules();
		printk(name_suspend "Failed to get load pageset1.\n");
		goto out_reset_console;
	}

	suspend_cond_pause(1, "About to restore original kernel.");
	result = 0;

	if (!test_action_state(SUSPEND_KEEP_IMAGE) &&
	    suspend_active_writer->mark_resume_attempted)
		suspend_active_writer->mark_resume_attempted();

out:
	free_page((unsigned long) header_buffer);
	return result;

out_reset_console:
	free_dyn_pageflags(&pageset1_map);
	free_dyn_pageflags(&pageset1_copy_map);
	suspend_cleanup_console();
	goto out;
}

/* read_pageset1()
 *
 * Description:	Attempt to read the header and pageset1 of a suspend image.
 * 		Handle the outcome, complaining where appropriate.
 */

int read_pageset1(void)
{
	int error;

	error = __read_pageset1();

	switch (error) {
		case 0:
		case -ENODATA:
		case -EINVAL:	/* non fatal error */
			return error;
		case -EIO:
			printk(KERN_CRIT name_suspend "I/O error\n");
			break;
		case -ENOENT:
			printk(KERN_CRIT name_suspend "No such file or directory\n");
			break;
		case -EPERM:
			printk(KERN_CRIT name_suspend "Sanity check error\n");
			break;
		default:
			printk(KERN_CRIT name_suspend "Error %d resuming\n",
					error);
			break;
	}
	abort_suspend("Error %d in read_pageset1",error);
	return error;
}

/*
 * get_have_image_data()
 */
char *get_have_image_data(void)
{
	char *output_buffer = (char *) get_zeroed_page(GFP_ATOMIC);
	struct suspend_header *suspend_header;

	if (!output_buffer) {
		printk("Output buffer null.\n");
		return NULL;
	}

	/* Check for an image */
	if (!suspend_active_writer->image_exists() ||
	    suspend_active_writer->read_header_init() ||
	    suspend_active_writer->rw_header_chunk(READ, NULL,
			output_buffer, sizeof(struct suspend_header)) !=
	    		sizeof(struct suspend_header)) {
		sprintf(output_buffer, "0\n");
		goto out;
	}

	suspend_header = (struct suspend_header *) output_buffer;

	sprintf(output_buffer, "1\n%s\n%s\n",
			suspend_header->machine,
			suspend_header->version);

	/* Check whether we've resumed before */
	if (test_suspend_state(SUSPEND_RESUMED_BEFORE))
		strcat(output_buffer, "Resumed before.\n");

out:
	noresume_reset_modules();
	return output_buffer;
}

/* read_pageset2()
 *
 * Description:	Read in part or all of pageset2 of an image, depending upon
 * 		whether we are suspending and have only overwritten a portion
 * 		with pageset1 pages, or are resuming and need to read them 
 * 		all.
 * Arguments:	Int. Boolean. Read only pages which would have been
 * 		overwritten by pageset1?
 * Returns:	Int. Zero if no error, otherwise the error value.
 */
int read_pageset2(int overwrittenpagesonly)
{
	int result = 0;

	if (!pagedir2.pageset_size)
		return 0;

	result = read_pageset(&pagedir2, 2, overwrittenpagesonly);

	suspend_update_status(100, 100, NULL);
	suspend_cond_pause(1, "Pagedir 2 read.");

	return result;
}
