/* 
 * kernel/power/extent.c
 * 
 * (C) 2003-2006 Nigel Cunningham <nigel@suspend2.net>
 *
 * Distributed under GPLv2.
 * 
 * These functions encapsulate the manipulation of storage metadata. For
 * pageflags, we use dynamically allocated bitmaps.
 */

#include <linux/module.h>
#include <linux/suspend.h>
#include "modules.h"
#include "extent.h"
#include "ui.h"

int suspend_extents_allocated = 0;

/* suspend_get_extent
 *
 * Returns a free extent. May fail, returning NULL instead.
 */
static struct extent *suspend_get_extent(void)
{
	struct extent *result;
	
	if (!(result = kmalloc(sizeof(struct extent), GFP_ATOMIC)))
		return NULL;

	suspend_extents_allocated++;
	result->minimum = result->maximum = 0;
	result->next = NULL;

	return result;
}

/* suspend_put_extent.
 *
 * Frees an extent. Assumes unlinking is done by the caller.
 */
void suspend_put_extent(struct extent *extent)
{
	BUG_ON(!extent);

	kfree(extent);
	suspend_extents_allocated--;
}

/* suspend_put_extent_chain.
 *
 * Frees a whole chain of extents.
 */
void suspend_put_extent_chain(struct extent_chain *chain)
{
	struct extent *this;

	this = chain->first;

	while(this) {
		struct extent *next = this->next;
		kfree(this);
		chain->frees++;
		suspend_extents_allocated --;
		this = next;
	}
	
	BUG_ON(chain->frees != chain->allocs);
	chain->first = chain->last = chain->last_touched = NULL;
	chain->size = chain->allocs = chain->frees = 0;
}

/* 
 * suspend_add_to_extent_chain
 *
 * Add an extent to an existing chain.
 */
int suspend_add_to_extent_chain(struct extent_chain *chain, 
		unsigned long minimum, unsigned long maximum)
{
	struct extent *new_extent = NULL, *start_at;

	/* Find the right place in the chain */
	start_at = (chain->last_touched && 
		    (chain->last_touched->minimum < minimum)) ?
		chain->last_touched : NULL;

	if (!start_at && chain->first && chain->first->minimum < minimum)
		start_at = chain->first;

	while (start_at && start_at->next && start_at->next->minimum < minimum)
		start_at = start_at->next;

	if (start_at && start_at->maximum == (minimum - 1)) {
		start_at->maximum = maximum;

		/* Merge with the following one? */
		if (start_at->next &&
		    start_at->maximum + 1 == start_at->next->minimum) {
			struct extent *to_free = start_at->next;
			start_at->maximum = start_at->next->maximum;
			start_at->next = start_at->next->next;
			chain->frees++;
			suspend_put_extent(to_free);
		}

		chain->last_touched = start_at;
		chain->size+= (maximum - minimum + 1);

		return 0;
	}

	new_extent = suspend_get_extent();
	if (!new_extent) {
		printk("Error unable to append a new extent to the chain.\n");
		return 2;
	}

	chain->allocs++;
	chain->size+= (maximum - minimum + 1);
	new_extent->minimum = minimum;
	new_extent->maximum = maximum;
	new_extent->next = NULL;

	chain->last_touched = new_extent;

	if (start_at) {
		struct extent *next = start_at->next;
		start_at->next = new_extent;
		new_extent->next = next;
		if (!next)
			chain->last = new_extent;
	} else {
		if (chain->first) {
			new_extent->next = chain->first;
			chain->first = new_extent;
		} else
			chain->last = chain->first = new_extent;
	}

	return 0;
}

/* suspend_serialise_extent_chain
 *
 * Write a chain in the image.
 */
int suspend_serialise_extent_chain(struct suspend_module_ops *owner,
		struct extent_chain *chain)
{
	struct extent *this;
	int ret, i = 0;
	
	if ((ret = suspend_active_writer->rw_header_chunk(WRITE, owner,
		(char *) chain,
		3 * sizeof(int))))
		return ret;

	this = chain->first;
	while (this) {
		if ((ret = suspend_active_writer->rw_header_chunk(WRITE, owner,
				(char *) this,
				2 * sizeof(unsigned long))))
			return ret;
		this = this->next;
		i++;
	}

	if (i != (chain->allocs - chain->frees)) {
		printk(KERN_EMERG "Saved %d extents but chain metadata says there should be %d-%d.\n",
				i, chain->allocs, chain->frees);
		BUG();
	}

	return ret;
}

/* suspend_load_extent_chain
 *
 * Read back a chain saved in the image.
 */
int suspend_load_extent_chain(struct extent_chain *chain)
{
	struct extent *this, *last = NULL;
	int i, ret;

	if (!(ret = suspend_active_writer->rw_header_chunk(READ, NULL,
		(char *) chain,
		3 * sizeof(int))))
		return ret;

	for (i = 0; i < (chain->allocs - chain->frees); i++) {
		this = kmalloc(sizeof(struct extent), GFP_ATOMIC);
		BUG_ON(!this); /* Shouldn't run out of memory trying this! */
		this->next = NULL;
		if (!(ret = suspend_active_writer->rw_header_chunk(READ, NULL,
				(char *) this, 2 * sizeof(unsigned long))))
			return ret;
		if (last)
			last->next = this;
		else
			chain->first = this;
		last = this;
	}
	chain->last = last;
	return ret;
}

/* suspend_extent_state_next
 *
 * Given a state, progress to the next valid entry. We may begin in an
 * invalid state, as we do when invoked after extent_state_goto_start below.
 *
 * When using compression and expected_compression > 0, we allocate fewer
 * swap entries, so we can validly run out of data to return.
 */
unsigned long suspend_extent_state_next(struct extent_iterate_state *state)
{
	if (state->current_chain > state->num_chains)
		return 0;

	if (state->current_extent) {
		if (state->current_offset == state->current_extent->maximum) {
			if (state->current_extent->next) {
				state->current_extent = state->current_extent->next;
				state->current_offset = state->current_extent->minimum;
			} else {
				state->current_extent = NULL;
				state->current_offset = 0;
			}
		} else
			state->current_offset++;
	}

	while(!state->current_extent) {
		int chain_num = ++(state->current_chain);

		if (chain_num > state->num_chains)
			return 0;

		state->current_extent = (state->chains + chain_num)->first;

		if (!state->current_extent)
			continue;

		state->current_offset = state->current_extent->minimum;
	}

	return state->current_offset;
}

/* suspend_extent_state_goto_start
 *
 * Find the first valid value in a group of chains.
 */
void suspend_extent_state_goto_start(struct extent_iterate_state *state)
{
	state->current_chain = -1;
	state->current_extent = NULL;
	state->current_offset = 0;
}

/* suspend_extent_start_save
 *
 * Given a state and a struct extent_state_store, save the crreutn
 * position in a format that can be used with relocated chains (at
 * resume time).
 */
void suspend_extent_state_save(struct extent_iterate_state *state,
		struct extent_iterate_saved_state *saved_state)
{
	struct extent *extent;

	saved_state->chain_num = state->current_chain;
	saved_state->extent_num = 0;
	saved_state->offset = state->current_offset;

	if (saved_state->chain_num == -1)
		return;
	
	extent = (state->chains + state->current_chain)->first;

	while (extent != state->current_extent) {
		saved_state->extent_num++;
		extent = extent->next;
	}
}

/* suspend_extent_start_restore
 *
 * Restore the position saved by extent_state_save.
 */
void suspend_extent_state_restore(struct extent_iterate_state *state,
		struct extent_iterate_saved_state *saved_state)
{
	int posn = saved_state->extent_num;

	if (saved_state->chain_num == -1) {
		suspend_extent_state_goto_start(state);
		return;
	}

	state->current_chain = saved_state->chain_num;
	state->current_extent = (state->chains + state->current_chain)->first;
	state->current_offset = saved_state->offset;

	while (posn--)
		state->current_extent = state->current_extent->next;
}
