/*
 * kernel/power/encryption.c
 *
 * Copyright (C) 2003-2006 Nigel Cunningham <nigel@suspend2.net>
 *
 * This file is released under the GPLv2.
 *
 * This file contains data encryption routines for suspend,
 * using cryptoapi transforms.
 *
 * ToDo:
 * - Apply min/max_keysize the cipher changes.
 * - Test.
 */

#include <linux/suspend.h>
#include <linux/module.h>
#include <linux/highmem.h>
#include <linux/vmalloc.h>
#include <linux/crypto.h>
#include <asm/scatterlist.h>

#include "suspend.h"
#include "modules.h"
#include "sysfs.h"
#include "io.h"

#define S2C_WRITE 0
#define S2C_READ 1

static struct suspend_module_ops suspend_encryption_ops;
static struct suspend_module_ops *next_driver;

static char suspend_encryptor_name[32];
static struct crypto_tfm *suspend_encryptor_transform;
static char suspend_encryptor_key[256];
static int suspend_key_len;
static char suspend_encryptor_iv[256];
static int suspend_encryptor_mode;
static int suspend_encryptor_save_key_and_iv;

static u8 *page_buffer = NULL;
static unsigned int bufofs;

static struct scatterlist suspend_crypt_sg[PAGE_SIZE/8];
       
/* ---- Local buffer management ---- */

/* allocate_local_buffer
 *
 * Description:	Allocates a page of memory for buffering output.
 * Returns:	Int: Zero if successful, -ENONEM otherwise.
 */
static int allocate_local_buffer(void)
{
	if (!page_buffer) {
		int i, remainder;
		
		page_buffer = (char *) get_zeroed_page(GFP_ATOMIC);
	
		if (!page_buffer) {
			printk(KERN_ERR
				"Failed to allocate the page buffer for "
				"suspend2 encryption driver.\n");
			return -ENOMEM;
		}

		for (i=0; i < (PAGE_SIZE / suspend_key_len); i++) {
			suspend_crypt_sg[i].page = virt_to_page(page_buffer);
			suspend_crypt_sg[i].offset = suspend_key_len * i;
			suspend_crypt_sg[i].length = suspend_key_len;
		}
		
		remainder = PAGE_SIZE % suspend_key_len;

		if (remainder) {
			suspend_crypt_sg[i].page = virt_to_page(page_buffer);
			suspend_crypt_sg[i].offset = suspend_key_len * i;
			suspend_crypt_sg[i].length = remainder;
		}
	}

	return 0;
}

/* free_local_buffer
 *
 * Description:	Frees memory allocated for buffering output.
 */
static void free_local_buffer(void)
{
	if (page_buffer)
		free_page((unsigned long) page_buffer);

	page_buffer = NULL;
}

/* suspend_encrypt_rw_cleanup
 *
 * Description:	Frees memory allocated for our labours.
 */
static int suspend_encrypt_rw_cleanup(int rw)
{
	if (suspend_encryptor_transform) {
		crypto_free_tfm(suspend_encryptor_transform);
		suspend_encryptor_transform = NULL;
	}

	free_local_buffer();

	return 0;
}

/* suspend_crypto_prepare
 *
 * Description:	Prepare to do some work by allocating buffers and transforms.
 * Returns:	Int: Zero if successful, 1 otherwise.
 */
static int suspend_encrypt_rw_prepare(int rw)
{
	if (!*suspend_encryptor_name) {
		printk("Suspend2: Encryptor enabled but no name set.\n");
		suspend_encryption_ops.enabled = 0;
		return 1;
	}

	suspend_encryptor_transform = crypto_alloc_tfm(suspend_encryptor_name,
					1 << suspend_encryptor_mode);
	if (!suspend_encryptor_transform) {
		printk("Suspend2: Failed to initialise the encryption "
				"transform (%s, mode %d).\n",
				suspend_encryptor_name, suspend_encryptor_mode);
		suspend_encryption_ops.enabled = 0;
		return 1;
	}

	if (rw == READ)
		bufofs = PAGE_SIZE;
	else
		bufofs = 0;

	suspend_key_len = strlen(suspend_encryptor_key);

	if (crypto_cipher_setkey(suspend_encryptor_transform, suspend_encryptor_key, 
				suspend_key_len)) {
		printk("%d is an invalid key length for cipher %s.\n",
					suspend_key_len,
					suspend_encryptor_name);
		return 1;
	}
	
	if (rw != READ) {
		crypto_cipher_set_iv(suspend_encryptor_transform,
				suspend_encryptor_iv,
				crypto_tfm_alg_ivsize(suspend_encryptor_transform));
	}
		
	return 0;
}

/* ---- Exported functions ---- */

/* suspend_encrypt_write_chunk()
 *
 * Description:	Encrypt a page of data, buffering output and passing on
 * 		filled pages to the next module in the pipeline.
 * Arguments:	Buffer_page:	Pointer to a buffer of size PAGE_SIZE, 
 * 				containing data to be encrypted.
 * Returns:	0 on success. Otherwise the error is that returned by later
 * 		modules, -ECHILD if we have a broken pipeline or -EIO if
 * 		zlib errs.
 */
static int suspend_encrypt_write_chunk(struct page *buffer_page)
{
	int ret; 
	unsigned int len;
	u16 len_written;
	char *buffer_start;
	
	if (!suspend_encryptor_transform)
		return next_driver->write_chunk(buffer_page);

	buffer_start = kmap(buffer_page);
	memcpy(page_buffer, buffer_start, PAGE_SIZE);
	kunmap(buffer_page);
	
	bytes_in += PAGE_SIZE;

	len = PAGE_SIZE;

	ret = crypto_cipher_encrypt(suspend_encryptor_transform,
			suspend_crypt_sg, suspend_crypt_sg, PAGE_SIZE);
	
	if (ret) {
		printk("Encryption failed.\n");
		return -EIO;
	}
	
	len_written = (u16) len;

	ret = next_driver->write_chunk(virt_to_page(page_buffer));

	return ret;
}

/* rw_init()
 *
 * Description:	Prepare to read a new stream of data.
 * Arguments:	int: Section of image about to be read.
 * Returns:	int: Zero on success, error number otherwise.
 */
static int suspend_encrypt_rw_init(int rw, int stream_number)
{
	int result;

	next_driver = suspend_get_next_filter(&suspend_encryption_ops);

	if (!next_driver) {
		printk("Encryption Driver: Argh! I'm at the end of the pipeline!");
		return -ECHILD;
	}
	
	if ((result = suspend_encrypt_rw_prepare(rw))) {
		set_result_state(SUSPEND_ENCRYPTION_SETUP_FAILED);
		suspend_encrypt_rw_cleanup(rw);
		return result;
	}
	
	if ((result = allocate_local_buffer()))
		return result;

	if (rw == WRITE && stream_number == 2)
		bytes_in = bytes_out = 0;
	
	bufofs = (rw == READ) ? PAGE_SIZE : 0;

	return 0;
}

/* suspend_encrypt_read_chunk()
 *
 * Description:	Retrieve data from later modules and deencrypt it until the
 * 		input buffer is filled.
 * Arguments:	Buffer_start: 	Pointer to a buffer of size PAGE_SIZE.
 * 		Sync:		Whether the previous module (or core) wants its
 * 				data synchronously.
 * Returns:	Zero if successful. Error condition from me or from downstream
 * 		on failure.
 */
static int suspend_encrypt_read_chunk(struct page *buffer_page, int sync)
{
	int ret; 
	char *buffer_start;

	if (!suspend_encryptor_transform)
		return next_driver->read_chunk(buffer_page, sync);

	/* 
	 * All our reads must be synchronous - we can't deencrypt
	 * data that hasn't been read yet.
	 */

	if ((ret = next_driver->read_chunk(
			virt_to_page(page_buffer), SUSPEND_SYNC)) < 0) {
		printk("Failed to read an encrypted block.\n");
		return ret;
	}

	ret = crypto_cipher_decrypt(suspend_encryptor_transform,
			suspend_crypt_sg, suspend_crypt_sg, PAGE_SIZE);

	if (ret)
		printk("Decrypt function returned %d.\n", ret);

	buffer_start = kmap(buffer_page);
	memcpy(buffer_start, page_buffer, PAGE_SIZE);
	kunmap(buffer_page);
	return ret;
}

/* suspend_encrypt_print_debug_stats
 *
 * Description:	Print information to be recorded for debugging purposes into a
 * 		buffer.
 * Arguments:	buffer: Pointer to a buffer into which the debug info will be
 * 			printed.
 * 		size:	Size of the buffer.
 * Returns:	Number of characters written to the buffer.
 */
static int suspend_encrypt_print_debug_stats(char *buffer, int size)
{
	int len;
	
	if (*suspend_encryptor_name)
		len = snprintf_used(buffer, size, "- Encryptor is '%s'.\n",
				suspend_encryptor_name);
	else
		len = snprintf_used(buffer, size, "- Encryptor is not set.\n");
	return len;
}

/* encryption_memory_needed
 *
 * Description:	Tell the caller how much memory we need to operate during
 * 		suspend/resume.
 * Returns:	Unsigned long. Maximum number of bytes of memory required for
 * 		operation.
 */
static unsigned long suspend_encrypt_memory_needed(void)
{
	return PAGE_SIZE;
}

static unsigned long suspend_encrypt_storage_needed(void)
{
	return 4 + strlen(suspend_encryptor_name) +
		(suspend_encryptor_save_key_and_iv ?
		 (4 + strlen(suspend_encryptor_key) +
		  strlen(suspend_encryptor_iv)) : 0);
}
	
/* suspend_encrypt_save_config_info
 *
 * Description:	Save informaton needed when reloading the image at resume time.
 * Arguments:	Buffer:		Pointer to a buffer of size PAGE_SIZE.
 * Returns:	Number of bytes used for saving our data.
 */
static int suspend_encrypt_save_config_info(char *buffer)
{
	int buf_offset, str_size;

	str_size = strlen(suspend_encryptor_name);
	*buffer = (char) str_size;
	strncpy(buffer + 1, suspend_encryptor_name, str_size + 1);
	buf_offset = str_size + 2;

	*(buffer + buf_offset) = (char) suspend_encryptor_mode;
	buf_offset++;

	*(buffer + buf_offset) = (char) suspend_encryptor_save_key_and_iv;
	buf_offset++;

	if (suspend_encryptor_save_key_and_iv) {
		
		str_size = strlen(suspend_encryptor_key);
		*(buffer + buf_offset) = (char) str_size;
		strncpy(buffer + buf_offset + 1, suspend_encryptor_key, str_size + 1);

		buf_offset+= str_size + 2;

		str_size = strlen(suspend_encryptor_iv);
		*(buffer + buf_offset) = (char) str_size;
		strncpy(buffer + buf_offset + 1, suspend_encryptor_iv, str_size + 1);

		buf_offset += str_size + 2;
	}

	return buf_offset;
}

/* suspend_encrypt_load_config_info
 *
 * Description:	Reload information needed for deencrypting the image at 
 * 		resume time.
 * Arguments:	Buffer:		Pointer to the start of the data.
 *		Size:		Number of bytes that were saved.
 */
static void suspend_encrypt_load_config_info(char *buffer, int size)
{
	int buf_offset, str_size;

	str_size = (int) *buffer;
	strncpy(suspend_encryptor_name, buffer + 1, str_size + 1);
	buf_offset = str_size + 2;
	
	suspend_encryptor_mode = (int) *(buffer + buf_offset);
	buf_offset++;

	suspend_encryptor_save_key_and_iv = (int) *(buffer + buf_offset);
	buf_offset++;

	if (suspend_encryptor_save_key_and_iv) {
		str_size = (int) *(buffer + buf_offset);
		strncpy(suspend_encryptor_key, buffer + buf_offset + 1, str_size + 1);

		buf_offset+= str_size + 2;

		str_size = (int) *(buffer + buf_offset);
		strncpy(suspend_encryptor_iv, buffer + buf_offset + 1, str_size + 1);

		buf_offset += str_size + 2;
	} else {
		*suspend_encryptor_key = 0;
		*suspend_encryptor_iv = 0;
	}
	
	if (buf_offset != size) {
		printk("Suspend Encryptor config info size mismatch (%d != %d): settings ignored.\n",
				buf_offset, size);
		*suspend_encryptor_key = 0;
		*suspend_encryptor_iv = 0;
	}
	return;
}

/*
 * data for our sysfs entries.
 */
static struct suspend_sysfs_data sysfs_params[] = {
	{
		SUSPEND2_ATTR("algorithm", SYSFS_RW),
		SYSFS_STRING(suspend_encryptor_name, 31, SYSFS_SM_NOT_NEEDED)
	},

	{
		SUSPEND2_ATTR("mode", SYSFS_RW),
		SYSFS_INT(&suspend_encryptor_mode, 0, 3)
	},

	{
		SUSPEND2_ATTR("save_key_and_iv", SYSFS_RW),
		SYSFS_INT(&suspend_encryptor_save_key_and_iv, 0, 1)
	},

	{
		SUSPEND2_ATTR("key", SYSFS_RW),
		SYSFS_STRING(suspend_encryptor_key, 255, SYSFS_SM_NOT_NEEDED)
	},

	{
		SUSPEND2_ATTR("iv", SYSFS_RW),
		SYSFS_STRING(suspend_encryptor_iv, 255, SYSFS_SM_NOT_NEEDED)
	},

	{
		SUSPEND2_ATTR("enabled", SYSFS_RW),
		SYSFS_INT(&suspend_encryption_ops.enabled, 0, 1)
	},
	
};

/*
 * Ops structure.
 */

static struct suspend_module_ops suspend_encryption_ops = {
	.type			= FILTER_MODULE,
	.name			= "Encryptor",
	.module			= THIS_MODULE,
	.memory_needed 		= suspend_encrypt_memory_needed,
	.print_debug_info	= suspend_encrypt_print_debug_stats,
	.save_config_info	= suspend_encrypt_save_config_info,
	.load_config_info	= suspend_encrypt_load_config_info,
	.storage_needed		= suspend_encrypt_storage_needed,
	
	.rw_init		= suspend_encrypt_rw_init,
	.rw_cleanup		= suspend_encrypt_rw_cleanup,

	.write_chunk		= suspend_encrypt_write_chunk,
	.read_chunk		= suspend_encrypt_read_chunk,
};

/* ---- Registration ---- */

static __init int suspend_encrypt_load(void)
{
	int result;
	int i, numfiles = sizeof(sysfs_params) / sizeof(struct suspend_sysfs_data);

	printk("Suspend2 Encryption Driver loading.\n");
	if (!(result = suspend_register_module(&suspend_encryption_ops))) {
		struct kobject *kobj = make_suspend2_sysdir("encryption");
		for (i=0; i< numfiles; i++)
			suspend_register_sysfs_file(kobj, &sysfs_params[i]);
		suspend_encryption_ops.enabled = 0;
	} else
		printk("Suspend2 Encryption Driver unable to register!\n");
	return result;
}

#ifdef MODULE
static __exit void suspend_compress_unload(void)
{
	printk("Suspend2 Encryption Driver unloading.\n");
	for (i=0; i< numfiles; i++)
		suspend_unregister_sysfs_file(&sysfs_params[i]);
	suspend_unregister_module(&suspend_encryption_ops);
}

module_init(suspend_encrypt_load);
module_exit(suspend_encrypt_unload);
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Nigel Cunningham");
MODULE_DESCRIPTION("Encryption Support for Suspend2");
#else
late_initcall(suspend_encrypt_load);
#endif
