/*
 * kernel/power/atomic_copy.c
 *
 * Copyright 2004-2006 Nigel Cunningham <nigel@suspend2.net>
 *
 * Distributed under GPLv2.
 *
 * Routines for doing the atomic save/restore.
 */

#include <linux/suspend.h>
#include <linux/highmem.h>
#include <linux/bootmem.h>
#include <linux/vt_kern.h>
#include <asm/setup.h>
#include "suspend.h"
#include "storage.h"
#include "power_off.h"
#include "ui.h"
#include "power.h"
#include "io.h"
#include "prepare_image.h"
#include "pageflags.h"
#include "extent.h"

static unsigned long state1 __nosavedata = 0;
static unsigned long state2 __nosavedata = 0;
static int state3 __nosavedata = 0;
static int io_speed_save[2][2] __nosavedata;
__nosavedata char suspend_resume_commandline[COMMAND_LINE_SIZE];

#define SUSPEND_FREQ_START 2000
#define FREQ_CHANGE -100
#define GROUP_SIZE 10
#define RESUME_FREQ_START (SUSPEND_FREQ_START + (GROUP_SIZE - 1) * FREQ_CHANGE)

static int __nosavedata suspend_beeping = 0;
static int __nosavedata this_freq = SUSPEND_FREQ_START;

extern void suspend_power_down(void);
extern int swsusp_resume(void);
extern int suspend2_in_suspend __nosavedata;
int extra_pd1_pages_used;

#ifdef CONFIG_HIGHMEM
static dyn_pageflags_t __nosavedata origmap;
static dyn_pageflags_t __nosavedata copymap;
static unsigned long __nosavedata origoffset;
static unsigned long __nosavedata copyoffset;
static __nosavedata int o_zone_num, c_zone_num;

struct zone_data {
	unsigned long start_pfn;
	unsigned long end_pfn;
	int is_highmem;
};

static __nosavedata struct zone_data *zone_nosave;
static __nosavedata unsigned long boot_max_pfn;

/**
 * suspend_init_nosave_zone_table: Set up nosave copy of zone table data.
 *
 * Zone information might be overwritten during the copy back, so we copy
 * the fields we need to a non-conflicting page and use it.
 **/
static void suspend_init_nosave_zone_table(void)
{
	struct zone *zone;
	
	zone_nosave = (struct zone_data *) suspend_get_nonconflicting_page();

	BUG_ON(!zone_nosave);

	/* It should all fit in one page. */
	BUG_ON(ZONETABLE_MASK > (PAGE_SIZE / sizeof(struct zone_data)));

	memset((char *) zone_nosave, 0, PAGE_SIZE);

	for_each_zone(zone) {
		if (populated_zone(zone)) {
			int zone_num = page_zone_id(pfn_to_page(zone->zone_start_pfn));
			zone_nosave[zone_num].start_pfn = zone->zone_start_pfn;
			zone_nosave[zone_num].end_pfn = zone->zone_start_pfn +
				zone->spanned_pages - 1;
			zone_nosave[zone_num].is_highmem = is_highmem(zone);
		}
	}

	boot_max_pfn = max_pfn;
}

/**
 * __suspend_get_next_bit_on: Atomic-copy safe location of next bit on in a bitmap.
 * 
 * @bitmap:	The bitmap we are traversing.
 * @zone_num:	Which zone we are in.
 * @counter:	The current pfn.
 *
 * A version of __get_next_bit_on that can be used when doing the atomic
 * copy. While doing it, we can't rely on the zone information being in
 * a constant location.
 **/
static unsigned long __suspend_get_next_bit_on(dyn_pageflags_t bitmap,
		int *zone_num, unsigned long counter)
{
	unsigned long *ul_ptr = NULL;
	int reset_ul_ptr = 1, pagebit;
	BUG_ON(*zone_num == (1 << ZONETABLE_SHIFT));

	if (counter == boot_max_pfn) {
		*zone_num = 0;
		while (!zone_nosave[*zone_num].end_pfn)
			(*zone_num)++;
		counter = zone_nosave[*zone_num].start_pfn - 1;
	}

	do {
		counter++;
		if (counter > zone_nosave[*zone_num].end_pfn) {
			(*zone_num)++;
			while (*zone_num < (1 << ZONETABLE_SHIFT) &&
					!zone_nosave[*zone_num].end_pfn)
				(*zone_num)++;
			
			if (*zone_num == (1 << ZONETABLE_SHIFT))
				return boot_max_pfn;
			counter = zone_nosave[*zone_num].start_pfn;
			reset_ul_ptr = 1;
		} else
			if (!(counter & BIT_NUM_MASK))
				reset_ul_ptr = 1;

		pagebit = PAGEBIT(counter);

		if (reset_ul_ptr) {
			reset_ul_ptr = 0;
			ul_ptr = PAGE_UL_PTR(bitmap, *zone_num,
				(counter - zone_nosave[*zone_num].start_pfn));
		}
		
		if (!(*ul_ptr & ~((1 << pagebit) - 1))) {
			counter += BITS_PER_LONG - pagebit - 1;
			continue;
		}
	} while(!test_bit(pagebit, ul_ptr));

	return counter;
}

/*
 * copyback_high: Restore highmem pages.
 *
 * Iterate through the source and destination bitmaps, restoring
 * highmem pages that were atomically copied.
 */
void copyback_high(void)
{
	unsigned long *origpage;
	unsigned long *copypage;

	origoffset = __suspend_get_next_bit_on(origmap, &o_zone_num, boot_max_pfn);
	copyoffset = __suspend_get_next_bit_on(copymap, &c_zone_num, boot_max_pfn);

	while (o_zone_num < (1 << ZONETABLE_SHIFT)) {
		if (zone_nosave[o_zone_num].is_highmem) {
			origpage = (unsigned long *) kmap_atomic(pfn_to_page(origoffset), KM_USER1);
			copypage = (unsigned long *) __va(copyoffset << PAGE_SHIFT);

			memcpy(origpage, copypage, PAGE_SIZE);

			kunmap_atomic(origpage, KM_USER1);
		}
		
		origoffset = __suspend_get_next_bit_on(origmap, &o_zone_num, origoffset);
		copyoffset = __suspend_get_next_bit_on(copymap, &c_zone_num, copyoffset);
	}
}
#else
void copyback_high(void) { }
#endif

static void suspend_do_beep(unsigned int count)
{
	/* enable counter 2 */
	outb_p(inb_p(0x61) | 3, 0x61);
	/* set command for counter 2, 2 byte write */
	outb_p(0xB6, 0x43);
	/* select desired HZ */
	outb_p(count & 0xff, 0x42);
	outb((count >> 8) & 0xff, 0x42);

	mdelay(100);

	/* disable counter 2 */
	outb(inb_p(0x61) & 0xFC, 0x61);
}

/* 
 * Beep in groups of three, with different frequencies so the individual
 * beeps can be distinguished. In addition, frequency rises when suspending
 * and drops when resuming.
 *
 * Yes, I know freq isn't really _the_ frequency.
 */

void suspend2_beep(int suspending)
{
	static int last_action = 0;
	int limit = suspending ? RESUME_FREQ_START : SUSPEND_FREQ_START;

	if (!suspend_beeping)
		return;

	if (suspending != last_action)
		this_freq = suspending ? SUSPEND_FREQ_START : RESUME_FREQ_START;
	
	suspend_do_beep(this_freq);

	if (this_freq == limit) {
		/* Time for a gap & freq reset */
		mdelay(300);
		this_freq = suspending ? SUSPEND_FREQ_START : RESUME_FREQ_START;
	} else
		this_freq += (suspending ? FREQ_CHANGE : -FREQ_CHANGE);

	last_action = suspending;
}

/*
 * prepare_suspend2_pbe_list
 *
 * Prepare pageset2 pages for doing the atomic copy. If necessary,
 * we allocate extra pages.
 *
 */

void prepare_suspend2_pbe_list(void)
{
	unsigned long orig_pfn, copy_pfn;
	int i = 1;
	struct pbe *this_pbe = NULL, *last_pbe = NULL;

	orig_pfn = copy_pfn = max_pfn;

	pagedir_nosave = NULL;

	do {
		if (!this_pbe ||
		    ((((unsigned long) this_pbe) & (PAGE_SIZE - 1)) 
		     + 2 * sizeof(struct pbe)) > PAGE_SIZE) {
			/* Get the next page for pbes */
			this_pbe = (struct pbe *) suspend_get_nonconflicting_page();
			BUG_ON(!this_pbe);
			BUG_ON(PagePageset1(virt_to_page(this_pbe)));
		} else
			this_pbe++;

		do {
			orig_pfn = get_next_bit_on(pageset1_map, orig_pfn);
			if (orig_pfn == max_pfn)
				return;
			copy_pfn = get_next_bit_on(pageset1_copy_map, copy_pfn);
		} while (PageHighMem(pfn_to_page(orig_pfn)));
		
		if (!last_pbe)
			pagedir_nosave = this_pbe;
		else
			last_pbe->next = this_pbe;

		last_pbe = this_pbe;
		this_pbe->orig_address = (unsigned long) page_address(pfn_to_page(orig_pfn));
		this_pbe->address = (unsigned long) page_address(pfn_to_page(copy_pfn));
		this_pbe->next = NULL; /* get_nonconflicting_page doesn't get zeroed pages */

		i++;

	} while (1);
}

/*
 * copyback_post: Post atomic-restore actions.
 *
 * After doing the atomic restore, we have a few more things to do:
 * 1) We want to retain some values across the restore, so we now copy
 * these from the nosave variables to the normal ones.
 * 2) Set the status flags.
 * 3) Resume devices.
 * 4) Get userui to redraw.
 * 5) Reread the page cache.
 */

void copyback_post(void)
{
	int loop;

	suspend2_beep(0);

	suspend_action = state1;
	suspend_debug_state = state2;
	console_loglevel = state3;

	for (loop = 0; loop < 4; loop++)
		suspend_io_time[loop/2][loop%2] =
			io_speed_save[loop/2][loop%2];

	set_suspend_state(SUSPEND_NOW_RESUMING);
	set_suspend_state(SUSPEND_PAGESET2_NOT_LOADED);

	if (pm_ops && pm_ops->finish && suspend_powerdown_method > 3)
		pm_ops->finish(suspend_powerdown_method);

	if (suspend_activate_storage(1))
		panic("Failed to reactivate our storage.");

	userui_redraw();

	suspend_cond_pause(1, "About to reload secondary pagedir.");

	if (read_pageset2(0))
		panic("Unable to successfully reread the page cache.");

	clear_suspend_state(SUSPEND_PAGESET2_NOT_LOADED);
	
	suspend_prepare_status(DONT_CLEAR_BAR, "Cleaning up...");
}

/*
 * suspend_post_context_save: Steps after saving the cpu context.
 *
 * Steps taken after saving the CPU state to make the actual
 * atomic copy.
 *
 * Called from swsusp_save in snapshot.c.
 */

int suspend_post_context_save(void)
{
	int old_ps1_size = pagedir1.pageset_size;
	int old_ps2_size = pagedir2.pageset_size;
	
	BUG_ON(!irqs_disabled());

	suspend2_beep(1);

	suspend_recalculate_image_contents(1);

	extra_pd1_pages_used = pagedir1.pageset_size - old_ps1_size;

	if ((pagedir1.pageset_size - old_ps1_size) > extra_pd1_pages_allowance) {
		abort_suspend("Pageset1 has grown by %d pages. "
			"extra_pages_allowance is currently only %d.\n",
			pagedir1.pageset_size - old_ps1_size,
			extra_pd1_pages_allowance);
		return -1;
	}

	BUG_ON(old_ps2_size != pagedir2.pageset_size);

	BUG_ON(!irqs_disabled());

	if (!test_action_state(SUSPEND_TEST_FILTER_SPEED) &&
	    !test_action_state(SUSPEND_TEST_BIO))
		suspend_copy_pageset1();


	suspend2_beep(1);

	return 0;
}

/* suspend_copy_pageset1: Do the atomic copy of pageset1.
 *
 * Make the atomic copy of pageset1. We can't use copy_page (as we once did)
 * because we can't be sure what side effects it has. On my old Duron, with
 * 3DNOW, kernel_fpu_begin increments preempt count, making our preempt
 * count at resume time 4 instead of 3.
 * 
 * We don't want to call kmap_atomic unconditionally because it has the side
 * effect of incrementing the preempt count, which will leave it one too high
 * post resume (the page containing the preempt count will be copied after
 * its incremented. This is essentially the same problem.
 */

void suspend_copy_pageset1(void)
{
	int i;
	unsigned long source_index, dest_index;

	source_index = get_next_bit_on(pageset1_map, max_pfn);
	dest_index = get_next_bit_on(pageset1_copy_map, max_pfn);

	for (i = 0; i < pagedir1.pageset_size; i++) {
		unsigned long *origvirt, *copyvirt;
		struct page *origpage;
		int loop = (PAGE_SIZE / sizeof(unsigned long)) - 1;

		origpage = pfn_to_page(source_index);
		
	       	if (PageHighMem(origpage))
			origvirt = kmap_atomic(origpage, KM_USER0);
		else
			origvirt = page_address(origpage);

		copyvirt = (unsigned long *) page_address(pfn_to_page(dest_index));

		while (loop >= 0) {
			*(copyvirt + loop) = *(origvirt + loop);
			loop--;
		}
		
		if (PageHighMem(origpage))
			kunmap_atomic(origvirt, KM_USER0);
		
		source_index = get_next_bit_on(pageset1_map, source_index);
		dest_index = get_next_bit_on(pageset1_copy_map, dest_index);
	}
}

int suspend2_suspend(void)
{
	int error;

	suspend2_beep(1);

	if (test_action_state(SUSPEND_PM_PREPARE_CONSOLE))
		pm_prepare_console();

	if ((error = arch_prepare_suspend()))
		return error;
	local_irq_disable();
	/* At this point, device_suspend() has been called, but *not*
	 * device_power_down(). We *must* device_power_down() now.
	 * Otherwise, drivers for some devices (e.g. interrupt controllers)
	 * become desynchronized with the actual state of the hardware
	 * at resume time, and evil weirdness ensues.
	 */
	if ((error = device_power_down(PMSG_FREEZE))) {
		set_result_state(SUSPEND_DEVICE_REFUSED);
		set_result_state(SUSPEND_ABORTED);
		printk(KERN_ERR "Some devices failed to power down, aborting suspend\n");
		goto enable_irqs;
	}

	suspend2_beep(1);

	save_processor_state();
	if ((error = swsusp_arch_suspend()))
		printk(KERN_ERR "Error %d suspending\n", error);
	/* Restore control flow appears here */
	restore_processor_state();

	suspend2_beep(1);

	if (!suspend2_in_suspend)
		copyback_high();
	device_power_up();
enable_irqs:
	local_irq_enable();
	if (test_action_state(SUSPEND_PM_PREPARE_CONSOLE))
		pm_restore_console();

	suspend2_beep(1);

	return error;
}

/*
 * suspend_atomic_restore
 *
 * Get ready to do the atomic restore. This part gets us into the same
 * state we are in prior to do calling do_suspend2_lowlevel while
 * suspending: hotunplugging secondary cpus and freeze processes,
 * before starting the thread that will do the restore.
 */
int suspend_atomic_restore(void)
{
	int error, loop;

	suspend2_beep(0);

	suspend_prepare_status(DONT_CLEAR_BAR,	"Atomic restore preparation");
	prepare_suspend2_pbe_list();

	suspend2_beep(0);

	if (test_action_state(SUSPEND_PM_PREPARE_CONSOLE))
		pm_prepare_console();

	disable_nonboot_cpus();

	if ((error = device_suspend(PMSG_FREEZE))) {
		printk("Some devices failed to suspend\n");
		if (test_action_state(SUSPEND_PM_PREPARE_CONSOLE))
			pm_restore_console();
		BUG();
	}

	suspend2_beep(0);

#ifdef CONFIG_HIGHMEM
	origmap = pageset1_map;
	copymap = pageset1_copy_map;
	suspend_init_nosave_zone_table();
#endif

	state1 = suspend_action;
	state2 = suspend_debug_state;
	state3 = console_loglevel;
	
	for (loop = 0; loop < 4; loop++)
		io_speed_save[loop/2][loop%2] =
			suspend_io_time[loop/2][loop%2];
	memcpy(suspend_resume_commandline, saved_command_line, COMMAND_LINE_SIZE);

	mb();

	local_irq_disable();
	if (device_power_down(PMSG_FREEZE)) {
		printk(KERN_ERR "Some devices failed to power down. Very bad.\n");
		BUG();
	}

	/* We'll ignore saved state, but this gets preempt count (etc) right */
	save_processor_state();
	suspend2_beep(0);
	error = swsusp_arch_resume();
	/* Code below is only ever reached in case of failure. Otherwise
	 * execution continues at place where swsusp_arch_suspend was called.
         */
	BUG();
	return 1;
}

static int suspend_beep_setup(char *str)
{
	suspend_beeping = !!simple_strtol(str, NULL, 0);
	return 1;
}

__setup("suspend_beep=", suspend_beep_setup);

