/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "widgets/view-base.h"

#include "cstr-intl.h"
#include "instance-manager.h"
#include "stock.h"
#include "undo-manager.h"
#include "ucompose.h"

#include "layout-base.h"
#include "disc-info-base.h"

#include <gtkmm/menu.h>
#include <gtkmm/actiongroup.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/stock.h>

namespace Coaster
{

namespace Widgets
{

View_Base::View_Base()
: m_pMenu_Popup(0),
  m_pLayout(0), m_pDiscInfo(0),
  m_refViewActionGroup(Gtk::ActionGroup::create("CoasterViewSpecificActions")),
  m_bModified(false), m_strDocTitle(_("Untitled Layout"))
{
  set_border_width(8);
  set_spacing(8);
}

View_Base::~View_Base()
{}

void View_Base::init()
{
  using namespace Gtk;
  undo_manager = UndoManager::create();
  undo_manager->signal_changed().connect(sigc::mem_fun(*this, &View_Base::on_undo_manager_changed));

  create_store();
  create_layout();
  create_disc_info();

  // Undo Actions:
  m_refUndo = Action::create("CoasterAction_Undo", Gtk::Stock::UNDO);
  m_refViewActionGroup->add(m_refUndo,
                            AccelKey("<control>Z"),
                            sigc::mem_fun(*this, &View_Base::on_edit_undo));

  m_refRedo = Action::create("CoasterAction_Redo", Gtk::Stock::REDO);
  m_refViewActionGroup->add(m_refRedo,
                            AccelKey("<control><shift>Z"),
                            sigc::mem_fun(*this, &View_Base::on_edit_redo));

  // Layout Actions:
  m_refViewActionGroup->add(Action::create("CoasterAction_Layout_Properties",
                                           Gtk::Stock::PROPERTIES,
                                           _("Proper_ties"),
                                           _("Change the properties of the current layout")),
                            sigc::mem_fun(*this, &View_Base::on_layout_properties));

  // Disc Actions:
  m_refBurn = Action::create("CoasterAction_Disc_Burn", Stock::DISC_BURN,
                             _("_Burn..."), _("Burn current layout to a disc"));
  m_refBurn->property_is_important() = true;
  m_refViewActionGroup->add(m_refBurn, Gtk::AccelKey("<control>B"),
                            sigc::mem_fun(*this, &View_Base::on_disc_burn));

  enable_selection_actions(false);
  enable_layout_item_actions(false);
  m_refUndo->set_sensitive(false);
  m_refRedo->set_sensitive(false);

  show_all();
}

void View_Base::create_store()
{}

void View_Base::create_layout()
{
  using namespace Gtk;
  ScrolledWindow* sw = new ScrolledWindow();
  sw->set_policy(POLICY_NEVER, POLICY_AUTOMATIC);
  sw->set_shadow_type(SHADOW_IN);

  m_pLayout->signal_selection_changed().connect(sigc::mem_fun(*this,
                                                    &View_Base::on_layout_selection_changed));
  sw->add(*manage(m_pLayout));
  pack_start(*manage(sw));
}

void View_Base::create_disc_info()
{
  pack_start(*manage(m_pDiscInfo), false, false);
}

Glib::ustring View_Base::get_document_title() const
{
  return m_strDocTitle;
}

Glib::ustring View_Base::get_ui_location() const
{
  return Glib::ustring();
}

Glib::RefPtr<Gtk::ActionGroup> View_Base::get_ui_action_group() const
{
  return m_refViewActionGroup;
}

void View_Base::set_popup_menu(Gtk::Menu* popup_menu)
{
  m_pMenu_Popup = popup_menu;
}

void View_Base::popup(guint button,
                      guint32 activate_time)
{
  m_pMenu_Popup->popup(button, activate_time);
}

void View_Base::set_modified(bool modified)
{
  m_bModified = modified;
  m_signal_modified.emit(modified);
}

bool View_Base::get_modified() const
{
  return m_bModified;
}

LayoutType View_Base::get_layout_type() const
{
  return COASTER_LAYOUT_UNKNOWN;
}

View_Base::type_signal_modified& View_Base::signal_modified()
{
  return m_signal_modified;
}

View_Base::type_signal_sensitive& View_Base::signal_sensitive()
{
  return m_signal_sensitive;
}

void View_Base::on_edit_undo()
{
  undo_manager->undo();
}

void View_Base::on_edit_redo()
{
  undo_manager->redo();
}

void View_Base::on_layout_properties()
{
  set_properties();
}

void View_Base::on_store_modified(bool modified)
{
  set_modified(modified);
}

void View_Base::on_layout_selection_changed(int number_selected)
{
  enable_selection_actions(number_selected);
}

void View_Base::on_undo_manager_changed()
{
  m_refUndo->property_label() = String::ucompose("_Undo%1", undo_manager->undo_label());
  m_refRedo->property_label() = String::ucompose("_Redo%1", undo_manager->redo_label());
  
  m_refUndo->set_sensitive(undo_manager->undo_size());
  m_refRedo->set_sensitive(undo_manager->redo_size());
}

void View_Base::enable_layout_item_actions(bool enable)
{
  m_refBurn->set_sensitive(enable);
}

} // namespace Widgets

} // namespace Coaster
