/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "widgets/tray-icon.h"

#include "cstr-debug.h"
#include "cstr-intl.h"
#include "exception.h"

#include <X11/Xatom.h>
#include <gdkmm/pixbuf.h>
#include <gtkmm/eventbox.h>
#include <gtkmm/image.h>
#include <gdkmm/drawable.h>
#include <gdkmm/window.h>

namespace
{

GdkFilterReturn manager_filter(GdkXEvent* xevent,
                               GdkEvent* event,
                               gpointer data)
{
  using namespace Coaster;

  Widgets::TrayIcon* tray_icon = static_cast<Widgets::TrayIcon*>(data);
  XEvent* xev = (XEvent*)xevent;

  if(xev->xany.type == ClientMessage &&
     xev->xclient.message_type == tray_icon->get_manager_atom() &&
     static_cast<unsigned int>(xev->xclient.data.l[1]) == tray_icon->get_selection_atom())
    tray_icon->update_manager_window();
  else if(xev->xany.window == tray_icon->get_manager_window())
  {
    if(xev->xany.type == PropertyNotify &&
       xev->xproperty.atom == tray_icon->get_orientation_atom())
      tray_icon->get_orientation_property();

    if(xev->xany.type == DestroyNotify)
      tray_icon->update_manager_window();
  }
  
  return GDK_FILTER_CONTINUE;
}

} // anonymous namespace

namespace Coaster
{

namespace Widgets
{

TrayIcon::TrayIcon()
: Gtk::Plug(static_cast<GdkNativeWindow>(0)),
  m_pEventBox(0), m_pImage(0)
{
  property_title() = _("Coaster Tray Icon");
  orientation = Gtk::ORIENTATION_HORIZONTAL;

  add_events(Gdk::PROPERTY_CHANGE_MASK);
  m_Tooltips.enable();

  m_pEventBox = new Gtk::EventBox();
  m_pImage = new Gtk::Image();
  
  m_pEventBox->add(*manage(m_pImage));
  add(*manage(m_pEventBox));

  m_pEventBox->set_events(Gdk::BUTTON_PRESS_MASK);
  m_pEventBox->signal_button_press_event().connect(sigc::mem_fun(*this, &TrayIcon::on_eb_button_press_event));

  show_all();
}

TrayIcon::~TrayIcon()
{}

void TrayIcon::set(const Glib::RefPtr<Gdk::Pixbuf>& pixbuf,
                     const Glib::ustring& tip,
                     const Glib::ustring& private_tip)
{
  m_pImage->set(pixbuf);
  m_Tooltips.set_tip(*m_pEventBox, tip, private_tip);
}

void TrayIcon::set(const Glib::ustring& tip,
                     const Glib::ustring& private_tip)
{
  if(m_pImage->get_storage_type() == Gtk::IMAGE_EMPTY)
    throw Exception(_("TrayIcon: Cannot set the tooltip of an empty image"));
  else
    m_Tooltips.set_tip(*m_pEventBox, tip, private_tip);
}

void TrayIcon::unset()
{
  m_pImage->clear();
  m_Tooltips.unset_tip(*m_pEventBox);
}

Atom TrayIcon::get_manager_atom() const
{
  return manager_atom;
}

Atom TrayIcon::get_selection_atom() const
{
  return selection_atom;
}

Atom TrayIcon::get_orientation_atom() const
{
  return orientation_atom;
}

C_X_Window TrayIcon::get_manager_window() const
{
  return manager_window;
}

void TrayIcon::on_realize()
{
  Gtk::Plug::on_realize();

  Glib::RefPtr<Gdk::Screen> screen = get_screen();
  Glib::RefPtr<Gdk::Display> display = screen->get_display();
  Display* xdisplay = gdk_x11_display_get_xdisplay(display->gobj());

  std::ostringstream oss;
  oss << "_NET_SYSTEM_TRAY_S" << screen->get_number();
  Glib::ustring buffer = oss.str();

  selection_atom = XInternAtom(xdisplay, buffer.c_str(), False);
  manager_atom = XInternAtom(xdisplay, "MANAGER", False);
  system_tray_opcode_atom = XInternAtom(xdisplay, "_NET_SYSTEM_TRAY_OPCODE", False);
  orientation_atom = XInternAtom(xdisplay, "_NET_SYSTEM_TRAY_ORIENTATION", False);

  update_manager_window();

  Glib::RefPtr<Gdk::Window> root_window = screen->get_root_window();

  if(root_window)
    root_window->add_filter(manager_filter, this);
}

void TrayIcon::on_unrealize()
{
  if(manager_window != None)
  {
    Glib::RefPtr<Gdk::Window> gdkwin = Glib::wrap(GDK_WINDOW_OBJECT(gdk_window_lookup_for_display(get_display()->gobj(), manager_window)));

    if(gdkwin)
      gdkwin->remove_filter(manager_filter, this);
  }

  Glib::RefPtr<Gdk::Window> root_window = get_screen()->get_root_window();
 
  if(root_window)
    root_window->remove_filter(manager_filter, this);
  
  Gtk::Plug::on_unrealize();
}

bool TrayIcon::on_delete_event(GdkEventAny* event)
{
  return true;
}

bool TrayIcon::on_eb_button_press_event(GdkEventButton* event)
{
  debug("TrayIcon::on_button_press_event");
  return false;
}

void TrayIcon::update_manager_window()
{
  Display* xdisplay = gdk_x11_display_get_xdisplay(get_display()->gobj());

  if(manager_window != None)
  {
    Glib::RefPtr<Gdk::Window> gdkwin =
      Glib::wrap(GDK_WINDOW_OBJECT(gdk_window_lookup_for_display(get_display()->gobj(), manager_window)));

    if(gdkwin)
      gdkwin->remove_filter(manager_filter, this);
  }

  XGrabServer(xdisplay);

  manager_window = XGetSelectionOwner(xdisplay, selection_atom);
  if(manager_window != None)
  {
    XSelectInput(xdisplay, manager_window, StructureNotifyMask|PropertyChangeMask);
  }

  XUngrabServer(xdisplay);
  XFlush(xdisplay);

  if(manager_window != None)
  {
    Glib::RefPtr<Gdk::Window> gdkwin =
      Glib::wrap(GDK_WINDOW_OBJECT(gdk_window_lookup_for_display(get_display()->gobj(),
                                                                 manager_window)));

    if(gdkwin)
      gdkwin->add_filter(manager_filter, this);

    send_dock_request();
    get_orientation_property();
  }
}

void TrayIcon::send_dock_request()
{
  send_manager_message(REQUEST_DOCK, manager_window, get_id(), 0, 0);
}

void TrayIcon::send_manager_message(SystemTrayMessage message,
                                        C_X_Window window,
                                        long data1,
                                        long data2,
                                        long data3)
{
  XClientMessageEvent ev;
  Display* display;

  ev.type = ClientMessage;
  ev.window = window;
  ev.message_type = system_tray_opcode_atom;
  ev.format = 32;
  ev.data.l[0] = gdk_x11_get_server_time(get_window()->gobj());
  ev.data.l[1] = message;
  ev.data.l[2] = data1;
  ev.data.l[3] = data2;
  ev.data.l[4] = data3;

  display = gdk_x11_display_get_xdisplay(get_display()->gobj());
  gdk_error_trap_push();
  XSendEvent(display, manager_window, False, NoEventMask, (XEvent*)&ev);
  XSync(display, False);
  gdk_error_trap_pop();
}

/*Gdk::FilterReturn TrayIcon::manager_filter(GdkXEvent* xevent,
                                               GdkEvent* event)
{
  XEvent* xev = (XEvent*)xevent;

  if(xev->xany.type == ClientMessage &&
     xev->xclient.message_type == manager_atom &&
     static_cast<unsigned int>(xev->xclient.data.l[1]) == selection_atom)
    update_manager_window();
  else if(xev->xany.window == manager_window)
  {
    if(xev->xany.type == PropertyNotify &&
       xev->xproperty.atom == orientation_atom)
      get_orientation_property();

    if(xev->xany.type == DestroyNotify)
      update_manager_window();
  }
  
  return Gdk::FILTER_CONTINUE;
}*/

void TrayIcon::get_orientation_property()
{
  Display* xdisplay = gdk_x11_display_get_xdisplay(get_display()->gobj());

  gdk_error_trap_push();
  Atom type = None;

  int format;
  gulong nitems;
  gulong bytes_after;
  union
  {
    gulong *prop;
    guchar *prop_ch;
  } prop = { NULL };

  int result = XGetWindowProperty(xdisplay, manager_window, orientation_atom,
                                  0, G_MAXLONG, FALSE, XA_CARDINAL, &type,
                                  &format, &nitems, &bytes_after, &(prop.prop_ch));
  int error = gdk_error_trap_pop();

  if(error || result != Success)
    return;

  if(type == XA_CARDINAL)
  {
    Gtk::Orientation orient;
    orient = (prop.prop[0] == ORIENTATION_HORZ) ?
      Gtk::ORIENTATION_HORIZONTAL : Gtk::ORIENTATION_VERTICAL;

    if(orientation != orient)
    {
      orientation = orient;
      g_object_notify(G_OBJECT(gobj()), "orientation");
    }
  }

  if(prop.prop)
    XFree(prop.prop);
}

} // namespace Widgets

} // namespace Coaster
