/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "widgets/combo-history.h"

#include "cstr-strings.h"

#include <gconfmm/value_listhelpers.h>
#include <gconfmm/client.h>

#include <libglademm/xml.h>

namespace Coaster
{

namespace Widgets
{

ComboHistory::ComboHistory(BaseObjectType* cobject,
                           const Glib::RefPtr<Gnome::Glade::Xml>& refXml)
: Gtk::ComboBoxEntryText(cobject), m_bSaving_history(false),
  m_bChanged(false)
{
  m_refClient = Gnome::Conf::Client::get_default_client();
  m_refClient->add_dir(conf_key_search_history);

  get_entry()->set_activates_default();

  load_history();

  notify_cnxn = m_refClient->notify_add(conf_key_search_history,
                                        sigc::mem_fun(*this, &ComboHistory::on_history_changed));
  get_entry()->signal_activate().connect(sigc::mem_fun(*this,
                                                       &ComboHistory::on_entry_activated));
  get_entry()->signal_changed().connect(sigc::mem_fun(*this,
                                                      &ComboHistory::on_entry_changed));
}

ComboHistory::~ComboHistory()
{
  save_history();
  m_refClient->suggest_sync();
  m_refClient->notify_remove(notify_cnxn);
  m_refClient->remove_dir(conf_key_search_history);
}

void ComboHistory::grab_focus()
{
  get_entry()->grab_focus();
}

void ComboHistory::set_entry_text(const Glib::ustring& text)
{
  m_entry_text = text;
}

Glib::ustring ComboHistory::get_entry_text() const
{
  return m_entry_text;
}

void ComboHistory::on_history_changed(guint cnxn,
                                      Gnome::Conf::Entry entry)
{
  if(m_bSaving_history)
  {
    m_bSaving_history = false;
    return;
  }

  load_history();
}

void ComboHistory::on_entry_activated()
{
  std::deque<Glib::ustring>::iterator iter;
  Glib::ustring entry_text = get_entry()->get_text();

  if(!m_bChanged || entry_text == "")
    m_bChanged = false;
  else
    prepend_history(entry_text);

  m_entry_text = "";
}

void ComboHistory::on_entry_changed()
{
  m_entry_text = get_entry()->get_text();
  m_bChanged = true;
}

void ComboHistory::prepend_history(const Glib::ustring& text)
{
  if(text.empty())
    return;

  add_history(text, false);
}

void ComboHistory::append_history(const Glib::ustring& text)
{
  if(text.empty())
    return;

  add_history(text, true);
}

void ComboHistory::add_history(const Glib::ustring& text,
                               bool append)
{
  bool changed = false;

  changed = false;

  for(std::deque<Glib::ustring>::iterator i = m_values.begin() ; i != m_values.end() ; ++i)
  {
    if(*i == text)
    {
      m_values.erase(i);
      changed = true;
      break;
    }
  }

  if(append)
    m_values.push_back(text);
  else
    m_values.push_front(text);

  set_history_items();

  if(changed)
    save_history();
}

void ComboHistory::load_history()
{
  m_values = m_refClient->get_string_list(conf_key_search_history);
  set_history_items();
}

void ComboHistory::clear_history()
{
  m_values.clear();
  set_history_items();

  save_history();
}

void ComboHistory::save_history()
{
  m_bSaving_history = true;

  m_refClient->set_string_list(conf_key_search_history, m_values);
}

void ComboHistory::set_history_items()
{
  clear();

  if(!m_values.empty())
  {
    for(std::deque<Glib::ustring>::iterator i = m_values.begin() ; i != m_values.end() ; ++i)
    {
      append_text(*i);
    }
  }
  else
    m_values.clear();

  set_active_text(m_values.front());
  m_bChanged = false;
}

bool ComboHistory::has_value(const Glib::ustring& string)
{
  return (std::find(m_values.begin(), m_values.end(), string) != m_values.end());
}

} // namespace Widgets

} // namespace Coaster
