/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "string-utils.h"

#include "ucompose.h"

#include <libgnomevfsmm/uri.h>
#include <libgnomevfsmm/utils.h>

namespace Coaster
{

namespace String
{

Glib::ustring create_full_uri(const Glib::ustring& path)
{
  if(!path.size())
    return Glib::ustring();

  if(is_uri(path))
    return path;

  if(path[0] == '/' || path[0] == '~')
  {
    Glib::ustring escaped = Gnome::Vfs::escape_path_string(path);
    Glib::ustring retval = Glib::ustring("file://") + escaped;
    return retval;
  }

  Glib::ustring curdir = Glib::get_current_dir();
  Glib::ustring escaped = Gnome::Vfs::escape_path_string(curdir);
  Glib::ustring curdir_withslash = String::ucompose("file://%1%2", escaped, G_DIR_SEPARATOR_S);

  escaped = Gnome::Vfs::escape_path_string(path);
  Glib::ustring retval = Gnome::Vfs::Uri::make_full_from_relative(curdir_withslash, escaped);
  return retval;
}

Glib::ustring create_full_path(const Glib::ustring& path)
{
  if(!path.size())
    return Glib::ustring();

  if(path[0] == '/' || path[0] == '~')
    return path;

  Glib::ustring curdir = Glib::get_current_dir();
  Glib::ustring curdir_withslash = String::ucompose("file://%1%2", curdir, G_DIR_SEPARATOR_S, path);
  return curdir_withslash;
}

bool is_uri(const Glib::ustring& path)
{
  return (path.find("file://") != Glib::ustring::npos);
}

Glib::ustring replace_home_dir_with_tilde(const Glib::ustring& uri)
{
  std::string tmp = Glib::get_home_dir();

  if(tmp.empty())
    return uri;

  Glib::ustring home = Glib::filename_to_utf8(tmp);

  if(home.empty())
    return uri;

  if(uri == home)
    return "~";

  Glib::ustring tmp_home = home;
  home = tmp_home + "/";

  if(Glib::str_has_prefix(uri, home))
    return Glib::ustring("~/" + uri.substr(home.size()));

  return uri;
}

Glib::ustring get_dirname(const Glib::ustring& uri)
{
  if(uri.empty())
    return "";

  using namespace Gnome::Vfs;
  //std::string str = Glib::path_get_dirname(uri);
  Glib::RefPtr<Uri> vfs_uri = Uri::create(uri);
  Glib::ustring str = vfs_uri->extract_dirname();

  if(str.empty())
    return ".";

  if((str.size() == 1) && (str[0] == '.'))
    return "";

  return replace_home_dir_with_tilde(str);
}

Glib::ustring middle_truncate(const Glib::ustring& string,
                                  unsigned int truncate_length)
{
  if(string.empty())
    return "";

  Glib::ustring delimiter = "…";
  unsigned int delimiter_len = delimiter.size();
  unsigned int min_trunc_len = delimiter_len + 2;

  if(truncate_length < min_trunc_len)
    return string;

  unsigned int length = string.size();

  if(length <= truncate_length)
    return string;

  unsigned int num_left_chars = (truncate_length - delimiter_len) / 2;
  unsigned int num_right_chars = truncate_length - num_left_chars - delimiter_len + 1;

  Glib::ustring truncated;
  truncated += string.substr(0, num_left_chars);
  truncated += delimiter;
  truncated += string.substr(length - num_right_chars + 1);

  return truncated;
}

Glib::ustring uri_get_basename(const Glib::ustring& uri)
{
  Glib::RefPtr<Gnome::Vfs::Uri> vfs_uri = Gnome::Vfs::Uri::create(uri);
  if(!vfs_uri)
    return "";

  return vfs_uri->extract_short_name();
}

type_list_ustrings uri_list_extract_uris(const Glib::ustring& uri_list)
{
  /* Note that this is mostly very stolen from nautilus's libnautilus-private/nautilus-icon-dnd.c */

	const gchar *p, *q;

  type_list_ustrings result;

	//g_return_val_if_fail (uri_list != NULL, NULL);

	p = uri_list.c_str();

	/* We don't actually try to validate the URI according to RFC
	 * 2396, or even check for allowed characters - we just ignore
	 * comments and trim whitespace off the ends.  We also
	 * allow LF delimination as well as the specified CRLF.
	 */
	while (p != NULL) {
		if (*p != '#') {
			while (g_ascii_isspace (*p))
				p++;

			q = p;
			while ((*q != '\0')
			       && (*q != '\n')
			       && (*q != '\r'))
				q++;

			if (q > p) {
				q--;
				while (q > p
				       && g_ascii_isspace (*q))
					q--;
          result.push_back(Glib::ustring(p, q+1));
			}
		}
		p = strchr (p, '\n');
		if (p != NULL)
			p++;
	}

	return result;
}

} // namespace String

} // namespace Coaster
