/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "stock.h"

#include <gtkmm/iconfactory.h>
#include <gtkmm/iconset.h>
#include <gtkmm/iconsource.h>
#include <gtkmm/stock.h>
#include <gtkmm/stockitem.h>
#include <gtkmm/window.h>

namespace
{

/* Include inlined raw pixbuf data generated by gdk-pixbuf-csource.
 */
#include <pixmaps/stockimages.h>


struct StockIconData
{
  const guint8*         data;
  unsigned int          length;
  Gtk::BuiltinIconSize  size;
};

struct StockItemData
{
  const char*           id;
  const StockIconData*  icons;
  unsigned int          n_icons;
  const char*           label;
};


const StockIconData gnome_audio[] =
{
  { gnome_audio48, sizeof(gnome_audio48), Gtk::ICON_SIZE_MENU }
};

const StockIconData gnome_regular[] =
{
  { gnome_regular48, sizeof(gnome_regular48), Gtk::ICON_SIZE_MENU }
};

const StockIconData stock_icon_add_files[] =
{
  { stock_add_files_24, sizeof(stock_add_files_24), Gtk::ICON_SIZE_MENU }
};

const StockIconData stock_icon_add_folder[] =
{
  { stock_add_folder_24, sizeof(stock_add_folder_24), Gtk::ICON_SIZE_MENU }
};

const StockIconData stock_icon_cd_burn[] =
{
	{ stock_cd_burn_24, sizeof(stock_cd_burn_24), Gtk::ICON_SIZE_MENU }
};

const StockIconData stock_icon_cd_cdrom[] = 
{
  { stock_cd_cdrom_24, sizeof(stock_cd_cdrom_24), Gtk::ICON_SIZE_MENU }
};

const StockIconData stock_icon_cd_copy[] =
{
	{ stock_cd_copy_24, sizeof(stock_cd_copy_24), Gtk::ICON_SIZE_MENU }
};

const StockIconData stock_icon_cd_erase[] =
{
  { stock_cd_erase_24, sizeof(stock_cd_erase_24), Gtk::ICON_SIZE_MENU }
};

const StockIconData stock_icon_cd_idle[] =
{
  { stock_cd_idle_24, sizeof(stock_cd_idle_24), Gtk::ICON_SIZE_MENU }
};

const StockIconData stock_icon_cd_info[] =
{
  { stock_cd_info_24, sizeof(stock_cd_info_24), Gtk::ICON_SIZE_MENU }
};

const StockIconData stock_icon_cd_mix[] =
{
  { stock_cd_mix_24, sizeof(stock_cd_mix_24), Gtk::ICON_SIZE_MENU }
};

const StockIconData stock_icon_cd_video[] =
{
  { stock_cd_video_24, sizeof(stock_cd_video_24), Gtk::ICON_SIZE_MENU }
};

const StockIconData stock_icon_session[] =
{
  { stock_session_24, sizeof(stock_session_24), Gtk::ICON_SIZE_MENU }
};

const StockIconData stock_icon_session_add[] =
{
  { stock_session_add_24, sizeof(stock_session_add_24), Gtk::ICON_SIZE_MENU }
};

const StockIconData stock_icon_session_remove[] =
{
  { stock_session_remove_24, sizeof(stock_session_remove_24), Gtk::ICON_SIZE_MENU }
};

const StockItemData coaster_stock_items[] =
{
  { "coaster-audio", gnome_audio, G_N_ELEMENTS(gnome_audio), "_Audio File" },
  { "coaster-data", gnome_regular, G_N_ELEMENTS(gnome_regular), "_Data File" },
  { "coaster-mix", stock_icon_cd_mix, G_N_ELEMENTS(stock_icon_cd_mix), "New _Multisession CD" },
  { "coaster-video", stock_icon_cd_video, G_N_ELEMENTS(stock_icon_cd_video), "New _Video CD" },

  { "coaster-add-files", stock_icon_add_files, G_N_ELEMENTS(stock_icon_add_files), "Add _Files..." },
  { "coaster-add-folder", stock_icon_add_folder, G_N_ELEMENTS(stock_icon_add_folder), "Add F_older..." },

  { "coaster-cdrom", stock_icon_cd_cdrom, G_N_ELEMENTS(stock_icon_cd_cdrom), "_CDROM" },
  { "coaster-burn", stock_icon_cd_burn, G_N_ELEMENTS(stock_icon_cd_burn), "_Burn CD" },
  { "coaster-copy", stock_icon_cd_copy, G_N_ELEMENTS(stock_icon_cd_copy), "_Copy a CD" },
  { "coaster-erase", stock_icon_cd_erase, G_N_ELEMENTS(stock_icon_cd_erase), "_Erase a CD" },
  { "coaster-idle", stock_icon_cd_idle, G_N_ELEMENTS(stock_icon_cd_idle), "CD Idle" },
  { "coaster-info", stock_icon_cd_info, G_N_ELEMENTS(stock_icon_cd_info), "CD _Info" },

  { "coaster-session", stock_icon_session, G_N_ELEMENTS(stock_icon_session), "_Session" },
  { "coaster-session-add", stock_icon_session_add, G_N_ELEMENTS(stock_icon_session_add), "Add _Session" },
  { "coaster-session-remove", stock_icon_session_remove, G_N_ELEMENTS(stock_icon_session_remove), "R_emove Session" }
};
  
} // anonymous namespace

namespace Coaster
{

namespace Stock
{

namespace
{

void register_items()
{
  const Glib::RefPtr<Gtk::IconFactory> factory = Gtk::IconFactory::create();

  for(unsigned item = 0; item < G_N_ELEMENTS(coaster_stock_items); ++item)
  {
    const StockItemData& stock = coaster_stock_items[item];
    Gtk::IconSet icon_set;

    for(unsigned icon = 0; icon < stock.n_icons; ++icon)
    {
      const StockIconData& icon_data = stock.icons[icon];
      Gtk::IconSource source;

      source.set_pixbuf(Gdk::Pixbuf::create_from_inline(icon_data.length, icon_data.data));
      source.set_size(icon_data.size);

      // Unset wildcarded for all but the the last icon.
      source.set_size_wildcarded(icon == stock.n_icons - 1);

      icon_set.add_source(source);
    }

    const Gtk::StockID stock_id (stock.id);

    factory->add(stock_id, icon_set);
    Gtk::Stock::add(Gtk::StockItem(stock_id, stock.label));
  }

  factory->add_default();
}

void set_window_icon()
{
  try
  {
    std::list< Glib::RefPtr<Gdk::Pixbuf> > icons;
    icons.push_back(Gdk::Pixbuf::create_from_file(COASTER_ICONDIR G_DIR_SEPARATOR_S "coaster.png"));
    Gtk::Window::set_default_icon_list(icons);
  }
  catch(const Glib::Error& error)
  {
    const Glib::ustring what = error.what();
    g_warning("%s", what.c_str());
  }
}

} // anonymous namespace

void init()
{
  register_items();
  set_window_icon();
}

const Gtk::BuiltinStockID AUDIO = { "coaster-audio" };
const Gtk::BuiltinStockID DATA = { "coaster-data" };
const Gtk::BuiltinStockID MIX = { "coaster-mix" };
const Gtk::BuiltinStockID VIDEO = { "coaster-video" };

const Gtk::BuiltinStockID ADD_FILES = { "coaster-add-files" };
const Gtk::BuiltinStockID ADD_FOLDER = { "coaster-add-folder" };

const Gtk::BuiltinStockID DISC = { "coaster-cdrom" };
const Gtk::BuiltinStockID DISC_BURN = { "coaster-burn" };
const Gtk::BuiltinStockID DISC_COPY = { "coaster-copy" };
const Gtk::BuiltinStockID DISC_ERASE = { "coaster-erase" };
const Gtk::BuiltinStockID DISC_IDLE = { "coaster-idle" };
const Gtk::BuiltinStockID DISC_INFO = { "coaster-info" };

const Gtk::BuiltinStockID SESSION = { "coaster-session" };
const Gtk::BuiltinStockID ADD_SESSION = { "coaster-session-add" };
const Gtk::BuiltinStockID REMOVE_SESSION = { "coaster-session-remove" };

} // namespace Stock

} // namespace Coaster
