/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "mainwindow.h"

#include "cstr-debug.h"
#include "cstr-intl.h"
#include "stock.h"
#include "exception.h"

#include "instance-manager.h"
#include "status-icon.h"
#include "sound.h"
#include "ucompose.h"
#include "string-utils.h"

#include "widgets/view-base.h"
#include "data/data-view.h"

#include "conf/conf-utils.h"

#include "dialogs/glade-dialog.h"
#include "dialogs/burn.h"
#include "dialogs/dialog-util.h"

#include "egg/egg-recent-view-gtk.h"
#include "egg/egg-recent-view.h"
#include "egg/egg-recent-model.h"

#include "backends/cd-io.h"

#include <gtkmm/alignment.h>
#include <gtkmm/box.h>
#include <gtkmm/handlebox.h>
#include <gtkmm/menu.h>
#include <gtkmm/menubar.h>
#include <gtkmm/statusbar.h>
#include <gtkmm/stock.h>
#include <gtkmm/toolbar.h>
#include <gtkmm/uimanager.h>
#include <gtkmm/main.h>

#include <gconfmm/client.h>

#include <libgnome/gnome-help.h>

#include <libxml++/document.h>
#include <libxml++/parsers/domparser.h>

#include <libgnomevfsmm/handle.h>
#include <libgnomevfsmm/utils.h>

#include <iostream>
#include <unistd.h>
#include <sys/types.h>

#define COASTER_UI COASTER_UIDIR G_DIR_SEPARATOR_S "coaster.ui"

namespace // Anonymous Namespace for this file only (kinda like C's static functions)
{

//////////////////////////////////////////////
// Function: can_read()                     //
// Purpose: Check if a file's permissions   //
//          and owner id match this         //
//          program's user id.              //
// Input: An integer for the UID and an     //
//        enumeration for the permissions.  //
// Output: A bool representing whether the  //
//         file is readable (true) or not.  //
//////////////////////////////////////////////
bool can_read (unsigned int uid,
               Gnome::Vfs::FilePermissions perms)
{
  bool canRead = false;
  uid_t tempUid = getuid();

  if ((uid == tempUid) && (perms & Gnome::Vfs::PERM_USER_READ))
    canRead = true;
  else if (perms & Gnome::Vfs::PERM_OTHER_READ)
    canRead = true;

  return canRead;
}

Coaster::LayoutType get_layout_type_from_uri(const Glib::ustring& uri)
{
  using namespace Coaster;
  Glib::ustring mime_type = Glib::convert_return_gchar_ptr_to_ustring(gnome_vfs_get_mime_type(uri.c_str()));

  if(mime_type == "application/x-blf")
  {
    debug("Document type: application/x-blf");
    return COASTER_LAYOUT_DATA;
  }
  else if(mime_type == "application/x-bla")
  {
    debug("Document type: application/x-bla");
    return COASTER_LAYOUT_AUDIO;
  }
  /*else if(mime_type == "application/x-blv")
    return new VideoDoc();*/
  else
    return COASTER_LAYOUT_UNKNOWN;
}

Glib::ustring get_file_uri_with_extension(const Glib::ustring& uri,
                                          const Coaster::LayoutType& layout_type)
{
  using namespace Coaster;
  Glib::ustring result = uri;
 
  Glib::ustring file_extension = "";
  if(layout_type == COASTER_LAYOUT_DATA)
    file_extension = ".blf";
  else if(layout_type == COASTER_LAYOUT_AUDIO)
    file_extension = ".bla";
  else if(layout_type == COASTER_LAYOUT_VIDEO)
    file_extension = ".blv";
  else
    file_extension = ".blf";

  bool bAddExt = false;
  
  if(result.size() < file_extension.size()) //It can't have the ext already if it's not long enough.
  {
    bAddExt = true; //It isn't there already.
  }
  else
  {
    Glib::ustring strEnd = result.substr(result.size() - file_extension.size());
    if(strEnd != file_extension) //If it doesn't already have the extension
      bAddExt = true;
  }

    //Add extension if necessay.
  if(bAddExt)
    result += file_extension;

  //Note that this does not replace existing extensions, so it could be e.g. 'something.blah.theext'

  return result;
}

void c_callback_recent_files_activate(EggRecentViewGtk* view, EggRecentItem* item, void* user_data)
{
  if(user_data)
  {
    Coaster::MainWindow::type_slot_recent_item* slot =
      static_cast<Coaster::MainWindow::type_slot_recent_item*>(user_data);
    slot->operator()(item);    
  } 
}



const char *const coaster_icon_filename =
  COASTER_ICONDIR G_DIR_SEPARATOR_S "coaster.png";

} // anonymous namespace


namespace Coaster
{

//////////////////////////////////////////////////////////////////////////////////////////
//  Coaster::MainWindow                                                                 //
//////////////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////
// Function: MainWindow() (constructor)     //
// Purpose: Initialize our window.          //
// Input: none                              //
// Output: none                             //
//////////////////////////////////////////////
MainWindow::MainWindow()
: Gtk::Window(),
  m_instmgr(InstanceMgr::instance()),
  m_pMenuBar(0), m_pToolbar(0), m_pView(0), m_pStatus(0)
{
  set_default_size(500, 420);
  m_refClient = Gnome::Conf::Client::get_default_client();
}

//////////////////////////////////////////////
// Function: ~MainWindow() (destructor)     //
// Purpose: Delete anything we need to.     //
// Input: none                              //
// Output: none                             //
//////////////////////////////////////////////
MainWindow::~MainWindow()
{
  m_refClient->notify_remove(m_toolbar_detachable_cnxn);

  if(m_pView)
  {
    delete m_pView;
    m_pView = 0;
  }

  if(m_pHandleBox)
  {
    delete m_pHandleBox;
    m_pHandleBox = 0;
  }

  if(m_pToolbar)
  {
    delete m_pToolbar;
    m_pToolbar = 0;
  }
}

void MainWindow::init(const Glib::ustring& file_uri)
{
  init_ui();
  
  if(file_uri != "")
  {
    LayoutType lt = get_layout_type_from_uri(file_uri);
    if(lt != COASTER_LAYOUT_UNKNOWN)
    {
      m_file_uri = file_uri;

      if(lt == COASTER_LAYOUT_DATA)
      {
        m_pView = new Data::View();
      }
    }
    else
    {
      // warn user

      // show blank document
      m_pView = new Data::View();
    }
  }
  else
    m_pView = new Data::View();

  //set_contents(*m_pView);
  m_pView_VBox->pack_start(*m_pView, true, true);
  update_window_title();

  m_refUIManager->insert_action_group(m_pView->get_ui_action_group());
  m_last_view_merge_ui = add_ui_from_file(m_pView->get_ui_location());
  m_pView->set_popup_menu(
            reinterpret_cast<Gtk::Menu*>(m_refUIManager->get_widget("/CoasterPopup")));
  m_pView->signal_modified().connect(sigc::mem_fun(*this, &MainWindow::on_view_modified));
  m_pView->signal_sensitive().connect(sigc::mem_fun(*this, &MainWindow::on_view_sensitive));

  show_all();
}

bool MainWindow::is_doc_new() const
{
  return (m_file_uri.empty());
}

Glib::ustring MainWindow::get_file_uri() const
{
  return m_file_uri;
}

void MainWindow::set_file_uri(const Glib::ustring& file_uri)
{
  m_file_uri = file_uri;
  update_window_title();
}

int MainWindow::add_ui_from_file(const Glib::ustring& ui_filename)
{
  int result;
  try
  {
    result = m_refUIManager->add_ui_from_file(ui_filename);
    return result;
  }
  catch(const Glib::Error& ex)
  {
    std::cerr << "building menus failed: " <<  ex.what();
  }

  return 0;
}

//////////////////////////////////////////////
// Function: init_ui_manager()              //
// Purpose: //
// Input: none                              //
// Output: none                             //
//////////////////////////////////////////////
void MainWindow::init_ui()
{
  m_refUIManager = Gtk::UIManager::create();

  // connect signals for menu tooltips
  m_refUIManager->signal_connect_proxy().connect(sigc::mem_fun(*this, &MainWindow::on_ui_connect));

  // initialize actions
  init_file_actions();
  init_edit_actions();
  init_other_actions();
  init_help_actions();

  add_ui_from_file(COASTER_UI);

  init_layout();
  init_recent_files();
}

void MainWindow::init_layout()
{
  m_pMain_VBox = new Gtk::VBox();
  add(*manage(m_pMain_VBox));

  m_pMenuBar = reinterpret_cast<Gtk::MenuBar*>(m_refUIManager->get_widget("/Coaster_MainMenu"));
  m_pMain_VBox->pack_start(*manage(m_pMenuBar), false, false);

  m_pAlignment = new Gtk::Alignment();
  m_pMain_VBox->pack_start(*manage(m_pAlignment), false, false);

  m_pHandleBox = new Gtk::HandleBox();
  m_pToolbar = reinterpret_cast<Gtk::Toolbar*>(m_refUIManager->get_widget("/Coaster_ToolBar"));

  show_handlebar(Conf::get_bool_full("/desktop/gnome/interface/toolbar_detachable"));
  m_toolbar_detachable_cnxn =
    m_refClient->notify_add("/desktop/gnome/interface/toolbar_detachable",
                            sigc::mem_fun(*this, &MainWindow::on_toolbar_detachable_changed));

  m_pView_VBox = new Gtk::VBox();
  m_pMain_VBox->pack_start(*manage(m_pView_VBox), true, true);

  m_pStatus = new Gtk::Statusbar();
  m_pMain_VBox->pack_start(*manage(m_pStatus), false, false);

  add_accel_group(m_refUIManager->get_accel_group());
}

void MainWindow::init_recent_files()
{
  // Add recent-files submenu:
  Gtk::MenuItem* pMenuItem = dynamic_cast<Gtk::MenuItem*>(m_refUIManager->get_widget("/Coaster_MainMenu/Coaster_Menu_File/Coaster_Menu_File_RecentFiles"));

  Gtk::Menu* pSubMenu_RecentFiles = pMenuItem->get_submenu();
  
  EggRecentViewGtk* view = egg_recent_view_gtk_new(GTK_WIDGET(pSubMenu_RecentFiles->gobj()), NULL);
  egg_recent_view_set_model(EGG_RECENT_VIEW(view), m_instmgr->get_recent_model());
  egg_recent_view_gtk_show_numbers(view, FALSE);
  egg_recent_view_gtk_show_icons(view, TRUE);

  m_slot_recent_files_activate = sigc::mem_fun(*this, &MainWindow::on_recent_files_activate);

  //Connect the signal handler:
  g_signal_connect(G_OBJECT(view),
                   "activate",
                   G_CALLBACK (c_callback_recent_files_activate),
                   &m_slot_recent_files_activate);
}

void MainWindow::init_file_actions()
{
  using namespace Gtk;

  // File menu

  //Build actions:
  m_refFileActionGroup = ActionGroup::create("CoasterFileActions");

  m_refFileActionGroup->add(Action::create("Coaster_Menu_File", _("_File")));
  m_refFileActionGroup->add(Action::create("Coaster_Menu_File_RecentFiles",
                                           _("_Recent Files")));

  //File actions
  Glib::RefPtr<Action> actionNew = Action::create("CoasterAction_File_New", Gtk::Stock::NEW,
                                                  _("_New"), _("Create a new layout"));
  actionNew->property_is_important() = true;
  m_refFileActionGroup->add(actionNew,
                            sigc::mem_fun(*this, &MainWindow::on_menu_file_new));
  m_refFileActionGroup->add(Action::create("CoasterAction_File_Open", Gtk::Stock::OPEN,
                                           _("_Open..."), _("Open a layout")),
                            sigc::mem_fun(*this, &MainWindow::on_menu_file_open));

  //Remember thes ones for later, so we can disable Save menu and toolbar items:
  m_refSaveAction = Action::create("CoasterAction_File_Save", Gtk::Stock::SAVE,
                                   _("_Save"), _("Save the current layout"));
  m_refFileActionGroup->add(m_refSaveAction,
                            sigc::mem_fun(*this, &MainWindow::on_menu_file_save));

  m_refSaveAsAction = Action::create("CoasterAction_File_SaveAs", Gtk::Stock::SAVE_AS,
                                     _("Save _As..."), 
                                     _("Save the current layout with a different name"));                   
  m_refFileActionGroup->add(m_refSaveAsAction,
                            sigc::mem_fun(*this, &MainWindow::on_menu_file_saveas));

  m_refRevertAction = Action::create("CoasterAction_File_Revert", Gtk::Stock::REVERT_TO_SAVED,
                                     _("Re_vert"),
                                     _("Revert the file to the last saved state"));
  m_refFileActionGroup->add(m_refRevertAction,
                            sigc::mem_fun(*this, &MainWindow::on_menu_file_revert));
                        
  m_refFileActionGroup->add(Action::create("CoasterAction_File_Close", Gtk::Stock::CLOSE,
                                           _("_Close"), _("Close the current layout")),
                            sigc::mem_fun(*this, &MainWindow::on_menu_file_close));
  m_refFileActionGroup->add(Action::create("CoasterAction_File_Exit", Gtk::Stock::QUIT,
                                           _("_Quit"), _("Quit the program")),
                            sigc::mem_fun(*this, &MainWindow::on_menu_file_exit));

  m_refSaveAction->set_sensitive(false);
  m_refSaveAsAction->set_sensitive(false);
  m_refRevertAction->set_sensitive(false);

  m_refUIManager->insert_action_group(m_refFileActionGroup);
}

void MainWindow::init_edit_actions()
{
  using namespace Gtk;

  m_refEditActionGroup = ActionGroup::create("CoasterEditActions");

  m_refEditActionGroup->add(Action::create("Coaster_Menu_Edit", _("_Edit")));

  m_refEditActionGroup->add(Action::create("CoasterAction_Preferences", Gtk::Stock::PREFERENCES,
                                           _("Pr_eferences"), _("Configure Coaster")),
                            sigc::mem_fun(*this, &MainWindow::on_preferences));

  m_refUIManager->insert_action_group(m_refEditActionGroup);
}

void MainWindow::init_other_actions()
{
  using namespace Gtk;

  m_refOtherActionGroup = ActionGroup::create("CoasterOtherActions");

  // Popup Menu
  m_refOtherActionGroup->add(Action::create("CoasterPopup"));

  //Extra Menus:
  m_refOtherActionGroup->add(Action::create("Coaster_Menu_Search", _("_Search")));
  m_refOtherActionGroup->add(Action::create("Coaster_Menu_Layout", _("_Layout")));
  m_refOtherActionGroup->add(Action::create("Coaster_Menu_Disc", _("_Disc")));
  
  //"Disc" actions:
  m_refOtherActionGroup->add(Action::create("CoasterAction_Disc_BurnISO",
                                            Stock::DISC_BURN,
                                            _("Burn _ISO..."), _("Burn an ISO image to disc")),
                             sigc::mem_fun(*this, &MainWindow::on_burn_iso));
  m_refOtherActionGroup->add(Action::create("CoasterAction_Disc_Copy", Stock::DISC_COPY,
                                            _("_Copy..."), _("Copy a disc to another")),
                             sigc::mem_fun(*this, &MainWindow::on_copy_disc));

  m_refEraseAction = Action::create("CoasterAction_Disc_Erase", Stock::DISC_ERASE,
                                    _("Erase..."), _("Erase a rewritable disc"));
  m_refOtherActionGroup->add(m_refEraseAction, sigc::mem_fun(*this, &MainWindow::on_erase_disc));

  m_refEjectAction = Action::create("CoasterAction_Disc_Eject", 
                                    _("Eject"), _("Eject the disc in CD/DVD drive"));
  m_refOtherActionGroup->add(m_refEjectAction);

  m_refEjectAction->set_sensitive(false);
  //m_refEraseAction->set_sensitive(false);

  m_refUIManager->insert_action_group(m_refOtherActionGroup);
}

void MainWindow::init_help_actions()
{
  using namespace Gtk;
  //Help menu

  //Build actions:
  m_refHelpActionGroup = ActionGroup::create("CoasterHelpActions");
  m_refHelpActionGroup->add(Action::create("Coaster_Menu_Help","_Help"));

  m_refHelpActionGroup->add(Action::create("CoasterAction_Help_Contents",
                                           Gtk::Stock::HELP),
                            AccelKey("F1"),
                            sigc::mem_fun(*this, &MainWindow::on_menu_help_contents));
  m_refHelpActionGroup->add(Action::create("CoasterAction_Help_About",
                                           Gtk::Stock::ABOUT),
                            sigc::mem_fun(*this, &MainWindow::on_menu_help_about));

  m_refUIManager->insert_action_group(m_refHelpActionGroup);
}

void MainWindow::open(const Glib::ustring& file_uri)
{
  if(!file_uri.empty())
  {
    OpenReturnValues orv = m_instmgr->open(file_uri,
                                           (m_pView->get_modified() || !is_doc_new()),
                                           this);
    update_window_title();
    if(orv == OPEN_FAILED)
    {
      // warn user the file didn't open
      return;
    }
    
    if(orv == OPEN_ALREADY)
    {
      // tell the user it was already open
      return;
    }
  }
}

bool MainWindow::save()
{
  if(m_pView->get_modified())
  {
    Glib::ustring strContents;

    try
    {
      strContents = m_pView->get_contents();
    }
    catch(xmlpp::exception& ex)
    {
      handle_exception("MainWindow::save(): exception caught: ", ex);
      return false;
    }

    /* we use create instead of open, because open will not create the file if it does
    not already exist. The last argument is the permissions to use if the file is created,
    the second to last tells GnomeVFS that its ok if the file already exists, and just open it */

    Gnome::Vfs::Handle write_handle;

    try
    {
      //0660 means "this user and his group can read and write this non-executable file".
      //The 0 prefix means that this is octal.
      write_handle.create(m_file_uri,
                          Gnome::Vfs::OPEN_WRITE,
                          false,
                          0660 /* leading zero means octal */);
    }
    catch(const Gnome::Vfs::exception& ex)
    {
      // If the operation was not successful, print the error and abort
      // Tell the user what happened!
      return false;
    }

    try
    {
      //Write the data to the output uri
      write_handle.write(strContents.data(), strContents.bytes());
    }
    catch(const Gnome::Vfs::exception& ex)
    {
      // If the operation was not successful, print the error and abort
      // Tell the user what happened!
      return false;
    }

    m_pView->set_modified(false);

    return true;
  }

  return true;
}

bool MainWindow::saveas()
{
  int response;
  Glib::ustring new_uri = Dialogs::offer_saveas(*this,
                                                m_file_uri,
                                                response);

  if(response == Gtk::RESPONSE_CANCEL)
  {
    return false;
  }
  else
  {
    new_uri = get_file_uri_with_extension(new_uri, m_pView->get_layout_type());
    debug("New URI: ", new_uri);

    bool bUseThisFileUri = true;

    //Check whether the file exists already:
    {
      Glib::RefPtr<Gnome::Vfs::Uri> vfs_uri = Gnome::Vfs::Uri::create(new_uri);
      if(vfs_uri->uri_exists())
      {
        bool bOverwrite = Dialogs::ask_overwrite(*this,
                                                 Gnome::Vfs::unescape_string(new_uri));

        bUseThisFileUri = bOverwrite;
      }
    }

    if(bUseThisFileUri)
    {
      set_file_uri(new_uri);
      bool bResult = save();

      if(!bResult)
      {
        // warn user it didn't save right
      }
      else
      {
        m_pView->set_modified(false);
        m_instmgr->document_history_add(m_file_uri);
      }

      update_window_title();
      return true;
    }
    else
    {
      saveas(); // recursive
    }

    return true;
  }
}

bool MainWindow::load()
{
  Glib::ustring strContents;
  Gnome::Vfs::Handle read_handle;

  if(is_doc_new())
    return false;

  try
  {
    read_handle.open(m_file_uri, Gnome::Vfs::OPEN_READ);
  }
  catch(const Gnome::Vfs::exception& ex)
  {
    // If the operation was not successful, print the error and abort
    return false; //print_error(ex, input_uri_string);
  }

  // Read data from the input uri:
  guint buffer[256] = {0, }; // For each chunk.
  Gnome::Vfs::FileSize bytes_read = 0;
  std::string data; //We use a std::string because we might not get whole UTF8 characters at a time. This might not be necessary.

  try
  {
    bool bContinue = true;
    while(bContinue)
    {
      bytes_read = read_handle.read(buffer, 256);

      if(bytes_read == 0)
        bContinue = false; //stop because we reached the end.
      else
      {
        // Add the text to the string:
        data += std::string((char*)buffer, bytes_read);
      }
    }
  }
  catch(const Gnome::Vfs::exception& ex)
  {
    // If the operation was not successful, print the error and abort
    return false; //print_error(ex, input_uri_string);
  }

  strContents = data;

  bool bResult = m_pView->load_contents(strContents);
  m_pView->set_modified(false);

  return bResult;
}

void MainWindow::close(bool warn)
{
  if(warn)
  {
    if(m_pView && m_pView->get_modified())
    {
      raise(); // to tell which instance we're showing this dialog for

      int response;

      Dialogs::offer_save_changes(*this,
                                  Gnome::Vfs::unescape_string(get_doc_short_name()),
                                  response);

      if(response == Gtk::RESPONSE_CANCEL)
      {
        m_instmgr->cancel_close_all();
        return; // don't change anything and keep open
      }
      if(response == Gtk::RESPONSE_ACCEPT)
      {
        if(is_doc_new())
        {
          if(!saveas())
            return;
        }

        save();
      }
    }
  }

  hide();
}

void MainWindow::show_handlebar(bool show)
{
  if(m_pAlignment->get_child())
    m_pAlignment->remove();
  
  if(m_pHandleBox->get_child())
    m_pHandleBox->remove();

  if(show)
  {
    m_pAlignment->add(*m_pHandleBox);
    m_pHandleBox->add(*m_pToolbar);
  }
  else
  {
    m_pAlignment->add(*m_pToolbar);
  }

  m_pAlignment->show_all();
}

void MainWindow::on_toolbar_detachable_changed(guint connection,
                                               Gnome::Conf::Entry entry)
{
  show_handlebar(Conf::get_bool_full("/desktop/gnome/interface/toolbar_detachable"));
}

bool MainWindow::on_delete_event(GdkEventAny* event)
{
  close(true);

  return true; // true = don't hide, don't destroy
}

void MainWindow::on_menu_file_new()
{
  m_instmgr->new_window();
}

void MainWindow::on_menu_file_open()
{
  // Bring window to front
  raise();

  // Ask user to choose file to open:
  Glib::ustring file_uri = Dialogs::offer_open_files(*this);
  if(!file_uri.empty())
    open(file_uri);
}

void MainWindow::on_recent_files_activate(EggRecentItem* item)
{
  unsigned int prefix_len = strlen ("file://");
  char* uri = egg_recent_item_get_uri(item);

  if (strlen(uri) > prefix_len)
  {
    Glib::ustring file_uri = Gnome::Vfs::unescape_string(uri + prefix_len);
  
    open(file_uri);
  }

  g_free (uri);
}

void MainWindow::on_menu_file_save()
{
  debug("MainWindow::on_menu_file_save()");
  if(is_doc_new())
  {
    saveas();
    // offer saveas
  }
  
  save();
}

void MainWindow::on_menu_file_saveas()
{
  raise();

  Glib::ustring new_file_uri;
  int response;
  new_file_uri = Dialogs::offer_saveas(*this,
                                       m_file_uri,
                                       response);

  if(response != Gtk::RESPONSE_CANCEL)
  {
    // Enforce the file extension:
    new_file_uri = get_file_uri_with_extension(new_file_uri, m_pView->get_layout_type());
    debug("New URI: ", new_file_uri);

    bool bUseThisFileUri = true;

    //Check whether the file exists already:
    {
      Glib::RefPtr<Gnome::Vfs::Uri> vfs_uri = Gnome::Vfs::Uri::create(new_file_uri);
      if(vfs_uri->uri_exists())
      {
        bool bOverwrite = Dialogs::ask_overwrite(*this,
                                                 Gnome::Vfs::unescape_string(new_file_uri));

        bUseThisFileUri = bOverwrite;
      }
    }

    if(bUseThisFileUri)
    {
      m_file_uri = new_file_uri;
      bool bResult = save();

      if(!bResult)
      {
        // warn user it didn't save right
      }
      else
      {
        m_pView->set_modified(false);
        m_instmgr->document_history_add(m_file_uri);
      }

      update_window_title();
    }
    else
    {
      on_menu_file_saveas(); // recursive
    }
  }
  else
  {
    m_instmgr->cancel_close_all();
  }
}

void MainWindow::on_menu_file_revert()
{
  if(Dialogs::ask_revert(*this, get_doc_short_name()))
  {
    m_pView->prepare_for_revert();
    load();
  }
}

void MainWindow::on_menu_file_close()
{
  close(true);
}

void MainWindow::on_menu_file_exit()
{
  // we don't want to quit directly as we should save our work
  // therefore we need to send close to each window.

  //Close each instance:
  m_instmgr->close_all();
}

void MainWindow::on_preferences()
{
  debug("MainWindow::on_preferences()");
  Dialogs::preferences(*this);
}

void MainWindow::on_burn_iso()
{
  debug("MainWindow::on_burn_iso()");

  // render these pixbufs before setting the interface insensitive
  // so we get a normal lookign icon
  Glib::RefPtr<Gdk::Pixbuf> pb_idle, pb_burn;
  pb_idle = render_icon(Stock::DISC_IDLE, Gtk::ICON_SIZE_MENU);
  pb_burn = render_icon(Stock::DISC_BURN, Gtk::ICON_SIZE_MENU);

  int response_file_chooser;
  int response_burn_dialog;
  std::string iso_filename;

  m_instmgr->set_sensitive(false);

  Dialogs::offer_open_iso(*this, response_file_chooser, iso_filename);

  if(response_file_chooser == Gtk::RESPONSE_OK)
  {
    StatusIcon::set(pb_idle, _("Idle"));

    std::auto_ptr<Dialogs::Burn> burndialog = 
      Dialogs::GladeDialog<Dialogs::Burn>::create(*this,
                                                  "diag_burn");

    response_burn_dialog = burndialog->run();
    
    if(response_burn_dialog == Gtk::RESPONSE_OK)
    {
      burndialog->hide();

      StatusIcon::set(pb_burn, _("Burning..."));

      IO::BurnDisc burn(*this,
                        burndialog->get_drive(),
                        burndialog->get_eject(),
                        burndialog->get_speed(),
                        burndialog->get_dummy(),
                        iso_filename);

      if(burn.start_burn())
        Sound::burn_finished();
      else
        Sound::burn_error();
    }

    StatusIcon::remove();
  }


  m_instmgr->set_sensitive(true);
  return;
}

void MainWindow::on_copy_disc()
{}

void MainWindow::on_erase_disc()
{}

void MainWindow::on_menu_help_contents()
{
  m_instmgr->show_help(*this);
}

void MainWindow::on_menu_help_about()
{
  m_instmgr->show_about(*this);
}

void MainWindow::on_view_modified(bool modified)
{
  update_window_title();
  m_refSaveAction->set_sensitive(modified);
  m_refSaveAsAction->set_sensitive(modified);
  if(!is_doc_new())
    m_refRevertAction->set_sensitive(modified);
}

void MainWindow::on_view_sensitive(bool sensitive)
{
  m_instmgr->set_sensitive(sensitive);
}

void MainWindow::on_ui_connect(const Glib::RefPtr<Gtk::Action>& action,
                                Gtk::Widget* widget)
{
  if(dynamic_cast<Gtk::MenuItem*>(widget))
  {
    (static_cast<Gtk::MenuItem*>(widget))->signal_select().connect(sigc::bind(sigc::mem_fun(*this, &MainWindow::on_show_menu_tip), action));
    (static_cast<Gtk::MenuItem*>(widget))->signal_deselect().connect(sigc::mem_fun(*this, &MainWindow::on_clear_menu_tip));
  }
}

void MainWindow::on_show_menu_tip(const Glib::RefPtr<Gtk::Action>& action)
{
  Glib::ustring tip = action->property_tooltip();
  if (tip == "") tip = " ";
  m_pStatus->push(" " + tip);
}

void MainWindow::on_clear_menu_tip()
{
  m_pStatus->pop();
}

void MainWindow::update_window_title()
{
  Glib::ustring dirname = get_doc_dirname();
  Glib::ustring name = get_doc_short_name();

  Glib::ustring title;

  if(m_pView->get_modified())
    title += "*";

  title += name;

  if(!dirname.empty())
    title += " (" + dirname + ")";

  title += " - ";
  title += _("Coaster");

  set_title(title);
}

Glib::ustring MainWindow::get_doc_dirname(bool for_title /* = true */) const
{
  if(is_doc_new())
    return Glib::ustring();

  Glib::RefPtr<Gnome::Vfs::Uri> vfs_uri = Gnome::Vfs::Uri::create(m_file_uri);
  Glib::ustring str = vfs_uri->extract_dirname();

  if(str.empty())
    return ".";

  if((str.size() == 1) && (str[0] == '.'))
    return Glib::ustring();

  Glib::ustring dirname = String::replace_home_dir_with_tilde(str);

  if(!for_title)
    return dirname;

  if(!dirname.empty())
    dirname = String::middle_truncate(dirname, 40);
  else
    dirname = "";

  return dirname;
}

Glib::ustring MainWindow::get_doc_short_name() const
{
  if(is_doc_new())
    return _("Unsaved Layout");
  
  Glib::ustring basename = String::uri_get_basename(m_file_uri);

  if(!basename.empty())
    return basename;
  else
    return _("Invalid file name");
}

} // namespace Coaster
