/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "instance-manager.h"

#include "cstr-debug.h"
#include "cstr-intl.h"
#include "ucompose.h"

#include "mainwindow.h"
#include "dialogs/about.h"
#include "egg/egg-recent-model.h"

#include <libgnomevfsmm.h>

#include <gtkmm/main.h>
#include <gtkmm/messagedialog.h>

#include <libgnome/gnome-help.h>

namespace Coaster
{

InstanceMgr::InstanceMgr()
: m_recent_files_model(0), m_bExiting(false)
{
  m_recent_files_model = egg_recent_model_new(EGG_RECENT_MODEL_SORT_MRU);
  
  egg_recent_model_set_filter_mime_types(m_recent_files_model,
                                         "application/x-blf",
                                         "application/x-bla",
                                         "application/x-blv", 0);
  egg_recent_model_set_limit(m_recent_files_model, 10);
}

InstanceMgr::~InstanceMgr()
{}

SharedPtr<InstanceMgr> InstanceMgr::instance()
{
  static SharedPtr<InstanceMgr> instmgr(new InstanceMgr());
  return instmgr;
}

void InstanceMgr::init(const type_vec_strings& filenames)
{

  if(filenames.empty())
  {
    MainWindow* pMW = 0;
    pMW = new MainWindow();

    pMW->init();
    add(pMW);
  }
  else
  {
    for(type_vec_strings::const_iterator i = filenames.begin() ; i != filenames.end() ; ++i)
    {
      if(i->empty())
        continue;

      MainWindow* pMW = 0;
      pMW = new MainWindow();

      pMW->init(*i);
      add(pMW);

      bool bResult = pMW->load();
      if(!bResult)
      {
        pMW->set_file_uri("");
        pMW->load();
      }
    }
  }
}

void InstanceMgr::new_window()
{
  MainWindow* pMW = new MainWindow();
  pMW->init();
  add(pMW);
  pMW->show();
}

OpenReturnValues InstanceMgr::open(const Glib::ustring& file_uri,
                                   bool might_need_new_instance,
                                   MainWindow* caller)
{
  if(!file_uri.empty())
  {
    MainWindow* pAppAO = is_already_open(file_uri);

    if(pAppAO)
    {
      pAppAO->raise();
      return OPEN_ALREADY;
    }
  }

  MainWindow* pMW = 0;

  if(might_need_new_instance)
  {
    // if it gets to this point, we definately need a new instance
    pMW = new MainWindow();
    pMW->init(file_uri);
  }
  else
  {
    // if it gets to this point, we can use the calling instance
    pMW = caller;
    pMW->set_file_uri(file_uri);
  }

  Glib::ustring oldUri = pMW->get_file_uri();
  bool bTest = pMW->load();

  if(!bTest)
  {
    if(might_need_new_instance)
      pMW->close(false);

    if(!might_need_new_instance)
    {
      pMW->set_file_uri(oldUri);
      pMW->load();
    }
      

    document_history_remove(file_uri);

    return OPEN_FAILED;
  }
  else
  {
    if(might_need_new_instance)
    {
      add(pMW);
      pMW->show();
    }

    document_history_add(file_uri);

    return OPEN_SUCCESS;
  }
}

void InstanceMgr::add(MainWindow* pMW)
{
  m_list_instances.push_back(pMW);

  pMW->signal_hide().connect(sigc::bind(sigc::mem_fun(*this, &InstanceMgr::on_hide), pMW));
}

void InstanceMgr::close_all()
{
  m_bExiting = true;

  type_list_coaster_apps::iterator i = m_list_instances.begin();
  while(m_bExiting && (i != m_list_instances.end()))
  {
    type_list_coaster_apps::iterator j = i;
    i++;

    MainWindow* mw = (*j);
    if(mw)
    {
      type_list_coaster_apps::size_type count = m_list_instances.size();
      mw->close(true);

      //The iterator is invalid if an element has been removed:
      if(count != m_list_instances.size())
      {
        i = m_list_instances.begin(); //There should not be a problem with asking again.
      }
    }
  }
}

void InstanceMgr::cancel_close_all()
{
  m_bExiting = false;
}

unsigned int InstanceMgr::get_count() const
{
  return m_list_instances.size();
}

InstanceMgr::type_list_coaster_apps InstanceMgr::get_instances() const
{
  return m_list_instances;
}

void InstanceMgr::show_about(Gtk::Window& window)
{
  if(m_pAbout.get())
  {
    m_pAbout->set_transient_for(window);
    m_pAbout->present();
  }
  else
  {
    //Re-create About box:
    std::auto_ptr<Dialogs::About> dialog = Dialogs::About::create();

    dialog->set_transient_for(window);
    dialog->signal_hide().connect(sigc::mem_fun(*this, &InstanceMgr::on_about_dialog_hide));
    dialog->show();

    m_pAbout = dialog;
  }
}

void InstanceMgr::show_help(Gtk::Window& window,
                            const Glib::ustring& link_id)
{
  GError* error = 0;

  if(!link_id.empty())
    gnome_help_display("coaster", link_id.c_str(), &error);
  else
    gnome_help_display("coaster", 0, &error);

  if(error)
  {
    if(m_pHelpError.get())
    {
      m_pHelpError->set_transient_for(window);
      m_pHelpError->set_markup(String::ucompose("<b>%1</b>\n\n%2",
                                                _("Could not display help for Coaster"),
                                                g_strdup(error->message)));
      m_pHelpError->present();
    }
    else
    {
      //Re-create HelpError box:
      std::auto_ptr<Gtk::MessageDialog> dialog =
        std::auto_ptr<Gtk::MessageDialog>(new Gtk::MessageDialog(window, String::ucompose("<b><span size='larger'>%1</span></b>\n\n%2", _("Could not display help for Coaster"), g_strdup(error->message)), true, Gtk::MESSAGE_ERROR, Gtk::BUTTONS_CLOSE));

      dialog->set_default_response(Gtk::RESPONSE_CLOSE);
      dialog->set_transient_for(window);
      dialog->signal_hide().connect(sigc::mem_fun(*this, &InstanceMgr::on_help_error_dialog_hide));
      dialog->show();

      m_pHelpError = dialog;
    }

    g_error_free(error);
  }
}

void InstanceMgr::on_about_dialog_hide()
{
  // Play safe and transfer ownership, and let the dtor do the delete.
  const std::auto_ptr<Gtk::Dialog> temp(m_pAbout);
}

void InstanceMgr::on_help_error_dialog_hide()
{
  // Play safe and transfer ownership, and let the dtor do the delete.
  const std::auto_ptr<Gtk::Dialog> temp(m_pHelpError);
}

EggRecentModel* InstanceMgr::get_recent_model() const
{
  return m_recent_files_model;
}

void InstanceMgr::document_history_add(const Glib::ustring& uri)
{
  if(m_recent_files_model && !uri.empty())
  { 
    //TODO: Wrap gnome_vfs_escape_path_string() in gnome-vfsmm.
    //Glib::ustring filename_e = Gnome::Vfs::escape_path_string(file_uri);
    Glib::ustring new_uri = uri; // "file://" + filename_e;

    egg_recent_model_add(m_recent_files_model , new_uri.c_str());
  }
}

void InstanceMgr::document_history_remove(const Glib::ustring& uri)
{
  if(m_recent_files_model && !uri.empty())
  {
    //Glib::ustring filename_e = Gnome::Vfs::escape_path_string(file_uri.c_str());
    Glib::ustring new_uri = uri; //"file://" + filename_e;

    egg_recent_model_delete(m_recent_files_model, new_uri.c_str());
  }
}

InstanceMgr::type_signal_sensitive& InstanceMgr::signal_sensitive()
{
  return m_signal_sensitive;
}

void InstanceMgr::set_sensitive(bool sensitive)
{
  for(type_list_coaster_apps::iterator i = m_list_instances.begin() ;
      i != m_list_instances.end() ;
      ++i)
  {
    (*i)->set_sensitive(sensitive);
  }
}

void InstanceMgr::on_hide(MainWindow* pMW)
{
  //If pApp is one of the remembered instances (it always should be):
  type_list_coaster_apps::iterator i = 
    std::find(m_list_instances.begin(), m_list_instances.end(), pMW);

  if(i != m_list_instances.end())
  {
    m_list_instances.erase(i);
    delete pMW;
    pMW = 0;
  }

  //When the last instance goes, the application closes.
  if(m_list_instances.empty())
  {
    Gtk::Main::quit();
  }
}

MainWindow* InstanceMgr::is_already_open(const Glib::ustring& file_uri)
{
  if(m_list_instances.size() == 0)
    return 0;

  for(type_list_coaster_apps::iterator i = m_list_instances.begin();
      i != m_list_instances.end();
      ++i)
  {
    MainWindow* pMW = *i;
    if(pMW && pMW->get_file_uri() == file_uri)
    {
      return pMW;
    }
  }

  return 0;
}

} // namespace Coaster
