/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "icon-cache.h"

#include "cstr-debug.h"

#include <gdkmm/pixbuf.h>
#include <gtkmm/treestore.h>
#include <gtkmm/icontheme.h>
#include <libgnomeui/gnome-icon-lookup.h>

#ifdef COASTER_HAS_HASH_MAP
#include <ext/hash_map>

namespace __gnu_cxx
{

using std::size_t;
template<> struct hash<const Glib::ustring>
{
  size_t operator()(const Glib::ustring& __s) const { return __stl_hash_string(__s.c_str()); }
};

} // namespace __gnu_cxx

#else
#include <map>
#endif

namespace
{

#ifdef COASTER_HAS_HASH_MAP
struct equstr
{
  bool operator()(const Glib::ustring& a,
                  const Glib::ustring& b) const
  {
    return a.collate_key() == b.collate_key();
  }
};

typedef __gnu_cxx::hash_map<const Glib::ustring,
                            Glib::RefPtr<Gdk::Pixbuf>,
                            __gnu_cxx::hash<const Glib::ustring>,
                            equstr> type_str_pix_hash;
#else
typedef std::map<const Glib::ustring,
                 Glib::RefPtr<Gdk::Pixbuf> > type_str_pix_hash;
#endif

// static objects
Glib::RefPtr<Gtk::IconTheme> refIconTheme;
type_str_pix_hash hash_StrPixbuf;

// static functions
Glib::RefPtr<Gdk::Pixbuf> gtk_lookup_icon(const Glib::ustring& mime_type,
                                          bool check_map = true)
{
  type_str_pix_hash::iterator pos =  hash_StrPixbuf.find(mime_type);
 
  if(!check_map || hash_StrPixbuf.empty() || pos  == hash_StrPixbuf.end())
  {
    Gtk::IconInfo icon_info;
    int width, height;
 
    Gtk::IconSize::lookup(Gtk::ICON_SIZE_MENU, width, height);
    Glib::ustring tempIconName =
      Glib::convert_return_gchar_ptr_to_ustring(gnome_icon_lookup(refIconTheme->gobj(),
                                                                  NULL, NULL, NULL, NULL,
                                                                  mime_type.c_str(),
                                                                  GNOME_ICON_LOOKUP_FLAGS_NONE,
                                                                  NULL));
    Coaster::debug("Icon name: ", tempIconName);
    
    Glib::RefPtr<Gdk::Pixbuf> result = refIconTheme->load_icon(tempIconName,
                                                               width,
                                                               Gtk::ICON_LOOKUP_USE_BUILTIN);

    if(result)
      hash_StrPixbuf.insert(std::make_pair(mime_type, result));
    else
    {
      Glib::RefPtr<Gdk::Pixbuf> regular_result =
        refIconTheme->load_icon("gnome-fs-regular",
                                width,
                                Gtk::ICON_LOOKUP_USE_BUILTIN);
      hash_StrPixbuf.insert(std::make_pair(mime_type, regular_result));
    }

    Coaster::debug("Icon Cache: ", "inserted " + mime_type + " successfully");
    return result;
  }
  else
  {
    Coaster::debug("Icon Cache: ", mime_type, " found already");
    return pos->second;
  }
}

void on_default_icon_theme_changed()
{
  Coaster::debug("IconCache map reset");

  for(type_str_pix_hash::iterator i = hash_StrPixbuf.begin() ; i != hash_StrPixbuf.end() ; ++i)
  {
    i->second = gtk_lookup_icon(i->first, false);
  }
}

} // anonymous namespace

namespace Coaster
{

namespace IconCache
{

void init()
{
  if(!refIconTheme)
    refIconTheme = Gtk::IconTheme::get_default();

  refIconTheme->signal_changed().connect(sigc::ptr_fun(&on_default_icon_theme_changed));
}

Glib::SignalProxy0<void> signal_icon_theme_changed()
{
  return refIconTheme->signal_changed();
}

Glib::RefPtr<Gdk::Pixbuf> lookup_icon(const Glib::ustring& mime_type)
{
  return gtk_lookup_icon(mime_type);
}

} // namespace IconCache

} // namespace Coaster
