/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "dialogs/dialog-util.h"

#include "cstr-intl.h"
#include "ucompose.h"
#include "stock.h"
#include "string-utils.h"

#include "widgets/stock-label-button.h"

#include "dialogs/chooser.h"
#include "dialogs/glade-dialog.h"
#include "dialogs/client-dialog.h"
#include "dialogs/find.h"
#include "dialogs/preferences.h"

#include <gtkmm/messagedialog.h>
#include <gtkmm/label.h>
#include <gtkmm/stock.h>

#include <libglademm/xml.h>

#include <libgnomevfsmm/utils.h>
#include <libgnomevfsmm/uri.h>

namespace Coaster
{

namespace Dialogs
{

Glib::ustring m_last_dir_ = "";
bool m_recursive_ = true;
  
type_list_ustrings offer_add_files(Gtk::Window& parent,
                                   bool& hidden,
                                   int& response)
{
  static Glib::ustring last_uri = "";
  static bool was_hidden = false;

  std::auto_ptr<Chooser> chooser(new Chooser(parent,
                                             Dialogs::CHOOSER_TYPE_FILES));
  
  if(!last_uri.empty())
    chooser->set_current_folder_uri(last_uri);

  chooser->set_show_hidden(was_hidden);

  response = chooser->run();

  if(response != Gtk::RESPONSE_CANCEL)
  {
    last_uri = chooser->get_current_folder_uri();
    was_hidden = chooser->get_show_hidden();
  }

  hidden = chooser->get_show_hidden();

  return chooser->get_uris();
}

Glib::ustring offer_add_folder(Gtk::Window& parent,
                               bool& hidden,
                               bool& recursive,
                               bool& follow_links,
                               int& response)
{
  static Glib::ustring last_uri = "";
  static bool was_hidden = false;
  static bool was_recursive = true;
  static bool was_follow_links = false;

  std::auto_ptr<Chooser> chooser(new Chooser(parent,
                                             Dialogs::CHOOSER_TYPE_FOLDER));

  Gtk::CheckButton* pRecursive = new Gtk::CheckButton(_("R_ecursive"), true);
  Gtk::CheckButton* pSymlinks = new Gtk::CheckButton(_("_Follow Symlinks"), true);

  pRecursive->set_active(was_recursive);
  pSymlinks->set_active(was_follow_links);
  chooser->set_show_hidden(was_hidden);

  chooser->hbox_pack_end(*Gtk::manage(pRecursive));
  chooser->hbox_pack_end(*Gtk::manage(pSymlinks));

  if(!last_uri.empty())
    chooser->set_current_folder_uri(last_uri);

  response = chooser->run();

  if(response != Gtk::RESPONSE_CANCEL)
  {
    Glib::RefPtr<Gnome::Vfs::Uri> vfs_uri =
      Gnome::Vfs::Uri::create(chooser->get_current_folder_uri());
    
    last_uri = vfs_uri->get_parent()->to_string();

    was_hidden = chooser->get_show_hidden();
    was_recursive = pRecursive->get_active();
    was_follow_links = pSymlinks->get_active();
  }
  
  hidden = chooser->get_show_hidden();
  recursive = pRecursive->get_active();
  follow_links = pSymlinks->get_active();

  return chooser->get_uri();
}

Glib::ustring offer_saveas(Gtk::Window& parent,
                           const Glib::ustring& old_filename,
                           int& response)
{
  std::auto_ptr<Chooser> chooser(new Chooser(parent,
                                             Dialogs::CHOOSER_TYPE_SAVE));

  if(!old_filename.empty())
  {
    chooser->set_current_folder_uri(old_filename);
    chooser->set_uri(old_filename);
  }

  response = chooser->run();

//  m_last_dir_ = chooser->get_current_folder_uri();
  
  return chooser->get_uri();
}

void offer_save_changes(Gtk::Window& parent,
                        const Glib::ustring& filename,
                        int& response)
{
  using namespace Gnome::Glade;
  const Glib::RefPtr<Xml> xml = Xml::create(glade_dialogs_filename, "diag_offer_save");

  Gtk::Dialog *tdialog = 0;
  std::auto_ptr<Gtk::Dialog> dialog(xml->get_widget("diag_offer_save", tdialog));
  dialog->set_transient_for(parent);
  dialog->set_default_response(Gtk::RESPONSE_CANCEL);
 
  Gtk::Label* pDocLabel = 0;
  xml->get_widget("offer_save_doc_label", pDocLabel);
  pDocLabel->set_text(filename);

  response = dialog->run();
}

Glib::ustring offer_open_files(Gtk::Window& parent)
{
  static Glib::ustring last_uri = "";

  std::auto_ptr<Chooser> chooser(new Chooser(parent,
                                             Dialogs::CHOOSER_TYPE_OPEN));

  if(!last_uri.empty())
    chooser->set_current_folder_uri(last_uri);

  int response = chooser->run();

  if(response != Gtk::RESPONSE_CANCEL)
  {
    last_uri = chooser->get_current_folder_uri();
    return chooser->get_uri();
  }
  else
    return Glib::ustring();
}

void offer_open_iso(Gtk::Window& parent,
                    int& response,
                    std::string& filename)
{
  Gtk::FileChooserDialog fc(parent, _("Open an ISO"));
  
  fc.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  fc.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_OK);
  fc.set_default_response(Gtk::RESPONSE_OK);

  Gtk::FileFilter* pIsoFiles = new Gtk::FileFilter();
  pIsoFiles->set_name(_("ISO Images"));
  pIsoFiles->add_mime_type("application/x-cd-image");

  Gtk::FileFilter* pAllFiles = new Gtk::FileFilter();
  pAllFiles->set_name(_("All Files"));
  pAllFiles->add_pattern("*");

  fc.add_filter(*Gtk::manage(pIsoFiles));
  fc.add_filter(*Gtk::manage(pAllFiles));

  response = fc.run();
  filename = Glib::filename_from_utf8(fc.get_filename());
}

void offer_export_iso(Gtk::Window& parent,
                      int& response,
                      std::string& filename)
{
  Gtk::FileChooserDialog fc(parent,
                            _("Export to..."),
                            Gtk::FILE_CHOOSER_ACTION_SAVE);
  
  fc.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  fc.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_OK);
  fc.set_default_response(Gtk::RESPONSE_OK);

  Gtk::FileFilter* pIsoFiles = new Gtk::FileFilter();
  pIsoFiles->set_name(_("ISO Images"));
  pIsoFiles->add_mime_type("application/x-cd-image");

  Gtk::FileFilter* pAllFiles = new Gtk::FileFilter();
  pAllFiles->set_name(_("All Files"));
  pAllFiles->add_pattern("*");

  fc.add_filter(*Gtk::manage(pIsoFiles));
  fc.add_filter(*Gtk::manage(pAllFiles));

  response = fc.run();
  filename = Glib::filename_from_utf8(fc.get_filename());
}

bool ask_overwrite(Gtk::Window& parent,
                   const Glib::ustring& file_uri)
{
  Glib::ustring filename_text = Gnome::Vfs::unescape_string_for_display(file_uri);
  filename_text = String::middle_truncate(filename_text, 50);
  Glib::ustring primary = String::ucompose(_("A file named \"%1\" already exists."),
                                           filename_text);
  Gtk::MessageDialog ask_overwrite(primary, false, Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_NONE, true);

  ask_overwrite.set_secondary_text(_("Do you want to replace it with the layout you are saving?"));
  ask_overwrite.set_title(_("Overwrite Layout?"));

  Widgets::SLButton *button = new Widgets::SLButton(Gtk::Stock::REFRESH, _("_Replace"));
  ask_overwrite.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  ask_overwrite.add_action_widget(*manage(button), Gtk::RESPONSE_YES);

  ask_overwrite.set_default_response(Gtk::RESPONSE_CANCEL);
  ask_overwrite.set_transient_for(parent);
  ask_overwrite.set_resizable(false);

  int response = ask_overwrite.run();

  return (response == Gtk::RESPONSE_YES);
}

bool ask_revert(Gtk::Window& parent,
                const Glib::ustring& filename)
{
  Glib::ustring primary_text = String::ucompose(_("Revert unsaved changes to layout \"%1\"?"),
                                                filename);
  Gtk::MessageDialog ask_revert(primary_text, false, Gtk::MESSAGE_QUESTION,
                                Gtk::BUTTONS_NONE, true);
  ask_revert.set_secondary_text(_("Changes made to the layout since the last time it was saved will be permanently lost."));
  ask_revert.set_title(_("Revert Layout?"));

  Widgets::SLButton *button = new Widgets::SLButton(Gtk::Stock::REFRESH, _("_Revert"));
  ask_revert.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  ask_revert.add_action_widget(*manage(button), Gtk::RESPONSE_YES);

  ask_revert.set_default_response(Gtk::RESPONSE_CANCEL);
  ask_revert.set_transient_for(parent);
  ask_revert.set_resizable(false);

  int response = ask_revert.run();

  return (response == Gtk::RESPONSE_YES);
}

bool warn_title_check(Gtk::Window& parent,
                      const sigc::slot<bool>& title_check_slot)
{
  using namespace Gnome::Glade;
  const Glib::RefPtr<Xml> xml = Xml::create(glade_dialogs_filename, "diag_title_check");

  ClientDialog *tdialog = 0;
  std::auto_ptr<ClientDialog> title_check(xml->get_widget_derived("diag_title_check", tdialog));
  title_check->set_transient_for(parent);
  title_check->set_default_response(Gtk::RESPONSE_ACCEPT);

  Gtk::Image* pBurnImage = 0;
  xml->get_widget("title_burn_image", pBurnImage);
  pBurnImage->set(Stock::DISC_BURN, Gtk::ICON_SIZE_BUTTON);

  title_check->connect_widget("layouts/title_check", "titlechk_check");

  bool title_check_return = false;

  int response;
  while(!title_check_return)
  {
    title_check->load();
    response = title_check->run();

    if(response != Gtk::RESPONSE_ACCEPT)
      break;
    else
      title_check_return = title_check_slot();
  }

  return (response != Gtk::RESPONSE_CANCEL);
}

bool warn_remove_files(Gtk::Window& parent)
{
  using namespace Gnome::Glade;
  const Glib::RefPtr<Xml> xml = Xml::create(glade_dialogs_filename, "diag_remove_warn");

  ClientDialog *tdialog = 0;
  std::auto_ptr<ClientDialog> remove_warn(xml->get_widget_derived("diag_remove_warn", tdialog));
  remove_warn->set_transient_for(parent);
  remove_warn->set_default_response(Gtk::RESPONSE_OK);
  remove_warn->connect_widget("layouts/remove_warn", "removewarn_check");
  remove_warn->load();

  int response = remove_warn->run();
  
  return (response == Gtk::RESPONSE_OK);
}

Glib::ustring find(Gtk::Window& parent)
{
  Glib::ustring search_string = "";

  std::auto_ptr<Find> find = GladeDialog<Find>::create(parent, "diag_find");
  int result = find->run();

  switch(result)
  {
    case Gtk::RESPONSE_OK:
      search_string = find->get_search_string();
      break;
    case Gtk::RESPONSE_CLOSE:
    default:
      break;
  }

  return search_string;
}

void preferences(Gtk::Window& parent)
{
  std::auto_ptr<Preferences> preferences =
    GladeDialog<Preferences>::create(parent,
                                     "diag_prefs",
                                     glade_prefs_filename);
  preferences->run();
}

} // namespace Dialogs

} // namespace Coaster
