/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "data/data-view.h"

#include "cstr-intl.h"
#include "cstr-debug.h"
#include "stock.h"
#include "exception.h"

#include "ucompose.h"

#include "undo-manager.h"

#include "widgets/busy-cursor.h"

#include "conf/conf-utils.h"

#include "data/data-layout.h"
#include "data/data-disc-info.h"
#include "data/data-prop-dialog.h"
#include "data/data-store.h"
#include "data/data-utils.h"

#include "backends/cd-io.h"
#include "backends/nautilus-make-iso.h"
#include "sound.h"
#include "status-icon.h"
#include "instance-manager.h"

#include "dialogs/glade-dialog.h"
#include "dialogs/burn.h"
#include "dialogs/add-progress.h"
#include "dialogs/dialog-util.h"

#include <gtkmm/actiongroup.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/stock.h>

#include <libgnomevfsmm/utils.h>

#include <libxml++/parsers/domparser.h>
#include <libxml++/document.h>

#include <iostream>

namespace
{

Glib::ustring get_node_value(const xmlpp::Element* node,
                             const Glib::ustring& strAttributeName)
{
  if(node)
  {
    const xmlpp::Attribute* attribute = node->get_attribute(strAttributeName);
    if(attribute)
    {      
      Glib::ustring value = attribute->get_value(); //Success.
      return value;
    }
  }

  return ""; //Failed.
}

void set_node_value(xmlpp::Element* node,
                    const Glib::ustring& strAttributeName,
                    const Glib::ustring& strValue)
{
  if(node)
  {
    xmlpp::Attribute* attribute = node->get_attribute(strAttributeName);
    if(attribute)
      attribute->set_value(strValue);
    else
      node->set_attribute(strAttributeName, strValue);
  }
}

} // anonymous namespace

namespace Coaster
{

namespace Data
{

View::View()
: View_Base()
{
  init();
}

View::~View()
{}

Glib::ustring View::get_ui_location() const
{
  return Glib::ustring(COASTER_UIDIR "/coaster-data.ui");
}

void View::disc_size_add(const Gnome::Vfs::FileSize& size)
{
  static_cast<DiscInfo*>(m_pDiscInfo)->disc_size_add(size);
}

void View::disc_size_remove(const Gnome::Vfs::FileSize& size)
{
  static_cast<DiscInfo*>(m_pDiscInfo)->disc_size_remove(size);
}

LayoutType View::get_layout_type() const
{
  return COASTER_LAYOUT_DATA;
}

Glib::ustring View::get_contents() const
{
  return m_refStore->get_contents();
}

bool View::load_contents(const Glib::ustring& contents)
{
  return m_refStore->set_contents(contents);
}

void View::prepare_for_revert()
{
  m_refStore->clear(true);
}

void View::init()
{
  debug("Data::View::init()");

  using namespace Gtk;

  // Set up the View's actions
  // File:
  m_refExport = Action::create("CoasterAction_ExportISO", Gtk::Stock::CONVERT,
                               _("E_xport to ISO..."), _("Export this layout to an ISO image"));
  m_refViewActionGroup->add(m_refExport,
                            sigc::mem_fun(*this, &View::on_file_export_iso));

  // Edit:
  m_refCut = Action::create("CoasterAction_Cut", Gtk::Stock::CUT);
  m_refViewActionGroup->add(m_refCut,
                            sigc::mem_fun(*this, &View::on_edit_cut));

  m_refCopy = Action::create("CoasterAction_Copy", Gtk::Stock::COPY);
  m_refViewActionGroup->add(m_refCopy,
                            sigc::mem_fun(*this, &View::on_edit_copy));

  m_refViewActionGroup->add(Action::create("CoasterAction_Paste", Gtk::Stock::PASTE),
                            sigc::mem_fun(*this, &View::on_edit_paste));
 
  m_refClear = Action::create("CoasterAction_Clear", Gtk::Stock::CLEAR,
                              _("C_lear"), _("Clear all files from the current layout"));
  m_refViewActionGroup->add(m_refClear,
                            sigc::mem_fun(*this, &View::on_edit_clear));
  
  m_refSelectAll = Action::create("CoasterAction_SelectAll", 
                            _("Select _All"), _("Select all files in current layout"));
  m_refViewActionGroup->add(m_refSelectAll, 
                            AccelKey("<control><shift>A"),
                            sigc::mem_fun(*this, &View::on_edit_select_all));

  // Search:
  m_refViewActionGroup->add(Action::create("CoasterAction_Search_Find", Gtk::Stock::FIND,
                              _("_Find"), _("Search for filename")),
                              sigc::mem_fun(*this, &View::on_search_find));
  m_refViewActionGroup->add(Action::create("CoasterAction_Search_FindNext", Gtk::Stock::FIND, 
                              _("Find Ne_xt"), _("Search forwards for the same filename")),
                              AccelKey("<control>G"));
  m_refViewActionGroup->add(Action::create("CoasterAction_Search_FindPrev", Gtk::Stock::FIND,
                              _("Find Ne_xt"), _("Search backwards for the same filename")),
                              AccelKey("<shift><control>G"));

  // Layout:
  Glib::RefPtr<Action> addFiles = Action::create("CoasterAction_Layout_Add_Files",
                        Stock::ADD_FILES, 
                        _("_Add..."), _("Add files to the current layout"));
  addFiles->property_is_important() = true;
  m_refViewActionGroup->add(addFiles,
                        Gtk::AccelKey("<control>A"),
                        sigc::mem_fun(*this, &View::on_layout_add_files));
  m_refViewActionGroup->add(Action::create("CoasterAction_Layout_Add_Folder",
                        Stock::ADD_FOLDER,
                        _("Add F_older..."),
                        _("Add a folder from your computer to the current layout")),
                        sigc::mem_fun(*this, &View::on_layout_add_folder));

  m_refViewActionGroup->add(Action::create("CoasterAction_Layout_Create_Folder",
                        _("_Create Folder"), _("Create a folder in the current layout")),
                        Gtk::AccelKey("<control><shift>N"),
                        sigc::mem_fun(*this, &View::on_layout_create_folder));

  m_refRename = Action::create("CoasterAction_Layout_Rename", _("R_ename"),
                               _("Rename the selected file in the current layout"));
  m_refViewActionGroup->add(m_refRename, Gtk::AccelKey("F2"),
                        sigc::mem_fun(*this, &View::on_layout_rename));

  m_refRemove = Action::create("CoasterAction_Layout_Remove", Gtk::Stock::REMOVE, 
                               _("_Remove"),
                               _("Remove the selected file(s) in the current layout"));
  m_refViewActionGroup->add(m_refRemove,
                        Gtk::AccelKey("Delete"),
                        sigc::mem_fun(*this, &View::on_layout_remove));
  
  m_refRename->set_sensitive(false);

  View_Base::init();
}

void View::create_store()
{
  Data::Columns columns;
  m_refStore = Store::create(columns,
                             sigc::mem_fun(*this, &View::disc_size_add),
                             sigc::mem_fun(*this, &View::disc_size_remove),
                             sigc::mem_fun(*undo_manager, &UndoManager::push));
  m_refStore->signal_modified().connect(sigc::mem_fun(*this, &View::on_store_modified));
}

void View::create_layout()
{
  m_pLayout = new Layout(m_refStore,
                         sigc::mem_fun(*this, &View::popup),
                         sigc::mem_fun(*this, &View::disc_size_add),
                         sigc::mem_fun(*this, &View::disc_size_remove));

  View_Base::create_layout();
}

void View::create_disc_info()
{
  m_pDiscInfo = new DiscInfo(sigc::mem_fun(*undo_manager, &UndoManager::push));

  View_Base::create_disc_info();
}

void View::save_to_document()
{
  debug("View: save to doc");
  /*get_document()->start_save();
  static_cast<Document*>(get_document())->save_tree(m_pLayout->get_tree());*/
}

void View::load_from_document()
{
  debug("View: load from doc");
  /*m_pDiscInfo->signal_disc_size_changed().connect(sigc::mem_fun(*this,
                                                                &View::on_disc_size_changed));

  static_cast<Document*>(get_document())->load_tree(m_pLayout->get_tree(),
                                                    sigc::mem_fun(*this, &View::disc_size_add));*/

  enable_layout_item_actions(m_refStore->has_files());
  //load_icons(true /* loading doc */);
}

void View::enable_selection_actions(bool enable)
{
  m_refRemove->set_sensitive(enable);
  m_refCut->set_sensitive(enable);
  m_refCopy->set_sensitive(enable);
}

void View::enable_layout_item_actions(bool enable)
{
  m_refClear->set_sensitive(enable);
  m_refSelectAll->set_sensitive(enable);
  m_refExport->set_sensitive(enable);

  View_Base::enable_layout_item_actions(enable);
}

void View::on_disc_size_changed(DiscSizeType size)
{
  set_modified(true);
}

void View::on_store_modified(bool modified)
{
  enable_layout_item_actions(m_refStore->has_files());

  View_Base::on_store_modified(modified);
}

void View::on_layout_selection_changed(int number_selected)
{
  if(number_selected == 1)
    m_refRename->set_sensitive(true);
  else
    m_refRename->set_sensitive(false);

  View_Base::on_layout_selection_changed(number_selected);
}

void View::on_file_export_iso()
{
  // set all instances so we can't do anything
  m_signal_sensitive.emit(false);

  int response;
  std::string iso_filename;
  
  Dialogs::offer_export_iso(get_win(), response, iso_filename);

  if(response == Gtk::RESPONSE_OK)
  {
    Widgets::BusyCursor cursor(get_win());
    Glib::ustring graft_file_contents = m_refStore->get_graft_contents();
    SharedPtr<IO::MakeIso> mkiso =
      SharedPtr<IO::MakeIso>(new IO::MakeIso(iso_filename,
                                             graft_file_contents,
                                             m_refStore->get_properties(),
                                             get_win()));

    mkiso->make_iso();
  }
  
  // return all instances to normal
  m_signal_sensitive.emit(true);
}

void View::on_edit_cut()
{
  m_refStore->cut();
}

void View::on_edit_copy()
{
  m_refStore->copy();
}

void View::on_edit_paste()
{
  m_refStore->paste();
}

void View::on_edit_clear()
{
  m_refStore->clear();
}

void View::on_edit_select_all()
{
  m_pLayout->select_all();
}

void View::on_search_find()
{
  Glib::ustring search_string = Dialogs::find(get_win());

  if(search_string != "")
  {
    m_pLayout->grab_focus();
    m_pLayout->unselect_all();

    m_refStore->search(search_string);
  }
}

void View::on_layout_add_files()
{
  debug("Data::View: on add files");
  bool hidden;
  int response;

  type_list_ustrings files =
    Dialogs::offer_add_files(get_win(), hidden, response);

  if(response != Gtk::RESPONSE_OK)
    return;

  Dialogs::AddProgress ap(get_win(), _("Files"));
  Widgets::BusyCursor cursor(get_win());
  m_refStore->add_files(files, hidden, sigc::mem_fun(ap, &Dialogs::AddProgress::show_filename));
}

void View::on_layout_add_folder()
{
  using namespace Gtk;
  
  debug("Layout: on add folder");
  bool hidden, recursive, follow_links;
  int response;
  
  Glib::ustring folder =
    Dialogs::offer_add_folder(get_win(), hidden, recursive, follow_links, response);

  if(response != RESPONSE_OK)
    return;

  Dialogs::AddProgress ap(get_win(), Gnome::Vfs::unescape_string(folder));
  Widgets::BusyCursor cursor(get_win());
  m_refStore->add_folder(folder, hidden, recursive, follow_links,
                         sigc::mem_fun(ap, &Dialogs::AddProgress::show_filename));
}

void View::on_layout_create_folder()
{
  m_refStore->create_folder();
}

void View::on_layout_remove()
{
  if(Conf::get_bool("layouts/title_check"))
  {
    if(!Dialogs::warn_remove_files(get_win()))
      return;
  }

  m_refStore->remove();
}

void View::on_layout_rename()
{
  m_pLayout->rename_selected();
}

bool View::set_properties()
{
  debug("Data::View::set_properties");

  using namespace Gtk;
  std::auto_ptr<PropDialog> dataprop =
    Dialogs::GladeDialog<PropDialog>::create(get_win(), "data_prop");
  dataprop->init(m_refStore->get_properties());
  int result = dataprop->run();

  if(result != Gtk::RESPONSE_CANCEL)
  {
    Properties tempDP = dataprop->get_properties();
    Properties tempDP_old = m_refStore->get_properties();

    /*std::cout << "Original Properties:\n"
              << "\tTitle: " << tempDP.title << "|\n"
              << "\tAppend Date: " << tempDP.append_date << "|\n"
              << "\tCatNum: " << tempDP.catnum << "|\n"
              << "\tPublisher: " << tempDP.publisher << "|\n"
              << "\tAuthor: " << tempDP.author << "|\n"
              << "\tIso Level: " << tempDP.isolevel << "|\n"
              << "\tJoliet: " << tempDP.joliet << "|\n"
              << "\tRockRidge: " << tempDP.rockridge << "|\n"
              << "\tSession Format: " << tempDP.sessionformat << "|" << std::endl;
    std::cout << "New Properties:\n"
              << "\tTitle: " << tempDP_old.title << "|\n"
              << "\tAppend Date: " << tempDP_old.append_date << "|\n"
              << "\tCatNum: " << tempDP_old.catnum << "|\n"
              << "\tPublisher: " << tempDP_old.publisher << "|\n"
              << "\tAuthor: " << tempDP_old.author << "|\n"
              << "\tIso Level: " << tempDP_old.isolevel << "|\n"
              << "\tJoliet: " << tempDP_old.joliet << "|\n"
              << "\tRockRidge: " << tempDP_old.rockridge << "|\n"
              << "\tSession Format: " << tempDP_old.sessionformat << "|" << std::endl;*/

    if(tempDP != tempDP_old)
    {
      m_refStore->set_properties(tempDP);
      return true;
    }
    else
    {
      return false;
    }
  }
  else
    return false;
}

void View::on_disc_burn()
{
  using namespace Gnome;
  debug("View: on burn clicked");

  Data::Properties properties = m_refStore->get_properties();
  if(properties.title == "")
  {
    bool check = Conf::get_bool("layouts/title_check");
    if(check)
    {
      // create and run dialog.  if they hit "No", return.  if they hit "Yes", go on.
      if(!Dialogs::warn_title_check(get_win(),
                                    sigc::mem_fun(*this, &View::set_properties)))
          return;
    } 
  }
    
  std::auto_ptr<Dialogs::Burn> burndialog = 
    Dialogs::GladeDialog<Dialogs::Burn>::create(get_win(),
                                       "diag_burn");

  // render this pixbuf before setting the interface insensitive
  // so we get a normal lookign icon
  Glib::RefPtr<Gdk::Pixbuf> pb_idle, pb_burn;
  pb_idle = render_icon(Stock::DISC_IDLE, Gtk::ICON_SIZE_MENU);
  pb_burn = render_icon(Stock::DISC_BURN, Gtk::ICON_SIZE_MENU);

  // set all instances so we can't do anything
  m_signal_sensitive.emit(false);

  StatusIcon::set(pb_idle, _("Idle"));

  int result = burndialog->run();

  if(result == Gtk::RESPONSE_OK)
  {
    burndialog->set_action_widgets_sensitive(false);
    burndialog->hide();
    
    StatusIcon::set(pb_burn, _("Burning..."));
    
    Glib::ustring graft_file_contents = m_refStore->get_graft_contents();
    IO::BurnDisc burn(get_win(),
                      burndialog->get_drive(),
                      burndialog->get_eject(),
                      burndialog->get_speed(),
                      burndialog->get_dummy(),
                      graft_file_contents,
                      properties);
    if(burn.start_burn())
      Sound::burn_finished();
    else
      Sound::burn_error();
  }

  StatusIcon::remove();

  // return all instances to normal
  m_signal_sensitive.emit(true);
}

} // namespace Data

} // namespace Coaster
