/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "data/data-undo-ops.h"

#include "cstr-intl.h"
#include "cstr-debug.h"

#include "ucompose.h"

#include "data/data-store.h"
#include "data/data-utils.h"

#include <libxml++/document.h>

namespace Coaster
{

namespace Data
{

/////////////////////
//  AddRemOp
/////////////////////
AddRemOp::AddRemOp(const xmlpp::Document* structure,
                   const Glib::ustring& label,
                   const sigc::slot<void,const xmlpp::Element*>& undo_slot,
                   const sigc::slot<void,const xmlpp::Element*>& redo_slot)
: UndoOp(label),
  undo_from_xml(undo_slot), redo_from_xml(redo_slot)
{
  m_pDocument = const_cast<xmlpp::Document*>(structure);
  debug_xml("Undo ", m_Label, " XML:");
  debug_xml(m_pDocument);
}

AddRemOp::~AddRemOp()
{}

AddRemOpPtr AddRemOp::create(const xmlpp::Document* structure,
                             const Glib::ustring& label,
                             const sigc::slot<void,const xmlpp::Element*>& undo_slot,
                             const sigc::slot<void,const xmlpp::Element*>& redo_slot)
{
  return AddRemOpPtr(new AddRemOp(structure, label, undo_slot, redo_slot));
}

void AddRemOp::undo()
{
  debug_xml("Undo: ",m_Label);
  undo_from_xml(m_pDocument->get_root_node());
}

void AddRemOp::redo()
{
  debug_xml("Redo: ",m_Label);
  redo_from_xml(m_pDocument->get_root_node());
}


/////////////////
//  RenameOp
/////////////////
RenameOp::RenameOp(const Glib::ustring& old_path_string,
                   const Glib::ustring& new_path_string,
                   const Glib::ustring& old_value,
                   const Glib::ustring& new_value,
                   const sigc::slot<void,const xmlpp::Element*,bool>& slot_rename)
: UndoOp(_("Rename")), rename_from_xml(slot_rename)
{
  m_pDocument = new xmlpp::Document();

  xmlpp::Element* rootNode = m_pDocument->create_root_node("coaster-rows");
  m_pRenameNode = rootNode->add_child("rename");
  m_pRenameNode->set_attribute("from", old_value);
  m_pRenameNode->set_attribute("to", new_value);
  m_pRenameNode->set_attribute("otpath", old_path_string);
  m_pRenameNode->set_attribute("ntpath", new_path_string);

  debug_xml("Undo Rename XML:");
  debug_xml(m_pDocument);
}

RenameOp::~RenameOp()
{}

RenameOpPtr RenameOp::create(const Glib::ustring& old_path_string,
                             const Glib::ustring& new_path_string,
                             const Glib::ustring& old_value,
                             const Glib::ustring& new_value,
                             const sigc::slot<void,const xmlpp::Element*,bool>& slot_rename)
{
  return RenameOpPtr(new RenameOp(old_path_string,
                                  new_path_string,
                                  old_value,
                                  new_value,
                                  slot_rename));
}

void RenameOp::undo()
{
  debug_xml("Undo Rename");
  rename_from_xml(m_pRenameNode, false);
}

void RenameOp::redo()
{
  debug_xml("Redo Rename");
  rename_from_xml(m_pRenameNode, true);
}


/////////////////
//  MoveOp
/////////////////
MoveOp::MoveOp(const xmlpp::Element* old_xml_root,
               const xmlpp::Element* new_xml_root,
               const sigc::slot<void,const xmlpp::Element*>& add_slot,
               const sigc::slot<void,const xmlpp::Element*>& remove_slot)
: UndoOp(_("Move Files")), add_back(add_slot), remove_from(remove_slot),
  new_node(0), old_node(0)
{
  m_pDocument = new xmlpp::Document();

  xmlpp::Element* rootNode = m_pDocument->create_root_node("coaster-move-rows");

  old_node = static_cast<xmlpp::Element*>(rootNode->import_node(old_xml_root));
  old_node->set_name("old-coaster-rows");

  new_node = static_cast<xmlpp::Element*>(rootNode->import_node(new_xml_root));
  new_node->set_name("new-coaster-rows");

  debug_xml(m_pDocument);
}

MoveOp::~MoveOp()
{}

MoveOpPtr MoveOp::create(const xmlpp::Element* old_xml_root,
                         const xmlpp::Element* new_xml_root,
                         const sigc::slot<void,const xmlpp::Element*>& add_slot,
                         const sigc::slot<void,const xmlpp::Element*>& remove_slot)
{
  return MoveOpPtr(new MoveOp(old_xml_root, new_xml_root, add_slot, remove_slot));
}

void MoveOp::undo()
{
  debug_xml("Undo Move");
  add_back(old_node);
  remove_from(new_node);
}

void MoveOp::redo()
{
  debug_xml("Redo Move");
  add_back(new_node);
  remove_from(old_node);
}

} // namespace Data

} // namespace Coaster
