/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "data/data-store.h"

#include "cstr-intl.h"
#include "cstr-debug.h"
#include "exception.h"

#include "edit-ops.h"
#include "undo-op.h"
#include "icon-cache.h"

#include "conf/conf-utils.h"

#include "data/data-file.h"
#include "data/data-utils.h"
#include "data/data-undo-ops.h"

#include <gtkmm/widget.h>
#include <gtkmm/treeselection.h>

#include <libgnomevfsmm/directory-handle.h>
#include <libgnomevfsmm/utils.h>

#include <libxml++/parsers/domparser.h>
#include <libxml++/document.h>

#include <sstream>

namespace
{

template<typename T>
Glib::ustring to_string(const T& x)
{
  std::ostringstream oss;
  oss.imbue(std::locale(""));
  oss << x;
  return Glib::locale_to_utf8(oss.str());
}

Glib::ustring to_graft_line(const Glib::ustring& one,
                            const Glib::ustring& two)
{
  std::ostringstream oss;
  oss.imbue(std::locale(""));  
  oss << one << "=" << two;
  return Glib::locale_to_utf8(oss.str());
}

Glib::ustring to_graft_line(const Glib::ustring& one,
                            const Glib::ustring& two,
                            const Glib::ustring& three)
{
  std::ostringstream oss;
  oss.imbue(std::locale(""));  
  oss << one << "/" << two << "=" << three;
  return Glib::locale_to_utf8(oss.str());
}

Glib::ustring to_graft_prefix(const Glib::ustring& one,
                              const Glib::ustring& two)
{
  std::ostringstream oss;
  oss.imbue(std::locale(""));  
  oss << one << "/" << two;
  return Glib::locale_to_utf8(oss.str());
}

Gnome::Vfs::FileSize to_size(const Glib::ustring& string)
{
  if(string == "")
    return 0;

  std::istringstream iss;
  iss.imbue(std::locale(""));
  iss.str(Glib::locale_from_utf8(string));

  Gnome::Vfs::FileSize result;
  iss >> result;

  return result;
}

int sort_name(const Gtk::TreeModel::iterator& a,
              const Gtk::TreeModel::iterator& b)
{
  using namespace Coaster;
  const Data::Columns& model_columns = Data::columns();

  bool a_isdir = Data::is_tree_dir(a);
  bool b_isdir = Data::is_tree_dir(b);

  if(a_isdir && b_isdir)
  {
    const std::string a_str = (*a).get_value(model_columns.m_col_name).collate_key();
    const std::string b_str = (*b).get_value(model_columns.m_col_name).collate_key();
    return a_str.compare(b_str);
  }
  else if(a_isdir && !b_isdir)
  {
    return -1;
  }
  else if(!a_isdir && b_isdir)
  {
    return 1;
  }
  else
  {
    const std::string a_str = (*a).get_value(model_columns.m_col_name).collate_key();
    const std::string b_str = (*b).get_value(model_columns.m_col_name).collate_key();
    return a_str.compare(b_str);
  }
}

int sort_size(const Gtk::TreeModel::iterator& a,
              const Gtk::TreeModel::iterator& b)
{
  using namespace Coaster;
  const Data::Columns& model_columns = Data::columns();

  bool a_isdir = Data::is_tree_dir(a);
  bool b_isdir = Data::is_tree_dir(b);

  if(a_isdir && b_isdir)
  {
    int a_childsize = (*a).children().size();
    int b_childsize = (*b).children().size();

    if(a_childsize > b_childsize)
      return -1;
    if(a_childsize < b_childsize)
      return 1;
    else
      return 0;
  }
  else if(a_isdir && !b_isdir)
  {
    return -1;
  }
  else if(!a_isdir && b_isdir)
  {
    return 1;
  }
  else
  {
    Gnome::Vfs::FileSize a_int = (*a).get_value(model_columns.m_col_size);
    Gnome::Vfs::FileSize b_int = (*b).get_value(model_columns.m_col_size);
    if(a_int > b_int)
      return -1;
    else if( a_int < b_int )
      return 1;
    else
      return 0;
  }
}

int sort_uri(const Gtk::TreeModel::iterator& a,
             const Gtk::TreeModel::iterator& b)
{
  using namespace Coaster;
  const Data::Columns& model_columns = Data::columns();

  bool a_isdir = Data::is_tree_dir(a);
  bool b_isdir = Data::is_tree_dir(b);

  if(a_isdir && b_isdir)
  {
    const std::string a_str = (*a).get_value(model_columns.m_col_uri).collate_key();
    const std::string b_str = (*b).get_value(model_columns.m_col_uri).collate_key();
    return a_str.compare(b_str);
  }
  else if(a_isdir && !b_isdir)
  {
    return -1;
  }
  else if(!a_isdir && b_isdir)
  {
    return 1;
  }
  else
  {
    const std::string a_str = (*a).get_value(model_columns.m_col_uri).collate_key();
    const std::string b_str = (*b).get_value(model_columns.m_col_uri).collate_key();
    return a_str.compare(b_str);
  }
}

Glib::ustring get_node_value(const xmlpp::Element* node,
                             const Glib::ustring& strAttributeName)
{
  if(node)
  {
    const xmlpp::Attribute* attribute = node->get_attribute(strAttributeName);
    if(attribute)
    {      
      Glib::ustring value = attribute->get_value(); //Success.
      return value;
    }
  }

  return ""; //Failed.
}

} // anonymous namespace

namespace Coaster
{

namespace Data
{

Store::Store(const Columns& cols,
             const type_slot_size_change& slot_disc_size_add,
             const type_slot_size_change& slot_disc_size_remove,
             const type_slot_push_undo& slot_push_undo)
: disc_size_add(slot_disc_size_add), disc_size_remove(slot_disc_size_remove)
{
  set_column_types(cols);
  set_push_undo(slot_push_undo);

  set_sort_func(m_Columns.m_col_name, sigc::ptr_fun(&sort_name));
  set_sort_func(m_Columns.m_col_size, sigc::ptr_fun(&sort_size));
  set_sort_func(m_Columns.m_col_uri, sigc::ptr_fun(&sort_uri));
  set_sort_column(m_Columns.m_col_name, Gtk::SORT_ASCENDING);

  IconCache::signal_icon_theme_changed().connect(
                                  sigc::mem_fun(*this, &Store::on_icon_theme_changed));
}

Store::~Store()
{}

Glib::RefPtr<Store> Store::create(const Columns& cols,
                                  const type_slot_size_change& slot_disc_size_add,
                                  const type_slot_size_change& slot_disc_size_remove,
                                  const type_slot_push_undo& slot_push_undo)
{
  return Glib::RefPtr<Store>( new Store(cols,
                                        slot_disc_size_add,
                                        slot_disc_size_remove,
                                        slot_push_undo) );
}

void Store::set_expand_funcs(const type_slot_expand& slot_expand,
                             const type_slot_expand_to& slot_expand_to)
{
  expand_path = slot_expand;
  expand_to_path = slot_expand_to;
}

bool Store::has_files() const
{
  return !(children().empty());
}

Gnome::Vfs::FileSize Store::get_size() const
{
  return get_size(children());
}

Gnome::Vfs::FileSize Store::get_size(const Gtk::TreeRow& row) const
{
  Gnome::Vfs::FileSize size = 0;

  if(Data::is_tree_dir(row))
  {
    size += get_size(row.children());
  }
  else
  {
    size += row.get_value(m_Columns.m_col_size);
  }

  return size;
}

Gnome::Vfs::FileSize Store::get_size(const type_list_rows& rows) const
{
  Gnome::Vfs::FileSize size = 0;

  for(type_list_rows::const_iterator i = rows.begin() ; i != rows.end() ; ++i)
  {
    size += get_size(*i);
  }

  return size;
}

Gnome::Vfs::FileSize Store::get_size(const Gtk::TreeNodeChildren& children) const
{
  Gnome::Vfs::FileSize size = 0;

  for(Gtk::TreeIter iter = children.begin() ; iter != children.end() ; ++iter)
  {
    if(Data::is_tree_dir(iter))
    {
      size += get_size((*iter).children());
    }
    else
    {
      size += (*iter).get_value(m_Columns.m_col_size);
    }
  }

  return size;
}

void Store::copy()
{
  type_list_paths selected_paths = m_refSelection->get_selected_rows();

  int sp_size = selected_paths.size();
  if(sp_size > 0)
  {
    xmlpp::Document document;
    xmlpp::Element* rootNode = document.create_root_node("coaster-data-rows");

    type_list_rows rows;
    
    if(sp_size == 1)
    {
      rows.push_back(*(get_iter(selected_paths.front())));
    }
    else if(sp_size > 1)
    {
      for(type_list_paths::iterator i = selected_paths.begin() ; i != selected_paths.end() ; ++i)
      {
        rows.push_back(*(get_iter(*i)));
      }
    }

    branch_to_xml(rows, rootNode, true);

    debug_xml(&document);

    Glib::ustring contents = document.write_to_string_formatted();

    Edit::copy_data_rows(contents);
  }
}

void Store::cut()
{
  type_list_paths selected_paths = m_refSelection->get_selected_rows();

  int sp_size = selected_paths.size();
  if(sp_size > 0)
  {
    copy();
    remove(true);
  }
}

void Store::paste()
{
  Edit::paste_data_rows(sigc::mem_fun(*this, &Store::paste_from_string));
}

void Store::add_files(const type_list_ustrings& files,
                      bool hidden,
                      const type_slot_report_file& slot_report)
{
  type_list_rows rows_added;
  
  bool changed = false;

  for(type_list_ustrings::const_iterator iter = files.begin() ; iter != files.end() ; ++iter)
  {
    File data_file(*iter);

    if(data_file.is_valid())
    {
      Glib::ustring tmpName = data_file.get_name();
      
      if(!slot_report(Gnome::Vfs::unescape_string(*iter))) // if the user hit cancel
        break;

      Glib::ustring firstChar = tmpName.substr(0,1);

      if(hidden == false && firstChar == ".")
        continue;

      changed = true;
      Gtk::TreeRow new_row = get_new_entity_row();
      rows_added.push_back(new_row);

      RowAttr row_attr(tmpName, data_file.get_size(),
                       COASTER_ENTITY_FILE, *iter,
                       data_file.get_mime_type());
      set_row_attribs(new_row, row_attr);
      disc_size_add(data_file.get_size());

      expand_to_path(get_path(new_row));
    }
  }

  if(changed)
  {
    push_undo(AddRemOp::create(to_document(rows_added, true, true),
                               _("Add Files"),
                               sigc::mem_fun(*this, &Store::remove_from_xml),
                               sigc::mem_fun(*this, &Store::add_back_from_xml)));
  }
}

void Store::add_folder(const Glib::ustring& directory,
                       bool hidden,
                       bool recursive,
                       bool follow_links,
                       const type_slot_report_file& slot_report)
{
  Glib::ustring::size_type posLastSlash = directory.find_last_of("/");
  Glib::ustring folder_name;
  if(posLastSlash == Glib::ustring::npos)
    folder_name = "/";
  else
    folder_name = directory.substr(posLastSlash+1);

  if(!slot_report(Gnome::Vfs::unescape_string(folder_name))) // if the user happens to hit cancel
    return;

  Gtk::TreeRow parent_row = get_new_entity_row();

  RowAttr row_attr(Gnome::Vfs::unescape_string(folder_name));
  set_row_attribs(parent_row, row_attr);

  bool changed = false;
  add_to_folder_from_dir(parent_row, directory, hidden, recursive, follow_links,
                         changed, slot_report);

  if(changed)
  {
    push_undo(AddRemOp::create(to_document(parent_row, true, true),
                               _("Add Directory"),
                               sigc::mem_fun(*this, &Store::remove_from_xml),
                               sigc::mem_fun(*this, &Store::add_back_from_xml)));
  }

  expand_path(get_path(parent_row), true);
}

void Store::create_folder()
{
  Gtk::TreeRow new_row = get_new_entity_row();
  
  RowAttr row_attr(_("untitled folder"));
  set_row_attribs(new_row, row_attr);

  // push create directory undo op onto stack
  push_undo(AddRemOp::create(to_document(new_row, true, true),
                             _("Create Directory"),
                             sigc::mem_fun(*this, &Store::remove_from_xml),
                             sigc::mem_fun(*this, &Store::add_back_from_xml)));

  expand_to_path(get_path(new_row));
}

void Store::remove(bool cut)
{
  type_list_rows rows_to_remove = remove_duplicate_rows();

  if(rows_to_remove.size() > 0)
  {
    push_undo(AddRemOp::create(to_document(rows_to_remove, true, true),
                               (cut ? _("Cut") : _("Remove")),
                               sigc::mem_fun(*this, &Store::add_back_from_xml),
                               sigc::mem_fun(*this, &Store::remove_from_xml)));

    disc_size_remove(get_size(rows_to_remove));

    for(type_list_rows::reverse_iterator i = rows_to_remove.rbegin() ;
        i != rows_to_remove.rend() ;
        ++i)
    {
      erase((*i));
    }
  }
}

void Store::clear(bool revert)
{
  if(!revert)
    push_undo(AddRemOp::create(to_document(true, true),
                               _("Clear"),
                               sigc::mem_fun(*this, &Store::add_back_from_xml),
                               sigc::mem_fun(*this, &Store::remove_from_xml)));

  disc_size_remove(get_size());
  Gtk::TreeStore::clear();
}

void Store::rename(const Glib::ustring& path_text,
                   const Glib::ustring& new_text)
{
  Gtk::TreeRow row = *(get_iter(path_text));
  
  Glib::ustring previous_name = row.get_value(m_Columns.m_col_name);

  if(previous_name != new_text)
  {
    debug(String::ucompose("Data::Store::rename(): From %1 to %2", previous_name, new_text));

    Gtk::TreePath old_path = get_path(row);
    row[m_Columns.m_col_name] = new_text;
    Gtk::TreePath new_path = get_path(row);

    push_undo(RenameOp::create(old_path.to_string(),
                               new_path.to_string(),
                               previous_name, new_text,
                               sigc::mem_fun(*this, &Store::rename_from_xml)));
  }
}

Glib::ustring Store::get_contents(bool internal,
                                  bool save_parent_path) const
{
  using namespace xmlpp;

  Document doc;
  xmlpp::Element* rootNode = 0;

  if(!internal)
  {
    doc.set_internal_subset("disclayout",
                            "-//COASTER//DTD Data",
                            "http://www.coaster-burn.org/coaster-data.dtd");
    rootNode = doc.create_root_node("disclayout");
    rootNode->set_attribute("type", "data");
    prop_to_xml(rootNode);
  }
  else
  {
    rootNode = doc.create_root_node("coaster-data-rows");
  }


  branch_to_xml(children(), rootNode, internal, save_parent_path);

  Glib::ustring result = doc.write_to_string_formatted();

  return result;
}

Glib::ustring Store::get_graft_contents() const
{
  using namespace xmlpp;

  Document doc;
  xmlpp::Element* rootNode = 0;

  rootNode = doc.create_root_node("coaster-data-rows");
  branch_to_xml(children(), rootNode);

  std::ostringstream oss;
  oss.imbue(std::locale(""));

  get_graft_contents(oss, rootNode);
  debug("Data::Store::get_graft_contents(): Graft File Contents: ", oss.str());
  return Glib::locale_to_utf8(oss.str());
}

void Store::get_graft_contents(std::ostringstream& graft_stream,
                               const xmlpp::Element* parent,
                               const Glib::ustring& prefix) const
{
  using namespace Gtk;

  xmlpp::Node::NodeList children = get_child_nodes(parent);

  if(!children.empty())
  {
    for(xmlpp::Node::NodeList::iterator i = children.begin() ; i != children.end() ; ++i)
    {
      xmlpp::Element* current_elem = static_cast<xmlpp::Element*>(*i);

      Glib::ustring name = get_node_value(current_elem, "name");

      if(current_elem->get_name() == "data")
      {
        Glib::ustring uri = get_node_value(current_elem, "path");
        Glib::RefPtr<Gnome::Vfs::Uri> vfs_uri = Gnome::Vfs::Uri::create(uri);

        if(vfs_uri->is_local())
          uri = Gnome::Vfs::get_local_path_from_uri(vfs_uri->to_string());
        else
        {
          // TODO: warn the user that this isn't supported right now!
          //
          // transfer the file to a temporary location
          // modify "uri" to point to temporary location
        }

        if(prefix.empty())
        {
          // output "name=uri" in file
          graft_stream << to_graft_line(name, uri) << std::endl;
          debug(to_graft_line(name, uri));
        }
        else
        {
          // output "prefix/name=uri" in file
          graft_stream << to_graft_line(prefix, name, uri) << std::endl;
          debug(to_graft_line(prefix, name, uri));
        }
      }
      else
      {
        if(!prefix.empty())
        {
          Glib::ustring new_prefix = to_graft_prefix(prefix, name);
          get_graft_contents(graft_stream, current_elem, new_prefix);
        }
        else
        {
          get_graft_contents(graft_stream, current_elem, name);
        }
      }
    }
  }
}

Glib::ustring Store::get_selected_rows(const Glib::ustring& root_node_name) const
{
  using namespace xmlpp;
  type_list_paths selected_paths = m_refSelection->get_selected_rows();
  type_list_rows rows = remove_duplicate_rows();

  Document doc;
  Element* rootNode = doc.create_root_node(root_node_name);
  branch_to_xml(rows, rootNode, true, true);

  return doc.write_to_string_formatted();
}

bool Store::set_contents(const Glib::ustring& contents)
{
  using namespace xmlpp;
  DomParser parser;
  Document* doc = 0;
  Element* rootNode = 0;
  
  parser.set_validate();

  try
  {
    if(contents.empty())
      g_warning("Data::Store::dnd_move(): parsing empty document.");

    parser.parse_memory(contents);
  }
  catch(const std::exception& ex)
  {
    handle_exception("XML Parser error: \n", ex);

    return false; //Failed.
  }

  doc = parser.get_document();
  rootNode = doc->get_root_node();
  xml_to_prop(rootNode);

  Gtk::TreeNodeChildren this_children = children();
  branch_from_xml(rootNode, this_children);

  return true;
}

bool Store::dnd_copy(const Glib::ustring& xmlstruct,
                     Gtk::TreeNodeChildren& children,
                     bool same_instance)
{
  using namespace xmlpp;
  DomParser parser;
  Document* doc = 0;
  Element* rootNode = 0; 

  try
  {
    if(xmlstruct.empty())
      g_warning("Data::Store::dnd_copy(): parsing empty document.");

    parser.parse_memory(xmlstruct);
  }
  catch(const std::exception& ex)
  {
    handle_exception("XML Parser error: \n", ex);

    return false; //Failed.
  }

  doc = parser.get_document();
  rootNode = doc->get_root_node();

  type_list_rows rows_added;

  Node::NodeList node_list = get_child_nodes(rootNode);
  if(!node_list.empty())
  {
    for(Node::NodeList::iterator i = node_list.begin() ; i != node_list.end() ; ++i)
    {
      Element *current_elem = static_cast<xmlpp::Element*>(*i);

      Gtk::TreeRow new_row = *append(children);
      row_from_elem(new_row, current_elem);

      rows_added.push_back(new_row);
    }
  }

  Glib::ustring label = (same_instance ? N_("Copy Files") : N_("Add Files"));
  push_undo(AddRemOp::create(to_document(rows_added, true, true),
                             _(label.c_str()),
                             sigc::mem_fun(*this, &Store::remove_from_xml),
                             sigc::mem_fun(*this, &Store::add_back_from_xml)));

  return true;
}

bool Store::dnd_move(const Glib::ustring& xmlstruct,
                     Gtk::TreeNodeChildren& children)
{
  using namespace xmlpp;
  DomParser parser;
  Element* old_root_node = 0;
  Element* new_root_node = 0;
  Document* old_doc = 0;
  Document* new_doc = 0;

  try
  {
    if(xmlstruct.empty())
      g_warning("Data::Store::from_string(): parsing empty document.");

    parser.parse_memory(xmlstruct);
  }
  catch(const std::exception& ex)
  {
    handle_exception("XML Parser error: \n", ex);

    return false; //Failed.
  }

  old_doc = parser.get_document();
  old_root_node = old_doc->get_root_node();


  type_list_rows rows_added;
  move_branch_from_xml(old_root_node, children, rows_added);

  new_doc = to_document(rows_added, true, true);
  new_root_node = new_doc->get_root_node();

  remove_from_xml(old_root_node);

  push_undo(MoveOp::create(old_root_node, new_root_node,
                           sigc::mem_fun(*this, &Store::add_back_from_xml),
                           sigc::mem_fun(*this, &Store::remove_from_xml)));
  delete new_doc;
  return true;
}

xmlpp::Document* Store::to_document(bool internal,
                                    bool save_parent_path) const
{
  using namespace xmlpp;

  Document* doc = new Document();
  xmlpp::Element* rootNode = 0;

  if(!internal)
  {
    doc->set_internal_subset("disclayout",
                             "-//COASTER//DTD Data",
                             "http://www.coaster-burn.org/coaster-data.dtd");
    rootNode = doc->create_root_node("disclayout");
    rootNode->set_attribute("type", "data");
  }
  else
  {
    rootNode = doc->create_root_node("coaster-data-rows");
  }


  branch_to_xml(children(), rootNode, internal, save_parent_path);

  return doc;
}

xmlpp::Document* Store::to_document(const type_list_rows& rows,
                                    bool internal,
                                    bool save_parent_path) const
{
  using namespace xmlpp;

  Document* doc = new Document();
  Element* rootNode = doc->create_root_node("coaster-data-rows");

  branch_to_xml(rows, rootNode, internal, save_parent_path);

  return doc;
}

xmlpp::Document* Store::to_document(const Gtk::TreeRow& row,
                                    bool internal,
                                    bool save_parent_path) const
{
  using namespace xmlpp;

  Document* doc = new Document();
  Element* rootNode = doc->create_root_node("coaster-data-rows");

  Gtk::TreePath path = get_path(row);

  elem_from_row(rootNode, row, internal, save_parent_path);

  return doc;
}

void Store::from_document(const xmlpp::Document* document)
{
  using namespace xmlpp;
  Element* rootNode = document->get_root_node();

  Gtk::TreeNodeChildren this_children = children();
  branch_from_xml(rootNode, this_children);
}

Data::Properties Store::get_properties() const
{
  return m_Properties;
}

void Store::set_properties(const Data::Properties& prop)
{
  if(m_Properties != prop)
    m_signal_modified.emit(true);

  m_Properties = prop;
}

void Store::xml_to_prop(const xmlpp::Element* node)
{
  m_Properties.title = get_node_value(node, "title");
  m_Properties.append_date = (get_node_value(node, "append_date") == "yes" ? true : false);
  m_Properties.catnum = get_node_value(node, "catnum");
  m_Properties.publisher = get_node_value(node, "publisher");
  m_Properties.author = get_node_value(node, "author");

  m_Properties.isolevel = (get_node_value(node, "isolevel") == "1" ? 
                 COASTER_ISO_LEVEL1 : COASTER_ISO_LEVEL2);
  m_Properties.joliet = (get_node_value(node, "joliet") == "no" ? false : true);
  m_Properties.rockridge = (get_node_value(node, "rockridge") == "no" ? false : true);

  SessionFormat tempFormat;
  Glib::ustring tempStringType = get_node_value(node, "sessionformat");

  if(tempStringType == "cdi")
    tempFormat = COASTER_SESSION_FMT_CDI;
  else if(tempStringType == "cdxa")
    tempFormat = COASTER_SESSION_FMT_CDXA;
  else
    tempFormat = COASTER_SESSION_FMT_CDROM;

  m_Properties.sessionformat = tempFormat;
}

void Store::prop_to_xml(xmlpp::Element* node) const
{
  node->set_attribute("title", m_Properties.title);

  {
    // append date
    Glib::ustring strAppDate = "no";

    if(m_Properties.append_date)
      strAppDate = "yes";

    node->set_attribute("append_date", strAppDate);
  }

  node->set_attribute("catnum", m_Properties.catnum);
  node->set_attribute("publisher", m_Properties.publisher);
  node->set_attribute("author", m_Properties.author);

  {
    // ISO Level
    Glib::ustring strISOLevel = "1";

    if(m_Properties.isolevel == COASTER_ISO_LEVEL2)
      strISOLevel = "2";

    node->set_attribute("isolevel", strISOLevel);
  }

  node->set_attribute("joliet", (m_Properties.joliet ? "yes" : "no"));
  node->set_attribute("rockridge", (m_Properties.rockridge ? "yes" : "no"));

  {
    // Session Format
    Glib::ustring strSession = "cdrom";

    switch(m_Properties.sessionformat)
    {
      case COASTER_SESSION_FMT_CDI:
        strSession = "cdi";
        break;
      case COASTER_SESSION_FMT_CDXA:
        strSession = "cdxa";
        break;
      case COASTER_SESSION_FMT_CDROM:
        strSession = "cdrom";
        break;
    }

    node->set_attribute("sessionformat", strSession);
  }
}

void Store::branch_to_xml(const Gtk::TreeNodeChildren& group,
                          xmlpp::Element* parent,
                          bool internal,
                          bool save_ppath) const
{
  if(!group.empty())
  {
    for(Gtk::TreeModel::iterator iter = group.begin() ; iter != group.end() ; ++iter)
    {
      elem_from_row(parent, (*iter), internal, save_ppath);
    }
  }
}

void Store::branch_to_xml(const type_list_rows& rows,
                          xmlpp::Element* parent,
                          bool internal,
                          bool save_ppath) const
{
  if(!rows.empty())
  {
    for(type_list_rows::const_iterator i = rows.begin() ; i != rows.end() ; ++i)
    {
      elem_from_row(parent, (*i), internal, save_ppath);
    }
  }
}

void Store::move_branch_from_xml(const xmlpp::Element* old_node,
                                 Gtk::TreeNodeChildren& children,
                                 type_list_rows& rows_added)
{
  xmlpp::Node::NodeList node_list = get_child_nodes(old_node);
  
  if(node_list.empty())
    return;

  for(xmlpp::Node::NodeList::iterator i = node_list.begin() ; i != node_list.end() ; ++i)
  {
    xmlpp::Element* current_elem = static_cast<xmlpp::Element*>(*i);
    Gtk::TreeRow new_row = *(append(children));

    row_from_elem(new_row, current_elem);

    rows_added.push_back(new_row);
  }
}

void Store::branch_from_xml(const xmlpp::Element* parent,
                            Gtk::TreeNodeChildren& children)
{
  using namespace xmlpp;

  Node::NodeList node_list = get_child_nodes(parent);
  if(!node_list.empty())
  {
    for(Node::NodeList::reverse_iterator i = node_list.rbegin() ; i != node_list.rend() ; ++i)
    {
      Element* current_elem = static_cast<Element*>(*i);
      if(current_elem->get_name() == "data")
      {
        File file(Glib::filename_to_utf8(get_node_value(current_elem, "path")));

        if(file.is_valid())
        {
          Gtk::TreeRow row = *(append(children));
          RowAttr row_attr(get_node_value(current_elem, "name"),
                           file.get_size(),
                           COASTER_ENTITY_FILE,
                           file.get_uri(),
                           file.get_mime_type());
          set_row_attribs(row, row_attr);

          disc_size_add(file.get_size());
        }
        else
        {
          // warn of a problem
        }
      }
      else if(current_elem->get_name() == "directory")
      {
        Gtk::TreeRow row = *(append(children));
        RowAttr row_attr(get_node_value(current_elem, "name"));
        set_row_attribs(row, row_attr);

        Gtk::TreeNodeChildren new_children = row.children();
        branch_from_xml(current_elem, new_children);
      }
    }
  }
}

void Store::paste_from_string(const Glib::ustring& string)
{
  using namespace xmlpp;
  DomParser parser;
  Document* doc = 0;
  Element* rootNode = 0;

  try
  {
    parser.parse_memory(string);
  }
  catch(const std::exception& ex)
  {
    handle_exception("Data::Store::add_from_string(): Xml Parser Exception: \n", ex);
  }

  doc = parser.get_document();
  rootNode = doc->get_root_node();

  debug_xml("Pasting:\n", doc->write_to_string_formatted());

  Node::NodeList nodeList = get_child_nodes(rootNode);

  type_list_rows rows_added;
  
  if(nodeList.size() > 0)
  {
    for(Node::NodeList::iterator i = nodeList.begin() ; i != nodeList.end() ; ++i)
    {
      Element* current_elem = static_cast<Element*>(*i);

      Gtk::TreeRow row = get_new_entity_row();
      row_from_elem(row, current_elem);
      rows_added.push_back(row);
    }

    push_undo(AddRemOp::create(to_document(rows_added, true, true),
                               _("Paste"),
                               sigc::mem_fun(*this, &Store::remove_from_xml),
                               sigc::mem_fun(*this, &Store::add_back_from_xml)));
  }
}

void Store::remove_from_xml(const xmlpp::Element* parent)
{
  using namespace xmlpp;

  Gnome::Vfs::FileSize size = 0;

  Node::NodeList node_list = get_child_nodes(parent);
  if(!node_list.empty())
  {
    for(Node::NodeList::reverse_iterator i = node_list.rbegin() ; i != node_list.rend() ; ++i)
    {
      Element* current_elem = static_cast<Element*>(*i);
     
      Gtk::TreePath path(current_elem->get_attribute("tpath")->get_value());
      Gtk::TreeRow row = *(get_iter(path));

      size += get_size(row);

      erase(row);
    }
  }

  disc_size_remove(size);
}

void Store::add_back_from_xml(const xmlpp::Element* parent)
{
  using namespace xmlpp;

  xmlpp::Node::NodeList node_list = get_child_nodes(parent);
  if(!node_list.empty())
  {
    for(xmlpp::Node::NodeList::iterator i = node_list.begin() ; i != node_list.end() ; ++i)
    {
      xmlpp::Element *current_elem = static_cast<xmlpp::Element*>(*i);

      Glib::ustring ppath_str = current_elem->get_attribute("ppath")->get_value();

      Gtk::TreeRow new_row;

      // get the new row
      if(ppath_str != "")
      {
        Gtk::TreePath ppath(ppath_str);
        Gtk::TreeRow prow = *(get_iter(ppath));

        new_row = *(append(prow.children()));
      }
      else
        new_row = *(append());
      
      // add the data to the row
      row_from_elem(new_row, current_elem);
    }
  }
}

void Store::add_children_from_xml(const xmlpp::Element* parent,
                                  Gtk::TreeNodeChildren& children)
{
  xmlpp::Node::NodeList node_list = get_child_nodes(parent);
  if(!node_list.empty())
  {
    for(xmlpp::Node::NodeList::iterator i = node_list.begin() ; i != node_list.end() ; ++i)
    {
      xmlpp::Element *current_elem = static_cast<xmlpp::Element*>(*i);

      Gtk::TreeRow new_row = *(append(children));

      row_from_elem(new_row, current_elem);
    }
  }
}

void Store::rename_from_xml(const xmlpp::Element* node,
                            bool redo)
{
  Glib::ustring tpath = get_node_value(node, (redo ? "otpath" : "ntpath"));
  Gtk::TreeRow row = *(get_iter(tpath));
  
  if(!redo) // undo
  {
    Glib::ustring from = node->get_attribute("from")->get_value();
    Data::set_name(row, from);
  }
  else
  {
    Glib::ustring to = node->get_attribute("to")->get_value();
    Data::set_name(row, to);
  }
}

void Store::search_case(const Glib::ustring& search_string)
{
  foreach_iter(sigc::bind(sigc::mem_fun(*this, &Store::on_search_foreach_case),
                          search_string));
}

void Store::search_nocase(const Glib::ustring& search_string)
{
  foreach_iter(sigc::bind(sigc::mem_fun(*this, &Store::on_search_foreach_nocase),
                          search_string));
}

bool Store::on_search_foreach_case(const Gtk::TreeIter& iter,
                                   const Glib::ustring& search_string) const
{
  if(search_string == "")
    return true;

  if(((*iter).get_value(m_Columns.m_col_name)).find(search_string) != Glib::ustring::npos)
    m_refSelection->select(iter);

  return false;
}

bool Store::on_search_foreach_nocase(const Gtk::TreeIter& iter,
                                     const Glib::ustring& search_string) const
{
  if(search_string == "")
    return true;

  Glib::ustring tmp_cell_name = ((*iter).get_value(m_Columns.m_col_name)).lowercase();

  if(tmp_cell_name.find(search_string.lowercase()) != Glib::ustring::npos)
    m_refSelection->select(iter);

  return false;
}

void Store::on_row_changed(const Gtk::TreePath& path,
                           const Gtk::TreeIter& iter)
{
  m_signal_modified.emit(true);
  TreeStore::on_row_changed(path, iter);
}

void Store::on_row_deleted(const Gtk::TreePath& path)
{
  m_signal_modified.emit(true);
  TreeStore::on_row_deleted(path);
}

void Store::on_row_inserted(const Gtk::TreePath& path,
                            const Gtk::TreeIter& iter)
{
  m_signal_modified.emit(true);
  TreeStore::on_row_inserted(path, iter);
}

void Store::on_rows_reordered(const Gtk::TreePath& path,
                              const Gtk::TreeIter& iter,
                              int* new_order)
{
  m_signal_modified.emit(true);
  TreeStore::on_rows_reordered(path, iter, new_order);
}

bool Store::row_drop_possible_vfunc(const Gtk::TreeModel::Path& dest,
                                    const Gtk::SelectionData& selection_data) const
{
  Gtk::TreeModel::Path dest_parent = dest;
  bool dest_is_not_top_level = dest_parent.up();

  if(!dest_is_not_top_level || dest_parent.empty())
  {
    //The user wants to move something to the top-level.
    //Let's always allow that.
    return true;
  }
  else
  {
    //Get an iterator for the row at this path:
    //We must unconst this. This should not be necessary with a future version of gtkmm.
    Store* unconstThis = const_cast<Store*>(this);
    const_iterator iter_dest_parent = unconstThis->get_iter(dest_parent);
    //const_iterator iter_dest_parent = get_iter(dest);
    if(iter_dest_parent)
    {
      if(Data::is_tree_dir(iter_dest_parent))
        return true;
      else
        return false;
    }
  }

  return Gtk::TreeStore::row_drop_possible_vfunc(dest, selection_data);
}

void Store::on_icon_theme_changed()
{
  foreach_iter(sigc::mem_fun(*this, &Store::icon_theme_changed_foreach));
}

bool Store::icon_theme_changed_foreach(const Gtk::TreeModel::iterator& iter)
{
  (*iter)[m_Columns.m_col_ref_pixbuf] = 
    IconCache::lookup_icon((*iter).get_value(m_Columns.m_col_mime_type));
  return false;
}

type_list_rows Store::remove_duplicate_rows() const
{
  type_list_paths selected_paths = m_refSelection->get_selected_rows();
  type_list_rows return_rows;

  if(selected_paths.size() > 0)
  {
    type_list_paths check_paths = selected_paths;

    for(type_list_paths::const_iterator i = selected_paths.begin() ;
        i != selected_paths.end() ; 
        ++i)
    {
      bool add = true;
      for(type_list_paths::iterator j = check_paths.begin() ; j != check_paths.end() ; ++j)
      {
        if((*i) == (*j))
          continue;

        if((*i).is_descendant((*j)))
        {
          add = false;
          break;
        }
      }

      if(add)
        // TODO: fix gtkmm to have a const get_iter() method
        return_rows.push_back(*(const_cast<Store*>(this)->get_iter(*i)));
    }
  }

  return return_rows;
}

Gtk::TreeRow Store::get_new_entity_row()
{
  Gtk::TreeRow new_row;
  type_list_paths selected_paths = m_refSelection->get_selected_rows();

  if(selected_paths.size() > 0)
  {
    Gtk::TreeRow selected_row = *(get_iter(selected_paths.back()));
    if(Data::is_tree_dir(selected_row))
      new_row = *(append(selected_row.children()));
    else
      new_row = *(append());
  }
  else
  {
    new_row = *(append());
  }

  return new_row;
}

Gtk::TreeRow Store::get_new_entity_row(const Gtk::TreeRow& folder_row)
{
  return *append(folder_row.children());
}

bool Store::add_to_folder_from_dir(const Gtk::TreeRow& parent_row,
                                   const Glib::ustring& folder,
                                   bool hidden,
                                   bool recursive,
                                   bool follow_links,
                                   bool& changed,
                                   const type_slot_report_file& slot_report)
{
  bool file_exists = true;
  
  Gnome::Vfs::DirectoryHandle dhandle;
  try
  {
    if(follow_links)
      dhandle.open(folder, COMMON_INFO_OPTIONS);
    else
      dhandle.open(folder, COMMON_INFO_OPTIONS_NO_LINKS);
  }
  catch(const Gnome::Vfs::exception& e)
  {
    //TODO: report this to the user!
    return false;
  }

  Glib::RefPtr<Gnome::Vfs::FileInfo> info = dhandle.read_next(file_exists);
  while(file_exists)
  {
    if(!((info->get_permissions() & Gnome::Vfs::PERM_USER_READ) ||
         (info->get_permissions() & Gnome::Vfs::PERM_GROUP_READ) ||
         (info->get_permissions() & Gnome::Vfs::PERM_OTHER_READ)))
    {
      info = dhandle.read_next(file_exists);
      continue; //TODO: report this to the user!
    }
    
    Glib::ustring new_row_name = info->get_name();
    if(hidden == false && new_row_name.substr(0,1) == ".")
    {
      info = dhandle.read_next(file_exists);
      continue;
    }

    if(!slot_report(Gnome::Vfs::unescape_string(folder + "/" + new_row_name)))
      return false;

    Glib::ustring mime_type = info->get_mime_type();
    if(mime_type != "x-directory/normal")
    {
      debug("Adding: ", new_row_name);
      Gtk::TreeRow child_row = get_new_entity_row(parent_row);

      RowAttr child_attr(new_row_name, info->get_size(), COASTER_ENTITY_FILE,
                         folder + "/" + new_row_name,
                         mime_type);

      set_row_attribs(child_row, child_attr);
      disc_size_add(info->get_size());

      if(!changed)
        changed = true;
    }
    else
    {
      if(recursive)
      {
        if(new_row_name != "." && new_row_name != "..")
        {
          debug("Adding: ", new_row_name);
          Gtk::TreeRow new_folder_row = get_new_entity_row(parent_row);

          RowAttr row_attr(new_row_name);
          set_row_attribs(new_folder_row, row_attr);

          if(!changed)
            changed = true;

          //Gtk::TreeRowReference new_folder_rowref(Glib::RefPtr<Store>(this), new_folder_path);
          if(!add_to_folder_from_dir(new_folder_row,
                                     folder + "/" + new_row_name,
                                     hidden,
                                     recursive,
                                     follow_links,
                                     changed,
                                     slot_report))
            return false;
        }
      }
    }

    info = dhandle.read_next(file_exists);
  }

  return true;
}

void Store::set_row_attribs(Gtk::TreeRow& row,
                            const RowAttr& row_attribs)
{
  row[m_Columns.m_col_name] = row_attribs.name;
  row[m_Columns.m_col_mime_type] = row_attribs.mime;
  row[m_Columns.m_col_size] = row_attribs.size;
  row[m_Columns.m_col_entity_type] = row_attribs.entity_type;
  row[m_Columns.m_col_uri] = row_attribs.uri;
  row[m_Columns.m_col_ref_pixbuf] = IconCache::lookup_icon(row_attribs.mime);
}

void Store::elem_from_row(xmlpp::Element* parent,
                          const Gtk::TreeRow& row,
                          bool internal,
                          bool save_ppath) const
{
  Gtk::TreePath path;
  if(internal)
  {
    path = get_path(row);
  }

  if(!Data::is_tree_dir(row))
  {
    xmlpp::Element* new_elem = parent->add_child("data");
    new_elem->set_attribute("name", row.get_value(m_Columns.m_col_name));
    new_elem->set_attribute("path", row.get_value(m_Columns.m_col_uri));
    if(internal)
    {
      new_elem->set_attribute("tpath", path.to_string());
      new_elem->set_attribute("size", to_string(row.get_value(m_Columns.m_col_size)));
      new_elem->set_attribute("mime", row.get_value(m_Columns.m_col_mime_type));

      if(save_ppath)
      {
        path.up();
        new_elem->set_attribute("ppath", path.to_string());
      }
    }
  }
  else
  {
    xmlpp::Element* new_elem = parent->add_child("directory");
    new_elem->set_attribute("name", row.get_value(m_Columns.m_col_name));
    
    if(internal)
    {
      new_elem->set_attribute("tpath", path.to_string());
      
      if(save_ppath)
      {
        path.up();
        new_elem->set_attribute("ppath", path.to_string());
      }
    }

    branch_to_xml(row.children(), new_elem, internal);
  }
}

void Store::row_from_elem(Gtk::TreeRow& row,
                          const xmlpp::Element* elem)
{
  if(elem->get_name() == "data")
  {
    Gnome::Vfs::FileSize elem_size = to_size(get_node_value(elem, "size"));

    RowAttr row_attr(get_node_value(elem, "name"),
                     elem_size,
                     COASTER_ENTITY_FILE,
                     get_node_value(elem, "path"),
                     get_node_value(elem, "mime"));
    set_row_attribs(row, row_attr);

    disc_size_add(elem_size);
  }
  else if(elem->get_name() == "directory")
  {
    RowAttr row_attr(get_node_value(elem, "name"));
    set_row_attribs(row, row_attr);

    Gtk::TreeNodeChildren new_children = row.children();
    add_children_from_xml(elem, new_children);
  }
}

xmlpp::Node::NodeList Store::get_child_nodes(const xmlpp::Element* elem) const
{
  using namespace xmlpp;

  Node::NodeList nl_data = elem->get_children("data");
  Node::NodeList nl_dir  = elem->get_children("directory");

  nl_data.merge(nl_dir);
  
  return nl_data;
}

} // namespace Data

} // namespace Coaster
