/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "data/data-layout.h"

#include "cstr-intl.h"
#include "cstr-debug.h"

#include "string-utils.h"
#include "ucompose.h"

#include "data/data-store.h"
#include "data/data-utils.h"

#include "dialogs/add-progress.h"

#include "widgets/busy-cursor.h"
#include "widgets/cellrenderer-uri.h"
#include "widgets/treecolumn-name.h"

#include <gdk/gdkkeysyms.h>
#include <gtk/gtkdnd.h>

#include <libgnomevfsmm/utils.h>

#include <vector>

namespace Coaster
{

namespace Data
{

//////////////////////////////////////////////////////////////////////////////////////////
//  Coaster::Data::Layout                                                               //
//////////////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
Layout::Layout(const Glib::RefPtr<Store>& refStore,
               const sigc::slot<void,guint,guint32>& slot_popup,
               const sigc::slot<void,const Gnome::Vfs::FileSize&>& slot_disc_size_add,
               const sigc::slot<void,const Gnome::Vfs::FileSize&>& slot_disc_size_remove)
: Layout_Base(), popup(slot_popup), disc_size_add(slot_disc_size_add),
  disc_size_remove(slot_disc_size_remove)
{
  m_refStore = refStore;
  m_refStore->set_tree_selection(m_refSelection);
  set_model(m_refStore);

  init();

  m_refStore->set_expand_funcs(sigc::mem_fun(*this, &Layout::expand_row),
                               sigc::mem_fun(*this, &Layout::expand_to_path));
  show_all_children();
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
Layout::~Layout()
{
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
void Layout::rename_selected()
{
  type_list_paths list_paths = m_refSelection->get_selected_rows();
  if(list_paths.size() == 1) // only do this if there is only one row selected
  {
    m_colName->property_editable() = true;
    set_cursor(*(list_paths.begin()), *m_colName, true);
  }
}

//////////////////////////////////////////////////////////////////////////////////////////
//  Coaster::Layout -- protected                                               //
//////////////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
void Layout::init_columns()
{
  using namespace Gtk;

  const Data::Columns& model_columns = Data::columns();

  {
    m_colName = new Widgets::TreeColumnName();
    m_colName->set_sort_column(model_columns.m_col_name);

    append_column(*manage(m_colName));

    m_colName->signal_edited().connect(sigc::mem_fun(*this, &Layout::on_text_edited));
    m_colName->signal_editing_canceled().connect(sigc::mem_fun(*this, &Layout::on_text_editing_canceled));

    m_colName->add_attribute(model_columns.m_col_name);
    m_colName->add_attribute(model_columns.m_col_ref_pixbuf);
  }

  {
    m_colSize = new Column(_("Size"));
    append_column(*manage(m_colSize));

    // sorting
    m_colSize->set_sort_column(model_columns.m_col_size);

    CellRendererText *const cell_size = new CellRendererText();
    m_colSize->pack_start(*manage(cell_size));
    cell_size->property_xalign() = 1.0;

    m_colSize->add_attribute(cell_size->property_text(), model_columns.m_col_size);
    m_colSize->set_cell_data_func(*cell_size, sigc::mem_fun(*this, &Layout::size_cell_data_func));
    m_colSize->set_sizing(Gtk::TREE_VIEW_COLUMN_AUTOSIZE);
    m_colSize->set_resizable(true);
  }

  {
    m_colUri = new Column(_("Location"));
    append_column(*manage(m_colUri));

    // sorting
    m_colUri->set_sort_column(model_columns.m_col_uri);

    Widgets::CellRendererUri *const cell_uri = new Widgets::CellRendererUri();
    m_colUri->pack_start(*manage(cell_uri));

    m_colUri->add_attribute(cell_uri->property_text(), model_columns.m_col_uri);
    m_colUri->set_sizing(Gtk::TREE_VIEW_COLUMN_AUTOSIZE);
  }
  
  set_headers_clickable(true);
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
void Layout::init_drag_n_drop()
{
  // Drag and drop
  std::vector<Gtk::TargetEntry> target_list_source;
  std::vector<Gtk::TargetEntry> target_list_dest;
  
  target_list_source.push_back(Gtk::TargetEntry("application/x-coaster/data-row"));

  target_list_dest.push_back(Gtk::TargetEntry("application/x-coaster/data-row"));
  target_list_dest.push_back(Gtk::TargetEntry("text/uri-list"));

  enable_model_drag_source(target_list_source, Gdk::MODIFIER_MASK, Gdk::ACTION_COPY|Gdk::ACTION_MOVE);
  enable_model_drag_dest(target_list_dest, Gdk::ACTION_COPY|Gdk::ACTION_MOVE);
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
bool Layout::on_button_press_event(GdkEventButton* event)
{
  using namespace Gtk;

  if(event->type == GDK_BUTTON_PRESS)
  {
    TreePath path;
    TreeViewColumn *column;
    int cell_x, cell_y;

    bool over_row = get_path_at_pos(static_cast<int>(event->x), static_cast<int>(event->y),
                                    path, column, cell_x, cell_y);

    if(event->button == 3)
    {
      if(!over_row)
        m_refSelection->unselect_all();
      else
      {
        if(m_refSelection->count_selected_rows() == 0)
          m_refSelection->select(path);
        else if(!m_refSelection->is_selected(path))
        {
          m_refSelection->unselect_all();
          m_refSelection->select(path);
        }
      }

      popup(event->button, event->time);

      return true;
    }
    else if(event->button == 1)
    {
      if(!over_row)
        m_refSelection->unselect_all();
      return TreeView::on_button_press_event(event);
    }
  }
  return TreeView::on_button_press_event(event);
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
bool Layout::on_key_press_event(GdkEventKey* event)
{
  if(m_refSelection->count_selected_rows() == 1)
  {
    if(event->type == GDK_KEY_PRESS)
    {
      if(event->keyval == GDK_Return)
      {
        Gtk::TreePath path = *(m_refSelection->get_selected_rows().begin());
        Gtk::TreeRow row = *(m_refStore->get_iter(path));
        if(!row.children().empty())
        {
          if(row_expanded(path))
            collapse_row(path);
          else
            expand_row(path, false);

          return true;
        }
      }
    }
  }
  
  return TreeView::on_key_press_event(event);
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
void Layout::on_drag_begin(const Glib::RefPtr<Gdk::DragContext>& context)
{
  debug_dnd("Layout::on_drag_begin");
  TreeView::on_drag_begin(context);
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
void Layout::on_drag_leave(const Glib::RefPtr<Gdk::DragContext>& context,
                           guint time)
{
  debug_dnd("Layout::on_drag_leave");
  TreeView::on_drag_leave(context, time);
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
bool Layout::on_drag_drop(const Glib::RefPtr<Gdk::DragContext>& context,
                          int x,
                          int y,
                          guint time)
{
  debug_dnd("Layout::on_drag_drop");

  // if we're dropping on the same widget the drag is coming from
  if(this == Gtk::Widget::drag_get_source_widget(context))
  {
    Gtk::TreePath drop_path;
    Gtk::TreeViewDropPosition drop_pos;
    
    bool over_row = get_dest_row_at_pos(x, y, drop_path, drop_pos);

    Gtk::TreePath orig_path = *(m_refSelection->get_selected_rows().begin());

    if(over_row)
    {
      // if the drop path is the same as the source path
      if(orig_path == drop_path)
      {
        debug_dnd("drop false");
        context->drop_reply(false, time);
        return false;
      }

      Gtk::TreeRow drop_row = *(m_refStore->get_iter(drop_path));
      if(Data::is_tree_dir(drop_row))
      {
        // if we're dropping into the directory
        if(drop_pos == Gtk::TREE_VIEW_DROP_INTO_OR_BEFORE ||
           drop_pos == Gtk::TREE_VIEW_DROP_INTO_OR_AFTER)
        {
          Gtk::TreePath orig_path_parent = orig_path;
          orig_path_parent.up();

          // if we're dropping into the same parent of the source
          if(!orig_path_parent.empty() && (orig_path_parent == drop_path))
          {
            debug_dnd("drop false");
            context->drop_reply(false, time);
            return false;
          }
          
          Gtk::TreePath drop_path_parent = drop_path;
          drop_path_parent.up();

          // if we're dropping into itself
          if(!drop_path_parent.empty() && orig_path == drop_path_parent)
          {
            debug_dnd("drop false");
            context->drop_reply(false, time);
            return false;
          }
        }
        else // if we're dropping before or after the directory
        {
          Gtk::TreePath orig_path_parent = orig_path;
          Gtk::TreePath drop_path_parent = drop_path;
          
          orig_path_parent.up();
          drop_path_parent.up();

          // if we're dropping into the same directory the source is in
          if(orig_path_parent.to_string() == drop_path_parent.to_string())
          {
            debug_dnd("drop false");
            context->drop_reply(false, time);
            return false;
          }
        }
      }
      else // if we're dropping on a file
      {
        if(orig_path.is_ancestor(drop_path))
        {
          debug_dnd("drop false");
          context->drop_reply(false, time);
          return false;
        }
    
        Gtk::TreePath orig_parent_path = orig_path;
        Gtk::TreePath drop_parent_path = drop_path;
        
        orig_parent_path.up();
        drop_parent_path.up();

        // if we're dropping into the same directory the source is in
        if(orig_parent_path.to_string() == drop_parent_path.to_string())
        {
          debug_dnd("drop false");
          context->drop_reply(false, time);
          return false;
        }
      }
    }
    else // if we're dropping into a blank area
    {
      Gtk::TreePath orig_parent_path = orig_path;
      orig_parent_path.up();

      if(orig_parent_path.empty()) // if the original row is already at the toplevel
      {
        debug_dnd("drop false");
        context->drop_reply(false, time);
        return false;
      }
    }
  }

  debug_dnd("drop true");
  return TreeView::on_drag_drop(context, x, y, time);
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
void Layout::on_drag_data_get(const Glib::RefPtr<Gdk::DragContext>& context,
                              Gtk::SelectionData& selection_data,
                              guint info,
                              guint time)
{
  debug_dnd("Layout::on_drag_data_get");
  debug_dnd("Get target: ", selection_data.get_target());

  Glib::ustring string = m_refStore->get_selected_rows("coaster-drag-rows");

  selection_data.set(selection_data.get_target(), 8, (const guchar*)string.c_str(), string.bytes());

  TreeView::on_drag_data_get(context, selection_data, info, time);
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
void Layout::on_drag_data_received(const Glib::RefPtr<Gdk::DragContext>& context,
                                   int x,
                                   int y,
                                   const Gtk::SelectionData& selection_data,
                                   guint info,
                                   guint time)
{
  debug_dnd("Layout::on_drag_data_received");
  debug_dnd("Recieved target: ", selection_data.get_target());


  Gtk::TreeRow drop_parent;
  
  if(selection_data.get_target() == "application/x-coaster/data-row")
  {
    Gtk::TreePath drop_path;
    Gtk::TreeViewDropPosition drop_pos;

    bool row_exists = get_dest_row_at_pos(x, y, drop_path, drop_pos);
    Gtk::TreeNodeChildren drop_children = get_drop_children(row_exists, drop_path, drop_pos);

    Glib::ustring recieved_string = selection_data.get_data_as_string();
    debug_dnd("Recieved XML: ", recieved_string);

    Gdk::DragAction action = context->get_action();

    if(this == Gtk::Widget::drag_get_source_widget(context))
    {
      if(action == Gdk::ACTION_MOVE)
      {
        debug_dnd("Moving stuff");
        
        bool result = m_refStore->dnd_move(recieved_string,
                                           drop_children);
        if(result)
        {
          // warn the user of a problem
        }
        context->drag_finish(result, false, time);

        return;
      }
      else
      {
        debug_dnd("Copying stuff within an instance.");

        m_refStore->dnd_copy(recieved_string,
                             drop_children,
                             true);
        context->drag_finish(true, false, time);

        return;
      }
    }
    else
    {
      debug_dnd("Copying stuff between two instances.");

      m_refStore->dnd_copy(recieved_string,
                           drop_children,
                           false);
      context->drag_finish(true, false, time);

      return;
    }

    return;
  }
  else if(selection_data.get_target() == "text/uri-list")
  {
    debug_dnd("Recieved URIs: ", selection_data.get_data_as_string());
   
    type_list_ustrings uri_list = 
      String::uri_list_extract_uris(selection_data.get_data_as_string());

    Dialogs::AddProgress pAP(get_win(), _("Files"));
    Widgets::BusyCursor cursor(get_win());
    // TODO: figure out how to handle dragging a directory to Coaster
    m_refStore->add_files(uri_list,
                          true,
                          sigc::mem_fun(pAP, &Dialogs::AddProgress::show_filename));

    context->drag_finish(true, false, time);
    return;
  }

  TreeView::on_drag_data_received(context, x, y, selection_data, info, time);
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
void Layout::on_drag_data_delete(const Glib::RefPtr<Gdk::DragContext>& context)
{
  debug_dnd("Layout::on_drag_data_delete");    
  TreeView::on_drag_data_delete(context);
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
void Layout::on_drag_end(const Glib::RefPtr<Gdk::DragContext>& context)
{
  debug_dnd("Layout::on_drag_end");
  TreeView::on_drag_end(context);
}


//////////////////////////////////////////////////////////////////////////////////////////
//  Coaster::Layout -- private                                                 //
//////////////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
void Layout::size_cell_data_func(Gtk::CellRenderer* cell,
                                 const Gtk::TreeModel::iterator& iter)
{
  using namespace Gtk;

  CellRendererText& renderer = dynamic_cast<CellRendererText&>(*cell);
  if(Data::is_tree_dir(iter))
  {
    int num_children = (*iter).children().size();
    renderer.property_text() = String::ucompose((num_children == 1 ? "%1 item" : "%1 items"), num_children);
  }
  else
  {
    const Data::Columns& model_columns = Data::columns();
    
    Gnome::Vfs::FileSize cell_file_size = (*iter).get_value(model_columns.m_col_size);
    renderer.property_text() = Gnome::Vfs::format_file_size_for_display(cell_file_size);
  } 
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
void Layout::do_nothing()
{
  debug("This does nothing!");
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
void Layout::on_text_edited(const Glib::ustring& path_text,
                            const Glib::ustring& new_text)
{
  m_refStore->rename(path_text, new_text);
  m_colName->property_editable() = false;
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
void Layout::on_text_editing_canceled()
{
  m_colName->property_editable() = false;
}

//////////////////////////////////////////////
// Function: //
// Purpose: //
// Input: //
// Output: //
//////////////////////////////////////////////
Gtk::TreeNodeChildren Layout::get_drop_children(bool exists,
                                                const Gtk::TreePath& path,
                                                const Gtk::TreeViewDropPosition& pos)
{
  if(exists)
  {
    Gtk::TreeRow drop_row = *(m_refStore->get_iter(path));
    if(Data::is_tree_dir(drop_row))
    {
      if(pos == Gtk::TREE_VIEW_DROP_INTO_OR_BEFORE ||
         pos == Gtk::TREE_VIEW_DROP_INTO_OR_AFTER)
      {
        return drop_row.children();
      }
      else
      {
        return (*(drop_row.parent())).children();
      }
    }
    else
    {
      return (*(drop_row.parent())).children();
    }
  }

  return m_refStore->children();
}

} // namespace Data

} // namespace Coaster
