/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "data/data-disc-info.h"

#include "cstr-intl.h"
#include "cstr-debug.h"

#include "ucompose.h"
#include "conf/conf-utils.h"

#include "widgets/combo-disc-size.h"
#include "widgets/disc-progress.h"

#include <gtkmm/label.h>
#include <gtkmm/progressbar.h>
#include <gtkmm/comboboxtext.h>

#include <libglademm/xml.h>
#include <libgnomevfsmm/utils.h>

#include <sstream>


namespace
{

void change_prog_text(const sigc::slot<void,const Glib::ustring&>& set_text,
                      Gnome::Vfs::FileSize used,
                      Gnome::Vfs::FileSize avail)
{
  set_text(Gnome::Vfs::format_file_size_for_display(used) + " / " + Gnome::Vfs::format_file_size_for_display(avail));
  //prog->set_text(Gnome::Vfs::format_file_size_for_display(avail - used));
}
  
} // anonymous namespace

namespace Coaster
{

namespace
{

class SizeChgOp : public UndoOp
{
public:
  typedef SharedPtr<SizeChgOp> SizeChgOpPtr;
  
  static SizeChgOpPtr create(DiscSizeType old_size,
                             DiscSizeType new_size,
                             const sigc::slot<void,DiscSizeType,bool>& size_chg_slot);
  virtual ~SizeChgOp();

  void undo();
  void redo();
protected:
  SizeChgOp(DiscSizeType old_size,
            DiscSizeType new_size,
            const sigc::slot<void,DiscSizeType,bool>& size_chg_slot);

private:
  DiscSizeType m_old;
  DiscSizeType m_new;
  sigc::slot<void,DiscSizeType,bool> set_disc_size;
};

typedef SharedPtr<SizeChgOp>  SizeChgOpPtr;

SizeChgOp::SizeChgOp(DiscSizeType old_size,
                     DiscSizeType new_size,
                     const sigc::slot<void,DiscSizeType,bool>& size_chg_slot)
: UndoOp(_("Disc Size Change")),
  m_old(old_size), m_new(new_size),
  set_disc_size(size_chg_slot)
{}

SizeChgOp::~SizeChgOp()
{}

SizeChgOpPtr SizeChgOp::create(DiscSizeType old_size,
                               DiscSizeType new_size,
                               const sigc::slot<void,DiscSizeType,bool>& size_chg_slot)
{
  return SizeChgOpPtr(new SizeChgOp(old_size, new_size, size_chg_slot));
}

void SizeChgOp::undo()
{
  set_disc_size(m_old,false);
}

void SizeChgOp::redo()
{
  set_disc_size(m_new,false);
}

} // anonymous namespace

namespace Data
{

//////////////////////////////////////////////////////////////////////////////////////////
//  Coaster::Data::DiscInfo                                                               //
//////////////////////////////////////////////////////////////////////////////////////////

DiscInfo::DiscInfo(const sigc::slot<void,const UndoOpPtr&>& slot_push_undo)
: Widgets::DiscInfo_Base(), push_undo(slot_push_undo),
  m_disc_size(0), m_size_used(0), m_bOver(false)
{
  using namespace Gtk;
  
  m_pComboBox->init_data();
  m_pComboBox->signal_changed().connect(sigc::mem_fun(*this, &DiscInfo::on_combo_box_changed));
  m_pComboBox->set_size(Conf::get_int("layouts/default_size"));


  change_prog_text(sigc::mem_fun(*m_pProgress, &Widgets::DiscProgress::set_text),
                   m_size_used, m_disc_size);
}

DiscInfo::~DiscInfo()
{}

void DiscInfo::disc_size_add(Gnome::Vfs::FileSize size)
{
  m_size_used += size;
  update_progress_bar();
}

void DiscInfo::disc_size_remove(Gnome::Vfs::FileSize size)
{
  m_size_used -= size;
  update_progress_bar();
}

DiscInfo::type_signal_disc_size_changed& DiscInfo::signal_disc_size_changed()
{
  return signal_disc_size_changed_;
}

Gnome::Vfs::FileSize DiscInfo::get_size_used() const
{
  return m_size_used;
}

DiscSizeType DiscInfo::get_disc_size() const
{
  int active_row = m_pComboBox->get_active_row_number();

  switch(active_row)
  {
    case 0:
      return COASTER_DISC_SIZE_185;
    case 1:
    default:
      return COASTER_DISC_SIZE_650;
    case 2:
      return COASTER_DISC_SIZE_700;
    case 3:
      return COASTER_DISC_SIZE_750;
    case 4:
      return COASTER_DISC_SIZE_800;
    case 5:
      return COASTER_DISC_SIZE_DVD5;
    case 6:
      return COASTER_DISC_SIZE_DVD9;
  }
}

void DiscInfo::set_disc_size(DiscSizeType size, bool create_undo_op)
{
  /*if(create_undo_op && size != COASTER_DISC_SIZE_NONE)
  {
    DiscSizeType old_size = get_disc_size();
    push_undo(SizeChgOp::create(old_size,
                                size,
                                sigc::mem_fun(*this, &DiscInfo::set_disc_size)));
  }*/

  /*debug(String::ucompose("DiscInfo: set_disc_size -> %1", size));
  switch(size)
  {
    case COASTER_DISC_SIZE_NONE:
      m_pComboBox->set_active(Conf::get_int("layouts/default_size"));
      return;
    case COASTER_DISC_SIZE_650:
      m_pComboBox->set_active(0);
      return;
    case COASTER_DISC_SIZE_700:
      m_pComboBox->set_active(1);
      return;
  }*/
}

//////////////////////////////////////////////////////////////////////////////////////////
//  Coaster::DiscInfo -- protected                                                   //
//////////////////////////////////////////////////////////////////////////////////////////

void DiscInfo::on_combo_box_changed()
{
  m_disc_size = m_pComboBox->get_byte_size();
  update_progress_bar();
  signal_disc_size_changed().emit(m_pComboBox->get_enum_size());
  debug("Data::DiscInfo::on_combo_box_changed(): changed to ", m_pComboBox->get_label_size());
}

void DiscInfo::update_progress_bar()
{
  m_pProgress->set_fraction(((long double)m_size_used) / ((long double)m_disc_size));
  change_prog_text(sigc::mem_fun(*m_pProgress, &Widgets::DiscProgress::set_text),
                   m_size_used, m_disc_size);
}

//////////////////////////////////////////////////////////////////////////////////////////
//  Coaster::DiscInfo -- private                                                     //
//////////////////////////////////////////////////////////////////////////////////////////

} // namespace Data

} // namespace Coaster
