/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "association.h"

#include "ncbmm/drive-selection.h"

#include <gtkmm/togglebutton.h>
#include <gtkmm/entry.h>
#include <gtkmm/spinbutton.h>
#include <gtkmm/combobox.h>
#include <gtkmm/filechooserbutton.h>

#include <gconfmm/client.h>

namespace Coaster
{

namespace Conf
{

//----------------------------------------------------------------------
// Association<T> specializations. These provide widget/key
// association behaviors that are specific to individual widget types.
//----------------------------------------------------------------------

// ---------------------------------------------------------
// SpinButton Association
void Association<Gtk::SpinButton>::connect_widget(Callback widget_changed)
{
  m_widget.signal_value_changed().connect(widget_changed);
}

void Association<Gtk::SpinButton>::load_widget()
{
  int val = get_conf_client()->get_int(get_key());
  if (m_widget.get_value() != val)
    m_widget.set_value(val);
}

void Association<Gtk::SpinButton>::save_widget()
{
  int val = static_cast<int>(m_widget.get_value());
  if (get_conf_client()->get_int(get_key()) != val)
    get_conf_client()->set(get_key(), val);
}

// ---------------------------------------------------------
// ComboBox Association
void Association<Gtk::ComboBox>::connect_widget(Callback widget_changed)
{
  m_widget.signal_changed().connect(widget_changed);
}

void Association<Gtk::ComboBox>::load_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  int val = get_conf_client()->get_int(get_key());
  if (m_widget.get_active_row_number() != val)
    m_widget.set_active(val);
}

void Association<Gtk::ComboBox>::save_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  int val = m_widget.get_active_row_number();
  if (get_conf_client()->get_int(get_key()) != val)
    get_conf_client()->set(get_key(), val);
}

// ---------------------------------------------------------
// Nautilus::Burn::DriveSelection Association
void Association<Nautilus::Burn::DriveSelection>::connect_widget(Callback widget_changed)
{
  m_widget.signal_device_changed().connect(sigc::hide(widget_changed));
}

void Association<Nautilus::Burn::DriveSelection>::load_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  Glib::ustring val = get_conf_client()->get_string(get_key());
  if (m_widget.get_device() != val)
    m_widget.set_device(val);
}

void Association<Nautilus::Burn::DriveSelection>::save_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  Glib::ustring val = m_widget.get_device();
  if (get_conf_client()->get_string(get_key()) != val)
    get_conf_client()->set(get_key(), val);
}

//----------------------------------------------------------------------
// Entry Association
void Association<Gtk::Entry>::connect_widget(Callback widget_changed)
{
  m_widget.signal_changed().connect(widget_changed);
}

void Association<Gtk::Entry>::load_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  Glib::ustring val = get_conf_client()->get_string(get_key());
  if (m_widget.get_text() != val)
    m_widget.set_text(val);
}

void Association<Gtk::Entry>::save_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  Glib::ustring val = m_widget.get_text();
  if (get_conf_client()->get_string(get_key()) != val)
    get_conf_client()->set(get_key(), val);
}

//----------------------------------------------------------------------
// ToggleButton Association
void Association<Gtk::ToggleButton>::connect_widget(Callback widget_changed)
{
  m_widget.signal_toggled().connect(widget_changed);
}

void Association<Gtk::ToggleButton>::load_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  bool val = get_conf_client()->get_bool(get_key());
  if (m_widget.get_active() != val)
    m_widget.set_active(val);
}

void Association<Gtk::ToggleButton>::save_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  bool val = m_widget.get_active();
  if (get_conf_client()->get_bool(get_key()) != val)
    get_conf_client()->set(get_key(), val);
}

//----------------------------------------------------------------------
// ChooserButton Association
void Association<Gtk::FileChooserButton>::connect_widget(Callback widget_changed)
{
  m_widget.signal_selection_changed().connect(widget_changed);
}

void Association<Gtk::FileChooserButton>::load_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  Glib::ustring val = get_conf_client()->get_string(get_key());
  if (Glib::filename_to_utf8(m_widget.get_filename()) != val)
    m_widget.set_filename(val);
}

void Association<Gtk::FileChooserButton>::save_widget()
{
  // Only set it if it has changed (avoids excess notifications).
  Glib::ustring val = Glib::filename_to_utf8(m_widget.get_filename());
  if (get_conf_client()->get_string(get_key()) != val)
    get_conf_client()->set(get_key(), val);
}

} // namespace Conf

} // namespace Coaster
