/* Copyright (C) 2002-2005  The Coaster Development Team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "cd-io.h"

#include "backends/nautilus-make-iso.h"

#include "cstr-intl.h"
#include "cstr-debug.h"
#include "status-icon.h"

#include "data/data-columns.h"
#include "dialogs/glade-dialog.h"
#include "dialogs/progress.h"

#include "ucompose.h"
#include "instance-manager.h"
#include "nautilus-burn-recorder.h"

#include <config.h>

#include <gtkmm/stock.h>
#include <gconfmm/client.h>

#include <libgnomevfsmm/handle.h>
#include <signal.h>

namespace Coaster
{
  
namespace IO
{

BurnDisc::BurnDisc(const Gtk::Window& parent_win,
                   const NautilusBurnDrive* rec,
                   bool eject_cd,
                   int speed,
                   bool dummy,
                   const Glib::ustring& graft_file_contents,
                   const Data::Properties& dp)
: instmgr(InstanceMgr::instance()), m_parent_win(const_cast<Gtk::Window*>(&parent_win)),
  m_pDrive(const_cast<NautilusBurnDrive*>(rec)), m_bEject_CD(eject_cd),
  m_dSpeed(speed), m_bDummy(dummy), m_graft_file_contents(graft_file_contents),
  m_DataProperties(dp)
{
  init();
}

BurnDisc::BurnDisc(const Gtk::Window& parent_win,
                   const NautilusBurnDrive* rec,
                   bool eject_cd,
                   int speed,
                   bool dummy,
                   const std::string& filename)
: instmgr(InstanceMgr::instance()), m_parent_win(const_cast<Gtk::Window*>(&parent_win)),
  m_pDrive(const_cast<NautilusBurnDrive*>(rec)), m_bEject_CD(eject_cd),
  m_dSpeed(speed), m_bDummy(dummy), m_graft_file_contents("")
{
  m_filename = filename;
  init();
}

BurnDisc::~BurnDisc()
{
  const std::auto_ptr<Dialogs::Progress> dialog = m_pBurnProgress;
}

void BurnDisc::init()
{
  m_progress_text = _("Initializing...");
  m_pBurnProgress = Dialogs::Progress::create(*m_parent_win,
    _("Burning Progress"),
    _("Burning files to disc progress"),
    _("Files in your layout are to be written to a CD or DVD.  This operation may take a long time, depending on data size and write speed."),
    m_progress_text);

  Gtk::Button* cancel = m_pBurnProgress->add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  cancel->signal_clicked().connect(sigc::mem_fun(*this, &BurnDisc::on_cancel_clicked));

  m_pBurnProgress->set_default_response(Gtk::RESPONSE_CANCEL);
}

bool BurnDisc::start_burn()
{
  debug("BurnDisc: start_burn");
  using namespace Nautilus::Burn;

  m_pBurnProgress->show();

  RecorderWriteFlags flags;
  GList *tracks = 0;
  Glib::ustring str;
  cancel = CANCEL_MAKE_ISO;

  bool result = true;
  std::string iso_filename = "";

  if(m_filename == "" && !m_graft_file_contents.empty())
  {
    debug("BurnDisc: making iso");
    mkiso = SharedPtr<MakeIso>(new MakeIso(iso_filename,
                                           m_graft_file_contents,
                                           m_DataProperties,
                                           *m_parent_win));

    sigc::connection set_prog_conn =
      mkiso->signal_progress_changed().connect(sigc::mem_fun(*this,
                                                             &BurnDisc::on_progress_changed));
    sigc::connection set_text_conn =
      mkiso->signal_text_changed().connect(sigc::mem_fun(*this,
                                                         &BurnDisc::on_text_changed));
  
    result = mkiso->make_iso();

    set_prog_conn.disconnect();
    set_text_conn.disconnect();
  }
  else
  {
    debug("BurnDisc: already have iso");
    iso_filename = m_filename;
  }
  
  if(!result)
  {
    return false;
  }
  else
  {
    NautilusBurnRecorderTrack* track;

    m_refRecorder = Recorder::create();
    cancel = CANCEL_CD_RECORD;

    m_refRecorder->signal_progress_changed().connect(sigc::mem_fun(*this,
                                                                   &BurnDisc::on_progress_changed));
    m_refRecorder->signal_action_changed().connect(sigc::mem_fun(*this,
                                                                 &BurnDisc::on_action_changed));
    //TODO:
    //m_refRecorder->signal_insert_media_request().connect(sigc::mem_fun(*this,
    //                                                        &BurnDisc::on_insert_media_request));

    track = g_new0(NautilusBurnRecorderTrack, 1);
    track->type = NAUTILUS_BURN_RECORDER_TRACK_TYPE_DATA;
    track->contents.data.filename = g_strdup(const_cast<char*>(iso_filename.c_str()));
    tracks = g_list_prepend(tracks, track);

    flags = (RecorderWriteFlags)(0);
    if(m_bEject_CD)
    {
      flags |= RECORDER_EJECT;
    }
    if(m_bDummy)
    {
      flags |= RECORDER_DUMMY_WRITE;
    }

    Glib::RefPtr<Gnome::Conf::Client> client = Gnome::Conf::Client::get_default_client();

    if(client->get_bool("/apps/nautilus-cd-burner/burnproof"))
    {
      flags |= RECORDER_BURNPROOF;
    }
    if(client->get_bool("/apps/nautilus-cd-burner/overburn"))
    {
      flags |= RECORDER_OVERBURN;
    }
    if(debug())
    {
      flags |= RECORDER_DEBUG;
    }

    m_refRecorder->write_tracks(m_pDrive, tracks, m_dSpeed, flags);

    if(m_filename == "")
      Gnome::Vfs::Handle::unlink (iso_filename);

    cancel = CANCEL_NONE;

    g_list_foreach(tracks, (GFunc)nautilus_burn_recorder_track_free, NULL);
    g_list_free(tracks);
  }
    
  return true;
}

void BurnDisc::on_cancel_clicked()
{
  if(cancel == CANCEL_NONE)
  {
    return;
  }

  if(cancel == CANCEL_MAKE_ISO)
  {
    mkiso->cancel();
    return;
  }
  
  if(m_refRecorder->cancel(true) == false)
  {
    m_refRecorder->cancel(false);
  }
}

void BurnDisc::on_progress_changed(double fraction)
{
  m_progress_fraction = fraction;
  m_pBurnProgress->set_progress(fraction);

  StatusIcon::set(m_progress_text, m_progress_fraction);
}

void BurnDisc::on_text_changed(const Glib::ustring& text)
{
  m_progress_text = text;
  m_pBurnProgress->set_status(text);
  
  StatusIcon::set(m_progress_text, m_progress_fraction);
}

void BurnDisc::on_action_changed(Nautilus::Burn::RecorderActions action,
                                 Nautilus::Burn::RecorderMedia media)
{
  using namespace Nautilus::Burn;
  Glib::ustring text;
  Glib::ustring media_str;
  
  switch(action)
  {
    case RECORDER_PREPARING_WRITE:
      text = _("Preparing to write ");
      break;
    case RECORDER_WRITING:
      text = _("Writing ");
      break;
    case RECORDER_FIXATING:
      text = _("Fixating ");
      break;
    case RECORDER_BLANKING:
      text = _("Erasing ");
      break;
    default:
      g_warning ("Unhandled action in on_action_changed");
      break;
  }

  switch(media)
  {
    case RECORDER_MEDIA_CD:
      media_str = _("CD");
      break;
    case RECORDER_MEDIA_DVD:
      media_str = _("DVD");
      break;
    default:
      g_warning ("Unhandled media in on_action_changed");
      break;
  }

  m_progress_text = text + media_str;
  
  m_pBurnProgress->set_status(m_progress_text);
  if(action != RECORDER_WRITING)
    StatusIcon::set(m_progress_text);
  else
    StatusIcon::set(m_progress_text, m_progress_fraction);
}

} // namespace IO

} // namespace Coaster
