;;; navi2ch-board.el --- subject list module for navi2ch

;; Copyright (C) 2000-2003 by Navi2ch Project

;; Author: Taiki SUGAWARA <taiki@users.sourceforge.net>
;; Keywords: network, 2ch

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.	If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Code:
(provide 'navi2ch-board)
(defvar navi2ch-board-ident
  "$Id: navi2ch-board.el,v 1.66 2003/07/16 12:30:48 mami Exp $")

(eval-when-compile (require 'cl))

(require 'navi2ch)

(defvar navi2ch-board-mode-map nil)
(unless navi2ch-board-mode-map
  (let ((map (make-sparse-keymap)))
    (set-keymap-parent map navi2ch-bm-mode-map)
    ;; (define-key map "q" 'navi2ch-board-exit)
    (define-key map "s" 'navi2ch-board-sync)
    (define-key map "r" 'navi2ch-board-select-view-range)
    (define-key map "a" 'navi2ch-board-add-bookmark)
    (define-key map "d" 'navi2ch-board-hide-article)
    (define-key map "h" 'navi2ch-board-toggle-hide)
    (define-key map "+" 'navi2ch-board-toggle-updated)
    (define-key map "b" 'navi2ch-board-toggle-bookmark)
    (define-key map "w" 'navi2ch-board-write-message)
    (define-key map "\M-e" 'navi2ch-board-expire)
    (define-key map "md" 'navi2ch-board-hide-mark-article)
    (setq navi2ch-board-mode-map map)))

(defvar navi2ch-board-mode-menu-spec
  (navi2ch-bm-make-menu-spec
   "Board"
   '(["Sync" navi2ch-board-sync]
     ["Add bookmark" navi2ch-board-add-bookmark]
     ["Add global bookmark" navi2ch-board-add-global-bookmark]
     ["Hide line" navi2ch-board-hide-article])))

(defvar navi2ch-board-regexp-list
  '("^\\([0-9]+\\)\\.dat,\\(.*\\)(\\([0-9]+\\))\n"
    "^\\([0-9]+\\)\\.dat<>\\(.*\\)<\\([0-9]+\\)>\n"
    "^\\([0-9]+\\)\\.dat<>\\(.*\\)(\\([0-9]+\\))\n"
    "^\\([0-9]+\\)\\.dat,\\(.*\\)$B!J(B\\([0-9]+\\)$B!K(B\n"))

(defvar navi2ch-board-current-board nil)
(defvar navi2ch-board-subject-list nil)
(defvar navi2ch-board-old-subject-list nil)
(defvar navi2ch-board-subject-alist nil)
(defvar navi2ch-board-old-subject-alist nil)
(defvar navi2ch-board-subject-file-name "subject.txt")
(defvar navi2ch-board-old-subject-file-name "old-subject.txt")
(defvar navi2ch-board-enable-readcgi nil)
(defvar navi2ch-board-last-seen-alist nil)

(defvar navi2ch-board-subback-file-name "subback.html")
(defvar navi2ch-board-use-subback-html nil)

(defvar navi2ch-board-minor-mode-list
  '(navi2ch-board-bookmark-mode
    navi2ch-board-hide-mode
    navi2ch-board-updated-mode))

;; bookmark mode
(defvar navi2ch-board-bookmark-mode nil)
(defvar navi2ch-board-bookmark-mode-map nil)
(unless navi2ch-board-bookmark-mode-map
  (setq navi2ch-board-bookmark-mode-map (make-sparse-keymap))
  (define-key navi2ch-board-bookmark-mode-map
    "d" 'navi2ch-board-delete-bookmark)
  (define-key navi2ch-board-bookmark-mode-map
    "md" 'navi2ch-board-delete-bookmark-mark-article)
  (define-key navi2ch-board-bookmark-mode-map
    "a" 'undefined))

;; hide mode
(defvar navi2ch-board-hide-mode nil)
(defvar navi2ch-board-hide-mode-map nil)
(unless navi2ch-board-hide-mode-map
  (setq navi2ch-board-hide-mode-map (make-sparse-keymap))
  (define-key navi2ch-board-hide-mode-map
    "d" 'navi2ch-board-cancel-hide)
  (define-key navi2ch-board-hide-mode-map
    "md" 'navi2ch-board-cancel-hide-mark-article)
  (define-key navi2ch-board-hide-mode-map
    "a" 'undefined))

;; updated mode
(defvar navi2ch-board-updated-mode nil)
(defvar navi2ch-board-updated-mode-map nil)
(unless navi2ch-board-updated-mode-map
  (setq navi2ch-board-updated-mode-map (make-sparse-keymap)))

;;; navi2ch-bm callbacks
(defun navi2ch-board-set-property (begin end item)
  (put-text-property begin end 'article item))

(defun navi2ch-board-get-property (point)
  (get-text-property (save-excursion (goto-char point)
				     (beginning-of-line)
				     (point))
		     'article))

(defun navi2ch-board-get-article (item)
  item)

(defun navi2ch-board-get-board (item)
  navi2ch-board-current-board)

(defun navi2ch-board-exit ()
  (run-hooks 'navi2ch-board-exit-hook)
  (navi2ch-board-save-info))

;; regist board
(navi2ch-bm-regist-board 'board 'navi2ch-board-select-board)

;; add hook
(add-hook 'navi2ch-save-status-hook 'navi2ch-board-save-info)

;;; navi2ch-board functions
(defun navi2ch-board-get-uri (board)
  "$B8e$m$N(B / $B$,IU$$$?(B uri $B$rJV$9!#(B"
  (let ((uri (cdr (assq 'uri board))))
    (when uri
      (when (string-match "[^/]$" uri)
	(setq uri (concat uri "/")))
      uri)))

(defun navi2ch-board-get-host (board)
  (navi2ch-url-to-host (cdr (assq 'uri board))))

(defun navi2ch-board-get-url (board &optional file-name)
  (if (and file-name (string-match "^/" file-name))
      (concat "http://" (navi2ch-board-get-host board) file-name)
    (concat (navi2ch-board-get-uri board)
	    (or file-name navi2ch-board-subject-file-name))))

(defun navi2ch-board-get-readcgi-url (board)
  "read.cgi $B$NHDL>$^$G$N(B url $B$rJV$9!#(B"
  (let ((uri (navi2ch-board-get-uri board)))
    (string-match "\\(.+\\)/\\([^/]+\\)/$" uri)
    (format "%s/test/read.cgi/%s/"
	    (match-string 1 uri) (match-string 2 uri))))

(defcustom navi2ch-board-default-bbscgi-path "/test/bbs.cgi"
  "*bbs.cgi $B$N%G%U%)%k%H$N(B path$B!#(B"
  :type 'string
  :group 'navi2ch-board)

(defcustom navi2ch-board-bbscgi-path-alist nil
  "*$BHD(B URL $B$+$i(B bbs.cgi $B$N(B path $B$X$N(B alist$B!#(B"
  :type '(repeat (cons (string :tag "URL") (string :tag "path")))
  :group 'navi2ch-board)

(defun navi2ch-board-get-bbscgi-path (board)
  "bbs.cgi $B$N(B path $B$rJV$9!#(B"
  (let ((uri (navi2ch-board-get-uri board)))
    (or (cdr (assoc uri navi2ch-board-bbscgi-path-alist))
	navi2ch-board-default-bbscgi-path)))

(defun navi2ch-board-get-bbscgi-url (board)
  "bbs.cgi $B$N(B url $B$rJV$9!#(B"
  (let ((uri (navi2ch-board-get-uri board)))
    (string-match "\\(.+\\)/[^/]+/$" uri)
    (concat (match-string 1 uri) (navi2ch-board-get-bbscgi-path board))))

(defun navi2ch-board-equal (board1 board2)
  (string= (cdr (assq 'uri board1))
	   (cdr (assq 'uri board2))))

(defun navi2ch-board-get-file-name (board &optional file-name)
  (navi2ch-multibbs-board-get-file-name board file-name))

(defsubst navi2ch-board-from-file-p (board)
  (string= (cdr (assq 'name board)) navi2ch-board-name-from-file))

(defsubst navi2ch-board-get-matched-article ()
  "match $B$7$?7k2L$+$i(B article $B$rF@$k(B"
  (let ((id (match-string 1))
	(str (match-string 2))
	(num (match-string 3)))
    ;; (setq str (navi2ch-replace-string "^ +" "" str)
    ;;       str (navi2ch-replace-string " +$" "" str))
    ;; $B$OCY$$!#(B
    (when (string-match "^ +" str)
      (setq str (replace-match "" nil t str)))
    (when (string-match " +$" str)
      (setq str (replace-match "" nil t str)))
    (list (cons 'subject str)
	  (cons 'response num)
	  (cons 'artid id))))

(defun navi2ch-board-url-to-board (url)
  "URL $B$+$i(B board $B$rF@$k(B"
  (let* ((alist (navi2ch-multibbs-url-to-board url))
	 (uri  (cdr (assq 'uri alist)))
	 (id   (cdr (assq 'id alist)))
	 board)
    (when id
      (dolist (x navi2ch-list-board-name-list)
	(when (string= (cdr (assq 'uri x)) uri)
	  (setq board x)))
      (unless board
	(setq board (list (cons 'uri uri)
			  (cons 'id id)
			  (cons 'type 'board)
			  (cons 'name "No Name"))))
      board)))

(defun navi2ch-board-to-url (board)
  "BOARD $B$+$i(B url $B$KJQ49(B"
  (navi2ch-board-get-uri board))

(defun navi2ch-board-get-subject-list (file)
  "file $B$+$i%9%l$N(B list $B$r:n$k!#(B"
  (when (file-exists-p file)
    (with-temp-buffer
      (navi2ch-insert-file-contents file)
      (run-hooks 'navi2ch-board-get-subject-list-hook)
      (navi2ch-apply-filters navi2ch-board-current-board navi2ch-board-filter-list)
      (navi2ch-replace-html-tag-with-buffer)
      (goto-char (point-min))
      (let ((regexp (navi2ch-board-regexp-test))
	    ;; (file-list (directory-files
	    ;;		   (navi2ch-board-get-file-name
	    ;;		    navi2ch-board-current-board "") nil "\\.dat$"))
	    list)
	(if (null regexp) nil
	  (while (re-search-forward regexp nil t)
	    (setq list (cons (navi2ch-board-get-matched-article) list)))
	  (nreverse list))))))

(defun navi2ch-board-get-updated-subject-list (board)
  "$B0l;~E*$K(B update $B$7$F$+$i%9%l$N(B list $B$r:n$k!#(B"
  (let ((file (navi2ch-board-get-file-name board)))
    (unwind-protect
	(progn
	  (navi2ch-board-save-old-subject-file board)
	  (navi2ch-board-update-file board)
	  (navi2ch-board-get-subject-list file))
      (ignore-errors (navi2ch-board-save-old-subject-file board 'restore)))))

(defsubst navi2ch-board-updated-article-p (article seen)
  (let* ((artid (cdr (assq 'artid article)))
	 (res (cdr (assoc artid navi2ch-board-subject-alist)))
	 old-res)
    (when res
      (if (setq old-res (cdr (assoc artid navi2ch-board-old-subject-alist)))
	  (when (> (string-to-number res)
		   (or seen (string-to-number old-res)))
	    'updated)
	'new))))

(defun navi2ch-board-regexp-test ()
  (save-excursion
    (beginning-of-line)
    (catch 'loop
      (dolist (regexp navi2ch-board-regexp-list)
	(when (looking-at regexp)
	  (throw 'loop regexp))))))


(defun navi2ch-board-insert-subjects (list)
  (let ((bookmark (cdr (assq 'bookmark navi2ch-board-current-board)))
	(hide (cdr (assq 'hide navi2ch-board-current-board)))
	(summary (navi2ch-article-load-article-summary
		  navi2ch-board-current-board))
	(i 1))
    (dolist (article list)
      (let* ((artid (cdr (assq 'artid article)))
	     (seen (navi2ch-article-summary-element-seen
		    (cdr (assoc artid summary))))
	     updated)
	(when (cond (navi2ch-board-bookmark-mode
		     (member artid bookmark))
		    (navi2ch-board-hide-mode
		     (member artid hide))
		    (navi2ch-board-updated-mode
		     (if navi2ch-board-hide-updated-article
			 (and (navi2ch-board-updated-article-p article seen)
			      (not (member artid hide)))
		       (navi2ch-board-updated-article-p article seen)))
		    (t
		     (not (member artid hide))))
	  (navi2ch-bm-insert-subject
	   article i
	   (cdr (assq 'subject article))
	   (let ((res (cdr (assq 'response article)))
		 (last (and navi2ch-board-insert-subject-with-diff
			    (or seen
				(cdr (assoc artid navi2ch-board-last-seen-alist))
				(catch 'break
				  (string-to-number
				   (or (cdr (assoc artid navi2ch-board-old-subject-alist))
				       (throw 'break nil)))))))
		 (read (and navi2ch-board-insert-subject-with-unread
			    (navi2ch-article-get-last-read-number
			     navi2ch-board-current-board
			     article))))
	     (concat "("
		     (format "%4s" res)
		     (and navi2ch-board-insert-subject-with-diff
			  (concat
			   "/"
			   (if last
			       (navi2ch-ifxemacs
				   (format "%+5d"
					   (- (string-to-number res) last))
				 (if (>= (string-to-number res) last)
				     (substring
				      (format "   +%d"
					      (- (string-to-number res) last))
				      -5)
				   (format "%5d"
					   (- (string-to-number res) last))))
			     "    -")))
		     (and navi2ch-board-insert-subject-with-unread
			  (concat
			   "/"
			   (if read
			       (substring
				(format "   $B&$(B%d"
					(max 0
					     (- (string-to-number res) read)))
				-5)
			     "     -")))
		     ")"))
	   (cond ((and navi2ch-board-check-updated-article-p
		       (setq updated
			     (navi2ch-board-updated-article-p article seen)))
		  updated)
		 (seen 'seen)))
	  (setq i (1+ i)))))))

(defun navi2ch-board-select-board (board &optional force)
  (let ((old-mode major-mode))
    (navi2ch-board-mode)
    (navi2ch-bm-setup 'navi2ch-board)
    (setq navi2ch-board-bookmark-mode nil)
    (save-excursion
      (if (and (eq navi2ch-board-current-board board)
	       (eq old-mode major-mode))
	  (navi2ch-board-sync force)
	(setq navi2ch-board-current-board (navi2ch-board-load-info board))
	(run-hooks 'navi2ch-board-select-board-hook)
	(navi2ch-board-sync force 'first)))))

(easy-menu-define navi2ch-board-mode-menu
  navi2ch-board-mode-map
  "Menu used in navi2ch-board"
  navi2ch-board-mode-menu-spec)

(defun navi2ch-board-setup-menu ()
  (easy-menu-add navi2ch-board-mode-menu))

(defun navi2ch-board-mode ()
  "\\{navi2ch-board-mode-map}"
  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'navi2ch-board-mode)
  (setq mode-name "Navi2ch Board")
  (setq buffer-read-only t)
  (buffer-disable-undo)
  (use-local-map navi2ch-board-mode-map)
  (navi2ch-board-setup-menu)
  (run-hooks 'navi2ch-bm-mode-hook 'navi2ch-board-mode-hook)
  (force-mode-line-update))

(defun navi2ch-board-save-old-subject-file (board &optional restore)
  (let ((from (navi2ch-board-get-file-name board))
	(to (navi2ch-board-get-file-name
	     board navi2ch-board-old-subject-file-name)))
    (when restore
      (setq from (prog1 to
		   (setq to from))))
    (when (file-exists-p from)
      (let ((coding-system-for-write navi2ch-coding-system))
	(with-temp-file to
	  (navi2ch-insert-file-contents from))))))

(defun navi2ch-board-update-seen-articles ()
  (let ((summary (navi2ch-article-load-article-summary
		  navi2ch-board-current-board)))
    (dolist (x summary)
      (let* ((element (cdr x))
	     (artid (car x))
	     (seen (navi2ch-article-summary-element-seen element)))
	(when (navi2ch-board-updated-article-p
	       (list (cons 'artid artid))
	       seen)
	  (setq navi2ch-board-last-seen-alist
		(navi2ch-put-alist
		 artid seen navi2ch-board-last-seen-alist))
	  (navi2ch-article-summary-element-set-seen element nil))
	(navi2ch-put-alist artid element summary)))
    (navi2ch-article-save-article-summary
     navi2ch-board-current-board summary)))

(defun navi2ch-board-get-readcgi-raw-url (board)
  (format "%s?raw=0.0" (navi2ch-board-get-readcgi-url board)))

(defun navi2ch-board-update-file (board)
  (unless navi2ch-offline
    (navi2ch-multibbs-board-update board)))

(defun navi2ch-board-sync (&optional force first)
  (interactive "P")
  (run-hooks 'navi2ch-board-before-sync-hook)
  (save-excursion
    (let* ((buffer-read-only nil)
	   (navi2ch-net-force-update (or navi2ch-net-force-update
					 force))
	   (board navi2ch-board-current-board)
	   (file (navi2ch-board-get-file-name board))
	   (old-file (navi2ch-board-get-file-name
		      board
		      navi2ch-board-old-subject-file-name))
	   time header)
      (unless navi2ch-offline
	(navi2ch-board-save-old-subject-file board)
	(setq header (navi2ch-board-update-file board))
	(setq time (and (not (navi2ch-net-get-state 'not-updated header))
			(not (navi2ch-net-get-state 'error header))
			(or (cdr (assoc "Last-Modified" header))
			    (cdr (assoc "Date" header)))))
	(when time
	  (setq board (navi2ch-put-alist 'time time board))))
      (setq navi2ch-board-current-board board)
      (when (or first time)
	(erase-buffer)
	(setq navi2ch-board-subject-list
	      (navi2ch-board-get-subject-list file))
	(setq navi2ch-board-subject-alist
	      (navi2ch-alist-list-to-alist
	       navi2ch-board-subject-list 'artid 'response))
	(setq navi2ch-board-old-subject-list
	      (navi2ch-board-get-subject-list old-file))
	(setq navi2ch-board-old-subject-alist
	      (navi2ch-alist-list-to-alist
	       navi2ch-board-old-subject-list 'artid 'response))
	(setq navi2ch-board-last-seen-alist nil)
	(when time
	  (navi2ch-board-update-seen-articles))
	(navi2ch-board-insert-subjects navi2ch-board-subject-list)
	(navi2ch-board-save-info)
	(navi2ch-board-set-mode-line))))
  (run-hooks 'navi2ch-board-after-sync-hook))

(defun navi2ch-board-make-subject-txt ()
  "subback.html $B$+$i(B (navi2ch $BMQ$N(B) subject.txt $B$r:n$k(B
`navi2ch-net-update-file' $B$N%O%s%I%i!#(B"
  (let ((coding-system-for-read 'binary)
	(coding-system-for-write 'binary)
	(case-fold-search t)
	(beg (point)))
    (while (re-search-forward
	    ;; $B$3$N@55,I=8=$b;EMMJQ99$G$@$a$K$J$k$N$+$b!#(B
	    ;;   $B!D!D$J$j$^$7$?!#(B
	    "<a +href=\"\\([0-9]+\\)[^>]*\">[0-9]+[^0-9].\\(.*\\)</a>" nil t)
      (let ((dat (match-string 1))
	    (title (match-string 2)))
	(delete-region beg (point))
	(insert (format "%s.dat<>%s\n" dat title))
	(setq beg (point))))
    (delete-region beg (point-max))))

(defun navi2ch-board-set-mode-line ()
  (let* ((board navi2ch-board-current-board)
	 (host (navi2ch-url-to-host (cdr (assq 'uri board)))))
    (setq navi2ch-mode-line-identification
	  (format "%s (%s) [%s]" (cdr (assq 'name board))
                  (cdr (assq 'id board))
                  host))
    (navi2ch-set-mode-line-identification)))

(defun navi2ch-board-write-message ()
  (interactive)
  (navi2ch-message-write-message navi2ch-board-current-board nil t))

(defun navi2ch-board-save-info (&optional board)
  (or board (setq board navi2ch-board-current-board))
  (when board
    (navi2ch-save-info
     (navi2ch-board-get-file-name
      board "board.info")
     (list (assq 'bookmark board)
	   (assq 'hide board)
	   (assq 'time board)
	   (assq 'logo board)))))

(defun navi2ch-board-load-info (&optional board)
  (or board (setq board navi2ch-board-current-board))
  (let ((alist (navi2ch-load-info
		(navi2ch-board-get-file-name
		 board "board.info"))))
    (dolist (x alist)
      (setq board
	    (navi2ch-put-alist (car x) (cdr x)
			       board)))
    board))

(defun navi2ch-board-save-spid (board spid)
  (navi2ch-save-info
   (navi2ch-board-get-file-name board "spid.txt")
   spid))

(defun navi2ch-board-load-spid (board)
  (navi2ch-load-info
   (navi2ch-board-get-file-name board "spid.txt")))

(defun navi2ch-board-select-view-range ()
  (interactive)
  (setq-default navi2ch-article-view-range
		(navi2ch-article-select-view-range-subr)))

(defun navi2ch-board-delete-line (sym func msg)
  (let ((artid (cdr (assq 'artid (get-text-property (point) 'article))))
	(list (cdr (assq sym navi2ch-board-current-board))))
    (if artid
	(progn
	  (setq list (funcall func artid list))
	  (setq navi2ch-board-current-board
		(navi2ch-put-alist sym list
				   navi2ch-board-current-board))
	  (let ((buffer-read-only nil))
	    (delete-region (save-excursion (beginning-of-line) (point))
			   (save-excursion (forward-line) (point))))
	  (and (eobp) (not (bobp))
	       (forward-line -1))
	  (message msg))
      (message "Can't select this line!"))))


;;; from Wanderlust(wl-expire.el)
(defsubst navi2ch-board-expire-date-p (key-datevec file access-time)
  (let ((datevec
	 (condition-case nil
	     (navi2ch-make-datevec
	      (or access-time
		  (nth 5 (file-attributes file))))
	   (error nil))))
    (and
     datevec (> (aref datevec 1) 0)
     (string<
      (navi2ch-make-sortable-date datevec)
      (navi2ch-make-sortable-date key-datevec)))))

(defun navi2ch-board-expire (&optional board ask)
  (interactive)
  (and (interactive-p) (setq ask t))
  (or board (setq board navi2ch-board-current-board))
  (unless (eq board navi2ch-board-current-board)
    (setq board (navi2ch-board-load-info board)))
  (let ((dir (navi2ch-board-get-file-name board "")))
    (when (and (file-exists-p dir)
	       (or (not ask) (y-or-n-p "Expire current borad?"))
	       navi2ch-board-expire-date)
      (let ((article-list (mapcar (lambda (file)
				    (when (string-match "[0-9]+" file)
				      (list (cons 'artid
						  (match-string 0 file)))))
				  (directory-files dir nil "\\.dat$")))
	    (summary (navi2ch-article-load-article-summary board))
	    (key-datevec (navi2ch-get-offset-datevec
			  (navi2ch-make-datevec (current-time))
			  navi2ch-board-expire-date))
	    (remove-list nil))
	(message "expiring %s..." (cdr (assq 'name board)))
	(dolist (article article-list)
	  (let ((artid (cdr (assq 'artid article)))
		(file (navi2ch-article-get-file-name board article)))
	    (when (and (or navi2ch-board-expire-bookmark-p
			   (not (or (member artid (cdr (assq 'bookmark board)))
				    (navi2ch-bookmark-exist-all board
								article))))
		       (navi2ch-board-expire-date-p
			key-datevec file
			(navi2ch-article-summary-element-access-time
			 (cdr (assoc artid summary)))))
	      (push article remove-list)
	      (when (assq 'hide board)
		(setcdr (assq 'hide board)
			(delq artid (cdr (assq 'hide board)))))
	      (when navi2ch-board-expire-bookmark-p
		(when (assq 'bookmark board)
		  (setcdr (assq 'bookmark board)
			  (delq artid (cdr (assq 'bookmark board)))))
		(navi2ch-bookmark-delete-article-all board article)))))
	(navi2ch-bm-remove-article-subr board remove-list)
	(navi2ch-board-save-info board)
	(message "expiring %s...done" (cdr (assq 'name board)))))))

(defun navi2ch-board-toggle-minor-mode (mode)
  (dolist (m navi2ch-board-minor-mode-list)
    (if (eq m mode)
	(set m (not (symbol-value m)))
      (set m nil)))
  (force-mode-line-update)
  (let ((buffer-read-only nil))
    (save-excursion
      (erase-buffer)
      (navi2ch-board-insert-subjects navi2ch-board-subject-list))))

;;; bookmark mode
(navi2ch-set-minor-mode 'navi2ch-board-bookmark-mode
			" Bookmark"
			navi2ch-board-bookmark-mode-map)

(defun navi2ch-board-update-buffer (buffer)
  (let (point)
    (with-current-buffer buffer
      (let ((buffer-read-only nil)
	    (artid (cdr (assq 'artid (get-text-property (point) 'article))))
	    p)
	(erase-buffer)
	(navi2ch-board-insert-subjects navi2ch-board-subject-list)
	(setq p (point-min))
	(while p
	  (if (equal artid (cdr (assq 'artid
				      (get-text-property p 'article))))
	      (progn
		(goto-char p)
		(setq point p
		      p nil))
	    (setq p (next-single-property-change p 'article))))))
    (if point
	;; $BI=<($7$F$k%P%C%U%!$N%+!<%=%k0LCV$r$9$Y$F99?7$9$k!#(B
	(save-selected-window
	  (dolist (win (get-buffer-window-list buffer nil t))
	    (select-window win)
	    (goto-char point))))))

(defun navi2ch-board-add-bookmark-subr (board article)
  (let ((buffer (get-buffer navi2ch-board-buffer-name))
	(artid (cdr (assq 'artid article)))
	current-board list)
    (setq board
	  (if (and (bufferp buffer)
		   (eq (with-current-buffer buffer major-mode)
		       'navi2ch-board-mode)
		   (equal (cdr (assq 'id navi2ch-board-current-board))
			  (cdr (assq 'id board))))
	      (progn
		(setq current-board t)
		navi2ch-board-current-board)
	    (navi2ch-board-load-info board)))
    (setq list (cdr (assq 'bookmark board)))
    (when (and artid (not (member artid list)))
      (setq board (navi2ch-put-alist 'bookmark (cons artid list) board))
      (message "Add bookmark")
      (if current-board
	  (with-current-buffer buffer
	    (setq navi2ch-board-current-board board)
	    (if navi2ch-board-bookmark-mode
		(navi2ch-board-update-buffer buffer)))
	(navi2ch-board-save-info board)))))

(defun navi2ch-board-add-bookmark ()
  (interactive)
  (unless navi2ch-board-bookmark-mode
    (let ((article (get-text-property (point) 'article)))
      (if article
	  (navi2ch-board-add-bookmark-subr navi2ch-board-current-board
					   article)
	(message "can't select this line!")))))

(defun navi2ch-board-delete-bookmark ()
  (interactive)
  (navi2ch-board-delete-line 'bookmark 'delete
			     "Delete bookmark"))

(defun navi2ch-board-delete-bookmark-mark-article ()
  (interactive)
  (navi2ch-bm-exec-subr 'navi2ch-board-delete-bookmark))

(defun navi2ch-board-toggle-bookmark ()
  (interactive)
  (navi2ch-board-toggle-minor-mode 'navi2ch-board-bookmark-mode))

;;; hide mode
(navi2ch-set-minor-mode 'navi2ch-board-hide-mode
			" Hide"
			navi2ch-board-hide-mode-map)

(defun navi2ch-board-hide-article ()
  (interactive)
  (navi2ch-board-delete-line 'hide
			     (lambda (artid list)
			       (if (member artid list)
				   list
				 (cons artid list)))
			     "Hide article"))

(defun navi2ch-board-hide-mark-article ()
  (interactive)
  (navi2ch-bm-exec-subr 'navi2ch-board-hide-article))

(defun navi2ch-board-cancel-hide ()
  (interactive)
  (navi2ch-board-delete-line 'hide 'delete
			     "Cancel hide article"))

(defun navi2ch-board-cancel-hide-mark-article ()
  (interactive)
  (navi2ch-bm-exec-subr 'navi2ch-board-cancel-hide))

(defun navi2ch-board-toggle-hide ()
  (interactive)
  (navi2ch-board-toggle-minor-mode 'navi2ch-board-hide-mode))

;;; updated mode
(navi2ch-set-minor-mode 'navi2ch-board-updated-mode
			" Updated"
			navi2ch-board-updated-mode-map)

(defun navi2ch-board-toggle-updated ()
  (interactive)
  (navi2ch-board-toggle-minor-mode 'navi2ch-board-updated-mode))

(run-hooks 'navi2ch-board-load-hook)
;;; navi2ch-board.el ends here
