;;; egg-remix.el --- Roman-Kana conversion input interface for Tamago4
;; Copyright (C) 2000 Hiroki Hayashi

;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2 of
;; the License, or (at your option) any later version.

;; This program is distributed in the hope that it will be
;; useful, but WITHOUT ANY WARRANTY; without even the implied
;; warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
;; PURPOSE.  See the GNU General Public License for more details.

;; You should have received a copy of the GNU General Public
;; License along with this program; if not, write to the Free
;; Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
;; MA 02111-1307 USA


;; Author: Hiroki Hayashi <payashi@dream.com>
;; Keywords: mule, egg, Japanese, input interface

;; Commentary:
;;   Egg ReMix $B$($C$0(B-$B$j$_$C$/$9(B
;;   Tamago V4 $B$G(B egg-mix.el $B$HF1MM$NF~NO4D6-$r<B8=$7$^$9!#(B
;;   $BF|K\8l$NF~NOCf$K!"%b!<%I$r@Z$jBX$($k$3$H$J$/%"%k%U%!%Y%C%HEy$,(B
;;   $BF~NO$G$-$k$h$&$K$J$j$^$9!#(B
;;   $B%U%'%s%9Fb$NJT=8$O!"I8=`$N$?$^$4$H$O0c$$!"F~NO$7$?%-!<$,C10L$K$J$j$^$9!#(B

;;   [$BF~NONc(B] $BF|K\8l$H(B alphabet $B$N:.:_(B
;;   $B!&I8=`$N(B modefull mode $B$N>l9g$NF~NO(B($B%b!<%I$r(B2$B2s@Z$jBX$($kI,MW$,$"$k(B)
;;     nihonngoto[SPC][RET][SPC]qalphabet[SPC][C-g]nokonnzai[SPC][RET]
;; 	                        ^             ^^^^^
;;   $B!&(BEgg ReMix $B$N>l9g$NF~NO(B($B%b!<%I$r@Z$jBX$($kA`:n$,$J$$(B)
;;     nihonngoto[SPC][RET][SPC]alphabet[SPC]nokonnzai[SPC][RET]

;;   $B%m!<%^;z$NDV$j$H1QC18l$,F1$8>l9g(B($BNc$($P!V$^$1!W$H!V(Bmake$B!W(B)$B$O!":G8e$K(B
;;   $B%f!<%6$,A*Br$7$?$b$N$,%U%'%s%9$K8=$l$^$9!#$3$l$,ITET9g$J>l9g$O(B
;;   C-r $B$G6/@)E*$K%"%k%U%!%Y%C%H$r!"(BC-w $B$G$R$i$,$J$rA*Br$7$^$9!#$3$NA*Br$O(B
;;   $B%f!<%6<-=q$KEPO?$5$l!"<!2s$NF~NO;~$K;2>H$5$l$^$9!#(B

;;   $B%m!<%I(B
;;   $BFbIt$G(B (require 'egg) $B$r$7$F$$$k$N$G!"4pK\E*$K$O(B require $B$d(B autoload $B$d(B
;;   load $B$r9T$C$F$$$kItJ,$G!"(Begg $B$NBe$j$K(B egg-remix $B$r;XDj$9$l$P$h$$$G$9!#(B
;;   $B$?$@$7!"(Begg-remix.el $B$,(B load-path $B$NDL$C$F$$$k$H$3$m$KCV$$$F$"$k$3$H$,(B
;;   $BI,MW$G$9!#(B

;;   $B@_Dj(B
;;   $B4pK\E*$K$O%3!<%ICf$N8D?M@_DjMQ$NJQ?t72$r%3%a%s%H$K$7$?$,$C$F@_Dj$7$^$9!#(B
;;   $B<B:]$NF~NO%$%s%?!<%U%'!<%9$O!"(Begg-mode-preference $B$NCM$G@Z$jBX$o$j$^$9!#(B
;;   $B@_Dj$G$-$kCM$O(B egg-input-mode-alist $B$N3FMWAG$N(B car $B$G$9!#(B

;;     egg-mode-preference $B$NCM(B    $B%$%s%?!<%U%'!<%9(B
;;       t $B$^$?$O(B "egg"              $B$?$^$4(B modefull $B%b!<%I(B
;;       nil $B$^$?$O(B "mlh"            $B$?$^$4(B modeless $B%b!<%I(B (mlh $B%b!<%I(B)
;;       "remix"                     Egg ReMix $B%b!<%I(B  

;; Change Log:
;;   Feb 11 2001: version 0.204
;;                $B!&(Bfont-lock-mode $B$,M-8z$J$H$-$K@5$7$/%U%'%s%9$rJT=8$G$-$J$$(B
;;                  $BLdBj$r(B fix
;;                $B!&%U%'%s%9$N(B face $BI=<($K(B overlay $B$rA*Br$G$-$k$h$&$K$7$?(B
;;                  ($BJQ?t(B remix-use-overlay $B$rDI2C(B)
;;                $B!&(Bfence $B$K;H$&J8;zNs$K(B remix-fence-* $B$,;D$C$F$$$?$N$r=$@5(B
;;   Oct 30 2000: version 0.203
;;                $B!&(Bisearch $B$GF|K\8l$rF~NOCf$K(B C-g $B$GCfCG$9$k$H!"0J8e$N(B
;;                  $B%_%K%P%C%U%!$X$NF~NO$,$G$-$J$/$J$kLdBj$KBP1~(B
;;                  (C-g $B$X(B abort $B$9$k4X?t$r%P%$%s%I$7$F2sHr(B)
;;                $B!&JQ49%b!<%I$+$i(B C-k $B$G%U%'%s%9%b!<%I$KLa$C$?$H$-$K!"%U%'%s%9$,(B
;;                  $B@5$7$/:n@.$5$l$J$$LdBj$r(B fix
;;   Oct 12 2000: version 0.202
;;                $B!&(Bremix-fence-open, remix-fence-continue, remix-fence-close $B$,(B
;;                  2$BJ8;z0J>e$N$H$-$K@5>o$KF0:n$7$J$$LdBj$KBP1~(B
;;                $B!&(Bfence $B$K;H$&J8;zNs$,(B remix-fence-* $B$K$J$C$F$$$?$N$r(B
;;                  its-fence-* $B$K=$@5(B
;;                $B!&(Bremix-sysdict $B$d(B remix-usrdict $B$,(B nil $B$N$H$-$K@5>o$KF0:n$7$J$$(B
;;                  $BLdBjE@$r(B fix
;;   Sep 27 2000: version 0.201
;;                $B!&:Y$+$J(B bug $B$r(B fix
;;   Sep 27 2000: version 0.200
;;                $B!&$H$j$"$($:8x3+HG(B

;; Code:

(require 'egg)
(activate-input-method "japanese-egg-wnn")
(activate-input-method nil)

;;;;====
;;;; $B5!G=3HD%$N$?$a$N(B Tamago4 native $B4X?t$N%*!<%P!<%i%$%I(B
;;;; $B>-MhE*$K$OITI,MW$K$J$C$FM_$7$$(B:-)
;;;;====

;;;; its-restart $B8F$S=P$7$rA*Br$G$-$k$h$&$K$9$k(B
(defun egg-decide-before-point ()
  (interactive)
  (let* ((inhibit-read-only t)
	 (start (if (get-text-property (1- (point)) 'egg-start)
		    (point)
		  (previous-single-property-change (point) 'egg-start)))
	 (end (if (get-text-property (point) 'egg-end)
		  (point)
		(next-single-property-change (point) 'egg-end)))
	 (decided (buffer-substring start (point)))
	 (undecided (buffer-substring (point) end))
	 i len bunsetsu source context)
    (delete-region
     (previous-single-property-change start 'egg-start nil (point-min))
     (next-single-property-change end 'egg-end nil (point-max)))
    (setq i 0
	  len (length decided))
    (while (< i len)
      (setq bunsetsu (nconc bunsetsu (list (egg-get-bunsetsu-info i decided)))
	    i (egg-next-bunsetsu-point i 1 decided len))
      (if (or (= i len)
	      (egg-get-bunsetsu-last (1- i) decided))
	  (progn
	    (insert (mapconcat 'egg-get-bunsetsu-converted bunsetsu nil))
	    (setq context (cons (cons (egg-bunsetsu-get-backend (car bunsetsu))
				      (egg-end-conversion bunsetsu nil))
				context)
		  bunsetsu nil))))
    (setq len (length undecided))
    (if (= len 0)
	(progn
	  (egg-do-auto-fill)
	  (run-hooks 'input-method-after-insert-chunk-hook)
	  context)
      (setq i 0)
      (while (< i len)
	(setq bunsetsu (egg-get-bunsetsu-info i undecided)
	      source (cons (egg-get-bunsetsu-source bunsetsu)
			   source))
	(put-text-property 0 (length (car source))
			   'egg-lang
			   (egg-get-source-language bunsetsu)
			   (car source))
	(setq i (egg-next-bunsetsu-point i 1 undecided len)))
      (funcall (nth 3 (assoc egg-mode-preference egg-input-mode-alist))
	       (apply 'concat (nreverse source)) t t context))))

;;; its-restart $B8F$S=P$7$rA*Br$G$-$k$h$&$K$9$k(B
(defun egg-abort-conversion ()
  (interactive)
  (let ((inhibit-read-only t)
	source context)
    (goto-char (previous-single-property-change
		(if (get-text-property (1- (point)) 'egg-start)
		    (point)
		  (previous-single-property-change (point) 'egg-start))
		'egg-start nil (point-min)))
    (setq source (get-text-property (point) 'egg-source)
	  context (get-text-property (point) 'egg-context))
    (delete-region (point) (next-single-property-change
			    (next-single-property-change (point) 'egg-end)
			    'egg-end nil (point-max)))
    (funcall (nth 3 (assoc egg-mode-preference egg-input-mode-alist))
	     source nil nil context)))

;;; egg-mode-preference $B$r3HD%$7!"3F<o$N%b!<%I$rDI2C$G$-$k$h$&$K$9$k(B
;;;###autoload
(defun egg-mode (&rest arg)
  "Toggle EGG mode.
\\[describe-bindings]
"
  (interactive "P")
  (if (null arg)
      ;; Turn off
      (unwind-protect
	  (progn
	    (funcall (nth 4 (assoc egg-mode-preference egg-input-mode-alist)))
	    (egg-exit-conversion))
	(setq describe-current-input-method-function nil)
	(let ((modelist egg-input-mode-alist)
	      mode)
	  (while (setq mode (car modelist))
	    (set (modename mode) nil)
	    (setq modelist (cdr modelist))))
	(remove-hook 'input-method-activate-hook 'its-set-mode-line-title t)
	(force-mode-line-update))
    ;; Turn on
    (if (null (string= (car arg) egg-last-method-name))
	(progn
	  (funcall (nth 1 arg))
	  (setq egg-default-language its-current-language)))
    (egg-set-conversion-backend (nthcdr 2 arg))
    (egg-set-conversion-backend
     (list (assq its-current-language (nthcdr 2 arg))) t)
    (setq egg-last-method-name (car arg)
	  egg-activated t)
    (egg-activate-keymap)

    (let ((mode (assoc egg-mode-preference egg-input-mode-alist)))
      (set (modename mode) t)
      (if (nth 3 mode)
	  (its-define-select-keys (symbol-value (mapname mode)))))

    (setq inactivate-current-input-method-function 'egg-mode)
    (setq describe-current-input-method-function 'egg-help)
    (make-local-hook 'input-method-activate-hook)
    (add-hook 'input-method-activate-hook 'its-set-mode-line-title nil t)
    (if (eq (selected-window) (minibuffer-window))
	(add-hook 'minibuffer-exit-hook 'egg-exit-from-minibuffer))
    (run-hooks 'egg-mode-hook)))

;;; $B:G8e$K(B input-method-after-insert-chunk-hook $B$r8F$S=P$9$h$&$K$9$k(B
(defun egg-decide-first-char ()
  (interactive)
  (let* ((inhibit-read-only t)
	 (start (if (get-text-property (1- (point)) 'egg-start)
		    (point)
		  (previous-single-property-change (point) 'egg-start)))
	 (end (if (get-text-property (point) 'egg-end)
		  (point)
		(next-single-property-change (point) 'egg-end)))
	 (bunsetsu (egg-get-bunsetsu-info start)))
    (delete-region
     (previous-single-property-change start 'egg-start nil (point-min))
     (next-single-property-change end 'egg-end nil (point-max)))
    (egg-end-conversion (list bunsetsu) nil)
    (insert (egg-string-to-char-at (egg-get-bunsetsu-converted bunsetsu) 0))
    (run-hooks 'input-method-after-insert-chunk-hook)))

;;;;====
;;;; $B5!G=3HD%MQ$NDI2C%3!<%I(B
;;;;====
(defsubst modename (mode)
  (intern (format "egg-%s-mode" (nth 1 mode))))
(defsubst mapname (mode)
  (intern (format "egg-%s-map" (nth 1 mode))))

(defvar egg-input-mode-alist '(("egg"   "modefull" t   its-restart   its-exit-mode)
			       ("mlh"   "modeless" nil its-restart   its-exit-mode)
			       ("remix" "remix"    t   remix-restart remix-exit-mode)
			       (t       "modefull" t   its-restart   its-exit-mode)
			       (nil     "modeless" nil its-restart   its-exit-mode))
  "Egg $B$NF~NO%b!<%I$N%j%9%H!#(B
$B3FMWAG$O(B\"$B%b!<%IL>(B\", \"$BFbIt<1JLL>(B\", modefull$B%U%i%0(B, $BJQ49%b!<%I$+$i$NI|5"4X?t(B, $B%U%'%s%9%b!<%I$N=*N;4X?t$N%j%9%H$G$"$k!#(B
\"$B%b!<%IL>(B\"$B$r(B egg-mode-preference $B$K@_Dj$9$k$3$H$G!"F~NO%b!<%I$r@Z$jBX$($k(B")


;;;;====
;;;; Egg-ReMix code
;;;;====
(defconst remix-version "0.204"
  "Egg ReMix $B$N%P!<%8%g%s(B")
(defconst remix-version-date "Feb 11 2001"
  "Egg ReMix $B$N:G=*99?7F|(B")
;;;###autoload
(defun remix-version ()
  "Egg ReMix $B$N%P!<%8%g%s$rJV$9(B"
  (interactive)
  (message "Egg ReMix version %s (%s)" remix-version remix-version-date))

;;;;====
;;;; $B8D?M@_DjMQ$NJQ?t(B
;;;;====
(defvar remix-quiet nil
  "*$B%a%C%;!<%8$rI=<($9$k$H$-$K(B ding $B$r9T$&$+$I$&$+$N%U%i%0!#(B
non-nil $B$@$H(B ding $B$r9T$o$J$$(B")

(defvar remix-remix-fence t
  "*$B%U%'%s%9$NI=<(7A<0$r(B Egg-ReMix $B$G@_Dj$7$?$b$N$K=i4|2=$9$k$+$I$&$+$N%U%i%0!#(B
non-nil $B$J$i$P!"=i4|2=$7!"(Bnil $B$J$i$J$K$b$7$J$$(B")

(defvar remix-fence-invisible t
  "*$B%U%'%s%9$NN>C<$rI=<($7$J$$$+$I$&$+$N%U%i%0!#(B
non-nil $B$J$i$P!"%U%'%s%9$NN>C<$OI=<($7$J$$(B")

(defvar remix-fence-open "|"
  "*$B%U%'%s%9$N3+;O$K;H$&J8;z!#(B
remix-remix-fence $B$,(B t $B$N;~$O!"(B1$BJ8;z0J>e$NJ8;zNs$r;XDj$9$kI,MW$,$"$k(B")

(defvar remix-fence-continue "+"
  "*$B7QB3$9$k%U%'%s%9$N3+;O$K;H$&J8;z!#(B
remix-remix-fence $B$,(B t $B$N;~$O!"(B1$BJ8;z0J>e$NJ8;zNs$r;XDj$9$kI,MW$,$"$k(B")

(defvar remix-fence-close "|"
  "*$B%U%'%s%9$N=*C<;~;H$&J8;z!#(B
remix-remix-fence $B$,(B t $B$N;~$O!"(B1$BJ8;z0J>e$NJ8;zNs$r;XDj$9$kI,MW$,$"$k(B")

(defvar remix-fence-face 'underline
  "*$B%U%'%s%9$NI=<($K;H$&(B face")

(defvar remix-sysdict '("/usr/dict/words")
  "*$B%7%9%F%`$G;H$&1QC18l<-=q$N@dBP%Q%9$N%j%9%H!#(B
$B$3$N%j%9%H$,(B nil $B$J$i$P%7%9%F%`$N<-=q$O;H$o$J$$!#(B
$B<-=q$O(B1$B9T(B1$BC18l$N7A<0$G$"$kI,MW$,$"$k(B")

(defvar remix-usrdict (expand-file-name "~/.remix-freq")
  "*$B%f!<%6MQ$NC18l<-=q$N%U%!%$%kL>!#(B
$B:G8e$K%m!<%^;zDV$j$H$_$J$7$?$+Hs%m!<%^;zDV$j$H$_$J$7$?$+$b5-O?$9$k!#(B
nil $B$J$i$P%f!<%6MQC18l<-=q$r;H$o$J$$(B")

(defvar remix-auto-add-entry t
  "*$B%f!<%6<-=q$K$J$$DV$j$r<+F0E*$KDI2C$9$k$+$I$&$+$N%U%i%0!#(B
$BCM$,(B non-nil $B$J$i!"%f!<%6<-=q$K$J$$DV$j$,%7%9%F%`<-=q$K$"$C$?$H$-$K(B
$B<+F0E*$K%f!<%6<-=q$KDI2C$9$k(B")

(defvar remix-use-overlay 10
  "*$B%U%'%s%9$N(B face $B$r(B overlay $B$GI=<($9$k$+$I$&$+$N%U%i%0!#(B
nil $B$J$i$P(B text property $B$r;H$&!#(B
$B?tCM$J$i$P$=$NCM$r(B priority $B$H$9$k(B overlay $B$r;H$&!#(B")

;;;;====
;;;; Egg-ReMix $B$NFbIt$G;H$&JQ?t(B
;;;;====
(defvar remix-sysdictbuf " *remix-sysdict*"
  "$B%7%9%F%`<-=q$r3JG<$9$k%P%C%U%!$NL>A0(B")

(defvar remix-usrdictbuf " *remix-usrdict*"
  "$B%f!<%6<-=q$r3JG<$9$k%P%C%U%!$NL>A0(B")

(defvar remix-state-indicator-alist '((remix-kanaseq .      "$B$"(B") ; $B%m!<%^;zDV$j(B
				      (remix-uncertain .    "$B!~(B") ; $B>uBVL$3NDj(B
				      (remix-nokanaseq .    "$B"!(B") ; $BHs%m!<%^;zDV$j(B
				      (remix-indictkana .   "$B!y(B") ; $B<-=qFb$N%m!<%^;zDV$j(B
				      (remix-indictnokana . "$B!z(B") ; $B<-=qFb$NHs%m!<%^;zDV$j(B
				      (remix-inedit .       "$B"&(B") ; $B%U%'%s%9JT=8Cf(B
				      )
  "*ReMix$B$N>uBV$H%b!<%I%i%$%s9T$N%$%s%8%1!<%?$NBP1~%j%9%H(B")

;;; $B=i4|>uBV(B($B%U%)!<%k%P%C%/(B)$B$N%$%s%8%1!<%?(B
(or (assq 'remix-default remix-state-indicator-alist)
    (setq remix-state-indicator-alist (append remix-state-indicator-alist
					      (list
					       (cons 'remix-default
						     (its-get-indicator
						      (symbol-value its-current-map)))))))

(define-egg-mode-map remix
  (define-key map "\C-^" 'egg-simple-input-method)
  (let ((i 33))
    (while (< i 127)
      (define-key map (vector i) 'remix-self-insert-char)
      (setq i (1+ i)))))

(defvar remix-its-mode-map
  (let ((map (make-sparse-keymap))
	(i 33))
     (define-key map "\C-a" 'remix-beginning-of-input-buffer)
     (define-key map "\C-b" 'remix-backward)
     (define-key map "\C-c" 'remix-cancel-input)
     (define-key map "\C-d" 'remix-delete)
     (define-key map "\C-e" 'remix-end-of-input-buffer)
     (define-key map "\C-f" 'remix-forward)
     (define-key map "\C-g" 'remix-abort)
     (define-key map "\C-]" 'remix-cancel-input)
     (define-key map "\C-h" 'remix-mode-help-command)
     (define-key map "\C-k" 'remix-kill-line)
     (define-key map "\C-m" 'remix-exit-mode)
     (define-key map [return] 'remix-exit-mode)
     (define-key map "\C-r" 'remix-select-nokana)
     (define-key map "\C-t" 'remix-transpose-chars)
     (define-key map "\C-w" 'remix-select-kana)
     (define-key map [backspace] 'remix-delete-backward)
     (define-key map [delete] 'remix-delete-backward)
     (define-key map [right] 'remix-forward)
     (define-key map [left] 'remix-backward)
    (while (< i 127)
      (define-key map (vector i) 'remix-its-self-insert-char)
      (setq i (1+ i)))
    (define-key map " "    'remix-kick-convert-region-or-self-insert)
    (define-key map "\177" 'remix-delete-backward)
    ;;
    (define-key map "\M-h" 'remix-hiragana)
    (define-key map "\M-k" 'remix-katakana)
    (define-key map "\M->" 'remix-full-width)
    map)
  "Keymap for EggReMix's fence")
(defvar remix-its-fence-mode nil)
(fset 'remix-its-mode-map remix-its-mode-map)

(make-variable-buffer-local 'remix-its-fence-mode)
(put 'remix-its-fence-mode 'permanent-local t)
(or (assq 'remix-its-fence-mode egg-sub-mode-map-alist)
    (setq egg-sub-mode-map-alist (cons '(remix-its-fence-mode . remix-its-mode-map)
				       egg-sub-mode-map-alist)))

(defvar remix-context nil)
(defvar remix-fence-operation nil)
(defvar remix-usrdict-modified nil)

;;; remix-remix-fence $B$,(B non-nil $B$J$i!"%U%'%s%9$N304Q$r(B
;;; remix $B$N$b$N$G=i4|2=$9$k(B
(if remix-remix-fence
    (setq its-fence-invisible remix-fence-invisible
	  its-fence-open      remix-fence-open
	  its-fence-continue  remix-fence-continue
	  its-fence-close     remix-fence-close
	  its-fence-face      remix-fence-face
	  egg-conversion-fence-invisible remix-fence-invisible
	  egg-conversion-fence-open      remix-fence-open
	  egg-conversion-fence-continue  remix-fence-continue
	  egg-conversion-fence-close     remix-fence-close
	  egg-conversion-face            remix-fence-face))

;;; remix-use-overlay $B$NBEEv@-%A%'%C%/(B
;;; nil: overlay $B$r;H$o$J$$(B
;;; number: overlay $B$N(B priority $B$K$9$k(B
(or
 (null remix-use-overlay)
 (numberp remix-use-overlay)
 (setq remix-use-overlay nil))

;;; $B4X?tDj5A(B
(defun remix-enter/leave-fence (&optional old new)
  (setq remix-its-fence-mode (remix-in-fence-p))
  (or remix-fence-operation
      (let* ((status (get-text-property (point) 'remix-status))
	     (title (cdr (or (assq status remix-state-indicator-alist)
			     (assq 'remix-default remix-state-indicator-alist)))))
	(following-char)
	(if (equal title current-input-method-title)
	    nil
	  (setq current-input-method-title title)
	  (force-mode-line-update)))))

(defun remix-in-fence-p ()
  (and (eq (get-text-property (point) 'intangible) 'remix-part-2)
       (get-text-property (point) 'read-only)))

(defun remix-beginning-of-input-buffer ()
  (interactive)
  (let* ((inhibit-read-only t)
	 (p (point))
	 (seq-before (get-text-property p 'remix-keyseq-before))
	 (seq-after (get-text-property p 'remix-keyseq-after)))
    (setq seq-after (concat seq-before seq-after))
    (setq seq-before nil)
    (add-text-properties p (1+ p) (list 'remix-keyseq-before seq-before
					'remix-keyseq-after seq-after))

    (remix-update-fence 'remix-inedit)))

(defun remix-end-of-input-buffer (&optional last)
  (interactive)
  (let* ((inhibit-read-only t)
	 (p (point))
	 (seq-before (get-text-property p 'remix-keyseq-before))
	 (seq-after (get-text-property p 'remix-keyseq-after)))
    (setq seq-before (concat seq-before seq-after))
    (setq seq-after nil)
    (add-text-properties p (1+ p) (list 'remix-keyseq-before seq-before
					'remix-keyseq-after seq-after
					'remix-status 'remix-uncertain))

    (remix-input last)))

(defun remix-forward (&optional n)
  (interactive)
  (setq n (or n 1))
  (if (> n 0)
      (let* ((inhibit-read-only t)
	     (p (point))
	     (seq-before (get-text-property p 'remix-keyseq-before))
	     (seq-after (get-text-property p 'remix-keyseq-after))
	     new-before new-after)
	(if seq-after
	    (progn
	      (setq n (min n (length seq-after)))
	      (setq new-before (concat seq-before (substring seq-after 0 n)))
	      (setq new-after (substring seq-after n))
	      (if (zerop (length new-after))
		  (setq new-after nil))
	      (add-text-properties p (1+ p) (list 'remix-keyseq-before new-before
						  'remix-keyseq-after new-after))

	      (if new-after
		  (remix-update-fence 'remix-inedit)
		(remix-input)))
	  (message "ReMix: end of fence")
	  (or remix-quiet (ding))))))

(defun remix-backward (&optional n)
  (interactive)
  (setq n (or n 1))
  (if (> n 0)
      (let* ((inhibit-read-only t)
	     (p (point))
	     (seq-before (get-text-property p 'remix-keyseq-before))
	     (seq-after (get-text-property p 'remix-keyseq-after))
	     new-before new-after)
	(if seq-before
	    (progn
	      (setq n (- (min n (length seq-before))))
	      (setq new-before (substring seq-before 0 n))
	      (if (zerop (length new-before))
		  (setq new-before nil))
	      (setq new-after (concat (substring seq-before n) seq-after))
	      (add-text-properties p (1+ p) (list 'remix-keyseq-before new-before
						  'remix-keyseq-after new-after))

	      (remix-update-fence 'remix-inedit))
	  (message "ReMix: beginnig of fence")
	  (or remix-quiet (ding))))))

(defun remix-cancel-input ()
  "Cancel all of input"
  (interactive)
  (if (remix-in-fence-p)
      (let ((inhibit-read-only t)
	    (p (point)))
	(add-text-properties p (1+ p) (list 'remix-keyseq-before nil
					    'remix-keyseq-after nil))
	(remix-update-fence 'remix-default))))

(defun remix-delete (&optional n)
  (interactive)
  (setq n (or n 1))
  (if (> n 0)
      (let* ((inhibit-read-only t)
	     (p (point))
	     (seq-after (get-text-property p 'remix-keyseq-after)))
	(if seq-after
	    (progn
	      (setq n (min n (length seq-after))
		    seq-after (substring seq-after n))
	      (if (zerop (length seq-after))
		  (setq seq-after nil))
	      (add-text-properties p (1+ p) (list 'remix-keyseq-after seq-after))
	      (if seq-after
		  (remix-update-fence 'remix-inedit)
		(add-text-properties p (1+ p) '(remix-status remix-uncertain))
		(remix-input)))
	  (message "ReMix: end of fence")
	  (or remix-quiet (ding))))))

(defun remix-delete-backward (&optional n)
  (interactive)
  (setq n (or n 1))
  (if (> n 0)
      (let* ((inhibit-read-only t)
	     (p (point))
	     (seq-before (get-text-property p 'remix-keyseq-before))
	     (seq-after (get-text-property p 'remix-keyseq-after)))
	(if seq-before
	    (progn
	      (setq n (- (min n (length seq-before)))
		    seq-before (substring seq-before 0 n))
	      (if (zerop (length seq-before))
		  (setq seq-before nil))
	      (add-text-properties p (1+ p) (list 'remix-keyseq-before seq-before))
	      (if seq-after
		  (remix-update-fence 'remix-inedit)
		(add-text-properties p (1+ p) '(remix-status remix-uncertain))
		(remix-input)))
	  (message "ReMix: beginning of fence")
	  (or remix-quiet (ding))))))

(defun remix-kill-line ()
  (interactive)
  (let ((inhibit-read-only t))
    (add-text-properties (point) (1+ (point)) '(remix-keyseq-after nil))
    (remix-input)))

(defun remix-transpose-chars ()
  (interactive)
  (let* ((inhibit-read-only t)
	 (p (point))
	 (seq-before (get-text-property p 'remix-keyseq-before))
	 (seq-after (get-text-property p 'remix-keyseq-after))
	 new-before new-after)
    (cond
     ((null seq-before)			           ; $B%U%'%s%9$N@hF,$N$H$-(B
      (message "ReMix: beginning of fence")
      (or remix-quiet (ding)))
     ((= (length (concat seq-before seq-after)) 1) ; $B%U%'%s%9$K(B1$BJ8;z$7$+$J$$$H$-(B
      (message "ReMix: there is no char to transpose")
      (or remix-quiet (ding)))
     ((null seq-after)			           ; $B%U%'%s%9$NKvHx$N$H$-(B
      (setq new-before (concat (substring seq-before 0 -2)
			       (substring seq-before -1)
			       (substring seq-before -2 -1)))
      (add-text-properties p (1+ p) (list 'remix-keyseq-before new-before
					  'remix-status 'remix-uncertain))
      (remix-input))
     (t
      (setq new-before (concat (substring seq-before 0 -1)
			       (substring seq-after 0 1)
			       (substring seq-before -1))
	    new-after (substring seq-after 1))
      (if (zerop (length new-after))
	  (setq new-after nil))
      (add-text-properties p (1+ p) (list 'remix-keyseq-before new-before
					  'remix-keyseq-after new-after))
      (remix-update-fence 'remix-inedit)))))

(defun remix-select-kana ()
  "$B8=:_$N%U%'%s%9A4BN$r2>L>$G$"$k$H%f!<%6<-=q$KEPO?$9$k!#(B
$B$?$@$7!"<-=q$K$J$$>l9g$OL5;k$9$k(B"
  (interactive)
  (remix-end-of-input-buffer)
  (let* ((p (point))
	 (seq (get-text-property p 'remix-keyseq-before))
	 dict)
    (when (and seq
	       (setq dict (remix-seq-indict-p seq)))
      (save-excursion
	(set-buffer remix-usrdictbuf)
	(if (zerop dict)
	    (progn
	      (goto-char (point-min))
	      (insert (format "%s 1\n" seq)))
	  (goto-char dict)
	  (end-of-line)
	  (delete-region dict (point))
	  (insert "1"))
	(set-buffer-modified-p nil)
	(setq remix-usrdict-modified t))
      (remix-input))))

(defun remix-select-nokana ()
  "$B8=:_$N%U%'%s%9A4BN$r2>L>$G$O$J$$$b$N$H$7$F3NDj$9$k!#(B
$B%U%'%s%9$,%m!<%^;zDV$j$H$7$F@5$7$1$l$P%f!<%6<-=q$KEPO?$9$k!#(B"
  (interactive)
  (remix-end-of-input-buffer)
  (let* ((p (point))
	 (seq (get-text-property p 'remix-keyseq-before))
	 (status (get-text-property p 'remix-status))
	 dict)
    (if (and seq
	     (or (eq status 'remix-kanaseq)
		 (eq status 'remix-uncertain)
		 (eq status 'remix-indictkana)))
	(let ((inhibit-read-only t)
	      (remix-fence-operation t))
	  (delete-region (remix-search-beginning-seq) (remix-search-end-seq))
	  (insert seq)
	  (remix-put-cursor)
	  (save-excursion
	    (set-buffer remix-usrdictbuf)
	    (setq dict (remix-seq-indict-p seq))
	    (if (null dict)
		(progn
		  (goto-char (point-min))
		  (insert (format "%s 0\n" seq)))
	      (goto-char dict)
	      (end-of-line)
	      (delete-region dict (point))
	      (insert "0"))
	    (set-buffer-modified-p nil)
	    (setq remix-usrdict-modified t))))
    (remix-exit-mode-internal)))

(defun remix-hiragana ()
  (interactive)
  (remix-end-of-input-buffer)
  (let* ((inhibit-read-only t)
	 (p1 (remix-search-beginning-seq))
	 (p2 (remix-search-end-seq))
	 (str (buffer-substring p1 (1- p2))))
    (delete-region p1 p2)
    (remove-text-properties 0 (length str) (text-properties-at 0 str) str)
    (remix-exit-mode-internal (japanese-hiragana str))))

(defun remix-katakana ()
  (interactive)
  (remix-end-of-input-buffer)
  (let* ((inhibit-read-only t)
	 (p1 (remix-search-beginning-seq))
	 (p2 (remix-search-end-seq))
	 (str (buffer-substring p1 (1- p2))))
    (delete-region p1 p2)
    (remove-text-properties 0 (length str) (text-properties-at 0 str) str)
    (remix-exit-mode-internal (japanese-katakana str))))

(defun remix-full-width ()
  (interactive)
  (remix-end-of-input-buffer)
  (let* ((inhibit-read-only t)
	 (str (get-text-property (point) 'remix-keyseq-before))
	 (full "")
	 (len (length str))
	 (i 0)
	 lang)
    (delete-region (remix-search-beginning-seq) (remix-search-end-seq))
    (egg-separate-languages str)
    (setq lang (get-text-property 0 'egg-lang str))
    (while (< i len)
      (setq full (concat full (char-to-string
			       (symbol-value (intern-soft (format "%s%c" lang (aref str i))
							  its-half-full-table)))))
      (setq i (1+ i)))

    (remix-exit-mode-internal full)))

(defun remix-self-insert-char ()
  (interactive)
  (remix-start last-command-char (and (eq last-command 'egg-use-context)
				      egg-context)))

(defun remix-its-self-insert-char ()
  (interactive)
  (remix-input last-command-char))

(defun remix-start (key context)
  (let ((remix-context context))
    (remix-initialize)
    (remix-setup-fence-mode)
    (remix-input key)))

(defun remix-restart (str set-prop begnning context)
  (let* ((remix-context context)
	p cursor
	(prop (text-properties-at (point)))
	(seq (plist-get prop 'remix-keyseq-before))
	(kana (remix-translate-keyseq seq t)))
    (setq seq (remix-chop-head-keyseq seq (string-match (concat str "\\'") kana)))
    (if (zerop (length seq))
	(setq seq nil))
    (setq prop (plist-put prop 'remix-keyseq-before seq))
    (remix-delete-cursor)
    (remix-setup-fence-mode)
    (setq p (point))
    (add-text-properties p (1+ p) prop)
    (remix-update-fence 'remix-uncertain)))

(defun remix-exit-mode ()
  "Exit ReMix input"
  (interactive)
  (remix-exit-mode-internal))

(defun remix-abort ()
  "Cancel current input and signal a quit"
  (interactive)
  (remix-cancel-input)
  (signal 'quit nil))

(defun remix-exit-mode-internal (&optional key)
  (if (remix-in-fence-p)
      (let ((inhibit-read-only t)
	    (remix-fence-operation t)
	    (status (get-text-property (point) 'remix-status))
	    (overlay (get-text-property (point) 'remix-overlay)))
	(if overlay
	    (delete-overlay overlay))
	(remix-delete-cursor)
	(remix-delete-fence)
	(if key
	    (insert key)))))

(defun remix-input (&optional key)
  (let* ((inhibit-read-only t)
	 (p (point))
	 (seq-before (get-text-property p 'remix-keyseq-before))
	 (last nil)
	 status
	 dict)
    (if key
      (when (numberp key)
	(setq seq-before (concat seq-before (vector key)))
	(add-text-properties p (1+ p) (list 'remix-keyseq-before seq-before))))
    (when seq-before
      (setq dict (remix-seq-indict-p seq-before)) ; $B<-=q$K$"$kDV$j$+D4$Y$k(B
      (cond
       ((null dict)			          ; $B<-=q$K$O$J$$DV$j(B
	(setq status 'remix-uncertain))
       ((zerop dict)			          ; $B%7%9%F%`<-=q$K$"$kC18l(B($B%f!<%6<-=q$KEPO?$;$:(B)
	(setq status 'remix-indictnokana))
       (t
	(let ((flag (save-excursion
		      (set-buffer remix-usrdictbuf)
		      (char-after dict))))
	  (if (= flag ?0)
	      (setq status 'remix-indictnokana)   ; $B<-=qFb$K$"$kHs%m!<%^;zDV$j(B
	    (setq status 'remix-indictkana))))))  ; $B<-=qFb$K$"$k%m!<%^;zDV$j(B
    (remix-update-fence status (and key (symbolp key)))))

(defun remix-update-fence (status &optional last)
  (let* ((inhibit-read-only t)
	 (remix-fence-operation t)
	 (p (point))
	 (p1 (remix-search-beginning-seq))
	 (p2 (remix-search-end-seq))
	 (previous-status (get-text-property p 'remix-status))
	 (seq-before (get-text-property p 'remix-keyseq-before))
	 (seq-after (get-text-property p 'remix-keyseq-after))
	 (tr (if (or (eq previous-status 'remix-nokanaseq)
		     (eq status 'remix-indictnokana))
		  seq-before
		(remix-translate-keyseq seq-before last)))
	 (cursor (buffer-substring p (1+ p)))
	 (overlay (get-text-property 0 'remix-overlay cursor)))

    ;; font-lock-mode $B$,(B active $B$@$H$$$A$$$A(B point $B$,JQ2=$9$k$N$G(B
    ;; $B%U%'%s%9$N99?7$,=*$o$k$^$GL58z$K$9$k(B
    ;;;; $B$J$<$+(B combine-after-change-calls $B$,(B Emacs-21.0.97 $B$G$O5!G=$7$J$$$N$G(B
    ;;;; let $B$G$4$^$+$7(B
    ;; (combine-after-change-calls
    (let ((after-change-functions nil))
      ;; $B%U%'%s%9Fb$r$$$C$?$s%/%j%"$9$k(B
      (delete-region p1 p2)

      ;; $B%-!<%7!<%/%'%s%9$,6u$J$i!"%U%'%s%9$r>C5n$7$F%$%s%8%1!<%?$r99?7$9$k(B
      (if (zerop (length (concat seq-before seq-after)))
	  (remix-delete-fence)
	;; $B%+!<%=%k$N<jA0ItJ,$rA^F~$9$k(B
	(when tr
	  (insert tr)
	  (add-text-properties p1 (point) (list 'intangible 'remix-part-1
						'read-only t
						'face its-fence-face)))
	(setq p (point))
	;; $B%+!<%=%k$rA^F~$9$k(B
	(insert cursor)
	(setq p2 (point))
	;; $B%+!<%=%k$N8e$m$,$"$l$PA^F~$9$k(B
	(when seq-after
	  (insert seq-after)
	  (add-text-properties p2 (point) (list 'intangible 'remix-part-2
						'read-only t
						'face its-fence-face))
	  (setq p2 (point)))

	;; overlay $B$N:F@_Dj(B
	(if remix-use-overlay
	    (move-overlay overlay p1 p2))

	;; $B>uBV$r99?7$9$k(B
	(goto-char p)
	(if (eq status 'remix-uncertain)
	    (cond
	     ((null its-translation-result)
	      (setq status 'remix-nokanaseq))
	     ((equal its-translation-result tr)
	      (setq status 'remix-kanaseq))))
	(if (eq previous-status 'remix-inedit)
	    (setq status 'remix-inedit))
	(add-text-properties p (1+ p) (list 'remix-status status)))
      (setq current-input-method-title
	    (cdr (or (assq status remix-state-indicator-alist)
		     (assq 'remix-default remix-state-indicator-alist))))
      (force-mode-line-update))))

(defun remix-setup-fence-mode ()
  (let ((open-props '(remix-start t intangible remix-part-1 front-sticky nil))
	(close-props '(remix-end t intangible remix-part-2 rear-nonsticky t))
	(p0 (point))
	p1)
    (if (or (null (stringp its-fence-open)) (zerop (length its-fence-open))
	    (null (stringp its-fence-continue)) (zerop (length its-fence-continue))
	    (null (stringp its-fence-close)) (zerop (length its-fence-close)))
	(error "invalid fence"))
    ;; Put open-fence before inhibit-read-only to detect read-only
    (insert (if remix-context its-fence-continue its-fence-open))
    (let ((inhibit-read-only t)
	  (p1 (point))
	  overlay)
      (add-text-properties p0 p1 open-props)
      (insert its-fence-close)
      (add-text-properties p1 (point) close-props)
      (if its-fence-invisible
	  (put-text-property p0 (point) 'invisible t))
      (put-text-property p0 (point) 'read-only t)
      (its-define-select-keys remix-its-mode-map)
      (goto-char p1)
      (remix-put-cursor)
      (when remix-use-overlay
	(setq overlay (make-overlay p1 (1+ p1)))
	(overlay-put overlay 'face its-fence-face)
	(overlay-put overlay 'evaporate t)
	(overlay-put overlay 'priority remix-use-overlay)
	(add-text-properties p1 (1+ p1) (list 'remix-overlay overlay)))
      (if remix-context
	  (add-text-properties p1 (1+ p1) (list 'remix-context remix-context))))))

(defun remix-put-cursor ()
  (let ((p (point))
	(str (copy-sequence "!")))
    (set-text-properties 0 1 (list
			      'remix-cursor        t
			      'read-only           t
			      'invisible           t
			      'intangible          'remix-part-2
			      'point-entered       'egg-enter/leave-fence
			      'point-left          'egg-enter/leave-fence
			      'modification-hooks  '(egg-modify-fence)
			      'remix-keyseq-before nil
			      'remix-keyseq-afer   nil
			      'remix-status        'remix-default
			      'rear-nonsticky      t)
			 str)
    (insert str)
    (goto-char p)))

(defun remix-delete-cursor ()
  (when (get-text-property (point) 'remix-cursor)
    (delete-region (point) (1+ (point)))))

(defun remix-delete-fence ()
  (let ((inhibit-read-only t)
	p1 p2 p3)
    ;; Open fence $B$r:o=|(B
    (setq p2 (remix-search-beginning-seq))
    (setq p1 (remix-search-beginning-fence p2))
    (delete-region p1 p2)

    ;; $B%U%'%s%9Fb$N%W%m%Q%F%#$r:o=|(B
    (setq p2 (remix-search-end-seq))
    (remove-text-properties p1 p2 (text-properties-at p1))

    ;; Close fence $B$r:o=|(B
    (setq p3 (remix-search-end-fence p2))
    (delete-region p2 p3)

    ;; $B4A;zJQ49$KHw$($F!"%U%'%s%9$NN>C<$rJV$9(B
    (cons p1 p2)))

(defun remix-search-beginning-fence (&optional from)
  (let ((p (or from (point))))
    (previous-single-property-change p 'remix-start nil (point-min))))

(defun remix-search-beginning-seq (&optional from)
  (let ((p (or from (point))))
    (or (get-text-property (1- p) 'remix-start)
	(setq p (previous-single-property-change p 'remix-start)))
    p))

(defun remix-search-end-fence (&optional from)
  (let ((p (or from (point))))
    (next-single-property-change p 'remix-end nil (point-max))))

(defun remix-search-end-seq (&optional from)
  (let ((p (or from (point))))
    (or (get-text-property p 'remix-end)
	(setq p (next-single-property-change p 'remix-end)))
    p))

(defun remix-initialize ()
  "$BI,MW$J$i$P%7%9%F%`<-=q$H%f!<%6<-=q$rFI$_9~$_!";2>H$KHw$($k(B"
  (save-excursion
    ;; $B%7%9%F%`<-=q$N%;%C%F%#%s%0(B
    (if remix-sysdict
	(let ((dictfiles remix-sysdict)
	      file
	      (buf (get-buffer remix-sysdictbuf)))
	  (if (null buf)
	      (progn
		(setq buf (get-buffer-create remix-sysdictbuf))
		(set-buffer buf)
		(while (setq file (car dictfiles))
		  (if (file-readable-p file)
		      (progn
			(insert-file-contents file t)
			(goto-char (point-max))
			;; $B<-=q%U%!%$%k$NKvHx$,2~9T$G$O$J$$>l9g$KHw$($k(B
			(or (bolp)
			    (insert ?\n)))
		    (message (format
			      "ReMix: $BJs9p(B: $B;XDj$5$l$?%7%9%F%`<-=q(B (%s) $B$rFI$a$^$;$s$G$7$?(B"
			      file)))
		  (setq dictfiles (cdr dictfiles)))
		(set-buffer-modified-p nil)
		(setq buffer-read-only t))))) ; $B%7%9%F%`<-=q$O=q49$($J$$(B

    ;; $B%f!<%6<-=q$N%;%C%F%#%s%0(B
    (if (and remix-usrdict (null (get-buffer remix-usrdictbuf)))
	(let ((buf (find-file-noselect remix-usrdict t)))
	  (set-buffer buf)
	  (rename-buffer remix-usrdictbuf)
	  (make-local-variable 'make-backup-files)
	  (setq make-backup-files nil))))) ; $B%f!<%6<-=q%U%!%$%k$N%P%C%/%"%C%W$O:n$i$J$$(B


(defun remix-seq-indict-p (seq)
  "SEQ $B$,<-=q$K$"$k$+$I$&$+$rD4$Y$k!#(B
$B%f!<%6<-=q$K$"$C$?>l9g$O!"(BSEQ $B$NDV$j%U%i%0$N(B point $B$rJV$9!#(B
$B%7%9%F%`<-=q$K$"$C$?>l9g$O(B remix-auto-add-entry $B$K=>$C$F%f!<%6<-=q$K(B SEQ $B$rEPO?$7!"(B
$BDV$j%U%i%0$N(B point $B$rJV$9!#EPO?$7$J$+$C$?>l9g$O(B 0 $B$rJV$9!#(B
$B8+$D$+$i$J$1$l$P(B nil $B$rJV$9(B"
  (save-excursion
    (let ((entry (format "^%s " (regexp-quote seq))))
      (or (if remix-usrdict
	      (progn
		(set-buffer remix-usrdictbuf)
		(goto-char (point-min))
		(re-search-forward entry nil t)) ; $B%f!<%6<-=q$+$i8!:w(B
	    nil)
	  (if (< (length seq) 2)
	      nil			; 2$BJ8;zL$K~$O8!:w$7$J$$(B
	    (if remix-sysdict
		(progn
		  (set-buffer remix-sysdictbuf)
		  (goto-char (point-min))
		  (setq entry (format "^%s$" (regexp-quote seq)))
		  (if (re-search-forward entry nil t) ; $B%7%9%F%`<-=q$+$i8!:w(B
		      (if remix-auto-add-entry ; $B%(%s%H%j(B(seq)$B$r<+F0$GDI2C$9$k$+(B?
			  (progn
			    (set-buffer remix-usrdictbuf)
			    (insert (format "%s 0\n" seq))
			    (forward-word -1)
			    (set-buffer-modified-p nil) ;$B%(%s%H%j$NDI2CH/@8(B
			    (point))
			0)
		    nil))
	      nil))))))

(defun remix-kick-convert-region-or-self-insert ()
  "$B>u67$K1~$8$FJ8;z$rF~NO$9$k$+4A;zJQ49$r8F$S=P$9!#(B
$B%U%'%s%9$,2>L>$NJB$S$J$i$P4A;zJQ49$r8F$S=P$9!#(B
$B$5$b$J$1$l$P%U%'%s%9$r3NDj$7!"F~NO$7$?J8;z$r$=$N$^$^A^F~$9$k(B"
  (interactive)
  (remix-end-of-input-buffer t)
  (let ((status (get-text-property (point) 'remix-status)))
    (if (or (eq status 'remix-kanaseq)
	    (eq status 'remix-indictkana))
	;; $B4A;zJQ49(B
	(let ((inhibit-read-only t)
	      (cursor (buffer-substring (point) (1+ (point))))
	      region)
	  (remix-delete-cursor)
	  (save-excursion
	    (goto-char (remix-search-end-fence))
	    (insert cursor))
	  (setq region (remix-delete-fence))
	  (if (/= (car region) (cdr region))
	      (egg-convert-region (car region) (cdr region)
				  (get-text-property 0 'remix-context cursor))))
      ;; self-insert
      (let ((inhibit-read-only t)
	    (seq (get-text-property (point) 'remix-keyseq-before)))
	(delete-region (remix-search-beginning-seq) (remix-search-end-seq))
	(insert seq)
	(remix-exit-mode-internal (this-command-keys))
	(egg-do-auto-fill)))))

(defun remix-pool-conversion (newsyl oldsyl cursor)
  (its-update-latest-SYL newsyl)
  (if (and newsyl
	   (consp (cdr newsyl))
	   (not (its-kst-p (its-get-kst/t newsyl))))
      ;; DSYL
      (let ((output (its-get-output newsyl)))
	(setq its-translation-result (concat its-translation-result output))))
  cursor)

(defun remix-translate-keyseq (seq last &optional cutoff)
  (setq its-translation-result ""
	its-latest-SYL (its-initial-ISYL))
  (let ((i 0)
	(syl (its-initial-ISYL))
	;; temporally disable DING
	(its-barf-on-invalid-keyseq nil)
	(len (length seq))
	cursor)
    (if (null cutoff)
	(setq cutoff len))
    (while (< i cutoff)
      (let ((key (aref seq i))
	    (its-disable-special-action t))                   ; ASCII $BF~NO%b!<%I$rM^@)$9$k(B
	(if (null (its-keyseq-acceptable-p (vector key) syl)) ; state-machine $B$K$J$$DV$j$+(B?
	    (setq its-translation-result nil
		  its-latest-SYL nil
		  i cutoff)		                      ; $BJQ49$rCf;_$9$k(B
	  (setq cursor (its-state-machine syl key 'remix-pool-conversion))
	  (setq i (1+ i))
	  (if cursor
	      (setq syl (its-initial-ISYL))
	    (setq syl its-latest-SYL)))))
    (if (and last (not cursor))
	(setq cursor (its-state-machine syl -1 'remix-pool-conversion)))
    (if (null its-translation-result)
	seq
      (concat its-translation-result
	      (if (not cursor)
		  (car its-latest-SYL))
	      (if (< cutoff len)
		  (substring seq cutoff))))))

(defun remix-chop-head-keyseq (seq count)
  (let ((i 0)
	(syl (its-initial-ISYL))
	(its-barf-on-invalid-keyseq nil)
	(len (length seq))
	(loop 0)
	cursor)
    (while (and (< i len) (not (zerop count)))
      (let ((key (aref seq i))
	    (its-disable-special-action t))
	(setq loop 0
	      cursor (its-state-machine syl key
					(lambda (newsyl oldsyl cursor)
					  (its-update-latest-SYL newsyl)
					  (if (and newsyl
						   (consp (cdr newsyl))
						   (not (its-kst-p (its-get-kst/t newsyl))))
					      (setq count (1- count)))
					  (setq loop (1+ loop))
					  cursor)))
	(setq i (1+ i))
	(if cursor
	    (setq syl (its-initial-ISYL))
	  (setq syl its-latest-SYL))))
    (if (> loop 1)
	(setq i (1- i)))
    (substring seq i)))

(defun remix-save-dict ()
  (if (and remix-usrdict-modified
	   (get-buffer remix-usrdictbuf))
      (save-excursion
	(set-buffer remix-usrdictbuf)
	(write-file remix-usrdict))))

(defun remix-mode ()
  "\\{remix-its-mode-map}"
  ;; dummy function to get docstring
  )

(defun remix-mode-help-command ()
  "Display documentation for Egg ReMix mode."
  (interactive)
  (with-output-to-temp-buffer "*Help*"
    (princ "Egg ReMix mode:\n")
    (princ (documentation 'remix-mode))
    (help-setup-xref (cons #'help-xref-mode (current-buffer)) (interactive-p))))

;;; $B%U%C%/@_Dj(B
(add-hook 'input-method-after-insert-chunk-hook 'remix-delete-cursor)
(add-hook 'egg-enter/leave-fence-hook 'remix-enter/leave-fence)
(add-hook 'kill-emacs-hook 'remix-save-dict)

(provide 'egg-remix)

;;; egg-remix.el ends here
