/*////////////////////////////////////////////////////////////////////////
Copyright (c) 2002 National Institute of Advanced Industrial Science and Technology (AIST)

Permission to use this material for evaluation, copy this material for
your own use, and distribute the copies via publically accessible on-line
media, without fee, is hereby granted provided that the above copyright
notice and this permission notice appear in all copies.
AIST MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	SockMux (socket multiplexer)
Author:		Yutaka Sato <ysato@delegate.org>
Description:
	Multiplex local sockets and tunnel to remote via a single
	bidrectional communication channel.

    Server binding:
	SERVER=proto[://server],-in  server for connection incoming from remote

TODO:
 - SERVER="sockmux:shio:script.shio"
 - SERVER=proto[://server],-out server for outogoing
 - SERVER="proto[://server]-out(optionList):-:-Pxxxx"
 - substitute "CRON=..." parameter ?

 - resumable commin/commout for movile usage
 - multiple commin/commout with encryption or not
 - 7bit mode comm relayable over telnet
 - SockMux as hidden background exchange like Telnet commands
 - SockMux for FTP data-connections like MODE XDC
 - symbolic message version of SockMux
 - remote administration commands like SNMP

 - multiple queue with priority
 - give the highest priority to Telnet, the lowest one to FTP, etc.
 - QoS control

 - forwarding originator's (and forwarder's) identity in chained SockMux
 - forwarding originator's certificate (singed and verified, or encrypted)
 - C1 -> S <- C2 ... and C1 <-> C2, relay as a reflector
 - SENDME/IHAVE or CONNECT2vaddr + ACCEPTvaddr or SEND2vaddr+RECVvaddr
 - HELLO assoc-name
 - multicasting on SockMux, like ViaBus

 - UTF-8 encoding including header ?

History:
	021215	created
//////////////////////////////////////////////////////////////////////#*/
#include <stdio.h>
#include <errno.h>
#include "vsocket.h"
#include "ystring.h"
#include "log.h"
extern int ACC_TIMEOUT;
extern int CFI_DISABLE;
extern char *numscanX();
extern char *getADMIN();

/*
 *	SIMPLE SOCKET TUNNELING MULTIPLEXER PROTOCOL
 *	Version: 0.3
 */
#define MAJVER	0
#define MINVER	3
#define MYVER	((MAJVER<<4) | MINVER)

/*
 *	PROTOCOL SPECIFICATION
 */
typedef unsigned char octet1;
typedef struct {
	octet1	ph_ver;	/* protocol version */
	octet1	ph_type; /* packet type */
	octet1	ph_Said[2]; /* sender's id */
	octet1	ph_Raid[2]; /* receiver's id */
	octet1	ph_leng[2]; /* payload length */
} PackV03;
#define PHSIZE	sizeof(PackV03)
#define PWSIZE	512

typedef struct {
	PackV03	p_head;
	char	p_data[PWSIZE-PHSIZE];
} Packet;

#define SOX_ACK		0x40
#define SOX_ECHO	0x00 /* ECHO data */
#define SOX_HELLO	0x01 /* HELLO hostname */
#define SOX_CONNECT	0x02 /* CONNECT bufsize client clif [bindif?] */
#define SOX_CONNECT2	0x03 /* CONNECT TO ... specifying destination port */
#define SOX_BIND	0x04
#define SOX_ACCEPT	0x05
#define SOX_SEND	0x06 /* SEND    L# R# data */
#define SOX_SENDOOB	0x07 /* SENDOOB L# R# data */
#define SOX_CLOSE	0x08 /* CLOSE   L# R# reason */
#define SOX_HELLOOK	(SOX_ACK|SOX_HELLO)   /* HELLOOK hostname */
#define SOX_CONNECTED	(SOX_ACK|SOX_CONNECT) /* CONNECTED bufsize server svif */
#define SOX_SENT	(SOX_ACK|SOX_SEND)    /* SENT length */
#define SOX_SENTOOB	(SOX_ACK|SOX_SENDOOB) /* SENTOOB length */
#define SOX_CLOSED	(SOX_ACK|SOX_CLOSE)   /* CLOSED reason */

/* internal status */
#define SOX_COMMOUT	0x0100
#define SOX_COMMIN	0x0200
#define SOX_LISTEN	0x0300
#define SOX_WAITING	0x1000 /* waiting remote, stop local input */
#define SOX_SENDING	(SOX_WAITING|SOX_SEND)
#define SOX_SENDINGOOB	(SOX_WAITING|SOX_SENDOOB)

#define p_Ver	p_head.ph_ver
#define p_Type	p_head.ph_type
#define GET_INT16(a)	((a[0]<<8)|a[1])
#define SET_INT16(a,v)	{a[0] = v>>8; a[1] = v;}
#define PK_Leng(p)	GET_INT16((p)->p_head.ph_leng)
#define PK_setLeng(p,l)	SET_INT16((p)->p_head.ph_leng,l)
#define PK_Said(p)	GET_INT16((p)->p_head.ph_Said)
#define PK_setSaid(p,l)	SET_INT16((p)->p_head.ph_Said,l)
#define PK_Raid(p)	GET_INT16((p)->p_head.ph_Raid)
#define PK_setRaid(p,l)	SET_INT16((p)->p_head.ph_Raid,l)

/*
 *	PROTOCOL INTERPRETER
 */
#define NSOCK	256
#define MAXPENDING	4
#define MAXBUFSIZE	((MAXPENDING+2)*PWSIZE)
typedef struct {
	int	b_fd;	/* socket file descriptor */
	int	b_rem;	/* remaining in buffer */
	char	b_buf[MAXBUFSIZE];
} Buffer;

typedef struct {
  unsigned int  s_rrecvN; /* received from remote agent */
  unsigned int  s_rrecv;  /* bytes received */
  unsigned int  s_rsentN; /* sent to remote agent */
  unsigned int  s_rsent;  /* bytes sent */
  unsigned int	s_rpendN; /* pending status caused */
} Stats;

typedef struct {
	int	a_abid;	/* agent buffer id */
  unsigned char	a_Laid;	/* local agent id */
  unsigned char	a_Xaid;	/* remote agent id */
  unsigned int	a_rpend; /* bytes pending in remote */
  unsigned int	a_stat;	/* local agent status */
	char	a_remote; /* initiated by remote */
	Buffer	a_lbuf; /* buffer for output to local socket */
	int	a_pid;	/* process id */
	Stats	a_stats;
	short	a_commin; /* Laid of commin */
	short	a_commout; /* Laid of commout */
} Agent;

#define a_recv	a_stats.s_rrecv
#define a_recvN	a_stats.s_rrecvN
#define a_sent	a_stats.s_rsent
#define a_sentN	a_stats.s_rsentN
#define a_pendN	a_stats.s_rpendN
#define a_sock	a_lbuf.b_fd
#define a_lpend	a_lbuf.b_rem

typedef struct {
	int	s_clock;
	int	s_nlaid;
	Stats	s_stats;
	int	s_Rpstats[256];	/* per received packet type */

	char	s_serverurl[256];
	char	s_myname[256];
	char	s_infifo[256];
	char	s_outfifo[256];
	char	s_proc[256];
	FILE   *s_comminfp;
	int	s_ocommin;

	int	s_agentserno;
	int	s_agents;
	Agent	s_agentbv[NSOCK];
	Agent  *s_agentpv[NSOCK];
	char	s_aidactive[NSOCK];

	int	s_qfdset;
	int	s_qtimeout;
	int	s_qfdc;
	int	s_qfdv[NSOCK];
	int	s_qcxv[NSOCK];
	int	s_qqev[NSOCK];
	int	s_commout;
	int	s_relayOOB;
} Sox;

#define Trecv	sox->s_stats.s_rrecv
#define TrecvN	sox->s_stats.s_rrecvN
#define Tsent	sox->s_stats.s_rsent
#define TsentN	sox->s_stats.s_rsentN
#define TpendN	sox->s_stats.s_rpendN

#define Serverurl	sox->s_serverurl
#define Myname	sox->s_myname
#define infifo	sox->s_infifo
#define outfifo	sox->s_outfifo
#define Comminfp	sox->s_comminfp
#define Ocommin	sox->s_ocommin

#define AgentSerno	sox->s_agentserno
#define Agents	sox->s_agents
#define Agentbv	sox->s_agentbv
#define Agentpv	sox->s_agentpv

#define Qfdset	sox->s_qfdset
#define Qtimeout	sox->s_qtimeout
#define Qfdc	sox->s_qfdc
#define Qfdv	sox->s_qfdv
#define Qcxv	sox->s_qcxv
#define Qqev	sox->s_qqev

#define CommoutFd	sox->s_commout
#define CommOutAp	Agentpv[0]
#define RelayOOB	sox->s_relayOOB


static int dbglev = 0;
static FILE *logfp;
static dbgprintf(fmt,a,b,c,d,e,f,g,h,i,j,k,l,m,n)
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k,*l,*m,*n;
{	char xfmt[256];
	int sec,usec;

	if( logfp == NULL ){
		if( 0 < curLogFd() )
			logfp = fdopen(curLogFd(),"a");
		else	logfp = stderr;
	}
	sec = Gettimeofday(&usec);
	sprintf(xfmt,"{S}%02d.%03d ",sec%60,usec/1000);
	strcat(xfmt,fmt);
	strcat(xfmt,"\n");
	fprintf(logfp,xfmt,a,b,c,d,e,f,g,h,i,j,k,l,m,n);
	fflush(logfp);
}
static int (*dbglog)() = dbgprintf;
#define Trace	(*dbglog)
#define Debug	(dbglev==0)?0:(*dbglog)

static writeb(sox,Ap1,buf,len)
	Sox *sox;
	Agent *Ap1;
	char *buf;
{	int wcc,rem;
	int wccs;
	Buffer *Buf = &Ap1->a_lbuf;

	wcc = 0;
	if( Buf->b_rem ){
		wcc = write(Buf->b_fd,Buf->b_buf,Buf->b_rem);
		Debug("## FLUSHED %d/%d (+%d)",wcc,Buf->b_rem,len);
		if( wcc < Buf->b_rem ){
			if( wcc < 0 )
				wcc = 0;
			rem = Buf->b_rem;
			if( 0 < wcc ){
				rem = rem - wcc;
				bcopy(Buf->b_buf+wcc,Buf->b_buf,rem);
			}
			bcopy(buf,Buf->b_buf+rem,len);
			Buf->b_rem = rem + len;
			return wcc;
		}
		Qfdset = 0;
		Buf->b_rem = 0;
	}
	wccs = wcc;

	if( buf != 0 && len ){
		wcc = write(Buf->b_fd,buf,len);
		if( wcc < len ){
			Debug("## SENDING %d/%d",wcc,len);
			Qfdset = 0;
			if( wcc < 0 )
				wcc = 0;
			rem = len - wcc;
			bcopy(buf+wcc,Buf->b_buf,rem);
			Buf->b_rem = rem;
		}
		wccs += wcc;
	}
	return wccs;
}
static writecomm(sox,Apc,pack)
	Sox *sox;
	Agent *Apc;
	Packet *pack;
{	int wcc;

	pack->p_Ver = MYVER;
	wcc = writeb(sox,Apc,pack,PHSIZE+PK_Leng(pack));
	return wcc;
}
static writesock(sox,ApR,pack)
	Sox *sox;
	Agent *ApR;
	Packet *pack;
{ 	int wcc;

	if( pack == NULL ) /* just to flush pending */
		wcc = writeb(sox,ApR,NULL,0);
	else	wcc = writeb(sox,ApR,pack->p_data,PK_Leng(pack));
	return wcc;
}
static sock2comm(sox,Api,sock,Apc,pack,OOB)
	Sox *sox;
	Agent *Api,*Apc;
	Packet *pack;
{	int laid = Api->a_Laid;
	int raid = Api->a_Xaid;
	int rcc,wcc,type;

	errno = 0;
	if( OOB ){
		rcc = recvOOB(sock,pack->p_data,sizeof(pack->p_data));
		type = SOX_SENDOOB;
	}else{
		rcc = read(sock,pack->p_data,sizeof(pack->p_data));
		type = SOX_SEND;
	}
	if( rcc <= 0 ){
		Debug("## sock2comm() EOF rcc=%d from client L#%d[%d]",
			rcc,laid,sock);
		pack->p_Type = SOX_CLOSE;
		return rcc;
	}
	pack->p_Type = type;
	PK_setLeng(pack,rcc);
	PK_setSaid(pack,laid);
	PK_setRaid(pack,raid);
	if( OOB && RelayOOB ){
		wcc = sendOOB(Apc->a_sock,pack,PHSIZE+rcc);
	}else	wcc = write(Apc->a_sock,pack,PHSIZE+rcc);
	if( wcc <= 0 ){
		Trace("## sock2comm() write %d/%d",wcc,PHSIZE+rcc);
	}
	Debug(" sock input...pack(#%d %02X %d) %d",
		PK_Raid(pack),pack->p_Type,PK_Leng(pack),wcc);
	return wcc;
}
static recvpack(sox,infp,pack,OOB)
	Sox *sox;
	FILE *infp;
	Packet *pack;
{	int rcc,wcc,leng;

	if( OOB && RelayOOB ){
		rcc = recvOOB(fileno(infp),pack,PHSIZE);
	}else	rcc = fread(pack,1,PHSIZE,infp);
	if( rcc <= 0 ){
		return -1;
	}
	if( 0 < (leng = PK_Leng(pack)) )
	{
		if( OOB && RelayOOB )
			rcc += recvOOB(fileno(infp),&pack->p_data,leng);
		else	rcc += fread(&pack->p_data,1,leng,infp);
	}
	Debug(" comm input...pack(#%d %02X %d) %d",
		PK_Raid(pack),pack->p_Type,PK_Leng(pack),rcc);
	return rcc;
}

static set_STR(pack,str)
	Packet *pack;
	char *str;
{
	Strncpy(pack->p_data,str,sizeof(pack->p_data));
	PK_setLeng(pack,strlen(pack->p_data)+1);
}
static set_INT(pack,ival)
	Packet *pack;
{
	sprintf(pack->p_data,"%d",ival);
	PK_setLeng(pack,strlen(pack->p_data)+1);
}
static get_INT(pack)
	Packet *pack;
{	int ival;

	sscanf(pack->p_data,"%d",&ival);
	return ival;
}
/*
 * CONNECT N.bufsize A.client A.clientif
 */
static set_CONNECT(Apn,pack,remote,local)
	Agent *Apn;
	Packet *pack;
	char *remote,*local;
{	char pair[512];

	sprintf(pair,"%d %s %s",MAXBUFSIZE,remote,local);
	linescanX(pair,pack->p_data,sizeof(pack->p_data));
	PK_setLeng(pack,strlen(pack->p_data)+1);
}
static get_CONNECT(pack,remote,local)
	Packet *pack;
	char *remote,*local;
{	int bsiz;

	bsiz = 0;
	remote[0] = local[0] = 0;
	sscanf(pack->p_data,"%d %s %s",&bsiz,remote,local);
}

static send_HELLO(sox,out,ack,pack)
	Sox *sox;
	Buffer *out;
	Packet *pack;
{	int wcc;

	pack->p_Ver = MYVER;
	pack->p_Type = ack?SOX_HELLOOK:SOX_HELLO;
	set_STR(pack,Myname);
	wcc = write(out->b_fd,pack,PHSIZE+PK_Leng(pack));
	return wcc;
}
static send_ACK(sox,Apc,pack)
	Sox *sox;
	Agent *Apc;
	Packet *pack;
{	int tid;

	pack->p_Type |= SOX_ACK;
	tid  = PK_Said(pack);
	PK_setSaid(pack,PK_Raid(pack));
	PK_setRaid(pack,tid);
	writecomm(sox,Apc,pack);
}

static newlaid(sox)
	Sox *sox;
{	int ci;

	for( ci = sox->s_nlaid; ci < NSOCK-1; ci++ )
		if( sox->s_aidactive[ci] == 0 )
			goto EXIT;

	for( ci = 0; ci < sox->s_nlaid; ci++ )
		if( sox->s_aidactive[ci] == 0 )
			goto EXIT;

	return -1;
EXIT:
	sox->s_aidactive[ci] = 1;
	sox->s_nlaid = ci + 1;
	return ci;
}
static Agent *newAgent(sox,raid,sock,stat)
	Sox *sox;
{	Agent *Apn;
	int bid;

	AgentSerno++;
	Apn = Agentpv[Agents++];
	bid = Apn->a_abid;
	bzero(Apn,sizeof(Agent));
	Apn->a_abid = bid;

	Apn->a_Laid = newlaid(sox);
	Apn->a_Xaid = raid;
	Apn->a_sock = sock;
	Apn->a_stat = stat;
	return Apn;
}
static delAgent(sox,ci)
	Sox *sox;
{	int cj;
	Agent *Ap1;

	if( ci != Agents-1 ){
		Ap1 = Agentpv[ci];
		for( cj = ci; cj < Agents-1; cj++ ){
			Agentpv[cj] = Agentpv[cj+1];
		}
		Agentpv[Agents-1] = Ap1;
	}
	Agents--;
}
static shutdownAgents(sox)
	Sox *sox;
{	int ci;
	Agent *Ap1;

	Trace("SHUTDOWN");
	for( ci = 0; ci < Agents; ci++ ){
		Ap1 = Agentpv[ci];
		if( Ap1->a_stat != SOX_LISTEN )
		{
			Trace("L#%d close %d",Ap1->a_Laid,Ap1->a_sock);
			close(Ap1->a_sock);
			Ap1->a_sock = -1;
			Ap1->a_stat = SOX_CLOSED;
		}
	}
}

static connects(ctx,local,remote,itvl)
	void *ctx;
	char *local,*remote;
{	int sock;

	for(;;){
		sock = VSocket(ctx,"CNCT/SOX*",-1,local,remote,"self");
		if( 0 <= sock)
			break;
		sleep(itvl);
	}
	Trace("CNCT %d %s -> %s",sock,local,remote);
	return sock;
}
static sox_init(ctx,ac,av,comminp,commoutp,server,ssiz)
	void *ctx;
	char *av[];
	int *comminp,*commoutp;
	char *server;
{	int commin,commout;
	int ai;
	char *arg,*hp;
	int stat;
	char local[256],remote[256],tmp[256];
	int sock;

	commin = *comminp;
	commout = *commoutp;
	for( ai = 0; ai < ac; ai++ ){
	    arg = av[ai];
	    if( strncasecmp(arg,"-A",2) == 0 ){
		/* accept commin/commout */
		remote[0] = 0;
		sock = VSocket(ctx,"BIND/SOX*",-1,arg+2,remote,"listen=1");
		Trace("BIND %d",sock);

		ACC_TIMEOUT = 0;
		local[0] = remote[0] = 0;
		commin = commout =
			VSocket(ctx,"ACPT/SOX*",sock,local,remote,"");
		Trace("ACPT %d",commin);

		close(sock);
	    }else
	    if( strncasecmp(arg,"-C",2) == 0 ){ /* connect commin/commout */
		int citvl = 5;

		hp = arg+2;
		if( *hp == '-' ){
			hp = numscanX(hp+1,tmp,sizeof(tmp));
			citvl = atoi(tmp);
			if( citvl <= 0 )
				citvl = 1;
			if( *hp == '-' )
				hp++;
		}
		strcpy(local,"*:*");
		commin = commout = connects(ctx,local,hp,citvl);
	    }else
	    if( strncasecmp(arg,"-B",2) == 0 ){
		/* accept() to be issued by remote agent */
		sock = VSocket(ctx,"BIND/SOX*",-1,arg+2,remote,"listen=8");
		Trace("## BIND for rmote: %d",sock);
	    }else
	    if( strncasecmp(arg,"-X",2) == 0 ){
		commin = commout = open(arg+2,2);
		Trace("COMM IN/OUT=%d %s",commin,arg+2);
	    }else
	    if( strncasecmp(arg,"-I",2) == 0 ){
		commin = open(arg+2,2);
		Trace("COMM IN=%d %s",commin,arg+2);
	    }else
	    if( strncasecmp(arg,"-O",2) == 0 ){
		commout = open(arg+2,2);
		Trace("COMM OUT=%d %s",commout,arg+2);
	    }else
	    if( strncasecmp(arg,"-Q",2) == 0 ){
		server[0] = 0;
		if( strstr(arg,"://") == 0 )
			wordscanX("tcprelay://",server,ssiz);
		wordscanX(arg+2,server+strlen(server),ssiz-strlen(server));
	    }else
	    if( strncasecmp(arg,"-v",2) == 0 ){
		switch(arg[2]){
			case 'd':
				dbglev = 1;
				break;
		}
	    }
	}
	*comminp = commin;
	*commoutp = commout;
}
static getserver(ctx,proto,serverurl,from,rclif,mbox,msiz)
	void *ctx;
	char *proto,*serverurl,*from,*rclif,*mbox;
{	char mboxb[256];
	char *dp,addr[256];
	int port;

	addr[0] = 0;
	port = 0;
	if( from != NULL ){
		if( dp = wordscanY(from,addr,sizeof(addr),"^:") )
		if( *dp == ':' )
			port = atoi(dp+1);
	}
	VA_setClientAddr(ctx,addr,port,1);

	HL_setClientIF(NULL,0,1);
	if( rclif != NULL ){
		wordscanX(rclif,addr,sizeof(addr));
		if( dp = strchr(addr,':') ){
			port = atoi(dp+1);
			*dp = 0;
		}else	port = 0;
		HL_setClientIF(addr,port,1);
	}

	if( mbox == 0 ){
		mbox = mboxb;
		msiz = sizeof(mboxb);
	}
	mbox[0] = 0;
	if( from == 0 ){
		wordscanX("-@-",mbox,msiz);
	}else{
		if( strchr(from,'@') ){
			wordscanX(from,mbox+strlen(mbox),msiz-strlen(mbox));
		}else{
			wordscanX("-@",mbox+strlen(mbox),msiz-strlen(mbox));
			wordscanX(from,mbox+strlen(mbox),msiz-strlen(mbox));
		}
	}

	setClientCert(ctx,"SOX",mbox);
	set_realproto(ctx,proto);
	if( find_CMAP(ctx,"XSERVER",serverurl) < 0 )
	{
		return 0;
	}
	return 1;
}

/*
 * accept from local socket, then connect at remote host
 */
static sox_accept1(sox,ctx,Api,pack)
	Sox *sox;
	void *ctx;
	Agent *Api;
	Packet *pack;
{	char local[256],remote[256];
	Agent *Apn;
	int clsock;
	int aid;
	int wcc;

	strcpy(local,"*:*");
	strcpy(remote,"*:*");
	errno = 0;
	clsock = VSocket(ctx,"ACPT/SOX",Api->a_sock,local,remote,"");
	Trace("L#%d accept(%d)=%d [%s]<-[%s] errno=%d",
		Api->a_Laid,Api->a_sock,clsock,remote,local,errno);
	if( clsock < 0 )
		return;

	set_ClientSock(ctx,clsock,remote,local);
	if( !source_permitted(ctx) ){
		Trace("Forbidden %s",remote);
		close(clsock);
		return;
	}

	setNonblockingIO(clsock,1);
	set_nodelay(clsock,1);

	Apn = newAgent(sox,-1,clsock,SOX_CONNECT);
	aid = Apn->a_Laid;
	Apn->a_remote = 0;
	Qfdset = 0;

	pack->p_Type = SOX_CONNECT;
	PK_setSaid(pack,aid);
	PK_setRaid(pack,-1);
	set_CONNECT(Apn,pack,remote,local);
	Trace("CONN> L#%d[%s]",aid,pack->p_data);
	wcc = writecomm(sox,CommOutAp,pack);
}
/*
 * SERVER=proto://host:port or SERVER=proto
 */
static sox_connect1(ctx,pack,serverURL,remote,local,pidp)
	void *ctx;
	Packet *pack;
	char *serverURL,*remote,*local;
	int *pidp;
{	int clsock,logfd;
	char servU[256],proto[64],serv[256],opt[256],param[256],mbox[256];
	char client[256],clif[256];
	int sockv[2];
	char *dp,*ep;
	int pid;
	int ai;

	linescanX(serverURL,servU);
	get_CONNECT(pack,client,clif);
	getserver(ctx,"vp_in",servU,client,clif,mbox,sizeof(mbox));

	if( servU[0] == 0 ){
		Trace("No server");
		return -1;
	}

	Trace("SERVER=%s",servU);
	param[0] = 0;
	if( dp = strchr(servU,'(') ){
		wordscanY(dp+1,param,sizeof(param),"^)");
		if( ep = strchr(servU,')')  )
			strcpy(dp,ep+1);
	}
	if( dp = strchr(servU,',') ){
		wordscanX(dp,opt,sizeof(opt));
		*dp = 0;
	}
	proto[0] = serv[0] = 0;
	sscanf(servU,"%[^:]://%[^/?]",proto,serv);
	if( serv[0] && strchr(serv,':') == 0 )
	if( !streq(serv,"-") && !streq(serv,"-.-") )
	{
		sprintf(serv+strlen(serv),":%d",serviceport(proto));
	}
	Trace("SERVER=%s://%s/%s(%s):-:%s",proto,serv,opt,param,mbox);

	if( streq(serv,"-") || streq(serv,"-.-") || param[0] ){
		char *av[32],ab[1024],*ap;
		int ac,fd,fd0,fd1,fd2;

		INET_Socketpair(sockv);
		clsock = sockv[0];
		setserversock(sockv[1]);
		logfd = curLogFd();
		for( fd = 0; fd < NSOCK; fd++ ){
			if( fd != sockv[1] )
			if( fd != logfd )
			if( fd != fileno(stderr) )
				setCloseOnExec(fd);
		}
		ac = 0;
		ap = ab;
		sprintf(ap,"SERVER=%s://%s",proto,serv);
		av[ac++] = ap; ap += strlen(ap) + 1;
		sprintf(ap,"-P-%s",clif);
		av[ac++] = ap; ap += strlen(ap) + 1;
		sprintf(ap,"-L0x%x/%d",LOG_type,logfd);
		av[ac++] = ap; ap += strlen(ap) + 1;
		sprintf(ap,"ADMIN=%s",getADMIN());
		av[ac++] = ap; ap += strlen(ap) + 1;

		/*
		sscanf(pack->p_data,"%s %s",rclient,rhost);
		sprintf(ap,"_remoteport=%s",rsvd);
		av[ac++] = ap; ap += strlen(ap) + 1;
		*/

		if( param[0] ){
			ac += decomp_args(&av[ac],32-ac,param,ap);
		}
		av[ac] = 0;
		for( ai = 0; ai < ac; ai++ ){
			Debug("[%d] %s",ai,av[ai]);
		}
		fd0 = dup(0); fd1 = dup(1); fd2 = dup(2);
		dup2(sockv[1],0);
		dup2(sockv[1],1);
		dup2(fileno(stderr),2);
		pid = spawnv_self1(ac,av);
		close(sockv[1]);
		dup2(fd0,0); dup2(fd1,1); dup2(fd2,2);
		close(fd0); close(fd1); close(fd2);
		Trace("Fork pid=%d",pid);
		*pidp = pid;
	}else{
		/* this connect should be in non-blocking */
		local[0] = 0;
		clsock = VSocket(ctx,"CNCT/SOX",-1,local,serv,"self");
		*pidp = 0;
	}
	if( 0 <= clsock ){
		gethostName(clsock,local,"%A:%P");
		getpeerName(clsock,remote,"%A:%P");
	}
	return clsock;
}
static getcommpath(fifo,path)
	char *fifo,*path;
{	char *dp;

	*path = 0;
	if( dp = strstr(fifo,"@") )
	{
		strcpy(path,dp+1);
	}
}
Sox *sox_setup1(ctx)
	void *ctx;
{	char fifo[256];
	Sox *sox;

	sox = (Sox*)calloc(1,sizeof(Sox));

	if( LOG_type & L_VERB)
		dbglev = 1;

	bzero(sox,sizeof(Sox));
	gethostname(Myname,sizeof(Myname));
	Ocommin = -1;
	Qtimeout = -1; /* infinite wait for PollInsOuts() */

	if( getserver(ctx,"vp_comm",fifo,NULL,NULL,NULL,0) ){
		getcommpath(fifo,infifo);
		strcpy(outfifo,infifo);
		Trace("comm=%s [%s]",infifo,fifo);
	}
	if( getserver(ctx,"vp_commin",fifo,NULL,NULL,NULL,0) ){
		getcommpath(fifo,infifo);
		Trace("commin=%s [%s]",infifo,fifo);
	}
	if( getserver(ctx,"vp_commout",fifo,NULL,NULL,NULL,0) ){
		getcommpath(fifo,outfifo);
		Trace("commout=%s [%s]",outfifo,fifo);
	}
	if( getserver(ctx,"vp_proc",fifo,NULL,NULL,NULL,0) ){
		getcommpath(fifo,sox->s_proc);
		Trace("proc=%s",sox->s_proc);
	}
	return sox;
}

sox_setup(ac,av,ctx,svsock,svport,sox)
	char *av[];
	void *ctx;
	Sox *sox;
{	char *isvproto,*isvhost;
	int isvport;
	int stat;
	int commin;
	int ci;
	int rcode;
	Packet packb,*pack = &packb;
	char local[256],remote[256];
	Agent *Ap1,*Apn,*Apb;
	int wcc,rcc,pid;
	int sockc,portv[NSOCK],sockv[NSOCK];
	Buffer out;

	isvport = CTX_get_iserver(ctx,&isvproto,&isvhost);
	sockc = getReservedPorts(portv,sockv);

RETRY:
	CFI_DISABLE = 0;
	if( 0 <= Ocommin ){
		Trace("## Close old commin hidden by FCL: %d",Ocommin);
		close(Ocommin);
		Ocommin = -1;
	}
	if( Comminfp != NULL ){
		Trace("## Close Comminfp %d/%X",fileno(Comminfp),Comminfp);
		fclose(Comminfp);
		Comminfp = NULL;
	}
	while( 0 < (pid = NoHangWait()) ){
		Trace("Exit pid=%d",pid);
	}
	set_ClientSock(ctx,-1);

	commin = CommoutFd = -1;
	if( commin < 0 && sox->s_proc[0] ){
		int sio[2];
		if( procSocket(ctx,sox->s_proc,sio) == 0 ){
			commin = sio[0];
			CommoutFd = sio[1];
		}
		Trace("PROC [%d,%d] %s",commin,CommoutFd,sox->s_proc);
	}
	if( commin < 0 && infifo[0] ){
		commin = open(infifo,2);
		Trace("COMMIN %d %s",commin,infifo);
	}
	if( CommoutFd < 0 && outfifo[0] ){
		if( strcmp(infifo,outfifo) == 0 )
			CommoutFd = commin;
		else{
			CommoutFd = open(outfifo,2);
			Trace("COMMOUT %d %s",CommoutFd,outfifo);
		}
	}
	if( commin < 0 )
	if( 0 < isvport && !streq(isvhost,"-") && !streq(isvhost,"-.-") )
	/* SERVER=sockmux://host:port given */
	{
		local[0] = 0;
		sprintf(remote,"%s:%d",isvhost,isvport);
		commin = CommoutFd = connects(ctx,local,remote,3);
	}
	if( commin < 0 && 0 < svport )
	/* -Pport given */
	{
		ACC_TIMEOUT = 0;
		local[0] = remote[0] = 0;
		commin = CommoutFd =
			VSocket(ctx,"ACPT/SOX*",svsock,local,remote,"");
		Trace("ACPT %d %s <- %s",commin,local,remote);
		set_ClientSock(ctx,commin);
	}

	sox_init(ctx,ac,av,&commin,&CommoutFd,Serverurl,sizeof(Serverurl));
	Trace("SOX reception[%d/%d] reserved(%d) channel[%d,%d]",
		svport,svsock,sockc,commin,CommoutFd);

	if( commin < 0 ){
		return -1;
	}
	if( file_issock(commin) ){
		int fcl;
		fcl = insertFCL(ctx,commin);
		if( 0 <= fcl ){
			Trace("Filter inserted: %d -> %d",fcl,commin);
			Ocommin = commin;
			commin = CommoutFd = fcl;
		}
	}
	Comminfp = fdopen(commin,"r");
	if( Comminfp == NULL ){
		Trace("## cannot open %d\n",commin);
		return -1;
	}
	out.b_fd = CommoutFd;
	out.b_rem = 0;

	CFI_DISABLE = 1;
	RES_CACHEONLY(1);

	/* must do exclusive lock here ... */
	wcc = send_HELLO(sox,&out,0,pack);
	Tsent += wcc;
	TsentN++;

	stat = 0;
	for( ci = 0; ci < 100; ci++ ){
		rcc = recvpack(sox,Comminfp,pack,0);
		Trecv += rcc;
		TrecvN++;
		stat = pack->p_Type;
		Trace("waiting HELLO V%02X > %02X",pack->p_Ver,stat);

		if( pack->p_Ver != MYVER ){
			Trace("SockMux protocol version mismatch: %X / %X",
				pack->p_Ver,MYVER);
			close(CommoutFd);
			sleep(3);
			goto RETRY;
		}

		if( stat == SOX_HELLO
		 || stat == SOX_HELLOOK
		){
			if( stat == SOX_HELLOOK )
				Trace("Got HELLO OK[%s]",pack->p_data);
			else	Trace("Got HELLO");
			send_HELLO(sox,&out,1,pack);

			if( stat == SOX_HELLOOK )
				break;
		}
	}
	if( stat != SOX_HELLOOK ){
		Trace("no HELLOOK");
		return -1;
	}

	/*
	 * set Nonblocking after HELLO done, or do HELLO after PollIn()
	 */
	/*
	setNonblockingIO(CommoutFd,1);
	*/
	set_nodelay(CommoutFd,1);

	for( ci = 0; ci < NSOCK; ci++ ){
		Ap1 = &Agentbv[ci];
		bzero(Ap1,sizeof(Agent));
		Agentpv[ci] = Ap1;
		Ap1->a_abid = ci;
		Ap1->a_sock = -1;
		Ap1->a_Laid = -1;
		Ap1->a_Xaid = -1;
		sox->s_aidactive[ci] = 0;
	}
	Agents = 0;

	Apn = newAgent(sox,-1,CommoutFd,SOX_COMMOUT);
	Trace("L#%d %d COMMOUT",Apn->a_Laid,Apn->a_sock);

	Apn = newAgent(sox,-1,commin,SOX_COMMIN);
	Trace("L#%d %d COMMIN(-P%d)",Apn->a_Laid,Apn->a_sock,
		sockPort(0<=Ocommin?Ocommin:commin));

	for( ci = 0; ci < sockc; ci++ ){
		Apn = newAgent(sox,-1,sockv[ci],SOX_LISTEN);
		Trace("L#%d %d PORT=%d",Apn->a_Laid,Apn->a_sock,portv[ci]);
	}
	return 0;
}

sox_main(ac,av,ctx,svsock,svport)
	char *av[];
	void *ctx;
{	Sox soxb,*sox = &soxb;

	sox = sox_setup1(ctx);
START:
	if( sox_setup(ac,av,ctx,svsock,svport,sox) < 0 )
		goto EXIT;

	Debug("---- START ----");
	Qfdset = 0;
	for(;;){
		if( sox_clock1(sox,ctx) < 0 )
			goto START;
	}
EXIT:
	Finish(0);
}

sox_clock1(sox,ctx)
	Sox *sox;
	void *ctx;
{	int nready;
	int rev[NSOCK];
	int ri,rx;
	Agent *Api;
	Packet packb,*pack = &packb;

	sox->s_clock++;

	if( Qfdset == 0 )
		sox_fdset(sox);

	Debug("%d poll(%d/%d) S=%d/%d R=%d/%d",sox->s_clock,Qfdc,Agents,
		Tsent,TsentN,Trecv,TrecvN);

	if( 0 < ready_cc(Comminfp) ){
		rx = -1;
		for( ri = 0; ri < Qfdc; ri++ ){
			rev[ri] = 0;
			if( Qfdv[ri] == fileno(Comminfp) ){
				rx = ri;
			}
		}
		if( 0 <= rx ){
			ri = rx;
			Api = Agentpv[Qcxv[ri]];
			rev[ri] = PS_IN; 
			goto COMMIN;
		}
	}

	nready = PollInsOuts(10*1000,Qfdc,Qfdv,Qqev,rev);
	if( nready == 0 ){
		/*
		Trace("poll(%d) = %d",Qfdc,0);
		for( ri = 0; ri < Agents; ri++ ){
			Api = Agentpv[ri];
			Trace("L#%d R#%d stat=%X pending-R=%d,L=%d",
				Api->a_Laid,Api->a_Xaid,
				Api->a_stat,
				Api->a_pend,
				Api->a_lbuf.b_rem);
		}
		for( ri = 0; ri < 256; ri++ ){
			if( sox->s_Rpstats[ri] ){
				Trace("%02X %d",ri,sox->s_Rpstats[ri]);
			}
		}
		*/
		nready = PollInsOuts(Qtimeout*1000,Qfdc,Qfdv,Qqev,rev);
	}
	if( nready <= 0 ){
		Trace("TIMEOUT");
		return -1;
	}

	for( ri = 0; ri < Qfdc; ri++ ){
		Api = Agentpv[Qcxv[ri]];
		if( rev[ri] == 0 )
			continue;

		if( rev[ri] & PS_OUT ){
			int rem,sent;
			rem = Api->a_lbuf.b_rem;
			writesock(sox,Api,NULL);
			if( sent = rem - Api->a_lbuf.b_rem ){
				pack->p_Type = SOX_SENT;
				PK_setRaid(pack,Api->a_Xaid);
				PK_setSaid(pack,Api->a_Laid);
				set_INT(pack,sent);
				writecomm(sox,CommOutAp,pack);
				Trace("## FLUSHED=%d L#%d stat=%X",
					sent,Api->a_Laid,Api->a_stat);
				Api->a_stat = SOX_SENT;
				Qfdset = 0;
			}
		}
		if( (rev[ri] & (PS_IN|PS_PRI)) == 0 )
			continue;

		if( Api->a_stat == SOX_COMMIN )
		COMMIN:{
		/* relay from remote to local */
			if( rev[ri] & PS_PRI ){
				if( sox_COMMIN(sox,ctx,pack,1) < 0 ){
					return -1;
				}
			}
			if( rev[ri] & PS_IN )
				if( sox_COMMIN(sox,ctx,pack,0) < 0 )
					return -1;
			continue;
		}
		if( Api->a_stat == SOX_LISTEN ){
		/* accept at reception socket(s) */
			sox_accept1(sox,ctx,Api,pack);
			continue;
		}
		/* relay local sockets to remote */
		if( rev[ri] & PS_PRI )
			sox_relay1(sox,Api,Qfdv[ri],pack,1);
		if( rev[ri] & PS_IN )
			sox_relay1(sox,Api,Qfdv[ri],pack,0);
	}
	return 0;
}

sox_fdset(sox)
	Sox *sox;
{	int ci,cj;
	Agent *Ap1;
	int rcode;
	int qev;

	for( ci = 0; ci < Agents; ci++ ){
		Ap1 = Agentpv[ci];
		if( Ap1->a_stat == SOX_CLOSED ){
			rcode = close(Ap1->a_sock);
	Trace("Fin L#%d[%d]%d S=%d/%d,%d/%d P=%d/%d R=%d/%d,%d/%d",
				Ap1->a_Laid,Ap1->a_abid,Ap1->a_sock,
				Ap1->a_sent,Ap1->a_sentN,Tsent,TsentN,
				Ap1->a_pendN,TpendN,
				Ap1->a_recv,Ap1->a_recvN,Trecv,TrecvN
			);
			Ap1->a_sock = -1;

			if( Ap1->a_pid ){
				int pid;
				while( 0 < (pid = NoHangWait()) ){
					Trace("Exit pid=%d",pid);
				}
			}
			sox->s_aidactive[Ap1->a_Laid] = 0;
			delAgent(sox,ci);
		}
	}
	Qfdc = 0;
	for( ci = 0; ci < Agents; ci++ ){
		Ap1 = Agentpv[ci];
		qev = 0;

		if( Ap1->a_stat != SOX_COMMOUT )
		if( Ap1->a_stat != SOX_CONNECT )
		if( Ap1->a_stat != SOX_CLOSE )
		if( Ap1->a_stat != SOX_SENDING )
		if( Ap1->a_stat != SOX_SENDINGOOB )
		{
			qev |= (PS_IN | PS_PRI);
		}
		if( Ap1->a_lbuf.b_rem ){
			qev |= PS_OUT;
		}
		if( qev ){
			Qcxv[Qfdc] = ci;
			Qfdv[Qfdc] = Agentpv[ci]->a_sock;
			Qqev[Qfdc] = qev;
			Qfdc++;
		}
	}
	Qfdset = 1;
}
sox_relay1(sox,Api,sin,pack,OOB)
	Sox *sox;
	Agent *Api;
	Packet *pack;
{	int wcc;

	wcc = sock2comm(sox,Api,sin,CommOutAp,pack,OOB);
	if( 0 < wcc ){
		Tsent += wcc;
		TsentN++;
		Api->a_sent += PK_Leng(pack);
		Api->a_sentN++;
		Api->a_rpend += PK_Leng(pack);
		if( OOB ){
			TpendN++;
			Api->a_pendN++;
			Api->a_stat = SOX_SENDINGOOB;
		}else{
			if( PWSIZE*MAXPENDING < Api->a_rpend ){
				TpendN++;
				Api->a_pendN++;
				Api->a_stat = SOX_SENDING;
			}else	Api->a_stat = SOX_SEND;
		}
		Qfdset = 0;
		Debug("SEND> L#%d -> R#%d (%d) pending=%d",
			Api->a_Laid,Api->a_Xaid,PK_Leng(pack),Api->a_rpend);
	}else
	if( errno == EAGAIN ){
		Trace("EAGAIN for read()");
	}else{
		Api->a_stat = SOX_CLOSE;
		if( 0 < Api->a_lbuf.b_rem ){
			Trace("##CLOSE>## L#%d discard pending output(%d)",
				Api->a_Laid,Api->a_lbuf.b_rem);
			Api->a_lbuf.b_rem = 0;
		}
		Qfdset = 0;

		pack->p_Type = SOX_CLOSE;
		PK_setSaid(pack,Api->a_Laid);
		PK_setRaid(pack,Api->a_Xaid);
		set_STR(pack,"DO close");
		wcc = writecomm(sox,CommOutAp,pack);
		Trace("CLOSE> L#%d -> R#%d (rcc=%d errno=%d)",
			Api->a_Laid,Api->a_Xaid, wcc,errno);
	}
}
sox_COMMIN(sox,ctx,pack,OOB)
	Sox *sox;
	void *ctx;
	Packet *pack;
{	int rcc;
	int stat;
	Agent *ApR,*Apn;
	int pid;
	int ci;
	int clsock;
	int aid;
	char local[256],remote[256],pair[512];
	int wcc;

	rcc = recvpack(sox,Comminfp,pack,OOB);
	if( rcc < 0 ){
		shutdownAgents(sox);
		return -1;
	}
	Trecv += rcc;
	TrecvN++;
	stat = pack->p_Type;
	sox->s_Rpstats[stat] += 1;

	ApR = 0;
	for( ci = 0; ci < Agents; ci++ ){
		if( Agentpv[ci]->a_Laid == PK_Raid(pack) ){
			ApR = Agentpv[ci];
			ApR->a_recv += PK_Leng(pack);
			ApR->a_recvN++;
			break;
		}
	}
	switch( stat ){
	case -1:
		Trace("BROKEN CONNECTION");
		/* broken connection */
		return -1;

	case SOX_HELLO:
		Trace("HELLO");
		shutdownAgents(sox);
		return -1;

	case SOX_HELLOOK:
		Trace("Got HELLO OK[%s]",pack->p_data);
		break;

	case SOX_CONNECT:
		Debug("Connect< R#%d[%s]",PK_Said(pack),pack->p_data);
		clsock = sox_connect1(ctx,pack,Serverurl,remote,local,&pid);
		if( clsock < 0 )
			goto SOXCLOSE;

		if( 0 <= clsock ){
			setNonblockingIO(clsock,1);
			set_nodelay(clsock,1);
			Apn = newAgent(sox,PK_Said(pack),clsock,SOX_CONNECTED);
			aid = Apn->a_Laid;
			Apn->a_remote = 1;
			Apn->a_pid = pid;
			Qfdset = 0;
			Trace("Connected %d L#%d[%s] <- R#%d[%s]",
				clsock,aid,local,PK_Said(pack),pack->p_data);
			PK_setRaid(pack,Apn->a_Laid);
			set_CONNECT(Apn,pack,remote,local);
			send_ACK(sox,CommOutAp,pack);
		}else SOXCLOSE:{
			pack->p_Type = SOX_CLOSE;
			PK_setRaid(pack,PK_Said(pack));
			PK_setSaid(pack,-1);
			set_STR(pack,"No server");
			wcc = writecomm(sox,CommOutAp,pack);
		}
		break;

	case SOX_CONNECTED:
		Trace("CONNed< L#%d <- R#%d[%s]",
			PK_Raid(pack),PK_Said(pack),pack->p_data);
		if( ApR ){
			ApR->a_Xaid = PK_Said(pack);
			ApR->a_stat = SOX_CONNECTED;
			Qfdset = 0;

/*
reserve port for BIND (data connection) for each CONNECT
L#NL BIND PL
R#NR BIND PR
FTP resp. PORT PL
accept at PL by L#NL
forward to R#NR
R#NR connect to PR
ready L#NL and R#NR

strcpy(rsvdport,"*:*");
sock = VSocket(ctx,"BIND/SOX",-1,rsvdport,remote,"listen=1");
Apb = newAgent(sox,-1,sock,SOX_LISTEN);
Apn->a_remote = 0;
Qfdset = 0;

or run SERVER=ftp at the entrance ?
*/
		}
		break;

	case SOX_SENDOOB:
		if( ApR ){
			wcc = sendOOB(ApR->a_sock,pack->p_data,PK_Leng(pack));
			Trace("SendOOB< L#%d <- R#%d (%d/%d)",
				PK_Raid(pack),PK_Said(pack),wcc,PK_Leng(pack));
			set_INT(pack,wcc);
			send_ACK(sox,CommOutAp,pack);
		}else{
			/* remove the obsolete remote agent by SOX_CLOSED */
		}
		break;

	case SOX_SENTOOB:
		if( ApR ){
			int sent = get_INT(pack);
			Trace("SENTOOB< L#%d/%X <- R#%d (%d) pending-R=%d,L=%d",
				PK_Raid(pack),ApR->a_stat,PK_Said(pack),sent,
				ApR->a_rpend,ApR->a_lpend);
			ApR->a_stat = SOX_SENTOOB;
			ApR->a_rpend -= sent;
			Qfdset = 0;
		}
		break;

	case SOX_SEND:
		Debug("Send< L#%d <- R#%d",PK_Raid(pack),PK_Said(pack));
		if( ApR ){
			if( ApR->a_stat == SOX_CLOSE ){
				Trace("##Send<## L#%d(CLOSED) R%d discard(%d)",
				PK_Raid(pack),PK_Said(pack),PK_Leng(pack));
				break;
			}
			wcc = writesock(sox,ApR,pack);
			if( wcc <= 0 ){
				break;
			}
			set_INT(pack,wcc);
			if( !OOB )
			send_ACK(sox,CommOutAp,pack);
		}else{
			/* return SOX_CLOSED */
		}
		break;

	case SOX_SENT:
		if( ApR ){
			int sent = get_INT(pack);
			ApR->a_rpend -= sent;
			if( ApR->a_rpend < PWSIZE*MAXPENDING ){
				if( ApR->a_stat == SOX_SENDING ){
					Debug("## RESUMED> L#%d",ApR->a_Laid);
				}
				ApR->a_stat = SOX_SENT;
				Qfdset = 0;
			}
			Debug("SENT< L#%d <- R#%d (%d) pending=%d",
			PK_Raid(pack),PK_Said(pack),sent,ApR->a_rpend);
		}else{
			/* return SOX_CLOSED */
		}
		break;

	case SOX_CLOSE:
		Debug("Close< L#%d <- R#%d",PK_Raid(pack),PK_Said(pack));
		if( ApR ){
			ApR->a_stat = SOX_CLOSED;
			Qfdset = 0;
		}
		set_STR(pack,"OK closed");
		send_ACK(sox,CommOutAp,pack);
		break;

	case SOX_CLOSED:
		Debug("CLOSED< L#%d <- R#%d",PK_Raid(pack),PK_Said(pack));
		if( ApR ){
			ApR->a_stat = SOX_CLOSED;
			Qfdset = 0;
		}
		break;

	default:
		Trace("unknown %d",stat);
		break;
	}
	return 0;
}
