/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1997-1999 Yutaka Sato
Copyright (c) 1997-1999 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, and distribute this material for any purpose
and without fee is hereby granted, provided that the above copyright
notice and this permission notice appear in all copies.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	smtpgw.c (SMTP to NNTP gateway)
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:

    WHAT IS THIS ?

	SMTP->{NNTP,SMTP} relay

		will be extended to

		NNTP->{NNTP,SMTP}
		SMTP->FTP
		...

    CONFIGURATION FILES

	DELEGATE_SMTPGATE/{admin,users}/To/conf  -- configuration file
	DELEGATE_SMTPGATE/{admin,users}/To/count -- counter file
	DELEGATE_SMTPGATE/{admin,users}/To/log   -- log file ?
	DELEGATE_SMTPGATE/admin/@default/conf    -- default configuration file

    QUESTION ?

	SERVER: nntp://host:port/newsgroup
		- partial inheritance becomes difficult in this format...

OWNER ... execute in the list owners effective UID ?

History:
	970829	created
//////////////////////////////////////////////////////////////////////#*/
#include <stdio.h>
#include <ctype.h>
#include "delegate.h"
#include "ystring.h"
#define LNSIZE  1024
#define lfprintf	SMTP_lfprintf

extern char *RFC822_fgetsHeaderField();
extern char *fgetsHeaderField();
extern char *fgetsTimeout();
extern char *getFieldValue2();
#define getFieldValue(str,fld,buf,siz) getFieldValue2(str,fld,buf,siz)
#define getFV(str,fld,buf)             getFieldValue2(str,fld,buf,sizeof(buf))

extern char *DELEGATE_ADMIN;
extern char *strip_spaces();
extern char *nextField();
extern char *malloc();
extern char *calloc();
extern FILE *SMTP_getEXPN();
extern char *gethostaddr();
extern FILE *TMPFILE();

#define POSTNEWS	"postnews"
#define SENDMAIL	"sendmail"
#define FTPMAIL		"ftpmail"
#define HTTPMAIL	"httpmail"

#define DEFAULT		"@default"

#define V_SENDER	"${sender}"
#define V_FROM		"${from}"
#define V_UNIXFROM	"${unixfrom}"
#define V_RECIPIENT	"${recipient}"
#define V_RECNAME	"${recipient.name}"
#define V_RECMX		"${recipient.mx}"
#define V_SUBJECT	"${subject}"
#define V_CLEANSUBJ	"${subject:hc}"
#define V_SEQNO		"${seqno}"
#define V_SEQNO10	"${seqno/10}"
#define V_SEQNO100	"${seqno/100}"
#define V_TIMEFORM	"${date+"
#define V_ERRORSTAT	"${error.status}"
#define V_ORIGHEADER	"${header."
#define V_PID		"${pid}"
#define V_SELF		"${self}"


#define GWTYPE			 0
#define INHERIT			 1

#define ACC_SENDER		 2
#define ACC_RECIPIENT		 3
#define ACC_FROM		 4
#define ACC_TO			 5

#define OUT_NEWSGROUPS		 6
#define OUT_DISTRIBUTION	 7
#define OUT_REPLY_TO		 8
#define OUT_TO			 9
#define OUT_SUBJECT		10
#define OUT_HEADER		11
#define OUT_FILTER		12

#define CTL_RECIPIENT		13
#define BCC			14
#define OPTION			15
#define SERVPROTO		16
#define SERVHOST		17
#define SERVPORT		18
#define COUNTER			19
#define ARCHIVE			20

#define ACC_MAX_BYTES		21
#define ACC_MIN_BYTES		22
#define ACC_MIN_BODY_BYTES	23
#define ACC_MAX_EXCLAMS		24
#define ACC_CLIENT_HOST		25
#define ACC_MESSAGE_ID		26
#define ACC_USERTEXT		27

#define CTL_SENDER		28
#define CTL_MYAUTH		29
#define NPARAMS			30 /* last element above + 1 */

typedef struct {
	char	*p_abbr;
	char	 p_pfix_opt;	/* prefix is optional */
	char	*p_pfix;
	char	*p_name;
} PSPEC;

static PSPEC param_spec[NPARAMS] = {
	{"gw",	1,"CONTROL",	"GATEWAY-TYPE"	},
	{"ih",	1,"CONTROL",	"INHERIT"	},

	{"se",	1,"ACCEPT",	"Sender"	},
	{"re",	0,"ACCEPT",	"Recipient"	},
	{"fr",	1,"ACCEPT",	"From"		},
	{"to",	0,"ACCEPT",	"To"		},

	{"ng",	1,"OUTPUT",	"Newsgroups"	},
	{"db",	1,"OUTPUT",	"Distribution"	},
	{"rt",	1,"OUTPUT",	"Reply-To"	},
	{"ot",  1,"OUTPUT",	"To"		},
	{"os",  1,"OUTPUT",	"Subject"	},
	{"oh",  1,"OUTPUT",	"Header"	},
	{"of",  1,"OUTPUT",	"FILTER"	},

	{"rp",	1,"CONTROL",	"RECIPIENT"	},
	{"bc",	1,"CONTROL",	"BCC"		},
	{"op",	1,"CONTROL",	"OPTION"	},
	{"sv",	1,"CONTROL",	"SERVER-PROTO"	},
	{"sh",	1,"CONTROL",	"SERVER-HOST"	},
	{"sp",	1,"CONTROL",	"SERVER-PORT"	},

	{"ct",	1,"CONTROL",	"COUNTER"	},
	{"ar",  1,"CONTROL",	"ARCHIVE"	},
	{"sz",	1,"ACCEPT",	"Max-Bytes"	},
	{"mb",	1,"ACCEPT",	"Min-Bytes"	},
	{"nb",	1,"ACCEPT",	"Min-Body-Bytes"},
	{"me",	1,"ACCEPT",	"Max-Exclams"	},
	{"ch",	1,"ACCEPT",	"Client-Host"	},
	{"mi",	1,"ACCEPT",	"Message-Id"	},
	{"at",  1,"ACCEPT",	"User-Text"	},

	{"cs",	0,"CONTROL",	"SENDER"	},
	{"ma",	1,"CONTROL",	"MYAUTH"	},
};
static param_idx(pname)
	char *pname;
{	int px;
	PSPEC *ps;
	char *dp;
	char pfix[128],name[128];

	pfix[0] = name[0] = 0;
	if( dp = strchr(pname,'/') )
		sscanf(pname,"%[^/]/%s",pfix,name);
	else	strcpy(name,pname);

	for( px = 0; px < NPARAMS; px++ ){
		ps = &param_spec[px];
		if( pfix[0] == 0 ){
			if( !ps->p_pfix_opt )
				continue;
		}else{
			if( !strcaseeq(pfix,ps->p_pfix) )
				continue;
		}
		if( strcasecmp(name,ps->p_name) == 0 )
			return px;
		if( strcasecmp(name,ps->p_abbr) == 0 )
			return px;
	}
	return -1;
}

typedef struct gateway {
	char	*g_pv[NPARAMS];
	char	 g_dontovw[NPARAMS];	/* don't overwrite */
	char	*g_outheaders[32];
	int	 g_outheaderN;
struct gateway	*g_parent;		/* parent class ... */

	char	*g_sender;
	char	*g_recipient;
	char	*g_rname; /* [xxx%] LOCAL [@domain] part of g_recipient */

	int	 g_option;
	char	 g_gwt[32];
	int	 g_count;

	FILE	*g_ts;
	FILE	*g_fs;
	char	*g_filter; /* saved global filter environment */
	char	 g_replyto[LNSIZE];
	char	 g_subject[LNSIZE];
} Gateway;

#define gv(px)		Gw->g_pv[px]
#define getp(px)	(gv(px) ? gv(px) : NULL)
#define getv(px)	((gv(px) && *gv(px)) ? gv(px) : NULL)

#define Option		Gw->g_option
#define Gwt		Gw->g_gwt
#define Sender		Gw->g_sender
#define Recipient	Gw->g_recipient
#define Rname		Gw->g_rname
#define Seqno		Gw->g_count
#define TS		Gw->g_ts
#define FS		Gw->g_fs
#define GFILTER		Gw->g_filter
#define ReplyTo		Gw->g_replyto
#define Subject		Gw->g_subject


static char *builtin_src[][8] = {
	/* MAIL TO NEWS */
	{
		"GATEWAY-TYPE: postnews",
		"SERVER-PROTO: nntp",
		"SERVER-HOST:  localhost",
		"Newsgroups:   junk",
		"Distribution: local",
	},
	/* MAIL TO MAIL */
	{
		"GATEWAY-TYPE: sendmail",
		"SERVER-PROTO: smtp",
		"SERVER-HOST:  localhost",
	},
	/* MAIL TO FTP */
	{
		"GATEWAY-TYPE: ftpmail",
		"SERVER-PROTO: ftp",
		"SERVER-HOST:  localhost",
	},
	/* MAIL TO HTTP */
	{
		"GATEWAY-TYPE: httpmail",
		"SERVER-PROTO: http",
		"SERVER-HOST:  localhost",
	},
	0
};
static Gateway *builtins;


static Gateway *new_conf()
{
	return (Gateway*)calloc(sizeof(Gateway),1);
}
static Gateway *builtinGw(gwname,silent)
	char *gwname;
{	int gwi,pi,px;
	int nlen;
	char *name;

	if( gwname == NULL ){
		sv1log("#### NO GATEWAY TYPE SPECIFIED\n");
		return NULL;
	}
	if( builtins == NULL ){
		builtins = (Gateway*)calloc(sizeof(Gateway),8);
	}
	if( builtins[0].g_pv[0] == NULL ){
		char **lines,*line,name[256],value[256],*vp;

		for( gwi = 0; (lines = builtin_src[gwi])[0]; gwi++ )
		for( pi = 0; line = lines[pi]; pi++ ){
			RFC822_decompField2(line,name,value,sizeof(value));
			vp = strip_spaces(value);
			px = param_idx(name);
			Verbose("SMTPGATE[%d][%d]=[%2d:%14s]=[%s]\n",
				gwi,pi,px,name,vp);
			if( px < 0 ){
				sv1log("#### UNKNOWN FILED IN BUILTIN SPEC: %s\n",
					line);
				return NULL;
			}
			builtins[gwi].g_pv[px] = stralloc(vp);
		}
	}

	for( gwi = 0; name = builtins[gwi].g_pv[GWTYPE];  gwi++ ){
		nlen = strlen(name);
		if( strncasecmp(gwname,name,nlen) == 0 )
			if( gwname[nlen] == ':' || gwname[nlen] == 0 )
				return &builtins[gwi];
	}
	if( !silent )
		sv1log("#### unknown GATEWAY TYPE: %s\n",gwname);
	return NULL;
}
static copy_ifundef(Gw,Gws)
	Gateway *Gw,*Gws;
{	int pi;
	char *p1,*d1;

	for( pi = 1; pi < NPARAMS; pi++ )
		if( (p1 = getp(pi)) == NULL && (d1 = Gws->g_pv[pi]) )
			Gw->g_pv[pi] = d1;
}

static scan_conf(Gw,fp,log)
	FILE *fp;
	Gateway *Gw;
	FILE *log;
{	int size,rcc;
	char *buff;
	char *line,*next,lbuf[1024],*comment;
	char name[1024],*np,value[2048],*vp;
	int px;

	size = file_size(fileno(fp));
	buff = malloc(size+1);
	rcc = fread(buff,1,size,fp);
	buff[rcc] = 0;

	for( line = buff; line && *line; line = next ){
		next = nextField(line,1);
		if( (comment = strpbrk(line,"#\r\n")) && *comment == '#' ){
			if( comment == line )
				continue;
			else	*comment = 0;
		}

		RFC822_decompField2(line,name,value,sizeof(value));
		if( np = strchr(name,'/') )
			np = np + 1;
		else	np = name;
		vp = strip_spaces(value);

		if( 0 <= (px = param_idx(name)) ){
			if( px == OUT_HEADER ){
				lfprintf(log,NULL,"conf + %s/%s[%d]:%s\r\n",
					param_spec[px].p_pfix,param_spec[px].p_name,
					Gw->g_outheaderN,vp);
				Gw->g_outheaders[Gw->g_outheaderN++] = stralloc(vp);
			}else
			if( Gw->g_pv[px] == 0 || px == INHERIT ){
				lfprintf(log,NULL,"conf + %s/%s:%s\r\n",
					param_spec[px].p_pfix,param_spec[px].p_name,vp);
				Gw->g_pv[px] = stralloc(vp);
			}else{
				lfprintf(log,NULL,"conf - %s:%s\r\n",name,vp);
			}
		}else{
			lfprintf(log,NULL,"unknown -- %s:%s\r\n",name,vp);
		}
	}

	free(buff);
}

static char *user_class[8] = {
	"admin",
	"users",
	"",
	0
};

extern char *DELEGATE_SMTPGATE;
static char *SMTPGATE_DIR;

FILE *open_file(name,type,mode,rpath)
	char *name,*type,*mode,*rpath;
{	char cpath[1024];
	char *class;
	FILE *fp;
	int ui;

	if( SMTPGATE_DIR == NULL ){
		sv1log("No SMTPGATE directory specified.\n");
		return NULL;
	}

	for( ui = 0; class = user_class[ui]; ui++ ){
		sprintf(rpath,"%s/%s/%s",class,name,type);
		sprintf(cpath,"%s/%s",SMTPGATE_DIR,rpath);
		if( fp = fopen(cpath,mode) )
			return fp;
	}
	return NULL;
}

static load_conf(Conn,Gw,name,afp,sender,recipient,log)
	Connection *Conn;
	Gateway *Gw;
	char *name;
	FILE *afp;
	char *sender,*recipient;
	FILE *log;
{	char rpath[1024];
	FILE *cfp;

	if( SMTPGATE_DIR == NULL ){
		lfprintf(log,NULL,"No SMTPGATE directory specified.\n");
		return 0;
	}

	cfp = open_file(name,"conf","r",rpath);
	if( cfp != NULL ){
		lfprintf(log,NULL,"load config. file -- SMTPGATE/%s\n",rpath);
		scan_conf(Gw,cfp,log);
		fclose(cfp);
		return 1;
	}else{
		lfprintf(log,NULL,"cannot read config. file -- SMTPGATE/%s\n",
			rpath);
		return 0;
	}
}

/*
 *	SMTPGATE=dir[:list-of-subdirs]
 */
extern int (*LOG_substfile)();
scan_SMTPGATE(Conn,conf)
	Connection *Conn;
	char *conf;
{	char pb[1024];
	char *pv[NPARAMS+1],*p0,*p1;
	int pc,pi;
	int gwi;
	char dir[1024];

	if( builtinGw(POSTNEWS,0) == NULL ){
		sv1tlog("ERROR IN BUILTIN SPECIFICATION\n");
		Finish(1);
	}

	strcpy(pb,conf);
	pc = stoV(pb,NPARAMS,pv,':');
	p0 = pv[0];
	p1 = pv[1];

	strcpy(dir,p0);
	(*LOG_substfile)(dir,"",NULL,NULL,NULL);
	if( !fileIsdir(dir) ){
		sv1tlog("Warning: non existent SMTPGATE=%s [%s]\n",conf,dir);
		return -1;
	}else{
		sv1tlog("SMTPGATE=%s [%s]\n",conf,dir);
		SMTPGATE_DIR = stralloc(dir);
		return 0;
	}
}

/*
 * OPTION
 */
#define RXO	0x01	/* Remove X-Original headers */
#define AXO	0x02	/* Append X-Original headers */
#define RES	0x04	/* Reject Empty Subject */
#define RNI	0x08	/* Reject No message-ID message */
#define REB	0x10	/* Reject Empby Body */
#define GWT	0x20	/* GateWay Trace */
#define ISN	0x40	/* Increment Sequence Number */
#define INH	0x80	/* INHibit to accepting at this name */
#define NTR	0x100	/* do NOT add trace field (Received:) */

#define CHK_ACC_TO	"tmn"	/* check To: name matching */
#define MSG_NUMBER	"msn"	/* message number by posted article number */

static scan_options(opts,gwt,log)
	char *opts,*gwt;
	FILE *log;
{	int opti;
	char optb[1024];
	char *ov[32],*o1;
	int oc,oi;
	char msg[1024];

	opti = 0;
	strcpy(optb,opts);
	if( strchr(optb,',') )
		oc = stoV(optb,32,ov,',');
	else	oc = stoV(optb,32,ov,'.');

	for( oi = 0; oi < oc; oi++ ){
		o1 = strip_spaces(ov[oi]);

		if( strcaseeq(o1,"res") ){
			sprintf(msg,"Reject-Empty-Subject");
			opti |= RES;
		}else
		if( strcaseeq(o1,"reb") ){
			sprintf(msg,"Reject-Empty-Body");
			opti |= REB;
		}else
		if( strcaseeq(o1,"rni") ){
			sprintf(msg,"Reject-No-messegaeID");
			opti |= RNI;
		}else
		if( strcaseeq(o1,"rxo") ){
			sprintf(msg,"Remove-X-Original");
			opti |= RXO;
		}else
		if( strcaseeq(o1,"axo") ){
			sprintf(msg,"Append-X-Original");
			opti |= AXO;
		}else
		if( strcaseeq(o1,"gwt") ){
			sprintf(msg,"Gateway-Trace");
			opti |= GWT;
			strcpy(gwt,o1);
		}else
		if( strcaseeq(o1,"isn") ){
			sprintf(msg,"Increment-Sequence-Number");
			opti |= ISN;
		}else
		if( strcaseeq(o1,"ntr") ){
			sprintf(msg,"DoNOT-add-Trace-Field");
			opti |= NTR;
		}else{
			sprintf(msg,"unknown, ignored.",o1);
		}
		lfprintf(log,NULL,"OPTION: %s -- %s ON\r\n",o1,msg);
	}
	return opti;
}

static append_log(Conn,Gw,ts,log)
	Connection *Conn;
	Gateway *Gw;
	FILE *ts,*log;
{	char proto[32];
	char *what;

	if( (Option & GWT) == 0 )
		return;

	fflush(log);
	ftruncate(fileno(log),ftell(log));
	fseek(log,0,0);

	strtoupper(DST_PROTO,proto);
	if( strcasecmp(proto,"smtp") != 0 )
		what = "gateway";
	else	what = "relay";

fprintf(ts,"\r\n");
fprintf(ts,"--MAILGATE %s/SMTP %s DeleGate/%s--\r\n",proto,what,DELEGATE_ver());
fprintf(ts,"Comment: This part including the CRLF at two lines above is a trace\r\n");
fprintf(ts,"  information appended by DeleGate which relayed this messaged from\r\n");
fprintf(ts,"  SMTP to %s.  The tracing is activated\r\n",proto);
 if( Gwt[0] ){
 fprintf(ts,"  by set \"gateway-trace\" option (%s) specified in the\r\n",Gwt);
 fprintf(ts,"  gateway for recipient address <%s>.\n",Recipient);
 }else{
 fprintf(ts,"  by a\r\n");
 fprintf(ts,"   X-Gateway-Control: trace-on\r\n");
 fprintf(ts,"  field included in the original message.\r\n");
 }
fprintf(ts,"\r\n");
copyfile1(log,ts);
fprintf(ts,"--\r\n");
}


subject_headclean(subj)
	char *subj;
{	char sc,*sp,*dp,tmp[2048];
	int withRe,numBracket;

	withRe = 0;
	numBracket = 0;

	for( sp = subj; sc = *sp; ){
		if( sc == ' ' || sc == '\t' ){
			sp++;
		}else
		if( sc == '[' && numBracket == 0 ){
			numBracket = 1;
			for( sp++; sc = *sp; sp++ )
				if( sc == ']' ){
					sp++;
					break;
				}
		}else
		if( strncasecmp(sp,"Re:",3) == 0 ){
			withRe++;
			sp += 3;
		}else
		if( strncasecmp(sp,"Re^",3) == 0 ){
			withRe++;
			for( sp += 3; *sp && isdigit(*sp); sp++ )
				;
			if( *sp == ':' )
				sp++;
		}else{
			break;
		}
	}
EXIT:
	strcpy(tmp,sp);
	subj[0] = 0;
	if( withRe )
		strcat(subj,"Re: ");
	strcat(subj,tmp);
}

extern char *TIMEFORM_ANSI_C;
static int starttime;
static make_unixfrom(unixfrom,sender)
	char *unixfrom;
{	char stime[128];

	StrftimeLocal(stime,sizeof(stime),TIMEFORM_ANSI_C,starttime,0);
	sprintf(unixfrom,"%s %s",sender,stime);
}
extern char *TIMEFORM_RFC822;
static put_trace(Conn,Gw,log,ts,recipients)
	Connection *Conn;
	Gateway *Gw;
	FILE *log,*ts;
{	char stime[128],clif[256];

	StrftimeLocal(stime,sizeof(stime),TIMEFORM_RFC822,time(0),0);
	ClientIF_name(Conn,FromC,clif);
	lfprintf(log,ts,
		"Received: from %s by %s (DeleGate/%s) for %s (%s); %s\r\n",
		Client_Host,clif,DELEGATE_ver(),recipients,Recipient,stime);
}

/*
 *  PATH NAME SUBSTITUTION
 *
 *     %5${seqno}
 *     %5${seqno/100}
 */
static subst_path(Gw,fmt,spath)
	Gateway *Gw;
	char *fmt,*spath;
{	char *fp,*np,*op,val[1024],numb[2];
	char tfmt[1024],*tp;
	int width,num;

	op = spath;
	for( fp = fmt; *fp; fp = np ){
		width = 0;
		np = fp;
		if( *fp == '%' && isdigit(fp[1]) ){
			numb[0] = fp[1];
			numb[1] = 0;
			width = atoi(numb);
			np = fp + 2;
		}

		if( strncmp(np,V_SEQNO,strlen(V_PID)) == 0 ){
			np += strlen(V_PID);
			num = getpid();
			goto PUTNUM;
		}
		if( strncmp(np,V_SEQNO,strlen(V_SEQNO)) == 0 ){
			np += strlen(V_SEQNO);
			num = Seqno;
			goto PUTNUM;
		}
		if( strncmp(np,V_SEQNO10,strlen(V_SEQNO10)) == 0 ){
			np += strlen(V_SEQNO10);
			num = (Seqno/10)*10;
			goto PUTNUM;
		}
		if( strncmp(np,V_SEQNO100,strlen(V_SEQNO100)) == 0 ){
			np += strlen(V_SEQNO100);
			num = (Seqno/100)*100;
			goto PUTNUM;
		}
		if( strncmp(np,V_TIMEFORM,strlen(V_TIMEFORM)) == 0 ){
			tp = tfmt;
			for( np += strlen(V_TIMEFORM); *np; np++ ){
				if( *np == '}' )
					break;
				*tp++ = *np;
			}
			if( *np == '}' ){
				*tp = 0;
				StrftimeLocal(val,sizeof(val),tfmt,time(0));
				np++;
				goto PUTSTR;
			}
		}
		*op++ = *fp;
		np = fp + 1;
		continue;

	PUTNUM:
		if( width )
			sprintf(val,"%0*d",width,num);
		else	sprintf(val,"%d",num);
	PUTSTR:
		strcpy(op,val);
		op += strlen(op);
	}
	*op = 0;
}

static substitute(Gw,mfp,src,dst,log,sender)
	Gateway *Gw;
	char *src,*dst;
	FILE *log;
	char *sender;
{	char tmp[1024],val[1024];
	char unixfrom[1024];
	char *bp,*ep,*sp,*dp;

	strcpy(dst,src);
	strcpy(tmp,dst);

	if( strstr(dst,V_ERRORSTAT) ){
		strsubst(dst,V_ERRORSTAT,"something wrong...");
		lfprintf(log,NULL,"substitute -- %s => %s\n",tmp,dst);
	}
	if( strstr(dst,V_RECIPIENT) ){
		strsubst(dst,V_RECIPIENT,Recipient);
		lfprintf(log,NULL,"substitute -- %s => %s\n",tmp,dst);
	}
	if( strstr(dst,V_RECNAME) ){
		strsubst(dst,V_RECNAME,Rname);
		lfprintf(log,NULL,"substitute -- %s => %s\n",tmp,dst);
	}
	if( strstr(dst,V_SENDER) ){
		strsubst(dst,V_SENDER,sender);
		lfprintf(log,NULL,"substitute -- %s => %s\n",tmp,dst);
	}
	if( strstr(dst,V_UNIXFROM) ){
		make_unixfrom(unixfrom,sender);
		strsubst(dst,V_UNIXFROM,unixfrom);
		lfprintf(log,NULL,"substitute -- %s => %s\n",tmp,dst);
	}
	if( strstr(dst,V_SEQNO) ){
		sprintf(val,"%d",Seqno);
		strsubst(dst,V_SEQNO,val);
		lfprintf(log,NULL,"substitute -- %s => %s\n",tmp,dst);
	}
	if( strstr(dst,V_FROM) ){
		fgetsHeaderField(mfp,"From",val,sizeof(val));
		strsubst(dst,V_FROM,val);
		lfprintf(log,NULL,"substitute -- %s => %s\n",tmp,dst);
	}
	if( strstr(dst,V_SUBJECT) ){
		fgetsHeaderField(mfp,"Subject",val,sizeof(val));
		strsubst(dst,V_SUBJECT,val);
		lfprintf(log,NULL,"substitute -- %s => %s\n",tmp,dst);
	}
	if( strstr(dst,V_CLEANSUBJ) ){
		fgetsHeaderField(mfp,"Subject",val,sizeof(val));
		subject_headclean(val);
		strsubst(dst,V_CLEANSUBJ,val);
		lfprintf(log,NULL,"substitute -- %s => %s\n",tmp,dst);
	}
	if( sp = strstr(dst,V_ORIGHEADER) )
	if( dp = strchr(sp,'}') ){
		char vname[256],fname[256];
		int len;
		len = dp + 1 - sp;
		bcopy(sp,vname,len);
		vname[len] = 0;
		sscanf(vname+strlen(V_ORIGHEADER),"%[^}]",fname);
		fgetsHeaderField(mfp,fname,val,sizeof(val));
		strsubst(dst,vname,val);
		lfprintf(log,NULL,"substitute -- %s => %s\n",tmp,dst);
	}
}

static rewrite_header(Conn,Gw,mfp,ts,sender,log)
	Connection *Conn;
	Gateway *Gw;
	FILE *mfp;
	FILE *ts;
	char *sender;
	FILE *log;
{	char *p1;
	int hi;
	char line[LNSIZE];

	if( p1 = getv(OUT_REPLY_TO) ){
		substitute(Gw,mfp,p1,ReplyTo,log,sender);
		lfprintf(log,ts,"Reply-To: %s\r\n",ReplyTo);
	}
	if( p1 = getv(OUT_SUBJECT) ){
		substitute(Gw,mfp,p1,Subject,log,sender);
		lfprintf(log,ts,"Subject: %s\r\n",Subject);
	}
	if( 0 < Gw->g_outheaderN ){
		for( hi = 0; hi < Gw->g_outheaderN; hi++ ){
			substitute(Gw,mfp,Gw->g_outheaders[hi],line,log,sender);
			lfprintf(log,ts,"%s\r\n",line);
		}
	}
}

static normalize_folding(line)
	char *line;
{	char *dp,*body;

	dp = strchr(line,':');
	if( dp == NULL )
		return;

	body = ++dp;
	while( *dp == ' ' || *dp == '\t' )
		dp++;
	if( *dp == '\r' || *dp == '\n' ){
		while( *dp == '\r' || *dp == '\n' )
			dp++;
		if( *dp == ' ' || *dp == '\t' ){
			sv1log("Removing unnecessary folding... %s",line);
			strcpy(body,dp);
		}
	}
}
static checkhead(Conn,Gw,fname,line,tc,log)
	Connection *Conn;
	Gateway *Gw;
	char *fname,*line;
	FILE *tc,*log;
{	char fvalue[1024];
	char *p1;

	if( strcmp(fname,"To")   == 0 && (p1 = getv(ACC_TO))
	||  strcmp(fname,"From") == 0 && (p1 = getv(ACC_FROM))
	){
		getFV(line,fname,fvalue);
		RFC822_addresspartX(fvalue,fvalue,sizeof(fvalue));
		if( tobe_rejected(tc,fname,fvalue,p1,log) != 0 )
			return -1;
	}
	return 0;
}

/*
 *   MAIL_TO_NEWS
 *	"tc" is a stream connected to the SMTP client
 *	"mfp" is a file containing the data sent by DATA command.
 *	"ts" is a stream connected to the NNTP server
 */
static mail_to_news(Conn,Gw,tc,mfp,sender,recipient,log)
	Connection *Conn;
	Gateway *Gw;
	FILE *tc,*mfp;
	char *sender,*recipient;
	FILE *log;
{	char line[1024],tmp[1024];
	char fname[1024],fvalue[1024];
	char myhost[256],clifhost[256],clhost[256],*dp;
	char path0[128];
	char To[1024],*to;
	char From[1024];
	char Newsgroups[1024],Distribution[1024];
	char EOH[8];
	char *p1;

	Newsgroups[0] = 0;
	Distribution[0] = 0;

	To[0] = 0;
	From[0] = 0;
	EOH[0] = 0;

	gethostname(myhost,sizeof(myhost));
	getFQDN(myhost,myhost);
	ClientIF_name(Conn,FromC,clifhost);
	getClientHostPort(Conn,clhost);

	if( (dp = strchr(sender,'@')) && dp[1] ){
		strcpy(path0,dp+1);
		strcat(path0,"!");
		sscanf(sender,"%[^@]",&path0[strlen(path0)]);
	}else	strcpy(path0,sender);
	lfprintf(log,TS,"Path: %s!SMTP-DeleGate%s!%s!%s!%s\r\n",
		myhost,DELEGATE_ver(),clifhost,clhost,path0);

	lfprintf(log,TS,
		"X-Forwarded: by %s (NNTP/SMTP gateway DeleGate/%s)\r\n",
		myhost,DELEGATE_ver());

	if( p1 = getv(OUT_NEWSGROUPS) ){
		substitute(Gw,mfp,p1,Newsgroups,log,sender);
		lfprintf(log,TS,"Newsgroups: %s\r\n",Newsgroups);
	}
	if( p1 = getv(OUT_DISTRIBUTION) ){
		substitute(Gw,mfp,p1,Distribution,log,sender);
		lfprintf(log,TS,"Distribution: %s\r\n",Distribution);
	}

	rewrite_header(Conn,Gw,mfp,TS,sender,log);

	/*
	 * should do MIME conversion ...
	 */
	for(;;){
		if( RFC822_fgetsHeaderField(line,sizeof(line),mfp) == NULL )
			break;
		normalize_folding(line);

		if( line[0] == '\r' || line[0] == '\n' ){
			strcpy(EOH,line);
			break;
		}

		sscanf(line,"%[^: ]",fname);

		if( checkhead(Conn,Gw,fname,line,tc,log) < 0 )
			return -1;

		if( strcasecmp(fname,"X-Gateway-Control") == 0 ){
			lfprintf(log,NULL,"control -- %s",line);
			if( strcasestr(line,"trace-on") )
				Option |=  GWT;
			else	Option &= ~GWT;
		}

		if( strcasecmp(fname,"Date") == 0 ){
			char oline[128];
			RFC822_strip_commentX(line,oline,sizeof(oline));
			if( strcmp(line,oline) != 0 ){
			lfprintf(log,NULL,"time zone erased -- %s",line);
			strcpy(line,oline);
			lfprintf(log,NULL,"time zone erased -- %s",line);
			}
		}

		if( strcasecmp(fname,"From") == 0 )
			getFV(line,"From",From);

		if( strncasecmp(fname,"X-Original-",11) == 0 ){
			if( (Option&RXO) ){
				lfprintf(log,NULL,"removed -- %s",line);
				continue;
			}
		}

		if( strcaseeq(fname,"Path")
		 || Newsgroups[0]   && strcaseeq(fname,"Newsgroups")
		 || Distribution[0] && strcaseeq(fname,"Distribution")
		 || ReplyTo[0]      && strcaseeq(fname,"Reply-To")
		 || Subject[0]      && strcaseeq(fname,"Subject")
		 || strcaseeq(fname,"NNTP-Posting-Host")
		 || strcaseeq(fname,"Errors-To")
		 || strcaseeq(fname,"Return-Receipt-To")
		 || strcaseeq(fname,"Return-Path")
		 || strcaseeq(fname,"Received")
		 || strcaseeq(fname,"Apparently-To")
		 || strcaseeq(fname,"To")
		 || strcaseeq(fname,"Cc")
		){
			if( !(Option&AXO) ){
				lfprintf(log,NULL,"removed -- %s",line);
				continue;
			}

			lfprintf(log,NULL,"escaped -- X-Original-%s",line);
			fputs("X-Original-",TS);
		}
		if( strcaseeq(fname,"Sender") ){
			lfprintf(log,NULL,"escaped -- X-Original-%s",line);
			fputs("X-Original-",TS);
		}

		fputs(line,TS);

		if( strcasecmp(fname,"Subject") == 0 && Subject[0] == 0 )
			strcpy(Subject,line);
	}
	if( EOH[0] ){
		char *admin;

		if( From[0] && strcmp(From,sender) != 0 )
			lfprintf(log,TS,"Sender: %s\r\n",sender);

		if( admin = DELEGATE_ADMIN )
			lfprintf(log,TS,"Resent-Sender: %s\r\n",admin);

		if( Subject[0] == 0 && !(Option&RES) )
			lfprintf(log,TS,"Subject: (no subject)\r\n");

		fputs(EOH,TS);
	}

	for(;;){
		if( fgets(line,sizeof(line),mfp) == NULL )
			break;
		fputs(line,TS);
	}

	append_log(Conn,Gw,TS,log);
	return 0;
}

static nntp_post(Conn,Gw,tc,mfp,sender,recipient,log)
	Connection *Conn;
	Gateway *Gw;
	FILE *tc,*mfp;
	char *sender;
	FILE *log;
{	char stat[1024];
	char msgid[1024],*dp;
	int off;

	if( fgetsHeaderField(mfp,"Message-Id",msgid,sizeof(msgid)) ){
		if( (dp = strchr(msgid,'@')) == NULL || dp[1] == 0 ){
			lfprintf(log,tc,"554 illegal Messae-ID: %s\r\n",msgid);
			return -1;
		}

		lfprintf(log,TS,"STAT %s\r\n",msgid);
		fflush(TS);

		fgets(stat,sizeof(stat),FS);
		lfprintf(log,NULL,"<<< %s",stat);
		if( atoi(stat) == 223 ){
			lfprintf(log,tc,"554 duplicate Mesage-ID: %s",stat);
			return -1;
		}
		lfprintf(log,NULL,"ok - Mesage-ID: %s\r\n",msgid);
	}else{
		if( Option & RNI ){
			lfprintf(log,tc,"554 no Message-Id\r\n");
			return -1;
		}
		lfprintf(log,NULL,"no original Message-Id\r\n");
	}

	lfprintf(log,TS,"POST\r\n");
	fflush(TS);

	/*
	if( fgets(stat,sizeof(stat),FS) == NULL ){
	*/
	if( fPollIn(FS,60*1000) <= 0 ){
		lfprintf(log,tc,"421 NNTP server didn't respond for POST.\r\n");
		return -1;
	}else
	if( fgetsTimeout(stat,sizeof(stat),FS,60) == NULL ){
		lfprintf(log,tc,"421 NNTP server closed before POST.\r\n");
		return -1;
	}else{
		lfprintf(log,NULL,"<<< %s",stat);
	}
	if( atoi(stat) != 340 ){
		fprintf(tc,"554 posting failed: %s",stat);
		fflush(tc);
		return -1;
	}

	/*
	if( mail_to_news(Conn,Gw,tc,mfp,sender,recipient,log) < 0 )
		return -1;
	*/
	{
		FILE *sTS,*xfp;
		int rcode;

		sTS = TS;
		TS = xfp = TMPFILE("MailToNews");
		rcode = mail_to_news(Conn,Gw,tc,mfp,sender,recipient,log);
		TS = sTS;
		if( rcode == 0 ){
			fseek(xfp,0,0);
			copyfile1(xfp,TS);
		}
		fclose(xfp);
		if( rcode != 0 ){
			return rcode;
		}
	}

	lfprintf(log,TS,".\r\n");
	fflush(TS);

	if( fgets(stat,sizeof(stat),FS) == NULL ){
		lfprintf(log,tc,"421 NNTP server closed during POST.\r\n");
		return -1;
	}

	lfprintf(log,NULL,"<<< %s",stat);
	if( atoi(stat) != 240 ){
		fprintf(tc,"554 posting failed: %s",stat);
		fflush(tc);
		return -1;
	}

	return 0;
}
static nntp_open(Conn,Gw,tc,mfp,recipient,pc,pv,log) 
	Connection *Conn;
	Gateway *Gw;
	FILE *tc;
	FILE *mfp;
	char *recipient;
	char *pv[];
	FILE *log;
{	int pi;
	char *p1;
	char *host;
	int port;
	char stat[1024];
	int server;
	char *auth,authb[128],user[128],pass[128];

	for( pi = 0; pi < pc; pi++ ){
		char *p1,tag[128],val[128];
		int px;

		p1 = pv[pi];
		sscanf(p1,"%[^.].%s",tag,val);
		px = param_idx(tag);
		if( px < 0 ){
			lfprintf(log,tc,"config. error -- unknown [%s]\n",p1);
			return -1;
		}
		Gw->g_pv[px] = stralloc(val);
	}

	host = getv(SERVHOST);
	if( p1 = getv(SERVPORT) )
		port = atoi(p1);
	else	port = 0;
	if( port == 0 )
		port = 119;

	lfprintf(log,NULL,"connecting to -- nntp://%s:%d\r\n",host,port);
	set_realserver(Conn,"nntp",host,port);
	Conn->no_dstcheck_proto = serviceport("nntp");

	if( setFilter(Conn,Gw,tc,mfp,log) != 0 )
		return -1;
	server = connect_to_serv(Conn,FromC,ToC,0);
	setFTOSV(GFILTER);
	if( server < 0 ){
		if( ConnError & CO_REJECTED )
			lfprintf(log,tc,"554 cannot connect NNTP (Forbidden)\r\n");
		else	lfprintf(log,tc,"421 %s... cannot connect NNTP\r\n",recipient);
		fflush(tc);
		return -1;
	}

	FS = fdopen(FromS,"r");
	TS = fdopen(ToS,"w");

	fgets(stat,sizeof(stat),FS);
	lfprintf(log,NULL,"<<< %s",stat);

	if( auth = getv(CTL_MYAUTH) ){
		nonxalpha_unescape(auth,authb,1);
		auth = authb;
	}else
	if( get_MYAUTH(Conn,authb,"nntp",host,port) )
		auth = authb;
	if( auth ){
		fieldScan(auth,user,pass);
		fprintf(TS,"AUTHINFO USER %s\r\n",user);
		fflush(TS);
		fgets(stat,sizeof(stat),FS);
		lfprintf(log,NULL,"AUTHINFO USER %s -> %s",user,stat);
		fprintf(TS,"AUTHINFO PASS %s\r\n",pass);
		fflush(TS);
		fgets(stat,sizeof(stat),FS);
		lfprintf(log,NULL,"AUTHINFO PASS **** -> %s",stat);
	}

	return 0;
}
static postnews(Conn,Gw,tc,mfp,sender,recipient,pc,pv,log)
	Connection *Conn;
	Gateway *Gw;
	FILE *tc,*mfp;
	char *sender;
	char *pv[];
	FILE *log;
{	int size;
	int off;

	off = ftell(mfp);

	/* must skip header and see the size of body ... */
	size = file_size(fileno(mfp)) - ftell(mfp);
	lfprintf(log,NULL,"#### BODY SIZE = %d\n",size);
	if( size == 0 && Option & REB ){
		/* ... */
	}

	if( nntp_open(Conn,Gw,tc,mfp,recipient,pc,pv,log) != 0 )
		return -1;
	else
	if( nntp_post(Conn,Gw,tc,mfp,sender,recipient,log) != 0 )
		return -1;

	fputs("QUIT\r\n",TS);
	fflush(TS);

	fseek(mfp,off,0);
	sendBCC(Conn,Gw,mfp,log);
	return 0;
}


static ftpmail(Conn,Gw,tc,mfp,sender,recipient,log)
	Connection *Conn;
	Gateway *Gw;
	FILE *tc;
	FILE *mfp;
	char *recipient;
	FILE *log;
{	FILE *fpv[2],*ts,*fs;
	char host[256];
	int port;
	int rsock;
	char resp[1024];

	rsock = -1;
	port = 25;

	if( !localsocket(ClientSock) ){
		getClientHostPort(Conn,host); /* should use HELO ? */
		rsock = SMTP_open(Conn,fpv,host,port,sender,recipient,1,log);
	}
	if( rsock < 0 ){
		gethostname(host,sizeof(host));
		rsock = SMTP_open(Conn,fpv,host,port,sender,recipient,1,log);
	}
	if( rsock < 0 ){
		lfprintf(log,tc,"421 cannot open SMTP for response.\r\n");
		return -1;
	}

	fs = fpv[0];
	ts = fpv[1];
	fprintf(ts,"Subject: TEST RESPONSE\r\n");
	fprintf(ts,"\r\n");
	fprintf(ts,"(TEST RESPONSE)\r\n");
	lfprintf(log,ts,".\r\n");
	fflush(ts);
	SMTP_relay_stat(fs,NULL,resp);

	lfprintf(log,ts,"QUIT\r\n");
	fflush(ts);
	SMTP_relay_stat(fs,NULL,resp);
	return 0;
}

static rewrite_maildata(Conn,Gw,ts,mfp,log)
	Connection *Conn;
	Gateway *Gw;
	FILE *ts,*mfp,*log;
{	char line[LNSIZE];
	char fname[LNSIZE];

	rewrite_header(Conn,Gw,mfp,ts,Sender,log);
	for(;;){
		if( RFC822_fgetsHeaderField(line,sizeof(line),mfp) == NULL )
			break;

		sscanf(line,"%[^: ]",fname);
		if( strcaseeq(fname,"Subject" ) && Subject[0]
		 || strcaseeq(fname,"Reply-To") && ReplyTo[0]
		){
			fprintf(ts,"X-Original-");
		}
		fputs(line,ts);
		if( line[0] == '\r' || line[0] == '\n' )
			break;
	}
	relayMSGdata(mfp,ts,1,0);
}
static put_maildata(Conn,Gw,ts,fs,mfp,recipients,resp,log)
	Connection *Conn;
	Gateway *Gw;
	FILE *ts,*fs,*mfp;
	char *recipients;
	char *resp;
	FILE *log;
{
	SMTP_putserv(log,fs,ts,resp,"DATA\r\n");
	if( atoi(resp) != 354 )
		return -1;

	smtp_datamark(Conn,ts);

	if( (Option & NTR) == 0 )
		put_trace(Conn,Gw,log,ts,recipients);
	rewrite_maildata(Conn,Gw,ts,mfp,log);
	
	append_log(Conn,Gw,ts,log);
	lfprintf(log,ts,".\r\n");
	fflush(ts);
	SMTP_relay_stat(fs,NULL,resp);
	return atoi(resp);
}

extern FILE *expand_aliases();

static mxaddr(host,saddr)
	char *host,*saddr;
{	char *dp,rechost[256],mxhost[256],*mxaddr;

	if( (dp = strchr(host,'@')) == 0 ){
		if( saddr )
			strcpy(saddr,"0.0.0.0");
		return 0;
	}
	sscanf(dp+1,"%[-.A-Za-z0-9]",rechost);
	sprintf(mxhost,"-MX.%s",rechost);
	if( mxaddr = gethostaddr(mxhost) )
		Verbose("#### MX-HOST(%s) = %s\n",host,mxaddr);
	else
	if( mxaddr = gethostaddr(rechost) ){
	}else{
		mxaddr = "255.255.255.255";
	}
	if( saddr )
		strcpy(saddr,mxaddr);
	return _inet_addrV4(mxaddr);
}
static sortbymx(a1,a2)
	char *a1,*a2;
{	int i1,i2;
	int rcode;

	i1 = mxaddr(a1,NULL);
	i2 = mxaddr(a2,NULL);
	if( i1 == i2 )
		return strcmp(a1,a2);
	else	return i1 - i2;
}
static sendmail_mx(Conn,Gw,tc,mfp,sender,recipient,log)
	Connection *Conn;
	Gateway *Gw;
	FILE *tc,*mfp;
	char *recipient,*sender;
	FILE *log;
{	char *recb,*recv[0x4000];
	int rc,ri,rj,rx;
	int st,rcode;
	char rechost[256],mxhost[256],addr[256],*dp,*rp;
	char reclist[256];
	int port;

	recb = stralloc(recipient);
	rc = list2vect(recb,',',0x4000,recv);
	if( rc == 0 ){
		lfprintf(log,tc,"554 no recipient\r\n");
		fflush(tc);
		return -1;
	}
	qsort(recv,rc,sizeof(char*),sortbymx);

	rcode = 0;
	port = 25;

	for( ri = 0; ri < rc; ri = rj + 1 ){
		for( rj = ri; rj < rc-1; rj++ )
		if( mxaddr(recv[rj],NULL) != mxaddr(recv[rj+1],NULL) )
			break;

		reclist[0] = 0;
		rp = reclist;
		for( rx = ri; rx <= rj; rx++ ){
			if( ri < rx )
				*rp++ = ',';
			strcat(rp,recv[rx]);
			rp += strlen(rp);
		}
		mxaddr(recv[ri],addr);
		st = sendmail_at(Conn,Gw,tc,mfp,addr,port,reclist,sender,log);
		if( st != 0 )
			rcode = -1;
	}
	free(recb);
	return rcode;
}

sendmail_at(Conn,Gw,tc,mfp,host,port,recipients,sender,log)
	Connection *Conn;
	Gateway *Gw;
	FILE *tc,*mfp;
	char *host,*recipients,*sender;
	FILE *log;
{	FILE *RCPT;
	char recipient[1024],*np;
	FILE *fpv[2],*ts,*fs;
	int rsock;
	char resp[1024];
	int nrec,nrec1,nerr,ri;
	int rcode = 0;

	RCPT = expand_aliases(recipients,log);
	if( RCPT == NULL ){
		lfprintf(log,tc,"421 error in alias expansion\r\n");
		return -1;
	}
	fgets(recipient,sizeof(recipient),RCPT);
	if( np = strpbrk(recipient,"\r\n") )
		*np = 0;
	rsock = SMTP_open(Conn,fpv,host,port,recipient,sender,0,log);
	setFTOSV(GFILTER);
	if( rsock < 0 ){
		lfprintf(log,tc,"421- cannot open SMTP for forwarding.\r\n");
		lfprintf(log,tc,"421  server = %s:%d.\r\n",host,port);
		return -1;
	}

	fs = fpv[0];
	ts = fpv[1];
	nrec = 1;
	nerr = 0;

	while( !feof(RCPT) ){
		for( nrec1 = 0; nrec1 < 64; nrec1++ ){
			if( fgets(recipient,sizeof(recipient),RCPT) == NULL )
				break;
			if( np = strpbrk(recipient,"\r\n") )
				*np = 0;
			lfprintf(log,ts,"RCPT To: %s\r\n",recipient);
			nrec++;
		}
		fflush(ts);
		for( ri = 0; ri < nrec1; ri++ ){
			SMTP_relay_stat(fs,NULL,resp);
			if( atoi(resp) != 250 ){
				lfprintf(log,NULL,"[%d] %s",ri,resp);
				nerr++;
			}
		}
	}
	if( nerr != 0 ){
		lfprintf(log,tc,"421 NOT sent: RCPT errors %d/%d\r\n",nerr,nrec);
		rcode = -1;
	}else{
		put_maildata(Conn,Gw,ts,fs,mfp,recipients,resp,log);
		lfprintf(log,NULL,"Sent to %d recipients\r\n",nrec);
	}

	lfprintf(log,ts,"QUIT\r\n");
	fflush(ts);
	SMTP_relay_stat(fs,NULL,resp);

	fclose(fpv[1]);
	fclose(fpv[0]);
	FromS = ToS = -1;
	fclose(RCPT);

	return rcode;
}
static sendmail(Conn,Gw,tc,mfp,sender,recipient,log)
	Connection *Conn;
	Gateway *Gw;
	FILE *tc;
	FILE *mfp;
	char *sender,*recipient;
	FILE *log;
{	char *p1;
	char *host;
	int port;
	char *to,tobuf[0x10000];
	char *rp;
	int off;
	char *from;

	if( from = getv(CTL_SENDER) )
		sender = from;

	to = getv(OUT_TO);
	rp = getv(CTL_RECIPIENT);
	if( (to == NULL || *to == 0) && (rp == NULL || *rp == 0) ){
		lfprintf(log,tc,"554 no forwarding info.\r\n");
		fflush(tc);
		return -1;
	}
	if( rp != NULL && *rp != 0 )
		substitute(Gw,mfp,rp,tobuf,log,sender);
	else	substitute(Gw,mfp,to,tobuf,log,sender);
	to = tobuf;

	host = getv(SERVHOST);
	if( streq(host,V_RECMX) ){
		off = ftell(mfp);
		if( sendmail_mx(Conn,Gw,tc,mfp,sender,to,log) < 0 )
			return -1;
		fseek(mfp,off,0);
	}else{
		if( p1 = getv(SERVPORT) ) 
			port = atoi(p1);
		else	port = 0;
		if( port == 0 )
			port = 25;

		if( setFilter(Conn,Gw,tc,mfp,log) != 0 )
			return -1;

		off = ftell(mfp);
		if( sendmail_at(Conn,Gw,tc,mfp,host,port,to,sender,log) < 0 )
			return -1;
		fseek(mfp,off,0);
	}

	sendBCC(Conn,Gw,mfp,log);
	return 0;
}

/*
 *	BCC: <user@host.domain:port>
 *
 *	TO-DO: multiple recipients which should be sent to the same SMTP
 *	server in BCC should be grouped in a single SMTP transmission...
 */
static sendBCC1(bcc,Conn,Gw,mfp,log)
	char *bcc;
	Connection *Conn;
	Gateway *Gw;
	FILE *mfp;
	FILE *log;
{	char host[256],*sender;
	int port;
	int off;
	int i1;
	char addr[64];

	host[0] = 0;
	port = 0;

	while( *bcc == ' ' || *bcc == '\t' )
		bcc++;

	sscanf(bcc,"%*[^@]@%[^:]:%d",host,&port);
	if( host[0] == 0 )
		strcpy(host,"localhost");
	if( port == 0 )
		port = 25;
	sender = DELEGATE_ADMIN;

	if( i1 = mxaddr(bcc,addr) )
	if( i1 != _inet_addrV4("255.255.255.255") ){
	lfprintf(log,NULL,"sending BCC -- %s [smtp://%s] MX=[%s]\n",
			bcc,host,addr);
		strcpy(host,addr);
	}

	lfprintf(log,NULL,"sending BCC -- %s [smtp://%s:%d] sender=%s\n",
		bcc,host,port,sender);

	off = ftell(mfp);
	sendmail_at(Conn,Gw,log,mfp,host,port,bcc,sender,log);
	fseek(mfp,off,0);
	return 0;
}
static sendBCC(Conn,Gw,mfp,log)
	Connection *Conn;
	Gateway *Gw;
	FILE *mfp;
	FILE *log;
{	char *bcc;
	char name[1024],rpath[1024];

	bcc = getv(BCC);
	if( bcc == NULL )
		return;
	scan_commaList(bcc,0,sendBCC1,Conn,Gw,mfp,log);
}

static lock_counter1(counter,rpath,timeout)
	FILE *counter;
	char *rpath;
{
	if( lock_exclusiveTO(fileno(counter),timeout,NULL) == 0 )
		return 0;
	sv1log("#### cannot lock COUNTER: %s\r\n",rpath);
	return -1;
}
static FILE *lock_counter(name,mfp,countp,log)
	char *name;
	FILE *mfp;
	int *countp;
	FILE* log;
{	char rpath[1024];
	FILE *counter;
	int opid,otime;
	char line[1024];
	int count;

	*countp = 0;
	if( *name == 0 )
		return NULL;

	if( counter = open_file(name,"count","r+",rpath) ){
		if( lock_counter1(counter,rpath,60*1000) != 0 ){
			fclose(counter);
			return NULL;
		}
		count = 0;
		opid = 0;
		fgets(line,sizeof(line),counter);
		sscanf(line,"%d pid=%d time=%d",&count,&opid,&otime);
		if( count == 0 )
			count = 1;
	}else
	if( counter = open_file(name,"count","w+",rpath) ){
		if( lock_counter1(counter,rpath,1) != 0 ){
			fclose(counter);
			return NULL;
		}
		lfprintf(log,NULL,"created counter -- SMTPGATE/%s\r\n",rpath);
		fprintf(counter,"1 pid=%d\r\n",getpid());
		fflush(counter);
		count = 1;
		opid = getpid();
		otime = time(0);
	}else{
		sv1log("#### cannot open COUNTER: %s\r\n",rpath);
		return NULL;
	}

	lfprintf(log,NULL,"got Counter -- %d : SMTPGATE/%s\n",count,rpath);
	*countp = count;
	return counter;
}
static unlock_counter(counter,count,inc,log)
	FILE *counter;
	FILE *log;
{
	if( inc ){
		fseek(counter,0,0);
		count += inc;
		fprintf(counter,"%d pid=%d time=%d\r\n",
			count,getpid(),time(0));
		fflush(counter);
		lfprintf(log,NULL,"set Counter -- %d\r\n",count);
	}
	fclose(counter);
}

static lfprintf1(log,fmt,a,b,c,d,e,f,g)
	char *fmt,*a,*b,*c,*d,*e,*f,*g;
{
	lfprintf(log,NULL,fmt,a,b,c,d,e,f,g);
}

static match_aclfile(what,addr,acladdr,afp,log)
	char *what,*addr,*acladdr;
	FILE *afp;
	FILE *log;
{	char line[1024],*vp,*dp;
	int nlines;

	for( nlines = 1; fgets(line,sizeof(line),afp) != NULL; nlines++ ){
		if( strcasestr(line,addr) == NULL )
			continue;

		sv1log("#### found -- <%s> at line#%d of %s\n",addr,nlines,acladdr);
		if( dp = strpbrk(line,"\r\n") )
			*dp = 0;
		lfprintf(log,NULL,"accept %s -- <%s> is in the ACL as '%s'\r\n",
			what,addr,line);
		return 1;
	}
	sv1log("#### not found -- <%s> in %s\n",addr,acladdr);
	return 0;
}
static check_infile(tc,what,addr,acl,log)
	FILE *tc;
	char *what,*addr,*acl;
	FILE *log;
{	FILE *afp;

	afp = fopen(acl,"r");
	if( afp == NULL ){
		lfprintf(log,tc,"554 cannot open the ACL file.\n");
		fflush(tc);
		return -1;
	}
	return match_aclfile(what,addr,acl,afp,log);
}
static check_byexpn(tc,what,addr,acladdr,log)
	FILE *tc;
	char *what,*addr,*acladdr;
	FILE *log;
{	FILE *afp;

	afp = SMTP_getEXPN(acladdr);
	if( afp == NULL ){
		lfprintf(log,tc,"554 cannot get EXPN <%s>\n",addr);
		fflush(tc);
		return -1;
	}
	return match_aclfile(what,addr,acladdr,afp,log);
}
static tobe_rejected(tc,what,addr,acli,log)
	FILE *tc;
	char *what,*addr,*acli;
	FILE *log;
{	char *av[256],*acla,*aclb,*aclc,*aclp,*acl0,*acl1;
	char acladdr[256];
	int ac,ai,negate,match,subst;

	acla = stralloc(acli);
	aclb = stralloc(acli);
	aclc = malloc(strlen(acli)+1);
	aclp = aclc;
	ac = stoV(acla,256,av,',');

	for( ai = 0; ai < ac; ai++ ){
		acl1 = av[ai];
		acl1 = acl0 = strip_spaces(acl1);
		if( negate = (*acl1 == '!') )
			acl1++;

		if( subst = isFullpath(acl1) ){
			match = check_infile(tc,what,addr,acl1,log);
			if( match < 0 )
				goto EXIT;
		}else
		if( subst = (strncmp(acl1,"${expn:",7)==0) ){
			acladdr[0] = 0;
			sscanf(acl1,"${expn:%[^}]}",acladdr);
			match = check_byexpn(tc,what,addr,acladdr,log);
			if( match < 0 )
				goto EXIT;
		}
		if( subst ){
			if( match ){
				if( aclp != aclc) *aclp++ = ',';
				if( negate )
					*aclp++ = '!';
				strcpy(aclp,"*");
			}
		}else{
			if( aclp != aclc) *aclp++ = ',';
			strcpy(aclp,acl0);
		}
		aclp += strlen(aclp);
	}
	lfprintf(log,NULL,"preprocessed ACL: [%s] -> [%s]\n",acli,aclc);

	if( match = strmatch_list(addr,aclc,"",lfprintf1,log) ){
		lfprintf(log,NULL,"accept %s -- %s is in [%s]\r\n",
			what,addr,aclc);
	}else{
		lfprintf(log,NULL,"reject %s -- %s is not in [%s]\r\n",
			what,addr,aclc);
		lfprintf(log,tc,"554 Forbidden by rule\r\n");
		fflush(tc);
	}
EXIT:
	free(acla);
	free(aclb);
	free(aclc);
	return match ? 0 : -1;
}

extern char *getFTOSV();
static setFilter(Conn,Gw,tc,mfp,log)
	Connection *Conn;
	Gateway *Gw;
	FILE *tc;
	FILE *mfp;
	FILE *log;
{	char *filter;

	filter = getv(OUT_FILTER);
	if( filter == NULL )
		return 0;

	lfprintf(log,NULL,"FILTER: %s\r\n",filter);
	/*
	scan_FTOSV(filter);
	*/
	scan_FTOSV(Conn,filter);
	return 0;
}

#define DEFAULT_CONF "/-/builtin/config/smtpgate/@default/conf";

SMTPgateway(Conn,tc,mfp,sender,recipient,log)
	Connection *Conn;
	FILE *tc;
	FILE *mfp;
	char *sender,*recipient;
	FILE *log;
{	Gateway *Gw,*BGw;
	char local[1024],domain[1024];
	char *pv[128],pb[1024],*p1,*gwname,*parent;
	int pc,pi;
	char params[1024],*local1,*local2,*pp;
	int lev;
	int rcode = 0;
	FILE *counter,*log1;
	int soff;

	starttime = time(0);
	soff = ftell(mfp);
	/*
	 *  search and select configuration file baed on
	 *  "sender", "recipients" and "mfp"
	 */
	local[0] = domain[0] = 0;
	sscanf(recipient,"%[^@]@%s",local,domain);
	strcpy(pb,local);
	pc = stoV(pb,128,pv,'%');
	if( pc < 1 ){
		lfprintf(log,NULL,"ERROR -- no local part ? [%s]\n",recipient);
		return 0;
	}
	gwname = local1 = local2 = pv[--pc];
	if( gwname == NULL )
		return 0;

	pp = params;
	params[0] = 0;
	for( pi = 0; pi < pc; pi++ ){
		sprintf(pp,"[%s]",pv[pi]);
		pp += strlen(pp);
	}
	lfprintf(log,NULL,"GWNAME = %s %s\n",gwname,params);

	Gw = new_conf();
	Sender = stralloc(sender);
	Recipient = stralloc(recipient);
	Rname = stralloc(local1);

	if( load_conf(Conn,Gw,gwname,mfp,sender,recipient,log) == 0 ){
		lfprintf(log,NULL,"GWNAME = %s (default gateway)\n",DEFAULT);
		if( load_conf(Conn,Gw,DEFAULT,mfp,sender,recipient,log) ){
			if( parent = getv(INHERIT) )
				gwname = parent;
			lfprintf(log,NULL,"GWNAME = %s\n",gwname);
			local2 = DEFAULT;
		}else{
			char *aurl,rurl[256],buf[2048];
			FILE *fp;
			aurl = DEFAULT_CONF;
			Verbose("SMTPGATE: loading builtin @default/conf ..\n");
			getBuiltinData(Conn,"SMTPGATE",aurl,buf,sizeof(buf),rurl);
			fp = TMPFILE("SMTPGATE"); fputs(buf,fp); fflush(fp);
			fseek(fp,0,0);
			scan_conf(Gw,fp,log);
			fclose(fp);
		}
	}
	Gw->g_pv[GWTYPE] = stralloc(gwname);

	/*
	 *  load parent if exists
	 */
	for( lev = 0; lev < 4; lev++ ){
		parent = getv(INHERIT);
		if( parent == NULL )
			break;

		if( BGw = builtinGw(parent,1) ){
			gwname = parent;
			break;
		}

		lfprintf(log,NULL,"GWNAME = %s\n",parent);
		if( load_conf(Conn,Gw,parent,mfp,sender,recipient,log) == 0 )
			break;
		parent = getv(INHERIT);
		if( parent != NULL )
			gwname = parent;
	}

	/*
	 *  builtin class
	 */
	if( (BGw = builtinGw(gwname,0)) == NULL ){
		lfprintf(log,NULL,"unknown Gateway Name -- %s\r\n",gwname);
		return 0;
	}
	Gw->g_parent = BGw;
	copy_ifundef(Gw,Gw->g_parent);
	lfprintf(log,NULL,"GWNAME = %s (built-in)\n",gwname);

	/*
	 *  access control
	 */
	if( p1 = getv(ACC_CLIENT_HOST) ){
		int hl,match;

		hl = makePathList("SMTPGATE",p1);
		match = matchPath1(hl,"",Client_Host,Client_Port);
		if( !match ){
			lfprintf(log,NULL,"Forbidden Client-Host -- %s\r\n",
				Client_Host);
			lfprintf(log,tc,"554 Forbidden by rule\r\n");
			fflush(tc);
			rcode = -1;
			goto EXIT;
		}
	}
	if( p1 = getv(ACC_SENDER) ){
		char acc_sender[1024];
		substitute(Gw,mfp,p1,acc_sender,log,sender);
		if( tobe_rejected(tc,"Sender",sender,acc_sender,log) != 0 ){
			rcode = -1;
			goto EXIT;
		}
	}
	if( p1 = getv(ACC_RECIPIENT) ){
		if( tobe_rejected(tc,"Recipient",recipient,p1,log) != 0 ){
			rcode = -1;
			goto EXIT;
		}
	}
	if( p1 = getv(ACC_TO) ){ /* case of lacking To: */
		char line[128];
		if( fgetsHeaderField(mfp,"To",line,sizeof(line))  == NULL ){
			if( tobe_rejected(tc,"To","",p1,log) != 0 ){
				rcode = -1;
				goto EXIT;
			}
		}
	}
	if( p1 = getv(ACC_MESSAGE_ID) ){
		char id[128];
		if( fgetsHeaderField(mfp,"Message-Id",id,sizeof(id)) != 0 ){
			if( tobe_rejected(tc,"Message-Id",id,p1,log) != 0 ){
				rcode = -1;
				goto EXIT;
			}
		}
	}
	if( p1 = getv(ACC_MAX_BYTES) ){
		int max,siz;
		max = kmxatoi(p1);
		siz = file_size(fileno(mfp)) - soff;
		if( max < siz ){
			lfprintf(log,NULL,"reject %s -- Max-Bytes: %d < %d\r\n",
				"TOO LARG",max,siz);
			lfprintf(log,tc,"554 message too large (%d < %d)\r\n",
				max,siz);
			fflush(tc);
			rcode = -1;
			goto EXIT;
		}
		lfprintf(log,NULL,"accept -- Max-Bytes: %d > %d bytes\r\n",
			max,siz);
	}
	if( p1 = getv(ACC_MIN_BYTES) ){
		int min,siz;
		min = kmxatoi(p1);
		siz = file_size(fileno(mfp)) - soff;
		if( siz < min ){
			lfprintf(log,NULL,"reject %s -- Min-Bytes: %d > %d\r\n",
				"TOO SMALL",min,siz);
			lfprintf(log,tc,"554 message too small (%d > %d)\r\n",
				min,siz);
			fflush(tc);
			rcode = -1;
			goto EXIT;
		}
	}
	if( p1 = getv(ACC_MIN_BODY_BYTES) ){
		int min,siz;
		min = kmxatoi(p1);
		RFC821_skipheader(mfp,NULL,NULL);
		siz = file_size(fileno(mfp)) - ftell(mfp);
		fseek(mfp,soff,0);
		if( siz < min ){
			lfprintf(log,NULL,
				"reject %s -- Min-Body-Bytes: %d > %d\r\n",
				"TOO SMALL",min,siz);
			lfprintf(log,tc,"554 message too small (%d > %d)\r\n",
				min,siz);
			fflush(tc);
			rcode = -1;
			goto EXIT;
		}
	}
	if( p1 = getv(ACC_MAX_EXCLAMS) ){
		int chcount[256],max;

		max = atoi(p1);
		msg_charcount(mfp,chcount);
		if( max < chcount['!'] ){
			lfprintf(log,NULL,"reject %s -- Max-Exclams: %d<%d\r\n",
				"TOO MANY EXCLAMATIONS",max,chcount['!']);
			lfprintf(log,tc,"554 too many exclamations\r\n");
			rcode = -1;
			goto EXIT;
		}
	}
	if( p1 = getv(ACC_USERTEXT) ){
		char line[128];
		int match;

		match = 0;
		if( fgetsHeaderField(mfp,"Subject",line,sizeof(line)) != 0 )
			match = strmatch_list(line,p1,"",NULL,NULL);
		if( match == 0 ){
			RFC821_skipheader(mfp,NULL,NULL);
			while( fgets(line,sizeof(line),mfp) != NULL ){
				if( strmatch_list(line,p1,"",NULL,NULL) ){
					match = 1;
					break;
				}
			}
			fseek(mfp,soff,0);
		}
		if( match == 0 ){
			lfprintf(log,tc,"554 not include keyword\r\n");
			rcode = -1;
			goto EXIT;
		}
	}

	/*
	 *  save FTOSV filter
	 */
	GFILTER = getFTOSV(Conn);

	/*
	 *  call gateways
	 */
	if( p1 = getv(OPTION) )
		Option = scan_options(p1,Gwt,log);
	else	Option = 0;
	counter = NULL;

	if( Option & ISN ){
		counter = lock_counter(local2,mfp,&Seqno,log);
		if( counter == NULL ){
			lfprintf(log,tc,"421 cannot open/lock counter.\r\n");
			rcode = -1;
			goto EXIT;
		}
	}

	if( strcaseeq(gwname,POSTNEWS) ){
		if( postnews(Conn,Gw,tc,mfp,sender,recipient,pc,pv,log) == 0 )
			rcode =  1;
		else	rcode = -1;
	}else
	if( strcaseeq(gwname,SENDMAIL) ){
		if( sendmail(Conn,Gw,tc,mfp,sender,recipient,log) == 0 )
			rcode =  1;
		else	rcode = -1;
	}else
	if( strcaseeq(gwname,FTPMAIL) ){
		if( ftpmail(Conn,Gw,tc,mfp,sender,recipient,log) == 0 )
			rcode =  1;
		else	rcode = -1;
	}

	if( counter != NULL  )
		unlock_counter(counter,Seqno,rcode==1,log);

	if( rcode == 1 ){
		char stime[1024],spath[1024],rpath[1024],unixfrom[1024];
		FILE *sfp;
		int toff;
		char *arcspec;

		StrftimeLocal(stime,sizeof(stime),"%Y%m%d-%H%M%S",starttime);
		sprintf(spath,"spool/%05d-%s-%05d",Seqno,stime,getpid());
		make_unixfrom(unixfrom,sender);

		if( sfp = open_file(local2,spath,"w",rpath) ){
			fprintf(sfp,"UNIX-From: %s\r\n",unixfrom);
			toff = ftell(mfp);
			fseek(mfp,soff,0);
			copyfile1(mfp,sfp);
			fclose(sfp);
			fseek(mfp,toff,0);
			Verbose("## wrote spool %s : %s\n",local1,rpath);
		}else{
			Verbose("## cannot open spool %s : %s\n",local1,rpath);
		}

		if( arcspec = getv(ARCHIVE) ){
			subst_path(Gw,arcspec,spath);
			if( sfp = open_file(local2,spath,"a",rpath) ){
				fprintf(sfp,"From %s\r\n",unixfrom);
				fseek(mfp,soff,0);
				rewrite_maildata(Conn,Gw,sfp,mfp,NULL);
				fclose(sfp);
				fseek(mfp,toff,0);
			}
		}
	}

EXIT:
	lfprintf(log,NULL,"END(%d) %s <- %s\n",rcode,recipient,sender);
	if( rcode != 0 ){
		char rpath[1024];
		char rfile[64];
		FILE *rfp;
		if( log1 = open_file(local2,"log","a",rpath) ){
			fflush(log);
			fseek(log,0,0);
			copyfile1(log,log1);
			close(log1);
			fseek(log,0,2);
			Verbose("## wrote logfile of %s : %s\n",local1,rpath);
		}else{
			Verbose("## cannot open logfile of %s : %s\n",local1,rpath);
		}
	    if( rcode < 0 ){
		StrftimeLocal(rfile,sizeof(rfile),"rejected/%y%m%d-%H%M%S",
			starttime,0);
		if( rfp = open_file(local2,rfile,"a",rpath) ){
			fprintf(rfp,"--%s--\n",rfile);
			fseek(mfp,0,0);
			copyfile1(mfp,rfp);
			fclose(rfp);
		}
	    }
	}
	return rcode;
}
