/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1994-1999 Yutaka Sato
Copyright (c) 1994-1999 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, and distribute this material for any purpose
and without fee is hereby granted, provided that the above copyright
notice and this permission notice appear in all copies.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	smtp.c (SMTP proxy)
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:
History:
	941016	created
//////////////////////////////////////////////////////////////////////#*/
#include <stdio.h>
#include <ctype.h>
#include "ystring.h"
#include <errno.h>
#include "ysocket.h" /* gethostname() for Win32 */
#include "fpoll.h" /* READYCC() */
#define LNSIZE  1024

#define sv1log	syslog_ERROR

extern char *TIMEFORM_RFC822;
#define SMT_NOHELO	0x01
#define SMT_NOFROM	0x02
int SMTP_tolerance = 0xFF;

extern double Scan_period();
int SMTP_thrudata = 30; /* max. time to wait DATA from client for buffering */
int SMTP_bgdatasize = 64*1024;
int SMTP_nomboxcase = 1;
int SMTP_maxrcpt = 0;

#define A_PLAIN	1
#define A_LOGIN	2
int SMTP_doauth;

#define lfprintf	SMTP_lfprintf

static scan1(conf)
	char *conf;
{	char nam[32],val[32];

	fieldScan(conf,nam,val);
	if( strcaseeq(nam,"reject") ){
		if( strcaseeq(val,"nohelo") )
			SMTP_tolerance &= ~SMT_NOHELO;
		else
		if( strcaseeq(val,"nofrom") )
			SMTP_tolerance &= ~SMT_NOFROM;
	}
	else
	if( strcaseeq(nam,"bgdatasize") ){
		SMTP_bgdatasize = atoi(val);
		if( strpbrk(val,"kK") )
			SMTP_bgdatasize *= 1024;
		sv1log("SMTPCONF=bgdatasize:%d (bytes)\n",SMTP_bgdatasize);
	}
	else
	if( strcaseeq(nam,"maxrcpt") ){
		SMTP_maxrcpt = atoi(val);
	}
	else
	if( strcaseeq(nam,"thrudata") ){
		SMTP_thrudata = Scan_period(val,'s',(double)30);
	}
	else
	if( strcaseeq(nam,"mboxcase") ){
		SMTP_nomboxcase = 0;
	}
	else
	if( strcaseeq(nam,"auth") ){
		strtolower(val,val);
		if( val[0] == 0 ){
			SMTP_doauth = A_PLAIN | A_LOGIN;
		}else{
			if( isinList(val,"plain") )
				SMTP_doauth |= A_PLAIN;
			if( isinList(val,"login") )
				SMTP_doauth |= A_LOGIN;
		}
	}
	return 0;
}
scan_SMTPCONF(_,conf)
	void *_;
	char *conf;
{
	scan_commaListL(conf,0,scan1);
}

SMTP_lfprintf(log,tosc,fmt,a,b,c,d,e,f,g,h,i,j,k,l,m,n)
	FILE *log,*tosc;
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k,*l,*m,*n;
{	char stime[32];
	char stat[0x8000];

	getTimestamp(stime);
	sprintf(stat,fmt,a,b,c,d,e,f,g,h,i,j,k,l,m,n);

	if( log != NULL ){
		fprintf(log,"%s [%d] ",stime,Getpid());
		if( tosc == NULL
		&& *stat != '<' && *stat != '>' && !isdigit(*stat) )
			fprintf(log,"* "); /* internal action */
		fprintf(log,fmt,a,b,c,d,e,f,g,h,i,j,k,l,m,n);
	}

	syslog_ERROR(fmt,a,b,c,d,e,f,g,h,i,j,k,l,m,n);

	if( tosc )
	fprintf(tosc,fmt,a,b,c,d,e,f,g,h,i,j,k,l,m,n);
}

SMTP_relay_stat(fs,tc,resp)
	FILE *fs,*tc;
	char *resp;
{	char rcode[4];
	char respb[1024];

	if( resp == 0 )
		resp = respb;

	rcode[0] = 0;
	for(;;){
		if( fgets(resp,LNSIZE,fs) == NULL ){
			syslog_ERROR("SC-EOF. errno=%d\r\n",errno);
			return -1;
		}
		syslog_ERROR("SMTP > %s",resp);
		if( tc != NULL )
			if( Fputs(resp,tc) == EOF )
				return -1;

		if( resp[3] == '-' ){
			if( rcode[0] == 0 )
				strncpy(rcode,resp,3);
		}else{
			if( rcode[0] == 0 || strncmp(resp,rcode,3) == 0 )
				break;
		}
	}
	return atoi(resp);
}

SMTP_putserv(log,fs,ts,resp,fmt,a,b,c,d)
	FILE *log;
	FILE *fs,*ts;
	char *fmt,*a,*b,*c,*d;
	char *resp;
{	char req[1024];

	sprintf(req,fmt,a,b,c,d);
	fputs(req,ts);
	fflush(ts);

	if( log != NULL )
		lfprintf(log,NULL,"<<< %s",req);
	else	syslog_ERROR("SMTP < %s",req);

	SMTP_relay_stat(fs,NULL,resp);
	if( log != NULL )
		lfprintf(log,NULL,"%s",resp);
}

#define SMQ_COM		0
#define SMQ_FIELD	1
#define SMQ_RCPT	2
#define SMQ_FROM	2
#define SMQ_CLRF	3
typedef struct {
	UTag   *sm_uv[5];
	UTag	sm_ub[4];
} SMTPreq;
static decomp_req(req,Req)
	char *req;
	SMTPreq *Req;
{
	uvinit(Req->sm_uv,Req->sm_ub,4);
	uvfromsf(req,0,"%s %[^:]: %[^\r\n]%[\r\n]",Req->sm_uv);
}

typedef struct {
	char	*al_what;
	char	*al_list;
	char	*al_command;
	FILE	*al_log;
	FILE 	*al_ts;
	FILE	*al_fs;
	char	*al_respbuf;
} AddrList;

static SMTPpath(addr,path,siz)
	char *addr,*path;
{
	*path = '<';
	RFC822_addresspartX(addr,path+1,siz-3);
	strcat(path,">");
}
static scan_addr1(to,addr)
	char *to;
	AddrList *addr;
{	char xto[1024];

	SMTPpath(to,xto,sizeof(xto));
	syslog_ERROR("SMTP : %s: %s -- %s\r\n",xto,addr->al_what,to);
	SMTP_putserv(addr->al_log,addr->al_fs,addr->al_ts,
		addr->al_respbuf, "%s%s\r\n",addr->al_command,xto);
	if( addr->al_respbuf[0] == '5' )
		return -1;
	else	return 0;
}
SMTP_openingX(fs,ts,myver,to,from,dodata,log,auth)
	FILE *fs,*ts;
	char *myver,*to,*from;
	FILE *log;
	char *auth;
{	char myhost[128];
	char xto[1024],xfrom[1024];
	char resp[2048];
	AddrList addrs;

	SMTP_relay_stat(fs,NULL,resp);
	if( gethostName(fileno(ts),myhost,"%H") <= 0 )
		gethostname(myhost,sizeof(myhost));
	sv1log("getting FQDN of %s ...\n",myhost);
	getFQDN(myhost,myhost);

	if( auth ){
		SMTP_putserv(log,fs,ts,resp,"EHLO %s\r\n",myhost);
	}else
	SMTP_putserv(log,fs,ts,resp,"HELO %s\r\n",myhost);

	if( from ){
		SMTPpath(from,xfrom,sizeof(xfrom));
		syslog_ERROR("SMTP : From: %s -- %s\r\n",xfrom,from);
		SMTP_putserv(log,fs,ts,resp,"MAIL From: %s\r\n",xfrom);
		if( resp[0] == '5' )
			return -1;
	}
	if( to && to[0] ){
		addrs.al_what = "To";
		addrs.al_list = to;
		addrs.al_command = "RCPT To: ";
		addrs.al_respbuf = resp;
		addrs.al_log = log;
		addrs.al_ts = ts;
		addrs.al_fs = fs;
		scan_commaList(to,0,scan_addr1,&addrs);
		if( resp[0] == '5' )
			return -1;
	}
	if( dodata )
		SMTP_putserv(log,fs,ts,resp,"DATA\r\n");
	return 0;
}

FILE *SMTP_post(host,port,myver,to,from,mfp,resp,rsize)
	char *host,*myver,*to,*from,*resp;
	FILE *mfp;
{	int server;
	FILE *ts,*fs,*fc;
	char line[1024],respbuff[1024],*ep;
	char topChar;
	int inHead,inSkip;

	if( resp == NULL ){
		resp = respbuff;
		if( sizeof(respbuff) < rsize )
			rsize = sizeof(respbuff);
	}

	syslog_ERROR("SMTP_POST at %s:%d to:%s from:%s\r\n",host,port,to,from);
	server = client_open("openMailPoster","smtp",host,port);
	if( server < 0 ){
		syslog_ERROR("SMTP: cannot connect: %s:%d.\r\n",host,port);
		return NULL;
	}

	if( mfp != NULL ){
		fc = mfp;
	}else{
		int io[2];
		Socketpair(io);
		if( Fork("SMTP_POST") != 0 ){
			close(io[0]);
			close(server);
			return fdopen(io[1],"w");
		}
		close(io[1]);
		fc = fdopen(io[0],"r");
	}

	fs = fdopen(server,"r");
	ts = fdopen(server,"w");

	if( SMTP_openingX(fs,ts,myver,to,from,1,NULL,NULL) != 0 ){
		if( mfp != NULL ){
			fclose(ts);
			fclose(fs);
			return NULL;
		}
		_Finish(-1);
	}

	inHead = 1;
	inSkip = 0;
	for(;;){
		if( fgets(line,sizeof(line),fc) == NULL )
			break;

		if( inHead ){
			topChar = line[0];
			if(  topChar == '\r' || topChar == '\n' ){
				inHead = 0;
				inSkip = 0;
			}else
			if( topChar == ' ' || topChar == '\t' )
				;
			else
			if( strncasecmp(line,"Bcc:",4) == 0 ){
				syslog_ERROR("SMTP : erased BCC header\r\n");
				inSkip = 1;
			}else	inSkip = 0; 
		}
		if( !inSkip ){
			if( ep = strpbrk(line,"\r\n") )
				*ep = 0;
			if( line[0] == '.' )
				fputc('.',ts);
			fputs(line,ts);
			fputs("\r\n",ts);
		}
	}
	SMTP_putserv(NULL,fs,ts,resp,".\r\n");
	SMTP_putserv(NULL,fs,ts,resp,"QUIT\r\n");

	if( mfp != NULL )
		return mfp;

	fclose(fc);
	fclose(ts);
	fclose(fs);
	_Finish(0); /* avoid duplicate flush of stream buffer of parent ... */
}

#ifndef LIBRARY

#include "delegate.h"
#include "filter.h"

/*
 *	RCPT To: recipients -> Recipients[]
 *	MAIL From: originator -> Sender[]
 */
typedef struct {
	char   *se_myver;
	char	se_myhost[128];
	char	se_clhost[256];
	char	se_Sender[256];
	char	se_RecipientLocal[256];
	char   *se_Recipients;
	int	se_RecipientsSize;
	int	se_respforEHLO;
	int	se_doAuth; /* AUTH must be done before starting */
} SmtpEnv;
static SmtpEnv *smtpEnv;
#define Myver		smtpEnv->se_myver
#define Myhost		smtpEnv->se_myhost
#define Clhost		smtpEnv->se_clhost
#define Sender		smtpEnv->se_Sender
#define RecipientLocal	smtpEnv->se_RecipientLocal
#define Recipients	smtpEnv->se_Recipients
#define RecipientsSize	smtpEnv->se_RecipientsSize
#define respforEHLO	smtpEnv->se_respforEHLO
#define doAuth		smtpEnv->se_doAuth
minit_smtp()
{
	if( smtpEnv == 0 )
		smtpEnv = NewStruct(SmtpEnv);
}
extern char *malloc();

static SMTP_mount_rcpt(Conn,tc,req,xreq)
	Connection *Conn;
	FILE *tc;
	char *req;
	char *xreq;
{	char rcpt[LNSIZE],url[LNSIZE];
	SMTPreq Req;
	UTag **reqv,xrcpt;
	char *opts,*CTX_mount_url_to();

	*xreq = 0;
	decomp_req(req,&Req);
	reqv = Req.sm_uv;
	if( reqv[SMQ_RCPT] == 0 )
		return 0;

	Utos(reqv[SMQ_RCPT],rcpt);
	RFC822_addresspartX(rcpt,url,sizeof(url));

	if( SMTP_nomboxcase )
		strtolower(url,url);
	opts =
	CTX_mount_url_to(Conn,NULL,"POST",url);
	if( opts && strstr(opts,"reject") )
		goto FORBIDDEN;

	if( strcmp(rcpt,url) != 0 ){
		sv1log("%s => %s\n",rcpt,url);
		if( ufromsf(url,0,"smtp://%[^:/]",&xrcpt)
		 || ufromsf(url,0,"mailto:%s",&xrcpt)
		){
			if( ustrcmp(&xrcpt,"-") == 0 )
				goto FORBIDDEN;

			reqv[SMQ_RCPT] = &xrcpt;
			uvtosf(xreq,LNSIZE,"%s %s:<%s>%s",reqv);
			sv1log(">>>> %s",xreq);
		}
	}
	return 0;

FORBIDDEN:
	sv1log("Rejected: %s",req);
	fprintf(tc,"553 Forbidden recipient (%s)\r\n",rcpt);
	fflush(tc);
	return -1;
}
static notAvailable(com,tc,log)
	char *com;
	FILE *tc,*log;
{
	if( doAuth ){
		if( !strcaseeq(com,"XECHO")
		 && !strcaseeq(com,"HELO")
		 && !strcaseeq(com,"EHLO")
		 && !strcaseeq(com,"AUTH")
		 && !strcaseeq(com,"QUIT") ){
			if( tc != NULL ){
				lfprintf(log,tc,"500 do AUTH first.\r\n");
				fflush(tc);
			}
			return 1;
		}
	}
	return 0;
}

static SMTP_relay_req(Conn,fc,tc,ts,req)
	Connection *Conn;
	FILE *fc,*tc,*ts;
	char *req;
{	char xreq[LNSIZE];
	char com[LNSIZE],arg[LNSIZE],CRLF[8],*dp;

	if( fgets(req,LNSIZE,fc) == NULL ){
		sv1log("CS-EOF.\n");
		Fputs("QUIT\r\n",ts);
		return -1;
	}
	dp = wordScan(req,com);
	dp = lineScan(dp,arg);
	strcpy(CRLF,dp);
	if( strcaseeq(com,"AUTH") ){
		char arg1[64];
		dp = wordScan(arg,arg1);
		sv1log("SMTP < %s %s %s\r\n",com,arg1,*dp?"***":"");
	}else{
		sv1log("SMTP < %s",req);
	}

	if( notAvailable(com,NULL,NULL) )
		return 0;

	if( strcasecmp(com,"XECHO") == 0 ){
		fprintf(tc,"%s%s",arg,CRLF);
		fflush(tc);
		return 0;
	}
	if( SMTP_doauth && strcaseeq(com,"AUTH") ){
		return 0;
	}
	if( SMTP_doauth && strcaseeq(com,"EHLO") ){
		strcpy(com,"HELO");
		respforEHLO = 1;
	}

	/*
	 * JIS (2byte) codes shuld not be passed to the SMTP server
	 * and must be removed before applying MOUNT ...
	 */
	TO_euc(req,xreq,sizeof(xreq));
	if( strcmp(req,xreq) != 0 ){
		del8bits(req,xreq);
		sv1log("SMTP < %s",req);
	}

	xreq[0] = 0;
	if( strcasecmp(com,"HELO") == 0 ){
		sprintf(xreq,"HELO %s%s",arg,CRLF);
	}else
	if( strcasecmp(com,"RCPT") == 0 ){
		if( SMTP_mount_rcpt(Conn,tc,req,xreq) < 0 )
			return -1;
	}
	if( xreq[0] )
		strcpy(req,xreq);

	if( Fputs(req,ts) == EOF )
		return -1;
	return 0;
}

static doRCPT(Conn,indata,tc,arg,urcpt,log)
	Connection *Conn;
	FILE *tc;
	char *arg;
	UTag *urcpt;
	FILE *log;
{	char recipient[256];

	Utos(urcpt,recipient);
	RFC822_addresspartX(recipient,recipient,sizeof(recipient));

	sscanf(recipient,"%[^@]",RecipientLocal);

	if( Recipients == NULL ){
		RecipientsSize = 0x8000;
		Recipients = malloc(RecipientsSize);
		Recipients[0] = 0;
	}
	if( Recipients[0] != 0 )
		strcat(Recipients,",");
	strcat(Recipients,recipient);

	sv1log("Recipient: <%s> %s\n",recipient,arg);
	lfprintf(log,tc,"250 %s... Recipient ok\r\n",arg);
	return 0;
}
static doMAIL(Conn,tc,arg,ufrom,log)
	Connection *Conn;
	FILE *tc;
	char *arg;
	UTag *ufrom;
{
	Utos(ufrom,Sender);
	RFC822_addresspartX(Sender,Sender,sizeof(Sender));

	sv1log("Sender: <%s> %s\n",Sender,arg);
	lfprintf(log,tc,"250 %s... Sender ok\r\n",Sender);
	return 0;
}
static doRSET(Conn)
	Connection *Conn;
{
	if( Recipients != NULL ){
		free(Recipients);
		Recipients = NULL;
	}
	Sender[0] = 0;
}

char *MAILGATE = "mailgate";
static doDATA(Conn,indata,tc,fc,myhost,clhost,log)
	Connection *Conn;
	FILE *indata;
	FILE *tc,*fc;
	char *myhost,*clhost;
	FILE *log;
{	char line[1024];
	int off1;
	int off2;
	char *cpath;
	int sent;
	char *rb,*rv[0x2000];
	int ri,rc;
	char header,Cte[256];

	if( Recipients == NULL ){
		lfprintf(log,tc,"503 Need RCPT (recipient)\r\n");
		fflush(tc);
		return -1;
	}

	if( (SMTP_tolerance & SMT_NOFROM) == 0 )
	if( Sender[0] == 0 ){
		lfprintf(log,tc,"503 Need MAIL (sender)\r\n");
		fflush(tc);
		return -1;
	}

	lfprintf(log,tc,
		"354 Enter mail, end with \".\" on a line by itself\r\n");
	fflush(tc);

	off1 = ftell(indata);
	Cte[0] = 0;
	header = 1;
	while( fgets(line,sizeof(line),fc) ){
		if( line[0] == '.' && (line[1] == '\r' || line[1] =='\n') ){
			lfprintf(log,NULL,">>> %s",line);
			break;
		}

		if( header )
		if( line[0] == '\r' || line[0] == '\n' )
			header = 0;
		else
		if( strncasecmp(line,"Content-Transfer-Encoding",25) == 0 ){
			char fname[256],fvalue[256];
			fieldScan(line,fname,fvalue);
			if( Cte[0] ){
				lfprintf(log,NULL,"ignored -- dup. %s: %s %s\n",
					fname,Cte,fvalue);
				continue;
			}
			strcpy(Cte,fvalue);
		}

		fputs(line,indata);
	}

	off2 = ftell(indata);
	sent = 0;

	if( Sender[0] && strcaseeq(RecipientLocal,MAILGATE) ){
		fseek(indata,0,0);
		sent = MAILGATE_returnUID(Conn,tc,indata,myhost,clhost,log);
		return sent;
	}

	rb = stralloc(Recipients);

#ifdef OPTIMIZE
	fseek(indata,off1,0);
	sent = SMTPgateway(Conn,tc,indata,Sender,Recipients,log);
	if( sent < 0 )
		goto EXIT;
#else
	rc = stoV(rb,0x2000,rv,',');
	for( ri = 0; ri < rc; ri++ ){
		fseek(indata,off1,0);
		sent = SMTPgateway(Conn,tc,indata,Sender,rv[ri],log);
		if( sent < 0 )
			goto EXIT;
	}
#endif
	if( 0 < sent )
		lfprintf(log,tc,"250 Mail accepted\r\n");
	else	lfprintf(log,tc,"550 <%s> User Unknown\r\n",Recipients);
	fflush(tc);

	free(Recipients);
	Recipients = 0;
	fseek(indata,off2,0);

EXIT:
	free(rb);
	return sent;
}

char *MailGate(Conn)
	Connection *Conn;
{	static char *mbox;
	char buf[128],host[128],dom[128];

	if( mbox )
		return mbox;
	ClientIF_H(Conn,host);
	getFQDN(host,dom);
	if( IsInetaddr(dom) )
		sprintf(buf,"%s@[%s]",MAILGATE,dom);
	else	sprintf(buf,"%s@%s",MAILGATE,dom);
	mbox = stralloc(buf);
	return mbox;
}

extern FILE *SMTP_POST();
MAILGATE_returnUID(Conn,tc,indata,myhost,clhost,log)
	Connection *Conn;
	FILE *tc;
	FILE *indata;
	char *myhost,*clhost;
{	FILE *fp;
	char from[128],muid[128],date[128];

	if( streq(myhost,clhost) && SERVER_PORT() == 25 ){
		lfprintf(log,tc,"500 don't connect with myself to deliver.\r\n");
		return -1;
	}

	mboxid(Sender,clhost,muid,date);
	if( (fp = SMTP_POST(clhost,25,Sender,MailGate(Conn))) == NULL ){
		lfprintf(log,tc,"500 cannot connect SMTP server to return.\r\n");
		return -1;
	}

lfprintf(log,fp,"To: %s\r\n",Sender);
lfprintf(log,fp,"Subject: receipt to your mail\r\n");
 fprintf(fp,"\r\n");
lfprintf(log,fp,"YOUR MAIL ADDRESS IS REGISTERED AND ASSIGNED A UNIQUE IDENTIFIER:\r\n");
 fprintf(fp,"\r\n");
lfprintf(log,fp,"MUID: %s\r\n",muid);
lfprintf(log,fp,"MBOX: <%s>\r\n",Sender);
lfprintf(log,fp,"DATE: %s\r\n",date);
 fprintf(fp,"\r\n");
 fprintf(fp,"--INPUT--\r\n");
	copyfile1(indata,fp);
 fprintf(fp,"--\r\n");
	pclose(fp);

	lfprintf(log,tc,"250 Mail accepted\r\n",Recipients);
	return 1;
}

extern FILE *openMbox();
mboxid(mbox,mailer,muid,date)
	char *mbox,*mailer,*muid,*date;
{	char src[256],md5id[128],sdate[32];
	char line[1024];
	int now;
	FILE *fp;
	int exist;

	muid[0] = date[0] = 0;
	fp = openMbox(0,mbox,muid);
	if( fp != NULL ){
		fgets(line,sizeof(line),fp); sscanf(line,"MUID: %s",muid);
		fgets(line,sizeof(line),fp); sscanf(line,"DATE: %[^\r\n]",date);
	}
	if( muid[0] && date[0] )
		return 1;

	now = time(0);
	StrftimeGMT(date,sizeof(date),TIMEFORM_RFC822,now);
	sprintf(src,"%d<%s>",now,Sender);
	toMD5(src,md5id);
	md5id[6] = 0;
	strtoupper(md5id,md5id);
	StrftimeGMT(sdate,sizeof(sdate),"%y%m%d",now);
	sprintf(muid,"%s-%s",sdate,md5id);

	fp = openMbox(1,mbox,muid);
	if( fp == NULL )
		return -1;

	fprintf(fp,"MUID: %s\r\n",muid);
	fprintf(fp,"DATE: %s\r\n",date);
	fprintf(fp,"MAILER: %s\r\n",mailer);
	fclose(fp);
	return 0;
}

static SMTP_sendMYAUTH(Conn,log,ts,fs)
	Connection *Conn;
	FILE *log,*ts,*fs;
{	char authb[256],*ap,*pp,plain[256],bplain[256];
	char resp[256];

	if( get_MYAUTH(Conn,authb,"smtp",DST_HOST,DST_PORT) ){
		char myhost[256];
		if( gethostNAME(fileno(ts),myhost) <= 0 )
			gethostname(myhost,sizeof(myhost));
		getFQDN(myhost,myhost);
		SMTP_putserv(log,fs,ts,resp,"EHLO %s\r\n",myhost);
		if( atoi(resp) != 250 ){
			return -1;
		}

		pp = plain;
		ap = authb;
		strcpy(pp,""); /* user-ID */
		pp += strlen(pp) + 1;
		ap = wordscanY(ap,pp,128,"^:");
		pp += strlen(pp) + 1;
		if( *ap == ':' ){
			wordscanY(ap+1,pp,128,"^\r\n");
		}
		pp += strlen(pp);

		str_to64(plain,pp-plain,bplain,sizeof(bplain),1);
		if( pp = strpbrk(bplain,"\r\n") )
			*pp = 0;

		SMTP_putserv(log,fs,ts,resp,"AUTH PLAIN %s\r\n",bplain);
		if( atoi(resp) != 235 )
			return -1;
	}
	return 0;
}
static SMTP_AUTH(Conn,log,fc,tc,arg)
	Connection *Conn;
	FILE *log,*fc,*tc;
	char *arg;
{	char buff[256],*wp,id[256],user[256],pass[256];
	int len,rcode;

	wp = wordScan(arg,buff);
	if( strncasecmp(buff,"PLAIN",5) == 0 ){
		wordScan(wp,buff);
		len = str_from64(buff,strlen(buff),id,sizeof(id));
		id[len] = id[len+1] = id[len+2] = 0;
		wp = strchr(id,'\0');
		wp = lineScan(wp+1,user);
		wp = lineScan(wp+1,pass);
	}
	else
	if( strcaseeq(arg,"LOGIN") ){
		lfprintf(log,tc,"334 VXNlcm5hbWU6\r\n"); /* Username: */
		fflush(tc);
		fgetsTIMEOUT(buff,sizeof(buff),fc);
		str_from64(buff,strlen(buff),user,sizeof(user));

		lfprintf(log,tc,"334 UGFzc3dvcmQ6\r\n"); /* Password: */
		fflush(tc);
		fgetsTIMEOUT(buff,sizeof(buff),fc);
		str_from64(buff,strlen(buff),pass,sizeof(pass));
	}
	else
	{
		rcode = 504;
		lfprintf(log,tc,"504 unsupported method: %s\r\n",arg);
		fflush(tc);
		return rcode;
	}

	if( CTX_auth(Conn,user,pass) < 0 ){
		rcode = 500;
		lfprintf(log,tc,"500 Not Authenticated.\r\n");
	}else{
		rcode = 0;
		lfprintf(log,tc,"235 Authenticated and Authorized.\r\n");
	}
	fflush(tc);
	return rcode;
}
static SMTPserver(Conn,fc,tc)
	Connection *Conn;
	FILE *fc,*tc;
{	char myhost[256],clhost[128],stime[128];
	char req[1024],*wp,com[1024],arg[1024];
	char xreq[LNSIZE];
	FILE *indata,*log,*TMPFILE();
	int sent = 0;
	char peerHELO[256];
	SMTPreq Req;
	int rcptn = 0;

	getClientHostPort(Conn,clhost);
	ClientIF_name(Conn,FromC,myhost);
	getFQDN(myhost,myhost);
	peerHELO[0] = 0;

	StrftimeLocal(stime,sizeof(stime),TIMEFORM_RFC822,time(0));
	indata = TMPFILE("SMTP/DATA");
	log = TMPFILE("SMTP/LOG");

	lfprintf(log,tc,"220 %s SMTP/DeleGate/%s ready at %s\r\n",
		myhost,DELEGATE_ver(),stime);
	fflush(tc);

	doAuth = SMTP_doauth != 0;
	for(;;){
		if( fgets(req,sizeof(req),fc) == 0 )
			break;
		fputs(req,indata);
		lfprintf(log,NULL,">>> %s",req);

		wp = wordScan(req,com);
		lineScan(wp,arg);
		sv1log("SMTP [%s][%s]\r\n",com,arg);

		if( notAvailable(com,tc,log) )
			continue;

		if( (SMTP_tolerance & SMT_NOHELO) == 0 )
		if( peerHELO[0] == 0 )
		if( !strcaseeq(com,"EHLO") && !strcaseeq(com,"AUTH") )
		if( !strcaseeq(com,"HELO") && !strcaseeq(com,"QUIT") ){
			lfprintf(log,tc,"503 Say HELO first (%s/%s)\r\n",
				clhost,gethostaddr(clhost));
			fflush(tc);
			continue;
		}

		if( strcasecmp(com,"XECHO") == 0 ){
			fprintf(tc,"%s\r\n",arg);
		}else
		if( strcaseeq(com,"HELO") ){
			lfprintf(log,tc,"250 %s Hello %s (%s)\r\n",myhost,arg,
				clhost);
			lineScan(arg,peerHELO);
		}else
		if( strcaseeq(com,"EHLO") ){
			lineScan(arg,peerHELO);
			lfprintf(log,tc,"250-%s Hello %s (%s)\r\n",
				myhost,arg,clhost);
			if( SMTP_doauth ){
				lfprintf(log,tc,"250-AUTH PLAIN LOGIN\r\n");
			}
			lfprintf(log,tc,"250 XECHO\r\n");
		}else
		if( strcaseeq(com,"AUTH") ){
			if( SMTP_AUTH(Conn,log,fc,tc,arg) == 0 )
				doAuth = 0;
		}else
		if( strcaseeq(com,"RCPT") ){ /* To */
			if( SMTP_maxrcpt && SMTP_maxrcpt < ++rcptn ){
				lfprintf(log,tc,"500 too many recipients.\r\n");
				break;
			}
			if( SMTP_mount_rcpt(Conn,tc,req,xreq) < 0 )
				continue;
			decomp_req(xreq[0]?xreq:req,&Req);
			if( (sent = doRCPT(Conn,indata,tc,arg,Req.sm_uv[SMQ_RCPT],log)) < 0 )
				break;
		}else
		if( strcaseeq(com,"MAIL") ){ /* From */
			decomp_req(req,&Req);
			if( (sent = doMAIL(Conn,tc,arg,Req.sm_uv[SMQ_FROM],log)) < 0 )
				break;
		}else
		if( strcaseeq(com,"DATA") ){
			if( (sent = doDATA(Conn,indata,tc,fc,myhost,clhost,log)) < 0 )
				break;
			fflush(tc);
			fseek(log,0,0);
		}else
		if( strcaseeq(com,"RSET") ){
			doRSET(Conn);
			lfprintf(log,tc,"250 Reset state\r\n");
		}else
		if( strcaseeq(com,"QUIT") ){
			lfprintf(log,tc,"221 %s closing connection\r\n",myhost);
			break;
		}else{
			lfprintf(log,tc,"500 Command unrecognized\r\n");
		}
		fflush(tc);
	}
	fflush(tc);
	fclose(indata);
	fclose(log);
}

extern FILE *TMPFILE();
static relay_DATA(fc,tc,fs,ts,stat)
	FILE *fc,*tc,*fs,*ts;
	char *stat;
{	FILE *data;
	int rcode,size;
	char buff[16*1024],req[32];
	int rcc,wcc,qc;
	int nready;

	if( SMTP_thrudata < 0 ){
		thruRESP(fc,ts);
		return SMTP_relay_stat(fs,tc,stat);
	}
	if( 0 < SMTP_thrudata ){
		FILE *indata;
		int start,till,rem;

		indata = TMPFILE("SMTP-DATA-IN");
		start = time(0);
		till = start + SMTP_thrudata;
		rcc = 0;
		for(;;){
			rem = till - time(0);
			if( rem <= 0 || fPollIn(fc,rem*1000) <= 0 ){
			sv1log("## SMTP unbuffering slow DATA (%dB/%ds).\n",
					rcc,time(0)-start);
				fflush(indata);
				fseek(indata,0,0);
				copyfile1(indata,ts);
				fclose(indata);
				thruRESP(fc,ts);
				fflush(ts);
				return SMTP_relay_stat(fs,tc,stat);
			}
			do {
				if( fgets(buff,sizeof(buff),fc) == NULL )
					goto ENDDATA;
				fputs(buff,indata);
				rcc += strlen(buff);

				if( buff[0] == '.' )
				if( buff[1] == '\r' || buff[1] == '\n' )
					goto ENDDATA;
			} while( 0 < READYCC(fc) );
		} ENDDATA:
		fflush(indata);
		fseek(indata,0,0);
		data = TMPFILE("SMTP-DATA");
		PGPencodeMIME(indata,data);
		fclose(indata);
	}else{
	data = TMPFILE("SMTP-DATA");
	PGPencodeMIME(fc,data);
	}
	fflush(data);
	size = ftell(data);
	fseek(data,0,0);

	if( size <= SMTP_bgdatasize ){
		copyfile1(data,ts);
		fflush(ts);
		fclose(data);
		return SMTP_relay_stat(fs,tc,stat);
	}

	sv1log("## SMTP relaying large DATA (%d < %d bytes)\n",
		SMTP_bgdatasize,size);
	fprintf(tc,"250 Mail accepted\r\n");
	fflush(tc);

	rcode = 0;
	for(;;){
	    if( (rcc = fread(buff,1,sizeof(buff),data)) <= 0 )
		break;
	    wcc = fwrite(buff,1,rcc,ts);

	    if( rcode == 0 && (nready = fPollIn(fc,1)) ){
		rcode = 1;
		if( nready < 0 ){
			sv1log("## SMTP client EOF ? during DATA relay\n");
			rcode = -1;
		}else{
		    qc = fgetc(fc);
		    if( feof(fc) ){
			sv1log("## SMTP client EOF during DATA relay\n");
			rcode = -2;
		    }else{
			ungetc(qc,fc);
			if( toupper(qc) == 'Q' ){
				fprintf(tc,"221 DeleGate delivering mail\r\n");
				fflush(tc);
				closeFDs(fc,tc);

				sv1log("## SMTP Quit sent during DATA relay\n");
				rcode = -3;
			}
		    }
		}
	    }
	}
	fflush(ts);
	fclose(data);
/*
	if( 0 <= rcode )
*/
	if( 0 <= rcode || 0 < fPollIn(fs,10*1000) )
		rcode = SMTP_relay_stat(fs,NULL,stat);
	return rcode;
}

smtp_datamark(Conn,ts)
	Connection *Conn;
	FILE *ts;
{
	if( filter_withCFI(Conn,XF_FTOSV)
	 || filter_withCFI(Conn,XF_FTOMD)
	){
		putMESSAGEline(ts,"mime","DATA");
	}
}

service_smtp(Conn)
	Connection *Conn;
{	FILE *fc,*ts,*fs,*tc;
	char req[LNSIZE],stat[LNSIZE];
	char com[64],arg[64],*wp;
	int rcode;
	int QUITdone = 0;
	int rcptn = 0;

	fc = fdopen(FromC,"r");
	tc = fdopen(ToC,  "w");
	fs = NULL;
	ts = NULL;

	getClientHostPort(Conn,Clhost);
	ClientIF_name(Conn,FromC,Myhost);
	getFQDN(Myhost,Myhost);

	if( SMTP_doauth == 0 ){
		if( CTX_auth(Conn,NULL,NULL) ){
			SMTP_doauth = A_PLAIN | A_LOGIN;
		}
	}

	if( ToS < 0 || FromS < 0 ){
		if( isMYSELF(DFLT_HOST) )
			SMTPserver(Conn,fc,tc);
		goto EXIT;
	}else{
		fs = fdopen(FromS,"r");
		if( SMTP_relay_stat(fs,tc,stat) < 0 )
			goto EXIT;
		ts = fdopen(ToS,  "w");
		if( SMTP_sendMYAUTH(Conn,NULL,ts,fs) < 0 )
		{
			fprintf(tc,"%s\r\n","500 Authentication error.");
			fflush(tc);
			goto EXIT;
		}
	}

	doAuth = SMTP_doauth != 0;
	for(;;){
		respforEHLO = 0;
		if( SMTP_relay_req(Conn,fc,tc,ts,req) < 0 )
			break;

		wp = wordScan(req,com);
		lineScan(wp,arg);
		if( notAvailable(com,tc,NULL) ){
			continue;
		}
		if( strncasecmp(req,"XECHO",5) == 0 )
			continue;

		if( respforEHLO ){
			respforEHLO = 0;
			rcode = SMTP_relay_stat(fs,NULL,stat);
			if( rcode != 250 ){
				fputs(stat,tc);
				fflush(tc);
				break;
			}
			fprintf(tc,"250-%s Hello %s (%s)\r\n",
				Myhost,arg,Clhost);
			fprintf(tc,"250-AUTH PLAIN LOGIN\r\n");
			fprintf(tc,"250-AUTH=LOGIN\r\n");
			fputs(stat,tc);
			fflush(tc);
			continue;
		}
		if( SMTP_doauth && strcaseeq(com,"AUTH") ){
			if( SMTP_AUTH(Conn,NULL,fc,tc,arg) == 0 )
				doAuth = 0;
			continue;
		}

		if( (rcode = SMTP_relay_stat(fs,tc,stat)) < 0 )
			break;

		if( strcaseeq(com,"RCPT") ){
			if( SMTP_maxrcpt && SMTP_maxrcpt < ++rcptn ){
				fprintf(tc,"500 too many recipients.\r\n");
				fflush(tc);
				break;
			}
		}
		if( rcode == 354 ){ /* DATA */
			smtp_datamark(Conn,ts);

			if( (rcode = relay_DATA(fc,tc,fs,ts,stat)) < 0 )
				break;
		}
		if( strcasecmp(req,"QUIT\r\n") == 0 ){
			sv1log("CS-QUIT\n");
			QUITdone = 1;
			break;
		}
	}
	if( !QUITdone ){
		fputs("QUIT\r\n",ts);
		fflush(ts);
		if( 0 < fPollIn(fs,10*1000) )
		SMTP_relay_stat(fs,NULL,stat);
	}
EXIT:
	fclose(fc);
	fclose(tc);
	if( fs != NULL ) fclose(fs);
	if( ts != NULL ) fclose(ts);
	return 0;
}

static char *myver(){
	char tmp[128];

	if( Myver == NULL ){
		sprintf(tmp,"SMTP-DeleGate/%s",DELEGATE_ver());
		Myver = StrAlloc(tmp);
	}
	return Myver;
}

SMTP_open(Conn,fpv,host,port,to,from,dodata,log)
	Connection *Conn;
	FILE *fpv[];
	char *host;
	char *to,*from;
	FILE *log;
{	FILE *fs,*ts;
	char *auth,authb[256];

	set_realserver(Conn,"smtp",host,port);
	if( connect_to_serv(Conn,FromC,ToC,0) < 0 )
		return -1;

	fs = fdopen(FromS,"r");
	ts = fdopen(ToS,"w");

	if( get_MYAUTH(Conn,authb,"smtp",host,port) )
		auth = authb;
	else	auth = NULL;

	if( SMTP_openingX(fs,ts,myver(),to,from,dodata,log,auth) != 0 ){
		char resp[1024];
		SMTP_putserv(log,fs,ts,resp,"QUIT\r\n");
		fclose(ts);
		fclose(fs);
		return -1;
	}

	fpv[0] = fs;
	fpv[1] = ts;
	return ToS;
}

extern FILE *TMPFILE();
typedef struct {
	int	va_checkuser;
	char   *va_addr;
	int	va_valid;
} validAddr;
static validAddr *validAddrs;

validateEmailAddr(addr,checkuser)
	char *addr;
{	Connection ConnBuf,*Conn = &ConnBuf;
	FILE *fpv[2],*log;
	int svsock;
	char user[256],host[256],mxhost[256];
	char addrb[256];
	validAddr *va;

	if( validAddrs == NULL )
		validAddrs = NewStruct(validAddr);

	va = &validAddrs[0];
	if( va->va_addr != 0 )
	if( strcmp(addr,va->va_addr) == 0 )
	if( checkuser == va->va_checkuser )
		return va->va_valid;

	RFC822_addresspartX(addr,addrb,sizeof(addrb));
	addr = addrb;

	if( sscanf(addr,"%[^@]@%s",user,host) != 2 )
		return -1;
	if( hostcmp("localhost",host) == 0 )
		return -1;

	sprintf(mxhost,"-MX.%s",host);
	if( IsResolvable(mxhost) )
		strcpy(host,mxhost);

	if( checkuser == 0 )
	{
		syslog_ERROR("DONT CHECK USER PART[%s]@%s\n",user,host);
		user[0] = 0;
	}

	ConnInit(Conn);
	Conn->from_myself = 1;
	Conn->co_mask |= CONN_NOPROXY | CONN_NOSOCKS;

	log = TMPFILE("emailValid");
	svsock = SMTP_open(Conn,fpv,host,25,user,DELEGATE_ADMIN,0,log);
	fclose(log);

	va = &validAddrs[0];
	va->va_checkuser = checkuser;
	va->va_addr = stralloc(addr);
	if( svsock < 0 )
	{
		va->va_valid = -1;
		return -1;
	}
	else{
		char resp[1024];
		SMTP_putserv(log,fpv[0],fpv[1],resp,"RSET\r\n");
		SMTP_putserv(log,fpv[0],fpv[1],resp,"QUIT\r\n");
		fclose(fpv[0]);
		fclose(fpv[1]);
		va->va_valid = 0;
		return 0;
	}
}

FILE *SMTP_POST(host,port,to,from)
	char *host,*to,*from;
{
	return SMTP_post(host,port,myver(),to,from,NULL,NULL);
}

char *DELEGATE_SMTPSERVER;
scan_SMTPSERVER(smtpserver)
	char *smtpserver;
{
	if( smtpserver )
		Strdup(&DELEGATE_SMTPSERVER,smtpserver);
}

#define MAILER	"/usr/lib/sendmail -t"
extern FILE *popen();
FILE *openMailPoster(to,from)
	char *to,*from;
{	FILE *out;
	char mailer[1024];
	int host[512];
	int port;

	out = NULL;
	if( DELEGATE_SMTPSERVER ){
		sprintf(mailer,"SMTPSERVER %s",DELEGATE_SMTPSERVER);
		port = 25;
		sscanf(DELEGATE_SMTPSERVER,"%[^:]:%d",host,&port);
		out = SMTP_POST(host,port,to,from);
		sv1log("#### SMTPSERVER [%x] %s\n",out,DELEGATE_SMTPSERVER);
	}
	if( out == NULL ){
		sprintf(mailer,MAILER,to);
		out = popen(mailer,"w");
	}
	if( out != NULL )
		fprintf(out,"X-Mailer: %s\n",mailer);
	return out;
}


FILE *SMTP_getEXPN(addr)
	char *addr;
{	char user[256],host[256];
	int server;
	FILE *ts,*fs;
	char stat[256];
	FILE *afp,*TMPFILE();
	int nlines;

	user[0] = host[0] = 0;
	sscanf(addr,"%[^@]@%s",user,host);
	server = client_open("SMTPGATE/EXPN","smtp",host,25);
	if( server < 0 ){
		sv1log("#### cannot EXPN <%s>\n",addr);
		return NULL;
	}

	fs = fdopen(server,"r");
	if( fgets(stat,sizeof(stat),fs) == NULL ){
		sv1log("#### EXPN error -- server closed\n");
		fclose(fs);
		return NULL;
	}
	sv1log("#### %s",stat);
	if( atoi(stat) != 220 ){
		fclose(fs);
		return NULL;
	}

	ts = fdopen(server,"w");
	fprintf(ts,"EXPN %s\r\n",user);
	fflush(ts);

	if( fgets(stat,sizeof(stat),fs) == NULL ){
		sv1log("#### EXPN error -- server closed\n");
		fclose(afp);
		afp = NULL;
	}else
	if( atoi(stat) != 250 ){
		sv1log("#### EXPN error -- %s",stat);
		fclose(afp);
		afp = NULL;
	}else{
		afp = TMPFILE("SMTPGATE/EXPN");
		fputs(stat,afp);
		for( nlines = 1; strncmp(stat,"250-",4) == 0; nlines++ ){
			if( fgets(stat,sizeof(stat),fs) == NULL )
				break;
			fputs(stat,afp);
		}
		fflush(afp);
		fseek(afp,0,0);
		sv1log("#### EXPN got <%s> -- %d lines\n",addr,nlines);
	}
	fclose(ts);
	fclose(fs);
	return afp;
}

/*
 *	-Fsendmail To URL
 *	-Fcopy srcURL dstURL
 *	-Fcopy nntp://server/group/number mailto:user@host.domain
 */
sendmail_main(ac,av,Conn)
	char *av[];
	Connection *Conn;
{	char *to,*from;
	FILE *afp,*log;

	to = "";
	from = DELEGATE_ADMIN;
	afp = stdin;
	log = stderr;

	if( 1 < ac )
		to = av[1];

	sendmail1(Conn,to,from,afp,log);
	if( afp != stdin )
		fclose(afp);
}

relayMSGdata(in,out,anl,eom)
	FILE *in,*out;
{	int cc;
	int nlend;
	char line[1024];

	cc = 0;
	nlend = 1;
	while( fgets(line,sizeof(line),in) != NULL ){
		nlend = strtailchr(line) == '\n';
		if( line[0] == '.' && (line[1] == '\r' || line[1] == '\n') )
			break;
		fputs(line,out);
		cc += strlen(line);
	}
	if( anl && !nlend ){
		sv1log("DATA ended without LF: size=%d\n",cc);
		fputs("\r\n",out);
	}
	if( eom ){
		fputs(".\r\n",out);
	}
	return cc;
}

sendmail1(Conn,to,from,afp,log)
	Connection *Conn;
	char *to,*from;
	FILE *afp,*log;
{	char hostport[256],host[256];
	FILE *fpv[2];
	int port;
	int cc;

	hostport[0] = 0;
	if( DELEGATE_SMTPSERVER && DELEGATE_SMTPSERVER[0] )
		strcpy(hostport,DELEGATE_SMTPSERVER);
	else	sscanf(to,"%*[^@]@%s",hostport);
	if( hostport[0] == 0 )
		strcpy(host,"localhost");

	port = scan_hostport("smtp",hostport,host);
	Conn->from_myself = 1;

	if( SMTP_open(Conn,fpv,host,port,to,from,1,log) < 0 ){
		fprintf(stderr,"COULD NOT OPEN %s:%d\n",host,port);
		return -1;
	}
	cc = relayMSGdata(afp,fpv[1],1,1);
	fflush(fpv[1]);
	if( 0 < fPollIn(fpv[0],1000) )
		SMTP_relay_stat(fpv[0],NULL,NULL);
	fprintf(fpv[1],"QUIT\r\n");
	fflush(fpv[1]);
	if( 0 < fPollIn(fpv[0],1000) )
		SMTP_relay_stat(fpv[0],NULL,NULL);
	fclose(fpv[1]);
	fclose(fpv[0]);
}

#endif /* !LIBRARY */
