/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1994-1999 Yutaka Sato
Copyright (c) 1994-1999 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, and distribute this material for any purpose
and without fee is hereby granted, provided that the above copyright
notice and this permission notice appear in all copies.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	mount.c (mount URL to/from)
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:

	MOUNT="virtualUrl realUrl optionList"

	=://=:=/*	... ///*
	=://=/*		... //=:0/*
	=://hostX/*	... //hostX:0/*
	=://hostX:=/*   ... //hostX:=/*
	=://=:portX/*   ... //=:portX/*

History:
	940806	created
	9907xx	full-URL in the left-hand (host name/addr matching)
		wild-card hostList like http://192.31.200.* in left-hand
		MOUNT="/-_-* * dstproto={protoList},dst={siteList}"
		MOUNT="/path/* http://* dst={siteList}"
ToDo:

	- proto1://site1/path1 proto2://site2/path2
	  where site1 is a "name of a hostList" (or Site in terms of URL)
	- proto1://+siteList/path1 proto2://site2/path2 vhost=siteList
	- MOUNT="http:// * /path1 http:// * /path2"
	- reverse matching in left hand (ex. *.gif)

//////////////////////////////////////////////////////////////////////#*/
#include "ystring.h"
#include "url.h"
extern char *CTX_clif_proto();
extern char *CTX_clif_host();
extern char *CTX_clif_hostport();
extern char *file_hostpath();
extern char *topofPath1();
extern char *getv();
extern char *gethostaddr();
extern char *gethostaddr_fromcache();
#define Strcpy(d,s)	(strcpy(d,s), d+strlen(d))
extern char *strtail();
extern char *frex_create(),*frex_match();
extern char *CTX_reqstr();


#ifndef NULL
#define NULL 0L
#endif

#ifndef FILE
#define FILE void
#endif
#include "log.h"
#define debug	((LOG_type&L_MOUNT)==0)?0:putLog0

#ifndef MTAB_SIZE
#define MTAB_SIZE	256
#endif

#ifdef DONT_SORT_MTAB
#define DO_MTAB_SORT 0
#else
#define DO_MTAB_SORT 1
#endif

#define ASIS		"="

#define C_CLHOST	0x00000001
#define C_CLFROM	0x00000002
#define C_CLVIA		0x00000004
#define C_CLPATH	0x00000008
#define C_CLCONDS	0x0000000F
#define C_METHOD	0x00000010
#define C_WITHQUERY	0x00000020
#define C_ASPROXY	0x00000040
#define C_DIRECTION	0x00000080
#define C_DSTPROTO	0x00000100
#define C_QHOST		0x00000200 /* destination host in the request URL */
#define C_RHOST		0x00000400 /* destination host in the response */
#define C_DSTHOST	(C_QHOST|C_RHOST)
#define C_DEFAULT	0x00000800 /* default mount derived from SERVER param. */
#define C_VHOST		0x00001000
#define C_REQMATCH	0x00002000 /* mathing by the contents of request */
#define C_NOCASE	0x00004000 /* ignore upper/lower case in matching */
#define C_ALL		0x0000FFFF

#define U_MOUNT		0x00010000
#define U_MOVED_TO	0x00020000
#define U_USE_PROXY	0x00040000
#define U_DISPPORT	0x00100000
#define U_PORTMAP	0x00200000
#define U_RESOLV	0x00400000
#define U_CGI		0x01000000
#define U_MYBASE	0x02000000
#define U_REQDUMP	0x04000000
#define U_PRIORITY	0x10000000
#define U_ALL		0xFFFF0000

typedef struct {
	char	*u_src;
	int	 u_fullurl;
	int	 u_remain;
	int	 u_uvfmt;
	char	*u_format;
	int	 u_format_toEOL; /* matching to the End Of Line */
	char	*u_rformat; /* reversed format for Dst match & Src gen. */
	char	*u_prefix;
	int	 u_emptysite;
	int	 u_path2site;	/* "/path/* scheme://*"	*/
	int	 u_any;		/* "/path/* *"		*/

	char	*u_proto;
	char	*u_userpass;
	char	*u_user;
	char	*u_pass;

/* caching and matching of host/addr should be done in hostlist.c */
	int	 u_hostList;
	char	*u_hostn;
	char	*u_hosta;
	int	 u_iport;
	int	 u_stdport;
	char	*u_portmap;
	int	 u_iport_mapped;

	char	*u_path;
	int	 u_plen;
	char	*u_query;
	int	 u_qlen;
	int	 u_asis;
	int	 u_proto_ASIS;
	int	 u_host_ASIS;
	int	 u_port_ASIS;
} Url;

typedef struct {
	int	c_direction;
	int	c_type;
	union {
	char   *c_STR;
	int	c_INT;
	char   *c_REX;
	} c_val;
} Cond1;
#define c_str	c_val.c_STR
#define c_int	c_val.c_INT
#define c_rex	c_val.c_REX

typedef struct {
	int	 u_serno;
	int	 u_compiled;
	int	 u_disabled;
	int	 u_flags;
	double	 u_priority;

	int	 u_conds;
	int	 u_conds_value;
	char	*u_conds_param;

	Cond1	 u_condList[20];
	Url	 Src;
	Url	 Dst;
	int	 u_xcond;
	int	 u_timeout;
} Mtab;

#define	L_VIA		1
#define L_PATH		2
#define L_CLIF		3
#define L_FROM		4
#define L_DST		5
#define L_VHOST		6
#define viaList		u_condList[L_VIA].c_int
#define pathList	u_condList[L_PATH].c_int
#define clifList	u_condList[L_CLIF].c_int
#define D_clifList	u_condList[L_CLIF].c_direction
#define fromList	u_condList[L_FROM].c_int
#define dstList		u_condList[L_DST].c_int
#define vhostList	u_condList[L_VHOST].c_int
#define D_vhostList	u_condList[L_VHOST].c_direction

#define R_REQMATCH	7

#define S_OPTIONS	8
#define S_USEPROXY	9
#define S_METHOD	10
#define S_DSTPROTO	11
#define S_FTOCL		12
#define S_WHOLE		13
#define S_MYBASE	14
#define S_REQDUMP	15

#define u_reqpat	u_condList[R_REQMATCH].c_rex
#define u_opts		u_condList[S_OPTIONS].c_str
#define u_useproxy	u_condList[S_USEPROXY].c_str
#define u_method	u_condList[S_METHOD].c_str
#define u_D_method	u_condList[S_METHOD].c_direction
#define u_dstproto	u_condList[S_DSTPROTO].c_str
#define u_ftocl		u_condList[S_FTOCL].c_str
#define u_whole		u_condList[S_WHOLE].c_str
#define u_mybase	u_condList[S_MYBASE].c_str

#define ENUMBASE	16
#define N_DIRECTION	16
#define u_direction	u_condList[N_DIRECTION].c_int

#define N_MOVED_TO	17
#define u_moved		u_condList[N_MOVED_TO].c_int

typedef struct {
	int	 me_do_sort;
	Mtab	*me_mtab[MTAB_SIZE];
	int	 me_mtabN;
	int	 me_init_from;
	int	 me_last_hit;
	int	 me_last_forw;
	char	 me_myhost[256]; /* myself(vhost) of the last forward MOUNT */
	int	 me_myport;
} MountEnv;
static MountEnv *mountEnv;
#define mount mountEnv[0]
minit_mount()
{
	if( mountEnv == 0 ){
		mountEnv = NewStruct(MountEnv);
		mount.me_do_sort = DO_MTAB_SORT; 
		mount.me_last_forw = -1;
	}
}

#define mtab		mount.me_mtab
#define mtabN		mount.me_mtabN
#define do_MTAB_SORT	mount.me_do_sort
#define init_from	mount.me_init_from
#define last_hit	mount.me_last_hit
#define last_forw	mount.me_last_forw
#define my_h		mount.me_myhost
#define my_p		mount.me_myport

clear_mtab(){
	mtabN = 0;
	init_from = 0;
	last_hit = 0;
	if( mtab[0] ){
		mtab[0] = 0;
	}
}

#define D_FORWONLY	1 /* fo: left to right only */
#define D_BACKONLY	2 /* bo: right to left only */
#define D_BACKIFFORW	3 /* bif: right to left if left to right */
#define V_DIRECTION	"fo,bo,bif"
static int I_DIRECTION[] = {D_FORWONLY,D_BACKONLY,D_BACKIFFORW};

#define V_MOVED_TO	",300,301,302,303"
static int I_MOVED_TO[] = {302,300,301,302,303};

static struct enumlist {
	char	*e_list;
	int	*e_value;
} enumlist[] = {
	{V_DIRECTION,	I_DIRECTION},
	{V_MOVED_TO,	I_MOVED_TO}
};
static enumopt(lidx,name,val)
	char *name,*val;
{	char *enums;
	int vidx,ival;
	struct enumlist *ep;

	ep = &enumlist[lidx-ENUMBASE];
	enums = ep->e_list;
	vidx = isinList(enums,val);
	if( 0 < vidx ){
		if( ep->e_value )
			ival = ep->e_value[vidx-1];
		else	ival = vidx;
		return ival;
	}
	sv1log("ERROR: %s='%s' ? must be one of {%s}\n",name,val,enums);
	return 0;
}

#define CFO	1	/* "-f.condition" applied to forwarding request */
#define CBO	2	/* "-b.condition" applied to backwarded response */
#define CFB	3	/* "condition" applied to both direction (default) */

struct {
	int	 o_direction; /* forced direction overriding -f,-b context */
	char	*o_name;
	int	 o_flags;
	int	 o_value;
	int	 o_listx;
	char	 o_enum;
	int	 o_rex;
} mount_opts[] = {
	{0,  "default",	C_DEFAULT},
	{0,  "pri",	U_PRIORITY},
	{0,  "dstproto",C_DSTPROTO,	S_DSTPROTO	},
	{0,  "dst",	C_DSTHOST,	0,L_DST		},
	{0,  "qhost",	C_QHOST,	0,L_DST		},
	{0,  "rhost",	C_RHOST,	0,L_DST		},
	{CFO,"vhost",	C_VHOST,	0,L_VHOST	},
	{0,  "from",	C_CLFROM,	0,L_FROM	},
	{0,  "host",	C_CLHOST,	0,L_CLIF	},
	{0,  "path",	C_CLPATH,	0,L_PATH	},
	{0,  "src",	C_CLVIA,	0,L_VIA		},
	{0,  "via",	C_CLVIA,	0,L_VIA		},
	{0,  "method",	C_METHOD,	S_METHOD	},
	{0,  "direction",C_DIRECTION,	0,0,N_DIRECTION	},
	{0,  "asproxy",	C_ASPROXY,			},
	{0,  "withquery",C_WITHQUERY,			},
	{0,  "ident",	}, /* client's user name got from Ident server */
	{0,  "qmatch",	C_REQMATCH,	0,0,0,R_REQMATCH},
	{0,  "nocase",	C_NOCASE,			},

	{0,  "moved",	U_MOVED_TO,	0,0,N_MOVED_TO	},
	{0,  "useproxy",U_USE_PROXY,	S_USEPROXY	},
	{0,  "resolv",	U_RESOLV,			},
	{0,  "mybase",	U_MYBASE,	S_MYBASE	},
	{0,  "qdump",	U_REQDUMP,	S_REQDUMP	},

	{0,  "htmlconv",	},/* META, SSI, DHTML */
	{0,  "uriconv",		},

	{0,  "ftocl",	0,		S_FTOCL		},
	{0,  "ftosv",		},
	{0,  "ro",		},/* read only */
	{0,  "rw",		},/* read & write (FTP) */

	{0,  "-f",		},/* conditions for forward follows */
	{0,  "-b",		},/* conditions for backward follows */

	{0,  "owner",		},/* dynamic mount from remote */

	{0,  "delay",		},/* scheduling, priority, ... */
	{0,  "forbidden",	},/* equiv. rcode=403 for HTTP */
	{0,  "unknown",		},/* equiv. rcode=404 for HTTP */

	/* cache control */
	{0,  "expire",		},
	{0,  "cache",		},

	/* routing by MOUNT ? */
	{0,  "master",		},
	{0,  "proxy",		},

	/* HTTP */
	{0,  "cgi",	U_CGI	},
	{0,  "genvhost",	},/* generate Host: in forwarding req. */
	{0,  "robots",		},/* /robots.txt control */
	{0,  "charcode",	},
	{0,  "rcode",		},/* response code */
	{0,  "rhead",		},/* response header in MIME format */
	{0,  "authorizer",	},/* AUTHORIZER parameter */

	/* NNTP */
	{0,  "hide",		},/* NNTP group mask */
	{0,  "upact",		},
	{0,  "pathhost",	},
	0
};

static scanopt(opt,mt,directp)
	char *opt;
	Mtab *mt;
	int *directp;
{	char buf[1024],*val,*nam1,what[32];
	int oi,flag1,val1,list1;
	char direction,*op,d1,d2;

	direction = *directp;
	d1 = 0;
	if( *opt == '-' ){
		switch( opt[1] ){
			case 'f': direction = CFO; break;
			case 'b': direction = CBO; break;
		}
		if( opt[2] == 0 ){
			*directp = direction;
			return 0;
		}
		if( op = strchr(opt,'.') ){
			op++;
			d1 = direction;
		}else{
			sv1log("ERROR: MountOption ? %s\n",opt);
			return 0;
		}
	}

	strcpy(buf,opt);
	if( val = strchr(buf,'=') )
		*val++ = 0;
	else	val = "";

	for( oi = 0; nam1 = mount_opts[oi].o_name; oi++ ){
		if( !streq(buf,nam1) )
			continue;

		if( d1 == 0 )
		if( d2 = mount_opts[oi].o_direction )
			direction = d2;

		if( flag1 = (mount_opts[oi].o_flags & U_ALL) )
			mt->u_flags |= flag1;
		if( flag1 = (mount_opts[oi].o_flags & C_ALL) )
			mt->u_conds |= flag1; 

		if( mount_opts[oi].o_flags == U_PRIORITY ){
			sscanf(val,"%lf",&mt->u_priority);
			continue;
		}
		if( val1 = mount_opts[oi].o_value ){
			mt->u_condList[val1].c_str = StrAlloc(val);
			mt->u_condList[val1].c_direction = direction;
		}
		if( list1 = mount_opts[oi].o_listx ){
			sprintf(what,"MOUNT.%s",buf);
			mt->u_condList[list1].c_int = makePathList(what,StrAlloc(val));
			mt->u_condList[list1].c_direction = direction;
		}
		if( val1 = mount_opts[oi].o_enum ){
			mt->u_condList[val1].c_int = enumopt(val1,nam1,val);
			mt->u_condList[val1].c_direction = direction;
		}
		if( val1 = mount_opts[oi].o_rex ){
			nonxalpha_unescape(val,buf,1);
			val = buf;
			mt->u_condList[val1].c_rex = frex_create(val);
			mt->u_condList[val1].c_direction = direction;
		}
		break;
	}
	return 0;
}
mount_disable_lasthit(){
	if( last_hit ){
		mtab[last_hit-1]->u_disabled = 1;
		return last_hit;
	}
	return 0;
}
mount_enable(last){
	if( last )
		mtab[last-1]->u_disabled = 0;
}
mount_nodefaults(iproto,on)
	char *iproto;
{	int mi;
	Mtab *mt;

	for( mi = 0; mi < mtabN; mi++ ){
		mt = mtab[mi];
		if( mt->u_conds & C_DEFAULT ){
			mt->u_disabled = on;
		}
	}
}
Mounted()
{	int mi,mn;

	mn = 0;
	for( mi = 0; mi < mtabN; mi++ )
		if( !mtab[mi]->Dst.u_asis )
			mn++;
	return mn;
}
MountedConditional()
{	int mi,mn;
	Mtab *mt;

	mn = 0;
	for( mi = 0; mi < mtabN; mi++ ){
		mt = mtab[mi];
		if( mt->u_conds & C_CLCONDS )
			mn++;
	}
	return mn;
}
scan_MOUNT(Conn,spec)
	char *Conn;
	char *spec;
{	char src[1024],dst[1024],opts[1024];
	char xdst[1024];

	opts[0] = 0;
	if( sscanf(spec,"%s %s %s",src,dst,opts) < 2 )
	{
		sv1log("MOUNT=\"%s\" ? (missing right hand)\n",spec);
		return;
	}
	Verbose("%s\n",spec);
	set_MOUNT(Conn,src,dst,opts);
}
char *getMountSrcByDst(dst)
	char *dst;
{	Mtab *mt;
	int mi;

	for( mi = 0; mi < mtabN; mi++ ){
		mt = mtab[mi];
		if( streq(mt->Dst.u_src,dst) )
			return mt->Src.u_src;
	}
	return NULL;
}

set_MOUNT(Conn,src,dst,opts)
	char *Conn;
	char *src,*dst,*opts;
{	Mtab *mt;
	int mi;
	int len;
	char dstbuf[1024];
	char whole[1024];
	char *dp;
	int direction;

	if( mtab[0] == NULL ){
		mtab[0] = (Mtab*)-1;

		/* These patterns should be able to be re-defined by user ... */
		/*
		set_MOUNT(Conn,"/-*",ASIS,"");
		set_MOUNT(Conn,"/=*",ASIS,"");
		if( streq(src,"/-*") && streq(dst,ASIS) && opts[0]==0 ) return;
		if( streq(src,"/=*") && streq(dst,ASIS) && opts[0]==0 ) return;
		*/
		set_MOUNT(Conn,"/-*",ASIS,"default");
		set_MOUNT(Conn,"/=*",ASIS,"default");
	}

	sprintf(whole,"%s %s %s",src,dst,opts);
	for( mi = 0; mi < mtabN; mi++ ){
		mt = mtab[mi];
		if( streq(mt->u_whole,whole) ){
			Verbose("IGNORE DUPLICATE MOUNT[%d] %s\n",mi,whole);
			return;
		}
	}

	mt = NewStruct(Mtab);
	if( mt == NULL )
		return;
	mtab[mtabN++] = mt;
	mt->u_whole = StrAlloc(whole);
	mt->u_serno = mtabN;
	mt->u_xcond = getCondition(Conn);

	mt->Src.u_src = StrAlloc(src);
	mt->Src.u_fullurl = isFullURL(src);
	len = strlen(src);

	if( dp = strstr(mt->Src.u_src,"*%") ){
		*dp = 0;
		mt->Src.u_remain = 1;
		mt->Src.u_uvfmt = 1;
		if( strtailchr(dp+1) == '$' ){
			*strtail(dp+1) = 0;
			mt->Src.u_format_toEOL = 1;
		}
		mt->Src.u_format = StrAlloc(dp+1);
	}else
	if( mt->Src.u_src[len-1] == '*' ){
		mt->Src.u_src[len-1] = 0;
		mt->Src.u_remain = 1;
		if( 1 < len && mt->Src.u_src[len-2] == ']' ){
			if( dp = strrchr(mt->Src.u_src,'[') ){
				mt->Src.u_format = StrAlloc(dp);
				*dp = 0;
			}
		}
	}

	if( strncmp(dst,"///",3) == 0 ){
		sprintf(dstbuf,"%s://%s:%s/%s",ASIS,ASIS,ASIS,dst+3);
		dst = dstbuf;
	}else
	if( strncmp(dst,"//",2) == 0 ){
		sprintf(dstbuf,"%s://%s",ASIS,dst+2);
		dst = dstbuf;
	}else
	if( *dst == '/' ){
		sprintf(dstbuf,"file://%s",dst);
		dst = dstbuf;
	}
	mt->Dst.u_src = StrAlloc(dst);
	if( dp = strstr(mt->Dst.u_src,"*%") ){
		*dp = 0;
		mt->Dst.u_remain = 1;
		mt->Dst.u_uvfmt = 1;
		mt->Dst.u_format = StrAlloc(dp+1);
		uvfmtreverse(mt->Src.u_format,mt->Dst.u_format,
			&mt->Src.u_rformat,&mt->Dst.u_rformat);
	}else{
	len = strlen(dst);
	if( mt->Dst.u_src[len-1] == ']' ){
		char *dp;
		if( dp = strrchr(mt->Dst.u_src,'[') )
		if( mt->Dst.u_src < dp && dp[-1] == '*' ){
			*dp = 0;
			len = strlen(mt->Dst.u_src);
			mt->Dst.u_format = StrAlloc(dp+1);
			*strchr(mt->Dst.u_format,']') = 0;
		}
	}
	if( mt->Dst.u_src[len-1] == '*' ){
		mt->Dst.u_src[len-1] = 0;
		mt->Dst.u_remain = 1;
	}
	}

	mt->u_opts = StrAlloc(opts);
	direction = CFB;
	scan_commaList(opts,1,scanopt,mt,&direction);
}
set_MOUNT_ifndef(Conn,src,dst,opts)
	char *Conn;
	char *src,*dst,*opts;
{	int mi;
	char srcbuf[1024];
	int len,remain;

	strcpy(srcbuf,src);
	len = strlen(src);
	if( srcbuf[len-1] == '*' ){
		srcbuf[len-1] = 0;
		remain = 1;
	}else	remain = 0;

	for( mi = 0; mi < mtabN; mi++ ){
		if( strcmp(mtab[mi]->Src.u_src,srcbuf) == 0 )
		if( strcmp(mtab[mi]->u_opts,opts) == 0 )
			/* if ``cond'' are different */
			return;
	}
	if( *opts == 0 )
		opts = "default";
	set_MOUNT(Conn,src,dst,opts);
}

/*
 *  - A rule with a left hand ``Path1*'' should be after the rule with
 *    a left hand ``Path1Path2''.
 *  - In rules of the same left hand ``Path1'' with each other should be
 *    sorted (or overwritten if without any condition) in reverse order
 *    of the definition.
 *
 *  return S1_S2;  s1 should be before than s2
 *  return S2_S1;  s2 should be before than s1
 */

#define S2_S1	 1
#define S1_S2	-1

static mtabcomp(mt1p,mt2p)
	Mtab **mt1p,**mt2p;
{	Mtab *mt1,*mt2;
	char *s1,*s2;
	char *fs1,*fs2;
	int r1,r2;
	int comp;

	mt1 = *mt1p; if( mt1 == NULL ) return S2_S1;
	mt2 = *mt2p; if( mt2 == NULL ) return S1_S2;

	if( mt1->u_priority < mt2->u_priority )
		return S2_S1;
	if( mt2->u_priority < mt1->u_priority )
		return S1_S2;

	s1 = mt1->Src.u_src;
	s2 = mt2->Src.u_src;
	r1 = mt1->Src.u_remain;
	r2 = mt2->Src.u_remain;

	/*if( mt1->Dst.u_asis && mt2->Dst.u_asis )*/
/*
	if( streq(s1,ASIS) && streq(s2,ASIS) )
	{
		s1 = mt1->Dst.u_src;
		s2 = mt2->Dst.u_src;
		r1 = mt1->Dst.u_remain;
		r2 = mt2->Dst.u_remain;
	}
*/

	comp = strcmp(s1,s2);

	if( comp == 0 ){
		if( r1 && !r2 ) return S2_S1;
		if( r2 && !r1 ) return S1_S2;

		/* rule with rexp-prefix first */
		if(  mt1->Src.u_format && !mt2->Src.u_format ) return S1_S2;
		if( !mt1->Src.u_format &&  mt2->Src.u_format ) return S2_S1;

		/* rule with scanf() matching pattern */
		if( (fs1 = mt1->Src.u_format) && (fs2 = mt2->Src.u_format) ){
			if( strstr(fs1,fs2) ) return S1_S2;
			if( strstr(fs2,fs1) ) return S2_S1;
		}

		/* default rule last */
		if( mt1->u_conds & C_DEFAULT ) return S2_S1;
		if( mt2->u_conds & C_DEFAULT ) return S1_S2;

		/* rule with src=xxx first */
		if( mt1->u_conds > mt2->u_conds ) return S1_S2;
		if( mt1->u_conds < mt2->u_conds ) return S2_S1;

		/* reverse entry order for overwriting */
		if( mt1->u_serno < mt2->u_serno )
			return S2_S1;
		else	return S1_S2;
	}else{	
		if( strstr(s1,s2) && r2) return S1_S2;
		if( strstr(s2,s1) && r1) return S2_S1;

		/* entry order */
		if( mt1->u_serno < mt2->u_serno )
			return S1_S2;
		else	return S2_S1;
	}
}
static sort_mtab()
{
	Bsort(mtab,mtabN,sizeof(Mtab*),mtabcomp);
}
static dump_mtab()
{	int mi;
	Mtab *mt;

	for( mi = 0; mi < mtabN; mi++ ){
		mt = mtab[mi];
		InitLog("MOUNT[%d]%s[%d] %s%s%s %s%s%s %s\n",
			mi,
			mi != mt->u_serno-1 ? "X":"=",
			mt->u_serno-1,
			mt->Src.u_src,mt->Src.u_remain?"*":"",
			mt->Src.u_format?mt->Src.u_format:"",
			mt->Dst.u_src,mt->Dst.u_remain?"*":"",
			mt->Dst.u_format?mt->Dst.u_format:"",
			mt->u_opts);
	}
}
init_mtab(){
	int mi;
	char *ssrc,*dsrc;
	char proto[64],hostport[256],host[256],port[256],path[1024],prefix[64];
	int plen;
	char *addr,*dpath;
	char login[256],*dp,userpass[256],user[64],pass[64];
	char loginpath[1024];
	int iport;
	Mtab *mt;
	char *opts;
	char dproto[256];
	char *upath;
	char *query;

	if( mtabN <= init_from )
		return;
	mi = init_from;
	init_from = mtabN;

	if( do_MTAB_SORT )
		sort_mtab();
	dump_mtab();

	for(mi = 0; mi < mtabN; mi++ ){
		mt = mtab[mi];
		if( mt->u_compiled )
			continue;
		mt->u_compiled = 1;

		ssrc = mt->Src.u_src;
		if( streq(ssrc,ASIS) )
			mt->Src.u_asis = 1;
		else
		if( isFullURL(ssrc) ){
			char norm_url[1024],hostport[1024];
			decomp_absurl(ssrc,proto,login,path,sizeof(path));
			mt->Src.u_proto = StrAlloc(proto);
			if( login[0] ){
			iport = scan_hostport(proto,login,host);
			sprintf(hostport,"%s:%d",host,iport);
			mt->Src.u_hostList = makePathList("MOUNT.vhost",
				StrAlloc(hostport));
			mt->Src.u_path = StrAlloc(path);
			mt->Src.u_plen = strlen(path);
			if( IsResolvable(host) ){
				sethostcache(host,1);
				sethostcache_predef(host);
			}
			InitLog("## MOUNT FULL-URL-SRC [%s]://[%s:%d]/[%s]\n",
				proto,host,iport,path);
			/* normalize URL string ... */
			}
		}

		proto[0] = path[0] = 0;
		dpath = path;
		dsrc = mt->Dst.u_src;

		dproto[0] = 0;
		scan_URI_scheme(dsrc,dproto,sizeof(dproto));

		if( mt->u_opts )
			opts = mt->u_opts;
		else	opts = "";

		if( streq(dsrc,ASIS) ){
			mt->Dst.u_asis = 1;
			continue;
		}else
		if( dsrc[0] == 0 && mt->Dst.u_remain ){
			mt->Dst.u_any = 1;
			continue;
		}else
		if( plen = isURN(dsrc) ){
			strcpy(proto,dproto);
			login[0] = 0;
			strcpy(path,dsrc+strlen(proto)+1);
			dpath = path;
		}else
		if( mt->Dst.u_remain && streq(dsrc+strlen(dproto),"://") ){
			mt->Dst.u_path2site = 1;
			strcpy(proto,dproto);
			login[0] = 0;
			dpath = "";
		}else
		if( localPathProto(dproto) ){
			if( upath = file_hostpath(dsrc,proto,login) ){
				if( login[0] == 0 )
					strcpy(login,"localhost");
				if( upath[0] == '/' )
					strcpy(path,upath+1);
				else
				if( !isFullpath(upath) ){
					char xupath[1024];
					char *query;
					if( query = strchr(upath,'?') )
						*query = 0;
					if( fullpathDATA(upath,"r",xupath) ){
						strcpy(path,xupath);
					}else{
					getcwd(path,sizeof(path));
					chdir_cwd(path,upath,0);
					strcat(path,"/");
					}
					if( query ){
						*query = '?';
						strcat(path,query);
					}
					sv1log("MOUNT relative file:%s -> %s\n",
						upath,path);
					if( *path == '/' )
						ovstrcpy(path,path+1);
				}
				else	strcpy(path,upath);
				dpath = path;
			}
		}else
		if( sscanf(dsrc,"%[^:]://%s",proto,loginpath) == 2 ){
			if( loginpath[0] == '/' || loginpath[0] == '?' ){
				mt->Dst.u_emptysite = 1;
				login[0] == 0;
				strcpy(path,loginpath+1);
			}else{
				sscanf(loginpath,"%[^/?]%s",login,path);
				if( path[0] == '/' )
					dpath = path+1;
				else	dpath = path;
			}
		}else{
			sv1log("ERROR MOUNT=\"%s %s\", right hand must be full-URL\n",
				mt->Src.u_src,mt->Dst.u_src);
			mt->u_disabled = 1;
			continue;
		}

		decomp_URL_siteX(login,userpass,user,pass,hostport,host,port);
		mt->Dst.u_userpass = StrAlloc(userpass);
		mt->Dst.u_user = StrAlloc(user);
		mt->Dst.u_pass = StrAlloc(pass);

		url_rmprefix(proto,prefix);
		mt->Dst.u_prefix = StrAlloc(prefix);

		if( login[0] ){
			iport = scan_hostport(proto,hostport,host);
			if( addr = gethostaddr(host) )
				mt->Dst.u_hosta = StrAlloc(addr);
			else	addr = "";
			Verbose("MOUNT HOST %s=%s\n",host,addr);
			/* multiple address should be cared ... */
		}else{
			host[0] = 0;
			iport = 0;
			addr = "";
		}

		mt->Dst.u_hostList = makePathList("MOUNT.rhost",
			StrAlloc(hostport));

		mt->Dst.u_proto = StrAlloc(proto);
		mt->Dst.u_hostn = StrAlloc(host);
		mt->Dst.u_stdport = serviceport(proto);
		mt->Dst.u_iport = iport;

		if( streq(proto,ASIS) ) mt->Dst.u_proto_ASIS = 1;
		if( streq(host,ASIS) ) mt->Dst.u_host_ASIS = 1;
		if( streq(port,ASIS) ) mt->Dst.u_port_ASIS = 1;

		if( dp = strchr(hostport,':') ){
			mt->u_flags |= U_DISPPORT;
			if( dp[1] == '+' || dp[1] == '-' ){
				mt->u_flags |= U_PORTMAP;
				mt->Dst.u_portmap = StrAlloc(dp+1);
			}
		}

		if( query = strchr(dpath,'?') ){
			*query++ = 0;
			mt->Dst.u_query = StrAlloc(query);
			mt->Dst.u_qlen = strlen(query);
		}
		mt->Dst.u_path = StrAlloc(dpath);
		mt->Dst.u_plen = strlen(dpath);

		if( login[0] )
			Verbose("[%d] MOUNT=%s %s%s://%s[%s]:%d%s %s\n",
				mi,mt->Src.u_src,prefix,proto,
				host,addr,iport,path,opts);
		else	Verbose("[%d] MOUNT=%s %s%s:%s %s\n",
				mi,mt->Src.u_src,prefix,proto,path,opts);


		if( IsResolvable(host) ){
			sethostcache(host,1);
			sethostcache_predef(host);
			/* preload cache from all of RESOLV=dns,nis,file ... */
		}
	}
}
static char *rewrite_path(ctx,out,fmt,in)
	void *ctx;
	char *out,*fmt,*in;
{	char *op,*fp,fc;
	char urlbuf[4096],*uv[256];
	int un,u1,u2,ui,tailis;
	char *ox;
	int omax;

	op = out;
	omax = strlen(in) + 64;
	if( omax < 256 )
		omax = 256;
	ox = out + omax;
	lineScan(in,urlbuf);
	tailis = strtailchr(urlbuf) == '/';
	un = stoV(urlbuf,256,uv,'/');

	for( fp = fmt; fc = *fp; fp++ ){
		if( ox <= op )
			break;

		if( fc != '%' ){
			*op++ = fc;
			continue;
		}
		fc = *++fp;
		if( fc == 0 )
			break;

		if( '0' <= fc && fc <= '9' ){
			u1 = fc - '0';
			u2 = u1;
			if( fp[1] == '-' ){
				fp++;
				u2 = un;
			}
			ui = u1;
			if( ui < un ) do {
				if( u1 < ui )
					*op++ = '/';
				strcpy(op,uv[ui]);
				op += strlen(op);
			} while ( ++ui < u2 );
			if( u2 == un && tailis )
				*op++ = '/';
		}
	}
	*op = 0;
	return op;
}

/*
static dstNoMatch(mt,clif,srcurl)
*/
static dstNoMatch(ctx,mt,clif,srcurl)
	void *ctx;
	Mtab *mt;
	char *clif,*srcurl;
{	char site[256],proto[256],host[256],port[256];
	int porti;
	int got;

	/* except for MOUNTs which extract destination server information
	 * from vURL dynamically, the destination is in rURL statically.
	 */
	if( !mt->Dst.u_path2site && *srcurl == '/' ){
		if( mt->Dst.u_hostn )
		if( mt->Dst.u_hostn[0] )
		if( matchPath1(mt->dstList,"-",mt->Dst.u_hostn,mt->Dst.u_iport))
			return 0;

		/* match dst=host with virtual host name in "Host:host" */
		if( clif != NULL ){
			wordScan(CTX_clif_proto(ctx),proto);
			decomp_URL_site(clif,host,port);
			got = 1;
			goto DOMATCH;
		}
		return 1;
	}

	if( mt->Dst.u_path2site ){
		got = 0;
		scan_URI_site(srcurl+strlen(mt->Src.u_src),site,sizeof(site));
		if( *site ){
			got = 1;
			strcpy(proto,mt->Dst.u_proto);
			decomp_URL_site(site,host,port);
		}
	}else{
		got = scan_protositeport(srcurl,proto,host,port);
	}

DOMATCH:
	if( got ){
		if( *port == 0 )
			porti = serviceport(proto);
		else	porti = atoi(port);
		if( !matchPath1(mt->dstList,"-",host,porti) )
			return 1;
	}
	return 0;
}
static evalCond1(ctx,direction,mt,clif,method,srcurl,rhost,rport)
	void *ctx;
	Mtab *mt;
	char *clif;
	char *method;
	char *srcurl;
	char *rhost;
{	char host[256];
	int port;

	if( mt->fromList && !matchFROM(ctx,mt->fromList,NULL) )
		return 0;

	if( direction & mt->D_vhostList )
	if( mt->vhostList && clif != NULL ){
		if( clif[0] != '-' )
			return 0;
		port = scan_Hostport1(clif,host);
		if( !matchPath1(mt->vhostList,"-",host,port) )
			return 0;
	}

	if( direction & mt->D_clifList )
	if( mt->clifList && clif != NULL ){
		port = scan_Hostport1(clif,host);
		if( !matchPath1(mt->clifList,"-",host,port) )
			return 0;
	}

	if( mt->dstList && srcurl != NULL ){
		if( (mt->u_conds & C_QHOST) == 0 )
			return 0;
		/*
		if( dstNoMatch(mt,clif,srcurl) )
		*/
		if( dstNoMatch(ctx,mt,clif,srcurl) )
			return 0;
	}

	if( mt->dstList && rhost != NULL ){
		int nomatch,co;
		if( (mt->u_conds & C_RHOST) == 0 )
			return 0;
		co = RES_CACHEONLY(1);
		nomatch = !matchPath1(mt->dstList,"-",rhost,rport);
		RES_CACHEONLY(co);
		if( nomatch )
			return 0;
	}

	if( direction & mt->u_D_method )
	if( mt->u_method && method != NULL ){
		/*sv1log("### MOUNT METHOD [%s][%s]\n",mt->u_method,method);*/
		if( strcmp(mt->u_method,"*") != 0 )
		if( strcasestr(mt->u_method,method) == NULL )
			return 0;
	}

	if( mt->pathList && !CTX_matchPATHs(ctx,mt->pathList,NULL) )
		return 0;

	if( mt->viaList && !CTX_matchPATH1(ctx,mt->viaList,NULL,"-") )
		return 0;

	if( mt->u_xcond && evalCondition(ctx,mt->u_xcond) == 0 )
		return 0;

	return 1;
}
static evalCond(ctx,direction,mt,clif,dstproto,method,srcurl,rhost,rport)
	void *ctx;
	Mtab *mt;
	char *clif;
	char *dstproto;
	char *method;
	char *srcurl;
	char *rhost;
{	int value;
	char param[1024],*pp;

	if( mt->u_dstproto && dstproto ){
		if( !wordIsinList(mt->u_dstproto,dstproto) )
			return 0;
	}

	param[0] = 0;
	pp = param;
	*pp++ = '0' + direction; *pp++ = '/';
	/*
	if( clif && (mt->clifList||mt->vhostList) ) pp = Strcpy(pp,clif);
	*/
	if( clif && (mt->clifList||mt->vhostList||mt->dstList) )
		pp = Strcpy(pp,clif);
	*pp++ = '/';
	if( rhost  && mt->dstList  ) pp = Strcpy(pp,rhost); *pp++ = '/';
	if( method && mt->u_method ) pp = Strcpy(pp,method); *pp++ = '/';
	*pp = 0;
	if( srcurl && mt->dstList  ) Strncpy(pp,srcurl,pp-param);

	if( mt->u_reqpat ){
		char *req;
		if( req = CTX_reqstr(ctx) ){
			if( frex_match(mt->u_reqpat,req) == 0 )
				return 0;
		}
	}

	if( mt->u_conds_value )
	if( strcmp(param,mt->u_conds_param) == 0 )
		return 0 < mt->u_conds_value ? 1 : 0;
		/* should check if input parameters (clif,method,srcurl)
		 * is identical with the previous evaluation...
		 */

	value = evalCond1(ctx,direction,mt,clif,method,srcurl,rhost,rport);
	if( clif ){
		if( value )
			mt->u_conds_value = 1;
		else	mt->u_conds_value = -1;
		Strdup(&mt->u_conds_param,param);
	}
	return value;
}
reset_MOUNTconds()
{	int mi;

	for( mi = 0; mi < mtabN; mi++ ){
		mtab[mi]->u_conds_value = 0;
		mtab[mi]->Dst.u_iport_mapped = 0;
	}
	last_forw = -1;
}
static portmap(ctx,mt)
	void *ctx;
	Mtab *mt;
{	int mport;
	Url *dst;

	dst = &mt->Dst;
	if( mport = dst->u_iport_mapped )
		return mport;
	mport = portMap(ctx,dst->u_proto,dst->u_hostn,dst->u_portmap);
	dst->u_iport_mapped = mport;
	return mport;
}

CTX_evalMountCond(ctx,opts,user,chost,cport,ihost,iport)
	void *ctx;
	char *opts,*user,*chost,*ihost;
{	int List;
	int match = 0;
	char path[1024];

	sprintf(path,"%s:%d!%s:%d",ihost,iport,chost,cport);

	if( strncmp(opts,"from=",5) == 0 ){
		List = makePathList("SERVER.from",opts+5);
		match = matchPath1(List,user,chost,cport);
	}else
	if( strncmp(opts,"host=",5) == 0 ){
		List = makePathList("SERVER.clif",opts+5);
		match = matchPath1(List,user,ihost,iport);
	}else
	if( strncmp(opts,"path=",5) == 0 ){
		List = makePathList("SERVER.path",opts+5);
		match = CTX_matchPATHs(ctx,List,path);
	}else{
		if( strncmp(opts,"via=",4)==0 || strncmp(opts,"src=",4)==0 )
			opts += 4;
		List = makePathList("SERVER.via",opts);
		match = CTX_matchPATH1(ctx,List,path,user);
	}
	return match;
}
static matchHost(up,host)
	Url *up;
	char *host;
{	char *ahostn,*ahosta,*hosta;

	ahostn = up->u_hostn;
	ahosta = up->u_hosta;
	if( strcaseeq(host,ahostn) || ahosta && streq(host,ahosta) ) 
		return 1;

	if( ahosta )
	if( hosta = gethostaddr_fromcache(host) )
		if( streq(hosta,ahosta) )
			return 1;

	/* when the host have several addresses ...
	 * `matchhostaddr_incache(host,ahosta)' is necessary ?
	 */
	return 0;
}
static matchURL(up,url)
	Url *up;
	char *url;
{	char proto[128],login[256],host[256],upath[1024];
	int iport;
	int hlen,plen;

	hlen = plen = 0;
	decomp_absurl(url,proto,login,upath,sizeof(upath));
	iport = scan_hostport(proto,login,host);

	if( streq(proto,up->u_proto) )
	if( matchPath1(up->u_hostList,"-",host,iport) )
	{
		hlen = strlen(url) - strlen(upath);
		plen = url_strstr(upath,up->u_path);
		Verbose("## MATCH FULL-URL-SRC [%s][%s] %d+%d\n",
			up->u_src,url,hlen,plen);
	}
	return hlen + plen;
}
static xmatch(mt,patn,url,lenp)
	Mtab *mt;
	char *patn,*url;
	int *lenp;
{
	*lenp = 0;
	if( *patn == 0 || (*lenp = url_strstr(url,patn)) ){
		if( mt->Src.u_remain || url[*lenp] == 0 )
			return 1;
	}
	return 0;
}
mount_lastforw(){ return last_forw; }
static char *mount_url_toX(ctx,clif,method,url,qtype,rmt)
	void *ctx;
	char *clif;
	char *method,*url;
	Mtab **rmt;
{	char *up,origin_iurl[URLSZ],*rp,src_rest[1024];
	char *iurl,nocase_iurl[URLSZ];
	char *iupath;
	int isfull;
	char *ourl;
	int match;
	int ai;
	char *addr;
	int len;
	Mtab *mt;
	int delput;
	char *dstproto,protob[128],*dp;
	UTag *uv[33],ub[32];
	int uc;
	char *myhostport;
	char *dproto;

	if( rmt ) *rmt = 0;

	my_h[0] = 0;
	if( clif == NULL )
		clif = CTX_clif_hostport(ctx);
	myhostport = clif;
	if( *myhostport == '-' )
		myhostport++;

	iurl = origin_iurl;
	rp = wordscanY(url,origin_iurl,sizeof(origin_iurl),"^ \t\r\n");
	Strncpy(src_rest,rp,sizeof(src_rest));

	isfull = isFullURL(iurl);
	/* must care full-path MOUNT ...
	if( isfull ){
		iupath = ...
	}else
	 */
	iupath = iurl;
	if( urlpath_normalize(iupath,iupath) )
		Verbose("MOUNT url-path normalized [%s]>[%s]\n",url,iurl);

	nocase_iurl[0] = 0;

	for( ai = 0; ai < mtabN; ai++ ){
		mt = mtab[ai];
		if( mt->Src.u_asis || mt->u_disabled )
			continue;
		if( mt->u_direction == D_BACKONLY )
			continue;

		if( mt->u_conds & C_NOCASE ){
			if( nocase_iurl[0] == 0 )
				strtolower(iurl,nocase_iurl);
			iurl = nocase_iurl;
		}else	iurl = origin_iurl;

		match = xmatch(mt,mt->Src.u_src,iurl,&len);
		if( !match && mt->Src.u_proto && isfull ){
			int hlen;
			if( hlen = matchURL(&mt->Src,iurl) )
			if( match = xmatch(mt,mt->Src.u_path,iurl+hlen,&len) )
				len += hlen;
		}

		if( !match )
			continue;

		if( mt->u_flags & U_REQDUMP ){
/*
CTX_dump_request(ctx);
			char *req;
			FILE *lfp;
			if( req = CTX_reqstr(ctx) ){
				if( lfp = fopen("/tmp/reqlog","a") ){
					fputs(req,lfp);
					fclose(lfp);
				}
			}
*/
		}

		if( mt->Dst.u_any ){
			ourl = iurl + strlen(mt->Src.u_src);
			scan_URI_scheme(ourl,protob,sizeof(protob));
			dstproto = protob;
		}else{
			ourl = iurl;
			dstproto = NULL;
		}
		if( evalCond(ctx,CFO,mt,clif,dstproto,method,ourl,NULL,0)==0 )
			continue;

		if( mt->Src.u_format )
		{
			if( mt->Src.u_uvfmt ){
				char *rsp,*rfp;
				uvinit(uv,ub,32);
				uc = uvfromsfX(&iurl[len],0,mt->Src.u_format,uv,&rsp,&rfp);
				if( uc < 1 || *rsp != 0 )
					continue;
/*
				if( *rfp != 0 && !streq(rfp,"$") && strtailchr(rfp)=='$' )
*/
				if( mt->Src.u_format_toEOL && *rfp != 0 )
					continue; /* not complete match */
			}else
			if( !RexpMatch(&iurl[len],mt->Src.u_format) )
				continue;
		}

		if( mt->u_conds & C_WITHQUERY )
		if( strchr(iurl,'?') == NULL )
			continue;

		if( mt->u_conds & C_ASPROXY )
		if( !isfull )
			continue;

		/*
		 * input URL matched with the URL pattern of this MOUNT point
		 */

		if( (qtype & U_MOVED_TO ) && (mt->u_flags & U_MOVED_TO ) == 0
		 || (qtype & U_USE_PROXY) && (mt->u_flags & U_USE_PROXY) == 0
		){
			Verbose("[%d] URL Matched but not for MovedTo[%x]:%s\n",
				ai,qtype,iurl);
			/* must exit without rewriting */
			goto FOUND;
		}

		if( mt->Dst.u_asis ){
			Verbose("[%d] MOUNT ASIS: %s\n",ai,mt->Src.u_src);
			if( mt->u_conds & C_DEFAULT ){
				/* treat "ASIS by default" as Not-Mounted */
			}else
			if( mt->u_opts && mt->u_opts[0] )
				goto FOUND;
			break;
		}

		if( mt->Dst.u_any ){
			up = Strcpy(url,iurl+len);
			up = Strcpy(up,src_rest);
			goto FOUND;
			break;
		}

		if((mt->u_flags & U_RESOLV) && mt->Dst.u_hosta)
			addr = mt->Dst.u_hosta;
		else	addr = mt->Dst.u_hostn;

		up = Strcpy(url,mt->Dst.u_prefix);
/*
		if( mt->Dst.u_proto_ASIS )
			up = Strcpy(up,CTX_clif_proto(ctx));
		else
		up = Strcpy(up,mt->Dst.u_proto);
*/
		if( mt->Dst.u_proto_ASIS )
			dproto = CTX_clif_proto(ctx);
		else	dproto = mt->Dst.u_proto;
		up = Strcpy(up,dproto);
		*up++ = ':';

		if( addr[0] ){ 
			up = Strcpy(up,"//");
			if( *mt->Dst.u_userpass ){
				up = Strcpy(up,mt->Dst.u_userpass);
				up = Strcpy(up,"@");
			}
			if( addr[0] == '-' && addr[1] == 0 ){
				my_p = scan_hostport(dproto,myhostport,my_h);
				up = Strcpy(up,myhostport);
			}else{
				if( mt->Dst.u_host_ASIS ){
					up = Strcpy(up,CTX_clif_host(ctx));
				}else
				up = Strcpy(up,addr);
				if( mt->u_flags & U_DISPPORT ){
					int mport;
					if( mt->Dst.u_port_ASIS ){
						mport = CTX_clif_port(ctx);
					}else
					if( mt->u_flags & U_PORTMAP )
						mport = portmap(ctx,mt);
					else	mport = mt->Dst.u_iport;
					sprintf(up,":%d",mport);
					up += strlen(up);
				}
			}
			delput = 0;
		}else
		if( mt->Dst.u_emptysite ){
			up = Strcpy(up,"//");
			delput = 0;
		}else
		if( mt->Dst.u_path2site ){
			up = Strcpy(up,"//");
			delput = 1;
		}else{
			delput = 1;
		}

		if( mt->Dst.u_path ){
			if( !delput ){
				up = Strcpy(up,"/");
				delput = 1;
			}
			up = Strcpy(up,mt->Dst.u_path);
			delput = 1;
		}
		if( mt->Dst.u_query ){
			*up++ = '?';
			up = Strcpy(up,mt->Dst.u_query);
			delput = 1;
		}

		if( iurl[len] )
		if( mt->Dst.u_remain ){
			if( !delput ){
				up = Strcpy(up,"/");
				delput = 1;
			}
			if( mt->Src.u_uvfmt && mt->Dst.u_uvfmt ){
				uvtosf(up,URLSZ-(up-url),mt->Dst.u_format,uv);
				up += strlen(up);
			}else
			if( mt->Dst.u_format )
				up = rewrite_path(ctx,up,mt->Dst.u_format,iurl+len);
			else	up = Strcpy(up,iurl+len);
		}

		Verbose("*** %s MOUNTED TO[%d] %s ***\n",
			mt->Src.u_src,ai,mt->Dst.u_src);
		sv1log("*** %s => %s ***\n",iurl,url);

		if( src_rest[0] )/* not a part of the URL */
			up = Strcpy(up,src_rest);
		last_hit = ai+1;
		goto FOUND;
	}
	return 0;

FOUND:
	/*
	 * MOVED_TO and USE_PROXY are not matching conditions about URL
	 * but the interpretations of the rule when the URL is matched.
	 * So they are here.
	 */
	if( qtype & U_MOVED_TO ){
		if((mt->u_flags & U_MOVED_TO) == 0)
		{
			/* matched non_MOVED MOUNT */
			if( rmt ) *rmt = mt;
			return 0;
		}
	}else
	if( qtype & U_USE_PROXY ){
		if((mt->u_flags & U_USE_PROXY) == 0)
			return 0;
	}else{
		if( mt->u_flags & (U_MOVED_TO|U_USE_PROXY) )
			return 0;
	}
	last_forw = ai;
	if( rmt ) *rmt = mt;
	return mt->u_opts;
}
char *CTX_mount_url_to(ctx,myhostport,method,url)
	void *ctx;
	char *myhostport,*method,*url;
{
	return mount_url_toX(ctx,myhostport,method,url,U_MOUNT,NULL);
}
int non_MOVED(){ return 100; }
CTX_moved_url_to(ctx,myhostport,method,url)
	void *ctx;
	char *myhostport,*method,*url;
{	Mtab *mt;

	if( mount_url_toX(ctx,myhostport,method,url,U_MOVED_TO,&mt) ){
		return mt->u_moved;
	}
	if( mt != 0 ){
		/* matched non_MOVED MOUNT */
		return non_MOVED();
	}
	return 0;
}
char *CTX_changeproxy_url_to(ctx,myhostport,method,url,proxy)
	void *ctx;
	char *myhostport,*method,*url,*proxy;
{	Mtab *mt;
	char *opt;

	*proxy = 0;
	if( opt = mount_url_toX(ctx,myhostport,method,url,U_USE_PROXY,&mt) )
	if( mt->u_useproxy )
		strcpy(proxy,mt->u_useproxy);
	return opt;
}

static dstmatch(ctx,mt,proto,host,iport,path,plen)
	void *ctx;
	Mtab *mt;
	char *proto,*host,*path;
{	char *aproto,*ahost;
	int mport;

	if( mt->Dst.u_proto_ASIS )
		aproto = CTX_clif_proto(ctx);
	else	aproto = mt->Dst.u_proto;
	if( !streq(proto,aproto) )
		return 0;

	if( mt->u_flags & U_PORTMAP )
		mport = portmap(ctx,mt);
	else	mport = mt->Dst.u_iport;
	if( mport != 0 && mport != iport )
		return 0;

	ahost = mt->Dst.u_hostn;
	if( strcmp(ahost,"-") == 0 && iport == my_p && strcmp(host,my_h) == 0 ){
		Verbose("** UNMOUNT[//-] %s:%d==%s:%d\n",host,iport,my_h,my_p);
	}else
	if( ahost[0] != 0 && !matchHost(&mt->Dst,host) )
		return 0;

	if( plen != 0 && strncmp(path,mt->Dst.u_path,plen) != 0 )
		return 0;

	return 1;
}

char *CTX_mount_url_fromL(ctx,url,proto,hostport,path,search,dproto,delegate)
	void *ctx;
	char *url,*proto,*hostport,*path,*search,*dproto,*delegate;
{	char *up,*dp1,*dp2,dhost[128];
	int ai;
	int plen;
	Mtab *mt;
	char host[128];
	char xpath[1024];
	int iport;
	int match;
	int mport;
	char mybase[128];

	debug("{M} %s://%s/%s ?%s\n",proto,hostport,path,search?search:"");

	up = url;

	if( proto == 0 || hostport == 0 )
		return 0;
	if( path == 0 )
		path = "";

	if( dproto == 0 || dproto[0] == 0 )
		dproto = "http";

	if( hostport[0] == 0 ){
		host[0] = 0;
		iport = 0;
	}else	iport = scan_hostport(proto,hostport,host);

	for( ai = 0; ai < mtabN; ai++ ){
		mt = mtab[ai];
		if( mt->Dst.u_asis || mt->u_disabled )
			continue;

		if( mt->u_flags & (U_MOVED_TO|U_USE_PROXY) )
			continue;
		if( mt->u_direction == D_FORWONLY )
			continue;
		if( mt->u_direction == D_BACKIFFORW && last_forw != ai )
			continue;

		plen = mt->Dst.u_plen;

		match = 0;
		if( mt->Dst.u_any )
			match = 1;
		else	match = dstmatch(ctx,mt,proto,host,iport,path,plen);

		if( !match )
			continue;

		if( mt->Dst.u_qlen != 0 ){
			if( search == 0 )
				continue;
			if( strncmp(search,mt->Dst.u_query,mt->Dst.u_qlen) != 0 )
				continue;
		 	if( !mt->Dst.u_remain && search[mt->Dst.u_qlen] != 0 )
				continue;
		}

		if( evalCond(ctx,CBO,mt,delegate,proto,NULL,NULL,host,iport)==0 )
			continue;

		if( mt->u_flags & U_MYBASE ){
			delegate = mt->u_mybase;
			/* should set proto also if u_mybase is in
			 * "proto://server" format ..
			 */
		}else
		if( mt->u_conds & C_VHOST ){
			delegate = topofPath1(mt->vhostList,delegate,mybase);
		}

		if( !mt->Dst.u_remain && path[plen] != 0 )
			continue;

		if( mt->Src.u_asis )
			break;

		Verbose("** %s UNMOUNTED FROM %s **\n",
			mt->Src.u_src,mt->Dst.u_src);

		if( mt->Dst.u_rformat ){
			UTag *uv[33],ub[32];
			char *rp,*rsp,*rfp;
			int uc,mlen;
			uvinit(uv,ub,32);
			uc = uvfromsfX(path,0,mt->Dst.u_rformat,uv,&rsp,&rfp);
			if( 0 < uc && *rfp == 0 ){
				mlen = rsp - path;
				if( strchr(" \t\r\n",path[mlen]) ){
					uvtosf(xpath,URLSZ,mt->Src.u_rformat,uv);
					strcat(xpath,&path[mlen]);
					Verbose("** %s -> %s\n",path,xpath);
					path = xpath;
				}
			}
		}

		if( dp1 = strchr(delegate,':') ){
			if( atoi(dp1+1) == serviceport(dproto) ){
				int hl;
				wordscanY(delegate,dhost,sizeof(dhost),"^:");
				hl = strlen(dhost);
				if( dp2 = strchr(dp1,'/') )
					wordscanX(dp2,dhost+hl,sizeof(dhost)-hl);
				delegate = dhost;
			}
		}

		if( !mt->Src.u_fullurl ){
			sprintf(up,"%s://%s",dproto,delegate);
			up += strlen(up);
		}
		up = Strcpy(up,mt->Src.u_src);
		if( mt->Dst.u_proto == 0 ){
			sprintf(up,"%s://%s/",proto,hostport);
			up += strlen(up);
		}
		if( mt->Dst.u_path2site ){
			sprintf(up,"%s/",hostport);
			up += strlen(up);
		}

		if( streq(proto,"gopher") ){
			if( up[-1] == '/' )
				up = Strcpy(up,"1");
			else	up = Strcpy(up,"/1");
		}

		if( plen )
			path += plen;
		up = Strcpy(up,path);

		if( search ){
			if( mt->Dst.u_qlen ){
				up = Strcpy(up,search+mt->Dst.u_qlen);
			}else{
			*up++ = '?';
			up = Strcpy(up,search);
			}
		}
		last_hit = ai+1;
		return mt->u_opts;
	}
	return 0;
}
CTX_scan_mtab(ctx,func,arg)
	void *ctx;
	int (*func)();
	char *arg;
{	Mtab *mt;
	int mi;
	char *clif;

	clif = CTX_clif_hostport(ctx);
	for( mi = 0; mi < mtabN; mi++ ){
		mt = mtab[mi];

		if( evalCond(ctx,CFB,mt,clif,NULL,NULL,NULL,NULL,0) == 0 )
			continue;

if( mt->Dst.u_hostn )
/* scanned but not compiled, caused by duplicat set of MOUNT list X-< */
		(*func)(arg,
			mt->Src.u_src,
			mt->Dst.u_proto,
			mt->Dst.u_user,
			mt->Dst.u_pass,
			mt->Dst.u_hostn,
			mt->Dst.u_iport,
			mt->Dst.u_path,
			mt->u_opts);
	}
}

trans_url(nurl,ourl,fmt)
	char *nurl,*ourl,*fmt;
{
	strcpy(nurl,fmt);
	sv1log("Transform-URL: %s -> %s\n",ourl,nurl);
}
