/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1994-1999 Yutaka Sato
Copyright (c) 1994-1999 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, and distribute this material for any purpose
and without fee is hereby granted, provided that the above copyright
notice and this permission notice appear in all copies.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	iotimeout.c
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:
History:
	940802	created

timeout could be the function of I/O buffer size...

//////////////////////////////////////////////////////////////////////#*/

#include <stdio.h>
#include "vsignal.h"
#include "ysocket.h"
#include "fpoll.h"
#include <errno.h>

#define sv1log	syslog_ERROR

#define CHECK_ITVL	(10*1000)

extern int (*CheckClosed)();
checkCloseOnTimeout(checktime){
	if( CheckClosed )
		return (*CheckClosed)(checktime);
	else	return 0;
}

xPollIn(fd,timeout)
{	int nready,ntry;
	int remain,timeout1;
	int closed;

	if( readyAlways(fd) )
		return 1;

	nready = -1;
	remain = timeout;
	closed = 0;

	for( ntry = 0; timeout == 0 && ntry < 1 || 0 < remain; ntry++ ){
		if( closed = checkCloseOnTimeout(1) )
			timeout1 = remain;
		else
		if( CHECK_ITVL < remain )
			timeout1 = CHECK_ITVL;
		else	timeout1 = remain;
		errno = 0;
		nready = PollIn(fd,timeout1);
		if( 0 < nready )
			break;

/*
this was introduced as a trial to reset possible POLL_PRI status
at DeleGate/5.7.5 (17Nov98), but seemed useless,
and was got rid of at DeleGate/5.9.1 because it is harmful; at least
it cause blocking on slow I/O on Win32.
		if( nready == 0 ){
			char buf[1];
			if( RecvOOB(fd,buf,0) == 0 )
			sv1log("#### POLL_PRI ? PollIn()==0, RecvOOB()==0\n");
		}
*/

		if( nready < 0 ){
			if( errno == EINTR )
				sv1log("#### INTERRUPTED PollIn()=%d\n",nready);
			else{
				sv1log("#### xPollIn(%d,%d)=%d errno=%d\n",
					fd,timeout,nready,errno);
				break;
			}
		}else	remain -= timeout1;
	}
	if( closed == 0 )
		checkCloseOnTimeout(1);
	if( 100 < ntry )
		sv1log("#### xPollIn(%d,%d)*%d : %d\n",fd,timeout,ntry,nready);
	return nready;
}

/*
 * I/O timeout in seconds
 */
int IO_TIMEOUT = (10*60);

static jmp_buf ioenv;
static int io_fd;
static int io_timer;
static int io_sig;
static void (*io_pipe)();

static void io_timeout(sig)
{
	io_sig = sig;
	longjmp(ioenv,sig);
}
#define	RETURN_ONTIMEOUTX(fd,retv,timeout) \
	checkCloseOnTimeout(1); \
	io_fd = fd; \
	io_pipe = Vsignal(SIGPIPE,io_timeout); \
	io_timer = pushTimer("Timered-IO",io_timeout,timeout); \
	if( setjmp(ioenv) != 0 ){ \
		Vsignal(SIGPIPE,io_pipe); \
		popTimer(io_timer); \
		checkCloseOnTimeout(1); \
		sv1log("IO_TIMEOUT(%d) SIG%s\n",timeout,sigsym(io_sig)); \
		return retv; \
	}

#define	RETURN_ONTIMEOUT(fd,retv) RETURN_ONTIMEOUTX(fd,retv,IO_TIMEOUT)

#define DONE_SUCCESSFULLY() \
	Vsignal(SIGPIPE,io_pipe); \
	popTimer(io_timer); \
	checkCloseOnTimeout(1);

writevTimeout(fd,iov,nv,timeout)
	char *iov;
{	int wcc;

	RETURN_ONTIMEOUTX(fd,-1,timeout);
	wcc = writev(fd,iov,nv);
	DONE_SUCCESSFULLY();
	return wcc;
}
sendTimeout(fd,buff,leng,flag,timeout)
	char *buff;
{	int wcc;

	RETURN_ONTIMEOUTX(fd,-1,timeout);
	wcc = send(fd,buff,leng,flag);
	DONE_SUCCESSFULLY();
	return wcc;
}

fwriteTIMEOUT(b,s,n,fp)
	char *b;
	FILE *fp;
{	int rc;

	RETURN_ONTIMEOUT(fileno(fp),0);
	rc = Fwrite(b,s,n,fp);
	DONE_SUCCESSFULLY();
	return rc;
}
fputsTIMEOUT(b,fp)
	char *b;
	FILE *fp;
{	int rcode;

	RETURN_ONTIMEOUT(fileno(fp),0);
	rcode = fputs(b,fp);
	DONE_SUCCESSFULLY();
	return rcode;
}
extern int LIN_TIMEOUT;
fflushTIMEOUT(fp)
	FILE *fp;
{	int rcode;

	if( feof(fp) )
		return EOF;

	RETURN_ONTIMEOUTX(fileno(fp),EOF,LIN_TIMEOUT+1);
	rcode = fflush(fp);
	DONE_SUCCESSFULLY();
	return rcode;
}
fcloseTIMEOUT(fp)
	FILE *fp;
{	int rcode;

	if( feof(fp) )
		return EOF;

	RETURN_ONTIMEOUTX(fileno(fp),EOF,LIN_TIMEOUT+2);
	rcode = fclose(fp);
	DONE_SUCCESSFULLY();
	return rcode;
}

static int readSERNO;
readTimeoutBlocked(fd,buf,len,timeout)
	char *buf;
{	int omask,nmask;
	int rcc;
	int serno;

	rcc = -1;
	omask = sigblock(sigmask(SIGCHLD));
	serno = ++readSERNO;
	if( 0 < PollIn(fd,timeout) ){
		rcc = read(fd,buf,len);
		if( rcc != len ){
		sv1log("##ERROR: readTimeoutB insufficient read %d/%d (%d)%X\n",
			rcc,len,errno,sigblock(0));
		}
	}
	if( serno != readSERNO ){
		sv1log("##ERROR: readTimeoutB broken %d/%d (%d)%X\n",
			serno,readSERNO,errno,sigblock(0));
		sleep(10);
	}
	nmask = sigsetmask(omask);
	return rcc;
}
readTimeout(fd,b,s,tout)
	char *b;
{	int nready,rc;

	if( 0 < (nready = xPollIn(fd,tout*1000)) )
		return read(fd,b,s);
	else{
		sv1log("read(%d) -- ready=%d IO_TIMEOUT(%d)\n",s,nready,tout);
		return 0;
	}
}
readTIMEOUT(fd,b,s)
	char *b;
{
	return readTimeout(fd,b,s,IO_TIMEOUT);
}
recvTIMEOUT(fd,b,s,f)
	char *b;
{	int rc;

	if( 0 < xPollIn(fd,IO_TIMEOUT*1000) )
		rc = recv(fd,b,s,f);
	else	rc = 0;
	return rc;
}
static recvPeekTIMEOUT1(fd,b,s)
	char *b;
{	int rc;

	RETURN_ONTIMEOUTX(fd,-1,10);
	rc = RecvPeek(fd,b,s);
	DONE_SUCCESSFULLY();
	return rc;
}
recvPeekTIMEOUT(fd,b,s)
	char *b;
{	int rc;

	if( 0 < xPollIn(fd,IO_TIMEOUT*1000) ){
		rc = recvPeekTIMEOUT1(fd,b,s);
		if( rc <= 0 )
			sv1log("#### recvPeek: failed: %d\n",rc);
	}else{
		sv1log("recvPeekTIMEOUT: %d\n",IO_TIMEOUT);
		rc = 0;
	}
	return rc;
}
freadTIMEOUT(b,s,n,fp)
	char *b;
	FILE *fp;
{	int rn;

	if( feof(fp) ){
		sv1log("-- Tried freadTIMEOUT() for EOF file.\n");
		return 0;
	}
	if( 0 < READYCC(fp) ){
		if( s == 1 ){
			rn = fgetBuffered(b,n,fp);
			if( rn == n )
				return rn;
			return rn + freadTIMEOUT(b+rn,s,n-rn,fp);
		}
	}
	if( 0 < READYCC(fp) || 0 < xPollIn(fileno(fp),IO_TIMEOUT*1000) )
		return fread(b,s,n,fp);
	else{
		sv1log("fread(%d*%d) -- IO_TIMEOUT(%d)\n",s,n,IO_TIMEOUT);
		return 0;
	}
}
discardBuffered(fp)
	FILE *fp;
{
	while( READYCC(fp) ){
		getc(fp);
	}
}
fgetsBuffered(b,n,fp)
	char *b;
	FILE *fp;
{	int mcc,cc,ch;

	n--;
	for( cc = 0; cc < n; cc++ ){
		if( READYCC(fp) <= 0 )
			break;
		b[cc] = ch = getc(fp);
		if( ch == 0 )
			break;
		if( ch == EOF )
			break;
		if( ch == '\n' ){
			cc++;
			break;
		}
	}
	b[cc] = 0;
	return cc;
}
fgetBuffered(b,n,fp)
	char *b;
	FILE *fp;
{	int mcc,cc,ch;

	for( cc = 0; cc < n; cc++ ){
		if( READYCC(fp) <= 0 )
			break;
		b[cc] = ch = getc(fp);
		if( ch == EOF )
			break;
	}
	if( cc < n )
		b[cc] = 0;
	return cc;
}
file_copyBuffered(in,out)
	FILE *in,*out;
{	int cc,ch;

	for( cc = 0; 0 < READYCC(in) && (ch = getc(in)) != EOF; cc++ ){
		if( putc(ch,out) == EOF )
			break;
	}
	return cc;
}

char *fgets0(b,n,fp)
	char *b;
	FILE *fp;
{	int ch,li;

	for( li = 0; li < n - 1; li++ ){
		b[li] = ch = getc(fp);
		if( ch == EOF || ch == '\n' )
			break;
		if( ch == 0 ){
			ungetc(ch,fp);
			break;
		}
	}
	b[li] = 0;
	if( li == 0 )
		return NULL;
	else	return b;
}

static char *fgetsIfReady(b,s,fp,rc)
	char *b;
	FILE *fp;
	int *rc;
{	char *bp;
	int bx,ch;

	bx = 0;
	while( bx < s-1 && 0 < READYCC(fp) ){
		ch = getc(fp);
		if( ch == EOF )
			break;
		b[bx++] = ch;
		if( ch == '\n' )
			break;
	}
	b[bx] = 0;
	if( rc ) *rc = bx;

	if( bx == 0 && feof(fp) )
		return NULL;
	else	return b;
}

char *fgetsTIMEOUT(b,s,fp)
	char *b;
	FILE *fp;
{	char *b0,*rs;
	int rc;

	b[0] = 0;
	if( feof(fp) ){
		sv1log("-- Tried fgetsTIMEOUT() for EOF file.\n");
		return NULL;
	}

	b0 = b;
	rc = 0;
	if( 0 < READYCC(fp) ){
		rs = fgetsIfReady(b,s,fp,&rc);
		if( 0 < rc && b[rc-1] == '\n' || feof(fp) || rc == s-1 )
			return rs;
		b += rc;
		s -= rc;
	}
	if( 0 < xPollIn(fileno(fp),IO_TIMEOUT*1000) )
		rs = fgets(b,s,fp);
	else{
		sv1log("fgets(%d) -- IO_TIMEOUT(%d)\n",s,IO_TIMEOUT);
		rs = NULL;
	}
	if( rs == NULL && rc == 0 )
		return NULL;
	else	return b0;
}
char *fgetsTimeout(b,s,fp,tout)
	char *b;
	FILE *fp;
{	int stout;
	char *rs;

	if( tout ){
		stout = IO_TIMEOUT;
		IO_TIMEOUT = tout;
		rs = fgetsTIMEOUT(b,s,fp);
		IO_TIMEOUT = stout;
	}else	rs = fgets(b,s,fp);
	return rs;
}

fgetcTIMEOUT(fp)
	FILE *fp;
{
	if( 0 < fPollIn(fp,IO_TIMEOUT*1000) )
		return getc(fp);
	else	return EOF;
}

char *fgetsByBlock(line,size,fs,niced,ltimeout,byline,fromcache,remlen,lengp,isbinp)
	char *line;
	FILE *fs;
	int *lengp,*isbinp;
{	int cc,ch,bc;
	int timeout;
	int stimeout;
	char *rcode = line;
	int insize;

	if( !fromcache ){
		stimeout = 30;
		timeout = ltimeout;
	}
	if( niced )
		insize = size / 2;
	else	insize = 1024;

	size--;
	bc = 0;

	for( cc = 0; cc < size; ){
		if( !fromcache && READYCC(fs)<1 && fPollIn(fs,timeout)<=0 ){
			if( cc == 0 ){
				syslog_ERROR("fgetsBB: TIMEOUT %dms\n",timeout);
				rcode = NULL;
			}
			break;
		}
		if( (ch = getc(fs)) == EOF ){
			if( cc == 0 )
				rcode = NULL;
			break;
		}

		line[cc++] = ch;
		if( ch == '\n' && (byline || insize < cc) )
			break;
		if( ch == 0 )
			bc++;

		if( !byline ){
			if( ch == '\n' || ch == '\r' )
				timeout = stimeout;
			else
			if( remlen <= cc ){
				if( timeout != stimeout )
				syslog_DEBUG("fgetsBB: %d/%d TIMEOUT=%d->%d\n",
					cc,remlen,timeout,stimeout);
				timeout = stimeout;
			}
			else	timeout = ltimeout;
		}
	}

	line[cc] = 0;
	*lengp = cc;
	*isbinp = bc;
	return rcode;
}

char *fgetsByLine(line,lsiz,in,timeout,rccp,isbinp)
	char *line;
	FILE *in;
	int *rccp,*isbinp;
{
	if( fPollIn(in,0) <= 0 )
		return NULL;
	return fgetsByBlock(line,lsiz,in,0,timeout,1,0,lsiz,rccp,isbinp);
}

extern double Time();
int MAX_NICE_VALUE = 10;
int MAX_NICE_SLEEP = 10;
int MIN_NICE_SLEEP =  1;
int MIN_SOCK_BUFSIZE = 1024;
int MAX_SOCK_BUFSIZE = 1024*64;
/*
 * try to do I/O every MAX_NICE_SLEEP seconds
 * to reduce the cost for process switch
 * without decreasing the speed of relay
 */
doNice(what,ifd,ifp,ofd,ofp,niced,bytes,count,since)
	char *what;
	FILE *ifp,*ofp;
	double since;
{	double now,elapsed;
	double sec1,sec;
	int bsize,isize,osize;

	now = Time();
	elapsed = now - since;

	if( 10 <= (now - since) )
	if( niced < MAX_NICE_VALUE && (0x010000 << niced) < bytes ){
		niced += 1;
		nice(1);

		bsize = ((int)((bytes/elapsed) * MAX_NICE_SLEEP*2)) & ~0x3FF;
		if( bsize == 0 )
			bsize = MIN_SOCK_BUFSIZE;
		if( MAX_SOCK_BUFSIZE < bsize )
			bsize = MAX_SOCK_BUFSIZE;

		if( 0 <= ifd ) expsockbuf(ifd,bsize,0);
		if( 0 <= ofd ) expsockbuf(ofd,0,bsize);

		isize = osize = 0;
		if( 0 <= ifd )
			getsockbuf(ifd,&isize,&osize);
		daemonlog("E","NICE-%d %dK/%d/%4.2fs %d/p %d/s buf=%d(%d)\n",
			niced,bytes/1024,count,elapsed,bytes/count,
			(int)(bytes/elapsed),isize,bsize);
	}
	if( 0 < niced )
	if( ifp == NULL || READYCC(ifp) <= 0 )
	if( 0 <= ifd && !readyAlways(ifd) && PollIn(ifd,1) == 0 )
	if( getsockbuf(ifd,&isize,&osize) == 0 ){
		sec = sec1 = (isize/4) / (bytes/elapsed);
		if( MIN_NICE_SLEEP < sec1 ){
			if( MAX_NICE_SLEEP < sec ) sec = MAX_NICE_SLEEP;
		daemonlog("E","NICE-%d %dK/%d/%4.2fs %d/p %d/s: %4.2fs/%4.2fs\n",
				niced,bytes/1024,count,elapsed,bytes/count,
				(int)(bytes/elapsed),sec,sec1);

			if( ofp != NULL )
				fflush(ofp);
			msleep((int)(sec*1000));
		}
	}
	return niced;
}
