/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1994-1999 Yutaka Sato
Copyright (c) 1994-1999 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, and distribute this material for any purpose
and without fee is hereby granted, provided that the above copyright
notice and this permission notice appear in all copies.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	cache.c (cache create, open and lock)
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:
	Cache file in a directory hierarchy.
	On creation of a cache file, if the name is already used by a
	directory, the cache file name is extended with "/=". 
	On creation of a directory, if the name is already used by a
	cache file, the cache file name is extended with "/=".
History:
	Mar1994	created
//////////////////////////////////////////////////////////////////////#*/
#define MKDIR_MODE	0755
#define FORCE_MODE	0

#include <errno.h>
#include <stdio.h>
#include "log.h"
#include <ctype.h>
#include "ystring.h"
#include <sys/types.h>
int CACHE_READONLY;
extern double Time();
extern FILE *fopenShared();

int MaxSymLink = 8;
static int NumSymLink;

mkdirRX(dir)
	char *dir;
{
	NumSymLink = 0;
	return mkdirR(dir,MKDIR_MODE);
}
linkRX(to,from)
	char *to,*from;
{
	NumSymLink = 0;
	return linkR(to,from,MKDIR_MODE);
}
renameRX(old,new)
	char *old,*new;
{
	NumSymLink = 0;
	return renameR(old,new,MKDIR_MODE);
}
Readlink(dir,xdir,xsiz)
	char *dir,*xdir;
{	char *dp,xtmp[1024];
	int scc;

	xdir[0] = 0;
	scc = readlink(dir,xdir,xsiz);
	if( scc < 0 )
		return scc;
	if( xsiz <= scc )
		return -1;
	if( xdir[0] == 0 )
		return -1;
	if( 8 < ++NumSymLink ){
		fprintf(stderr,"DeleGate: too many symbolic links: %s\n",dir);
		return -1;
	}
	xdir[scc] = 0;
	if( isFullpath(xdir) ){
		if( strcmp(dir,xdir) == 0 )
			return -1; /* symbolic link to self */
		else	return scc;
	}

	strcpy(xtmp,xdir);
	strcpy(xdir,dir);
	if( dp = strrpbrk(xdir,"/\\") )
		dp[1] = 0;
	chdir_cwd(xdir,xtmp,1);

	if( strcmp(dir,xdir) == 0 ){
		/* symbolic link to self */
		return -1;
	}
	return scc;
}

#define DIR_ESCAPE	"@"
#define DIR_DESCRIBER	"="

mkdir1(dir,mode)
	char *dir;
{	char xpath[1024],ypath[1024];
	int rcode;
	char xdir[1024];
	int scc;

	if( 0 < (scc = Readlink(dir,xdir,sizeof(xdir))) ){
		dir = xdir;
	}
	if( mkdirShared(dir,mode) == 0 ){
		if( FORCE_MODE ){ if( mode != -1 ) chmod(dir,mode); }
		return 0;
	}
	if( fileIsflat(dir) ){
		sprintf(xpath,"%s.%s",dir,DIR_ESCAPE);
		sprintf(ypath,"%s/%s",dir,DIR_ESCAPE);
		rename(dir,xpath);
		mkdirShared(dir,mode);
		if( FORCE_MODE ){ if( mode != -1 ) chmod(dir,mode); }
		if( rename(xpath,ypath) == 0 ){
			sv1log("mkdir1: ESCAPED `%s' to `%s'\n",dir,ypath);
			return 0;
		}
	}
	if( 0 < scc )
		return mkdirR(dir,mode);

	return -1;
}
static mkdirR(dir,mode)
	char *dir;
{	char *dp,*tp;
	int isdir;

	if( fileIsdir(dir) )
		return 0;

	dp = dir;
	for( tp = &dir[strlen(dir)-1]; dir < tp; tp-- ){
		if( *tp != '/' )
			continue;
		*tp = 0;
		isdir = fileIsdir(dir);
		*tp = '/';
		if( isdir ){
			dp = tp + 1;
			break;
		}
	}

	while( dp = strchr(dp,'/') ){
		*dp = 0;
		mkdir1(dir,mode);
		*dp++ = '/';
	}
	return mkdir1(dir,mode);
}
static linkR(to,from,mode)
	char *to,*from;
{	char dir[1024],*dp;

	if( link(to,from) == 0 )
		return 0;

	strcpy(dir,from);
	if( dp = strrchr(dir,'/') ){
		*dp = 0;
		mkdirR(dir,mode);
		return link(to,from);
	}
	return -1;
}
static renameR(old,new,mode)
	char *old,*new;
{	char dir[1024],*dp,path[1024];

	strcpy(path,new);
	path_escchar(path);
	new = path;

	if( File_size(old) < 0 )
		return -1;

	if( rename(old,new) == 0 )
		return 0;

	if( unlink(new) == 0 )
		if( rename(old,new) == 0 )
			return 0;

	strcpy(dir,new);
	if( dp = strrchr(dir,'/') ){
		*dp = 0;
		mkdirR(dir,mode);
		if( rename(old,new) == 0 )
			return 0;
	}
	return -1;
}

static cachelog(fp,path,mode)
	FILE *fp;
	char *path,*mode;
{	char dir[2048],*dp;
	int pino,ino;
	int now,ctime,mtime,atime;

/*
	strcpy(dir,path);
	if( dp = strrchr(dir,'/') ){
		*dp = 0;
		pino = File_ino(dir);
	}else	pino = 0;
	ino = file_ino(fileno(fp));
	file_times(fileno(fp),&ctime,&mtime,&atime);
	now = time(0);

fprintf(stderr,"#### %6d %6d [%8d %8d %8d] [%-2s] %s\n",
pino,ino,
now-ctime,now-mtime,now-atime,
mode,path);
*/
}

FILE *dirfopen(what,file,mode)
	char *what,*file,*mode;
{	char *dp;
	char xpath[2048];
	FILE *fp;

	if( file[0] == 0 )
		return NULL;

	path_escchar(file);

	if( file[strlen(file)-1] == '/' )
		strcat(file,DIR_DESCRIBER);

	if( dp = strrchr(file,'/') ){
		*dp = 0;
		mkdirRX(file);
		*dp = '/';
	}

	if( fileIsdir(file) ){
/*
BUG?941027
sprintf(xpath,"%s/%s",file,DIR_DESCRIBER);
*/
sprintf(xpath,"%s/%s",file,DIR_ESCAPE);
		sv1log("dirfopen: ESCAPED `%s' to `%s'\n",file,xpath);
		strcpy(file,xpath);
	}
	fp = fopenShared(file,mode);
	Verbose("dirfopen(%s,%s): %x [%d]\n",file,mode,fp,fp?fileno(fp):-1);

	if( fp != NULL )
		cachelog(fp,file,mode);
	return fp;
}

/*
 * check the date and expire
 */
static expfopen_log(what,Start,file,atime,fmt,a,b)
	char *what;
	double Start;
	char *file;
	char *fmt;
{	double Now,Lap;
	int iatime; /* seconds since the last access */
	char log[2048];

	Now = Time();
	Lap = Now - Start;
	if( atime == 0 )
		iatime = -1;
	else	iatime = Now - atime;
	sprintf(log,fmt,a,b);
	daemonlog("E","[%3.2f,%d][%s %s] %s\n",Lap,iatime,what,log,file);
}
FILE *expfopen(what,expire,file,mode,datep)
	char *what,*file,*mode;
	int *datep;
{	FILE *fp;
	int stat,size,mtime,atime,etime;
	char xpath[2048];
	double Start;

	Start = Time();
	if( file[0] == 0 )
		return NULL;

	path_escchar(file);

	if( file[strlen(file)-1] == '/' )
		strcat(file,DIR_DESCRIBER);
	else
	if( fileIsdir(file) ){
		sprintf(xpath,"%s/%s",file,DIR_ESCAPE);
		strcpy(file,xpath);
	}
	Verbose("expfopen: %s\n",file);

	if( (fp = fopen(file,mode)) == NULL ){
		expfopen_log(what,Start,file,0,"cache-NONE");
		return NULL;
	}

	stat = file_stat(fileno(fp),&size,&mtime,&atime);
	if( stat != 0 || size <= 0 ){
		expfopen_log(what,Start,file,atime,"cache-EMPTY: %d",size);
		fclose(fp);
		fp = NULL;
	}else{
		if( datep )
			*datep = mtime;
		etime = time(0) - mtime;
		if( expire < etime ){
			expfopen_log(what,Start,file,atime,
				"cache-EXPIRED: %d > %d",etime,expire);
			fclose(fp);
			fp = NULL;
		}else{
			expfopen_log(what,Start,file,atime,
				"cache-VALID: %d < %d",etime,expire);
		}
	}
	if( fp != NULL )
		cachelog(fp,file,mode);
	return fp;
}

extern char *cachedir(),*cachefmt();
CTX_cache_path(ctx,proto,server,iport,path1,cachepath)
	void *ctx;
	char *proto,*server,*path1,*cachepath;
{	char *rp;
	char pathb[4096],xpath[4096];
	int plen;
	char sc,lproto[256],aserver[256],lserver[256];
	int si;

	*cachepath = 0;
	if( cachedir() == 0 )
		return 0;

	for( si = 0; sc = proto[si]; si++ )
		lproto[si] = isupper(sc) ? tolower(sc) : sc;
	lproto[si] = 0;

	if( VSA_strisaddr(server) ){
		gethostbyAddr(server,aserver);
		sv1log("CACHE hostname: %s -> %s\n",server,aserver);
		server = aserver;
	}
	for( si = 0; sc = server[si]; si++ )
		lserver[si] = isupper(sc) ? tolower(sc) : sc;
	lserver[si] = 0;

	if( strcmp(proto,lproto) != 0 || strcmp(server,lserver) != 0 )
		sv1log("CACHE: case changed %s://%s -> %s://%s\n",
			proto,server,lproto,lserver);

	if( rp = strpbrk(path1,"\r\n\t ") ){
		plen = rp - path1;
		strncpy(pathb,path1,plen);
		pathb[plen] = 0;
		path1 = pathb;
	}
	xpath[0] = 0;
	chdir_cwd(xpath, path1, 0);
	if( strtailchr(path1) == '/' && strtailchr(xpath) != '/' ) 
		strcat(xpath,"/");
	evalPATHexp(cachepath,cachefmt(),lproto,lserver,iport,xpath);
	return 1;
}
cache_path(proto,server,iport,path1,cachepath)
	char *proto,*server,*path1,*cachepath;
{
	return CTX_cache_path(NULL,proto,server,iport,path1,cachepath);
}

FILE *cache_fopen_rd(what,cpath,expire,datep)
	char *what,*cpath;
	int *datep;
{	FILE *cfp;
	char *cache_dir;

	if( cachedir() == 0 )
		return NULL;

	cfp = expfopen(what,expire,cpath,"r",datep);
	if( cfp != NULL ){
		if( file_isdir(fileno(cfp)) ){
			fclose(cfp);
			cfp = NULL;
		}
	}
	return cfp;
}
FILE *cache_fopen_wr(what,cpath)
	char *what,*cpath;
{	FILE *fp;

	if( CACHE_READONLY )
		return NULL;

	if( cachedir() == 0 )
		return NULL;

	fp = dirfopen(what,cpath,"w");
	return fp;
}
FILE *cache_fopen_rw(what,cpath)
	char *what,*cpath;
{	FILE *fp;

	if( cachedir() == 0 )
		return NULL;

	if( (fp = dirfopen(what,cpath,"r+")) == NULL )
		fp = dirfopen(what,cpath,"w+");

	return fp;
}

cache_expire(sexp,dflt)
	char *sexp;
{	int clock;

	clock = scan_period(sexp,'d',dflt);
	Verbose("EXPIRE %s = %d\n",sexp,clock);
	return clock;
}

cache_delete(cpath)
	char *cpath;
{
	unlink(cpath);
}

CTX_cache_remove(ctx,proto,host,port,path)
	void *ctx;
	char *proto,*host,*path;
{	char base[1024],cpath[1024],*dp;
	int len;

	if( cachedir() == 0 )
		return -1;

	CTX_cache_path(ctx,proto,host,port,"",base);
	len = strlen(base);
	if( base[len-1] == '/' )
		base[len-1] = 0;

	if( strstr(path,base) == NULL )
		return -1;

	if( unlink(path) != 0 )
		return -1;

	strcpy(cpath,path);
	while( dp = strrchr(cpath,'/') ){
		*dp = 0;
		if( strstr(cpath,base) == NULL )
			break;

		if( rmdir(cpath) != 0 )
			break;
	}
	return 0;
}


stripPATHexp(path,spath)
	char *path,*spath;
{	char *dp,*sp,ch;
	int in;

	dp = spath;
	in = 1;
	for( sp = path; ch = *sp; sp++ ){
		if( ch == '$' && sp[1] == '[' ){
			sp++;
			in = 0;
		}else
		if( ch == ']' && in == 0 )
			in = 1;
		else
		if( in )
			*dp++ = ch;
	}
	*dp = 0;
}


/*
 *	$[client:...]
 *		%A    user-agent
 */
static evalClient(xpath,fmt,host,user)
	char *xpath,*fmt,*host,*user;
{
	return 0;
}
static evalQuery(xpath,fmt,host,user)
	char *xpath,*fmt,*host,*user;
{
	return 0;
}

static char *getdomN(name,lev)
	char *name;
{	char *dp;
	int lv;

	if( name[0] == 0 || name[1] == 0 )
		return name;

	dp = &name[strlen(name)-1];
	lv = 0;
	for(; name < dp; dp-- ){
		if( *dp == '.' )
			if( ++lv == lev )
				return &dp[1];
	}
	return name;
}
static char *getdomN1(dst,src,lev)
	char *dst,*src;
{	char *dp,*sp,buf[512];
	int lx;

	if( VSA_strisaddr(src) ){
		for( lx = 1, sp = src; lx < lev && *sp; sp++ )
			if( *sp == '.' )
			if( ++lx == lev )
				break;
	}else{
		for( lx = 0, sp = &src[strlen(src)-1]; lx < lev && src < sp; sp-- )
			if( *sp == '.' )
			if( ++lx == lev )
				break;
	}
	if( *sp == '.' )
		sp++;

	dp = dst;
	while( *sp && *sp != '.' )
		*dp++ = *sp++;
	*dp = 0;
}

static layered_name(dst,src,addhost)
	char *dst,*src;
{	char *dp,*sp,buf[512];

	if( VSA_strisaddr(src) ){
		strcpy(dst,src);
	}else{
		strcpy(buf,src);
		sp = &buf[strlen(src)-1];
		dp = dst;
		for(; buf <= sp; sp-- ){
			if( *sp == '.' ){
				*sp = 0;
				strcpy(dp,sp+1);
				dp += strlen(dp);
				*dp++ = '.';
				*dp = 0;
			}
			if( sp == buf && addhost )
				strcpy(dp,sp);
		}
	}
	for( dp = dst; *dp; dp++ )
		if( *dp == '.' )
			*dp = '/';
}

int FQDN_HSIZE = 32;
static evalHash(xpath,fmt,proto,server,port,path)
	char *xpath,*fmt;
	char *proto,*server,*path;
{	char pathbuf[256];

	evalServer(pathbuf,fmt,proto,server,port,path);
	sprintf(xpath,"%02x",((unsigned int )FQDN_hash(pathbuf)) % FQDN_HSIZE);
	return 1;
}

static evalServer(xpath,fmt,proto,server,port,path)
	char *xpath,*fmt,*proto,*server,*path;
{	char *p1,*fp,*xp;
	char tmp[256];
	char fqdn[256];
	int neval = 0;
	int tomd5;

	xpath[0] = 0;
	xp = xpath;
	for( fp = fmt; *fp; fp++ )
	if( *fp == '%' && fp[1] ){
		fp++;
		if( tomd5 = (fp[0] == 'm') && (fp[1] != 0) )
			fp++;
		switch( *fp ){
		    default:  return -1;
		    case '%': strcpy(xp,"%"); break;
		    case 'Q': getFQDN(server,fqdn); server = fqdn; break;
		    case 'P': strcpy(xp,proto); break;
		    case 'H': strcpy(xp,server); break;
		    case 'd': layered_name(xp,server,0); break;
		    case 'h': layered_name(xp,server,1); break;
		    case 'T': sprintf(xp,"%d",port); break;
		    case 'L':
			if( port != serviceport(proto) )
				sprintf(xp,"%s:%d",server,port);
			else	strcpy(xp,server);
			break;
		    case '1':
		    case '2':
		    case '3':
		    case '4':
		    case '5':
/*
			if( p1 = strrchr(server,'.') ){
				if( isdigits(p1+1) ) 
					strcpy(xp,"no-name");
				else	strcpy(xp,getdomN(server,atoi(fp)));
			}else	strcpy(xp,"no-domain");
*/
			getdomN1(xp,server,atoi(fp));
			break;
		    case 'p':
			strcpy(xp,path);
			break;
		}
		neval++;
		if( tomd5 ){
			toMD5(xp,tmp);
			strcpy(xp,tmp);
		}
		xp += strlen(xp);
	}else{
		*xp++ = *fp;
		*xp = 0;
	}
	return neval;
}
static struct {
	char	*f_begin;
	char	*f_end;
	int	(*f_func)();
} tab_formats[] = {
	{"$[server:",	"]",	evalServer},
	{"$[hash:",     "]",    evalHash},
	{"$[query:",	"]",	evalQuery},
	{"$[cleint:",	"]",	evalClient},
	0
};
#define formats	tab_formats

evalPATHexp(xpath,pathfmt,proto,server,port,path)
	char *xpath,*pathfmt,*proto,*server,*path;
{	char *dp,*fp,*tp;
	int fi;
	char fmtv[2][1024],*fmts,*fmtd,*fmtt;
	char fmtb[1024];
	int cfmt,nfmt,dlen,flen;
	char *bsym,*esym;
	int (*ffunc)();

	strcpy(fmtv[0],pathfmt);
	fmts = fmtv[0];
	fmtd = fmtv[1];

	Verbose("CACHE: `%s'\n",fmts);
	for(;;){
		ffunc = NULL;
		for( fi = 0; bsym = formats[fi].f_begin; fi++ ){
			esym = formats[fi].f_end;
			ffunc = formats[fi].f_func;
			if( (dp = strstr(fmts,bsym)) && (tp = strstr(dp,esym)) )
				break;
		}
		if( bsym == 0 )
			break;

		dlen = dp - fmts;
		strncpy(fmtd,fmts,dlen);
		fp = dp +strlen(bsym);
		flen = tp - fp;
		strncpy(fmtb,fp,flen);
		fmtb[flen] = 0;
		if( (*ffunc)(&fmtd[dlen],fmtb,proto,server,port,path) <= 0 )
			break;
		strcat(fmtd,tp+1);
		fmtt = fmts;
		fmts = fmtd;
		fmtd = fmtt;
		Verbose("CACHE: `%s'\n",fmts);
	}
	strcpy(xpath,fmts);
}

extern FILE *TMPFILE();
static FILE *sharedfp;
FILE *new_shared()
{
	return sharedfp = TMPFILE("new_shared");
}
get_shared(buf,size,fp)
	char *buf;
	FILE *fp;
{	int rc;

	if( fp == NULL )
		fp = sharedfp;
	if( fp == NULL ){
		sv1log("get_shared: not opened.\n");
		return -1;
	}
	fseek(fp,0,0);
	lock_shared(fileno(fp));
	rc = fread(buf,1,size-1,fp);
	lock_unlock(fileno(fp));
	buf[rc] = 0;
	return rc;
}
put_shared(buf,size,fp)
	char *buf;
	FILE *fp;
{	int wc;

	if( fp == NULL )
		fp = sharedfp;
	if( fp == NULL ){
		sv1log("put_shared: not opened.\n");
		return -1;
	}
	lock_exclusive(fileno(fp));
	wc = fwrite(buf,1,size,fp);
	fflush(fp);
	lock_unlock(fileno(fp));
	return wc;
}
get_equiv_user(clhost,clport,eqhost,equser)
	char *clhost;
	char *eqhost,*equser;
{	char buff[2048],*bp,*np;
	char equserhost[1024],chost1[256];

	if( get_shared(buff,sizeof(buff),NULL) <= 0 )
		return 0;

	for( bp = buff; *bp; ){
		if( np = strstr(bp,"\r\n") )
			*np = 0;

		sscanf(bp,"%s %s\n",equserhost,chost1);
		if( strcasecmp(chost1,clhost) == 0 ){
			if( sscanf(equserhost,"%[^@]@%s",equser,eqhost) == 2 ){
				fprintf(stderr,"EQUIVE %s -> %s@%s\n",clhost,
					equser,eqhost);
				return 1;
			}
		}
		if( np == NULL )
			break;
		bp = np + 2;
	}
	return 0;
}


#define CREATING	"#CREATING"
FILE *cache_make(what,cpath,xcpath)
	char *what,*cpath,*xcpath;
{	FILE *cachefp;

/*
to invalidate cache ???
it is halmful for whom is reading it now, and for
whom has different EXPIRE period.

	cachefp = cache_fopen_wr(what,cpath);
	if( cachefp == NULL ){
		sv1log("CACHE: can't create (%d) = %s\n",errno,cpath);
		return NULL;
	}
	if( rename(cpath,xcpath) == 0 ){
		sv1log("CACHE: created = %s\n",xcpath);
	}else{
		sv1log("CACHE: can't rename (%d) = %s\n",errno,xcpath);
		fclose(cachefp);
		cachefp = NULL;
	}
*/

	sprintf(xcpath,"%s%s",cpath,CREATING);
	cachefp = cache_fopen_wr(what,xcpath);
	if( cachefp != NULL )
		sv1log("CACHE: created %s\n",xcpath);
	else	sv1log("CACHE: can't create (%d) = %s\n",errno,xcpath);
	return cachefp;
}

cache_done(gotok,cachefp,cpath,xcpath)
	FILE *cachefp;
	char *cpath,*xcpath;
{	int size;

	if( cachefp == NULL ){
		sv1log("CACHE: no cache opened.\n");
		return;
	}
	size = ftell(cachefp);
	fclose(cachefp);

	if( gotok ){
		if( renameRX(xcpath,cpath) == 0 )
			sv1log("CACHE: got = [%d] %s\n",size,cpath);
		else	sv1log("CACHE: can't link %s => %s\n",cpath,xcpath);
	}else{
		sv1log("CACHE: err = [%d] %s\n",size,xcpath);
		unlink(xcpath);
	}
}
