/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1992 Electrotechnical Laboratry (ETL)

Permission to use, copy, modify, and distribute this material
for any purpose and without fee is hereby granted, provided
that the above copyright notice and this permission notice
appear in all copies, and that the name of ETL not be
used in advertising or publicity pertaining to this
material without the specific, prior written permission
of an authorized representative of ETL.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY
OF THIS MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS",
WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type: program/C; charset=US-ASCII
Program:      str_stdio.h
Author:       Yutaka Sato <ysato@etl.go.jp>
Description:

     This program redirects the file I/O from/to strings on memory.
     Include "str_stdio.h" file after <stdio.h>

History:
	92.05.18   created
//////////////////////////////////////////////////////////////////////#*/
#include <stdio.h>
typedef unsigned char Uchar;

#define Str_MAGIC 0x12345678
typedef struct {
	int	sh_magic;
	char	sh_id[16];
	char    sh_mode[16];
	Uchar  *sh_base;
	int	sh_peak;
	int	sh_maxsize;		/* limit of auto expansion */
	int	sh_size;
} StrHead;
typedef struct {
	FILE	s_FILE;
	StrHead	s_STRING;
} String;
#define s_magic	s_STRING.sh_magic
#define s_mode	s_STRING.sh_mode
#define s_base	s_STRING.sh_base
#define s_peak	s_STRING.sh_peak
#define s_maxsize	s_STRING.sh_maxsize
#define s_size	s_STRING.sh_size

#define str_isSTR(Str)	(Str->s_magic == Str_MAGIC)

str_isStr(Str)
	register String *Str;
{
	return str_isSTR(Str);
}
str_sopen(StrH,id,buf,size,peak,mode)
	StrHead *StrH;
	char *id;
	unsigned char *buf;
	char *mode;
{
	StrH->sh_magic = Str_MAGIC;
	if( id ){
		raw_Strncpy(StrH->sh_id,id,sizeof(StrH->sh_id));
	}else{
		sprintf(StrH->sh_id,"%X",buf);
	}
	strcpy(StrH->sh_mode,mode);
	StrH->sh_base = buf;
	StrH->sh_size = size;
	StrH->sh_peak = peak;
}
String *
str_fopen(buf,size,mode)
	unsigned char *buf;
	char *mode;
{	String *Str;

	Str = (String*)calloc(1,sizeof(String));
	str_sopen(&Str->s_STRING,NULL,buf,size,0,mode);
	return Str;
}
str_fclose(Str)
	String *Str;
{
	if( !str_isSTR(Str) )
		return fclose((FILE*)Str);

	str_fflush(Str);
	free(Str);
	return 0;
}

str_fgetc(Str)
	register String *Str;
{
	if( !str_isSTR(Str) )
		return fgetc((FILE*)Str);

	if( Str->s_size <= Str->s_peak )
		return EOF;

	return Str->s_base[Str->s_peak++];
}
str_feof(Str)
	String *Str;
{
	if( !str_isSTR(Str) )
		return feof((&Str->s_FILE));
	return Str->s_size <= Str->s_peak;
}
str_fungetc(ch,Str)
	String *Str;
{	char pch;

	/*if( ch == EOF )
		return EOF;*/

	if( !str_isSTR(Str) )
		return ungetc(ch,(FILE*)Str);

	if( Str->s_peak <= 0)
		return EOF;

	pch = Str->s_base[--Str->s_peak];
	if( pch != ch )
		Str->s_base[Str->s_peak] = ch;
	return ch;
}
char *
str_fgets(buf,size,Str)
	char *buf;
	String *Str;
{	int rsize,nlx;
	Uchar *top,*nlp;

	if( !str_isSTR(Str) )
		return fgets(buf,size,(FILE*)Str);

	rsize = Str->s_size - Str->s_peak;
	if( rsize <= 0 )
		return NULL;
	if( rsize < size )
		size = rsize;

	top = &Str->s_base[Str->s_peak];
	for(nlx = 0; nlx < rsize; nlx++)
		if( top[nlx] == '\n' ){
			size = nlx+1;
			break;
		}
	strncpy(buf,top,size);
	Str->s_peak += size;
	buf[size] = 0;
	return buf;
}

str_fputc(ch,Str)
	String *Str;
{
	if( !str_isSTR(Str) )
		return fputc(ch,(FILE*)Str);

	return str_sputc(ch,&Str->s_STRING);
}
str_sputc(ch,StrH)
	StrHead *StrH;
{
	if( StrH->sh_size <= StrH->sh_peak )
		return EOF;

	StrH->sh_base[StrH->sh_peak++] = ch;
	return ch;
}
str_fputs(buf,Str)
	char *buf;
	String *Str;
{	int size,rsize;

	if( !str_isSTR(Str) )
		return fputs(buf,(FILE*)Str);

	rsize = Str->s_size - Str->s_peak;
	if( rsize <= 0 )
		return EOF;

	size = strlen(buf);
	if( size == 0 )
		return;
	if( rsize < size )
		size = rsize;

	strncpy(&Str->s_base[Str->s_peak],buf,size);
	Str->s_peak += size;

	return 0;
}

str_sflush(StrH)
	StrHead *StrH;
{
	if( strpbrk(StrH->sh_mode,"+wa") )
		StrH->sh_base[StrH->sh_peak] = 0;
	return 0;
}
str_fflush(Str)
	String *Str;
{
	if( !str_isSTR(Str) )
		return fflush((FILE*)Str);
	return str_sflush(&Str->s_STRING);
}
str_fprintf(Str,form,a,b,c,d,e,f)
	String *Str;
	char *form,*a,*b,*c,*d,*e,*f;
{
	if( !str_isSTR(Str) )
		return fprintf((FILE*)Str,form,a,b,c,d,e,f);
	return str_sprintf(&Str->s_STRING,form,a,b,c,d,e,f);
}

static int fmtDBG;
static int fmtOK;
static int fmtNO;

static fmtdebug(StrH,who,why,fmt,xpeak)
	StrHead *StrH;
	char *who,*why,*fmt;
{	char fc,*fp;

	fprintf(stderr,"##%s:%s:%s:%d/%d:%d/%d:",
		who,why,StrH->sh_id,
		StrH->sh_peak,StrH->sh_size,fmtNO,fmtOK);
	for( fp = fmt; fc = *fp; fp++ ){
		switch( fc ){
			case '\r': fputs("^M",stderr); break;
			case '\n': fputs("^J",stderr); break;
			default: putc(fc,stderr);
		}
	}
	fputs("##\r\n",stderr);
}

Str_sprintf(StrH,fmt,a0,a1,a2,a3,a4,a5,a6,a7,a8,a9,aA,aB,aC,aD,aE,aF)
	StrHead *StrH;
	unsigned char *fmt;
 unsigned char *a0,*a1,*a2,*a3,*a4,*a5,*a6,*a7,*a8,*a9,*aA,*aB,*aC,*aD,*aE,*aF;
{	unsigned char *fp,fc,*dp,*xp,*fdp,fmt1[8];
	unsigned char *vv[16],*v1;
	int vi,leng;

	vv[0x0] = a0; vv[0x1] = a1; vv[0x2] = a2; vv[0x3] = a3;
	vv[0x4] = a4; vv[0x5] = a5; vv[0x6] = a6; vv[0x7] = a7;
	vv[0x8] = a8; vv[0x9] = a9; vv[0xA] = aA; vv[0xB] = aB;
	vv[0xC] = aC; vv[0xD] = aD; vv[0xE] = aE; vv[0xF] = aF;
	vi = 0;

	xp = &StrH->sh_base[StrH->sh_size] - 1;
	dp = &StrH->sh_base[StrH->sh_peak];

	for( fp = fmt; fc = *fp; fp++ ){
		if( xp <= dp ){
			break;
		}
		if( fc != '%' ){
			*dp++ = fc;
			continue;
		}
		fc = *++fp;
		if( fc == 0 )
			break;
		if( fc == '%' ){
			*dp++ = fc;
			continue;
		}
		fdp = fmt1;
		*fdp++ = '%';
		if( fc == '-' ){
			*fdp++ = '-';
			fc = *++fp;
		}
		for(;;){
			if( '0' <= fc && fc <= '9' || fc == '.' ){
				*fdp++ = fc;
				fc = *++fp;
			}else	break;
		}
		*fdp++ = fc;
		*fdp = 0;

		if( fc == 0 ){
			break;
		}

		v1 = vv[vi++];
		switch( fc ){
		  case 's':
			if( v1 == 0 ){
				strcpy(dp,"(null)");
				dp += strlen(dp);
				continue;
			}
			leng = strlen(v1);
			if( xp < dp+leng ){
				if( 0 < fmtDBG )
				fmtdebug(StrH,"Str_sprintf","overflow",fmt,
					&dp[leng]-StrH->sh_base);
				leng = xp - dp - 4;
				if( 0 < leng ){
					bcopy(v1,dp,leng);
					strcpy(dp+leng,"=-=\n");
					dp += leng + 4;
				}
				goto OUT;
			}
			break;
		  case 'c':
		  case 'd':
		  case 'x':
		  case 'X':
			break;
		  default:
			fmtNO++;
			if( 1 < fmtDBG )
			fmtdebug(StrH,"Str_sprintf","unsupported",fmt,0);
			/* this includes "%*X" format */
			return -1;
		}
		sprintf(dp,fmt1,v1);
		dp += strlen(dp);
	}
OUT:
	if( dp <= xp )
		*dp = 0;
	fmtOK++;
	return 1;
}
str_sprintf(StrH,form,a,b,c,d,e,f,g,h,i,j,k,l,m,n)
	StrHead *StrH;
	char *form,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k,*l,*m,*n;
{	unsigned char *peakp;
	int size,peak,rem,wlen;

	size = StrH->sh_size;
	peak = StrH->sh_peak;
	rem = size - peak;

	peakp = &StrH->sh_base[peak];
	if( Str_sprintf(StrH,form,a,b,c,d,e,f,g,h,i,j,k,l,m,n) < 0 )
		sprintf((char*)peakp,form,a,b,c,d,e,f,g,h,i,j,k,l,m,n);

	wlen = strlen(peakp);
	if( size <= peak + wlen )
		fmtdebug(StrH,"str_sprintf","overflow",form,peak+wlen);

	StrH->sh_peak += wlen;
	str_sflush(StrH);
	return  wlen;
}
str_fseek(Str,off,where)
	String *Str;
{
	if( !str_isSTR(Str) )
		return fseek((FILE*)Str,off,where);
	return str_sseek(&Str->s_STRING,off,where);
}
str_sseek(StrH,off,where)
	StrHead *StrH;
{	int noff;

	switch( where ){
		case 0: noff = off; break;
		case 1: noff = StrH->sh_peak + off; break;
		case 2: noff = StrH->sh_size-1 + off; break;
		default: return -1;
	}

	if( noff < 0 || StrH->sh_size <= noff )
		return -1;
	StrH->sh_peak = noff;
}
str_ftell(Str)
	String *Str;
{
	if( !str_isSTR(Str) )
		return ftell((FILE*)Str);

	return Str->s_peak;
}
str_stell(StrH)
	StrHead *StrH;
{
	return StrH->sh_peak;
}
Uchar *
str_sptell(StrH)
	StrHead *StrH;
{
	return &StrH->sh_base[StrH->sh_peak];
}
