/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1997 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, modify, and distribute this material for any
purpose and without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies, and
that the name of ETL not be used in advertising or publicity pertaining
to this material without the specific, prior written permission of an
authorized representative of ETL.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	passwd.c
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:
History:
	970514	extracted from misc.c
//////////////////////////////////////////////////////////////////////#*/
#include <stdio.h>
#include "ystring.h"

#include <sys/types.h>
#include <sys/stat.h>
#ifdef S_ISUID /* set UID on execution */
#include <pwd.h>
#include <grp.h>
#else
#include "passwd.h"
static struct passwd passwd0 = { "?", "", 0, 0, 0, "", "", "/", "" };
#define getpwuid(uid)	(&passwd0)
#define getpwnam(name)	(&passwd0)

static struct group group0 = { "?", "", 0, 0 };
#define getgrgid(gid)	(&group0)
#define getgrnam(gid)	(&group0)
#endif


usedFD(usedfd)
{	int fd;

	fd = dup(0);
	close(fd);
	if( fd != usedfd ){
		/*fprintf(stderr,"SETFD[%d]\n",usedfd);*/
		syslog_DEBUG("usedFD():setCloseOnExec(%d)\n",fd);
		setCloseOnExec(usedfd);
	}
}
nextFD(){
	int fd;

	fd = dup(0);
	close(fd);
	return fd;
}


/* Solaris2.X leaves NIS+? relevant fd be opened after exec() */
struct passwd *GETpwuid(uid)
{	struct passwd *pw;
	int fd;

	fd = nextFD();
	pw = getpwuid(uid);
	usedFD(fd);
	return pw;
}
struct group *GETgrgid(gid)
{	struct group *gr;
	int fd;

	fd = nextFD();
	gr = getgrgid(gid);
	usedFD(fd);
	return gr;
}
struct passwd *GETpwnam(name) char *name;
{	struct passwd *pw;
	int fd;

	fd = nextFD();
	pw = getpwnam(name);
	usedFD(fd);
	return pw;
}
struct group *GETgrnam(group) char *group;
{	struct group *gr;
	int fd;

	fd = nextFD();
	gr = getgrnam(group);
	usedFD(fd);
	return gr;
}

char *getUsername(uid,name)
	char *name;
{	struct passwd *passwd;

	if( passwd = GETpwuid(uid) )
		strcpy(name,passwd->pw_name);
	else	sprintf(name,"#%d",uid);
	return name;
}

typedef struct {
	int ue_LastUid;
	char ue_LastUname[32];
} UidnameCache;
static UidnameCache *uidnameCache;
#define LastUid   uidnameCache->ue_LastUid
#define LastUname uidnameCache->ue_LastUname
char *getUsernameCached(uid,name)
	char *name;
{
	if( uidnameCache == 0 ){
		uidnameCache = NewStruct(UidnameCache);
		LastUid = -1;
	}
	if( uid == LastUid && LastUname[0] != 0 )
		strcpy(name,LastUname);
	else{
		getUsername(uid,name);
		strcpy(LastUname,name);
		LastUid = uid;
	}
	return name;
}

getHOME(uid,home)
	char *home;
{	struct passwd *passwd;

	if( passwd = GETpwuid(uid) )
		strcpy(home,passwd->pw_dir);
	else	strcpy(home,"/");
}
getSHELL(uid,shell)
	char *shell;
{	struct passwd *passwd;

	if( passwd = GETpwuid(uid) ){
		strcpy(shell,passwd->pw_shell);
		return 1;
	}
	return 0;
}
char *getGroupname(gid,name)
	char *name;
{	struct group *group;

	if( group = GETgrgid(gid) )
		strcpy(name,group->gr_name);
	else	sprintf(name,"#%d",gid);
	return name;
}
char *getusernames(names)
	char *names;
{	char uname[32],euname[32];
	char gname[32],egname[32];

	uname[0] = euname[0] = 0;
	getUsername(getuid(),uname);
	getUsername(geteuid(),euname);

	gname[0] = egname[0] = 0;
	getGroupname(getgid(),gname);
	getGroupname(getegid(),egname);

	sprintf(names,"%s/%s(%s/%s)",uname,gname,euname,egname);
	return names;
}

/*
 *	user_group:	user / group ( euser / egroup )
 */
scan_guid(user_group,uidp,gidp)
	char *user_group;
	int *uidp,*gidp;
{	char *dp,user[256],group[256];
	struct passwd *pw;
	struct group *gr;
	int len;
	int uid,gid;

	if( dp = strchr(user_group,'/') ){
		if( len = dp - user_group ){
			strncpy(user,user_group,len);
			user[len] = 0;
		}else	getUsername(getuid(),user);
		strcpy(group,dp+1);
	}else{
		strcpy(user,user_group);
		group[0] = 0;
	}

	uid = gid = -1;

	if( *user == '#' ){
		uid = atoi(user+1) & 0xFFFF;
		if( pw = GETpwuid(uid) )
			gid = pw->pw_gid;
	}else
	if( pw = GETpwnam(user) ){
		uid = pw->pw_uid;
		gid = pw->pw_gid;
	}else	return -1;

	if( *group == '#' )
		gid = atoi(group+1);
	else
	if( gr = GETgrnam(group) )
		gid = gr->gr_gid;

	if( uidp ) *uidp = uid;
	if( gidp ) *gidp = gid;
	return 0;
}

static struct passwd *_pwuids;
#define _pwuid	_pwuids[0]
struct passwd *getpwuid_nis(map,domain,uid)
	char *map,*domain;
{	char *ypdomain,*key,*val,*vp;
	char uids[32];
	int vlen;
	int rcode;

	if( _pwuids == 0 )
		_pwuids = NewStruct(struct passwd);

	if( _pwuid.pw_name ){
		if( _pwuid.pw_uid == uid )
			return &_pwuid;
		free(_pwuid.pw_name);
		bzero(&_pwuid,sizeof(struct passwd));
	}

	if( map == NULL ){
		map = "passwd.byuid";
	}

	if( domain != NULL && domain[0] != 0 )
		ypdomain = domain;
	else{
		ypdomain = NULL;
		rcode = yp_get_default_domain(&ypdomain);
		if( rcode != 0 || ypdomain == NULL || *ypdomain == 0 )
			return NULL;
		if( strcmp(ypdomain,"(none)") == 0 )
			return NULL;
	}

	sprintf(uids,"%d",uid);
	rcode = yp_match(ypdomain,map,uids,strlen(uids),&val,&vlen);
	if( rcode != 0 )
		return NULL;

	val[vlen] = 0;

	_pwuid.pw_name = vp = stralloc(val);
	if( vp = strchr(vp,':') ){
		*vp++ = 0;
		_pwuid.pw_passwd = vp;
	if( vp = strchr(vp,':') ){
		*vp++ = 0;
		_pwuid.pw_uid = atoi(vp);
	if( vp = strchr(vp,':') ){
		*vp++ = 0;
		_pwuid.pw_gid = atoi(vp);
	if( vp = strchr(vp,':') ){
		*vp++ = 0;
		_pwuid.pw_gecos = vp;
	if( vp = strchr(vp,':') ){
		*vp++ = 0;
		_pwuid.pw_dir = vp;
	if( vp = strchr(vp,':') ){
		*vp++ = 0;
		_pwuid.pw_shell = vp;
	}}}}}}

	return &_pwuid;
}
struct passwd *Getpwuid(uid)
{	struct passwd *pw;

	if( pw = getpwuid(uid) )
		return pw;
	else	return getpwuid_nis(NULL,NULL,uid);
}

#ifdef MAIN
main(ac,av)
	char *av[];
{	struct passwd *pp;

	if( pp = Getpwuid(atoi(av[1])) ){
		printf("name=	%s\n",pp->pw_name);
		printf("pass=	%s\n",pp->pw_passwd);
		printf("uid=	%d\n",pp->pw_uid);
		printf("gid=	%d\n",pp->pw_gid);
		printf("gecos=	%s\n",pp->pw_gecos);
		printf("dir=	%s\n",pp->pw_dir);
		printf("shell=	%s\n",pp->pw_shell);
	}
}
#endif
